/*  file: keyboard.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stdio.h>
#include <stddef.h>
#include <string.h>
#ifndef SUNOS4
#include <unistd.h> /* For _POSIX_VDISABLE  (not in <unixstd.h> on sunos4) */
#endif

#include "kehpager.h"
#include "memory.h"
#include "charset.h"

#include "keyboard.h"
#include "rc.h"
#include "esc.h"
#include "terminal.h"
#include "control.h"

int key_quit = -1;
int key_tab = -1;
int key_prompt = -1;
int key_pagemode = -1;
int key_time = -1;
int key_wrap = -1;
int key_spc = -1;
int key_next = -1;
int key_page = -1;
int key_prev = -1;
int key_help = -1;
int key_redraw = -1;
int key_lnext = -1;
int key_lprev = -1;
int key_top = -1;
int key_bottom = -1;
int key_mquit = -1;
int key_goto = -1;
int key_copyright = -1;

int key_genter = -1;
int key_gpercnt = -1;
int key_search = -1;
int key_searchn = -1;
int key_searchf = -1;
int key_ssearchn = -1;
int key_ssearchf = -1;
int key_sinsert = -1;
int key_senter = -1;
int key_debug = -1;

int key_reread = -1;
int key_forward = -1;

int key_nl = -1;
int key_nl_unix = -1;
int key_nl_network = -1;
int key_nl_enter = -1;

int key_ct = -1;
int key_ct_insert = -1;
int key_ct_enter = -1;

int key_ts = -1;
int key_ts_insert = -1;
int key_ts_enter = -1;

int key_is = -1;
int key_is_insert = -1;
int key_is_enter = -1;

int key_charset = -1;
int key_cinsert = -1;
int key_center = -1;

int key_edel = -1;
int key_eerase = -1;

/* Control keys */
int key_cancel  = -1;
int key_compose = -1;
int key_kill    = -1;
int key_suspend = -1;
int key_flush   = -1;
int key_abort   = -1;
int key_res1    = -1;
int key_res2    = -1;


int key_nogroup = -1;

init_item keyboard_items[] = {
  /* Control keys */
  { &key_cancel, "keyboard.cancel",        V_KEY },
  { &key_compose,"keyboard.compose",       V_KEY },
  { &key_kill,   "keyboard.kill",          V_KEY },
  { &key_suspend,"keyboard.suspend",       V_KEY },
  { &key_flush,  "keyboard.flush",         V_KEY },
  { &key_abort,  "#(Abort key)",           V_KEY },
  { &key_res1,   "#(Reserved 1)",          V_KEY },
  { &key_res2,   "#(reserved 2)",          V_KEY },

  /* Action keys */
  { &key_quit,  "keyboard.quit",           V_KEY },
  { &key_copyright, "keyboard.copyright",  V_KEY },
  { &key_debug, "keyboard.debug_screen",   V_KEY },
  { &key_help,  "keyboard.give_help",      V_KEY },
  { &key_tab,   "keyboard.toggle_tabbing", V_KEY },
  { &key_prompt,"keyboard.toggle_prompt",  V_KEY },
  { &key_pagemode,"keyboard.toggle_pagemode",V_KEY },
  { &key_time,  "keyboard.toggle_time",    V_KEY },
  { &key_wrap,  "keyboard.toggle_wrapping",V_KEY },
  { &key_spc,   "keyboard.toggle_eatspc",  V_KEY },
  { &key_next,  "keyboard.next_file",      V_KEY },
  { &key_page,  "keyboard.next_page",      V_KEY },
  { &key_prev,  "keyboard.prev_page",      V_KEY },
  { &key_redraw,"keyboard.redraw",         V_KEY },
  { &key_lnext, "keyboard.next_line",      V_KEY },
  { &key_lprev, "keyboard.prev_line",      V_KEY },
  { &key_top,   "keyboard.top",            V_KEY },
  { &key_bottom,"keyboard.bottom",         V_KEY },
  { &key_mquit, "keyboard.message.clear",  V_KEY },
  { &key_reread, "keyboard.reread_file",   V_KEY },
  { &key_forward, "keyboard.forward",      V_KEY },
  { &key_nl,         "keyboard.eoln",                 V_KEY },
  { &key_nl_unix,    "keyboard.eoln.unix",            V_KEY },
  { &key_nl_network, "keyboard.eoln.network",         V_KEY },
  { &key_nl_enter,   "keyboard.eoln.enter",           V_KEY },
  { &key_ct,         "keyboard.content",              V_KEY },
  { &key_ct_insert,  "keyboard.content.self_insert",  V_KEY },
  { &key_ct_enter,   "keyboard.content.enter",        V_KEY },
  { &key_ts,         "keyboard.tab_step",             V_KEY },
  { &key_ts_insert,  "keyboard.tab_step.self_insert", V_KEY },
  { &key_ts_enter,   "keyboard.tab_step.enter",       V_KEY },
  { &key_is,         "keyboard.indent_step",             V_KEY },
  { &key_is_insert,  "keyboard.indent_step.self_insert", V_KEY },
  { &key_is_enter,   "keyboard.indent_step.enter",       V_KEY },

  { &key_goto,  "keyboard.goto",           V_KEY },
  { &key_genter,"keyboard.goto.enter",     V_KEY },
  { &key_gpercnt,"keyboard.goto.percent",   V_KEY },
  { &key_search,"keyboard.search",         V_KEY },
  { &key_searchf,"keyboard.search_first", V_KEY },
  { &key_searchn,"keyboard.search_next",   V_KEY },
  { &key_ssearchn,"keyboard.search.next",         V_KEY },
  { &key_ssearchf,"keyboard.search.first",        V_KEY },
  { &key_sinsert,  "keyboard.search.self_insert", V_KEY },
  { &key_senter,"keyboard.search.enter",          V_KEY },
  { &key_charset,"keyboard.charset",              V_KEY },
  { &key_cinsert,"keyboard.charset.self_insert",  V_KEY },
  { &key_center, "keyboard.charset.enter",        V_KEY },

  { &key_edel,   "keyboard.edit.del",             V_KEY },
  { &key_eerase, "keyboard.edit.erase_line",      V_KEY },

  /* OBSOLENT: */
  { &key_edel,    "keyboard.goto.del",            V_KEY },
  { &key_eerase,  "keyboard.goto.erase_line",     V_KEY },
  { &key_edel,    "keyboard.search.del",          V_KEY },
  { &key_eerase,  "keyboard.search.erase_line",   V_KEY },

  { NULL, NULL, V_LAST }
};

static int * groups [GROUP_count][40] = { 
  /* G_PAGER */
  { &key_quit, &key_help, &key_tab, &key_prompt, &key_wrap, &key_spc, 
      &key_time, &key_pagemode,
      &key_next, &key_page, &key_prev, &key_redraw,
      &key_lnext, &key_lprev, &key_top, &key_bottom, 
      &key_goto, &key_search, &key_searchn, &key_searchf,
      &key_debug, &key_reread, &key_forward, &key_charset, 
      &key_cancel, &key_copyright, &key_nl, &key_ct, &key_ts, &key_is,
      NULL },
  /* G_MESSAGE */
  { &key_quit, &key_help, &key_tab, &key_prompt, &key_wrap, &key_spc, 
      &key_time, &key_pagemode,
      &key_next, &key_page, &key_prev, &key_redraw,
      &key_lnext, &key_lprev, &key_top, &key_bottom, 
      &key_goto, &key_search, &key_searchn, &key_searchf,
      &key_debug, &key_cancel, NULL },
  /* G_MESSAGE_LAST */
  { &key_quit, &key_help, &key_tab, &key_prompt, &key_wrap, &key_spc, 
      &key_time, &key_pagemode,
      &key_next,&key_prev, &key_redraw, &key_mquit,
      &key_lprev, &key_top, &key_bottom, 
      &key_goto, &key_search, &key_searchn, &key_searchf,
      &key_debug, &key_cancel, NULL },
  /* G_GOTO */
  { &key_quit, &key_goto, &key_edel, &key_genter, &key_gpercnt, 
      &key_redraw, &key_eerase, &key_cancel, NULL },
  /* G_SEARCH */
  { &key_quit, &key_senter, &key_edel, &key_ssearchn, &key_ssearchf,
      &key_redraw, &key_eerase, &key_sinsert, &key_cancel, NULL },
  /* G_CHARSET */
  { &key_quit, &key_edel, &key_center, &key_cinsert,
      &key_redraw, &key_eerase, &key_cancel, NULL },
  /* G_NEWLINE */
  { &key_quit, &key_edel, &key_eerase, &key_nl_unix, &key_nl_network,
      &key_nl_enter, &key_cancel, NULL },
  /* G_CONTENT */
  { &key_quit, &key_edel, &key_eerase, &key_ct_insert, &key_ct_enter,
      &key_cancel, NULL },
  /* G_TABSTEP */
  { &key_quit, &key_edel, &key_eerase, &key_ts_insert, &key_ts_enter,
      &key_cancel, NULL },
  /* G_INDENTSTEP */
  { &key_quit, &key_edel, &key_eerase, &key_is_insert, &key_is_enter,
      &key_cancel, NULL },
};

static struct KL {
  char * keyname;
  struct command cm;
} keys [] = {
  { "UP",    { C_escape, 'A', 0 , NULL } },
  { "DOWN",  { C_escape, 'B', 0 , NULL } },
  { "RIGHT",  { C_escape, 'C', 0 , NULL } },
  { "LEFT", { C_escape, 'D', 0 , NULL } },
  { "PF1",   { C_single, 'P', 1 , "\217" /* SS3 */ } },
  { "PF2",   { C_single, 'Q', 1 , "\217" /* SS3 */ } },
  { "PF3",   { C_single, 'R', 1 , "\217" /* SS3 */ } },
  { "PF4",   { C_single, 'S', 1 , "\217" /* SS3 */ } },
  { "F6",    { C_escape, '~', 2 , "17" } },
  { "F7",    { C_escape, '~', 2 , "18" } },
  { "F8",    { C_escape, '~', 2 , "19" } },
  { "F9",    { C_escape, '~', 2 , "20" } },
  { "F10",   { C_escape, '~', 2 , "21" } },
  { "F11",   { C_escape, '~', 2 , "23" } },
  { "F12",   { C_escape, '~', 2 , "24" } },
  { "F13",   { C_escape, '~', 2 , "25" } },
  { "F14",   { C_escape, '~', 2 , "26" } },
  { "HELP",  { C_escape, '~', 2 , "28" } },
  { "Do",    { C_escape, '~', 2 , "29" } },
  { "F17",   { C_escape, '~', 2 , "31" } },
  { "F18",   { C_escape, '~', 2 , "32" } },
  { "F19",   { C_escape, '~', 2 , "33" } },
  { "F20",   { C_escape, '~', 2 , "34" } },
  { "FIND",  { C_escape, '~', 1 , "1" } },
  { "INSERT HERE", { C_escape, '~', 1 , "2" } },
  { "NEXT SCREEN", { C_escape, '~', 1 , "6" } },
  { "PREV SCREEN", { C_escape, '~', 1 , "5" } },
  { "REMOVE",{ C_escape, '~', 1 , "3" } },
  { "SELECT", { C_escape, '~', 1 , "4" } },
  { "HOME",   { C_escape, 'H', 0 , "" } },
  { "Ctrl-SPACE", { C_ctl, ' ' & 31, 0 , NULL } },    /* NUL */
  { "Ctrl-A", { C_ctl, 'A' & 31, 0 , NULL } },
  { "Ctrl-B", { C_ctl, 'B' & 31, 0 , NULL } },
  { "Ctrl-C", { C_ctl, 'C' & 31, 0 , NULL } },
  { "Ctrl-D", { C_ctl, 'D' & 31, 0 , NULL } },
  { "Ctrl-E", { C_ctl, 'E' & 31, 0 , NULL } },
  { "Ctrl-F", { C_ctl, 'F' & 31, 0 , NULL } },
  { "Ctrl-G", { C_ctl, 'G' & 31, 0 , NULL } },
  { "BACKSPACE", { C_ctl, BS, 0 , NULL } },
  { "TAB",   { C_ctl, TAB, 0 , NULL } },
  { "LINEFEED",   { C_ctl, LF, 0 , NULL } },
  { "Ctrl-K", { C_ctl, 'K' & 31, 0 , NULL } },
  { "Ctrl-L", { C_ctl, 'L' & 31, 0 , NULL } },
  { "RETURN",{ C_ctl, CR, 0 , NULL } },
  { "Ctrl-N", { C_ctl, 'N' & 31, 0 , NULL } },
  { "Ctrl-O", { C_ctl, 'O' & 31, 0 , NULL } },
  { "Ctrl-P", { C_ctl, 'P' & 31, 0 , NULL } },
  { "Ctrl-Q", { C_ctl, 'Q' & 31, 0 , NULL } },
  { "Ctrl-R", { C_ctl, 'R' & 31, 0 , NULL } },
  { "Ctrl-S", { C_ctl, 'S' & 31, 0 , NULL } },
  { "Ctrl-T", { C_ctl, 'T' & 31, 0 , NULL } },
  { "Ctrl-U", { C_ctl, 'U' & 31, 0 , NULL } },
  { "Ctrl-V", { C_ctl, 'V' & 31, 0 , NULL } },
  { "Ctrl-W", { C_ctl, 'W' & 31, 0 , NULL } },
  { "Ctrl-X", { C_ctl, 'X' & 31, 0 , NULL } },
  { "Ctrl-Y", { C_ctl, 'Y' & 31, 0 , NULL } },
  { "Ctrl-Z", { C_ctl, 'Z' & 31, 0 , NULL } },
  { "Ctrl-\\", { C_ctl, '\\' & 31, 0 , NULL } },
  { "Ctrl-]", { C_ctl, ']' & 31, 0 , NULL } },
  { "Ctrl-^", { C_ctl, '^' & 31, 0 , NULL } },
  { "Ctrl-_", { C_ctl, '_' & 31, 0 , NULL } },
  { "DEL",   { C_ctl, DEL, 0 , NULL } },

};

static struct NL {
  char *name;
  char c;
} namelist[] = {
  { "SPACE", ' ' },
  { "NUMBER SIGN", '#' },
  { "QUOTATION MARK", '"' }
};

static int key_count = 0;
static struct SN {
  enum KT { K_null,K_keys, K_namelist, K_char, K_letter } type;
  int indx;
  int next;
  int *pos;
} *bindings = NULL;

int map_key(CHAR *key, int *pos) {
  enum KT type = K_null;
  CHAR *ptr;
  CHAR_IDX indx = -1;
  if  (0 == strcasecmp(Cs(key),"LETTER")) {
    type = K_letter;
    indx = 0;
  } else  if (3 == strlen(Cs(key)) && '"' == key[0] && '"' == key[2]) {
    type = K_char;
    map_input(MAP_ASCII,1,key+1,&indx);
  } else if (NULL != (ptr = rCs(strchr(Cs(key),':')))) {
    CHAR buffer[100];
    int l = ptr-key < 99 ? ptr-key : 99;
    strncpy(Cs(buffer),Cs(key),l);
    while (l > 0 && ' ' == buffer[l-1]) l--;
    buffer[l] = '\0';
    while (' ' == *(ptr+1)) ptr++;
    ptr++;
    if (3 == strlen(Cs(ptr)) && '"' == ptr[0] && '"' == ptr[2]) {
      int map = find_map(buffer);
      if (-1 == map) return -1;
      map_input(map,1,ptr+1,&indx);
      type = K_char;
    } else return -1;
  } else { 
    int i;
    for (i = 0; i < sizeof (namelist) / sizeof (struct NL); i++)
      if (0 == strcasecmp(namelist[i].name,Cs(key))) {
	indx = i;
	type = K_namelist;
	break;
      }
    for (i = 0; i < sizeof (keys) / sizeof (struct KL); i++)
      if (0 == strcasecmp(keys[i].keyname,Cs(key))) {
	indx = i;
	type = K_keys;
	break;
      }
  }
  if (-1 == indx) return -1;

  bindings = (struct SN *) REALLOC((void *)bindings,
				   (key_count+1) * sizeof (struct SN));
  bindings[key_count].type = type;
  bindings[key_count].indx = indx;
  bindings[key_count].pos  = pos;
  bindings[key_count].next = *pos;
  return key_count++; /* -> *pos */ 
}

static int compose = 0;
static int cmp[2];

static int *accept_compose(int group, int ch) {
  int wrong_group = 0;
  int i;

  compose = 0;

  for (i = key_count-1; i >= 0; i--) 
    if (K_letter ==  bindings[i].type || 
	(bindings[i].type == K_char &&
	bindings[i].indx == ch)) {
      int l;
      for(l = 0; NULL != groups[group][l]; l++)
	if (bindings[i].pos == groups[group][l])
	  return bindings[i].pos;
      wrong_group = 1;
    }
  
  if (wrong_group)
    return &key_nogroup;
  return NULL;
}

static int scan_composed(CHAR_IDX *ch) {
  CHAR ch1, ch2;
  int i,rev=-1;
  if (3 != compose) {
    print_notify("Error/scan_composed: 3 != compose=%d",compose);
    compose = 0;
    return 0;
  }
  compose = 0;
  if (-1 == chars[cmp[0]].vector[MAP_ASCII] ||
      -1 == chars[cmp[1]].vector[MAP_ASCII]) 
    return 0;

  ch1 = chars[cmp[0]].vector[MAP_ASCII];
  ch2 = chars[cmp[1]].vector[MAP_ASCII];

  for (i = 0; i < CHARS_COUNT_; i++) 
    if (Cc(ch1) == chars[i].compose[0] && 
	Cc(ch2) == chars[i].compose[1]) {
      *ch = i;
      return 1;
    } else if (Cc(ch2) == chars[i].compose[0] && 
	Cc(ch1) == chars[i].compose[1]) 
      rev = i;

  if (rev >= 0) {
    *ch = rev;
    return 1;
  }

  return 0;
}


int *parse_key(int cmd, int group, CHAR_IDX *ch) {
  int i, wrong_group = 0;
  struct command k = commands[cmd];
  CHAR *A = rCs(k.char_arr ? k.char_arr : "");
  map_input(MAP_ASCII,1,&k.cmd,ch);

  if (1 == compose) 
    return accept_compose(group,*ch);

  for (i = key_count-1; i >= 0; i--) {
    struct command j;
    CHAR *B;
    if (K_keys != bindings[i].type) continue;
    j = keys[bindings[i].indx].cm;
    B = rCs(j.char_arr ? j.char_arr : "");
    if (j.what == k.what &&
	j.cmd == k.cmd &&
	j.len == k.len &&
	0 == strcmp(Cs(A),Cs(B))) {
      int l;

      if (0 == compose && bindings[i].pos == &key_compose) {
	compose = 1;
	return &key_nogroup;
      }

      for(l = 0; NULL != groups[group][l]; l++)
	if (bindings[i].pos == groups[group][l]) {
	  compose = 0;
	  return bindings[i].pos;
	}
      wrong_group = 1;
    }
  }
  compose = 0;
  if (wrong_group) 
    return &key_nogroup;
  return NULL;
}

int *parse_char(CHAR c, int group, CHAR_IDX *ch) {
  int i, wrong_group = 0;
  enum KT type = K_null;
  int indx = -1;
  map_input(keyboard_charset,1,&c,ch);

  if (compose > 0) {
    cmp[compose-1] = *ch;
    compose++;
    if (compose < 3)
      return &key_nogroup;
    
    if (scan_composed(ch))
      return accept_compose(group,*ch);
    else {
      compose = 0;
      return &key_nogroup;
    }
  }

  for (i = 0; i < sizeof (namelist) / sizeof ( struct NL); i++) 
    if (namelist[i].c == Cc(c)) {
      type = K_namelist;
      indx = i;
      break;
    }
  if (-1 == indx) {
    indx = *ch;
    type = K_char;
  }
    
  for (i = key_count-1; i >= 0; i--) 
    if (K_letter ==  bindings[i].type || 
	(bindings[i].type == type &&
	 bindings[i].indx == indx)) {
      int l;
      for(l = 0; NULL != groups[group][l]; l++)
	if (bindings[i].pos == groups[group][l])
	  return bindings[i].pos;
      wrong_group = 1;
    }
     
  if (wrong_group)
    return &key_nogroup;
  return NULL;
}

static CHAR *char_in_ascii(int ch) {
  static CHAR buffer[6];
  if (ch >= 0) {
    if (-1 != chars[ch].vector[MAP_ASCII]) {
      buffer[0] = chars[ch].vector[MAP_ASCII];
      buffer[1] = '\0';
      return buffer;
    }
    return rCs(chars[ch].fallback);
  }
  sprintf(Cs(buffer),"`%02X",-ch);
  return buffer;
}

CHAR *give_ctl_mapping(int *pos) {
  int i;
  for (i = *pos; -1 != i; i = bindings[i].next)
    if (K_keys == bindings[i].type &&
	C_ctl == keys[bindings[i].indx].cm.what) 
      return rCs(keys[bindings[i].indx].keyname);
  return NULL;
}

CHAR *give_mapping(int *pos) {
  if (-1 == *pos) return NULL;
  switch(bindings[*pos].type) {
  case K_letter:
    return rCs("LETTER");
  case K_char:
    return char_in_ascii(bindings[*pos].indx);
  case K_namelist:
    return rCs(namelist[bindings[*pos].indx].name);
  case K_keys:
    return rCs(keys[bindings[*pos].indx].keyname);
  default:
    return NULL;
  }
}

int control_key(int *pos, CHAR *ch) {
  int ptr;
  for (ptr = *pos; -1 != ptr; ptr = bindings[ptr].next)
    if (K_keys == bindings[ptr].type &&
	C_ctl == keys[bindings[ptr].indx].cm.what) {
      *ch = keys[bindings[ptr].indx].cm.cmd;
      return 1;
    }
  return 0;
}

static void *var_name(int * pos) {
  int i;
  for (i = 0; V_LAST != keyboard_items[i].what; i++)
    if (pos == keyboard_items[i].variable)
      return keyboard_items[i].name;
  return NULL;
}

void make_default(int *pos, CHAR ch) {
  CHAR tmp;
  int i,indx=-1;
  if (control_key(pos,&tmp)) return;
#ifdef SUNOS4
  /* SunOS 4 not include _POSIX_VDISABLE in <unistd.h> */
  if (0 == ch) return;
#else
  if (_POSIX_VDISABLE == ch) return;
#endif

  for (i = 0; i < sizeof (keys) / sizeof (struct KL); i++)
    if (C_ctl == keys[i].cm.what &&
	ch == keys[i].cm.cmd) indx = i;

  if (-1 == indx) {
    print_notify("Can't generate default binding for %s (char=%02X)",
		 var_name(pos), ch);
  }

  bindings = (struct SN *) REALLOC((void *)bindings,
				   (key_count+1) * sizeof (struct SN));
  bindings[key_count].type = K_keys;
  bindings[key_count].indx = indx;
  bindings[key_count].pos  = pos;
  bindings[key_count].next = *pos;
  *pos = key_count++; 

}

CHAR *composing(void) {
  static CHAR buffer[20];
  if (0 == compose) return NULL;
  strcpy(Cs(buffer),Cs(give_mapping(&key_compose)));
  if (1 == compose) return buffer;
  strcat(Cs(buffer)," ");
  strcat(Cs(buffer),Cs(char_in_ascii(cmp[0])));
  if (2 == compose) return buffer;
  strcat(Cs(buffer)," ");
  strcat(Cs(buffer),Cs(char_in_ascii(cmp[1])));
  return buffer;
  
}


CHAR *key_name(int cmd) {
  int i;
  struct command k = commands[cmd];
  CHAR *A = rCs(k.char_arr ? k.char_arr : "");
  for (i = 0; i < sizeof(keys) / sizeof(struct KL); i++) {
    struct command j = keys[i].cm;
    CHAR *B = rCs(j.char_arr ? j.char_arr : "");
    if (j.what == k.what &&
	j.cmd == k.cmd &&
	j.len == k.len &&
	0 == strcmp(Cs(A),Cs(B))) return rCs(keys[i].keyname);
  }
  return NULL;
}
