/*  file: kehpager.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>  /* ANSI C stuff */
/* #include <signal.h> terminal.h includes this */

#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#if defined(SUNOS4) || defined(SUNOS5)
#include <sys/filio.h>
#else
#include <sys/ioctl.h>
#endif
#include <unistd.h>

#include <time.h>

#include <errno.h>
extern int errno;

#include "kehpager.h"
#include "memory.h"
#include "charset.h"

#include "terminal.h"
#include "esc.h"
#include "control.h"
#include "window.h"
#include "rc.h"
#include "env.h"
#include "keyboard.h"
#include "content.h"
#include "edit.h"

char *Version = "kehpager V1.2";
/* Author: Kari E. Hurtta <Kari.Hurtta@Helsinki.FI> */
char *prog="Unknown";
int quitflag = 0;
int debugflag = 0;

/* Versions:
 * 0.1  93-05-27 First version
 * 0.11 93-05-27 telentd/rlogind don't longer die when start kehpager
 * 0.12 93-05-27 Printing optimizing, 
 *               keys: b = previous page, DEL = backward one line
 * 0.13 93-06-06 option: - refers standard input, 
 *               binary clean (reading of NUL), background reading of file
 *               (or standard input)
 * 0.14 93-06-07 printing bug fixed, better error display
 * 0.15 93-06-10 .kehpagerrc init file, kehpager -e and KEHPAGER_TERM,
 *               keys: ? prints help page
 *               options: -e -et -eu for print defination of KEHPAGER_TERM
 *                        -f disables using of KEHPAGER_TERM
 *                        -t disables interpreting of TAB character
 *                        -t+ enables interpreting of tab character
 * 0.16 93-06-29 bug fig for character table, option: -T
 * 0.17 93-06-30 bug fix in visible_len (causes illegal indexing of array)
 * 0.18 93-07-12 bug fix for character table, do_escape
 *               cleaning MALLOC, REALLOC
 *               option: -Dn
 * 0.19 93-08-28 scrolling regions
 * 0.2  93-08-28 word wrap
 *               options: -w disables word wrap
 *                        -w+ enables word wrap
 *                        -t disables interpreting of TAB,SHY,NBSP -characters
 *                        -t+ enables interpreting of TAB,SHY,NBSP -characters
 *               keys: t  = Toggle interpreting of TAB, NBSP, SHY -characters
 *                     w  = Toggle word wrap
 * 0.21 93-08-29 bug fix for printing, down_line (set_hard)
 *               prompt
 *               options: -p disables prompt
 *                        -p+ enables prompt
 * 0.22 93-08-31 better handling of screen size change (recalculate prompt)
 *               bug fix in insert_line, wrap_line
 * 0.23 93-09-01 do_kayboard,printable_sequence,action
 *               module keyboard, global init file kehpager.init
 * 0.24 93-09-02 force latin/1 in xterm
 *               Options: -X, -X+, -X++
 * 0.25 93-09-02 UK-ASCII (option: -UK), Norwegian (option: -NR),
 *               Swedish (option: -SW)
 *               Option: -Nn
 *               key_bottom, key_top
 *               key_name
 *               bug fix in error's printing
 * 0.26 93-09-04 line wrapping rewritten, key groups, goto line, search
 *               Option: -CS cs
 *               Latin/4 (option: -L4), Latin/5 (option: -L5)
 *               MAP_LATIN_ANY (matches ISO-8859-*)
 * 0.27 93-09-09 cut_window_line, read_win_char, win_line_len
 *               key_serase, key_gerase
 *               Bug fix in calculate_prompt
 *               Options: -b, -b+, -a, -a+
 * 0.28 93-09-10 Bug fix in key defination table, show always search and
 *               goto prompt
 * 1.0  93-09-10 * Put to /usr/local/contrib as version 1.0 *
 * 1.01 93-09-11 Fix bad behauviour with Wyse vt320 (or NOT !!!)
 *               Notice if Y was bigger than terminal size on startup
 *               FL_uncertain flag
 *               send shoft reset command to terminal 
 * 1.02 93-09-19 Compiles on SunOS 5 (well, it dosn't work)
 *               use.terminal.lines, use.terminal.columns
 *               Option: -DS
 *               debug_page, give_debug_page
 *               key_debug, key_ssearchn, key_ssearchf, key_gpercnt,
 *               key_searchf, key_compose
 *               key_group changes follows better to commands
 *               Default search action changed (key_senter)
 *               Compose characters -feature
 *               Bug fix in find_buffer_text
 *               Optimize clearing rest of line
 *               Control keys bindings and defaults to/from terminal driver
 *               Compiles on HPUX
 *               Don't complain if can't set file to nonblocking mode
 * 1.03 93-10-03 composed input for BREVE, OGONEK and CARON
 *               LATIN2, French/Belgian, French Canadian
 *               key_reread, key_forward, keyedel, key_eerase, key_charset,
 *               key_center, key_cinsert
 *               (key_gdel,key_gerase,key_sdel,key_serase obsoleted)
 *               map_name, print_help, reread_file, reset_file_flags, eq_pair
 *               Info for composed input in internal help page
 *               find_buffer_text handles now CH_SHY and CH_NBSP specially
 * 1.04 93-10-16 bug fix in setup of c_cc[VMIN]
 *               Added CAPITAL LETTER Y WITH DIARESES to Multinational table 
 *               Changing of charset name is now possible (use.charset.label 
 *                  and use.renamed.charset)
 *               Added HOME -key
 *               Latin/3, MS-DOS CP437 (partially)
 *               options: -7, -8, -q, -q+, -L3, -PC
 *               cs_names, give_ctl_binding
 *               key_cancel
 * 1.05 93-10-18 Fixed singal handling on SYSV machines (SunOS 5 and HP-UX).
 *                 Now uses sigaction instead of signal.
 *               options: -s, -s+, -wt, -wt+
 *               use.scrolling,use.wait_quit
 *               set_win_prompt, calculate_prompt updated (now window = -2
 *                 means absolute cursor position)
 * 1.06 93-12-06 Better handling of terminals DA (device attributes) response
 *               - plain VT100 don't support ins/del charracters/lines, but
 *                 VT102 does - both supports scrolling regions
 *               scroll_area, T_scroll_area, T_scroll_area_R
 *                 flush_error_buffer, feed_queued_error,
 *                 put_stderr, deliver_signal, stop_me, friend_have_died,
 *                 kill_friend, friend, handle_sigchld, BLOCK, UNBLOCK,
 *                 got_from_TAIL, handle_sighup, friend_exit, copyright_page
 *               Tried use DEC Technical charset (MAP_TECHNICAL) but seems
 *                 that I don't get it working just now
 *               charset: Spanish
 *               options: -SP, -pl, -tr, -ct, -nu, -nn
 *               initfile field: input.content_type, use.technical,
 *                 force.feature.enabled, force.feature.disabled,
 *                 keyboard.copyright, keyboard.eoln, keyboard.eoln.enter,
 *                 keyboard.eoln.unix, keyboard.eoln.network, keyboard.content,
 *                 keyboard.content.enter, keyboard.content.self_insert,
 *                 use.vt100.national_available
 *               Banks assignments are done more dynamically
 *               bug fix in printable_len, reserve_line, print_window,
 *                 map_window, calculate_prompt
 *               new modules: content (content.c, content.h),
 *                 edit (edit.c, edit.h)
 *               kehpager now puts itself to own process group -
 *                 this fixes problem that, when kehpager is called from 
 *                 another program, that program isn't suspend (and shell not
 *                   get control) before kehpager returns terminal's correct 
 *                   state when user hits Ctrl-Z.
 *                 this isn't done when kehpager is it's process group leader
 *                 kehpager forks also another process when it creates new 
 *                   process group. This process guarantees that original 
 *                   process group is here when kehpager suspends itself
 *                   (otherwise it isn't availabel if all other processes
 *                    are died in job.)
 *                 In SunOS 4 this worked without that arragement, but not in
 *                   in SUnOS 5 or HP-UX
 *                 This wasn't problem when kehpager was invoked from shell
 *               Enviroment variable: KEHPAGER_DEBUG - it have format
 *                   <debug level>:<log file name>
 *                 this can be use instead of 
 *                   kehpager -D<debug level> ... >& <debug file name>
 *                 so debugging of kehpager is possible when kehpager acts as
 *                   pager
 *               Copyright notice
 *               Handle SIGHUP
 *               Seems that I got Technical charset working, but I don't still
 *                 understand these terminals...
 *               Newline conventions: unix, network
 *               Now compiles with HP's ANSI C compiler
 * 1.1  94-01-05 Charset: Macintosh
 *               Content-type: Enriched (Text/Enriched)
 *               Options: -MAC, -er
 *               Bug fix in reserve_line
 *               Many fields to window -structure (for enriched support)
 * 1.11 94-01-08 Buffering terminal output
 *               Options: -sp, -sp+, -j, -j+, -st, -st+, -rt, -tn, -in
 *               Initfile fields: keyboard.toggle_eatspc, input.eat_spaces,
 *                  input.full_justify, keyboard.toggle_time, input.tab_step,
 *                  input.indent_step
 *               key_eatspc, key_time
 *               Prettier FlushRight (and also other output)
 *               New field to window -structure
 *               Full justify of enriched text, nested excerpts now supported,
 *                 now multiple spaces are translated to one space except in
 *                 verbatin and nofill environments, hanging indentation is
 *                 now supported
 *               Shows current time in prompt
 *               Richtext content-type
 *               update_content (better handle of eof)
 *               Now compiles with Sun's acc (on SunOS 4)
 * 1.12 94-01-09 Avoid output loops with buffered output 
 *                 when incomplete input detected
 *               Fix bug what cause core on invalid input
 * 1.13 94-02-17 Bug fix in close_terminal (job control, companion prosess
 *                 was killed too early)
 *               Support for rc -shell, now $SHELL is matched agaist to table
 *               Options: -erc, -pm, -pm+, --, -nl, -nl+, -rs, -rs+
 *               init file fields: keyboard.toggle_pagemode, input.pagemode,
 *                 keyboard.tab_step,keyboard.tab_step.self_insert,
 *                 keyboard.tab_step.enter, keyboard.indent_step,
 *                 keyboard.indent_step.self_insert, 
 *                 keyboard.indent_step.enter, env.size, input.ignore_nl,
 *                 exit.newline, use.query_size
 *               content: If later (or nested) attribute is conflict with
 *                 previous attribute, only later is accepted; notably now
 *                 <Bold> works inside of <Signature> in Text/richtext
 *               clear_attr_conflict, sum_fl_attr, is_pagebreak,
 *                 win_is_page_top, repage_window, flush_window,
 *                 StrInit, StrFree, StrSet, StrAdd, StrCat, 
 *                 set_terminal_size, saved_undef, neg_undef,
 *                 terminal_hazards, update_priority, need_line_flush,
 *                 flush_line, flush_some_lines, flush_to_priority,
 *                 defogus_priorities, flush_S_buffer, ER_flushboth_enter,
 *                 ER_flushboth_exit, unblock, not_selectable, TABSET_ready,
 *                 INDENTSTEP_ready, give_syntax, import_vars
 *               key_ts, key_ts_insert, key_ts_enter, key_pagemode, key_is,
 *                 key_is_insert, key_is_enter
 *               Lazy update of screen
 *               Handling of pagebreaks (FF, richtext's <np>)
 *               Now handles no selectable inputs
 *               Option processing is disabable now ('option' --)
 *               Handling of terminal/tty driver 'hazards'
 *               Added some changes posted by 
 *                 Jrg Wunsch <joerg_wunsch@tcd-dresden.de> so this propably
 *                 compiles on 386BSD, FreeBSD and NetBSD
 *               VLNEXT_preserved
 *               $LINES overdrives use.terminal.lines and $COLUMNS overdrives
 *                  use.terminal.columns (if env.size is set)
 *               Updated Text/Enriched according RFC1563 (previous version
 *                 of enriched specification was RFC1523):
 *                 - Added FlushBoth environment
 *                 - Now Text/Enriched allows initial state justification 
 *                   to be any of FlushLeft, FlushRight or FlushBoth (in 
 *                   RFC1523 it was FlushBoth ie. fully justified). 
 *                 > Because kehpager heve difficulties with fully justified
 *                   text (kehpager may print justified text twise), I changed
 *                   default state to be FlushLeft. That is: default value
 *                   of input.full_justify is now false.
 *               Ignore newline immediately after <Nofill> of <Verbatim>.
 *                 This violates RFC1563, but does enriched text to prettier
 *                 (disabable with input.ignore_nl: false or with -nl).
 *               content.c simplified (or not)
 *               I marked VT500 as unsupported, because I don't have seen
 *                 manuals yet.
 *               Now kehpager asks terminal size from terminal itself (as
 *                 'resize' does it). Disable with -rs or 
 *                 use.query_size: Disabled
 *               Now esc.c don't eat nulls and binding to Ctrl-space
 *                 (ie. NUL) is possible
 * 1.14 94-03-26 charset: Italian, Mnemonics encoding, more characters from
 *                  MS-DOS CP437
 *               options: -IT, -MN, -MN:n
 *               composed input for HORN, RING BELOW, LINE BELOW and 
 *                 CIRCUMFLEX BELOW
 *               Ignoring of newline immediately after <Nofill> of <Verbatim>
 *                 REMOVED.
 *               open_table,close_table,map2_input,copy_table,name_table
 *               MAP_ITAL, MAP_RFC1345, MAP_Mnem (ENT_Mnem), ENT_name
 * 1.2  94-07-08 Avoid unnecessary redrawing of window when remapped by setting
 *                 windows priority temporary to zero (so now SEARCH: -prompt
 *                 works little nicer)
 *               Support for Unicode 1.1 with UTF-7 and UTF-8 encodings. 
 *                 Only small part of Unicode is supported. No support for
 *                 Unicode decomposition/composition.
 *               Options: -U7, -U8
 *               MAP_RFC134 replaced with MAP_UNICODE11
 *               MAP_UTF7, MAP_UTF8
 *               Now internal characters ire type CHAR_IDX
 *               Latin/3 213 fixed
 *                 from LATIN CAPITAL LETTER O WITH DOT ABOVE
 *                 to LATIN CAPITAL LETTER G WITH DOT ABOVE
 *               Latin/4 241 fixed
 *                 from LATIN SMALL LETTER N WITH OGONEK
 *                 to LATIN SMALL LETTER N WITH CEDILLA
 *               Latin/4 209 fixed
 *                 from LATIN CAPITAL LETTER N WITH OGONEK
 *                 to LATIN CAPITAL LETTER N WITH CEDILLA
 */

static int use_scrolling = 1;
static int wait_quit = 1;

static void goto_window(int win, int line) {
 int Y,offset,roffset,pagenum,pages;
 int LIMIT = use_scrolling ? lines*3/4 : lines/3;

 window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);
 if (roffset < line-1 && roffset > line-LIMIT) {
   while(roffset < line-1) {
     scroll_window(win);
     window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);     
     flush_some_lines();
   }
 } else if (roffset > line-1 && roffset < line+LIMIT) {
   while (roffset > line-1) {
     scroll_window_R(win);
     window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);     
     flush_some_lines();
   }
 } else if (roffset != line-1) {
   map_window(win,line-1,Y);
 }
 while (flush_some_lines());
}

static void go_next_page(int win) {
  int prev,offset,Y,roffset,pagenum,pages,prevp;

  set_win_prompt(-1,-1,-1);
  window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);
  if (use_scrolling) {
    for (window_pos(win,&Y,&offset,&roffset,&pagenum,&pages),
	 prev=roffset,prevp=pagenum; 
	 (window_rlines(win) - roffset > lines -Y || 
	  (0 < pagenum && pagenum < pages)) && 
	 roffset - prev <lines -1 && prevp == pagenum;
	 window_pos(win,&Y,&offset,&roffset,&pagenum,&pages)) {
      scroll_window(win); 
      flush_some_lines();
    }
  } else if (pages > 1) {
    unmap_window(win);
    /* currently works only with scrolling */
    for (window_pos(win,&Y,&offset,&roffset,&pagenum,&pages),
	 prev=roffset,prevp=pagenum; 
	 (window_rlines(win) - roffset > lines -Y || 
	  (0 < pagenum && pagenum < pages)) && 
	 roffset - prev <lines -1 && prevp == pagenum;
	 window_pos(win,&Y,&offset,&roffset,&pagenum,&pages)) 
      scroll_window(win); 
    map_window(win,roffset,Y);
  } else {
    int next;
    window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);
    next = roffset + lines -1;
    if (next > window_rlines(win) - lines + Y) 
      next = window_rlines(win) - lines + Y;
    map_window(win,next,Y);
  }
  while (flush_some_lines());
}

static void go_prev_page(int win) {
  int prev,offset,Y,roffset,pagenum,pages,top=0;
  int next;

  set_win_prompt(-1,-1,-1);
  window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);
  if (use_scrolling) {
    for (window_pos(win,&Y,&offset,&roffset,&pagenum,&pages),
	 prev=roffset; 
	 roffset > 0 && prev - roffset < lines-1 && !top;
	 window_pos(win,&Y,&offset,&roffset,&pagenum,&pages),
	 top = win_is_page_top(win)) {
      scroll_window_R(win); 
      flush_some_lines();
    }
  } else if (pages > 1) {
    /* only scrolling currently works with pagebreaks */
    unmap_window(win);
    for (window_pos(win,&Y,&offset,&roffset,&pagenum,&pages),
	 prev=roffset; 
	 roffset > 0 && prev - roffset < lines-1 && !top;
	 window_pos(win,&Y,&offset,&roffset,&pagenum,&pages),
	 top = win_is_page_top(win))
      scroll_window_R(win); 
    map_window(win,roffset,Y);
  } else {
    window_pos(win,&Y,&offset,&roffset,&pagenum,&pages);
    next = roffset - lines +1;
    if (next < 0) next = 0;
    map_window(win,next,Y);
  }
  while(flush_some_lines());
}

static void print_and_scroll(int win,int map,int flags, int len, 
			     CHAR *text) {
  CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));
  
  scroll_it(win);

  if (len > 0) {
    map_input(map,len,text,mapped);
    print_to_window(win,-1,-1,flags,len,mapped);
  } else {
    clear_window_line(win,-1);
  }
  
  FREE(mapped);
}


static void print_to_column(int win,int map,int flags, int X,int len, 
			    unsigned char *text) {
  CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));

  int Y = window_lines(win);
  if (Y < 1) Y = 1;

  map_input(map,len,text,mapped);
  print_to_window(win,X,Y,flags,len,mapped);

  FREE(mapped);

}


static int have_prompt = 1;
static int pline = 0;
static int prev_prompt[1000];
static int prev_prompt_len = 0;
static int key_group = G_PAGER;
static int map = MAP_LATIN1;
static CHAR *map_param = NULL;

#define MAX_GOTO 5
#define MAX_SEARCH 950
static int search_X = 1, search_Y = 1;
#define MAX_CHARSET 40

static int showtime = 1;

static int current_time(int *h, int *m, int *s) {
  time_t p;
  struct tm *T;
  if ((time_t)-1 == time(&p)) {
    int code=errno;
    print_error(code,"(%s) time: Failed to get current time",prog);
    showtime = 0;
    return 0;
  }
  T = localtime(&p);
  if (NULL == T) {
    print_notify("(%s) localtime: Failed to get current time",prog);
    showtime = 0;
    return 0;
  }
  *h = T -> tm_hour;
  *m = T -> tm_min;
  *s = T -> tm_sec;
  return 1;
}

int seconds_to_next_minute(void) {
  int h, m, s,val;
  if (!current_time(&h,&m,&s)) return 0;
  val = 60 - s;
  if (val < 0) val = 0;   /* s can be > 60 if leap seconds ... */
  return val;
}

static void update_prompt(int win, int mode) {
  int Y,off,roff,rl,bot,prompt_line,tgroup,pagenum,pages;
  int T,W,SP,TS,IS,PR,PG,prompt_X = 1;
  char buffer[100],S[50],pbuffer[30];
  int have_edit,b2len,line;
  CHAR_IDX buffer2[100];

  strcpy(S,"???");
  window_pos(win,&Y,&off,&roff,&pagenum,&pages);
  rl = window_rlines(win);
  get_window_status(win,&T,&W,&SP,&TS,&IS,&PR,&PG);

  switch(mode) {
  case 3: 
    sprintf(S,"MESSAGE (press %s for next page)",give_mapping(&key_page));
    key_group = G_MESSAGE;    
    break;
  case 2: case 1: case 4:
    sprintf(S,"MORE (press %s for next page)",give_mapping(&key_page));
    key_group = G_PAGER;
    break;
  }

  pbuffer[0]='\0';
  if (pages > 1 || pagenum > 1) {
    sprintf(pbuffer," page %d/%d",pagenum,pages);
    if (win_is_page_top(win)) strcat(pbuffer," (TOP)");
  }

  bot = rl - roff + Y;
  prompt_line = bot;
  if (prompt_line < Y) prompt_line = Y;
  if (prompt_line > lines) prompt_line = lines;
  else if (pagenum == 0 || pagenum == pages) switch (mode) { 
    case 1: sprintf(S,"TAIL (press %s for next file)",
		    give_mapping(&key_next)); break;
    case 2: strcpy(S,"READ (please wait)"); break;
    case 3: 
      sprintf(S,"OK (press %s to clear message)",
	      give_mapping(&key_mquit)); 
      key_group = G_MESSAGE_LAST;
      break;
    case 4: sprintf(S,"TAIL (press %s to quit kehpager)",
		    give_mapping(&key_quit)); 
      if (!wait_quit) {
	quitflag = 1;
	mode = 0;
      }
      break;
    }

  if (5 == mode) {
    strcpy(S,"WRAP (please wait)"); 
  } else if (6 == mode) {
    strcpy(S,"SEARCH (please wait)"); 
  } else if (7 == mode) {
    strcpy(S,"SKIP (please wait)"); 
  }
  
  line = roff + prompt_line - Y;

  if (showtime) {
    int h=0,m=0,s=0;
    current_time(&h,&m,&s);
    sprintf(buffer,"%s %d:%02d [%c%c%c%c] %d-%d/%d%s",S,h,m,
	    T ? 'T' : '-', 
	    W ? 'W' : '-',
	    SP ? '-' : 'S',
	    PG ? 'P' : '-',
	    roff > 0 ? roff+1 : 1,
	    line,rl,
	    pbuffer);
  } else {  
    sprintf(buffer,"%s [%c%c%c%c] %d-%d/%d%s",S,
	    T ? 'T' : '-', W ? 'W' : '-',
	    SP ? '-' : 'S',
	    PG ? 'P' : '-',
	    roff > 0 ? roff+1 : 1,
	    line,rl,pbuffer);
  }
  b2len = strlen(buffer);
  map_input(MAP_ASCII,b2len,rCs(buffer),buffer2);

  set_window_status(WIN_PROMPT,1,1,0,8,4,PRI_WAIT,0);

  if (b2len < prev_prompt_len)
    cut_window_line(WIN_PROMPT,b2len+1,1);
  if (b2len > prev_prompt_len && 
      0 == memcmp((void *)buffer2,(void *)prev_prompt,
		  prev_prompt_len * sizeof (int)))
    print_to_window(WIN_PROMPT,prev_prompt_len+1,1,
		    FL_BOLD,b2len-prev_prompt_len,buffer2+prev_prompt_len);
  else if (b2len != prev_prompt_len || 
	   0 != memcmp((void *)buffer2,(void *)prev_prompt,
		       prev_prompt_len * sizeof (int)))
    print_to_window(WIN_PROMPT,1,1,FL_BOLD,b2len,buffer2);
  memcpy((void *)prev_prompt,(void *)buffer2,b2len*sizeof(int));
  prev_prompt_len = b2len;

  tgroup = key_group;
  have_edit = edit_pos(prompt_line,&key_group);

#define HAVE_PROMPT (have_prompt && (!have_edit || (5 <= mode && mode <= 7)))

  if (HAVE_PROMPT) key_group = tgroup;

  if (pline != prompt_line && HAVE_PROMPT && mode != 0) {
    map_window(WIN_PROMPT,0,prompt_line);
    pline = prompt_line;
  } else if (pline != 0 && (!HAVE_PROMPT || mode == 0)) {
    unmap_window(WIN_PROMPT);
    pline = 0;
  }

  if (mode && HAVE_PROMPT)
    set_win_prompt(WIN_PROMPT,prompt_X,1);
  else if (mode && !have_edit)
    set_win_prompt(-2,1,down_line > 1 ? down_line -1 : 1);

#undef HAVE_PROMPT

}

#define MAX_ERRORS 100
static int have_error = 0;
static int error_line = 1;
static volatile CHAR error_pool[MAX_ERRORS][ERROR_LEN+1];
static volatile int head = 0, tail = 0;

#define ADD_P(x) (((x)+1)%MAX_ERRORS)

static void clear_errors(void) {
  unmap_window(WIN_ERROR);
  clear_window(WIN_ERROR); 
  have_error = 0;
}

static void map_errors(void) {
  if (!have_error) {
    int Y,offset,roffset,pagenum,pages;
    clear_window(WIN_ERROR); 
    set_window_status(WIN_ERROR,0,0,0,8,4,PRI_NORMAL,0);
    error_line = 1;
    have_error = 1;
    window_pos(WIN_TEXT,&Y,&offset,&roffset,&pagenum,&pages); 
    if (roffset > 0) roffset = 0;
    map_window(WIN_ERROR,roffset,Y);
  } else {
    error_line = window_lines(WIN_ERROR)+1;
  }
}

static int got_from_TAIL(int *ptr) {
  int result;
  block_state S;

  /* This can check outside of blocking first:
   * do not call sigprocmask without reason!
   */
  if (head == tail) return 0;

  BLOCK(&S);
  result = head != tail;
  if (result) {
    *ptr = tail;
    tail = ADD_P(tail);
  }
  UNBLOCK(&S);
  return result;
}

void flush_error_buffer(void) {
  int ptr;

  while (got_from_TAIL(&ptr)) {

    feed_queued_error((CHAR *) (error_pool[ptr]));
  }
}

static void print_errors(int ptr) {
  volatile CHAR *buffer = error_pool[ptr];
  CHAR_IDX *mapped;
  int len;
  CHAR *p,*n;
  
  map_errors();
  
  len=strlen(Cs((CHAR *)buffer));
  mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));
  
  for (p=(CHAR *)buffer; p < (CHAR *)buffer + len; p = n) {
    
    for (n=p; ('\n' != *n) && (n < buffer+len); n++);
    
    map_input(MAP_ASCII,n-p,p,mapped);
    print_to_window(WIN_ERROR,-1,error_line,FL_BOLD,n-p,mapped);
    error_line++;
    n++;
  }

  FREE(mapped);

  have_error = 1;
}

int error_callback(CHAR *buffer) {
  block_state S;
  if (!pager_state) return 0;

  BLOCK(&S);
  if (ADD_P(head) == tail) {
    UNBLOCK(&S);
    return 0;
  }
  
  strcpy(Cs((CHAR *)error_pool[head]),Cs(buffer));
  head = ADD_P(head);

  UNBLOCK(&S);
  return 1;
}

static int is_unblocking = 0;
static char * const STD_INPUT = "standard input";
static CHAR *cur_filename = NULL;
static int NOT_SELECTABLE = 0;

static void unblock(int *fp) {
  int fionbio_arg = 0;

  if (is_unblocking) {
    if (DEBUG_FILIO) {
      print_debug("Unsetting nonblocking mode for %s%s (fd=%d)",
		  Cs(cur_filename) != STD_INPUT ? "file " : "",
		  cur_filename ? cur_filename : rCs("(null)"),
		  *fp);

    }
    if (-1 == ioctl(*fp,FIONBIO,&fionbio_arg)) {
      int code = errno;
      print_error(code,"Can't unset nonblocking mode: %s%s",
		  Cs(cur_filename) != STD_INPUT ? "file " : "",
		  cur_filename ? cur_filename : rCs("(null)")
		  );
    } else
      is_unblocking = 0;
  }

}

static void not_selectable(int *fp) {
  unblock(fp);
  if (DEBUG_FILIO) {
    print_debug("Marking %s%s (fd=%d) not selectable",
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)"),
		*fp);

  }
  NOT_SELECTABLE = 1;
}

static void close_file(int *fp) {

  unblock(fp);
  NOT_SELECTABLE = 1;
  if (DEBUG_FILIO) {
    print_debug("Closing %s%s (fd=%d)",
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)"),
		*fp);
  }

  close(*fp); 
  *fp = -1;
}

#define MAX_BUFFERED 5

typedef struct {
  int buffered;
  CHAR buffer[MAX_BUFFERED];
} eoln_state;

static const eoln_state NULL_EOLN_STATE = { 0, "" };
#define EOLN_unix    0
#define EOLN_network 1

static struct {
  int len;
  char *data;
} eoln_array[] = {
  { 1, "\n" },
  { 2, "\r\n" }
};

static int eoln_type = EOLN_unix;

static int seek_eoln(int len,CHAR *ptr) {
  int s = 0;
  CHAR *ch1;

  for (ch1 = ptr; ch1 < ptr + len; ch1++) {
  again:
    if (rCc(eoln_array[eoln_type].data[s]) == *ch1) s++;
    else if (s) { s = 0; goto again; }

    if (s >= eoln_array[eoln_type].len)
      return ch1 - ptr - s +1; /* eof have found */
  }
  /* incomplete eof or eof not found */
  return len - s;
}

static int get_eoln(int len,CHAR *ptr) {
  CHAR *ch1;
  int s = 0;

  for (ch1 = ptr; ch1 < ptr + len; ch1++) {
    if (rCc(eoln_array[eoln_type].data[s]) == *ch1) s++;
    else return 0; /* not eof */

    if (s >= eoln_array[eoln_type].len)
      return s; /* got eof */
  }
  return -1; /* incomplete eof */
}

static void handle_file(content *CT,int *fp,int *eof, eoln_state *S) {
  int size = is_unblocking ? 4000 : 500;
  CHAR buffer [4000],*ch1,*ch2,*ch0;
  int len;

  if (S->buffered) 
    memcpy((void *)buffer,(void *)(S->buffer),S->buffered);
  ch0 = buffer + S-> buffered;

  if (DEBUG_FILIO) {
    print_debug("Reading %d bytes from %s%s (fd=%d)",
		size-S->buffered,
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)"),
		*fp);
  }

  len = read(*fp,ch0,size-S->buffered);

  if (-1 == len) {
    int code = errno;
    print_error(code,"Reading error: %s%s",
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)")
		);
    len = 0;
  }

  if (DEBUG_FILIO) {
    print_debug("Got %d bytes from %s%s (fd=%d)",
		len,
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)"),
		*fp);
  }

  if (0 == len) *eof = 1;
  

  len += S->buffered; 
  S-> buffered = 0;

  if (0 == len) update_content(CT);

  /* This code will lost last incomplete eoln !! */
  if (0 == len) return;

  for (ch1 = buffer; ch1 < buffer + len; ch1 = ch2) {
    int len1,len2;

    len1  = seek_eoln(len - (ch1 - buffer), ch1);

    if (len1)
      handle_content(CT,len1,ch1);
    ch2 = ch1 + len1;

    len2 = get_eoln(len - (ch2 - buffer), ch2);

    if (-1 == len2) {
      /* buffer it */
      if (len - (ch2 - buffer) > MAX_BUFFERED) {
	reset_terminal_state();
	print_notify("(%s) need buffer = %d > %d "
		     "- aborting (will produce core-file)...",
		     prog,len - (ch2 - buffer),MAX_BUFFERED);
	close_files();
	close_terminal();
	abort();	
      }
      memcpy((void *)S->buffer,(void *)ch2,len - (ch2 - buffer));
      S->buffered=len - (ch2 - buffer);
      return;
    }

    if (len2 > 0)
      eoln_content(CT);

    ch2 += len2;
  }
}

static void reset_file_flags(int *fp,int *eof, eoln_state *S) {
  search_X = 1;
  search_Y = 1;

  *eof = 0;
  *S = NULL_EOLN_STATE;
}

static void read_file(CHAR *name, int *fp, int *eof, eoln_state *S,
		      content *CT, 
		      int map, CHAR *map_param,
		      int ct) {
  int ft;
  int fionbio_arg = 1;

  if (-1 != *fp) {
    exit_content(CT);
    close_file(fp);
  }
  *eof = 1;
  cur_filename = NULL;

  if (Cs(name) == STD_INPUT) {
    if (DEBUG_FILIO) {
      print_debug("Duplicating standard input");
    }

    ft = dup(0); /* standard input */
  } else {
    if (DEBUG_FILIO) {
      print_debug("Opening %s%s",
		  Cs(name) != STD_INPUT ? "file " : "",
		  name ? name : rCs("(null)"));
    }
    ft = open(Cs(name),O_RDONLY);		   
  }

  if (-1 == ft) {
    int code = errno;
    print_error(code,"Can't read %s%s",
		Cs(name) != STD_INPUT ? "file " : "",
		name);
    return;
  }
  NOT_SELECTABLE = 0; /* first guess */

  if (isatty(ft)) {
    print_notify("Can't read from terminal: %s%s",
		 Cs(name) != STD_INPUT ? "file " : "",
		 name);
    close(ft);
    return;
  }

  if (DEBUG_FILIO) {
    print_debug("Unblocking %s%s (fd=%d)",
		Cs(name) != STD_INPUT ? "file " : "",
		name ? name : rCs("(null)"),
		ft);
  }
  
  is_unblocking = 1;
  if (-1 == ioctl(ft,FIONBIO,&fionbio_arg)) {
    int code = errno;
    is_unblocking = 0;
    /* HPUX allows only terminal to put nonblocking mode */
    if (EINVAL != code && ENOTTY != code && ENXIO != code ) 
      print_error(code,"Can't set %s%s to nonblocking mode",
		  Cs(name) != STD_INPUT ? "file " : "",
		  name);
    else if (DEBUG_FILIO) {
      print_debug("Got errno=%d (EINVAL or ENOTTY or ENXIO) for %s%s (fd=%d)",
		  code,
		  Cs(name) != STD_INPUT ? "file " : "",
		  name ? name : rCs("(null)"),
		  ft);
    }
  }

  cur_filename = name;
  *fp = ft;
  reset_file_flags(fp,eof,S);
  init_content(ct,fp,WIN_TEXT,map,map_param,CT);
}

static int reread_file(int *FP,int *eof, eoln_state *S,
		       content *CT, int map, CHAR *map_param,
		       int ct) {
  int tab,wrap,showspc,tab_step,indent_step,priority,pagemode;
  if (-1 == *FP) return 0;

  if (DEBUG_FILIO) {
    print_debug("Seeking beginning of %s%s (fd=%d)",
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)"),
		*FP);
  }
  
  if (-1 == lseek(*FP,0,SEEK_SET)) {
    int code = errno;
    if (ESPIPE == code) {
      if (DEBUG_FILIO) {
	print_debug("Got errno=%d (ESPIPE) for %s%s (fd=%d)",
		    code,
		    Cs(cur_filename) != STD_INPUT ? "file " : "",
		    cur_filename ? cur_filename : rCs("(null)"),
		    *FP);
      }
      return 0;
    }
    print_error(code,"Can't seek to start of file: %s%s",
		Cs(cur_filename) != STD_INPUT ? "file " : "",
		cur_filename ? cur_filename : rCs("(null)")
		);
    return 0;
  }
  get_window_status(CT->win,&tab,&wrap,&showspc,&tab_step,&indent_step,
		    &priority,&pagemode);
  reset_file_flags(FP,eof,S);
  exit_content(CT);
  init_content(ct,FP,WIN_TEXT,map,map_param,CT);
  /* exit / init context can chance some modes, so return they back! */
  set_window_status(CT->win,tab,wrap,showspc,tab_step,indent_step,
		    priority,pagemode);

  return 1;
}

static int arg_left = 0;
static char **arg_vec = NULL;
static int FP = -1;
static int EOF_IN_FILE = 1;
static eoln_state EOLN_STATE;
static content CT;

void close_files(void) {
  if (-1 != FP) close_file(&FP);
}

static int map_null_name = MAP_ASCII;
static int enable_tab = 1;
static int enable_wrapping = 1;
static int enable_eatspc = 1;
static int input_tabstep = 8;
static int enable_pages  = 1;
static int input_indentstep = 4;
static int content_type = CT_terminal;
static int NOT_IN_USE = 0;

init_item kehpager_items[] = {
  { &map,               "input.charset",       V_MAP },
  { &content_type,      "input.content_type",  V_CONTENT },
  { &map_null_name,     "input.charset.null",  V_MAP },
  { &enable_tab,        "input.interpret_tab", V_BOOL },
  { &input_tabstep,     "input.tab_step",      V_NUM },
  { &input_indentstep,  "input.indent_step",   V_NUM },
  { &enable_wrapping,   "input.wordwrap",      V_BOOL },
  { &enable_eatspc,     "input.eat_spaces",    V_BOOL },
  { &enable_fulljust,   "input.full_justify",  V_BOOL },
  { &enable_pages,      "input.pagemode",      V_BOOL },
  { &have_prompt,       "use.prompt",          V_BOOL },
  { &showtime,          "use.prompt.time",     V_BOOL },
  { &use_scrolling,     "use.scrolling",       V_BOOL },
  { &wait_quit,         "use.wait_quit",       V_BOOL },

  /* OBSOLETED */
  { &content_type,      "input.interpret_bs",  V_BOOL },
  { &content_type,      "input.interpret_ansi_attributes",  V_BOOL },
  { &NOT_IN_USE,        "input.ignore_nl",    V_BOOL },

  { NULL, NULL, V_LAST }
};

static void print_help(int win, char *s, ...) {
  va_list walker;
  char *b,*b1;
  int flag = 0;
  int col=-1;

  va_start(walker,s);

  print_and_scroll(win,MAP_ASCII,0,0,rCs("")); 
  for (b = s; '\0' != *b; b = b1) {
    for (b1 = b; '\0' != *b1 && '%' != *b1; b1++);
    if (b1>b) {
      print_to_column(win,MAP_ASCII,flag,col,b1-b,rCs(b));
      col=-1;
    }

    if ('%' == *b1) {
      CHAR *MM;
      int *key,Y;
      CHAR_IDX num;

      b1++;
      switch (*b1++) {
      case 's': case 'm': 
	key = va_arg(walker, int *);
	MM = give_mapping(key);
	print_to_column(win,MAP_ASCII,flag,col,strlen(Cs(MM)),MM);
	col=-1;
	break;
      case 'S':
	MM=va_arg(walker,CHAR *);
	print_to_column(win,MAP_ASCII,flag,col,strlen(Cs(MM)),MM);
	col=-1;
	break;
      case 'M':
	num=va_arg(walker,int);
	Y = window_lines(win);
	if (Y < 1) Y = 1;
	print_to_window(win,col,Y,flag,1,&num);
	col=-1;
	break;
      case 'c':
	num=va_arg(walker,int);
	if (num>0) col=num;
	break;
      case 'u':
	flag |= FL_UNDER; break;
      case 'U':
	flag &= ~FL_UNDER; break;
      case 'b':
	flag |= FL_BOLD; break;
      case 'B':
	flag &= ~FL_BOLD; break;
      case '0':
	flag = 0; break;
      case '%':
	print_to_column(win,MAP_ASCII,flag,col,1,rCs("%")); 
	col=-1; break;
      }
    }
  }

  va_end(walker);

}


static int order[TOTAL_MAP_COUNT];
static CHAR *tmp_name[TOTAL_MAP_COUNT];

static int cl_cmp(const void *A,const void *B) {
  const int *A1 = (int *)A;
  const int *B1 = (int *)B;
  return strcasecmp(Cs(tmp_name[*A1]),Cs(tmp_name[*B1]));
}

static void charset_listing(int win) {
  int map,i;

#define P(F,S) print_and_scroll(win,MAP_ASCII,F,strlen(S),rCs(S))
#define P0(F,S) print_and_scroll(win,MAP_ASCII,F,strlen(Cs(S)),S)
#define CL(X,S) print_to_column(win,MAP_ASCII,0,X,strlen(Cs(S)),S)
#define C(S,K) print_help(win,S,K)
  
  C("%bKehpager charset listing:%0",NULL);
  P(0,"");
  P(0,"");
  
  print_to_column(win,MAP_ASCII,FL_UNDER,1,strlen("Charset"),rCs("Charset"));
  print_to_column(win,MAP_ASCII,FL_UNDER,30,strlen("MIME name"),rCs("MIME name"));

  /* Make sorting by using first mime name */
  for (map = 0; map < TOTAL_MAP_COUNT; map++) {
    order[map] = map;
    tmp_name[map] = map_name(map,1);
  }
  qsort((void *)order,MAP_COUNT,sizeof (int),cl_cmp);

  for (i = 0; i < TOTAL_MAP_COUNT; i++) {
    int map = order[i];
    CHAR *name= map_name(map,0);
    CHAR *vector[15];
    int count = cs_names(map,1,20,vector);
    int j;

    if (0 == strcmp(Cs(name),"Unknown")) continue;

    P0(0,name);
    for (j = 0; j < count; j++) {
      if (0 != j) P(0,"");
      CL(30,vector[j]);
    }
  }

#undef CL
#undef P0
}

static void copyright_page(int win) {
  set_window_status(win,1,1,0,8,4,PRI_NORMAL,0);

  P(0,"");
  P(0,"kehpager, Charset aware pager, Kari E. Hurtta");
  P(0,"");
  P(0,"Copyright (c) 1993, 1994 Kari E. Hurtta");
  P(0,"");
  P(0,
    "Redistribution and use in source and binary forms are permitted "
    "provided that the above copyright notice and this paragraph are "
    "duplicated in all such forms. This software is provided 'as is' "
    "and without any warranty."
    );
}

static void help_page(int win) {
  CHAR buffer[2048],*MM;


#define M(S,K) if (NULL != (MM = give_mapping(K))) { sprintf(Cs(buffer),S,MM); \
	       print_and_scroll(win,MAP_ASCII,0,strlen(Cs(buffer)),buffer); }
#define CM(S,K) if (NULL != (MM = give_ctl_mapping(K))) { sprintf(Cs(buffer),S,MM); \
	       print_and_scroll(win,MAP_ASCII,0,strlen(Cs(buffer)),buffer); }
    
  set_window_status(win,1,1,0,8,4,PRI_NORMAL,0);
  P(FL_BLINK,Version);
  print_to_column(win,MAP_ASCII,0,30,48,
		  rCs("Author: Kari E. Hurtta <Kari.Hurtta@Helsinki.FI>"));
  P(0,"");
  C("To see copyright message, type %s",&key_copyright);
  P(0,"");
  P(FL_BOLD,"Keyboard commands:");
  P(0,"");
  M(" %s\tRedraw",&key_redraw);
  M(" %s\tCancel key",&key_cancel);
  M(" %s\tCompose/Quote key (See COMPOSED INPUT)",&key_compose);
  M(" %s\tNext file",&key_next);
  M(" %s\tReread file (file must be seekable)",&key_reread);
  M(" %s\tChanges charset of file (file must be seekable)",&key_charset);
  M(" %s\tChanges content-type fo file (file must be seekable)",&key_ct);
  print_help(win," %s%s\tSets eoln convention of file to unix "
	    "(file must be seekable)",&key_nl,&key_nl_unix);
  print_help(win," %s%s\tSets eoln convention of file to network "
	    "(file must be seekable)",&key_nl,&key_nl_network);
  M(" %s\tChanges tabbing step fo file",&key_ts);
  M(" %s\tChanges indentation step fo file",&key_is);
  M(" %s\tQuit",&key_quit);
  M(" %s\tNext page",&key_page);
  M(" %s\tNext page (checks if file is growing)",&key_forward);
  M(" %s\tNext line",&key_lnext);
  M(" %s\tPrevious page",&key_prev);
  M(" %s\tPrevious line",&key_lprev);
  M(" %s\tTop of file",&key_top);
  C(" %un%0%s\tGoto line",&key_genter);
  C(" %un%0%s\tGoto relative(%%) line",&key_gpercnt);
  M(" %s\tBottom of file",&key_bottom);
  M(" %s\tSearch text from current page of file",&key_search);
  M(" %s\tSearch next instance of text",&key_searchn);
  M(" %s\tSearch first instance of text",&key_searchf);
  M(" %s\tToggle interpreting of TAB, NBSP, SHY -characters",&key_tab);
  M(" %s\tToggle interpreting of FF -character (toggle pagemode)",
    &key_pagemode);
  M(" %s\tToggle word wrap",&key_wrap);
  M(" %s\tToggle time in prompt on/off",&key_time);
  M(" %s\tToggle eating of spaces",&key_spc);
  M(" %s\tToggle prompt on/off",&key_prompt);
  M(" %s\tDebug screen",&key_debug);
  M(" %s\tCopyright message",&key_copyright);
  M(" %s\tThis text",&key_help);
  P(0," "); 
  P(FL_BOLD,"Control keys:");
  P(0,"");
  CM(" %s\tRefresh screen",&key_redraw);
  CM(" %s\tCancels prompts",&key_cancel);
  CM(" %s\tErases line",&key_eerase);
  CM(" %s\tDeletes char",&key_edel);
#if defined(VLNEXT_preserved)
  CM(" %s\tQuotes next control character",&key_compose);
#endif
  CM(" %s\tKills kehpager",&key_kill);
  CM(" %s\tKills kehpager (don't restore terminal driver state)",&key_abort);
  CM(" %s\tSuspends kehpager",&key_suspend);
  CM(" %s\tFlushes output",&key_flush);
  CM(" %s\tReserved",&key_res1);
  CM(" %s\tReserved",&key_res2);
  P(0,"");
  P(FL_BOLD,"Charset options:");
  P(0,"");
  P(0," -A\tCharset is Ascii (US-ASCII)");
  P(0," -L[1]\tCharset is Latin/1 (ISO-8859-1)");
  P(0," -L2\tCharset is Latin/2 (ISO-8859-2)");
  P(0," -L3\tCharset is Latin/3 (ISO-8859-3)");
  P(0," -L4\tCharset is Latin/4 (ISO-8859-4)");
  P(0," -L5\tCharset is Latin/5 (ISO-8859-9)");
  P(0," -U7\tCharset is Unicode 1.1 with UTF7 (UNICODE-1-1-UTF7)");
  P(0," -U8\tCharset is Unicode 1.1 with UTF8 (UNICODE-1-1-UTF8)");
  P(0," -M\tCharset is Multinational");
  P(0," -PC\tCharset is MS-DOS CP437");
  P(0," -MAC\tCharset is Macintosh");
  P(0," -F\tCharset is National (Finnish)");
  P(0," -UK\tCharset is National (UK-ASCII)");
  P(0," -NR\tCharset is National (Norwegian, Danish)");
  P(0," -SW\tCharset is National (Swedish)");
  P(0," -CA\tCharset is National (French Canadian)");
  P(0," -FR\tCharset is National (French/Belgian, Flemish)");
  P(0," -GR\tCharset is National (German)");
  P(0," -SP\tCharset is National (Spanish)");
  P(0," -IT\tCharset is National (Italian)");
  C(" -MN\tCharset is with Mnemonics encoding (RFC1345)",NULL);
  C(" -MN:%un%0\tMnemonics encoding with given escape character value",NULL);
  C(" -CS %ucs%0\tSet charset",NULL);
  P(0,"");
  P(FL_BOLD,"Content-type options:");
  P(0,"");
  P(0," -tr\tContent-type is Terminal (interprets BS and some ANSI controls)");
  P(0," -pl\tContent-type is Plain (Text/Plain)");
  P(0," -er\tContent-type is Enriched (Text/Enriched)");
  P(0," -rt\tContent-type is Richtext (Text/Richtext)");
  C(" -ct %uct%0\tSet content-type",NULL);
  P(0,"");
  P(FL_BOLD,"Other interpreting options:");
  P(0,"");
  P(0," -nu\teoln convention is unix (LF)");
  P(0," -nn\teoln convention is network (CR LF)");
  P(0," -t\tDisable interpreting of TAB, NBSP, SHY -characters");
  P(0," -t+\tEnable interpreting of TAB, NBSP, SHY -characters");
  P(0," -pm\tDisable interpreting of FF -character (pagemode off)");
  P(0," -pm+\tEnable interpreting of FF -character (pagemode on)");
  P(0,"LINE SEPARATOR and PARAGRAPH SEPARATOR is interpreted when " 
    "pagemode is on or interpreting of TAB, NBSP, SHY -characters is " 
    "turned on.");
  C(" -t%un%0\tSet tab step",NULL);  
  C(" -i%un%0\tSet indentation step for exriched",NULL);
  P(0," -sp\tDisable eating of spaces");
  P(0," -sp+\tEnable eating os spaces");
  P(0," -j\tSets default justification of enriched text to <FlushLeft>");
  P(0," -j+\tSets default justification of enriched text to <FlushBoth>");
  P(0,"");
  P(FL_BOLD,"Terminal options:");
  P(0," -e\tfor eval `kehpager -e` (-ec for csh -eu for Bourne Shell)");
  P(0,"\t\tThis is for prepare quick start");
  P(0," -f\tDisable quick start (previous eval `kehpager -e` don't effect)");
  P(0," -X\tDisable forced Latin/1 on Xterm");
  P(0," -X+\tEnable forced Latin/1 on Xterm");
  P(0," -X++\tForce Latin/1 (and Multinational mode)");
  P(0," -7\tForce National mode");
  P(0," -8\tForce Multinational mode");
  P(0," -q\tLimit queries from terminal (according terminal's type)");
  P(0," -q+\tAsk all queries from terminal");
  P(0," -s\tNext and prev page is printing by repainting");
  P(0," -s+\tNext and prev page is scrolled to visible");
  P(0," -rs\tDon't resize terminal (don't query size from terminal)");
  P(0," -rs+\tResize terminal (query size from terminal)");
  C(" -N%un%0\tSet (default) keyboard language",NULL);

  P(0,"");
  P(FL_BOLD,"Options:");
  P(0," --\tDisables option processing");
  P(0," -H\tThis text");
  P(0," -T\tTest Page");
  P(0," -T2\tTest Page 2");
  P(0," -T3\tTest Page 3");
  P(0," -DS\tPrint debug screen");
  P(0," -cs\tPrint list of charsets");
  P(0," -st\tDisable time in prompt");
  P(0," -st+\tEnable time in prompt");  
  P(0," -w\tDisable word wrap");
  P(0," -w+\tEnable word wrap");
  P(0," -p\tDisable prompt");
  P(0," -p+\tEnable prompt");
  M(" -wt\tDon't wait that user press %s key for quiting",&key_quit);
  M(" -wt+\tWait that users press %s key in end of last file",&key_quit);

  P(0," -\tStandard input");
  C(" -D%un%U\tPrint debug output to standard error",NULL);
  P(0," -DS\tPrint debug page");
  P(0,"");
  P(FL_BOLD,"COMPOSED INPUT:");
  P(0,"");
  M("Some letters can enter with typing compose key (%s) "
    "and two printable characters.",&key_compose);
  P(0,"");
  C("%m %uletter%0 '\t\t%uletter%U with ACUTE ACCENT",&key_compose);
  C("%m %uletter%0 ,\t\t%uletter%U with CEDILLA",&key_compose);
  C("%m %uletter%0 `\t\t%uletter%U with GRAVE ACCENT",&key_compose);
  C("%m %uletter%0 ^\t\t%uletter%U with CIRCUMFLEX ACCENT",&key_compose);
  C("%m %uletter%0 ~\t\t%uletter%U with TILDE",&key_compose);
  C("%m %uletter%0 \"\t\t%uletter%U with DIAERESIS",&key_compose);
  C("%m \" %uletter%0\t\t%uletter%U with DOUBLE ACUTE ACCENT",&key_compose);
  C("%m %uletter%0 o\t\t%uletter%U with RING ABOVE",&key_compose);
  C("%m %uletter%0 .\t\t%uletter%U with DOT ABOVE",&key_compose);
  C("%m %uletter%0 (\t\t%uletter%U with BREVE",&key_compose);
  C("%m %uletter%0 ;\t\t%uletter%U with OGONEK",&key_compose);
  C("%m %uletter%0 <\t\t%uletter%U with CARON",&key_compose);
  C("%m %uletter%0 9\t\t%uletter%U with HORN",&key_compose);
  C("%m %uletter%0 0\t\t%uletter%U with RING BELOW",&key_compose);
  C("%m %uletter%0 _\t\t%uletter%U with LINE BELOW",&key_compose);
  C("%m %uletter%0 >\t\t%uletter%U with CIRCUMFLEX BELOW",&key_compose);
  C("%m %uletter%0 |\t\t%uletter%U with BAR",&key_compose);
  C("%m A A\t\tCOMMERCIAL AT (@)",&key_compose);
  C("%m ( (\t\tLEFT SQUARE BRACKET ([)",&key_compose);
  C("%m / /\t\tREVERSE SOLIDUS (\\)",&key_compose);
  C("%m ' '\t\tGRAVE ACCENT (`)",&key_compose);
  C("%m ( -\t\tLEFT CURLY BRACKET ({)",&key_compose);
  C("%m ) -\t\tRIGHT CURLY BRACKET (})",&key_compose);
  C("%m ^ /\t\tVERTICAL LINE (|)",&key_compose);
  C("%m ^ ^\t\tTILDE (~)",&key_compose);
  C("%m SPC SPC\t\tNO-BREAK SPACE (NBSP)",&key_compose);
  C("%m - -\t\tSHOFT HYPHEN (SHY)",&key_compose);
  P(0,"");
#undef P
#undef M
#undef CM
#undef C
}

static int DW = -1;
static void DP(CHAR *line,int header) {
  if (header > 0) {
    print_and_scroll(DW,MAP_ASCII,0, 0, rCs(""));
    print_and_scroll(DW,MAP_ASCII,FL_UNDER, strlen(Cs(line)), line);
    print_and_scroll(DW,MAP_ASCII,0, 0, rCs(""));
  } else {
    print_and_scroll(DW,MAP_ASCII,0, strlen(Cs(line)), line);
    if (header < 0)
      set_line_status(DW,window_lines(DW),header,0,0,0,0,0,0);
  }
}

static void debug_page(int win) {
  DW = win;
  set_window_status(win,0,1,0,8,4,PRI_NORMAL,0);
  give_debug_page(DP);
}

static void test_page(int win) {
  CHAR c;
  int i;
  char temp[5];
#define P(C,S) print_and_scroll(win,C,0,strlen(S),rCs(S))
#define T(C,X,S) print_to_column(win,C,0,X,strlen(S),rCs(S))
#define K(C,X,c) print_to_column(win,C,0,X,1,&c)

  set_window_status(win,0,0,0,8,4,PRI_NORMAL,0);
  print_and_scroll(win,MAP_ASCII,FL_UNDER,strlen("Kehpager test page:"), 
		   rCs("Kehpager test page:"));
  P(MAP_ASCII,"");
  P(MAP_LATIN1,        "A Diaresis (Latin/1 0xC4):         \304");
  P(MAP_LATIN1,        "O Diaresis (Latin/1 0xD6):         \326");
  P(MAP_LATIN1,        "NBSP (Latin/1 0xA0):               \240");
  P(MAP_LATIN1,        "SHY  (Latin/1 0xAD):               \255");
  P(MAP_LATIN1,        "Multiplication (Latin/1 0xD7):     \327");
  P(MAP_LATIN1,        "Division (Latin/1 0xF7):           \367");
  P(MAP_MULTINATIONAL, "OE (Multinational 0xD7):           \327");
  P(MAP_MULTINATIONAL, "oe (Multinational 0xF7):           \367");
  P(MAP_ASCII,         "Left square bracket (Ascii 0x91):  \133");
  P(MAP_ASCII,         "Right square bracket (Ascii 0x93): \135");
  P(MAP_ASCII,"");
  P(MAP_ASCII, "Code"); T(MAP_ASCII, 10, "Ascii"); ;
                        T(MAP_MULTINATIONAL, 22, "Multinational"); 
  T(MAP_UK, 36, "British"); T(MAP_SWE, 48, "Swedish");
  T(MAP_LATIN5,60,"Latin/5");   T(MAP_LATIN2,72, "Latin/2");
  P(MAP_ASCII, ""); T(MAP_LATIN1, 16, "Latin/1");
                    T(MAP_FINNISH, 30, "Finnish");   
                    T(MAP_NOR, 42, "Norwegian");
                    T(MAP_LATIN4, 54, "Latin/4");
                    T(MAP_LATIN_ANY, 66, "ISO-8859-*");

  for (i = 0; i < 256; i++) {
    c = i;
    sprintf(temp,"0x%02X",i);
    P(MAP_ASCII,temp);

    K(MAP_ASCII, 10, c); K(MAP_LATIN1, 16, c);
    K(MAP_MULTINATIONAL, 22, c); K(MAP_FINNISH, 30, c);
    K(MAP_UK,36, c); K(MAP_NOR,42, c); K(MAP_SWE, 48, c);
    K(MAP_LATIN4,54, c); K(MAP_LATIN5,60,c);
    K(MAP_LATIN_ANY,66,c); K(MAP_LATIN2,72,c);
  }
#undef P
#undef T
#undef K
}


static void test_page2(int win) {
  unsigned char c;
  int i;
  char temp[5];
#define P(C,S) print_and_scroll(win,C,0,strlen(S),rCs(S))
#define T(C,X,S) print_to_column(win,C,0,X,strlen(S),rCs(S))
#define K(C,X,c) print_to_column(win,C,0,X,1,&c)

  set_window_status(win,0,0,0,8,4,PRI_NORMAL,0);
  print_and_scroll(win,MAP_ASCII,FL_UNDER,strlen("Kehpager test page (2):"), 
		   rCs("Kehpager test page (2):"));
  P(MAP_ASCII,"");
  P(MAP_ASCII, "Code"); T(MAP_LATIN1, 10, "Latin/1"); 
  T(MAP_GER, 22, "German"); T(MAP_CAN,30, "French Canadian");
  T(MAP_SPA, 50, "Spanish"); T(MAP_ITAL,63,"Italian");
   P(MAP_ASCII, ""); T(MAP_BEL, 16, "French/Belgian");
  T(MAP_LATIN3,36, "Latin/3");  T(MAP_CP437, 44, "MS-DOS CP437");
  T(MAP_MAC,57,"Macintosh");

  for (i = 0; i < 256; i++) {
    c = i;
    sprintf(temp,"0x%02X",i);
    P(MAP_ASCII,temp);

    K(MAP_LATIN1, 10, c); K(MAP_BEL, 16, c);
    K(MAP_GER, 22, c); K(MAP_CAN, 30, c);
    K(MAP_LATIN3, 36, c); K(MAP_CP437, 44, c);
    K(MAP_SPA,50,c); K(MAP_MAC,57,c);
    K(MAP_ITAL,63,c);
  }
#undef P
#undef T
#undef K
}

static int tp3_cmp(const void *A,const void *B) {
  const int *A1 = (int *)A;
  const int *B1 = (int *)B;

  return chars[*A1].vector[MAP_UNICODE11] - chars[*B1].vector[MAP_UNICODE11];
}

static void test_page3(int win) {
  int *vector=MALLOC(CHARS_COUNT_*sizeof(int));
  
  int i;

  set_window_status(win,0,0,0,8,4,PRI_NORMAL,0);
  print_help(win,"%uKehpager char listing:%0");
  print_help(win,"");

  for (i=0; i < CHARS_COUNT_; i++) vector[i]=i;
  qsort((void *)vector,CHARS_COUNT_,sizeof(int),tp3_cmp);
  print_help(win,"Unicode 1.1%cLatin/1%cchar%cmnemonics%cname",14,22,28,38);
  
  if (DEBUG_CHARMAP) {
    print_debug("Charmap listing: ");
  }
  for(i=0; i < CHARS_COUNT_; i++) {
    int j=vector[i];
    char s1[8],s2[6],*s3,*s4;

    if (DEBUG_CHARMAP) {
    print_debug(" %5d: %8X %8X %s %s",
		j,chars[j].vector[MAP_UNICODE11],
		chars[j].vector[MAP_LATIN1],
		chars[j].evector[ENT_Mnem] ? 
		chars[j].evector[ENT_Mnem] : "null",
		chars[j].evector[ENT_name] ?
		chars[j].evector[ENT_name] : "null"
		);
    }

    if (chars[j].vector[MAP_UNICODE11] >= 0) {
      if (0 != (chars[j].vector[MAP_UNICODE11]& FC_fallback) ||
	  0 != (chars[j].vector[MAP_UNICODE11]& FC_alias))
	sprintf(s1,"[0x%04X]",chars[j].vector[MAP_UNICODE11]&CHAR_code);
      else 
	sprintf(s1,"0x%04X",chars[j].vector[MAP_UNICODE11]&CHAR_code);
    }
    else s1[0]='\0';

    if (chars[j].vector[MAP_LATIN1] >= 0) {
      if (0 != (chars[j].vector[MAP_LATIN1]& FC_fallback) ||
	  0 != (chars[j].vector[MAP_LATIN1]& FC_alias))
	sprintf(s2,"[0x%02X]",chars[j].vector[MAP_LATIN1]&CHAR_code);
      else
	sprintf(s2,"0x%02X",chars[j].vector[MAP_LATIN1]&CHAR_code);
    }
    else s2[0]='\0';

    if (chars[j].evector[ENT_Mnem]) s3=chars[j].evector[ENT_Mnem];
    else s3="";

    if (chars[j].evector[ENT_name]) s4=chars[j].evector[ENT_name];
    else s4="";

    print_help(win,"%S%c%S%c%M%c%S%c%S",s1,14,s2,22,j,28,s3,38,s4);
    
  }

  FREE(vector);
}

static int get_next = 1;
static int do_forwarding = 0;

static int NEWLINE_ready(int *pos,int value_len,CHAR_IDX *value) {
  int new_type = eoln_type;
  if (&key_nl_unix == pos) new_type = EOLN_unix;
  else if (&key_nl_network == pos) new_type = EOLN_network;
  else if (&key_nl_enter != pos) return 0;

  if (reread_file(&FP,&EOF_IN_FILE,&EOLN_STATE,
		  &CT,map,map_param,
		  content_type)) {
    eoln_type = new_type;
    return 1;
  }
  return 0;
}

#define MAX_CONTENT 40

static int CONTENT_ready(int *pos,int value_len,CHAR_IDX *value) {
  CHAR name[MAX_CONTENT+2];
  int ncontent;
  if (0 == value_len) ncontent = content_type;
  else {
    int len = unmap_string(MAP_ASCII,value_len,value,name);
    name[len] = '\0';
    if (len < value_len) 
      return 0;
    ncontent = search_content(name,0);
    if (ncontent < 0) 
      return 0;
    }
  if (reread_file(&FP,&EOF_IN_FILE,&EOLN_STATE,
		  &CT,map,map_param,
		  ncontent)) {
    content_type = ncontent;
    return 1;
  }
  return 0;
}

#define MAX_NUMLEN 5

static int TABSTEP_ready(int *pos,int value_len,CHAR_IDX *value) {
  int visible = have_error ? WIN_ERROR : WIN_TEXT;
  int nvalue;
  
  int tab, wrap,showspc,tab_step,indent_step,priority,pagemode;
  
  get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		    &priority,&pagemode);
  
  nvalue = tab_step;

  if (value_len > 0) {
    int rvalue = string_value(value_len,value);
    nvalue = rvalue;
  }
    
  if (nvalue >= 0 && nvalue <= MAX(MAX(8,columns/3),tab_step)) {
    set_window_status(visible,tab,wrap,showspc,nvalue,indent_step,
		      priority,pagemode);
    return 1;
  }
  return 0;
}


static int INDENTSTEP_ready(int *pos,int value_len,CHAR_IDX *value) {
  int visible = have_error ? WIN_ERROR : WIN_TEXT;
  int nvalue;
  
  int tab, wrap,showspc,tab_step,indent_step,priority,pagemode;
  
  get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		    &priority,&pagemode);
  
  nvalue = tab_step;

  if (value_len > 0) {
    int rvalue = string_value(value_len,value);
    nvalue = rvalue;
  }
    
  if (nvalue >= 0 && nvalue <= MAX(MAX(8,columns/3),indent_step)) {
    set_window_status(visible,tab,wrap,showspc,tab_step,nvalue,
		      priority,pagemode);
    return 1;
  }
  return 0;
}

static int CHARSET_ready(int *pos,int value_len,CHAR_IDX *value) {
  CHAR name[MAX_CHARSET+2],*nvalue;
  static CHAR value_store[MAX_CHARSET+1];
  int nmap;
  if (0 == value_len) {
    name_table(&CT.cs,&nmap,&nvalue);
  }
  else {
    char *npos = NULL;
    int len = unmap_string(MAP_ASCII,value_len,value,name);
    nvalue = NULL;
    name[len] = '\0';
    if (len < value_len) 
      return 0;
    if (NULL != (npos = strchr(Cs(name),';')) )
      *npos = '\0';
    nmap = find_map(name);
    if (nmap < 0) 
      return 0;
    if (npos && ' ' == *(npos+1))
      npos++;
    if (npos) {
      strcpy(Cs(value_store),npos+1);
      nvalue = value_store;
    }
  }

  if (reread_file(&FP,&EOF_IN_FILE,&EOLN_STATE,
		  &CT,nmap,
		  nvalue,
		  content_type)) {
    map = nmap;
    map_param = nvalue;
    return 1;
  }
  return 0;
}

static int search_value_len = 0;
static CHAR_IDX search_value[MAX_SEARCH];

void do_search(int mode) {
  int visible = have_error ? WIN_ERROR : WIN_TEXT;
  int rY,mlen;
  switch (mode) {
    int Y,offset,roffset,pagenum,pages;
  case 0:
    window_pos(visible,&Y,&offset,&roffset,&pagenum,&pages);
    if (search_Y == offset+1) search_X++;
    else {
      search_Y = offset+1;
      search_X = 1;
    }
    break;
  case 1:
    search_X = 1;
    search_Y = 1;
    break;
  case 2:
    search_X++;
    break;
  }
  update_prompt(visible,6);
  if (find_buffer_text(visible,search_value_len,search_value,
		       &search_X,&search_Y,&rY,&mlen)) {
    set_win_hl(visible,FL_BOLD,search_X,search_X+mlen-1,search_Y);
    goto_window(visible,rY);
  } else 
    set_win_hl(-1,0,-1,-1,-1);
}

static int SEARCH_ready(int *pos,int value_len,CHAR_IDX *value) {
  memcpy((void *)search_value,(void *)value,value_len * sizeof (CHAR_IDX));
  search_value_len = value_len;
  
  if (&key_senter == pos) do_search(0);
  else if (&key_ssearchf == pos || &key_searchf == pos) do_search(1);
  else do_search(2);

  return 1;
}

static int GOTO_ready(int *pos,int value_len,CHAR_IDX *value) {
  int visible = have_error ? WIN_ERROR : WIN_TEXT;
  if (&key_genter == pos) {
    int rvalue = string_value(value_len,value);
    update_prompt(visible,7);
    goto_window(visible,rvalue);
    return 1;
  } else if (&key_gpercnt == pos) {
    int len = window_rlines(visible);
    int rvalue = string_value(value_len,value);
    if (rvalue <= 100) {
      update_prompt(visible,7);
      if (0 == rvalue) goto_window(visible,1);
      else goto_window(visible,rvalue*len/100);
      return 1;
    }
  }
  return 0;
}

static void action (int *pos, int c, int *cancel) {
  int visible = have_error ? WIN_ERROR : WIN_TEXT;

  if (&key_quit == pos) {
    quitflag = 1;
  } else if (edit_action(pos,c,&key_group)) {
    return;
  } else if (&key_charset == pos) {
    char buffer[100],buffer2[50];
    CHAR *nvalue;
    int nmap;

    if (-1 == FP) return;
    name_table(&CT.cs,&nmap,&nvalue);

    buffer2[0] = '\0';
    if (nvalue)
      sprintf(buffer2,"; %s",nvalue);
    sprintf(buffer,"CHARSET (Default: %s%s): ",map_name(nmap,0),buffer2);
    edit_query(rCs(buffer),G_CHARSET,MAX_CHARSET,
	       CHARSET_ready,&key_group);
    return;
    
  } else if (&key_nl == pos) {
    char *S = "EOLN CONVENTION (Default: ???): ";
    if (-1 == FP) return;
    if (EOLN_unix == eoln_type)    S = "EOLN CONVENTION (Default: Unix): ";
    if (EOLN_network == eoln_type) S = "EOLN CONVENTION (Default: Network): ";
    edit_query(rCs(S),G_NEWLINE,1,NEWLINE_ready,&key_group);
    return;
  } else if (&key_search == pos) {
    edit_query(rCs("SEARCH: "),G_SEARCH,MAX_SEARCH,SEARCH_ready,&key_group);
    return;
  } else if (&key_ct == pos) {
    char buffer[50];
    if (-1 == FP) return;
    sprintf(buffer,"CONTENT-TYPE (Default: %s): ",
	    content_name(content_type,0));
    edit_query(rCs(buffer),G_CONTENT,MAX_CONTENT,
	       CONTENT_ready,&key_group);
    return;
  } else if (&key_ts == pos) {
    int tab, wrap,showspc,tab_step,indent_step,priority,pagemode;
    char buffer[50];

    get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		      &priority,&pagemode);

    sprintf(buffer,"Tabbing step (Default: %d): ",tab_step);
    edit_query(rCs(buffer),G_TABSTEP,MAX_NUMLEN,
	       TABSTEP_ready,&key_group);
    return;
  } else if (&key_is == pos) {
    int tab, wrap,showspc,tab_step,indent_step,priority,pagemode;
    char buffer[50];

    get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		      &priority,&pagemode);

    sprintf(buffer,"Indentation step (Default: %d): ",indent_step);
    edit_query(rCs(buffer),G_INDENTSTEP,MAX_NUMLEN,
	       INDENTSTEP_ready,&key_group);
    return;
  } else if (&key_cancel == pos) {
    *cancel = 1;
    edit_cancel(&key_group);
  } else if (&key_senter == pos ||
	     &key_searchn == pos || 
	     &key_ssearchn == pos ||
	     &key_ssearchf == pos ||
	     &key_searchf == pos) {
    if (&key_senter == pos) do_search(0);
    else if (&key_ssearchf == pos || &key_searchf == pos) do_search(1);
    else do_search(2);
  } else if (&key_goto == pos) {
    edit_query(rCs("GOTO: "),G_GOTO,MAX_GOTO,GOTO_ready,&key_group);
    edit_action(pos,c,&key_group);
  } else if (&key_tab == pos) {
    int tab, wrap,showspc,tab_step,indent_step,priority,pagemode;
    get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		      &priority,&pagemode);
    tab = !tab;
    update_prompt(visible, 5);
    set_window_status(visible,tab,wrap,showspc,tab_step,indent_step,
		      priority,pagemode);
  } else if (&key_prompt == pos) {
    have_prompt = !have_prompt;
  } else if (&key_time == pos) {
    showtime = !showtime;
  } else if (&key_wrap == pos) {
    int tab,wrap,showspc,tab_step,indent_step,priority,pagemode;
    get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		      &priority,&pagemode);
    wrap = !wrap;
    update_prompt(visible, 5);
    set_window_status(visible,tab,wrap,showspc,tab_step,indent_step,
		      priority,pagemode);
  } else if (&key_spc == pos) {
    int tab,wrap,showspc,tab_step,indent_step,priority,pagemode;
    get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		      &priority,&pagemode);
    showspc = !showspc;
    update_prompt(visible, 5);
    set_window_status(visible,tab,wrap,showspc,tab_step,indent_step,
		      priority,pagemode);
  } else if (&key_pagemode == pos) {
    int tab,wrap,showspc,tab_step,indent_step,priority,pagemode;
    get_window_status(visible,&tab,&wrap,&showspc,&tab_step,&indent_step,
		      &priority,&pagemode);
    pagemode = !pagemode;
    update_prompt(visible, 5);
    set_window_status(visible,tab,wrap,showspc,tab_step,indent_step,
		      priority,pagemode);
  } else if (&key_next == pos) {
    if (have_error) clear_errors();
    else get_next = 1;
  } else if (&key_forward == pos) {
    if (!have_error) {
      EOF_IN_FILE = 0;
      do_forwarding = 1;
    }
  } else if (&key_reread == pos) {
    if (!have_error) {
      reread_file(&FP,&EOF_IN_FILE,&EOLN_STATE,
		  &CT,map,map_param,
		  content_type);
    }
    
  } else if (&key_page == pos) {
    update_prompt(visible,7);
    go_next_page(visible);
  } else if (&key_prev == pos) {
    update_prompt(visible,7);
    go_prev_page(visible);
  } else if (&key_copyright == pos) {
    key_group = G_MESSAGE;
    map_errors();
    copyright_page(WIN_ERROR);
  } else if (&key_help == pos) {
    key_group = G_MESSAGE;
    map_errors();
    help_page(WIN_ERROR);
  } else if (&key_debug == pos) {
    key_group = G_MESSAGE;
    map_errors();
    debug_page(WIN_ERROR);
  } else if(&key_redraw == pos) {
    do_redraw();
  } else if(&key_lnext == pos) {
    int offset,Y,roffset,pagenum,pages;
    window_pos(visible,&Y,&offset,&roffset,&pagenum,&pages);
    if (window_rlines(visible) - roffset > lines -Y)
      scroll_window(visible); 
  } else if(&key_mquit == pos) {
    clear_errors();
    key_group = G_PAGER; 
  } else if(&key_lprev == pos) {
    int offset,Y,roffset,pagenum,pages;
    window_pos(visible,&Y,&offset,&roffset,&pagenum,&pages);
    if (roffset > 0) scroll_window_R(visible);
  } else if(&key_top == pos) {
    int offset,Y,roffset,pagenum,pages;
    update_prompt(visible,7);
    window_pos(visible,&Y,&offset,&roffset,&pagenum,&pages);
    if (roffset > 0) roffset = 0;
    map_window(visible,roffset,Y);
    while(flush_some_lines());
  } else if(&key_bottom == pos) {
    int offset,Y,roffset,roffset_old,rl,pagenum,pages;
    update_prompt(visible,7);
    window_pos(visible,&Y,&offset,&roffset_old,&pagenum,&pages);
    rl = window_rlines(visible);
    roffset = rl - lines + Y;
    if (roffset_old > roffset) roffset = roffset_old;
    map_window(visible,roffset,Y);
    while (window_pos(visible,&Y,&offset,&roffset_old,&pagenum,&pages),
	   pagenum < pages && pagenum > 0) {
      go_next_page(visible);
    }
    while(flush_some_lines());
  }
}

static void message_to_error(int len, CHAR_IDX *buffer) {
  map_errors();
  print_to_window(WIN_ERROR,-1,error_line,0,len,buffer);
}

static int do_keyboard(int cmd, int *cancel) {
  CHAR_IDX ch;
  switch (commands[cmd].what) {
    CHAR *c;
    int *result ;
  case C_text: 
    for ( c= rCs(commands[cmd].char_arr); 
	 c < rCs(commands[cmd].char_arr + commands[cmd].len);
	 c++) {
      result = parse_char(*c, key_group,&ch);
      if (NULL == result) {
	char *buffer1 = "Char \"", *buffer3 = "\" haven't action";
	int bl1 = strlen(buffer1), bl3 = strlen(buffer3);
	CHAR_IDX buffer[50];
	map_input(MAP_ASCII,bl1,rCs(buffer1),buffer);
	buffer[bl1] = ch;
	map_input(MAP_ASCII,bl3,rCs(buffer3),buffer+bl1+1);
	message_to_error(bl1+1+bl3,buffer);
      }
      else action(result,ch,cancel);
      
      if (*cancel) break;
    }
    return 1;
    break;
  default:
    result = parse_key (cmd, key_group,&ch);
    if (result) action(result,ch,cancel);
    else return 0;
    return 1;
    break; 
  }
}

static int auto_input = 1;
static CHAR *LAST_UNPARSED = NULL;

void schedule(void) {
#ifdef _HPUX_SOURCE
  int readfs,writefs;
#else
  fd_set readfs,writefs;
#endif
  struct timeval *timeout,Timeout;
  long atime;
  int res;
  int last_columns=0, last_lines = 0;
  int remove = 0; /* Unparsed */
  int cancel = 0, no_option = 0;

  while (!quitflag) {
    int biggest = terminal;
    int skip = 0, was_input = 0;
    int tail_ptr;

    if (need_redraw) {
      need_redraw = 0;
      do_redraw();
    }

    if (last_columns != columns) {
      last_columns = columns;
      if (pager_state && !remove)
	update_prompt(have_error ? WIN_ERROR : WIN_TEXT, 5);
      rewrap_windows();
      skip = 1; /* Prompt need recalculating */
    } 
    if (last_lines != lines) { 
      last_lines = lines;
      skip = 1; /* Prompt need recalculating */
    } 

    /* With this way we can lost some messages 
     * but it is very diffuclt guarantee that message is really
     * printed before signal occurs (because messag can go out
     * of display in window anyway)
     */
    if (got_from_TAIL(&tail_ptr)) {
      print_errors(tail_ptr);
      skip = 1; /* Prompt need recalculating */
    }

#ifdef _HPUX_SOURCE
    readfs = 1 << terminal;    
    writefs = 0;
#else
    FD_ZERO(&readfs);
    FD_SET(terminal,&readfs);
    FD_ZERO(&writefs);
#endif

    if (need_line_flush() || need_flush_buffer()) {
#ifdef _HPUX_SOURCE
      writefs |= 1 << terminal;
#else
      FD_SET(terminal,&writefs);
#endif
    }

    if (-1 != FP && !EOF_IN_FILE) {
      if (!NOT_SELECTABLE) {
#ifdef _HPUX_SOURCE
	readfs |= 1 << FP;
#else
	FD_SET(FP,&readfs);
#endif
	if (FP > biggest) biggest = FP;
      } else skip = 1;
    } 
    
    if (skip) {
      timeout = &Timeout;
      Timeout.tv_sec = 0;
      Timeout.tv_usec = 0;
    } else if (ask_timeout(&atime)) {
      timeout = &Timeout;
      Timeout.tv_sec = atime / 10L;
      Timeout.tv_usec = ( atime % 10L) * 100000L;
    } else timeout = NULL;

    if (pager_state && showtime && !timeout) {
      timeout = &Timeout;
      Timeout.tv_sec = seconds_to_next_minute();
      Timeout.tv_usec = 1;
      skip = 2;
    } else if (pager_state && showtime) {
      int s = seconds_to_next_minute();
      if (s < timeout -> tv_sec) {
	timeout = &Timeout;
	Timeout.tv_sec = s;
	Timeout.tv_usec = 1;
	skip = 2;
      }
    }

    if (DEBUG_FILIO || DEBUG_TERM || DEBUG_BUFFER) {
      print_debug("Doing select: max fd=%d, %s (skip=%d)",
		  biggest, timeout ? "timeout" : "no timeout",
		  skip);
    }

#ifdef _HPUX_SOURCE
    res = select(biggest+1,&readfs,&writefs,(int *)NULL,
		     timeout);
#else
    res = select(biggest+1,&readfs,&writefs,(fd_set *)NULL,
		     timeout);
#endif
    if (-1 == res) {
      int code = errno;
      if (DEBUG_FILIO) {
	print_debug("Got errno=%d from select",
		    code);
      }
      if (EINTR == code) continue;
      if (ENXIO == code && !NOT_SELECTABLE) {
	not_selectable(&FP);
	continue;
      }
      print_error(code,
		  "(%s) Error when waiting response of terminal from /dev/tty",
		  prog);
      quitflag = 1;
      continue;
    }
    if (0 == res && !skip) do_timeout();

#ifdef _HPUX_SOURCE
    if (readfs & (1 << terminal))
#else
    if (FD_ISSET(terminal,&readfs)) 
#endif
      {
	
	CHAR input_buffer[INPUT_BUFFER_SIZE+1];
	int len = read_from_terminal(input_buffer);
	if (len > 0) parse_input(input_buffer, len); 
	was_input = 1;
      }
    
    cancel = 0;
    while (command_len > 0) {
      CHAR *name;
      if (DEBUG_TERM) {
	CHAR *R1=printable_command(0);
	print_debug("Command: %s",R1);
	FREE(R1);
      }
      if (do_escape(0) || 
	  cancel || 
	  do_keyboard(0,&cancel)) {

      } else if (0 != (name = key_name(0))) {
	  char *buffer1 = "Key ", *buffer3 = " haven't action";
	  int bl1 = strlen(buffer1), bl3 = strlen(buffer3);
	  int bl2 = strlen(Cs(name));
	  CHAR_IDX buffer[50];
	  map_input(MAP_ASCII,bl1,rCs(buffer1),buffer);
	  map_input(MAP_ASCII,bl2,name,buffer+bl1);
	  map_input(MAP_ASCII,bl3,rCs(buffer3),buffer+bl1+bl2);
	  message_to_error(bl1+bl2+bl3,buffer);
      } else {
	CHAR *R1=printable_sequence(0);
	print_notify("Unexpected escape sequence: %s",R1);
	free(R1);
      }

      rem_command(0);
    }

    if (pager_state)
    {
      CHAR *R2=unparsed_printable();
      CHAR *T=composing();

      if (cancel) {
	clear_errors();
	key_group = G_PAGER; 
      }

      if (T) {
	CHAR *TMP = concat_text(NULL,T);
	if (R2) {
	  TMP = concat_text(TMP,rCs(" "));
	  TMP = concat_text(TMP,R2);
	  FREE(R2);
	}
	R2 = TMP;
      }

      if (R2) {
	int len = strlen(Cs(R2));

	if (LAST_UNPARSED && 0 == strcmp(Cs(R2),Cs(LAST_UNPARSED))) {
	  FREE(R2);

	} else {
	  CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));

	  if (LAST_UNPARSED) {
	    FREE(LAST_UNPARSED); LAST_UNPARSED = NULL;
	  }
	  
	  clear_window(WIN_UNPARSED);
	  set_window_status(WIN_UNPARSED,0,0,0,8,4,PRI_COMPLETE,0);
	  map_window(WIN_UNPARSED,0,lines);

	  map_input(MAP_ASCII,len,R2,mapped);
	  print_to_window(WIN_UNPARSED,1,1,FL_BOLD,len,mapped);
	
	  set_win_prompt(WIN_UNPARSED,len+1,1);
	  FREE(mapped);
	  LAST_UNPARSED = R2;
	  remove = 1;
	}
      } else if (remove) {
	if (LAST_UNPARSED) {
	  FREE(LAST_UNPARSED); LAST_UNPARSED = NULL;
	}
	unmap_window(WIN_UNPARSED);
	remove = 0;
      }
    }
    
    if (pager_state && get_next) {
      CHAR *name = NULL;

      set_win_prompt(-1,-1,-1);
      update_prompt(WIN_TEXT, 0);
      clear_window(WIN_TEXT);
      map_window(WIN_TEXT,1-down_line,1);
      
      if (0 == arg_left) { 
	if (auto_input) {
	  if (!isatty(0)) name = rCs(STD_INPUT);
	  else {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    copyright_page(WIN_TEXT);
	    auto_input = 0;
	    if (-1 != FP) {
	      exit_content(&CT);
	      close_file(&FP);
	    }
	    EOF_IN_FILE = 1;
	  }
	} else quitflag = 1; 
      } else while(arg_left--> 0) {
	if ('-' == **arg_vec && !no_option) {
	  if (0 == strcmp("-",*arg_vec)) name = rCs(STD_INPUT);
	  else if (0 == strcmp("--",*arg_vec)) no_option = 1;
	  else if (0 == strcmp("-A",*arg_vec)) { map = MAP_ASCII;
						 map_param = NULL; }
	  else if (0 == strcmp("-L",*arg_vec)) { map = MAP_LATIN1;
						 map_param = NULL; }
	  else if (0 == strcmp("-L1",*arg_vec)) { map = MAP_LATIN1;
						  map_param = NULL; }
	  else if (0 == strcmp("-L2",*arg_vec)) { map = MAP_LATIN2;
						  map_param = NULL; }
	  else if (0 == strcmp("-L3",*arg_vec)) { map = MAP_LATIN3;
						  map_param = NULL; }
	  else if (0 == strcmp("-L4",*arg_vec)) { map = MAP_LATIN4;
						  map_param = NULL; }
	  else if (0 == strcmp("-L5",*arg_vec)) { map = MAP_LATIN5;
						  map_param = NULL; }
	  else if (0 == strcmp("-M",*arg_vec)) { map = MAP_MULTINATIONAL;
						 map_param = NULL; }
	  else if (0 == strcmp("-PC",*arg_vec)) { map = MAP_CP437;
						  map_param = NULL; }
	  else if (0 == strcmp("-MAC",*arg_vec)) { map = MAP_MAC;
						   map_param = NULL; }
	  else if (0 == strcmp("-F",*arg_vec)) { map = MAP_FINNISH;
						 map_param = NULL; }
	  else if (0 == strcmp("-UK",*arg_vec)) { map = MAP_UK;
						  map_param = NULL; }
	  else if (0 == strcmp("-NR",*arg_vec)) { map = MAP_NOR;
						  map_param = NULL; }
	  else if (0 == strcmp("-SW",*arg_vec)) { map = MAP_SWE;
						  map_param = NULL; }
	  else if (0 == strcmp("-FR",*arg_vec)) { map = MAP_BEL;
						  map_param = NULL; }
	  else if (0 == strcmp("-CA",*arg_vec)) { map = MAP_CAN;
						  map_param = NULL; }
	  else if (0 == strcmp("-GR",*arg_vec)) { map = MAP_GER;
						  map_param = NULL; }
	  else if (0 == strcmp("-SP",*arg_vec)) { map = MAP_SPA;
						  map_param = NULL; }
	  else if (0 == strcmp("-IT",*arg_vec)) { map = MAP_ITAL;
						  map_param = NULL; }
	  else if (0 == strcmp("-U7",*arg_vec)) { map = MAP_UTF7;
						  map_param = NULL; }
	  else if (0 == strcmp("-U8",*arg_vec)) { map = MAP_UTF8;
						  map_param = NULL; }
	  else if (0 == strcmp("-MN",*arg_vec)) { map = MAP_Mnem;
						  map_param = rCs("29"); }
	  else if (0 == strncmp("-MN:",*arg_vec,4)) { map = MAP_Mnem;
						  map_param = rCs(*arg_vec+4); }
	  else if (0 == strcmp("-t",*arg_vec)) enable_tab = 0;
	  else if (0 == strcmp("-t+",*arg_vec)) enable_tab = 1;
	  else if (0 == strncmp("-t",*arg_vec,2) && isdigit((*arg_vec)[2])) {
	    input_tabstep = atoi((*arg_vec)+2);
	  }
	  else if (0 == strncmp("-i",*arg_vec,2) && isdigit((*arg_vec)[2])) {
	    input_indentstep = atoi((*arg_vec)+2);
	  }
	  else if (0 == strcmp("-w",*arg_vec)) enable_wrapping = 0;
	  else if (0 == strcmp("-w+",*arg_vec)) enable_wrapping = 1;
	  else if (0 == strcmp("-sp",*arg_vec)) enable_eatspc = 0;
	  else if (0 == strcmp("-sp+",*arg_vec)) enable_eatspc = 1;
	  else if (0 == strcmp("-st",*arg_vec)) showtime = 0;
	  else if (0 == strcmp("-st+",*arg_vec)) showtime = 1;
	  else if (0 == strcmp("-j",*arg_vec)) enable_fulljust = 0;
	  else if (0 == strcmp("-j+",*arg_vec)) enable_fulljust = 1;
	  else if (0 == strcmp("-p",*arg_vec)) have_prompt = 0;
	  else if (0 == strcmp("-p+",*arg_vec)) have_prompt = 1;
	  else if (0 == strcmp("-pm",*arg_vec)) enable_pages = 0;
	  else if (0 == strcmp("-pm+",*arg_vec)) enable_pages = 1;
	  else if (0 == strcmp("-nu",*arg_vec)) eoln_type = EOLN_unix;
	  else if (0 == strcmp("-nn",*arg_vec)) eoln_type = EOLN_network;
	  else if (0 == strcmp("-pl",*arg_vec)) content_type = CT_plain;
	  else if (0 == strcmp("-tr",*arg_vec)) content_type = CT_terminal;
	  else if (0 == strcmp("-er",*arg_vec)) content_type = CT_enriched;
	  else if (0 == strcmp("-rt",*arg_vec)) content_type = CT_richtext;
	  else if (0 == strcmp("-H",*arg_vec)) {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    help_page(WIN_TEXT);
	    arg_vec++;
	    auto_input = 0;
	    if (-1 != FP) {
	      exit_content(&CT);
	      close_file(&FP);
	    }
	    EOF_IN_FILE = 1;
	    break;
	  } else if (0 == strcmp("-cs",*arg_vec)) {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    charset_listing(WIN_TEXT);
	    arg_vec++;
	    auto_input = 0;
	    if (-1 != FP) {
	      exit_content(&CT);
	      close_file(&FP);
	    }
	    EOF_IN_FILE = 1;
	    break;
	  } else if (0 == strcmp("-DS",*arg_vec)) {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    debug_page(WIN_TEXT);
	    arg_vec++;
	    auto_input = 0;
	    if (-1 != FP) {
	      exit_content(&CT);
	      close_file(&FP);
	    }
	    EOF_IN_FILE = 1;
	    break;
	  } else if (0 == strcmp("-T",*arg_vec)) {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    test_page(WIN_TEXT);
	    arg_vec++;
	    auto_input = 0;
	    EOF_IN_FILE = 1;
	    break;
	  } else if (0 == strcmp("-T2",*arg_vec)) {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    test_page2(WIN_TEXT);
	    arg_vec++;
	    auto_input = 0;
	    if (-1 != FP) {
	      exit_content(&CT);
	      close_file(&FP);
	    }
	    EOF_IN_FILE = 1;
	    break;
	  } else if (0 == strcmp("-T3",*arg_vec)) {
	    set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			      !enable_eatspc,input_tabstep,input_indentstep,
			      PRI_NORMAL,enable_pages);
	    test_page3(WIN_TEXT);
	    arg_vec++;
	    auto_input = 0;
	    if (-1 != FP) {
	      exit_content(&CT);
	      close_file(&FP);
	    }
	    EOF_IN_FILE = 1;
	    break;
	  } else if (0 == strcmp("-CS",*arg_vec)) {
	    if (arg_left > 0) {
	      if ('\0' == (*++arg_vec)[0]) {
		arg_left--;
		map = map_null_name;
		map_param = NULL;
	      } else {
		int cs = find_map(rCs(*arg_vec));
		arg_left--;
		map = MAP_ASCII;
		map_param = NULL;
		if (cs >= 0) map = cs;
		else print_notify("Bad charset: %s",*arg_vec);
	      }
	    } else print_notify("Argument expected for -CS");
	  } else if (0 == strcmp("-ct",*arg_vec)) {
	    if (arg_left > 0) {
	      int ct = search_content(rCs(*++arg_vec),0);
	      arg_left--;
	      content_type = CT_plain;
	      if (ct >= 0) content_type = ct;
	      else print_notify("Bad content-type: %s",*arg_vec);
	    } else print_notify("Argument expected for -ct");
	  } 
	  /* OBSOLETED: */
	  else if (0 == strcmp("-b",*arg_vec)) content_type = CT_plain;
	  else if (0 == strcmp("-b+",*arg_vec)) content_type = CT_terminal;
	  else if (0 == strcmp("-a",*arg_vec)) content_type  = CT_plain;
	  else if (0 == strcmp("-a+",*arg_vec)) content_type = CT_terminal;
	  else if (0 == strcmp("-nl",*arg_vec)) NOT_IN_USE = 0;
	  else if (0 == strcmp("-nl+",*arg_vec)) NOT_IN_USE = 0;
	  else print_notify("Bad option: %s",*arg_vec);

	} else {
	  set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			    !enable_eatspc,input_tabstep,input_indentstep,
			    PRI_NORMAL,enable_pages);
	  name = rCs(*arg_vec);
	  arg_vec++;
	  break;
	}
	arg_vec++;
      }

      if (NULL == name && auto_input)
	name = rCs(STD_INPUT);
      
      if (name) {
	if (-1 != FP) {
	  exit_content(&CT);
	  close_file(&FP);
	}
	auto_input = 0;
	read_file(name,&FP,&EOF_IN_FILE,&EOLN_STATE,
		  &CT,map,map_param,
		  content_type);
	
	set_window_status(WIN_TEXT,enable_tab,enable_wrapping,
			  !enable_eatspc,input_tabstep,input_indentstep,
			  PRI_NORMAL,enable_pages);

      }
      
      get_next = 0;
    }

#ifdef _HPUX_SOURCE
    if (-1 != FP && !EOF_IN_FILE && 
	(NOT_SELECTABLE || (readfs & (1 << FP))))
#else
    if (-1 != FP && !EOF_IN_FILE && 
	(NOT_SELECTABLE || FD_ISSET(FP,&readfs)))
#endif
      {
	handle_file(&CT,&FP,&EOF_IN_FILE,&EOLN_STATE);

	if (do_forwarding) {
	  int prev,offset,Y,roffset,pagenum,pages;
	  for (window_pos(WIN_TEXT,&Y,&offset,&roffset,&pagenum,&pages),
	       prev=roffset; 
	       window_rlines(WIN_TEXT) - roffset > lines -Y && 
	       roffset - prev <lines;
	       window_pos(WIN_TEXT,&Y,&offset,&roffset,&pagenum,&pages))
	    scroll_window(WIN_TEXT); 
	  do_forwarding = 0;
	}

      }

    if (pager_state && !remove)
      update_prompt(have_error ? WIN_ERROR : WIN_TEXT,
		    !quitflag ? 
		       have_error ? 3 : 
		         !EOF_IN_FILE ? 2 : 
		           arg_left > 0 ? 1 : 
                           4 : 
                    0);


#ifdef _HPUX_SOURCE
    if (writefs & (1 << terminal))
#else
    if (FD_ISSET(terminal,&writefs)) 
#endif
      { 
	int val = need_line_flush();
	if (was_input && 2 != val) {
	  if (DEBUG_BUFFER || DEBUG_LBUFFER) 
	    print_debug("BUFFER: was_input=%d, "
			"need_line_flush()=%d - no flush",
			was_input,val);
	} else {
	  if (DEBUG_BUFFER || DEBUG_LBUFFER) 
	    print_debug("BUFFER: was_input=%d, "
			"need_line_flush()=%d - flushing...",
			was_input,val);
	  if (!flush_some_lines())
	    flush_buffer(0);
	}
      }
  }

  if (-1 != FP) {
    exit_content(&CT);
    close_file(&FP);
  }

  if (LAST_UNPARSED) {
    FREE(LAST_UNPARSED); LAST_UNPARSED = NULL;
  }

  while (flush_some_lines());

}

extern gid_t getgid(void);
extern gid_t getegid(void);
extern uid_t getuid(void);
extern uid_t geteuid(void);

static int is_unsecure(void) {
  return
    getgid() != getegid() ||
      getuid() != geteuid();
}

/* 0 = unknown
   1 = bourne
   2 = csh
   3 = rc 
   */

static struct stbl {
  char *sh;
  int syntax;
} syntax_tbl[] = {
  { "sh",   1 },   /* Bourne or Posix shell */
  { "jsh",  1 },   /* Bourne shell with job control */
  { "rsh",  1 },   /* Restricted shell */
  { "bash", 1 },   /* Bourne again shell */
  { "csh",  2 },   /* C shell */
  { "tcsh", 2 },   /* C shell with with file name completion 
		      and command  line editing */
  { "rc",   3 } }; /* shell */

static int give_syntax(CHAR *shell) {
  int result = 0,i;
  char *A=Cs(concat_text(NULL,shell)),*B,*C=A;

  if (NULL != (B = strrchr(A,'/'))) C = B+1;

  for (B=strtok(C,".-:"); NULL != B; B=strtok(NULL,".-:")) {
    for (i = 0; i < sizeof(syntax_tbl)/sizeof(struct stbl); i++) {
      if (0 == strcmp(syntax_tbl[i].sh,B)) {
	if (0 == result) result = syntax_tbl[i].syntax;
	else result = -1;
      }
    }
  }

  FREE((void *)A);
  return result;
}

int main(int argc, char *argv[]) {
  CHAR *home=rCs(getenv("HOME"));
  CHAR *shell=rCs(getenv("SHELL"));
  char *debug=getenv("KEHPAGER_DEBUG");
  int shell_syntax = 0,print_it = 0;
  int unsecure = is_unsecure();

  CT=NULL_content;
  EOLN_STATE = NULL_EOLN_STATE;

  if (argc > 0) prog=argv[0];
  arg_left = argc -1;
  arg_vec = argv + 1;

  if (debug && unsecure) 
    print_notify("Ignoring KEHPAGER_DEBUG ...");
  else if (debug) {
    char *B=Cs(concat_text(NULL,rCs(debug)));
    char *A=strtok(B,":");
    int flag = atoi(A);
    char *N=strtok(NULL,":");
    int debugfile = open(N,O_WRONLY|O_CREAT,0600);

    if (flag && -1 != debugfile && -1 != dup2(debugfile,2))
      debugflag = flag;
    
    FREE(B);
  }

  if (shell) shell_syntax = give_syntax(shell);

  if (arg_left > 0 && 0 == strncmp("-D",*arg_vec,2) && isdigit((*arg_vec)[2]))
    {
      debugflag = atoi((*arg_vec)+2);
      arg_left--;
      arg_vec++;
    }

  /* malloc_debug(2); */

  init_tables();
  init_windows();

  read_rc_file (rCs(GLOBAL_INIT_FILE));

  if (home) {
    CHAR *name = concat_text(NULL,home);
    if (0 != strcmp(Cs(name),"/")) name = concat_text(name,rCs("/"));
    name = concat_text(name,rCs(".kehpagerrc"));
    read_rc_file(name);
    FREE(name);
  }

  if (!give_mapping(&key_quit)) {
    print_notify("No defination of quit character!\n"
		 "   Add keyboard.quit: key \n"
		 "   to global init file: " GLOBAL_INIT_FILE );
    exit(8);
  }

  import_vars(); /* reads $COLUMNS and $LINES */
  
  while (arg_left > 0 ) {
    if (0 == strcmp(*arg_vec,"-f")) disable_fast_start = 1;
    else if (0 == strcmp(*arg_vec,"-e")) {
      disable_fast_start = 1;
      print_it = 1;
      auto_input = 0;
    } else if (0 == strcmp(*arg_vec,"-eu")) {
      disable_fast_start = 1;
      shell_syntax = 1;
      print_it = 1;
      auto_input = 0;
    } else if (0 == strcmp(*arg_vec,"-ec")) {
      disable_fast_start = 1;
      shell_syntax = 2;
      print_it = 1;
      auto_input = 0;
    } else if (0 == strcmp(*arg_vec,"-erc")) {
      disable_fast_start = 1;
      shell_syntax = 3;
      print_it = 1;
      auto_input = 0;
    } else if (0 == strcmp("-X",*arg_vec)) set_xterm_latin1 = 0;
    else if (0 == strcmp("-X+",*arg_vec)) set_xterm_latin1 = 1;
    else if (0 == strcmp("-X++",*arg_vec)) set_xterm_latin1 = 2;
    else if (0 == strcmp("-7",*arg_vec))  force_mode = 1;
    else if (0 == strcmp("-8",*arg_vec))  force_mode = 2;
    else if (0 == strcmp("-q",*arg_vec))  limit_query = 1;
    else if (0 == strcmp("-q+",*arg_vec))  limit_query = 0;
    else if (0 == strcmp("-s",*arg_vec))  use_scrolling = 0;
    else if (0 == strcmp("-s+",*arg_vec)) use_scrolling = 1;
    else if (0 == strcmp("-rs",*arg_vec))  query_terminal_size = 0;
    else if (0 == strcmp("-rs+",*arg_vec)) query_terminal_size = 1;
    else if (0 == strcmp("-wt",*arg_vec))  wait_quit = 0;
    else if (0 == strcmp("-wt+",*arg_vec)) wait_quit = 1;
    else if (0 == strncmp("-N",*arg_vec,2) && isdigit((*arg_vec)[2])) 
      default_language = atoi((*arg_vec)+2);
    else break;
    arg_left--;
    arg_vec++;
  }

  if (!open_terminal()) {
    exit(5);    /* No terminal */
  }

  get_terminal_state();
  schedule();
  close_files();
  reset_terminal_state();
  close_terminal();
  if (print_it) {
    switch(shell_syntax) {
    case -1: 
      print_notify("(%s) Ambiguous shell name '%s'. "
		   "Use option -eu, -ec or -erc.",
		   prog,shell);
      exit(10);
    case 0:
      print_notify("(%s) Unrecognized shell name '%s'. "
		   "Use option -eu, -ec or -erc.",
		   prog,shell ? shell : rCs(""));
      exit(10);
    case 1: /* bourne shell */
      print_kehpager_term(NULL,
			  "%s=\"%s\";", "export %s;",
			  "\n");
      break;
    case 2: /* csh */
      print_kehpager_term(NULL,
			  "setenv %s \"%s\";", NULL,
			  "\n");
      break;
    case 3: /* rc */
      print_kehpager_term(NULL,
			  "%s='%s';", NULL,
			  "\n");
      break;
    }
  }
  exit(0);
}
