/*  file: esc.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stddef.h>
#include <stdio.h>
#include <stdarg.h>

#include "kehpager.h"
#include "memory.h"
#include "charset.h"
#include "esc.h"
#include "terminal.h"
#include "control.h"

static struct charname {
  int chr;
  char *name;
} charnames[] = {
  { ESC, "ESC" },
  { CAN, "CAN" },
  { BS,  "BS" },
  { SI,  "SI" },
  { SO,  "SO" },
  { LF,  "LF" },
  { CR,  "CR" },
  { ' ', "SPC" },
  { TAB, "TAB" },
  { DEL, "DEL" },
  { IND, "IND" },
  { NEL, "NEL" },
  { HTS, "HTS" },
  { RI,  "RI" },
  { SS2, "SS2" },
  { SS3, "SS3" },
  { DCS, "DCS" },
  { CSI, "CSI" },
  { ST,  "ST" },
  { OSC, "OSC" },
  { PM,  "PM" },
  { APC, "APC" } };

#define charnames_count (sizeof charnames / sizeof (struct charname))

static int map7to8bit(String *source, String *target,
	       int *is8bit) {
  /* Mapping 
        ESC + ( 0x40 - 0x5F ) -> ( 0x80 - 0x9F )
     Returns number of charters ins source string processed
       
   */
  int len = StrLEN(*source);
  CHAR *S = StrPTR(*source);

  StrFree(target);
  StrInit(target,len+1);

  while (S < StrPTR(*source) + StrLEN(*source)) {
    if (ESC == *S) {
      if (S+1 == StrPTR(*source) + StrLEN(*source)) break;   /* incomplete */
      if (*(S+1) < 0x40 || *(S+1) > 0x5F) { StrAdd(target,*S++); continue; }
      S++;
      StrAdd(target,*S + 0x40);
      S++;
      if (is8bit) *is8bit = 0;
      continue;
    } 
    if (*S >= 0x80 && *S <= 0x9F) if (is8bit) *is8bit = 1;
    StrAdd(target,*S++);
  }
  return S - StrPTR(*source);
}

CHAR * map_buffer(CHAR *buffer, int need_mapping,
		  CHAR *static_buffer, int static_len) {
  /* Mapping
     ( 0x80 - 0x9F ) -> ESC + ( 0x40 - 0x5F )
     
   */

  int alloced= static_buffer ? static_len : strlen(buffer)+1;
  int len=0;
  CHAR *result = (static_buffer ? static_buffer : 
			   (CHAR *)MALLOC(alloced)),
    *ptr;
#define ADD_CHR(c) { \
  if (len > alloced) { \
     alloced += 10; \
     result = (result == static_buffer) ? (CHAR *)MALLOC(alloced) : \
       (CHAR *)REALLOC((void *)result,alloced); \
  } \
  result[len++] = (c); \
}
  for (ptr = buffer; '\0' != *ptr; ptr++) {
    if ( 0x80 <= *ptr && *ptr <= 0x9F && need_mapping) {
      ADD_CHR(ESC); ADD_CHR(*ptr - 0x40);
    } else ADD_CHR(*ptr);
  }
  ADD_CHR(0);
#undef ADD_CHR
  return result;
}

void char2names(String *source,CHAR **target) {
  CHAR *S;
  if (*target) FREE (*target);
  *target = NULL;
  
  for (S = StrPTR(*source); S < StrPTR(*source) + StrLEN(*source); S++) {
    int i,found = -1;
    if (*target) *target = concat_text(*target,rCs(" "));
    for (i = 0; i < charnames_count; i++) 
      if (*S == charnames[i].chr) { found = i; break; }
    if (-1 != found) {
      *target = concat_text(*target,rCs(charnames[found].name));
    } else if (*S > ' ' && *S < 127) {
      CHAR buffer[2];
      buffer[0] = *S;
      buffer[1] = 0;
      *target = concat_text(*target,buffer);
    } else {
      CHAR buffer[5];
      sprintf(Cs(buffer),"\\%03o",(int)*S);
      *target = concat_text(*target,buffer);
    }
  }
  *target = concat_text(*target,rCs(""));
}

struct command commands[MAX_COMMANDS];
int command_len = 0;

static void clear_buffer(void) {
  int i;
  for (i = 0; i < command_len; i++) {
    commands[i].what = C_text;
    commands[i].len = 0;
    if (commands[i].char_arr) FREE((void *)commands[i].char_arr);
    commands[i].char_arr = NULL;
  }
  command_len = 0;
}

void rem_command(int ptr) {
  if (ptr < 0 || ptr >= command_len) {
    print_notify("[%s] Bug(rem_command): ptr = %d, command_len = %d\n",
		 prog,ptr,command_len);
    close_terminal();
    exit(6);
  }
  commands[ptr].what = C_text;
  commands[ptr].len = 0;
  if (commands[ptr].char_arr) FREE((void *)commands[ptr].char_arr);
  commands[ptr].char_arr = NULL;
  while (++ptr < command_len) commands[ptr-1] = commands[ptr];
  commands[command_len].what = C_text;
  commands[command_len].len = 0;
  commands[command_len].char_arr = NULL;
  command_len--;
}

static void put_command (CHAR cmd,
			 c_type what, 
			 String *char_arr) {
  CHAR * char_ptr = NULL;

  if (command_len >= MAX_COMMANDS) {
    print_notify("Command buffer overflow! Clearing buffer !!");
    clear_buffer();
  }

  commands[command_len].len = StrLEN(*char_arr);
  commands[command_len].what = what;
  commands[command_len].cmd = cmd;
  if (StrLEN(*char_arr)) {
    char_ptr = (CHAR *)MALLOC(StrLEN(*char_arr)+1);
    memcpy(Cs(char_ptr),Cs(StrPTR(*char_arr)),StrLEN(*char_arr));
    char_ptr[StrLEN(*char_arr)] = '\0';
  }
  commands[command_len].char_arr = Cs(char_ptr);
  command_len++;
}

CHAR * printable_command(int ptr) {
  char *ptr1 = "";
  CHAR *ptr2=NULL;
  CHAR *ptr3=NULL;
  CHAR *result;
  String buffer;
  int len;
  StrInit(&buffer,2);

  switch (commands[ptr].what) {
  case C_text: ptr1 = "C_text"; break;
  case C_ctl: ptr1 = "C_ctl"; break;
  case C_escape: ptr1 = "C_escape"; break;
  case C_escpair: ptr1 = "C_escpair"; break;
  case C_string: ptr1 = "C_string"; break;
  case C_single: ptr1 = "C_single"; break;
  default: ptr1="{unknown}"; break;
  }
  StrAdd(&buffer,commands[ptr].cmd);
  char2names(&buffer,&ptr2);
  if (commands[ptr].char_arr) {
    String S;
    S.buffer = rCs(commands[ptr].char_arr);
    S.len = commands[ptr].len;
    S.alloced = 0;
    char2names(&S,&ptr3);
  }
  else ptr3 = NULL;
  len = strlen(ptr1) + strlen(ptr2) + 
    (ptr3 ? strlen(ptr3) : 0) + 40;
  result = (CHAR *)MALLOC(len);
  sprintf(Cs(result),"commands[]: what=%s len=%d char_arr=%s cmd=%s",
	  ptr1,commands[ptr].len, ptr3 ? ptr3 : rCs("") ,
	  ptr2);
  if (ptr3) FREE(ptr3);
  if (ptr2) FREE(ptr2);
  StrFree(&buffer);

  return result;
}

CHAR * printable_sequence(int ptr) {
  CHAR *ptr2=NULL;
  CHAR *ptr3=NULL;
  CHAR *result;
  String buffer;
  int len;

  StrInit(&buffer,2);
  StrAdd(&buffer,commands[ptr].cmd);
  char2names(&buffer,&ptr2);

  if (commands[ptr].char_arr) {
    String S;
    S.buffer = rCs(commands[ptr].char_arr);
    S.len = commands[ptr].len;
    S.alloced = 0;
    char2names(&S,&ptr3);
  }
  else ptr3 = NULL;
  
  len = strlen(Cs(ptr2)) + 
    (ptr3 ? strlen(Cs(ptr3)) : 0) + 10;
  result = (CHAR *)MALLOC(len);

  switch (commands[ptr].what) {
  case C_text: sprintf(Cs(result),"%s",ptr3); break;
  case C_ctl: sprintf(Cs(result),"%s",ptr2); break;
  case C_escape: sprintf(Cs(result),"CSI %s %s",ptr3 ? ptr3 : rCs(""),
			 ptr2); break;
  case C_escpair: sprintf(Cs(result),"ESC %s %s",ptr3 ? ptr3 : rCs(""),
			  ptr2); break;
  case C_string: sprintf(Cs(result),"DCS %s %s",ptr3 ? ptr3 : rCs(""),
			 ptr2); break;
  case C_single: sprintf(Cs(result),"%s %s",ptr3,ptr2); break;
  default: sprintf(Cs(result),"{unknown}"); break;
  }

  StrFree(&buffer);
  return result;
}

static int eat_one(String *ptr) {
  CHAR cmd = ' ';
  c_type what = C_text;
  String char_arr;
  CHAR *P = StrPTR(*ptr);

  StrInit(&char_arr,10);

#define ADD_CHR(c) StrAdd(&char_arr,c)
#define CLEAR StrFree(&char_arr)

#define NEXT { P++; \
  if (P == StrPTR(*ptr) + StrLEN(*ptr)) { CLEAR; return 0; } \
}
#define DONE(ptr,P) { \
  memcpy(Cs((ptr)->buffer),Cs(P),StrLEN(*ptr)-(P-StrPTR(*ptr))); \
  (ptr)->len = StrLEN(*ptr)-(P-StrPTR(*ptr)); }

 again:
  if (*P == CSI) {
    CHAR *MSG = 0,*CN = 0;
    NEXT;
    while ( 0x30 <= *P && *P <= 0x3F) { ADD_CHR(*P); NEXT; }
    while ( 0x20 <= *P && *P <= 0x2F) { ADD_CHR(*P); NEXT; }
    if ( 0x40 <= *P && *P <= 0x7E) { 
      cmd = *P++;
      what = C_escape;
      put_command(cmd,what,&char_arr);
      DONE(ptr,P); 
      CLEAR;
      return 1;
    }
    char2names(&char_arr,&MSG);
    { String S;
      S.buffer=P;
      S.len= StrLEN(*ptr)-(P-StrPTR(*ptr));
      S.alloced=0;
      char2names(&S,&CN);
    }
    print_notify("Parse Error: CSI %s\n"
		 "        Got: %s",MSG,CN);
    FREE(MSG);
    FREE(CN);
    CLEAR;
    goto again;
  } else if (DCS == *P) {
    CHAR *MSG = 0, *CN = 0;
    NEXT;
    while (!(*P < ' ' || *P == 0x7F || (0x80 <= *P && *P <= 0x9F))) { 
      ADD_CHR(*P); NEXT; 
    }
    if (ST == *P) {
      cmd = *P++;
      what = C_string;
      put_command(cmd,what,&char_arr);
      DONE(ptr,P); 
      CLEAR;
      return 1;
    }
    char2names(&char_arr,&MSG);
    { String S;
      S.buffer=P;
      S.len= StrLEN(*ptr)-(P-StrPTR(*ptr));
      S.alloced=0;
      char2names(&S,&CN);
    }
    print_notify("Parse Error: DCS %s\n"
		 "        Got: %s",MSG,CN);
    FREE(MSG);
    FREE(CN);
    CLEAR;
    goto again;
  } else if (ESC == *P) {
    CHAR *MSG = 0, *CN = 0;
    NEXT;
    while (0x20 <= *P && *P <= 0x2F) { ADD_CHR(*P); NEXT; }
    if (0x30 <= *P && *P <= 0x5F) { 
      cmd = *P++;
      what = C_escpair;    /* not always pair */
      put_command(cmd,what,&char_arr);
      DONE(ptr,P); 
      CLEAR;
      return 1;
    }
    char2names(&char_arr,&MSG);
    { String S;
      S.buffer=P;
      S.len= StrLEN(*ptr)-(P-StrPTR(*ptr));
      S.alloced=0;
      char2names(&S,&CN);
    }
    print_notify("Parse Error: ESC %s\n"
		 "        Got: %s",MSG,CN);
    FREE(MSG);
    FREE(CN);
    CLEAR;
    goto again;
  } else if (*P == SS2 || *P == SS3) {
    CHAR *MSG = 0, *CN = 0;
    ADD_CHR(*P); 
    NEXT;
    if (*P >  ' ' && *P < 0x7F) { 
      cmd = *P++;
      what = C_single;
      put_command(cmd,what,&char_arr);
      DONE(ptr,P);
      return 1;
    }
    char2names(&char_arr,&MSG);
    { String S;
      S.buffer=P;
      S.len= StrLEN(*ptr)-(P-StrPTR(*ptr));
      S.alloced=0;
      char2names(&S,&CN);
    }
    print_notify("Parse Error: (single) %s\n"
		 "        Got: %s",MSG,CN);
    FREE(MSG);
    FREE(CN);
    CLEAR;
    goto again;
  } else if (*P < ' ' || *P == 0x7F || (0x80 <= *P &&*P <= 0x9F)) {
    CHAR ch = *P;
    if (HAZ_MAY(HAZ_newline) && ch == CR) {
      /* Eat LF if LF follows CR */
      CHAR *tmp = P;
      NEXT;
      if (LF != *P) P = tmp;   /* It wasn't LF */
    }

    cmd = ch; P++;
    what = C_ctl;
    put_command(cmd,what,&char_arr);
    DONE(ptr,P);
    return 1;
  }
  while (P < StrPTR(*ptr) + StrLEN(*ptr)
	 && !(*P < ' ' || *P == 0x7F || (0x80 <= *P &&*P <= 0x9F))) {
    ADD_CHR(*P); P++; /* no NEXT; */
  }
  cmd = 0;
  what = C_text;
  put_command(cmd,what,&char_arr);
  DONE(ptr,P);
  return 1;

#undef ADD_CHR 
#undef CLEAR
#undef NEXT
#undef DONE
}

static String unparsed = { 0, 0, NULL};
static String unparsed_mapped = { 0, 0, NULL};
int personality_8bit = 0;

void parse_input(CHAR *buffer, int len) {
  String base;
  String mapped;
  String base_mapped;
  String ptr;
  int reslen;

  StrInit(&base,len);
  StrInit(&mapped,len);
  StrInit(&base_mapped,len);
  StrInit(&ptr,len);

  if (NULL != StrPTR(unparsed)) {
    StrCat(&base,&unparsed);
    StrFree(&unparsed);
  }
  { String S;
    S.buffer=buffer;
    S.len=len;
    S.alloced = 0;
    StrCat(&base,&S);
  }

  if (DEBUG_TERM)
    {
      CHAR *T = NULL;
      char2names(&base,&T);
      print_debug("parse_input: input: %s",T);
      FREE(T); T = NULL;
    }

  reslen = map7to8bit(&base,&mapped,&personality_8bit);
  if (StrLEN(base) > reslen) {
    String S;
    S.buffer = StrPTR(base)+reslen;
    S.len = StrLEN(base)-reslen;
    S.alloced=0;
    StrCat(&unparsed,&S);
  }

  if (NULL != StrPTR(unparsed_mapped)) {
    StrCat(&base_mapped,&unparsed_mapped);
    StrFree(&unparsed_mapped);
  }
  StrCat(&base_mapped,&mapped);

  StrCat(&ptr,&base_mapped);

  if (DEBUG_TERM)
    {
      CHAR *T = NULL;
      char2names(&base_mapped,&T);
      print_debug("parse_input: Mapped: %s",T);
      FREE(T); T = NULL;
    }


  while (StrLEN(ptr) && eat_one(&ptr));
  if (StrLEN(ptr)) {
    StrCat(&unparsed_mapped,&ptr);
  }
  StrFree(&base);
  StrFree(&base_mapped);
  StrFree(&mapped);
  StrFree(&ptr);
}

CHAR *unparsed_printable(void) {
  CHAR *result = NULL;
  CHAR *p1=NULL,*p2=NULL;
  if (!StrLEN(unparsed) && !StrLEN(unparsed_mapped)) return NULL;
  if (StrLEN(unparsed_mapped)) {
    char2names(&unparsed_mapped,&p1);
    result = concat_text(result,p1);
    FREE(p1); p1 = NULL;
  }
  if (StrLEN(unparsed)) {
    if (result) result = concat_text(result,rCs(" "));
    char2names(&unparsed,&p2);
    result = concat_text(result,p2);
    FREE(p2); p2 = NULL;
  }
  return result;
}
