/*  file: edit.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stdio.h>
#include <stdlib.h>

#include "kehpager.h"
#include "memory.h"
#include "charset.h"

#include "edit.h"
#include "keyboard.h"
#include "control.h"
#include "window.h"
#include "terminal.h"


static int NULL_HANDLER(int *pos,int len,CHAR_IDX *test) {
  reset_terminal_state();
  print_notify("NULL_HANDLER called: FATAL error "
	       "- aborting (will produce core-file)...");
  close_files();
  close_terminal();
  abort();
}

static int prev_group = -1;
static int edit_group = -1;
static int edit_Y     = -1;
static int edit_max   = -1;
static int edit_prev_Y = -1;
static edit_ready *edit_handler = NULL_HANDLER;

#define MAX_EDIT 1000
static int edit_value_len = 0;
static CHAR_IDX edit_value[MAX_EDIT];
static int edit_message_len = 0;
static CHAR_IDX edit_message[MAX_EDIT];

static struct {
  int * insert_keys[10];
  int * ready_keys[10];
} action_table[GROUP_count] = {
  /* G_PAGER */
  { { NULL }, { NULL } },
  /* G_MESSAGE */
  { { NULL }, { NULL } },
  /* G_MESSAGE_LAST */
  { { NULL }, { NULL } },
  /* G_GOTO */
  { { &key_goto, NULL }, 
      { &key_genter, &key_gpercnt, NULL } },
  /* G_SEARCH */
  { { &key_sinsert, NULL }, 
      { &key_senter, &key_ssearchn, &key_ssearchf, NULL } },
  /* G_CHARSET */
  { { &key_cinsert, NULL }, 
      { &key_center, NULL } },
  /* G_NEWLINE */
  { { NULL },
      { &key_nl_enter, &key_nl_unix, &key_nl_network, NULL } },
  /* G_CONTENT */
  { { &key_ct_insert, NULL },
      { &key_ct_enter, NULL } },
  /* G_TABSTEP */
  { { &key_ts_insert, NULL },
      { &key_ts_enter, NULL } },
  /* G_INDENTSTEP */
  { { &key_is_insert, NULL },
      { &key_is_enter, NULL } },
};

int edit_pos(int Y, int *group) { /* return state if editing is going */
  int ln;

  if (-1 == edit_group) return 0;
  
  *group = edit_group;
  edit_Y = Y;
  ln = window_rlines(WIN_EDIT);

  if (edit_Y + ln > lines+1) {
    edit_Y = lines - ln +1;
    if (edit_Y < 1) edit_Y = 1;
  }

  if (edit_prev_Y != edit_Y) {
    map_window(WIN_EDIT,0,edit_Y);
    set_window_status(WIN_EDIT,0,1,1,8,4,PRI_EDIT,0);
    edit_prev_Y = edit_Y;
  }

  set_win_prompt(WIN_EDIT,edit_message_len+edit_value_len+1,1);  
  return 1;
}

static void clear_edit(void) {
  clear_window_line(WIN_EDIT,1);
  unmap_window(WIN_EDIT);
  edit_group = -1;
  edit_prev_Y = -1;
  edit_value_len = 0;
}

static void update_line(void) {
  print_to_window(WIN_EDIT,1,1,FL_BOLD,edit_message_len,edit_message);
  set_win_prompt(WIN_EDIT,edit_message_len+edit_value_len+1,1);  
}

int edit_action(int *pos, CHAR_IDX c, int *group) {
  int have_insert = 0;
  int have_ready = 0;
  int i;

  if (-1 == edit_group) return 0;

  for (i = 0; NULL != action_table[edit_group].insert_keys[i]; i++)
    if (pos == action_table[edit_group].insert_keys[i]) {
      have_insert = 1;
      break;
    }

  for (i = 0; NULL != action_table[edit_group].ready_keys[i]; i++)
    if (pos == action_table[edit_group].ready_keys[i]) {
      have_ready = 1;
      break;
    }

  if (&key_eerase == pos) {
    clear_edit();
    *group = prev_group;
    return 1;
  } else if (&key_edel == pos) {
    if (edit_value_len > 0)
      edit_value_len--;
    if (0 == edit_value_len) {
      clear_edit();
      *group = prev_group;
      return 1;      
    }
    set_win_prompt(WIN_EDIT,edit_message_len+edit_value_len+1,1);
    cut_window_line(WIN_EDIT,edit_message_len+edit_value_len+1,1);
    flush_window(WIN_EDIT);
    return 1;
  } 

  if (!have_insert && !have_ready) return 0;

  if (have_insert && edit_value_len < edit_max) {
    edit_value[edit_value_len++] = c;
    print_to_window(WIN_EDIT,edit_message_len+edit_value_len,1,
		    FL_UNDER|FL_BOLD,1,&c);
    set_win_prompt(WIN_EDIT,edit_message_len+edit_value_len+1,1);  
    flush_window(WIN_EDIT);
  }

  if (have_ready) {
    int ok = (*edit_handler)(pos,edit_value_len,edit_value);
    if (ok) {      
      clear_edit();
      *group = prev_group;
      return 1;      
    }
  }
  return 1;
}

void edit_query(CHAR *msg, int newgroup, int max_len,
		edit_ready *handler,int *group ) {
  int len = strlen(Cs(msg));
  if (-1 != edit_group) {
    clear_edit();
    *group = prev_group;
  }
  map_input(MAP_ASCII,len,msg,edit_message);
  edit_message_len = len;
  set_line_status(WIN_EDIT,1,-len,0,0,0,0,0,0);

  prev_group = *group;
  edit_group = newgroup;
  *group = newgroup;
  edit_value_len = 0;
  edit_handler = handler;
  edit_max = max_len;
  if (edit_max > MAX_EDIT)
    edit_max = MAX_EDIT;
  update_line();
}

void edit_cancel(int *group) {
  if (-1 == edit_group) return;

  clear_edit();
  *group = prev_group;
}

