/*  file: control.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <string.h>
#include <stdio.h>
#include <stdarg.h>


#include "kehpager.h"
#include "memory.h"
#include "charset.h"

#include "control.h"
#include "terminal.h"
#include "esc.h"
#include "rc.h"
#include "env.h"

volatile int need_redraw = 0;
volatile int pager_state = 0;
volatile int down_line = 1; /* !! */
static int vt_type = -1;
static terminal_type_pass = 0;
/* 1 = asking terminal type
   2 = trying multinational
   3 = ask attributes
 */

#define READY_PASS (4)

#define SET_PASS(a) { terminal_type_pass = (a); \
     if (DEBUG_TERM) print_debug("Terminal pass: %d",terminal_type_pass); }

static int in_panic = 0;
#define PANIC(text) if (!in_panic) { \
     in_panic = 1; reset_terminal_state(); \
     print_notify("(%s) " #text,prog);  \
     close_files(); close_terminal(); exit(1); }

static int wait_count = 0;

#define ASKED { wait_count++; \
     if (DEBUG_TERM) print_debug("wait_count: %d",wait_count); }
#define GOT   { if (DEBUG_TERM) print_debug("wait_count: %d",wait_count); \
     if (wait_count && !--wait_count) pass_4(); }

static int saved_personality = -1;
static int saved_multinational = -1;
static int saved_autowrap = -1;
static int saved_application = -1;
static int saved_language = -1;
static int saved_legend = -1;    /* 1 = data processing keys */
static int saved_newline = -1;

static int default_personality = 0;
static int default_multinational = -1;
static int default_autowrap = 1;
static int default_application = 0;
int default_language = -1;
static int default_legend = 0;    /* 1 = data processing keys */
static int default_newline = 0;

static int default_latin1 = 0;

static int set_vt100_is_G2 = 0;
static int set_vt100_is_ncr = 0;

int set_xterm_latin1 = 1;
int query_terminal_size = 1;
int limit_query = 0;
int force_mode = 0;          /* 1 = National
			      * 2 = Multinational 
			      */

int timer_per_char = 1;
static int timer_id_1 = 50;
static int timer_id_2 = 50;
static int timer_states_1 = 30;
static int timer_states_2 = 10;
static int use_technical = 1;

static int n_SUP = 0;
static int n_DISAB = 1;

init_item control_items[] = {
  { &n_SUP,                 "force.feature.enabled", V_ATTR },
  { &n_DISAB,               "force.feature.disabled", V_ATTR },
  { &default_personality,   "exit.personality_8bit", V_BOOL },
  { &default_multinational, "exit.multinational",    V_BOOL },
  { &default_autowrap,      "exit.autowrap",         V_BOOL },
  { &default_application,   "exit.application_cursor_keys", V_BOOL },
  { &default_legend,        "exit.data_processing_keys", V_BOOL },
  { &default_latin1,        "exit.vt300.8bit_latin1",V_BOOL },
  { &default_newline,       "exit.newline",          V_BOOL },
  { &default_language,      "use.keyboard_language", V_NUM },
  { &set_vt100_is_G2,       "use.vt100.G2_available", V_BOOL },
  { &set_vt100_is_ncr,      "use.vt100.national_available", V_BOOL },
  { &use_technical,         "use.technical",         V_BOOL },
  { &limit_query,           "use.limited.queries",   V_BOOL },
  { &query_terminal_size,   "use.query_size",        V_BOOL },
  { &set_xterm_latin1,      "force.xterm.latin1",    V_BOOL },
  { &force_mode,            "force.mode",            V_MODE },
  { &timer_per_char,        "timer.per_char",        V_TIMER },
  { &timer_id_1,            "timer.identification",  V_TIMER },
  { &timer_id_2,            "timer.test_8bit",       V_TIMER },
  { &timer_states_1,        "timer.states_first",    V_TIMER },
  { &timer_states_2,        "timer.states_interval", V_TIMER },
  { NULL, NULL, V_LAST }
};

static int starting_line = -1;
static int starting_col = -1;
static int pos_count = 0;  

/*
static int saved_undef(int value,int def) {
 if (-1 == value) value = def;
 return value;
}
*/

static int saved(int value,int def) {
  if (-1 == value) value = def;
  if (-1 == value) value = 0;
  return value;
}

static int neg_undef(int value) {
  if (-1 == value) return -1;
  return !value;
}

static int have_8bit_clean = 0; 
static int use_national = -1; 
static int keyboard_language = -1;
              /* 1 North American
	       * 6 Finnish
	       */

static int xterm_latin1_magic = -1;

static int GL_set = -1;
static int GR_set = -1;
static int FLAGS_set = -1;

static long PRIM_attr = 0;
static long OPT_attr = 0;
static long SUP_attr = 0;
static long DISAB_attr = 0;

static long *attrs[] = { &SUP_attr, &DISAB_attr };


#define HAVE(x) ((x) == ((x) & (SUP_attr | PRIM_attr | OPT_attr) \
			 & ~DISAB_attr))
#define SUPPOSE(x)  (SUP_attr |= (x))

#define ATR_insdel       (1L<<0)  /* Insert/delete characters/lines */
/* #define ATR_xxx       (1L<<1)  */
#define ATR_cs_latin1    (1L<<2)  /* (14) Have Latin/1 and possible some other
				  *      Latin charset 
				  */
#define ATR_cs_technical  (1L<<3)  /* (15) Have technical character set */
#define ATR_cs_ncr       (1L<<4)  /* (9) Have national charsets */
#define ATR_soft_char    (1L<<5)  /* (7) Have shoft characters */
#define ATR_132          (1L<<6)  /* (1) Have 132 columns */
#define ATR_sel_erase    (1L<<7)  /* (6) Have selective erase */
#define ATR_def_key      (1L<<8)  /* (8) Have user defined keys */
/* #define ATR_windows   (1L<<9)   (18) Have user windows */
/* #define ATR_2_sess    (1L<<10)  (19) Have two sessions */
#define ATR_hor_scroll   (1L<<11) /* (21) Have horizontal scrolling */
#define ATR_ReGIS        (1L<<12) /* (3) Have ReGIS graphics */
#define ATR_SIXEL        (1L<<13) /* (4) Have SIXEL graphics */
/* #define ATR_status_d  (1L<<14)  (11) Status display */
#define ATR_printer      (1L<<15) /* (2) Printer port */
#define ATR_cs_multinational (1L<<16) /* Have multinational */
#define ATR_G2           (1L<<17) /*  Have G2 & G3 available */
#define ATR_vt100_magic  (1L<<18) /* Unnamed negative feature */
#define ATR_C1_controls  (1L<<19) /* Have C1 controls */
#define ATR_terminal_rq  (1L<<20) /* Terminal interrogation commands
				   * DECRQM, DECRQSS, ...
				   */

static struct features {
  long flag;
  char *name;
} ftrtable[] = {
  { ATR_vt100_magic, "(vt100 magic)" },
  { ATR_insdel,      "Ins/Del characters/lines" },
  { ATR_G2,          "Banks G2 & G3 (changeable)" },
  { ATR_cs_latin1,   "Latin/1 charset" },
  { ATR_cs_multinational, "Multinational charset" },
  { ATR_cs_technical, "Technical charset" },
  { ATR_C1_controls, "C1 control characters" },
  { ATR_cs_ncr,      "National charset" },
  { ATR_soft_char,   "Shoft characters" },
  { ATR_132,         "132 columns" },
  { ATR_sel_erase,   "Selective erase" },
  { ATR_def_key,     "User defined keys" },
  { ATR_hor_scroll,  "Horizontal scrolling" },
  { ATR_ReGIS,       "ReGIS graphic" },
  { ATR_SIXEL,       "SIXEL graphic" },
  { ATR_printer,     "Printer port" },
  { ATR_terminal_rq, "Terminal interrogation commands" },
  { -1, NULL }
};

static long flag_name (CHAR *s) {
  int i;
  for (i = 0; -1 != ftrtable[i].flag; i++)
    if (0 == strcasecmp(ftrtable[i].name,Cs(s))) return ftrtable[i].flag;
  return -1;
}

int map_attribute(CHAR *value,int *ptr)  {
  long flag;
  long *fp = attrs[*ptr];
  if (0 == strcasecmp("None",Cs(value)))
    (*fp) = 0L;
  else {
    flag  = flag_name(value);
    if (-1 == flag) return -1;
    (*fp) |= flag;
  }
  return *ptr;
}

static void do_magic() {
  if (HAVE(ATR_vt100_magic)) {
    if (set_vt100_is_G2) SUPPOSE(ATR_G2);
    if (set_vt100_is_ncr) SUPPOSE(ATR_cs_ncr);
  }
}

static CHAR * ftr_string(long features) {
  CHAR *ptr = NULL;
  int i;
  for (i = 0; -1 != ftrtable[i].flag; i++) 
    if (ftrtable[i].flag & features) {
      if (ptr) ptr = concat_text(ptr,rCs(", "));
      ptr = concat_text(ptr,rCs(ftrtable[i].name));
    }
  if (!ptr) 
    ptr = concat_text(ptr,rCs("None"));
  return ptr; 
}

typedef struct ftr {
  int basic;
  int pos;
  int value;
  long flags;
} ftrs;

static ftrs DA_table[] = {
  { -1, -1,  1,  ATR_132 },
  { -1, -1,  2,  ATR_printer },
  { -1, -1,  3,  ATR_ReGIS },
  { -1, -1,  4,  ATR_SIXEL },
  { -1, -1,  5,  0          /* AVO option (132 char wide) NOT  installed */ },
  { -1, -1,  6,  ATR_sel_erase },
  { -1, -1,  7,  ATR_soft_char },
  { -1, -1,  8,  ATR_def_key },
  { -1, -1,  9,  ATR_cs_ncr },
  { -1, -1, 10,  0              /* Text ruling  vector */ },
  { -1, -1, 11,  0              /* 25th Status line */ },
  { -1, -1, 12,  0              /* Terminal is a VT125 */ },
  { -1, -1, 13,  0              /* Local editing mode */ },
  { 63, -1, 14,  ATR_cs_latin1  /* 8-bit characters */ }, 
  { 62, -1, -1,  ATR_cs_multinational }, 
  { -1, -1, 15,  ATR_cs_technical /* Technical character set */ },
  { -1, -1, 16,  0              /* Locator device port (ReGIS) [=mouse?] */ },
  { -1, -1, 17,  0              /* Terminal state reports */ },
  { -1, -1, 18,  0              /* Windowing capability */ },
  { -1, -1, 19,  0              /* Dual sessions */ },
  { -1, -1, 21,  ATR_hor_scroll /* Horizontal scrolling */ },
  { -1, -1, 29,  0              /* ANSI text locator [=mouse?] */ },

  { 61, -1, -1,  ATR_insdel },
  { 61, -1, -1,  ATR_G2 },
  { 62, -1, -1,  ATR_C1_controls },
  { 63, -1, -1,  ATR_terminal_rq },
  { -1, -1, -1, -1 }
};

static ftrs vt100_table[] = {
  { 0, -1,  ATR_vt100_magic },
  { -1, 1, 0,  0                     /* NO options */ },
  { -1, 1, 1,  0                     /* Processor Option (STP)  */ },
  { -1, 1, 2,  ATR_132 | ATR_insdel  /* Advanced Video Option (AVO) */ },
  { -1, 1, 3,  ATR_132 | ATR_insdel  /* AVO and STP */ },
  { -1, 1, 4,  0                     /* Graphics Option (GPO) */ },
  { -1, 1, 5,  0                     /* GPO and STP */ },
  { -1, 1, 6,  ATR_132 | ATR_insdel  /* GPO and AVO */ },
  { -1, 1, 7,  ATR_132 | ATR_insdel  /* GPO, AVO, and STP */ },
  /* I don't know if these are correct - don't seem to formed from bits: */
  { -1, 1, 9,  ATR_printer | ATR_132 | ATR_insdel  /* PP and AVO */ },
  { -1, 1, 11, ATR_printer | ATR_132 | ATR_insdel  /* PP, GPO and AVO */ },
  /* VT125 uses: */
  { -1, 2, 0,  0                     /* No printer present */ },
  { -1, 2, 1,  ATR_printer           /* printer present */ },
  { -1, -1, -1, -1 }				     
};

static ftrs LPC02_table[] = {
  { -1, -1, 1,  ATR_ReGIS      /* ReGIS graphics software loaded */ },
  { -1, -1, 2,  ATR_SIXEL      /* Color Sixel graphics software loaded */ },
  { -1, -1, 3,  0              /* NAPLPS ???? */ },
  { -1, -1, 4,  0              /* GIDIS */ },

  { -1, -1, -1, -1 }				     
};

static ftrs basic_table[] = {
  { 0, -1, -1,   ATR_vt100_magic },
  { 0, -1, -1,   ATR_insdel },
  { -1, -1,  4,  ATR_SIXEL },
  { -1, -1,  3,  ATR_ReGIS },
  { -1, -1,  2,  ATR_printer },
  { -1, -1,  1,  ATR_132 },
  { -1, -1, -1, -1 }				     
};

static struct lvl {
  int type;
  ftrs *table;
  char *name;
} levels[] = {
  {  1,  vt100_table, "VT100" },
  {  2,  vt100_table, "VT102" },
  {  3,  vt100_table,  "DECwrite IV" },
  {  4,  vt100_table, "VT132" },
  {  5,  vt100_table, "VK100" },
  {  6,  basic_table, "VT102" },
  {  7,  basic_table, "VT131" },
  {  12, vt100_table, "VT125" },
  {  28, LPC02_table, "LPC01 (Unsupported)" },
  {  37, LPC02_table, "LG02  (Unsupported)" },
  {  61, DA_table,    "Level 1 (VT100 series)" },
  {  62, DA_table,    "Level 2 (VT200 series)" },
  {  63, DA_table,    "Level 3 (VT300 series)" },
  {  64, DA_table,    "Level 4 (VT400 series)" },
  /* These are unsupported, because basic features for 
   * these levels is not know 
   */
  {  65, DA_table,    "Level 5 (Unsupported)" },
  {  66, DA_table,    "Level 6 (Unsupported)" },
  {  67, DA_table,    "Level 7 (Unsupported)" },
  {  68, DA_table,    "Level 8 (Unsupported)" },
  {  69, DA_table,    "Level 9 (Unsupported)" },
  {  -1, NULL, NULL }
  };

static void parse_DEC_DA (CHAR * S) {
  char *level,*ptr;
  int L,i,pos;
  static ftrs *table;
  
  PRIM_attr = 0;
  OPT_attr = 0;
  if (NULL == (level = strtok(Cs(S),";")) || 
      0 == (vt_type = atoi(level))) {
    print_notify("Warning: Unsupported DA response");
    vt_type = -1;
    return;
  }
  L=-1;
  for (i = 0; -1 != levels[i].type; i++) 
    if (vt_type == levels[i].type) {
      L = i; 
      break;
    }
  if (-1 == L) {
    print_notify("Warning: Unknown terminal model (type/level field=%d)",
		 vt_type);
    return;
  }
  table = levels[L].table;
  /* Calculate primary features */
  for (i = 0; -1 != table[i].flags; i++)
    if (-1 != table[i].basic &&
	vt_type >= table[i].basic)
      PRIM_attr |= table[i].flags;
  if (DEBUG_TERM) {
    CHAR *P = ftr_string(PRIM_attr);
    if (strlen(Cs(P)) < ERROR_LEN -20)
      print_debug("Primary features: %s",P);
    else
      print_debug("Primary features list too long to print...");
    free(P);
  }
  /* Calculate optional features */
  pos=0;
  for (ptr = strtok(NULL,";"); NULL != ptr; ptr = strtok(NULL,";")) {
    int val = atoi(ptr);
    pos++;
    for (i = 0; -1 != table[i].flags; i++)
      if (-1 != table[i].value &&
	  val == table[i].value &&
	  (-1 == table[i].pos || pos == table[i].pos))
	OPT_attr |= table[i].flags;
  }  
  if (DEBUG_TERM) {
    CHAR *P = ftr_string(OPT_attr);
    if (strlen(Cs(P)) < ERROR_LEN -20)
      print_debug("Optional features: %s",P);
    else
      print_debug("Optional features list too long to print...");
    free(P);
  }
}

static CHAR * terminal_type(void) {
  CHAR *ptr=NULL;
  int i;
  if (-1 == vt_type) {
    ptr = concat_text(ptr,rCs("Undefined"));
  } else {
    CHAR T[10];
    sprintf(Cs(T),"(%d) ",vt_type);
    ptr = concat_text(ptr,T);
    for (i = 0; -1 != levels[i].type; i++) 
      if (vt_type == levels[i].type) 
	ptr = concat_text(ptr,rCs(levels[i].name));
  }
  return ptr;
}

long hazards = 0L;

static struct Hazard {
  long value;
  char *text;
} hazard_table[] = {
  { HAZ_wrap,     "Terminal may wrap in right" },
  { HAZ_national, "Terminal is always in national mode" },
  { HAZ_ONLCR,    "Tty driver may wrap LF -> CR LF on output" },
  { HAZ_TAB,      "Tty driver may expand TAB -> SPCs on output" },
  { HAZ_newline,  "Terminal is always in newline mode (LF produces new line, "
      " RETURN produces CR LF)" },
  { HAZ_application, "Cursor keys are always in application mode" },

  { -1, NULL }
};

static CHAR * terminal_hazards(void) {
  CHAR *ptr=NULL;
  int i;
  for (i = 0; -1 != hazard_table[i].value; i++) 
    if (HAZ_MAY(hazard_table[i].value)) {
      if (ptr) ptr = concat_text(ptr,rCs(", "));
      ptr = concat_text(ptr,rCs(hazard_table[i].text));
    }
  if (!ptr) ptr = concat_text(ptr,rCs("None"));
  return ptr;
}

struct termitem kehpager_term[] = {
  { "type",                 &vt_type },
  { "s.personality",        &saved_personality },
  { "s.multinational",      &saved_multinational },
  { "d.multinational",      &default_multinational },
  { "s.application",        &saved_application },
  { "s.language",           &saved_language },
  { "s.legend",             &saved_legend },
  { "s.autowrap",           &saved_autowrap },
  { "u.8bit_clean",         &have_8bit_clean },
  { NULL, NULL }
};

struct termitem2 kehpager_term2[] = {
  { "attr.prim",            &PRIM_attr },
  { "attr.supp",            &SUP_attr }, 
  { "attr.opt",             &OPT_attr },
  { "attr.hazards",         &hazards },
  { NULL, NULL }
};

#define A_ASCII 1
#define A_8bit  2
#define A_8bit_n (A_ASCII | A_8bit)
#define A_96set 4

struct one_set {
  int language;
  char *set;
  char *name;
  long flags;
  int use_latin;
  int area;
  int charset;
} sets[] = {

  /* National character sets */
  { 1, "B", "ASCII",           0  , -1 ,        A_ASCII,  MAP_ASCII }, /* 0 */
  { 2, "A", "UK-ASCII",        0  , -1 ,        0,  MAP_UK },
  { 4, "9", "French Canadian", ATR_cs_ncr, -1 , 0,  MAP_CAN },
  { 6, "5", "Finnish",         ATR_cs_ncr, -1 , 0,  MAP_FINNISH },
  { 7, "K", "German",          ATR_cs_ncr, -1 , 0,  MAP_GER },
  { 9, "Y", "Italian",         ATR_cs_ncr, -1 , 0,  MAP_ITAL },
  { 5, "E", "Danish",          ATR_cs_ncr, -1 , 0,  MAP_NOR },
  { 13,"E", "Norwegian",       ATR_cs_ncr, -1 , 0,  MAP_NOR },
  { 12,"7", "Swedish",         ATR_cs_ncr, -1 , 0,  MAP_SWE },
  { 3, "R", "Flemish",         ATR_cs_ncr, -1 , 0,  MAP_BEL },
  { 14,"R", "French/Belgian",  ATR_cs_ncr, -1 , 0,  MAP_BEL },
  { 15,"Z", "Spanish",         ATR_cs_ncr, -1 , 0,  MAP_SPA },

  { -1, ">", "Technical",      ATR_cs_technical, -1, 0, MAP_TECHNICAL },
  { -1, "<", "Multinational" , ATR_cs_multinational  
                                          , 0 , A_8bit_n, MAP_MULTINATIONAL },
  { -1, "%5", "Multinational", ATR_cs_multinational | ATR_cs_latin1
                                          , -1, A_8bit_n, MAP_MULTINATIONAL },

  { -1, "<", "Latin/1",        ATR_cs_latin1 , 1 , A_8bit_n,   MAP_LATIN1 }, /* Must be last -1 */
  { -1, "0", "Special",        0  , -1 , 0,  MAP_SPECIAL }   /* Must be last */

};

#define SET_NUM (sizeof (sets) / sizeof (struct one_set))

int find_set_by_name (CHAR *name, int is_8bit) {
  int i,result=-1;
  for (i = 0; i < SET_NUM; i++) {
    if (-1 != result && 0 == strcmp(sets[i].name,sets[result].name) &&
	HAVE(sets[i].flags)) result = -1;
    if (0 == strcasecmp(sets[i].name,Cs(name)) &&
	HAVE(sets[i].flags) &&
	(0 == (sets[i].area & A_8bit) || is_8bit)) result = i;
  }
  return result;
}

int find_set_by_language (int language, int is_8bit) {
  int i,result=-1;
  for (i = 0; i < SET_NUM; i++) {
    if (-1 != result && 0 == strcmp(sets[i].name,sets[result].name) &&
	HAVE(sets[i].flags)) result = -1;
    if (sets[i].language == language &&
	HAVE(sets[i].flags) &&
	(0 == (sets[i].area & A_8bit) || is_8bit)) result = i;
  }
  return result;
}

typedef pattern[4];

/* G0 -> national  |  national | ascii         | ascii
 * G1 -> special   |  ascii    | special       | special
 * G2 -> ascii     |           | multinational | latin/1
 * G3 -> uk-ascii  |           | latin/1       |
 */

static pattern default_sets = { -1, (SET_NUM -1), -1, 0 };
static pattern used_sets = { -1, (SET_NUM -1), -1, -1 };

static void set_mode(char *frm, ...) {
  va_list args;
  CHAR buffer[MAX_OUTPUT+1],*ptr,*ptr2=NULL;
  CHAR sbuffer[MAX_OUTPUT+1];

  va_start(args,frm);
  vsprintf(Cs(buffer),frm,args);
  va_end(args);

  ptr = map_buffer(buffer,!personality_8bit,sbuffer,MAX_OUTPUT);
  print_to_terminal(ptr);

  if (DEBUG_TERM) {
    String S;
    S.buffer = ptr;
    S.len = strlen(Cs(ptr));
    S.alloced=0;
    char2names(&S,&ptr2);
    print_debug("To terminal: %s",ptr2);
    FREE(ptr2);
  }

  if (ptr != sbuffer) FREE(ptr);

}

static void clear_attr_conflict(int *flag) {

   if (((*flag) & (FL_BOLD | FL_DIM)) == (FL_BOLD | FL_DIM))
     (*flag) &= ~(FL_BOLD | FL_DIM);

}

int sum_fl_attr(int flags, int new) {
  flags |= new;
  clear_attr_conflict(&flags);
  flags |= new;
  return flags;
}

static void set_attr(int flags) {
   CHAR buffer[32];
   char tmp[4];

   buffer[0] = '\0';

#define ADD_VAL(p) { sprintf(tmp,"%d",p); \
                     if ('\0' != buffer[0]) strcat(Cs(buffer),";"); \
		     strcat(Cs(buffer),tmp); }

  clear_attr_conflict(&flags);

   if (0 == flags && 0 != FLAGS_set) {
     set_mode("%cm",CSI);
     FLAGS_set = 0;
     return;
   }

   if (-1 == FLAGS_set) { ADD_VAL(0); FLAGS_set = 0; }
   if (FLAGS_set & ~flags) {ADD_VAL(0); FLAGS_set = 0; }

   if (FL_BOLD & flags & ~FLAGS_set) { ADD_VAL(1); FLAGS_set |= FL_BOLD; }
   if (FL_DIM & flags & ~FLAGS_set) { ADD_VAL(2); FLAGS_set |= FL_DIM; }
   if (FL_UNDER & flags & ~FLAGS_set) { ADD_VAL(4); FLAGS_set |= FL_UNDER; }
   if (FL_REVERSE & flags & ~FLAGS_set) { ADD_VAL(7); FLAGS_set |= FL_REVERSE; }
   if (FL_BLINK & flags & ~FLAGS_set) { ADD_VAL(5); FLAGS_set |= FL_BLINK; }

   if ('\0' == buffer[0]) return;

   set_mode("%c%sm",CSI,buffer);

#undef ADD_VAL
}


static int set_banks(int GL, int GR) {
  int flag = 1;
  if (-1 != GL && GL != GL_set) switch(GL) {
  case 0: set_mode("%c",SI); GL_set = 0; break;
  case 1: set_mode("%c",SO); GL_set = 1; break;
  case 2: 
    if (HAVE(ATR_cs_latin1) || HAVE(ATR_cs_multinational)) {
      set_mode("%cn",ESC); GL_set = 2; }
    else { set_mode("%cN",ESC); flag = 0; } break;
  case 3: 
    if (HAVE(ATR_cs_latin1) || HAVE(ATR_cs_multinational)) {
      set_mode("%co",ESC); GL_set = 3; }
    else { set_mode("%cO",ESC); flag = 0; } break;
  }

  if (-1 != GR && GR != GR_set) switch(GR) {
  case 0:
    PANIC("Software failure: trying map GR to G0");
    break;
  case 1: set_mode("%c~",ESC); GR_set = 1; break;
  case 2: set_mode("%c}",ESC); GR_set = 2; break;
  case 3: set_mode("%c|",ESC); GR_set = 3; break;
  }
  return flag;
}

static int X_set = -1, Y_set = -1;
static int X_hard = -1, Y_hard = -1;

static void print_viaset(int set, CHAR *buffer) {
  int i, bank = -1, left,len = strlen(Cs(buffer));
  int lo = 0, hi = 0, GL,GR,repeat,maxx=columns;
  CHAR *c;

  if (HAZ_MAY(HAZ_wrap) && Y_set == lines) maxx = columns-1;

  if (X_set > maxx) X_set = -1;
  if (-1 == X_set) return;
  left = maxx - X_set +1;

  for (i = 0; i < 4; i++) if (used_sets[i]==set) { bank = i; break; }
  if (-1 == bank) {
    PANIC("Charset not in any bank!");
    bank = 0;
  }
  for (c= buffer; *c; c++)
    if (*c & 128) hi = 1;
    else lo = 1;
  GL = lo ? bank : -1;
  GR = hi ? bank : -1;
  repeat = !set_banks(GL,GR);
  if (repeat) for (c=buffer;*c && left ;c++,set_banks(GL,GR)) {
    CHAR tbuffer[2];
    tbuffer[0] = *c; tbuffer[1] = 0;
    print_to_terminal(tbuffer);
    if (DEBUG_TERM) {
      print_debug("Text to terminal: %s",tbuffer);
    }
    X_set++; left--;
  } else if (left < len) {
    CHAR sbuffer[MAX_OUTPUT+1];
    CHAR *tbuffer = left < MAX_OUTPUT ? sbuffer : MALLOC(left+1);

    memcpy((void *)tbuffer,(void *)buffer,left);
    tbuffer[left] = '\0';
    print_to_terminal(tbuffer);
    
    if (tbuffer != sbuffer) FREE(tbuffer);

    if (DEBUG_TERM) {
      print_debug("Text to terminal: %s",tbuffer);
    }
    X_set += left;
  } else {
    print_to_terminal(buffer);
    if (DEBUG_TERM) {
      print_debug("Text to terminal: %s",buffer);
    }
    X_set += len;
  }
  if (HAZ_MAY(HAZ_wrap) && X_set == columns) {
    X_set = -1;
    Y_set = -1;
  } else if (X_set > columns) X_set = -1;
}

static void change_banks(pattern char_sets) {
  int i;
  /* Set char sets */

  for (i = 0; i < 4; i++) if (-1 != char_sets[i]) {
    if (sets[char_sets[i]].area & A_96set) {
      if (i == 0) 
	PANIC("Can't assign 96 char sets to bank 0 !");
      set_mode("%c%c%s",ESC,"\0-./"[i],sets[char_sets[i]].set);
    } else
      set_mode("%c%c%s",ESC,"()*+"[i],sets[char_sets[i]].set);
  }
}

static void change_state(int setup,
		    int personality,
		    int multinational,
		    int autowrap,
		    int application,
		    pattern char_sets,
		    int legend,
		    int new_line) {
  int use_latin1 = -1, i;
  static int latin1_prev = -1;

  for (i = 0; i < 4; i++) if (-1 != char_sets[i])
    if (-1 != sets[char_sets[i]].use_latin)
      use_latin1 = sets[char_sets[i]].use_latin;

  set_mode("%c%c%c!p",CAN,ST,CSI); /* shoft reset */

  if (HAVE(ATR_C1_controls)) {
    set_mode("%c[%d;%d\"p",ESC,vt_type,personality ? 2 : 1);

    if (-1 != multinational) {
      set_mode ("%c?42%c",CSI,multinational ? 'l' : 'h');
      personality_8bit = personality && multinational;      
    } else
      personality_8bit = 0;

  } else {
    if (-1 != multinational)
      set_mode ("%c?42%c",CSI,multinational ? 'l' : 'h');
    personality_8bit = 0;
  }

  if (-1 != autowrap)
    set_mode ("%c?7%c",CSI,autowrap ? 'h' : 'l');
  if (-1 != application)
    set_mode ("%c?1%c",CSI,application ? 'h' : 'l' );
  set_mode ("%c?68%c",CSI,legend ? 'h' : 'l' );

  /* Insert mode off, New line mode off,
     Control excution on
   */
  set_mode ("%c4;13l",CSI);
  /* Local echo off */
  set_mode ("%c12h",CSI);

  if (-1 != new_line)
    set_mode("%c20%c",CSI,new_line ? 'h' : 'l');

  /* Set Latin/1 mode */
  if (-1 != use_latin1 && (latin1_prev != use_latin1 || setup)) {
    set_mode("%c%s%c",DCS,use_latin1 ? "1!uA" : "0!u%5" , ST);
    latin1_prev = use_latin1; /* for avoid garbage when suspending */
  } else if (-1 == use_latin1 && -1 != default_latin1 &&
	     -1 != latin1_prev && latin1_prev != default_latin1) {
    set_mode("%c%s%c",DCS,default_latin1 ? "1!uA" : "0!u%5" , ST);
    latin1_prev = default_latin1;
  }

  change_banks(char_sets);
}

static int Y1_set = -1;
static int Y2_set = -1;
static int Y1_region = -1;
static int Y2_region = -1;

static void set_region(int Y1, int Y2) {
  if (-1 == Y1 || -1 == Y2) {
    Y1 = 1; Y2 = lines;
  }
  if (DEBUG_CURSOR) {
    print_debug("Region to (Y1,Y2) = (%d,%d) from (%d,%d)",
		Y1,Y2,Y1_set,Y2_set);
  }
  if (Y1 != Y1_set || Y2 != Y2_set) {
    set_mode("%c%d;%dr",CSI,Y1,Y2);
    Y1_set = Y1; Y2_set = Y2;
    Y_set = -1;
  }
}

static void update_region(void) {
  int Y1 = Y1_region, Y2=Y2_region;

  if (Y1 > lines) PANIC("Software failure/Region out of screen");
  if (Y2 > lines) Y2 = lines;

  if (-1 == Y1) {
    Y1 = -1; Y2  = -1;
  }
  set_region(Y1,Y2);
}

static void check_region(void) {
  if (-1 != Y1_region || -1 != Y2_region)
    PANIC("Software error/Regions not supported here!");
}

static int is_region(int line) {
  if (-1 == Y1_region || -1 == Y2_region) return 0;
  if (line < Y1_region || line > Y2_region)
    PANIC("Software error/Line out of region");
  return 1;
}

void set_margins(int Y1, int Y2) {
  if (Y1 > Y2) PANIC("Software failure/set_margins: Y1 > Y2");

  Y1_region = Y1; Y2_region = Y2;
}

static void set_down_line(int line) {
  if (DEBUG_CURSOR) {
    print_debug("Down line to %d from %d",line,down_line);
  }
  down_line = line;
}
  
static void goto_XY(int X, int Y) {

  if (-1 == X_set  && -1 == Y_set) {
    /* Origin mode off */
    set_mode ("%c?6l",CSI);
  }

  if (-1 == Y1_set || -1 == Y2_set) 
    set_region(-1,-1);

  if (Y < Y1_set || Y > Y2_set)
    PANIC("Software error/Going out of region");

  if (DEBUG_CURSOR) {
    print_debug("Cursor to (X,Y) = (%d,%d) from (%d,%d)",X,Y,X_set,Y_set);
  }

  if (X == X_set && Y == Y_set) return;
  if (Y >= down_line) set_down_line(Y+1);

  if (Y == Y_set && X == X_set-1) {
    set_mode("%c",BS);
    X_set--;
    return;
  }

  if (1 == Y && 1 == X) {
    set_mode("%cH",CSI);
    X_set = 1;
    Y_set = 1;
    return;
  }

  if (Y == Y_set+1 && X == X_set) {
    /* IND is two characters in 7-bit output !! */
    if (personality_8bit || HAZ_MAY(HAZ_ONLCR) ||
	HAZ_MAY(HAZ_newline)) set_mode("%c",IND);
    else set_mode("%c",LF);
    Y_set++;
    return;
  }

  if (Y == Y_set+1 && X == 1 && -1 != X_set) {
    set_mode("%c",NEL);
    Y_set++;
    X_set = 1;
    return;
  }

  if (Y == Y_set && -1 != X_set) {
    if (1 == X) set_mode("%c",CR);
    else if (X < X_set) set_mode("%c%dD",CSI,X_set-X);
    else set_mode("%c%dC",CSI,X-X_set);
    X_set = X; return;
  }
  if (X == X_set && -1 != Y_set && Y_set >= Y1_set && Y_set <= Y2_set) {
    if (Y < Y_set) set_mode("%c%dA",CSI,Y_set-Y);
    else set_mode("%c%dB",CSI,Y-Y_set);
    Y_set = Y; return;
  }
  set_mode("%c%d;%dH",CSI,Y,X); X_set = X; Y_set = Y;
}

static void update_hard(void) {
  set_region(-1,-1);
  if (-1 != X_hard && -1 != Y_hard &&
      X_hard <= columns && Y_hard <= lines) goto_XY(X_hard,Y_hard);
}

void reset_terminal_state(void) {
  pager_state = 0;
  change_state(0,
	       saved(saved_personality,default_personality),
	       saved(saved_multinational,default_multinational),
	       saved(saved_autowrap,default_autowrap),
	       saved(saved_application,default_application),
	       default_sets,
	       saved(saved_legend,default_legend),
	       saved(saved_newline,default_newline));
  set_banks(0,saved(saved_multinational,default_multinational) ? 2 : -1);
  set_attr(0);
  set_region(-1,-1);
  goto_XY(1,down_line <= lines ? down_line : lines);
  if (down_line > lines) set_mode("%c",LF);

  flush_error_buffer(); 
}

static int fast_start = 0;

static void pass_2(void);
void get_terminal_state(void) {
  set_attr(FL_BLINK);
  print_to_terminal(rCs("\rInitializing terminal: "));
  set_attr(0);
  fast_start = init_from_kehpager_term();
  xterm_latin1_magic = (set_xterm_latin1 && is_xterm()) || 
    2 == set_xterm_latin1;

  SET_PASS(1);

  if (!fast_start) {
    set_mode("%c%c%c[0c",CAN,ST,ESC); /* Ask terminal status report */
  } else {
    pass_2();
  }
}

static void pass_3(void);
static void pass_2(void) {
  int tmp = personality_8bit;
  SET_PASS(2);
  if (force_mode > 0) {
    /* force 7 or 8bit */
    switch(force_mode) {
    case 1: /* force National */
      have_8bit_clean = 0;
      if (!HAVE(ATR_cs_ncr)) SUPPOSE(ATR_cs_ncr);
      if (-1 == default_multinational) default_multinational = 0;
      break;
    case 2: /* force Multinational */
      have_8bit_clean = 1;
      if (!HAVE(ATR_cs_multinational)) SUPPOSE(ATR_cs_multinational);
      if (!HAVE(ATR_G2)) SUPPOSE(ATR_G2);
      if (-1 == default_multinational) default_multinational = 1;
      break;
    }
    pass_3();
  } else if (xterm_latin1_magic) {
    /* force 8bit */
    have_8bit_clean = 1;
    if (!HAVE(ATR_cs_latin1)) SUPPOSE(ATR_cs_latin1);
    if (!HAVE(ATR_G2)) SUPPOSE(ATR_G2);
    if (-1 == default_multinational) default_multinational = 1;
    pass_3();
  } else if (!fast_start) {
    personality_8bit = 1;

    set_mode("%c0c",CSI);

    personality_8bit = tmp;
  } else {
    pass_3();
  }
}

static CHAR *Boolean(int v) {
  char *S = "??"; switch(v) { 
  case -1: S = "Undefined"; break; 
  case 0:  S = "False"; break; 
  case 1:  S = "True"; break; } 
  return rCs(S);
}

static CHAR *Number(int v) {
  static int B=0;
  static CHAR buffer[5][10];
  if (-1 == v) return rCs("Undefined");
  B = (B+1)%5;
  sprintf(Cs(buffer[B]),"%d",v);
  return buffer[B];
}

void give_debug_page(void (*printer)(CHAR *line, int header)) {
  CHAR buffer[80];
#define PV(T,v) { if (-1 == v) sprintf(Cs(buffer),"%-30s: Undefined",T); \
		  else sprintf(Cs(buffer),"%-30s: %d",T,v); printer(buffer,0); }
#define PV1(T,v,v1) { sprintf(Cs(buffer),"%-30s: %-15s Default: %s",T,\
			      Number(v),Number(v1)); printer(buffer,0); }

#define PB(T,v) { sprintf(Cs(buffer),"%-30s: %s",T,Boolean(v)); printer(buffer,0); }
#define PB1(T,v,v1) { sprintf(Cs(buffer),"%-30s: %-15s Default: %s",T,\
			      Boolean(v),Boolean(v1)); printer(buffer,0); }
#define PF(T,v) { char *S = "??"; switch(v) { \
					    case -1: S = "Undefined"; break; \
					    default: S = sets[v].name; } \
		  sprintf(Cs(buffer),"%-30s: %-1s",T,S); printer(buffer,0); }
#define PMP(T,v) { CHAR *S = rCs("??"); switch(v) { \
					    case -1: S = rCs("Undefined"); break; \
					    default: S = map_name(v,0); } \
		  sprintf(Cs(buffer),"%-30s: %-1s",T,S); printer(buffer,0); }
#define MES(T,s) { CHAR *S = NULL; sprintf(Cs(buffer),"%-30s: ",T); \
		   S = concat_text(S,buffer); S = concat_text(S,s); \
                   printer(S,-32); FREE(S); FREE(s); }
  printer(rCs("Terminal state"),1); 
  MES("Terminal type",terminal_type());
  printer(rCs("Features"),0);
  MES("   - basic",ftr_string(PRIM_attr));
  MES("   - optional",ftr_string(OPT_attr));
  MES("   - supposed",ftr_string(SUP_attr));
  MES("   - disabled",ftr_string(DISAB_attr));
  MES("Terminal/driver hazards",terminal_hazards());

  PB("8 bit clean", have_8bit_clean);
  PB("Use national mode", use_national);
  PB1("8 bit personality", personality_8bit,default_personality);
  PB("XTerm mode", xterm_latin1_magic);
  PV1("Keyboard language", keyboard_language,default_language);
  PMP("Keyboard charset",keyboard_charset);
  PF("Bank G0 font", used_sets[0]);
  PF("Bank G1 font", used_sets[1]);
  PF("Bank G2 font", used_sets[2]);
  PF("Bank G3 font", used_sets[3]);
  PV1("Lines",lines,def_lines);
  PV1("Columns",columns,def_columns);
  printer(rCs("Saved (or exit) terminal state"),1);
  PB1("8 bit characters",     saved_multinational,default_multinational);  
  PB1("8 bit personality",    saved_personality,default_personality);
  PB1("Autowrap",             saved_autowrap,default_autowrap);
  PB1("App. cursor keys",     saved_application,default_application);
  PB1("Data processing keys", saved_legend,default_legend);
  PB1("Newline mode",         saved_newline,default_newline);
  PF("Bank G0 font", default_sets[0]);
  PF("Bank G1 font", default_sets[1]);
  PF("Bank G2 font", default_sets[2]);
  PF("Bank G3 font", default_sets[3]);

#undef MES
#undef PMP
#undef PV
#undef PB
#undef PB1
#undef PF
}

static int ASCII_set = 0; /* ASCII or it's superset */

static int find_ASCII(void) {
  int i;
  for (i = 0; i < 4; i++)
    if (-1 != used_sets[i] && (sets[used_sets[i]].area & A_ASCII))
      return used_sets[i];
  
  PANIC("Software failure/ASCII compatible font not found from any bank!");
  return 0;
}

int keyboard_charset = MAP_ASCII;

void set_terminal_state(void) {
  if (terminal_type_pass == READY_PASS) {
    int i;
    int LANG = saved(saved_language,default_language);
    use_national = -1;

    if (HAZ_MAY(HAZ_national)) {
      use_national = 1;
      have_8bit_clean = 0;
      default_multinational = 0;
    }
      
    /* DEFAULT */
    default_sets[0] = saved(saved_multinational,default_multinational)
      ? find_set_by_name(rCs("ASCII"),0)
	: find_set_by_language(LANG,
			       saved(saved_personality,default_personality));
    if (-1 == default_sets[0])
      default_sets[0] = find_set_by_name(rCs("ASCII"),0);
    
    default_sets[2] = -1;
    if (default_latin1 >1)
      default_sets[2] = find_set_by_name(rCs("Latin/1"),
					 saved(saved_multinational,
					       default_multinational));
    if (-1 == default_sets[2])
      default_sets[2] = find_set_by_name(rCs("Multinational"),
					 saved(saved_multinational,
					       default_multinational));
    if (-1 == default_sets[2])
      default_sets[2] = find_set_by_name(rCs("ASCII"),
					 saved(saved_multinational,
					       default_multinational));
    default_sets[3] = default_sets[2];

    /* USED */
    /* G0 or G2 indicates keyboard charset */
    if (!have_8bit_clean || 
	(LANG > 0 && !HAVE(ATR_cs_multinational) && !HAVE(ATR_cs_latin1)))
      used_sets[0] = find_set_by_language(LANG,0);
    else 
      used_sets[0] = -1;
    if (-1 == used_sets[0])
      used_sets[0] = find_set_by_name(rCs("ASCII"),have_8bit_clean);
    if (-1 != used_sets[0] && (sets[used_sets[0]].flags & ATR_cs_ncr)) {
      use_national = 1;
      /* When national mode some terminals haven't 8bit clean !! */
      have_8bit_clean = 0;
    }
    
    used_sets[1] = find_set_by_name(rCs("Special"),have_8bit_clean);
    if (!HAVE(ATR_G2)) {
      /* VT100 without G2 and G3 banks */
      used_sets[1] = find_set_by_name(rCs("ASCII"),have_8bit_clean);
      if (used_sets[0] == used_sets[1])
	used_sets[1] = find_set_by_name(rCs("Special"),have_8bit_clean);
    } else {
      /* set banks G2-G3 */
      used_sets[2] = find_set_by_name(rCs("ASCII"),have_8bit_clean);
      used_sets[3] = find_set_by_name(rCs("ASCII"),have_8bit_clean);
      /* xterm don't know Latin/1 (or multinational) code 
         so we put it first to Ascii */
      change_banks(used_sets); 
      used_sets[2] = find_set_by_name(rCs("Latin/1"),have_8bit_clean);
      if (-1 == used_sets[2])
	used_sets[2] = find_set_by_name(rCs("Multinational"),have_8bit_clean);
      if (-1 == used_sets[2])
	used_sets[2] = find_set_by_name(rCs("Ascii"),have_8bit_clean);
      used_sets[3] = find_set_by_name(rCs("Multinational"),have_8bit_clean);
      if (-1 == used_sets[3])
	used_sets[3] = find_set_by_name(rCs("UK-ASCII"),have_8bit_clean);
      if (use_technical)
	{
	  int tmp = used_sets[3];
	  used_sets[3] = find_set_by_name(rCs("Technical"),0);
	  if (-1 == used_sets[3]) used_sets[3] = tmp;
	}
    }	  
    if (-1 == used_sets[0]) {
      PANIC("Unsupported keyboard language or software failure");
    } else {
      keyboard_charset = sets[used_sets[0]].charset;
      if (-1 != used_sets[2])
	keyboard_charset = sets[used_sets[2]].charset;
    }
    ASCII_set = find_ASCII();

    GL_set = -1; GR_set = -1;
    FLAGS_set = -1;
    Y1_set = -1; Y2_set = -1;

    for (i = 0; i < 4; i++) if (-1 != used_sets[i]) {
      int wanna = -1;
      if ((sets[used_sets[i]].flags & ATR_cs_ncr) == ATR_cs_ncr)
	wanna = 1; /* national */
      if ((sets[used_sets[i]].area & A_8bit) == A_8bit)
	wanna = 0; /* multinational */
      if (-1 != wanna) {
	if (-1 != use_national && wanna != use_national) {
	  PANIC("Software failure/set_terminal state: Required both "
		"national and multinational mode!");
	} else use_national = wanna;
      }
    }


    change_state(1,
		 have_8bit_clean && HAVE(ATR_C1_controls),
		 neg_undef(use_national),
		 HAZ_MAY(HAZ_wrap) ? -1 : 0,
		 HAZ_MAY(HAZ_application) ? -1 : 0,
		 used_sets,
		 0,
		 HAZ_MAY(HAZ_newline) ? -1 : 0);

    pager_state = 1;

  }

}

static void pass_3(void) {
  SET_PASS(3);
  if (!fast_start) {
    if (1 != limit_query || HAVE(ATR_cs_ncr)) {
      set_mode("%c?26n",CSI); ASKED;  /* Ask keyboard language */
    }
    if (1 != limit_query || HAVE(ATR_terminal_rq)) {
      set_mode("%c?1$p",CSI); ASKED;  /* Ask cursor key application */
      set_mode("%c?7$p",CSI); ASKED;  /* Ask autowrap */
      set_mode("%c?42$p",CSI); ASKED;  /* Ask mational replacement set */
      set_mode("%c?68$p",CSI); ASKED;  /* Ask key legend */
      set_mode("%c20$p",CSI); ASKED; /* Ask new-line mode */
    }
  }
  set_mode("%c6n",CSI); ASKED;    /* Ask cursor position */
  if (!fast_start && query_terminal_size) {
    /* Origin mode off, goto 999,999 and ask position of cursor */
    set_mode("%c?6l%c999;999H%c6n",CSI,CSI,CSI); ASKED;
    X_set = -1; Y_set = -1;
  }

}

static void T_clear_from_line(int line) {
  /* check_region(); */
  set_attr(0);
  goto_XY(1,line);
  set_mode("%c0J",CSI);
  set_down_line(line);
}

static void pass_4(void) {
  SET_PASS(4);
  if (-1 != starting_line) {
    if (starting_line > lines) {
      print_notify("Incorrect terminal size or something is wrong! Y=%d lines=%d",
		   starting_line,lines);
      starting_line=lines;
    }
    set_down_line(starting_line);
  }
  /* Put possible garbage after Initializing terminal text: */
  if (-1 != starting_line && -1 != starting_col && query_terminal_size) 
    /* goto_XY updates down_line: can't use */
    set_mode("%c%d;%dH",CSI,starting_line,starting_col);
  set_terminal_state();
  T_clear_from_line(down_line);
}

static void T_clear_rest_of_line(void) {
  set_attr(0);
  set_mode("%c0K",CSI);
}

static void T_scroll_area(int Y1, int Y2) {
  if (DEBUG_CURSOR) print_debug("  Scrolling UP (%d,%d) with IND",Y1,Y2);
  set_region(Y1,Y2);
  goto_XY(1,Y2);
  set_mode("%c",IND);
}

static void T_scroll_area_R(int Y1, int Y2) {
  if (DEBUG_CURSOR) print_debug("  Scrolling DOWN (%d,%d) with RI",Y1,Y2);
  set_region(Y1,Y2);
  set_attr(0);
  goto_XY(1,Y1);
  set_mode("%c",RI);
  if (down_line <= Y2)
    set_down_line(down_line+1);
}

static void T_delete_line(int line) {
  update_region();
  if (HAVE(ATR_insdel)) {
    goto_XY(1,line);
    set_mode("%cM",CSI);
  } else {
    int bot = Y2_region;
    if (-1 == bot) bot = lines;
    if (bot > lines) bot = lines;
    if (DEBUG_CURSOR) {
      print_debug("Simulated delete line (%d) -> region=(%d,%d) cursor=%d",
		  line,line,bot,bot);
      print_debug("   original (soft) region=(%d,%d)",Y1_region,Y2_region);
    }
    /* Simulate this with scrolling regions */
    T_scroll_area(line,bot);
  }
  if (down_line <= Y2_set+1) 
    set_down_line(down_line-1);
}

static void T_insert_line(int line) {
  update_region();
  if (HAVE(ATR_insdel)) {
    goto_XY(1,line);
    set_attr(0);
    set_mode("%cL",CSI);
  } else {
    int bot = Y2_region;
    if (-1 == bot) bot = lines;
    if (bot > lines) bot = lines;
    /* Simulate this with scrolling regions */
    if (DEBUG_CURSOR) {
      print_debug("Simulated (with insert line (%d) -> region=(%d,%d) cursor=%d",
		  line,line,bot,line);
      print_debug("   original (soft) region=(%d,%d)",Y1_region,Y2_region);
    }
    T_scroll_area_R(line,bot);
  }
  if (down_line <= Y2_set) 
    set_down_line(down_line+1);
}

static struct oneline {
  int len;
  int is_cleaned;
  int priority,X1,X2;
  struct onechar {
    CHAR c;
    unsigned short set;
    unsigned short flags;
  } *chars;
} *image_lines = NULL;
static int line_count = 0;

#define PRI_COUNT 7
static struct PRI {
  int first_line;
  int last_line;
} priorities[PRI_COUNT] = {
  { 0, 0 },
  { 0, 0 },
  { 0, 0 },
  { 0, 0 },
  { 0, 0 },
  { 0, 0 },
  { 0, 0 }
};

static void add_char(int X,int Y, int set, int flags, CHAR c,
		     int *is_done, int *first_X) {

  if (((int)c & 127) < 32) {
    close_files();
    close_terminal();
    flush_error_buffer();
    print_notify("(%d) add_char(X=%d,Y=%d,set=%d,flags=%d,c=%d, \n"
		 "              *is_done=%d,*first_X=%d): \n"
		 "     Software error: char (c=%d) in control range!",
		 X,Y,set,flags,c,*is_done,*first_X,
		 c);
    exit(10);
  }

  if (set < 0 || set >= SET_NUM) {
    close_files();
    close_terminal();
    flush_error_buffer();
    print_notify("(%d) add_char(X=%d,Y=%d,set=%d,flags=%d,c=%d, \n"
		 "              *is_done=%d,*first_X=%d): \n"
		 "     Software error: charset (set=%d) out of range (0-%d)!",
		 X,Y,set,flags,c,*is_done,*first_X,
		 set,SET_NUM-1);
    exit(10);

  }

  if (line_count < Y) {
    int y;
    image_lines = (struct oneline *)REALLOC((void *)image_lines,
					    Y*sizeof(struct oneline));

    for (y = line_count; y < Y; y++) {
      image_lines[y].len = 0;
      image_lines[y].is_cleaned = 0;
      image_lines[y].chars = NULL;
      image_lines[y].priority = 0;
      image_lines[y].X1 = 0;
      image_lines[y].X2 = 0;
    }
    line_count = Y;
  }
  
  if (image_lines[Y-1].X1 > 0 && *is_done) {
    *first_X = image_lines[Y-1].X1;
    *is_done = 0;
  }

  if (image_lines[Y-1].len < X) {
    int x;

    /* if pending clear !! */ 
    if (image_lines[Y-1].is_cleaned == 2) {
      if (*is_done) *first_X = image_lines[Y-1].len+1;
      *is_done = 0;
      if (0 == image_lines[Y-1].X1 || 
	  image_lines[Y-1].X1 > image_lines[Y-1].len+1) 
	image_lines[Y-1].X1 = X;
      if (image_lines[Y-1].X2 < image_lines[Y-1].len+1) 
	image_lines[Y-1].X2 = image_lines[Y-1].len+1;
    }



    image_lines[Y-1].chars = (struct onechar *)
      REALLOC((void *)image_lines[Y-1].chars,
	      X*sizeof(struct onechar));

    for (x = image_lines[Y-1].len; x < X; x++) {
      image_lines[Y-1].chars[x].c = ' ';
      image_lines[Y-1].chars[x].set = 0;
      image_lines[Y-1].chars[x].flags = 0;
    }

    image_lines[Y-1].len = X;
  }

  if (image_lines[Y-1].chars[X-1].c != c ||
      image_lines[Y-1].chars[X-1].set != set ||
      image_lines[Y-1].chars[X-1].flags != flags) {
    image_lines[Y-1].chars[X-1].c = c;
    image_lines[Y-1].chars[X-1].set = set;
    image_lines[Y-1].chars[X-1].flags = flags;

    if (*is_done) *first_X = X;
    *is_done = 0;
    if (0 == image_lines[Y-1].X1 || image_lines[Y-1].X1 > X) 
      image_lines[Y-1].X1 = X;
    if (image_lines[Y-1].X2 < X) 
      image_lines[Y-1].X2 = X;
  }
}


static void update_priority(int Y, int priority) {
  if (0 == priorities[priority].first_line ||
      priorities[priority].first_line > Y)
    priorities[priority].first_line = Y;
  if (priorities[priority].last_line < Y)
    priorities[priority].last_line = Y;

  if (image_lines[Y-1].priority < priority)
    image_lines[Y-1].priority = priority;
  if (DEBUG_LBUFFER) 
    print_debug("update_priority(Y=%d, priority=%d): \n"
		"  priorities[priority=%d].first_line=%d\n"
		"  priorities[priority=%d].last_line=%d\n"
		"  image_lines[Y-1=%d].priority=%d",
		Y,priority,
		priority,priorities[priority].first_line,
		priority,priorities[priority].last_line,
		Y-1,image_lines[Y-1].priority);
}


void clear_line(int line) {
  if (line_count >= line && image_lines[line-1].chars) {
    FREE((void *)image_lines[line-1].chars);
    image_lines[line-1].chars = NULL;
    image_lines[line-1].len = 0;
    image_lines[line-1].X1 = 0;
    image_lines[line-1].X2 = 0;

    if (image_lines[line-1].priority < PRI_WAIT && line <= lines &&
	line < down_line) {
      if (DEBUG_LBUFFER)
	print_debug("clear_line: buffering clearing for line %d "
		    "  (priority %d)",
		    line,image_lines[line-1].priority);
      update_priority(line,image_lines[line-1].priority);
      image_lines[line-1].is_cleaned = 2;
      return;
    }

    if (line <= lines && pager_state) {

      goto_XY(1,line);
      T_clear_rest_of_line();
      update_hard();

      image_lines[line-1].is_cleaned = 1;
    }
  } else if (line <= lines && pager_state && line < down_line) {

    if (line_count >= line &&
	image_lines[line-1].is_cleaned == 1) return;

    goto_XY(1,line);
    T_clear_rest_of_line();
    update_hard();

    if (line_count >= line)
      image_lines[line-1].is_cleaned = 1;

  }
}

void clear_rest_of_line(int x,int line) {
  if (1 == x) {
    clear_line(line);
    return;
  }

  if (line_count >= line && 
      image_lines[line-1].chars && image_lines[line-1].len >= x) {
    image_lines[line-1].chars = (struct onechar *)
      REALLOC((void *)image_lines[line-1].chars,
	      (x-1) * sizeof(struct onechar));
    image_lines[line-1].len = x-1;

    if (image_lines[line-1].priority < PRI_WAIT && line <= lines &&
	line < down_line) {
      if (DEBUG_LBUFFER)
	print_debug("clear_rest_of_line: buffering clearing for line %d "
		    "  (priority %d)",
		    line,image_lines[line-1].priority);
      update_priority(line,image_lines[line-1].priority);
      image_lines[line-1].is_cleaned = 2;
      return;
    }

    if (line <= lines && x <= columns && pager_state) {

      goto_XY(x,line);
      T_clear_rest_of_line();
      update_hard();

      image_lines[line-1].is_cleaned = 1;
    }
  } else if (line <= lines && x <= columns && pager_state 
	     && line < down_line) {

    if (line_count >= line &&
	image_lines[line-1].is_cleaned == 1) return;

    goto_XY(x,line);
    T_clear_rest_of_line();
    update_hard();

    if (line_count >= line)
      image_lines[line-1].is_cleaned = 1;
  }
}

static void T_print(int X,int Y, int len) {
  int x,x1;
  int flags, set;
  CHAR sbuffer[MAX_OUTPUT+1];
  CHAR *buffer;

  if (Y > lines) return;

  if (line_count < Y) {
    T_clear_from_line(line_count+1);
    return;
  }

  if (image_lines[Y-1].len < X) {
    goto_XY(image_lines[Y-1].len+1,Y);
    T_clear_rest_of_line();
    return;
  }

  if (len < 1) return;

  buffer = len < MAX_OUTPUT ? sbuffer : (CHAR *) MALLOC(len+1);
  goto_XY(X,Y);

  for(x = X; x < X + len && x <= image_lines[Y-1].len; x=x1) {
    CHAR *ptr = buffer;

    flags = image_lines[Y-1].chars[x-1].flags;
    set   = image_lines[Y-1].chars[x-1].set;

    for (x1=x;
	 x1 < X + len && x1 <= image_lines[Y-1].len &&
	 flags == image_lines[Y-1].chars[x1-1].flags &&
	 set == image_lines[Y-1].chars[x1-1].set;
	 x1++) {
      *ptr++ = image_lines[Y-1].chars[x1-1].c;
    }
    *ptr = '\0';
    set_attr(flags);
    print_viaset(set,buffer);
  }

  if (x - X < len && -1 != X_set) {
    T_clear_rest_of_line();
  }

  if (buffer != sbuffer) FREE(buffer);

  if (image_lines[Y-1].X1 >= X && image_lines[Y-1].X2 <= X + len) {
    image_lines[Y-1].X1 = 0;
    image_lines[Y-1].X2 = 0;
    image_lines[Y-1].priority = 0;
    if (DEBUG_LBUFFER) 
      print_debug("T_print: reset priority for line %d",Y);
  }
}

static int add_mapped(int X, int Y, int flags, CHAR_IDX mapped,
		      int *is_done,int *first_X) {
  CHAR buffer[5+MAX_FALLBACK],*p;
  int set,x=X;

  if (mapped < 0) {
    flags |= FL_REVERSE;
    set = ASCII_set;  /* ASCII */
    if (-mapped > 255)
      sprintf(Cs(buffer),"`%04X",(int)-mapped);
    else
      sprintf(Cs(buffer),"`%02X",(int)-mapped);
  } else if (!chars[mapped].printable) {
    int tmp;
    if ((tmp=strlen(chars[mapped].fallback)) > MAX_FALLBACK) {
      print_notify("(%s) Software error/Length for fallback string "
		   "for mapped=%d too long: %d > %d",
		   prog,(int)mapped,tmp,MAX_FALLBACK);
      sprintf(Cs(buffer),"`(0x%X)",(unsigned int)mapped);
    } else 
      strcpy(Cs(buffer),chars[mapped].fallback);

    flags |= FL_REVERSE;
    set = ASCII_set;  /* ASCII */

  } else {
    int i,set2 = -1,set3=-1;
    set = -1;

    for (i = 0; i < 4; i++) if (-1 != used_sets[i])
      if (-1 != chars[mapped].vector[sets[used_sets[i]].charset]) {
	if (chars[mapped].vector[sets[used_sets[i]].charset]
	    & FC_uncertain) 
	  set2 = used_sets[i];
	else if (chars[mapped].vector[sets[used_sets[i]].charset]
	    & FC_fallback) 
	  set3 = used_sets[i];
	else
	  set = used_sets[i];
	break;
      }

    if (-1 == set) {
      set = set3;
      flags |= FL_REVERSE;
    }
    if (-1 == set) set = set2;
    if (-1 == set) {
      int tmp;
      flags |= FL_REVERSE;
      set = ASCII_set;

      if ((tmp=strlen(chars[mapped].fallback)) > MAX_FALLBACK) {
	print_notify("(%s) Software error/Len for fallback string for "
		     "mapped=%d too long: %d > %d",
		     prog,mapped,tmp,MAX_FALLBACK);
	sprintf(Cs(buffer),"`%02X",(unsigned int)-mapped);
      } else 
	strcpy(Cs(buffer),chars[mapped].fallback);
    } else {
      buffer[0] = chars[mapped].vector[sets[set].charset] & CHAR_code;
      buffer[1] = '\0';
    }
  }

  for (p = buffer; *p; p++)
    add_char(x++,Y,set,flags,*p,is_done,first_X);

  return x-X;
}

static int mapped_char_len(CHAR_IDX mapped) {
   if (mapped < -0xFF) return 5;
   else if (mapped < 0) return 3;
   else if (!chars[mapped].printable) return strlen(chars[mapped].fallback);
   else {
    int i;
    int set = -1;

    for (i = 0; i < 4; i++) if (-1 != used_sets[i])
      if (-1 != chars[mapped].vector[sets[used_sets[i]].charset]) {
	set = used_sets[i];
	break;
      }

    if (-1 == set) return strlen(chars[mapped].fallback);
    else return 1;
   }
}

int need_line_flush(void) {
  int p, result = 0;

  if (!pager_state) return 0;

  for (p=0; p < PRI_COUNT; p++)
    if (priorities[p].first_line) {
      if (!result) result = 1;
      if (1 == priorities[p].first_line &&
	  lines <= priorities[p].last_line)
	result = 2;
      if (DEBUG_LBUFFER) {
	print_debug("need_line_flush: priority=%p lines=%d-%d",
		    p,priorities[p].first_line,priorities[p].last_line);
      }
    }

  if (DEBUG_LBUFFER)
    print_debug("need_line_flush: result=%d",result);

  return result;
}

static void flush_line(int Y, int pri) {
  if (Y > line_count || image_lines[Y-1].priority != pri) return;

  if (image_lines[Y-1].X1 > 0) {
    if (Y <= lines) {
      if (DEBUG_LBUFFER)
	print_debug("flush_line: flushing %d (%d-%d)",Y,
		    image_lines[Y-1].X1,image_lines[Y-1].X2);
      
      T_print(image_lines[Y-1].X1,Y,
	      image_lines[Y-1].X2-image_lines[Y-1].X1+1);
    } else {
      if (DEBUG_LBUFFER)
	print_debug("flush_line: clearing %d (out of screen)",Y);
      image_lines[Y-1].X1=0;
      image_lines[Y-1].X2=0;
      image_lines[Y-1].priority=0;
    }
  }

  if (2 == image_lines[Y-1].is_cleaned) {
    if (Y <= lines && image_lines[Y-1].len < columns) {
      if (DEBUG_LBUFFER)
	print_debug("flush_line: clearing %d from %d (is_cleaned=%d)",
		    Y,image_lines[Y-1].len+1,image_lines[Y-1].is_cleaned);
      goto_XY(image_lines[Y-1].len+1,Y);
      T_clear_rest_of_line();
      image_lines[Y-1].is_cleaned = 1;
    } 
  }
}

int flush_some_lines(void) {
  int p,pri=-1,Y;

  if (!pager_state) return 0;

  for (p=0; p < PRI_COUNT; p++)
    if (priorities[p].first_line) pri=p;
  
  if (-1 == pri) {
    if (DEBUG_LBUFFER)
      print_debug("flush_some_lines: result=0");

    return 0;
  }

  if (DEBUG_LBUFFER)
    print_debug("flush_some_lines: priority=%d lines=%d-%d",
		pri,priorities[pri].first_line,
		priorities[pri].last_line);

  

  for (Y = priorities[pri].first_line;
       Y <= priorities[pri].last_line;
       Y++) flush_line(Y,pri);
  
  priorities[pri].first_line=0;
  priorities[pri].last_line=0;

  if (DEBUG_LBUFFER)
    print_debug("flush_some_lines: result=1\n"
		"  priorities[pri=%d].first_line=%d\n"
		"  priorities[pri=%d].last_line=%d",
		pri,priorities[pri].first_line,
		pri,priorities[pri].last_line);
  update_hard();  
  return 1;
}

void flush_to_priority(int priority) {
  if (!pager_state) return;

  while (priorities[priority].first_line && flush_some_lines());
}

int print_XY(int X, int Y, int flags,int len,CHAR_IDX *mapped, int showspc,
	     int priority) {
  int x=X,y=Y, space = 1, extra = 0, is_done=1,first_X = X;

#if defined(MAX_LINES)
  if (Y > MAX_LINES)
    return 0;
#endif

  while(len--) {
    if (space && *mapped == CH_SPC && extra > 0 && !showspc) {
      extra--;
    } else {
      int n;
      space = *mapped == CH_SPC;
      n = add_mapped(x,y,flags,*mapped,&is_done,&first_X);
      extra += n -1;
      x += n;
    }
    mapped++;
  }

  if (!is_done && Y <= lines)
    update_priority(Y,priority);

  if (!is_done && pager_state && 
      MAX(priority,image_lines[Y-1].priority) >= PRI_WAIT) {
    /* T_print(first_X,Y,x-first_X); */
    flush_line(Y,MAX(priority,image_lines[Y-1].priority));
    flush_buffer(0);
  }

  return x-X;
}

int printable_len(int len,CHAR_IDX *mapped, int *list, int showspc) {
  int n=0,extra = 0,space=1,i;

  for (i=0; i < len; i++) {
    if (space && mapped[i] == CH_SPC && extra > 0 && !showspc) {
      extra--;

    } else {
      int I;
      
      space = mapped[i] == CH_SPC;
      I = mapped_char_len(mapped[i]);
      extra += I -1;
      n += I;
    }
    if (list) 
      list[i] = n;
  }
  return n;
}

void clear_from_line(int line) {
  int i,is_done=1;

  check_region();

  if (line_count < line) return;

  for (i = line -1; i < line_count; i++)  {
    if (image_lines[i].chars) {
      image_lines[i].len = 0;
      image_lines[i].X1 = 0;
      image_lines[i].X2 = 0;
      image_lines[i].priority = 0;
      FREE(image_lines[i].chars);
      image_lines[i].chars = NULL;
      is_done = 0;
    }
    if (1 != image_lines[i].is_cleaned)
      is_done = 0;
  }

  if (1 == line && image_lines) {
    line_count = 0;
    FREE(image_lines);
    image_lines = NULL;
  } else {
    for (i = line -1; i < line_count; i++) 
      image_lines[i].is_cleaned = 1;
  }

  if (!is_done && pager_state && line <= lines) {
    while(flush_some_lines());

    T_clear_from_line(line);
    update_hard();
  }
}

static void defogus_priorities(void) {
  int p;
  for (p = 0; p < PRI_COUNT; p++) 
    if (priorities[p].first_line) {
      priorities[p].first_line = 1;
      priorities[p].last_line = line_count+1;
      if(DEBUG_LBUFFER)
	print_debug("defogus_priorities: \n"
		    "  priorities[%d].first_line=%d \n"
		    "  priorities[%d].last_line=%d",
		    p,priorities[p].first_line,
		    p,priorities[p].last_line);
    }

}

void scroll_area(int Y1,int Y2) {
  int bot = Y2,i,rbot;
  check_region();
  if (bot > line_count) bot = line_count;
  if (Y1 > bot) return;

  defogus_priorities();

  rbot = bot;
  if (rbot > lines) rbot = lines;

  if (image_lines[Y1-1].chars) {
    FREE(image_lines[Y1-1].chars);
    image_lines[Y1-1].chars = NULL;
    image_lines[Y1-1].len = 0;
  }

  for (i = Y1; i < bot; i++)
    image_lines[i-1] = image_lines[i];

  image_lines[bot-1].chars = NULL;
  image_lines[bot-1].len = 0;
  image_lines[bot-1].is_cleaned = 0;
  if (bot == line_count) line_count--;

  if (Y1 <= rbot && pager_state) {
    T_scroll_area(Y1,rbot);
    T_print(1,rbot,columns);
    update_hard();
  }  
}

void delete_line(int line) {
  int i, R=is_region(line), B=line_count, SB=lines;

  if (line > line_count) return;

  defogus_priorities();

  if (R && line_count > Y2_region) B = Y2_region;
  if (R && lines > Y2_region) SB = Y2_region;

  if (image_lines[line-1].chars) {
    FREE(image_lines[line-1].chars);
    image_lines[line-1].chars = NULL;
    image_lines[line-1].len = 0;
  }

  for (i = line; i < B; i++)
    image_lines[i-1] = image_lines[i];

  image_lines[B-1].chars = NULL;
  image_lines[B-1].len = 0;
  image_lines[B-1].is_cleaned = 0;
  if (B == line_count) line_count--;

  if (line <= SB && pager_state) {
    T_delete_line(line);
    T_print(1,SB,columns);
    update_hard();
  }
}

void insert_line(int line) {
  int i, R=is_region(line), B=line_count, last;

  if (line > line_count) return;

  defogus_priorities();

  if (R && line_count > Y2_region) B = Y2_region;

  if (B == line_count) {
    image_lines = (struct oneline *)
      REALLOC((void *)image_lines,(line_count+1) *sizeof(struct oneline));
    last = line_count++;
  }
  else if (image_lines[B-1].chars) {
    FREE(image_lines[B-1].chars);
    image_lines[B-1].chars = NULL;
    image_lines[B-1].len = 0;
    last = B-1;
  } else 
    last = B-1;

  for (i = last ; i >= line; i--)
    image_lines[i] = image_lines[i-1];

  image_lines[line-1].chars = NULL;
  image_lines[line-1].len = 0;
  image_lines[line-1].is_cleaned = 0;


  if (line <= lines && pager_state) {
    T_insert_line(line);
    update_hard();
  }

#if defined(MAX_LINES)
  while (line_count > MAX_LINES)
    delete_line(line_count);
#endif

}

void do_redraw(void) {
  int y;
  if (!pager_state) set_terminal_state();
  if (!pager_state) return;

  set_margins(-1,-1);
  update_region();

  T_clear_from_line(1);
  for (y = 1; y <= line_count; y++) {
    image_lines[y-1].is_cleaned = 0; /* need recleaning */
    T_print(1,y,columns);
  }
  update_hard();
}

int ask_timeout(long *timeout) {
  static int states_first = 1;
  /* 1/10 seconds */
  if (terminal_type_pass && terminal_type_pass < READY_PASS) {
    switch(terminal_type_pass) {
    case 1: *timeout = timer_id_1; return 1;
    case 2: *timeout = timer_id_2; return 1;
    case 3: if (states_first) {
      states_first = 0; *timeout = timer_states_1;
    } else *timeout = timer_states_2;
      return 2;
    }
  }
  return 0;
}

void do_timeout(void) {
  if (DEBUG_TERM) {
    print_debug("Timeout!");
  }
  switch (terminal_type_pass) {
  case 1:
    print_notify("(%s) Terminal init failed",prog);
    close_terminal();
    exit(1);
    break;
  case 2:    /* We don't know if terminal can get 8bit characters */
    have_8bit_clean = 0;
    if (-1 == default_multinational) default_multinational = 0;
    pass_3();
    break;
  case 3:
    GOT;
    break;
  }

}

int do_escape(int cmd) {
  CHAR chr; 
  CHAR *args;
  int len;
  CHAR *ptr;
  
  if (C_escape != commands[cmd].what)  return 0;
  chr = commands[cmd].cmd;
  args = rCs(commands[cmd].char_arr);
  len = commands[cmd].len;
  
  switch(chr) {
  case 'R':
    if ('?' == *args) {
      return 0;
    } else { /* Cursor position */
      int X,Y;
      pos_count++;
      if (2 == sscanf(Cs(args),"%d;%d",&Y,&X)) {
	switch(pos_count) {
	case 1: 
	  starting_line = Y;
	  starting_col=X;
	  break;
	case 2: if (X < 999 && Y < 999) {
	  set_terminal_size(X,Y);
	  set_lines=Y;
	  set_columns=X;
	}
	  break;
	}
	GOT;
      }
      return 1;
    }
    break;
  case 'c':
    if ('?' == *args) { /* terminal status report */
      args++;
      
      switch (terminal_type_pass) {
      case 1:
	parse_DEC_DA(args);
	do_magic(); /* vt100 magic ... */
	saved_personality = personality_8bit;
	if (personality_8bit) {
	  have_8bit_clean = 1;
	  if (!HAVE(ATR_cs_multinational) && !HAVE(ATR_cs_latin1)) 
	    SUPPOSE(ATR_cs_multinational);
	  if (!HAVE(ATR_G2)) 
	    SUPPOSE(ATR_G2);
	  if (!HAVE(ATR_C1_controls)) 
	    SUPPOSE(ATR_C1_controls);

	  if (-1 == default_multinational) default_multinational = 1;
	  pass_3();
	  break;
	}
	pass_2();
	break;
      case 2:
	have_8bit_clean = 1;
	if (!HAVE(ATR_cs_multinational) && !HAVE(ATR_cs_latin1)) 
	  SUPPOSE(ATR_cs_multinational);
	if (!HAVE(ATR_G2)) 
	  SUPPOSE(ATR_G2);
	if (!HAVE(ATR_C1_controls)) 
	  SUPPOSE(ATR_C1_controls);

	if (-1 == default_multinational) default_multinational = 1;
	pass_3();
	break;
      default: ;
      }
      return 1;
    } else {
      return 0;
    }
    break;
  case 'n':
    if ('?' == *args) { /* terminal status report */
      int group;
      args++;
      ptr = rCs(strtok(Cs(args),";")); if (!ptr) break;
      group = atoi(Cs(ptr));
      switch (group) {
      case 27: /* keyboard language reported */
	ptr = rCs(strtok(NULL,";")); if (!ptr) break;
	keyboard_language = atoi(Cs(ptr));
	
	if (terminal_type_pass != 3) return 1;
	
	if (keyboard_language) {
	  saved_language = keyboard_language;
	  if (!HAVE(ATR_cs_ncr)) SUPPOSE(ATR_cs_ncr);
	}
	GOT;
	break;
      }
      return 1;
    } else {
      return 0;
    }
    break;
  case 'y':
    if (len > 1 && '$' == args[len-1]) {
      int mode = '?' == args[0];
      int range,val;
      char c;
      if (mode) args++;
      if (3 == sscanf(Cs(args),"%d;%d%c",&range,&val,&c) && '$' == c) {
	if (terminal_type_pass != 3) return 1;
	
	switch (mode) {
	case 0:
	  switch(range) {
	  case 20: /* New line mode */
	    if (1 == val || 3 == val) saved_newline = 1;
	    if (2 == val || 4 == val) saved_newline = 0;
	    if (3 == val) HAZ_DOES(HAZ_newline);
	    break;
	  }
	  break;
	case 1:
	  switch(range) {
	  case 42: /* National replacement character set */
	    if (1 == val || 3 == val) saved_multinational = 0;
	    if (3 == val) HAZ_DOES(HAZ_national);
	    if (2 == val || 4 == val) saved_multinational = 1;
	    break;
	  case 7: /* Autowrap */
	    if (1 == val || 3 == val) saved_autowrap = 1;
	    if (3 == val) HAZ_DOES(HAZ_wrap);
	    if (2 == val || 4 == val) saved_autowrap = 0;	
	    break;
	  case 1: /* Cursor key application */
	    if (1 == val || 3 == val) saved_application = 1;
	    if (2 == val || 4 == val) saved_application = 0;	
	    if (3 == val) HAZ_DOES(HAZ_application);
	    break;
	  case 68: /* key legend */
	    if (1 == val || 3 == val) saved_legend = 1;
	    if (2 == val || 4 == val) saved_legend = 0;	
	  }
	  break;
	}
	GOT;
      }
      return 1;
    } else {
      return 0;
    }
    break;
  default:
    return 0;
    break;
  }
  return 0;
}

void set_hard(int X,int Y) {
  X_hard = X; Y_hard = Y;
  if (pager_state) update_hard();
}
