/*  file: content.c
 *
 *  kehpager, Charset aware pager, Kari E. Hurtta
 *
 *  Copyright (c) 1993, 1994 Kari E. Hurtta
 *
 *  Redistribution and use in source and binary forms are permitted
 *  provided that the above copyright notice and this paragraph are
 *  duplicated in all such forms. This software is provided 'as is'
 *  and without any warranty. 
 */

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "kehpager.h"
#include "memory.h"
#include "charset.h"
#include "content.h"
#include "window.h"
#include "control.h"
#include "terminal.h"

#include <stdlib.h>
#include <string.h>

int enable_fulljust = 0;

void scroll_it(int win) {
  int Y,offset,roffset, pagenum,pages,was_updated=0;
  set_win_prompt(-1,-1,-1);  

  for (window_pos(win,&Y,&offset,&roffset,&pagenum,&pages); 
       window_rlines(win) - roffset > lines -Y && roffset < 0;
       window_pos(win,&Y,&offset,&roffset,&pagenum,&pages)) {
    if (!was_updated) {
      /* update lines from previous scroll_it call !! */
      while(flush_some_lines());
      was_updated = 1;
    }
    scroll_window(win); 
  }
}

static void flush_S_buffer(content *CT,String *buffer,int flag) {
  CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(StrLEN(*buffer)	* sizeof(CHAR_IDX));

  map_input(MAP_ASCII,StrLEN(*buffer),StrPTR(*buffer),mapped);
  print_to_window(CT->win,CT->last_line_X+1,
		  CT->last_line_Y,
		  flag,
		  StrLEN(*buffer),mapped);
  CT->last_line_X += StrLEN(*buffer);

  FREE(mapped);
  StrFree(buffer);
  
}

/* Terminal content-type -------------------------------------------------- */

static int special_char(content *CT,unsigned ch) {
  if (StrLEN(CT->c.terminal.attrib_buffer)) return 0;
  if ('\r' == ch) return 1;
  if ('\b' == ch) return 1;
  if (ESC == ch) return 1;
  return 0;
}

static void flush_attrib_buffer(content *CT) {
  flush_S_buffer(CT,
		 &(CT->c.terminal.attrib_buffer),
		 CT->c.terminal.attrib_char_flag);
}

static int attrib_handle(content *CT,CHAR ch) {
  if (0 == StrLEN(CT->c.terminal.attrib_buffer)) return 0;
  if (1 == StrLEN(CT->c.terminal.attrib_buffer)) {
    if ('[' == ch) {
      StrAdd(&(CT->c.terminal.attrib_buffer),ch);
      return 1;
    } else if ('>' == ch || '=' == ch) {
      /* HACK: forgot keypad commands */
      StrFree(&(CT->c.terminal.attrib_buffer));
      return 1;
    } else {
      flush_attrib_buffer(CT);
      return 0;
    }
  } else if (isdigit(ch) || ';' == ch || 
	     /* HACK: forgot keypad commands */
	     ('?' == ch && StrLEN(CT->c.terminal.attrib_buffer) == 2)) {
    StrAdd(&(CT->c.terminal.attrib_buffer),ch);
    return 1;
  } else if ('m' == ch && 
	     (StrLEN(CT->c.terminal.attrib_buffer) < 3 || 
	      '?' != StrPTR(CT->c.terminal.attrib_buffer)[2])) {
    CHAR *c,*d;
    c=StrPTR(CT->c.terminal.attrib_buffer)+2;
    if ('\0' == c[0]) CT->c.terminal.attrib_char_flag = 0;
    else for (d=rCs(strtok(Cs(c),";")); NULL != d; d=rCs(strtok(NULL,";"))) {
      if ('\0' == d[0]) CT->c.terminal.attrib_char_flag = 0;
      else switch(atoi(Cs(d))) {
      case 0: CT->c.terminal.attrib_char_flag = 0; break;
      case 1: CT->c.terminal.attrib_char_flag = 
	sum_fl_attr(CT->c.terminal.attrib_char_flag,FL_BOLD); break;
      case 2: CT->c.terminal.attrib_char_flag = 
	sum_fl_attr(CT->c.terminal.attrib_char_flag,FL_DIM); break;
      case 4: CT->c.terminal.attrib_char_flag = 
	sum_fl_attr(CT->c.terminal.attrib_char_flag,FL_UNDER); break;
      case 5: CT->c.terminal.attrib_char_flag = 
	sum_fl_attr(CT->c.terminal.attrib_char_flag,FL_BLINK); break;
      case 7: CT->c.terminal.attrib_char_flag = 
	sum_fl_attr(CT->c.terminal.attrib_char_flag,FL_REVERSE); break;
      case 8: CT->c.terminal.attrib_char_flag &= ~(FL_BOLD | FL_DIM) /* ??? */;
	break;
      case 22: CT->c.terminal.attrib_char_flag &= ~(FL_BOLD | FL_DIM); break;
      case 24: CT->c.terminal.attrib_char_flag &= ~FL_UNDER; break;
      case 25: CT->c.terminal.attrib_char_flag &= ~FL_BLINK; break;
      case 27: CT->c.terminal.attrib_char_flag &= ~FL_REVERSE; break;
      }
    }
    StrFree(&(CT->c.terminal.attrib_buffer));
    return 1;
  } else if ((ch == 'l' || ch == 'h') &&
	     ((StrLEN(CT->c.terminal.attrib_buffer) == 4 && 
	      0 == strncmp(Cs(StrPTR(CT->c.terminal.attrib_buffer))+2,
			   "?1",2)) ||
	      (StrLEN(CT->c.terminal.attrib_buffer) == 5 && 
	      0 == strncmp(Cs(StrPTR(CT->c.terminal.attrib_buffer))+2,
			   "?66",3)))) {
    /* HACK: forgot keypad commands */
    StrFree(&(CT->c.terminal.attrib_buffer));
    return 1;
  } else {
    flush_attrib_buffer(CT);
    return 0;
  }
}

static int TR_basic_handle(content *CT,int len, CHAR *buffer,int flush) {
  int ret,mlen;
  CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));
  int flag = 0;
  
  ret = map2_input(&(CT->cs),len,buffer,&mlen,mapped,CT->multi,flush); 
  if (0 == ret) {
    CT->multi = !CT->multi;
    FREE(mapped);
    return 0;
  }
  else if (ret < 0) { 
    StrSet2(&(CT->multi_buffer),len,buffer);
    FREE(mapped);
    return -1;
  } 
  
  if (CT->c.terminal.owerwrite) {
    int max = win_line_len(CT->win,CT->last_line_Y);
    if (CT->last_line_X >= max) CT->c.terminal.owerwrite = 0;
    else {
      CHAR_IDX CH = read_win_char(CT->win,
			     CT->last_line_X+1,
			     CT->last_line_Y);
      
      if (CH == *mapped) flag = FL_BOLD;
      else if (CH == CH_UL) flag = FL_UNDER;
    }
  }
      
  print_to_window(CT->win,
		  CT->last_line_X+1,
		  CT->last_line_Y,
		  flag
		  |(CT->c.terminal.attrib_char_flag),
		  mlen,mapped);
  CT->last_line_X += mlen;
  
  FREE(mapped);

  return ret;
}

static int TR_handle_multi(content *CT,int len, CHAR *buffer,int flush) {
  int ret = TR_basic_handle(CT,len,buffer,flush);
  if (ret < 0) return len;
  return ret;
}

static int TR_handle(content *CT,int len, CHAR *buffer) {
  CHAR *ch1,*ch2;
  
  set_win_prompt(-1,-1,-1);
  
  for (ch1 = buffer; ch1 < buffer + len; ch1 = ch2) {
    CHAR *ch3;

    for (ch2 = ch1; ch2 < buffer + len && attrib_handle(CT,*ch2); ch2++);
       
    ch3 = ch2;
    for (; ch2 < buffer + len && !special_char(CT,*ch2) &&
	 (!CT->c.terminal.owerwrite || ch2 == ch1) ; ch2++);
    
    if (ch2 != ch3) {
      int ret = TR_basic_handle(CT,ch2-ch3,ch3,0);      
      if (ret < 0) { CrashMe ("multibyte store with no multi data"); }
      if (0 == ret) {
	break; /* will return 0 is nothing handled */
	/* return 0; */
      }
      ch2 = ch3 + ret;
    }
    
    for (;ch2 < buffer + len && special_char(CT,*ch2); ch2++) {
      switch(*ch2) {
      case '\r':
	CT->last_line_X = 0;
	CT->c.terminal.owerwrite = 1;
	break;
      case '\b':
	if (CT->last_line_X > 0)  {
	  CT->last_line_X--;
	  CT->c.terminal.owerwrite = 1;
	}
	break;
      case ESC:
	StrAdd(&(CT->c.terminal.attrib_buffer),ESC);
	break;
      }
    }
    scroll_it(CT->win); /* for autowrap */
  }
  return ch1 - buffer;
}

static void TR_handle_eoln (content *CT) {
  scroll_it(CT->win);
  CT->c.terminal.owerwrite = 1;
  CT->last_line_Y++;
  CT->last_line_X = 0;
  clear_window_line(CT->win,
		    CT->last_line_Y);
}

static void TR_init (content *CT) {

  CT->c.terminal.magic = CT_terminal;
  CT->c.terminal.attrib_char_flag = 0;
  StrInit(&(CT->c.terminal.attrib_buffer),10);
  CT->c.terminal.owerwrite = 0;
}

static void TR_exit(content *CT) {
  if(StrLEN(CT->c.terminal.attrib_buffer))
    flush_attrib_buffer(CT);
  StrFree(&(CT->c.terminal.attrib_buffer));
  CT->c.terminal.magic = -1;
}

static void TR_update(content *CT) {

}

/* Plain content-type ------------------------------------------------ */

static int PL_basic_handle(content *CT,int len, CHAR *buffer, int flush) {
  set_win_prompt(-1,-1,-1);
  
  if (len > 0) {
    int ret,mlen;
    CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));
    
    ret=map2_input(&(CT->cs),len,buffer,&mlen,mapped,CT->multi,flush); 
    if (0 == ret) {
      CT->multi = !CT->multi;
      FREE(mapped);
      return 0;
    }
    else if (ret < 0) { 
      StrSet2(&(CT->multi_buffer),len,buffer);
      FREE(mapped);
      return len;
    }

    print_to_window(CT->win,
		    CT->last_line_X+1,
		    CT->last_line_Y,
		    0,
		    mlen,mapped);
    CT->last_line_X += mlen;
    
    FREE(mapped);
    return ret;
  }
  scroll_it(CT->win);
  return 0;
}

static int PL_handle_multi(content *CT,int len, CHAR *buffer, int flush) {
  return PL_basic_handle(CT,len,buffer,flush);
}

static int PL_handle(content *CT,int len, CHAR *buffer) {
  return PL_basic_handle(CT,len,buffer,0);
}

static void PL_handle_eoln(content *CT) {
  scroll_it(CT->win);
  CT->last_line_Y++;
  CT->last_line_X = 0;
  clear_window_line(CT->win,
		    CT->last_line_Y);
}

static void PL_init (content *CT) {
  CT->c.plain = CT_plain;
}

static void PL_exit(content *CT) {
  CT->c.plain = -1;
}

static void PL_update(content *CT) {

}

/* Routines for Enriched and Richtext content-type ------------------------ */


static void flush_cmd_buffer(content *CT) {
  flush_S_buffer(CT,&(CT->c.enriched.cmd_buffer),
		 CT->c.enriched.stack_top->attrib_flag);
  CT->c.enriched.spc_count = 0;
}


typedef void ER_function (content *CT, struct er_STACK *current);

static void  ER_main_enter (content *CT, struct er_STACK *current) {

}

static void  ER_main_exit (content *CT, struct er_STACK *current) {

}

static void  ER_unknown_enter (content *CT, struct er_STACK *current) {

}

static void  ER_unknown_exit (content *CT, struct er_STACK *current) {

}

static void  ER_verbatim_enter (content *CT, struct er_STACK *current) {
  CT->c.enriched.verbatim = 1;  
  current->force_wrap = 0;
  current->justify = JUST_left;
}

static void  ER_verbatim_exit (content *CT, struct er_STACK *current) {
  CT->c.enriched.verbatim = 0;  
}


static void  ER_bold_enter (content *CT, struct er_STACK *current) {
  current->attrib_flag = sum_fl_attr(current->attrib_flag,FL_BOLD);
}

static void  ER_bold_exit (content *CT, struct er_STACK *current) {

}

static void  ER_underline_enter (content *CT, struct er_STACK *current) {
  current->attrib_flag = sum_fl_attr(current->attrib_flag,FL_UNDER);
}

static void  ER_underline_exit (content *CT, struct er_STACK *current) {

}

static void  ER_indent_enter (content *CT, struct er_STACK *current) {
  current->indent += 1;
  if (0 != CT->last_line_X) {
    print_to_window(CT->win,
		    CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    1,&CH_IND_B);
    CT->last_line_X += 1;
  }


}

static void  ER_indent_exit (content *CT, struct er_STACK *current) {
  if (0 != CT->last_line_X) {
    print_to_window(CT->win,
		    CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    1,&CH_IND_E);
    CT->last_line_X += 1;
  }

}

static void  ER_indentright_enter (content *CT, struct er_STACK *current) {
  current->submargin += 1;
  if (0 != CT->last_line_X) {
    print_to_window(CT->win,
		    CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    1,&CH_MAR_B);
    CT->last_line_X += 1;
  }
}

static void  ER_indentright_exit (content *CT, struct er_STACK *current) {
  if (0 != CT->last_line_X) {
    print_to_window(CT->win,
		    CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    1,&CH_MAR_E);
    CT->last_line_X += 1;
  }
}

static void  ER_param_enter (content *CT, struct er_STACK *current) {
  current->param = 1;
}

static void  ER_param_exit (content *CT, struct er_STACK *current) {

}


static void  RT_reject_enter (content *CT, struct er_STACK *current) {
  current->param = 3;
}

static void  RT_reject_exit (content *CT, struct er_STACK *current) {

}


static void  RT_comment_enter (content *CT, struct er_STACK *current) {
  current->param = 2;
}

static void  RT_comment_exit (content *CT, struct er_STACK *current) {

}


static void  ER_nofill_enter (content *CT, struct er_STACK *current) {
  current->nofill = 1;
  current->justify = JUST_left;
  current->force_wrap = 0;
}

static void  ER_nofill_exit (content *CT, struct er_STACK *current) {

}

static void justification_update(content *CT) {
  int A,I,S,J,E1,E2,W;
  /* set justification */
  get_line_status(CT->win,CT->last_line_Y,&A,&I,&S,&J,&E1,&E2,&W);
  if (J != CT->c.enriched.stack_top->justify) {
    J = CT->c.enriched.stack_top->justify;
    set_line_status(CT->win,CT->last_line_Y,A,I,S,J,E1,E2,W);
  }
}

static void do_line_break(content *CT) {
  justification_update(CT);
  scroll_it(CT->win);
  CT->last_line_Y++;
  CT->last_line_X = 0;
  CT->c.enriched.spc_count = 1; /* Eat spaces also in beginnin of line */
  clear_window_line(CT->win,
		    CT->last_line_Y);    
}

static void force_line_break(content *CT) {
  if (0 != CT->last_line_X) {
    do_line_break(CT);
  }
}

static void  ER_center_enter (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
  current->justify = JUST_center;
}

static void  ER_center_exit (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
}

static void  ER_flushleft_enter (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
  current->justify = JUST_left;
}

static void  ER_flushleft_exit (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
}

static void  ER_flushright_enter (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
  current->justify = JUST_right;
}

static void  ER_flushright_exit (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
}

static void  ER_flushboth_enter (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
  current->justify = JUST_both;
}

static void  ER_flushboth_exit (content *CT, struct er_STACK *current) {
  if (CT->c.enriched.stack_top->nofill) return;

  force_line_break(CT);
}

static void  ER_excerpt_enter (content *CT, struct er_STACK *current) {
  force_line_break(CT);

  current->excerpt_count++;
  current->excerpt_flag = current->attrib_flag;;
}

static void  ER_excerpt_exit (content *CT, struct er_STACK *current) {

  force_line_break(CT);
}

static void  RT_signature_enter (content *CT, struct er_STACK *current) {
  CHAR_IDX mapped[2];
  force_line_break(CT);
  do_line_break(CT);

  current -> attrib_flag = FL_DIM;
  current->nofill = 1;
  current->justify = JUST_left;
  current->force_wrap = 0;

  map_input(MAP_ASCII,2,rCs("--"),mapped);

  print_to_window(CT->win,
		  CT->last_line_X+1,
		  CT->last_line_Y,
		  CT->c.enriched.stack_top->attrib_flag,
		  2,mapped);
  CT->last_line_X += 2;

  CT->c.enriched.spc_count = 0;

  do_line_break(CT);	

}

static void  RT_signature_exit (content *CT, struct er_STACK *current) {
  force_line_break(CT);
  do_line_break(CT);

}


static void  RT_paragraph_enter (content *CT, struct er_STACK *current) {
  force_line_break(CT);

  current -> pindent = 3;

}

static void  RT_paragraph_exit (content *CT, struct er_STACK *current) {
  current -> pindent = 0;
  force_line_break(CT);
  do_line_break(CT);
  CT->c.enriched.no_nl = 1;  
}


static void attrib_update(content *CT) {
  int A,I,S,J,E1,E2,W;

  get_line_status(CT->win,CT->last_line_Y,&A,&I,&S,&J,&E1,&E2,&W);
  if (A != CT->c.enriched.stack_top->pindent ||
      I != CT->c.enriched.stack_top->indent ||
      S != CT->c.enriched.stack_top->submargin ||
      E1 != CT->c.enriched.stack_top->excerpt_count ||
      E2 != CT->c.enriched.stack_top->excerpt_flag ||
      W != CT->c.enriched.stack_top->force_wrap) {
    A = CT->c.enriched.stack_top->pindent;
    I = CT->c.enriched.stack_top->indent;
    S = CT->c.enriched.stack_top->submargin;
    E1 = CT->c.enriched.stack_top->excerpt_count;
    E2 = CT->c.enriched.stack_top->excerpt_flag;
    W = CT->c.enriched.stack_top->force_wrap;
    set_line_status(CT->win,CT->last_line_Y,A,I,S,J,E1,E2,W);
  }
}

static void  RT_superscript_enter (content *CT, struct er_STACK *current) {

  if (0 == CT->last_line_X)
    attrib_update(CT);
  print_to_window(CT->win,
		  CT->last_line_X+1,
		  CT->last_line_Y,
		  CT->c.enriched.stack_top->attrib_flag,
		  1,&CH_SUP_M);
  CT->last_line_X++;
  CT->c.enriched.spc_count = 0;
}

static void  RT_superscript_exit (content *CT, struct er_STACK *current) {

  if (0 == CT->last_line_X)
    attrib_update(CT);
  print_to_window(CT->win,
		  CT->last_line_X+1,
		  CT->last_line_Y,
		  CT->c.enriched.stack_top->attrib_flag,
		  1,&CH_END_M);
  CT->last_line_X++;
  CT->c.enriched.spc_count = 0;
}


static void  RT_subscript_enter (content *CT, struct er_STACK *current) {

  if (0 == CT->last_line_X)
    attrib_update(CT);
  print_to_window(CT->win,
		  CT->last_line_X+1,
		  CT->last_line_Y,
		  CT->c.enriched.stack_top->attrib_flag,
		  1,&CH_SUB_M);
  CT->last_line_X++;
  CT->c.enriched.spc_count = 0;
}

static void  RT_subscript_exit (content *CT, struct er_STACK *current) {

  if (0 == CT->last_line_X)
    attrib_update(CT);
  print_to_window(CT->win,
		  CT->last_line_X+1,
		  CT->last_line_Y,
		  CT->c.enriched.stack_top->attrib_flag,
		  1,&CH_END_M);
  CT->last_line_X++;
  CT->c.enriched.spc_count = 0;
}

#define ER_main_f 0
#define ER_unknown_f 1
#define ER_verbatim_f 2
#define RT_comment_f 3

#define M_er 1
#define M_rt 2

static struct ER_functions {
  int mode;
  char *name;
  ER_function *enter_region;
  ER_function *exit_region;
} ER_functions[] = {
  { M_er|M_rt,  NULL,       ER_main_enter,      ER_main_exit },
  { M_er|M_rt,  NULL,       ER_unknown_enter,   ER_unknown_exit },
  { M_er,       "Verbatim", ER_verbatim_enter,  ER_verbatim_exit },
  { M_rt,       "Comment",  RT_comment_enter,   RT_comment_exit },

  { M_er|M_rt,  "Bold",     ER_bold_enter,      ER_bold_exit },
  { M_er|M_rt,  "Italic",   ER_underline_enter, ER_underline_exit },
  { M_er|M_rt,  "Underline",ER_underline_enter, ER_underline_exit },
  { M_er|M_rt,  "Indent",   ER_indent_enter,    ER_indent_exit },
  { M_er|M_rt,  "IndentRight",ER_indentright_enter,ER_indentright_exit },
  { M_er,       "Param",    ER_param_enter,     ER_param_exit },
  { M_er,       "NoFill",   ER_nofill_enter,    ER_nofill_exit },
  { M_er|M_rt,  "Center",   ER_center_enter,    ER_center_exit },
  { M_er|M_rt,  "FlushLeft",ER_flushleft_enter, ER_flushleft_exit },
  { M_er|M_rt,  "FlushRight",ER_flushright_enter,ER_flushright_exit },
  { M_er,       "FlushBoth",ER_flushboth_enter, ER_flushboth_exit },
  { M_er|M_rt,  "Excerpt",  ER_excerpt_enter,   ER_excerpt_exit },
  { M_rt,       "Paragraph",RT_paragraph_enter, RT_paragraph_exit },
  { M_rt,       "Signature",RT_signature_enter, RT_signature_exit },
  { M_rt,       "Heading",  RT_reject_enter,    RT_reject_exit },
  { M_rt,       "Footing",  RT_reject_enter,    RT_reject_exit },
  { M_rt,       "Subscript", RT_subscript_enter, RT_subscript_exit },
  { M_rt,       "Superscript",RT_superscript_enter,RT_superscript_exit },

};

static void push_er_stack(content *CT, int fnc,CHAR *main) {
  struct er_STACK *prev = CT->c.enriched.stack_top;
  struct er_STACK *new = (struct er_STACK *) MALLOC(sizeof (struct er_STACK));
  CHAR *ptr = main ? concat_text(NULL,main) : NULL;

  if (DEBUG_ENRICHED) {
    print_debug("ENRICHED/RICHTEXT: -> %d: %s",fnc,Cs(main) ? Cs(main) : "");
  }

  new-> previous = prev;
  new-> er_function = fnc;
  new-> name = ptr;
  new-> cs = NULL_chtable;
  if (prev) {    
    copy_table(&(new -> cs),&(prev -> cs));
    new -> indent = prev -> indent;
    new -> pindent = prev -> pindent;
    new -> submargin = prev -> submargin;
    new -> param = prev -> param;
    new -> justify = prev -> justify;
    new -> nofill = prev -> nofill;
    new -> excerpt_count = prev -> excerpt_count;
    new -> excerpt_flag = prev -> excerpt_flag;
    new -> force_wrap = prev -> force_wrap;
    new -> attrib_flag = prev -> attrib_flag;
  } else {
    copy_table(&(new -> cs),&(CT -> cs));
    new -> indent = 0;
    new -> pindent = 0;
    new -> submargin = 0;
    new -> param = 0;
    new -> justify = enable_fulljust ? JUST_both : JUST_left;
    new -> nofill = 0;
    new -> excerpt_count = 0;
    new -> excerpt_flag = 0;
    new -> force_wrap = 1;
    new -> attrib_flag = 0;
  }
  CT->c.enriched.stack_top = new;
  if (2 == CT->c.enriched.stack_top->param && fnc != RT_comment_f) {
    if (DEBUG_ENRICHED) 
      print_debug("ENRICHED/RICHTEXT -> richtext comment: enter not executed");

  } else {
    ER_functions[fnc].enter_region(CT,new);
  }
}

static void pop_er_stack(content *CT) {
  struct er_STACK *current = CT->c.enriched.stack_top;
  struct er_STACK *prev = current->previous;

  if (DEBUG_ENRICHED) {
    print_debug("ENRICHED/RICHTEXT <- %d: %s",
		current->er_function,Cs(current->name) ? 
		Cs(current->name) : "");
  }


  if (2 == CT->c.enriched.stack_top->param && 
      current->er_function != RT_comment_f) {
    if (DEBUG_ENRICHED) 
      print_debug("ENRICHED/RICHTEXT <- richtext comment: exit not executed");

  } else {
    ER_functions[current->er_function].exit_region(CT,current);
  }
  close_table(&(current->cs));
  if (current->name) FREE(current->name);
  FREE((void *)current);
  CT->c.enriched.stack_top = prev;
}

static void unroll_er_stack(content *CT, int fnc) {
  while(CT->c.enriched.stack_top->er_function != fnc &&
	CT->c.enriched.stack_top->previous != NULL)
    pop_er_stack(CT);
}

static void unroll_er_stack_n(content *CT, CHAR *fnc) {
  while((!CT->c.enriched.stack_top->name ||
	 0 != strcasecmp(Cs(CT->c.enriched.stack_top->name),Cs(fnc))) &&
	CT->c.enriched.stack_top->previous != NULL)
    pop_er_stack(CT);
}

static int ER_basic_handle(content *CT,int len, CHAR *buffer,int flush) {
  CT->c.enriched.no_nl = 0;
  if (1 < CT->c.enriched.stack_top->param) {
    /* richtext comment or heading/footing - ignore */

    return len;
  } else if (CT->c.enriched.stack_top->param) {
    /* param not currently stored */

    /* mast also add handling for multibyte text to here !!! */

    return len;
  } else {
    int ret,mlen;
    CHAR_IDX *mapped = (CHAR_IDX *)MALLOC(len * sizeof(CHAR_IDX));
	
    if (CT->last_line_X == 0) {
      attrib_update(CT);
    }
	
    /* in richtext (but not in enriched) changin of charset is possible */
    ret=map2_input(&(CT->c.enriched.stack_top->cs),len,buffer,
		   &mlen,mapped,CT->multi,flush); 
    if (0 == ret) {
      CT->multi = !CT->multi;
      FREE(mapped);
      return 0;
    }
    else if (ret <0) { 
      StrSet2(&(CT->multi_buffer),len,buffer);
      FREE(mapped);
      return -1;
    } 
	
    print_to_window(CT->win,
		    CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    mlen,mapped);
    CT->last_line_X += mlen;
    CT->c.enriched.spc_count = 0;
    
    FREE(mapped);
    return ret;
  }
}

static int ER_handle_multi(content *CT,int len, CHAR *buffer,int flush) {
  int ret = ER_basic_handle(CT,len,buffer,flush);
  if (ret < 0) return len;
  return ret;
}


static int handle_common(content *CT,int len, CHAR *buffer,
			 CHAR **ch1, CHAR **ch2, 
			 int (*handle_it)(content *CT,CHAR ch),
			 int (*handle_sp)(content *CT,CHAR ch)) {
  CHAR *ch3;
  
  for ((*ch2) = (*ch1); 
       (*ch2) < buffer + len && handle_it(CT,*(*ch2)); 
       (*ch2)++);
  
  ch3 = (*ch2);
  for (; (*ch2) < buffer + len && !handle_sp(CT,*(*ch2)); (*ch2)++);
  
  if ((*ch2) != ch3) {
    int ret = ER_basic_handle(CT,(*ch2)-ch3,ch3,0);
    if (ret < 0) { CrashMe ("multibyte store with no multi data"); }
    if (0 == ret) return 0;
    (*ch2) = ch3 + ret;
  }    
  return 1;
}

static void add_space(content *CT) {
  if (CT->last_line_X == 0) {
    attrib_update(CT);
  }
  CT->c.enriched.spc_count++;
  if (CT->c.enriched.spc_count == 1 || CT->c.enriched.verbatim ||
      CT->c.enriched.stack_top->nofill)
    print_to_window(CT->win,CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    1,&CH_SPC);
  else 
    print_to_window(CT->win,CT->last_line_X+1,
		    CT->last_line_Y,
		    CT->c.enriched.stack_top->attrib_flag,
		    1,&CH_SPC_C);
  CT->last_line_X++;
  CT->c.enriched.no_nl = 0;
}

/* Enriched content-type --------------------------------------------------- */

static int er_special_char(content *CT,CHAR ch) {
  if (' ' == ch) return 1;
  if (StrLEN(CT->c.enriched.cmd_buffer) > 1) return 0;
  if ('<' == ch) return 1;
  return 0;
}

static int cmd_handle(content *CT,CHAR ch) {
  if (0 == StrLEN(CT->c.enriched.cmd_buffer)) return 0;
  if (1 == StrLEN(CT->c.enriched.cmd_buffer) && '<' == ch) return 0;
  if (1 == StrLEN(CT->c.enriched.cmd_buffer) && '/' == ch) {      
    StrAdd(&(CT->c.enriched.cmd_buffer),ch);
    return 1;
  }
  if (isdigit(ch) || ('a' <= ch && ch <= 'z') || ('A' <= ch && ch <= 'Z')
      || '-' == ch) {
    StrAdd(&(CT->c.enriched.cmd_buffer),ch);
    return 1;
  }


  if (CT->c.enriched.verbatim &&  '>' == ch &&
      0 == strncasecmp("</verbatim",
		       Cs(StrPTR(CT->c.enriched.cmd_buffer)),10)) {

    if (DEBUG_ENRICHED) {
      print_debug("ENRICHED token: </verbatim>");
    }


      unroll_er_stack(CT,ER_verbatim_f);
      if (CT->c.enriched.stack_top->previous != NULL)
	pop_er_stack(CT);

    StrFree(&(CT->c.enriched.cmd_buffer));
    return 1;
  }
  
  if (!CT->c.enriched.verbatim && '>' == ch) {
    CHAR *ptr;
    int neg = 0;

    ptr = StrPTR(CT->c.enriched.cmd_buffer);

    if (DEBUG_ENRICHED) {
      print_debug("ENRICHED token: %s>",ptr);
    }
    
    ptr++;

    if ('/' == *ptr) {
      neg = 1;
      ptr++;
    }

    if (neg) {
      unroll_er_stack_n(CT,ptr);
      if (CT->c.enriched.stack_top->previous != NULL)
	pop_er_stack(CT);
    } else {
      int fnc = ER_unknown_f,i;
      
      for (i = 0; i < sizeof(ER_functions) / sizeof(struct ER_functions); i++)
	if ((ER_functions[i].mode & M_er) == M_er && 
	    ER_functions[i].name &&
	    0 == strcasecmp(Cs(ptr),ER_functions[i].name)) 
	  fnc = i;
      
      push_er_stack(CT,fnc,ptr);
    }
    
    StrFree(&(CT->c.enriched.cmd_buffer));
    return 1;
  }

  flush_cmd_buffer(CT);
  return 0;
}

static int ER_handle(content *CT,int len, CHAR *buffer) {
  CHAR *ch1,*ch2;

  set_win_prompt(-1,-1,-1);

  if (1 == CT->c.enriched.nl_count) {
    add_space(CT); 
  }
  CT->c.enriched.nl_count = 0;

  for (ch1 = buffer; ch1 < buffer + len; ch1 = ch2) {

    if (!handle_common(CT,len,buffer,&ch1,&ch2,cmd_handle,er_special_char))
      break;
       
    for (;ch2 < buffer + len && er_special_char(CT,*ch2); ch2++) {
      if (' ' == *ch2) add_space(CT);
      else if (StrLEN(CT->c.enriched.cmd_buffer)) {
	if (CT->c.enriched.stack_top->param) {
	  /* param not currently stored */

	  StrFree(&(CT->c.enriched.cmd_buffer));
	} else {
	  if (CT->c.enriched.verbatim) 
	    StrAdd(&(CT->c.enriched.cmd_buffer),'<');
	  flush_cmd_buffer(CT);
	}
      } else StrAdd(&(CT->c.enriched.cmd_buffer),'<');
    }
    scroll_it(CT->win); /* for autowrap */
  }
  return ch1-buffer;
}

static void ER_handle_eoln(content *CT) {
  CT->c.enriched.no_nl = 0;

  if (CT->c.enriched.verbatim || CT->c.enriched.stack_top->nofill) {
    do_line_break(CT);
    return;
  }

  if (CT->c.enriched.stack_top->param) {
    /* Storing of param not implemented */
    return;
  }

  CT->c.enriched.nl_count++;
  if (CT->c.enriched.nl_count > 1) {
    do_line_break(CT);
  }
}

static void ER_init (content *CT) {
  CT->c.enriched.magic = CT_enriched;
  StrInit(&(CT->c.enriched.cmd_buffer),60);
  CT->c.enriched.nl_count = 0;
  CT->c.enriched.spc_count = 1;   /* eats spaces also in beginning of line */
  CT->c.enriched.no_nl = 0;
  CT->c.enriched.verbatim = 0;
  CT->c.enriched.stack_top = NULL;
  push_er_stack(CT,ER_main_f,NULL);
}

static void ER_exit(content *CT) {

  if(StrLEN(CT->c.enriched.cmd_buffer))
    flush_cmd_buffer(CT);
  StrFree(&(CT->c.enriched.cmd_buffer));

  justification_update(CT);

  unroll_er_stack(CT,ER_main_f);
  pop_er_stack(CT);
  if (NULL != CT->c.enriched.stack_top) {
    CrashMe("ER_exit: stack not empty - FATAL error ");
  }
  CT->c.enriched.magic = -1;
}

static void ER_update(content *CT) {
  justification_update(CT);
}

/* Richtext content-type --------------------------------------------- */

static int rt_special_char(content *CT,CHAR ch) {
  if (' ' == ch) return 1;
  if (StrLEN(CT->c.enriched.cmd_buffer) > 0) return 0;
  if ('<' == ch) return 1;
  return 0;
}

static int rt_cmd_handle(content *CT,CHAR ch) {
  if (0 == StrLEN(CT->c.enriched.cmd_buffer)) return 0;

  if (1 == StrLEN(CT->c.enriched.cmd_buffer) && '/' == ch) {      
    StrAdd(&(CT->c.enriched.cmd_buffer),ch);
    return 1;
  }
  if (isdigit(ch) || ('a' <= ch && ch <= 'z') || ('A' <= ch && ch <= 'Z')
      || '-' == ch) {
    StrAdd(&(CT->c.enriched.cmd_buffer),ch);
    return 1;
  }

  if ('>' == ch) {
    CHAR *ptr;
    int neg = 0;

    ptr = StrPTR(CT->c.enriched.cmd_buffer);


    if (DEBUG_ENRICHED) {
      print_debug("RICHTEXT token: %s>",ptr);
    }
    
    ptr++;

    if (0 == strcasecmp(Cs(ptr),"lt")) {
      if (CT->last_line_X == 0) 
	attrib_update(CT);

      print_to_window(CT->win,CT->last_line_X+1,
		      CT->last_line_Y,
		      CT->c.enriched.stack_top->attrib_flag,
		      1,&CH_LT);
      CT->last_line_X++;
      CT->c.enriched.spc_count = 0;
      
      StrFree(&(CT->c.enriched.cmd_buffer));
      return 1;
      
    } else if (0 == strcasecmp(Cs(ptr),"nl") || 
	       0 == strcasecmp(Cs(ptr),"np")) {
      do_line_break(CT);
      CT->c.enriched.no_nl = 1;      

      if (0 == strcasecmp(Cs(ptr),"np")) {
	print_to_window(CT->win,
			CT->last_line_X+1,
			CT->last_line_Y,
			CT->c.enriched.stack_top->attrib_flag,
			1,&CH_NP);
	CT->last_line_X += 1;
      }

      StrFree(&(CT->c.enriched.cmd_buffer));
      return 1;
    }
    CT->c.enriched.no_nl = 0;      

    if ('/' == *ptr) {
      neg = 1;
      ptr++;
    }

    if (neg) {
      unroll_er_stack_n(CT,ptr);
      if (CT->c.enriched.stack_top->previous != NULL)
	pop_er_stack(CT);
    } else {
      int fnc = ER_unknown_f,i;
      
      for (i = 0; i < sizeof(ER_functions) / sizeof(struct ER_functions); i++)
	if ((ER_functions[i].mode & M_rt) == M_rt && 
	    ER_functions[i].name &&
	    0 == strcasecmp(Cs(ptr),ER_functions[i].name)) 
	  fnc = i;

      push_er_stack(CT,fnc,ptr);

      if (fnc == ER_unknown_f) {
	int map = find_mime_map(ptr);
	if (-1 != map) {
	  if (!open_table(&(CT->c.enriched.stack_top->cs),map,NULL))
	    copy_table(&(CT->c.enriched.stack_top->cs), &(CT->cs));
	}
      }
    }
    
    StrFree(&(CT->c.enriched.cmd_buffer));
    return 1;
  }
  
  flush_cmd_buffer(CT);
  return 0;

}

static void RT_init (content *CT) {
  CT->c.enriched.magic = CT_richtext;
  StrInit(&(CT->c.enriched.cmd_buffer),40);
  CT->c.enriched.nl_count = 0;
  CT->c.enriched.spc_count = 1;  /* eat spacesc also in beginning of line */
  CT->c.enriched.no_nl = 0;
  CT->c.enriched.verbatim = 0;
  CT->c.enriched.stack_top = NULL;
  push_er_stack(CT,ER_main_f,NULL);
}

static void RT_handle_eoln(content *CT) {
  if (CT->c.enriched.stack_top->param) return;
  if (!CT->c.enriched.no_nl) add_space(CT); 
  CT->c.enriched.no_nl = 0;
}

static int RT_handle(content *CT,int len, CHAR *buffer) {
  CHAR *ch1,*ch2;

  set_win_prompt(-1,-1,-1);

  for (ch1 = buffer; ch1 < buffer + len; ch1 = ch2) {

    if (!handle_common(CT,len,buffer,&ch1,&ch2,rt_cmd_handle,rt_special_char))
      break;
       
    for (;ch2 < buffer + len && rt_special_char(CT,*ch2); ch2++) {
      if (' ' == *ch2) add_space(CT);
      else StrAdd(&(CT->c.enriched.cmd_buffer),'<');
    }
    scroll_it(CT->win); /* for autowrap */
  }
  return ch1-buffer;
}


/* NULL content-type ------------------------------------------------- */

static void NL_init (content *CT) {
  CrashMe("NL_init");
}

static void NL_handle_eoln(content *CT) {
  CrashMe("NL_handle_eoln");
}


static void NL_exit(content *CT) {
  CrashMe("NL_exit");
}

static void NL_update(content *CT) {
  CrashMe("NL_update");
}

static int NL_handle(content *CT,int len, CHAR *buffer) {
  CrashMe("NL_handle");
}

static int NL_handle_multi(content *CT,int len, CHAR *buffer, int flush) {
  CrashMe("NL_handle_multi");
}

/* ------------------------------------------------------------------- */

static content content_models[] = {
  /* CT_plain */
  { CT_plain, NULL, -1, -1, 0,
      PL_init, PL_handle, PL_handle_multi,
      PL_handle_eoln, PL_exit, PL_update, 1, 0 },
  /* CT_terminal */
  { CT_terminal, NULL, -1, -1, 0,
      TR_init, TR_handle, TR_handle_multi,
      TR_handle_eoln, TR_exit, TR_update, 1, 0 },
  /* CT_enriched */
  { CT_enriched, NULL, -1, -1, 0,
      ER_init, ER_handle, ER_handle_multi,
      ER_handle_eoln, ER_exit, ER_update, 1, 0 },
  /* CT_richtext */
  { CT_richtext, NULL, -1, -1, 0,
      RT_init, RT_handle, ER_handle_multi,
      RT_handle_eoln, ER_exit, ER_update, 1, 0 },
};

const content NULL_content = 
{ -1, NULL, -1, 0, 0,
    NL_init, NL_handle, NL_handle_multi,
    NL_handle_eoln, NL_exit, NL_update, 1, 0 };

static struct CN {
  int mime;
  char *name;
  int content;
} ct_names[] = {
  { 0,  "Terminal",     CT_terminal },
  { 0,  "Plain",        CT_plain },
  { 1,  "Text/Plain",   CT_plain },
  { 0,  "Enriched",     CT_enriched },
  { 1,  "Text/Enriched",CT_enriched },
  { 0,  "Richtext",     CT_richtext },
  { 1,  "Text/Richtext",CT_richtext },

  { -1, NULL, -1 }
};

int search_content(CHAR *name,int mime) {
  int i;
  for (i=0; NULL != ct_names[i].name; i++)
    if ((!mime || ct_names[i].mime) &&
	0 == strcasecmp(Cs(name),ct_names[i].name))
      return ct_names[i].content;
  return -1;
}

CHAR *content_name(int content, int mime) {
  int i;
  for (i=0; NULL != ct_names[i].name; i++)
    if (mime == ct_names[i].mime &&
	content == ct_names[i].content)
      return rCs(ct_names[i].name);
  return rCs("Unknown");
}

INLINE static void turn_flag(content *CT) {
  if (!CT->cs_flag) {
    StrInit(&(CT->multi_buffer),10);
    CT->cs=NULL_chtable;
    CT->cs_flag=1;
  }
}

void init_content(int content_type,int *fp,int win, 
		  int cs, CHAR *cs_param,
		  content *CT) {
  *CT = content_models[content_type];
  turn_flag(CT);
  CT->fp = fp;
  CT->win=win;
  if (!open_table(&(CT->cs),cs,cs_param)) {
    int ok = 0;
    if (cs_param) {
      if (open_table(&(CT->cs),cs,NULL)) {
	print_notify("(%s) Unsupported charset: %s; %s\n"
		     "     Using charset: %s",
		     prog,map_name(cs,0),cs_param,
		     map_name(cs,0));
	ok = 1;	
      } else {
	print_notify("(%s) Unsupported charset: %s; %s\n"
		     "\tUsing charset: %s",
		     prog,map_name(cs,0),cs_param,
		     map_name(MAP_ASCII,0));
      }
    }
    else
      print_notify("(%s) Unsupported charset: %s\n",
		   "\tUsing charset: %s",
		   prog,map_name(cs,0),
		   map_name(MAP_ASCII,0));
    if (!ok && !open_table(&(CT->cs),MAP_ASCII,NULL))
      CrashMe("Opening of Ascii failed !");
  }
  CT->multi = 0;
  CT->last_line_Y = 1;
  CT->last_line_X = 0;
  clear_window(CT->win);    
  CT->init(CT);
}


void handle_content(content *CT, int len, CHAR *buffer) {
  int ret=0;
  int rlen=len;
  CHAR *rbuffer = buffer,*ptr;
  turn_flag(CT);
  if (StrLEN(CT->multi_buffer) > 0) {
    rbuffer = MALLOC(len+StrLEN(CT->multi_buffer));
    memcpy((void *)rbuffer,(void *)buffer,len);
    memcpy((void *)(rbuffer+len),(void *)StrPTR(CT->multi_buffer),
	   StrLEN(CT->multi_buffer));
    rlen = len + StrLEN(CT->multi_buffer);
    StrFree(&(CT->multi_buffer));
  }
  
  for (ptr=rbuffer;ptr < rbuffer+rlen; ptr += ret) {
    if (CT->multi) 
      ret=CT->handle_multi(CT,rlen-(ptr-rbuffer),ptr,0);
    else
      ret=CT->handle(CT,rlen-(ptr-rbuffer),ptr);
  }

  if (rbuffer != buffer) FREE(rbuffer);
}

static void flush_pending_multi(content *CT) {
  if (StrLEN(CT->multi_buffer) > 0) {
    int ret=0;
    CHAR *ptr;
    for (ptr=StrPTR(CT->multi_buffer);
	 ptr < StrPTR(CT->multi_buffer)+StrLEN(CT->multi_buffer); ptr += ret) {
      if (!CT->multi) { CrashMe("Pending data with no multi mode!!"); }
      ret=CT->handle_multi(CT,
			   StrLEN(CT->multi_buffer)
			   -(ptr-StrPTR(CT->multi_buffer)),ptr,1);
    }
    StrFree(&(CT->multi_buffer));		      
  }
}

void eoln_content(content *CT) {
  turn_flag(CT);
  flush_pending_multi(CT);
  CT->handle_eoln(CT);
}

void exit_content(content *CT) {
  turn_flag(CT);
  flush_pending_multi(CT);
  turn_flag(CT);
}

void update_content(content *CT) {   /* shoft eof - eof detected, but
				        here can be more data incoming 
					when reread */
  turn_flag(CT);
  /* can here to be flush_pending_multi(CT) ? */
  CT->update(CT);
}
