/*
  Definitionsdateien
*/
#include <ii/i_dl.h>
#include <ii/i_su.h>
#include <ii/i_vm.h>
#include <ii/i_sfm.h>

#include <limits.h>

#include <sys/log.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <sys/tihdr.h>

#define DRIVER
#include "vaxmhs_data.h"

/*
  STREAMS Funktionen
*/
static int sosopen(),sosclose(),sosuwput(),soslrput(),sosursrv();

/*
  STREAMS Informationsblocks
*/
static struct module_info sos_minfo = 
       {
	0, "sos", 0, sizeof(MAILBOX), 10000, 4000 
       };

static struct qinit sos_urinit =
       {
	NULL, sosursrv, sosopen, sosclose, NULL, &sos_minfo, NULL
       };

static struct qinit sos_uwinit =
       {
	sosuwput, NULL, NULL, NULL, NULL, &sos_minfo, NULL
       };

static struct qinit sos_lrinit =
       {
	soslrput, NULL, NULL, NULL, NULL, &sos_minfo, NULL
       };

static struct qinit sos_lwinit =
       {
	NULL, NULL, NULL, NULL, NULL, &sos_minfo, NULL
       };

struct streamtab sosinfo =
       {
	&sos_urinit, &sos_uwinit, &sos_lrinit, &sos_lwinit
       };

/*
  Devicetreiber Funktionen
*/
static status_type sos_configure();
static status_type sos_name_to_device(),sos_device_to_name();

/*
  Devicetreiber Informationsblocks
*/
su_driver_routines_vector_type sfm_sos_routines_vector =
			       {
				SU_ROUTINES_VECTOR_VERSION_1,
				sfm_nodevice_init,
				sos_configure,sfm_nodevice_deconfigure,
				sos_name_to_device,sos_device_to_name,
			       };

/*
  Konfigurierbare Variablen
*/
extern uint32_type cf_sos_chan_cnt;
extern uint16_type cf_sos_protocoll_type;

/*
  Macros
*/
#define ISIOC(x)	(((x)-xids) == IOCXID)
#define IOCRD		(xids[IOCXID].uread)

#define MALLOC(t,n)	((t *)vm_get_wired_memory(usizeof(t)*(n),VM_DEFAULT_ALIGNMENT))

#define MCAST(n)	((logaddr *)(((char *)mcast)+(n)*MC_SIZE))

#define MC_INDEX(n)	((n)>>5)
#define MC_BIT(n)	((n)&31)
#define MC_MASK(n)	(1<<MC_BIT(n))

#define MC_SET(l,n)	((l)->users[MC_INDEX(n)] |= MC_MASK(n))
#define MC_CLR(l,n)	((l)->users[MC_INDEX(n)] &= ~MC_MASK(n))
#define MC_ISSET(l,n)	((l)->users[MC_INDEX(n)]&MC_MASK(n))
#define MC_ISCLR(l,n)	(!MC_ISSET(l,n))

#define MC_FIELDS	(MC_INDEX(cf_sos_chan_cnt)+1)
#define MC_SIZE		(usizeof(logaddr)+(MC_FIELDS-1)*usizeof(long))
#define MCAST_SIZE	(MC_SIZE*SOS_MULTICASTS)

#define MINNETSIZE	100

#define EXRDBIT		(EXBIT|RDBIT)

/*
  Nicht Konfigurierbare Variablen und Konstanten
*/
typedef struct sosconn
        {
	 queue_t	*uread;
	 mblk_t		*mt;
	 int		tid;
        } exchange;

typedef struct mcinfo
	{
	 char		addr[6];
	 long		users[1];
	} logaddr;

static int32_type confsos = -1,clonesos = -1;
static mblk_t *link = NULL,*mcact = NULL;
static exchange *xids = NULL;
static logaddr *mcast = NULL;
static queue_t *inen = NULL;
static char paddr[6];

static char sos_multicasts[][6] = 
       {
	{ 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },		/* BROADCAST		*/
	{ 0xab, 0x00, 0x04,  'E',  'R',  'R' },		/* ERROR		*/
	{ 0xab, 0x00, 0x04,  'E',  'V',  'E' },		/* EVENT		*/
	{ 0xab, 0x00, 0x04,  'M',  'U',  'O' },		/* MUON			*/
	{ 0xab, 0x00, 0x04,  'R',  'E',  'S' },		/* RESERVED		*/
	{ 0xab, 0x00, 0x04,  'S',  'T',  'A' },		/* STATUS		*/
	{ 0xab, 0x00, 0x04,  'T',  'E',  'S' }		/* TEST			*/
       };
#define SOS_MULTICASTS		(usizeof(sos_multicasts)/usizeof(sos_multicasts[0]))

/*
  Interne Routinen.
*/
static int initinen(),closeinen(),bindmc(),unbindmc(),nouser();
static int settime(),cleartime(),nodata(),timedout();
static int localMail(),sendether(),sendlocal();
static int sendaddr(),freeit();
static int doioctl(),putioc();
static logaddr *findmc();

/*
  Devicetreiber Routinen
*/

/*
  /dev/sos konfigurieren
*/
status_type sos_configure(name,majnum)
char_ptr_type name;
io_major_device_number_type majnum;
{
 char_ptr_type lname = "sos()";
 fs_dev_request_type credev;
 io_device_number_type cdev;
 uint32_type length,dev,n;
 status_type res;

 /* Der Name muss stimmen */
 do
  {
   if ( *lname != *name++ ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;
  }
 while ( *lname++ );
 /* Darf nur einmal konfiguriert werden */
 if ( confsos != -1 ) return IO_ENXIO_DEVICE_IS_ALREADY_CONFIGURED;
 /* Und benoetigt zum Betrieb das 'clone' Device */
 if ( (res = sfm_name_to_device("clone()",&cdev)) != OK ) return res;

 /* Defaultwerte aufsetzen und Speicher fuer die Kanaele reservieren */
 if ( cf_sos_chan_cnt <= 0 ) cf_sos_chan_cnt = 1;
 if ( !cf_sos_protocoll_type ) cf_sos_protocoll_type = VIPTYPE;
 xids = MALLOC(exchange,cf_sos_chan_cnt);

 /* Speicher fuer die Multicastadressen reservieren */
 mcast = (logaddr *)MALLOC(char,MCAST_SIZE);

 /* Situation festhalten */
 confsos = majnum;
 clonesos = cdev.major;

 /* inode fuer /dev/sos als cloned Device erzeugen */
 credev.operation = Fs_Dev_Request_Operation_Create;
 credev.dirname[0] = '\0';
 length = usizeof(credev.filename);
 misc_string_copy("sos",credev.filename,&length);
 credev.op.create.mode_bits = SFM_PIPE_NODE_ACCESS_PERMISSIONS;
 credev.op.create.device.major = clonesos;
 credev.op.create.device.minor = confsos;
 fs_submit_dev_request(&credev);

 /* Kanaele initialisieren */
 for ( dev = cf_sos_chan_cnt ; dev-- ; )
  {
   xids[dev].uread = NULL;
   xids[dev].mt = NULL;
  }

 /* Multicasts initialisieren */
 for ( n = SOS_MULTICASTS ; n-- ; )
  {
   bcopy(sos_multicasts[n],MCAST(n)->addr,6);
   bzero(MCAST(n)->users,MC_FIELDS*usizeof(long));
  }

 /* Erfolg melden */
 return OK;
}

/*
  Devicename in ein Nummernpaar (Major,Minor) umsetzen.
*/
status_type sos_name_to_device(name,nump)
char_ptr_type name;
io_device_number_ptr_type nump;
{
 char_ptr_type lname = "sos(";
 uint32_type dev;

 if ( (confsos == -1) || (clonesos == -1) ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

 while ( *lname )
  if ( *name++ != *lname++ )
   return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

 for ( lname = name ; (*name >= '0') && (*name <= '9') ; name++ );

 if ( lname == name )
  {
   nump->major = clonesos;
   nump->minor = confsos;
  }
 else
  {
   if ( (name-lname) > 5 ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

   for ( dev = 0 ; lname < name ; ) dev = 10*dev+*lname++-'0';

   if ( dev >= cf_sos_chan_cnt ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

   nump->major = confsos;
   nump->minor = dev;   
  }

 if ( *name && ((*name++ != ')') || *name) ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

 return OK;
}

/*
  Nummernpaar (major,minor) in einen Devicenamen umsetzen
*/
status_type sos_device_to_name(dev,namep,size)
io_device_number_type dev;
char_ptr_type namep;
uint32_type size;
{
 if ( (dev.major == clonesos) && (dev.minor == confsos) )
  misc_string_copy("sos()",namep,&size);
 else
  {
   if ( dev.major != confsos ) return IO_ENXIO_DEVICE_IS_NOT_CONFIGURED;

   misc_format_line(namep,size,"sos(%d)",(uint32_type)dev.minor);
  }

 return OK;
}

/*
  STREAMS Routinen
*/

/*
  Oeffnen des Devices
*/
static int sosopen(q,dev,flag,sflag)
queue_t *q;
int dev,flag,sflag;
{

 /* Bei einem cloned open freien Eintrag suchen */
 if ( sflag != CLONEOPEN )
  dev = minor(dev);
 else
  for ( dev = 0 ; (dev < cf_sos_chan_cnt) && xids[dev].uread ; dev++ );

 /* Eintrag verifizieren */
 if ( ((dev != IOCXID) && (!inen || link)) || (dev >= cf_sos_chan_cnt) ) return OPENFAIL;

 /* Informationen festhalten */
 q->q_ptr = WR(q)->q_ptr = (char *)(xids+dev);

 /* Devicekanal reservieren */
 xids[dev].uread = q;
 xids[dev].mt = NULL;

 /* Hardwareadresse melden */
 if ( dev != IOCXID ) sendaddr(q);
 
 /* Hat funktioniert */
 return dev;
}

/*
  Schliessen des Devices
*/
static int sosclose(q,oflag)
queue_t *q;
int oflag;
{
 exchange *xid;
 logaddr *mc;
 int n,xn;

 /* Aktuellen Kanal ermitteln */
 xid = (exchange *)q->q_ptr;

 /* Timeout loeschen */
 cleartime(xid);

 /* Kanal freigeben */
 xid->uread = NULL;

 /* Multicasts loeschen */
 for ( xn = xid-xids, n = SOS_MULTICASTS ; n-- ; )
  if ( MC_ISSET(mc = MCAST(n),xn) )
   {
    MC_CLR(mc,xn);
    if ( nouser(mc) ) unbindmc(mc);
   }
}

/* 
  Daten vom Benutzer (upper) entgegennehmen
*/
static int sosuwput(q,mp)
queue_t *q;
mblk_t *mp;
{
 exchange *xid;

 /* Zugeordneten Kanal ermitteln */
 xid = (exchange *)q->q_ptr;

 /* Daten auswerten */
 switch (mp->b_datap->db_type)
  {
   case M_IOCTL : /* Kontrollbefehl auswerten */
     		  doioctl(xid,mp);
		  return;
   case M_FLUSH : /* Datenqueues loeschen */
     		  if ( *mp->b_rptr&FLUSHW ) flushq(q,FLUSHDATA);
		  if ( *mp->b_rptr&FLUSHR )
		   {
		    flushq(RD(q),FLUSHDATA);
		    /* Fuer den STREAM-Head */
		    *mp->b_rptr &= ~FLUSHW;
		    qreply(q,mp);
		   }
		  else
		   /* Fertig */
		   freemsg(mp);
		  return;
   case M_PROTO : /* Eventuell Timeout fuer ein IREQX setzen */
     		  settime(xid,mp);
		  return;
   case M_DATA  : /* Nur, wenn das ETHERNET dranhaengt */
     		  if ( inen && !link )
		   {
     		    /* Werden einfach durchgereicht */
		    localMail(mp,0);
		    /* Fertig */
		    return;
		   }
   default      : /* Unbekannte STREAM-Codes */
     		  freemsg(mp);
		  putctl1(RD(q)->q_next,M_ERROR,EINVAL);
  }
}

/*
  Daten vom Netzwerk (lower) entgegennehmen
*/
static int soslrput(q,mp)
queue_t *q;
mblk_t *mp;
{
 union DL_primitives *dp;
 dl_unitdata_ind_t *ui;
 char header[14];
 logaddr *mc;
 int ok = 0;
 long dev;

 /* Nachrichtentyp ermitteln */
 switch (mp->b_datap->db_type)
  {
   case M_FLUSH   : /* Aktionen eines STREAMS-Heads ausfuehren */
     		    if ( *mp->b_rptr&FLUSHR ) flushq(q,FLUSHDATA);
		    if ( *mp->b_rptr&FLUSHW )
		     {
		      *mp->b_rptr &= ~FLUSHR;
		      qreply(q,mp);
		     }
		    else
		     freemsg(mp);
		    /* Fertig */
		    return;
   case M_ERROR   :
   case M_HANGUP  : /* Kanal darf nicht mehr benutzt werden */
     		    closeinen();
   default        : /* Nachricht ignorieren */
     		    freemsg(mp);
		    return;
   case M_PROTO   :
   case M_PCPROTO : break;
  }
 
 /* Kontrollfeld auswerten */
 dp = (union DL_primitives *)mp->b_rptr;
 ui = &dp->unitdata_ind;

 /* Je nach Zustand verfahren */
 if ( !link )
  switch (dp->dl_primitive)
   {
    case DL_OK_ACK	 : /* Multicastadresse hinzugefuegt oder geloescht */
      			   if ( !mcact ) break;
			   bcopy(mcact->b_cont->b_rptr+6,&dev,4);
			   if ( mc = findmc(mcact->b_cont->b_rptr) )
      			    switch (dp->ok_ack.dl_correct_primitive)
			     {
			      case DL_BIND_MC_REQ   : MC_SET(mc,dev);
			    			      mcact->b_datap->db_type = M_IOCACK;
			        		      break;
			      case DL_UNBIND_MC_REQ : MC_CLR(mc,dev);
			    			      mcact->b_datap->db_type = M_IOCACK;
			        		      break;
			    }
    case DL_ERROR_ACK	 : /* Fehler melden */
			   putioc(&mcact);
			   break;
    case DL_UNITDATA_IND : /* Datenpacket verifizieren */
      			   if ( (ui->dl_dest_addr_length != 8) ||
      			        (ui->dl_src_addr_length != 8) ||
				!ui->dl_dest_addr_offset || !ui->dl_src_addr_offset )
			    break;
      			   /* ETHERNET-Header zusammenstellen */
      			   bcopy(((char *)ui)+ui->dl_dest_addr_offset,header+0,6);
 			   bcopy(((char *)ui)+ui->dl_src_addr_offset,header+6,8);
			   /* Protokollkopf umwandeln */
			   bcopy(header,ui,14);
			   mp->b_datap->db_type = M_DATA;
			   mp->b_wptr = mp->b_rptr+14;
			   /* Mail bearbeiten */
			   localMail(mp,1);
			   /* Fertig */
			   return;
   }
 else
  {
   /* Hardwareadresse auslesen */
   if ( ok = ((dp->dl_primitive == DL_BIND_ACK) &&
              (dp->bind_ack.dl_sap == cf_sos_protocoll_type) &&
	      dp->bind_ack.dl_addr_offset && (dp->bind_ack.dl_addr_length == 8)) )
    {
     /* Adresse kopieren */
     bcopy(((char *)dp)+dp->bind_ack.dl_addr_offset,paddr,6);
     /* Alles in Ordnung */
     link->b_datap->db_type = M_IOCACK;
    }
   /* Fertig */
   putioc(&link);
   /* Hardwareadresse verschicken, falls gesetzt */
   if ( ok && IOCRD ) sendaddr(IOCRD);
  }
 /* Nachricht freigeben */
 freemsg(mp);
}

/*
  Timeouts durchreichen.
*/
static int sosursrv(q)
queue_t *q;
{
 mblk_t *mp;

 /* Ohne Flowcontrol zu beachten weitergeben */
 while ( mp = getq(q) ) putnext(q,mp);
}

/*
  Timeout zu einem Kanal loeschen.
*/
static cleartime(xid)
exchange *xid;
{
 /* Timeout abbrechen */
 if ( !xid->mt ) return;
  
 /* Eintrag loeschen */
 untimeout(xid->tid);
 freeit(&xid->mt);
}

/*
  Physikalische Hardwareadresse als Kontrollpaket an eine Queue schicken. Davor werden
  Flowcontrol Informationen des STREAMS heads modifiziert.
*/
static sendaddr(q)
queue_t *q;
{
 struct stroptions *so;
 mblk_t *mb;

 /* Message allokatieren */
 if ( !(mb = allocb(usizeof(*so),BPRI_MED)) ) return;

 /* Messageblock fuellen und abschicken */
 mb->b_datap->db_type = M_SETOPTS;
 so = (struct stroptions *)mb->b_wptr;
 so->so_flags = SO_HIWAT|SO_LOWAT;
 so->so_hiwat = sos_minfo.mi_hiwat;
 so->so_lowat = sos_minfo.mi_lowat;
 mb->b_wptr += usizeof(*so);
 putnext(q,mb);

 /* Message allokatieren */
 if ( !(mb = allocb(6,BPRI_MED)) ) return;

 /* Messageblock fuellen und abschicken */
 mb->b_datap->db_type = M_PROTO;
 bcopy(paddr,mb->b_wptr,6);
 mb->b_wptr += 6;
 putnext(q,mb);
}

/*
  IOCTL Befehl auswerten.
*/
static doioctl(xid,mp)
exchange *xid;
mblk_t *mp;
{
 int ilen,new = -1,unl = 0,tlen,fields;
 struct linkblk *lp;
 struct iocblk *ip;
 mblk_t *fp,*cp;
 logaddr *mc;
 long dev;
	
 /* Strukturen ermitteln */
 ip = (struct iocblk *)mp->b_rptr;
 lp = (struct linkblk *)mp->b_cont->b_rptr;
 ilen = ip->ioc_count;
 /* Alles fuer ein IOCNAK vorbereiten */
 mp->b_datap->db_type = M_IOCNAK;
 ip->ioc_count = 0;
 /* IOCTLs sind nur vom IOC aus erlaubt */
 if ( !ISIOC(xid) )
  {
   putnext(xid->uread,mp);
   return;
  }
 /* Befehl auswerten */
 switch (ip->ioc_cmd)
  {
   case SOS_BIND    : /* Multicastadresse suchen */
		      if ( (ilen != 10) || !(mc = findmc(mp->b_cont->b_rptr)) ) break;
		      new = nouser(mc);
		      /* Exchange auslesen */
		      bcopy(mp->b_cont->b_rptr+6,&dev,4);
		      if ( (dev < 0) || (dev >= cf_sos_chan_cnt) ) break;
		      /* Kanal eintragen */
		      if ( !new ) MC_SET(mc,dev);
		      break;
   case SOS_UNBIND  : /* Multicaseadresse suchen */
		      if ( (ilen != 10) || !(mc = findmc(mp->b_cont->b_rptr)) ) break;
		      /* Exchange auslesen */
		      bcopy(mp->b_cont->b_rptr+6,&dev,4);
		      if ( (dev < 0) || (dev >= cf_sos_chan_cnt) ) break;
		      /* Kanal eintragen */
		      MC_CLR(mc,dev);
		      /* Eventuell Multicastadresse freigeben */
		      if ( new = nouser(mc) ) MC_SET(mc,dev);
		      break;
   case SOS_REQUEST : /* Alte Messageblocks freigeben */
     		      for ( cp = mp->b_cont ; fp = cp ; freeb(fp) ) cp = fp->b_cont;
     		      /* Multicastadressen auslesen */
     		      if ( !(mp->b_cont = allocb(4+(tlen = MCAST_SIZE),BPRI_LO)) )  break;
		      /* Daten zusammenstellen */
		      fields = MC_FIELDS;
		      bcopy(&fields,mp->b_cont->b_rptr,4);
		      bcopy(mcast,mp->b_cont->b_rptr+4,tlen);
		      mp->b_cont->b_wptr = mp->b_cont->b_rptr+(ip->ioc_count = tlen+4);
		      /* Erfolg melden */
		      mp->b_datap->db_type = M_IOCACK;
		      /* Fertig */
		      break;
   case I_LINK      : /* Verbindung darf noch nicht bestehen */
		      if ( inen ) break;
		      /* Informationen uebertragen */
		      inen = lp->l_qbot;
		      link = mp;
		      /* Device initialisieren */
		      initinen();
		      /* Weitere Bearbeitung erfolgt asynchron */
		      return;
   case I_UNLINK    : /* Verbindung loesen */
		      unl = 1;
		      /* Alles in Ordnung */
		      mp->b_datap->db_type = M_IOCACK;
		      /* Fertig */
		      break;
  }
 /* Multicast an- oder abmelden */
 if ( !new )
  /* Alles in Ordung */
  mp->b_datap->db_type = M_IOCACK;
 else if ( new == 1 )
  {
   /* Asynchron bearbeiten */
   freeit(&mcact);
   mcact = mp;
   /* Entsprechende Routine aufrufen */
   if ( ip->ioc_cmd == SOS_BIND )
    bindmc(mc);
   else
    unbindmc(mc);
   /* Fertig */
   return;
  }
 /* Antwort senden */
 putnext(xid->uread,mp);
 /* Aufraeumen */
 if ( unl ) closeinen();
}

/*
  ETHERNET Device initialisieren.
*/
static initinen()
{
 union DL_primitives *dp;
 mblk_t *mb;

 /* Nachrichtenblock reservieren */
 if ( !(mb = allocb(DL_BIND_REQ_SIZE,BPRI_MED)) )
  {
   /* Nachricht verwerfen */
   putioc(&link);
   inen = NULL;
   /* Fertig */
   return;
  }

 /* Message fuellen und abschicken */
 mb->b_datap->db_type = M_PROTO;
 dp = (union DL_primitives *)mb->b_wptr;
 dp->bind_req.dl_primitive = DL_BIND_REQ;
 dp->bind_req.dl_sap = cf_sos_protocoll_type;
 dp->bind_req.dl_max_conind = 0;
 dp->bind_req.dl_service_mode = DL_CLDLS;
 dp->bind_req.dl_conn_mgmt = 0;
 mb->b_wptr += DL_BIND_REQ_SIZE;
 putnext(inen,mb);
}

/*
  Multicastadresse anbinden.
*/
static bindmc(mc)
logaddr *mc;
{
 dl_bind_mc_req_t *bm;
 mblk_t *mb;

 /* Nicht ganz am Ende */
 if ( !inen ) return;

 /* Nachrichtenblock reservieren */
 if ( !(mb = allocb(DL_BIND_MC_REQ_SIZE,BPRI_MED)) )
  {
   /* Nachricht verwerfen */
   putioc(&mcact);
   /* Fertig */
   return 0;
  }

 /* Message fuellen und abschicken */
 mb->b_datap->db_type = M_PROTO;
 bm = (dl_bind_mc_req_t *)mb->b_wptr;
 bm->dl_primitive = DL_BIND_MC_REQ;
 bm->dl_bind_mc_num_of_addrs = 1;
 bm->dl_bind_mc_addr_length = 6;
 bm->dl_bind_mc_first_addr_offset = DL_BIND_MC_REQ_SIZE;
 mb->b_wptr += DL_BIND_MC_REQ_SIZE;
 bcopy(mc->addr,mb->b_wptr,6);
 mb->b_wptr += 6;
 putnext(inen,mb);
}

/*
  Multicastadresse anbinden.
*/
static unbindmc(mc)
logaddr *mc;
{
 dl_unbind_mc_req_t *um;
 mblk_t *mb;

 /* Nicht ganz am Ende */
 if ( !inen ) return;

 /* Nachrichtenblock reservieren */
 if ( !(mb = allocb(DL_UNBIND_MC_REQ_SIZE,BPRI_MED)) )
  {
   /* Nachricht verwerfen */
   putioc(&mcact);
   /* Fertig */
   return 0;
  }

 /* Message fuellen und abschicken */
 mb->b_datap->db_type = M_PROTO;
 um = (dl_unbind_mc_req_t *)mb->b_wptr;
 um->dl_primitive = DL_UNBIND_MC_REQ;
 um->dl_unbind_mc_num_of_addrs = 1;
 um->dl_unbind_mc_addr_length = 6;
 um->dl_unbind_mc_first_addr_offset = DL_UNBIND_MC_REQ_SIZE;
 mb->b_wptr += DL_UNBIND_MC_REQ_SIZE;
 bcopy(mc->addr,mb->b_wptr,6);
 mb->b_wptr += 6;
 putnext(inen,mb);
}

/*
  Timeout fuer einen Exchange setzen.
*/
static settime(xid,mp)
exchange *xid;
mblk_t *mp;
{
 long *data = (long *)mp->b_rptr;

 /* Parameter sind ein von Null verschiedener Identifier und eine Zeit in pSOS-Ticks */
 if ( !mp->b_cont && ((mp->b_wptr-mp->b_rptr) == 8) && (data[1] > 0) ) 
  {
   /* Alten Timeout eleminieren */
   cleartime(xid);

   /* Neuen Timeout aufsetzen */
   if ( nodata(xid->uread) && (xid->mt = allocb(4,BPRI_MED)) )
    {
     xid->mt->b_datap->db_type = M_PROTO;
     bcopy(data,xid->mt->b_wptr,4);
     xid->mt->b_wptr += 4;
     xid->tid = timeout(timedout,xid,data[1]);
    }
  }

 /* Fertig */
 freemsg(mp);
}

/*
  Feststellen, ob in einem STREAM mindestens eine M_DATA Nachricht ist.
*/
static nodata(q)
queue_t *q;
{
 mblk_t *mb,*cb;

 /* Alle Queues durchgehen */
 for ( ; q ; q = q->q_next )
  /* Alle Messages durchgehen */
  for ( mb = q->q_first ; mb ; mb = mb->b_next )
   /* Alle Messageblocks durchgehen */
   for ( cb = mb ; cb ; cb = cb->b_cont )
    if ( cb->b_datap->db_type == M_DATA )
     return 0;
 /* In der Tat, keine M_DATA Messages */
 return 1;
}

/*
  Timeout ist abgelaufen.
*/
static timedout(xid)
exchange *xid;
{
 queue_t *rd = xid->uread;
 mblk_t *mb = xid->mt;

 /* Timeout ist bearbeitet */
 xid->mt = 0;

 /* Nachsehen, ob abschicken moeglich ist */
 if ( !mb ) return;
 if ( !rd )
  {
   freemsg(mb);
   return;
  }

 /* Muss mit der Serviceroutine bearbeitet werden */
 putq(rd,mb);
 qenable(rd);
}

/*
  Kein Benutzer fuer eine logische Adresse.
*/
static nouser(mc)
logaddr *mc;
{
 int n;

 /* Suchen */
 for ( n = MC_FIELDS ; n-- && !mc->users[n] ; );
 /* Ergebnis melden */
 return (n < 0) ? 1 : 0;
}

/*
  Multicastadresse suchen.
*/
static logaddr *findmc(addr)
char *addr;
{
 int n;

 /* Suchen */
 for ( n = SOS_MULTICASTS ; n-- ; )
  if ( !bcmp(MCAST(n)->addr,addr,6) )
   return MCAST(n);
 /* Nichts gefunden */
 return 0;
}

/*
  Mail Dispatch Routine
*/
static localMail(mb,ether)
mblk_t *mb;
int ether;
{
 int tonet,fromioc,len,sendit,first,n,code;
 mblk_t *next = mb,*fp;
 MAILBOX *mbox,*fbox;
 logaddr *mc;
 short slen;

 /* Alle Mails durchgehen */
 for ( mb->b_next = 0 ; mb = next ; ether = 0 )
  {
   /* Zeiger auf die naechste Nachricht */
   next = mb->b_next;
   /* Mailboxheader sammeln */
   if ( !pullupmsg(mb,HEADERSIZE) )
    {
     freemsg(mb);
     continue;
    }
   /* Mailboxheader ermitteln */
   mbox = (MAILBOX *)mb->b_rptr;
   /* Netzwerkcharakteristik feststellen */
   tonet = (mbox->mbrxid == NETXID);
   /* Diverse Netzwerkflags auswerten */
   if ( ether && ((mbox->mbstatus&EXRDBIT) == EXRDBIT) )
    {
     LONG(&mbox->mbaddr,&mbox->mbsaddr);
     if ( !(mbox->mbstatus&ERBIT) ) mbox->mbstatus |= ERBIT|FRBIT;
     mbox->mbstatus |= NRBIT;
     mbox->mbstatus &= ~EXBIT;
    }
   /* Eigene ETHERNET Adresse eintragen */
   if ( !ether && tonet )
    {
     bcopy(paddr,mbox->mbtnet,6);
     mbox->mbtype = cf_sos_protocoll_type;
    }
   /* IOC Mails gesondert behandeln */
   mbox->mbstatus &= ~(fromioc = mbox->mbstatus&FIBIT);
   /* Schreibdaten bearbeiten */
   sendit = code = first = 0;
   /* Laenge ermitteln */
   len = MINNETSIZE;
   if ( mbox->mbstatus&WRBIT )
    {
     SWORD(&mbox->mblen,&slen);
     if ( (slen < 0) || (slen > MAILMAX) )
      code = 10;
     else if ( (slen += HEADERSIZE) > MINNETSIZE )
      len = (slen+1)&~1;     
    }
   /* Multicastadressen bearbeiten */
   if ( !code )
    {
     if ( ((*(char *)mbox->mbrnet)&1) && (ether || (tonet && !fromioc)) )
      {
       first = ether;
       if ( (mc = findmc(mbox->mbrnet)) && !nouser(mc) )
	/* Alle angemeldeten Benutzer durchgehen */
	for ( n = cf_sos_chan_cnt ; n-- ; )
	 if ( MC_ISSET(mc,n) )
	  {
	   if ( first )
	    {
	     fp = mb;
	     first = 0;
	    }
	   else if ( !(fp = copymsg(mb)) )
	    continue;
	   /* Neuer Header */
	   fbox = (MAILBOX *)fp->b_rptr;
	   /* Informationen einsetzen */
	   fbox->mbicom = NM_COM;
	   fbox->mbixid = fbox->mbrxid = n;
	   /* Verketten */
	   fp->b_next = next;
	   next = fp;
	  }
      }
     /* Mail verschicken */
     if ( ether && !((*(char *)mbox->mbrnet)&1) )
      {
       if ( mbox->mbicom != NM_COM )
        code = 18;
       else
        {
         mbox->mbrxid = mbox->mbixid;
         sendit = 1;
        }
      }
     else if ( !ether )
      if ( tonet )
       {
        if ( !bcmp(mbox->mbrnet,paddr,6) ) 
         if ( mbox->mbicom != NM_COM )
          code = 18;
         else
          {
           mbox->mbrxid = mbox->mbixid;
           sendit = 1;
          }
        else 
	 /* Auf ETHERNET verschicken */
	 sendether(mb); 
       }
      else
       sendit = 1;
    }
   /* Mail lokal verschicken */
   if ( !code )
    if ( sendit )
     {
      if ( sendlocal(mb) ) code = 19;
     }
    else if ( first && ether )
     freemsg(mb);
   /* Fehler behandeln */
   if ( !code ) continue;
   if ( (mbox->mbstatus&ERBIT) && (mbox->mbrnet[0] || (mbox->mbscom == NM_COM)) &&
	(fp = allocb(HEADERSIZE,BPRI_MED)) )
    {
     /* Readymailbox ermitteln */
     fbox = (MAILBOX *)fp->b_wptr;
     fp->b_wptr += HEADERSIZE;
     /* Mailbox fuellen */
     fbox->mbscom = NM_COM;
     if ( !mbox->mbrnet[0] )
      {
       fbox->mbrnet[0] = 0;
       fbox->mbrxid = mbox->mbsxid;
       fbox->mbstatus = 0;
      } 
     else
      {
       bcopy(mbox->mbtnet,fbox->mbrnet,6);
       fbox->mbrxid = NETXID;
       fbox->mbicom = mbox->mbscom;
       fbox->mbixid = mbox->mbsxid;
       LONG(mbox->mbsaddr,fbox->mbsaddr);
       fbox->mbstatus = mbox->mbstatus&(~(FRBIT|NRBIT));
      }
     fbox->mbexpd = mbox->mbexpd;
     fbox->mbnum = mbox->mbnum;
     fbox->mbfid = mbox->mbfid;
     fbox->mblen = fbox->mbres = 0;
     fbox->mbstatus |= RYBIT|SEBIT;
     fbox->mbcode = code;
     /* Mailbox verketten */
     fp->b_next = next;
     next = fp;
    }
   freemsg(mb);
  }
}

/*
  Mailbox an das ETHERNET Device weitergeben.
*/
static sendether(mb)
mblk_t *mb;
{
 /* Nachsehen, ob Speichern moeglich ist */
 if ( !inen || !canput(inen->q_next) )
  {
   freemsg(mb);
   return;
  }
 /* Nachricht an das ETHERNET-Device weitergeben */
 putnext(inen,mb);
}

/*
  Mailbox lokal verschicken.
*/
static sendlocal(mb)
mblk_t *mb;
{
 int dev = ((MAILBOX *)mb->b_rptr)->mbrxid;
 exchange *xid;

 /* Device testen */
 if ( (dev < 0) || (dev >= cf_sos_chan_cnt) || !(xid = xids+dev)->uread ||
      !canput(xid->uread->q_next) )
  return 1;
 /* Timeout loeschen */
 cleartime(xid);
 /* Nachricht verschicken */
 putnext(xid->uread,mb);
 /* Hat geklappt */
 return 0;
}

/*
  IOCTL-Block an den IOC zurueckgeben.
*/
static putioc(mpp)
mblk_t **mpp;
{
 /* Nachricht nicht gegeben */
 if ( !*mpp ) return;
 /* Nachricht freigeben */
 if ( !IOCRD ) return freeit(mpp);
 /* Nachricht verschicken */
 putnext(IOCRD,*mpp);
 *mpp = NULL;
}

/*
  Nachricht freigeben.
*/
static freeit(mpp)
mblk_t **mpp;
{
 /* Nur falls vorhanden */
 if ( *mpp ) freemsg(*mpp);
 /* Immer frei */
 *mpp = NULL;
}

/*
  ETHERNET-Kanal schliessen
*/
static closeinen()
{
 int n;

 /* Struktur freigeben */
 inen = NULL;
 /* Verbindungsstrukturen freigeben */
 freeit(&link);
 freeit(&mcact);
 /* An alle Kanaele weiterleiten */
 for ( n = cf_sos_chan_cnt ; n-- ; )
  if ( xids[n].uread )
   putctl1(xids[n].uread->q_next,M_ERROR,EINVAL);
}
