/* GETARGS.C */

#include <stdio.h>
#include "getargs.h"

/*
  Variablen zur Kommunikation mit dem aufrufenden Programm:
	OPTARG		zustzlicher Parameter einer Option
	OPTIND		aktuell untersuchte Option in ARGV
	OPTERR		Fehlercode
*/
char *optarg;
int optind = 0;
int opterr = GETERR_NONE;

/*
  Interne Variablen, die den Kontext zwischen zwei Aufrufen der GETARGS
  Routine speichern.
	OPTEXT		Zhler der Parameter
	OPTCUR		Aktuell zu untersuchender Text
	OPTCON		Ist 1 innerhalb einer Optionskette
*/
static int optext = 0;
static char *optcur;
static char optcon = 0;

/*
  int getargs(int argc,char **argv,char *fmt)
	argc	Wie einem Hauptprogramm bergeben
	argv	Wie einem Hauptprogramm bergeben
	fmt	Erlaubte Optionen in folgendem Format:
		  <fmt> = [<oliste>] [!<pliste>[*]]
		  <oliste> = <option> {<option>}
		  <option> = <buchstabe> [:]
		  <pliste> = <buchstabe> {<buchstabe>}
		Die verwendeten Buchstaben drfen nicht die Zeichen
		!, : und * sein und sollten nicht doppelt verwendet
		werden.

  Analysiert die durch ARGC und ARGV gegebene Eingabezeile gem
  dem spezifizierten Format FMT. Alle Parameter werden nur gelesen.
*/
int getargs(argc,argv,fmt)
int argc;
char *argv[],*fmt;
{
 char opt;
 int pcnt;

 /*
   Wenn OPTCON 0 ist, so beginnt die Analyse gerade oder es wurde
   zuletzt ein Parameter eingelesen. Ansonsten ist OPTCUR die
   Adresse der zunchst zu bearbeitenden Option aus einem Element
   von ARGV. Ist das Ende des aktuellen Elements erreicht, so mu
   wie bei den Fllen oben das nchste Element aktiviert werden.
 */
 if ( !optcon || !*optcur )
  {
   if ( ++optind >= argc ) return EOF;
   optcur = argv[optind];
   optcon = 0;
  }
 /*
   Wenn jetzt OPTCON immer noch 0 ist, so wurde ein neues Element
   aktiviert. Beginnt es mit einem -, so handelt es sich um eine
   Optionskette. In diesem Fall darf das Element nicht nur aus
   einem - bestehen.
 */
 if ( !optcon && (*optcur == '-') )
  {
   if ( !*++optcur )
    {
     opterr = GETERR_NOOPT;
     return EOF;
    }
   optcon = 1;
  }
 /*
   Wird eine Optionskette bearbeitet, so werden die aufgelisteten
   Optionen zeichenweise bearbeitet und dem aufrufenden Programm
   bergeben. Wenn eine Option gewhlt wurde, hinter der im Format
   ein : steht, so mu in dem ARGV Feld noch mindestens ein Element
   vorhanden sein. Die Adresse dieser Zeichenkette wird dann in
   OPTARG gespeichert und die Option dem aufrufenden Programm zu-
   rckgemeldet.
 */
 if ( optcon )
  {
   opt = *optcur++;
   while ( *fmt && (*fmt != '!') && (*fmt != opt) ) fmt++;
   if ( (opt == ':') || (opt == '!') || (opt == '*') || (*fmt != opt) )
    {
     opterr = GETERR_ILLOPT;
     return EOF;
    }
   if ( fmt[1] != ':' ) return opt;
   if ( ++optind >= argc )
    {
     opterr = GETERR_NOPAR;
     return EOF;
    }
   optarg = argv[optind];
   return opt;
  }
 /*
   Wird aber keine Optionskette bearbeitet, so folgt ein Parameter
   im aktuellen Argument. Parameter werden im Format hinter einem
   ! angegeben. OPTEXT beschreibt, der wievielte Parameter gerade
   bearbeitet wird, um dem aufrufenden Programm die aus den Zeichen
   hinter dem ! abgeleitete Option korrekt zurckzugeben. Endet eine
   Formatzeichenkette mit ! mit einem *, so knnen beliebig viele
   Parameter angegeben werden. Jeder einzelne liefert dem aufrufen-
   den Programm ein * mit geflltem OPTARG.
 */
 while ( *fmt && (*fmt != '!') ) fmt++;
 if ( *fmt )
  {
   for ( pcnt = optext ; *++fmt && pcnt-- ; );
   if ( !*fmt && (fmt[-1] == '*') ) fmt--;
  }
 if ( !*fmt )
  {
   opterr = GETERR_ILLPAR;
   return EOF;
  }
 optext++;
 optarg = optcur;
 return *fmt;
}

/*
  char *getargs_error()

  Setzt den Fehlercode aus OPTERR in einen entsprechenden Text um.
  Alle nicht bekannten Fehler werden als ein 0 Zeiger gemeldet und
  sollten im Normalbetrieb nicht auftreten.
*/
char *getargs_error()
{
 switch (opterr)
  {
   case GETERR_NOOPT  : return "Fehlende Option";
   case GETERR_ILLOPT : return "Illegale Option";
   case GETERR_NOPAR  : return "Fehlender Parameter";
   case GETERR_ILLPAR : return "Illegaler Parameter";
   default	      : return 0;
  }
}
