#define LOGOPEN			0x00000001
#define LOGPUT			0x00000002
#define LOGCTRL			0x00000004
#define LOGSRV			0x00000010
#define LOGIOCTL		0x00000020
#define LOGFLOW			0x00000040
#define LOGFLUSH		0x00000080
#define LOGWDATA		0x00000008
#define LOGRDATA		0x00000100
#define LOGUDATA		0x00000200
#define LOGDDATA		0x00000400

#if 0
#define LOGLEVEL		(LOGIOCTL|LOGCTRL)
#endif

#ifndef LOGLEVEL
#define LOGLEVEL	        0
#endif

/*
  Definitionsdateien
*/
#include <ii/i_vm.h>
#include <ii/i_sfm.h>

#include <sys/log.h>
#include <sys/stat.h>
#include <sys/errno.h>
#include <sys/tihdr.h>

#include <netinet/tcp.h>
#define TCP_LOCAL_ADDRESS_SIZE	(2*sizeof(long)+sizeof(struct sockaddr_in))

#include "parallel.h"

/*
  STREAMS Funktionen
*/
static int parallelopen(),parallelclose();
static int paralleluwput(),parallelursrv();
static int parallellrput(),parallellwsrv();

/*
  Hilfsfunktionen
*/
static int par_connect(),par_disconnect();

/*
  STREAMS Informationsblocks
*/
static struct module_info minfo = 
       {
	0, "parallel", 0, INFPSZ, 4096, 512
       };

static struct qinit urinit =
       {
	NULL, parallelursrv, parallelopen, parallelclose, NULL, &minfo, NULL
       };

static struct qinit uwinit =
       {
	paralleluwput, NULL, NULL, NULL, NULL, &minfo, NULL
       };

static struct qinit lrinit =
       {
	parallellrput, NULL, NULL, NULL, NULL, &minfo, NULL
       };

static struct qinit lwinit =
       {
	NULL, parallellwsrv, NULL, NULL, NULL, &minfo, NULL
       };

struct streamtab parallelinfo =
       {
	&urinit, &uwinit, &lrinit, &lwinit
       };

/*
  Devicetreiber Funktionen
*/
static status_type parallel_configure();
static status_type parallel_name_to_device(),parallel_device_to_name();

/*
  Devicetreiber Informationsblocks
*/
su_driver_routines_vector_type sfm_parallel_routines_vector =
			       {
				SU_ROUTINES_VECTOR_VERSION_1,
				sfm_nodevice_init,
				parallel_configure,sfm_nodevice_deconfigure,
				parallel_name_to_device,parallel_device_to_name,
			       };

/*
  Konfigurierbare Variablen
*/
extern uint32_type cf_parallel_chan_cnt;

/*
  Nicht Konfigurierbare Variablen und Konstanten
*/
enum parstate { 
  	       PS_IDLE, PS_CONNECTED, PS_ERROR,
	       PS_CONNECTING, PS_CONFIRMING
	      };

struct parconn
       {
	enum parstate   state;
	queue_t		*qptr;
	mblk_t		*iow;
	struct linkblk  tcp;
	mblk_t		*sync;
       };

static int32_type confmaj = -1,clonemaj = -1;
static struct parconn *conn = NULL;


/*
  MACROs
*/
#define putdown(c,q,m)			{ putq(q,m); qenable((c)->tcp.l_qbot); }
#define putup(c,q,m)			{ putq(q,m); qenable((c)->qptr); }


/*
  Devicetreiber Routinen
*/

/*
  /dev/parallel konfigurieren
*/
status_type parallel_configure(name,majnum)
char_ptr_type name;
io_major_device_number_type majnum;
{
 char_ptr_type lname = "parallel()";
 fs_dev_request_type credev;
 io_device_number_type cdev;
 uint32_type length,dev;
 status_type res;

 /* Der Name muss stimmen */
 do
  {
   if ( *lname != *name++ ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;
  }
 while ( *lname++ );
 /* Darf nur einmal konfiguriert werden */
 if ( confmaj != -1 ) return IO_ENXIO_DEVICE_IS_ALREADY_CONFIGURED;
 /* Und benoetigt zum Betrieb das clone-Device */
 if ( (res = sfm_name_to_device("clone()",&cdev)) != OK ) return res;

 /* Speicher fuer die Kanaele reservieren */
 if ( length = usizeof(conn[0])*cf_parallel_chan_cnt )
  conn = (struct parconn *)vm_get_wired_memory(length,VM_DEFAULT_ALIGNMENT);

 /* Situation festhalten */
 confmaj = majnum;
 clonemaj = cdev.major;

 /* inode fuer /dev/parallel als cloned Device erzeugen */
 credev.operation = Fs_Dev_Request_Operation_Create;
 credev.dirname[0] = '\0';
 length = usizeof(credev.filename);
 misc_string_copy("parallel",credev.filename,&length);
 credev.op.create.mode_bits = SFM_PIPE_NODE_ACCESS_PERMISSIONS;
 credev.op.create.device.major = clonemaj;
 credev.op.create.device.minor = confmaj;
 fs_submit_dev_request(&credev);

 /* inodes fuer /dev/parallel%d als normale Devices erzeugen und Speicher initialisieren */
 for ( dev = cf_parallel_chan_cnt ; dev-- > 0 ; )
  {
   /* Speicher initialisieren */
   conn[dev].state = PS_IDLE;
   conn[dev].qptr = NULL;
   conn[dev].iow = NULL;
   conn[dev].tcp.l_qbot = NULL;
   conn[dev].sync = NULL;
   /* Devicenode erzeugen */
   misc_format_line(credev.filename,usizeof(credev.filename),"parallel%d",dev);
   credev.op.create.device.major = confmaj;
   credev.op.create.device.minor = dev;
   fs_submit_dev_request(&credev);
  }

 /* Erfolg melden */
 return OK;
}

/*
  Devicename in ein Nummernpaar (Major,Minor) umsetzen.
*/
status_type parallel_name_to_device(name,nump)
char_ptr_type name;
io_device_number_ptr_type nump;
{
 char_ptr_type lname = "parallel";
 uint32_type dev;

 if ( (confmaj == -1) || (clonemaj == -1) ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

 while ( *lname )
  if ( *name++ != *lname++ )
   return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

 for ( lname = name ; (*name >= '0') && (*name <= '9') ; name++ );

 if ( lname == name )
  {
   nump->major = clonemaj;
   nump->minor = confmaj;
  }
 else
  {
   if ( (name-lname) > 5 ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

   for ( dev = 0 ; lname < name ; ) dev = 10*dev+*lname++-'0';

   if ( dev >= cf_parallel_chan_cnt ) return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

   nump->major = confmaj;
   nump->minor = dev;   
  }

 if ( *name && ((*name++ != '(') || (*name++ != ')') || *name) )
  return IO_ENXIO_DEVICE_NAME_NOT_RECOGNIZED;

 return OK;
}

/*
  Nummernpaar (major,minor) in einen Devicenamen umsetzen
*/
status_type parallel_device_to_name(dev,namep,size)
io_device_number_type dev;
char_ptr_type namep;
uint32_type size;
{
 if ( (dev.major == clonemaj) && (dev.minor == confmaj) )
  misc_string_copy("parallel()",namep,&size);
 else
  {
   if ( dev.major != confmaj ) return IO_ENXIO_DEVICE_IS_NOT_CONFIGURED;

   misc_format_line(namep,size,"parallel%d()",(uint32_type)dev.minor);
  }

 return OK;
}

/*
  STREAMS Routinen
*/

/*
  Oeffnen des Devices
*/
static int parallelopen(q,dev,flag,sflag)
queue_t *q;
int dev,flag,sflag;
{
#if LOGLEVEL&LOGOPEN
 strlog(50,(sflag == CLONEOPEN) ? -1 : minor(dev),0,SL_ERROR,"open start");
#endif

 /* Bei einem cloned open freien Eintrag suchen */
 if ( sflag != CLONEOPEN )
  dev = minor(dev);
 else
  for ( dev = 0 ; dev < cf_parallel_chan_cnt ; dev++ )
   if ( (conn[dev].state == PS_IDLE) && !conn[dev].qptr && !conn[dev].tcp.l_qbot )
    break;

 /* Eintrag verifizieren */
 if ( dev >= cf_parallel_chan_cnt ) return OPENFAIL;

 /* Informationen festhalten */
 q->q_ptr = WR(q)->q_ptr = (char *)(conn+dev);

 /* Devicekanal reservieren */
 conn[dev].qptr = q;

 /* Hat funktioniert */
#if LOGLEVEL&LOGOPEN
 strlog(50,dev,0,SL_ERROR,"open done");
#endif
 return dev;
}

/*
  Schliessen des Devices
*/
static int parallelclose(q,oflag)
queue_t *q;
int oflag;
{
 struct parconn *pc;

#if LOGLEVEL&LOGOPEN
 strlog(50,0,0,SL_ERROR,"close start");
#endif

 /* Aktuellen Kanal ermitteln */
 pc = (struct parconn *)q->q_ptr;

 /* Kanal freigeben */
 pc->qptr = NULL;

 /* Keine Synchronisation mehr */
 if ( pc->sync ) 
  {
#if LOGLEVEL&LOGIOCTL
   strlog(50,0,0,SL_ERROR,"close ioctl sync");
#endif
   freemsg(pc->sync);
   pc->sync = NULL;
  }

 /* IOCTL-Anfrage loeschen */
 if ( pc->iow )
  {
#if LOGLEVEL&LOGIOCTL
   strlog(50,0,0,SL_ERROR,"close ioctl");
#endif
   freemsg(pc->iow);
   pc->iow = NULL;
  }

 /* Fertig */
#if LOGLEVEL&LOGOPEN
 strlog(50,0,0,SL_ERROR,"close done");
#endif
}

/* 
  Daten vom Benutzer (upper) entgegennehmen
*/
static int paralleluwput(q,mp)
queue_t *q;
mblk_t *mp;
{
 struct parconn *pc;
 struct linkblk *lp;
 struct iocblk *ip;
 int ilen;

#if LOGLEVEL&LOGPUT
 strlog(50,0,0,SL_ERROR,"uwput start");
#endif

 /* Zugeordneten Kanal ermitteln */
 pc = (struct parconn *)q->q_ptr;

 /* Daten auswerten */
 switch (mp->b_datap->db_type)
  {
   case M_IOCTL : /* Struktur ermitteln */
     		  ip = (struct iocblk *)mp->b_rptr;
		  ilen = ip->ioc_count;
		  /* Alles fuer ein IOCNAK vorbereiten */
		  mp->b_datap->db_type = M_IOCNAK;
		  ip->ioc_count = 0;
		  /* Normale IOCTLs */
		  if ( (ip->ioc_cmd != I_LINK) && (ip->ioc_cmd != I_UNLINK) )
		   {
		    /* Verbindung aufbauen */
		    if ( ip->ioc_cmd == PARALLEL_CONNECT )
		     {
#if LOGLEVEL&LOGIOCTL
  		      strlog(50,0,0,SL_ERROR,"uwput ioctl connect");
#endif
		      if ( (ilen == sizeof(struct parallel_connect)) &&
		           par_connect(pc,mp,ip,mp->b_cont->b_rptr) )
		       return;
		     }
		    /* Verbindung abbauen */
		    else if ( ip->ioc_cmd == PARALLEL_SYNCHRONIZE )
		     {
		      /* Alte Nachricht freigeben */
		      if ( pc->sync ) freemsg(pc->sync);
		      pc->sync = NULL;
		      /* Neue wird immer positiv beantortet */
		      mp->b_datap->db_type = M_IOCACK;
		      /* Nachsehen, ob noch Sendedaten ausstehen */
		      if ( qsize(q) )
		       {
		        /* Wird spaeter bearbeitet */
		        pc->sync = mp;
		        /* Nicht beantworten (Benutzer hat das Timeout zu setzen) */
		        return;
		       }
		     }
		    else if ( ip->ioc_cmd != PARALLEL_DISCONNECT )
		     {
#if LOGLEVEL&LOGIOCTL
  		      strlog(50,ip->ioc_cmd,0,SL_ERROR,"uwput ioctl failed");
#endif
		      /* Eventuell nach unten weitergeben */
		      if ( pc->tcp.l_qbot )
		       {
			/* Felder wieder einsetzen */
			mp->b_datap->db_type = M_IOCTL;
			ip->ioc_count = ilen;
			/* Abschicken */
			putnext(pc->tcp.l_qbot,mp);
		        return;  
		       }
		     }
		    else if ( par_disconnect(pc,0) )
		     {
#if LOGLEVEL&LOGIOCTL
  		      strlog(50,0,0,SL_ERROR,"uwput ioctl disconnect");
#endif
		      mp->b_datap->db_type = M_IOCACK;
		     }
#if LOGLEVEL&LOGIOCTL
		    else
  		     strlog(50,0,0,SL_ERROR,"uwput ioctl disconnect failed");
#endif
		    /* Befehl beantworten */
		    qreply(q,mp);
		    return;
		   }
		  /* Sturktur ermitteln */
		  lp = (struct linkblk *)mp->b_cont->b_rptr;
		  /* Funktion ausfuehren */
		  if ( ip->ioc_cmd == I_UNLINK )
		   {
#if LOGLEVEL&LOGIOCTL
  		    strlog(50,0,0,SL_ERROR,"uwput ioctl unlink");
#endif
		    if ( pc->tcp.l_qbot )
		     {
#if LOGLEVEL&LOGIOCTL
  		      strlog(50,0,0,SL_ERROR,"uwput ioctl unlink disconnect");
#endif
		      /* Verbindung sauber herunterfahren */
		      par_disconnect(pc,1);
		      /* Kanal freigeben */
		      pc->tcp.l_qbot = NULL;
		     }		    
		    /* Alles in Ordnung */
		    mp->b_datap->db_type = M_IOCACK;
		   }
		  else if ( !pc->tcp.l_qbot )
		   {
#if LOGLEVEL&LOGIOCTL
  		    strlog(50,0,0,SL_ERROR,"uwput link");
#endif
		    /* Informationen uebertragen */
		    lp->l_qbot->q_ptr = RD(lp->l_qbot)->q_ptr = (char *)pc;
		    /* Eintragen */
		    pc->tcp = *lp;
		    /* Alles in Ordnung */
		    mp->b_datap->db_type = M_IOCACK;
		   } 
#if LOGLEVEL&LOGIOCTL
		  else
  		   strlog(50,0,0,SL_ERROR,"uwput link failed");
#endif
		  /* Antwort senden */
		  qreply(q,mp);
		  return;
   case M_FLUSH : /* Datenqueues loeschen */
#if LOGLEVEL&LOGFLUSH
  		  strlog(50,0,0,SL_ERROR,"uwput flush");
#endif
     		  if ( *mp->b_rptr&FLUSHW ) flushq(q,FLUSHDATA);
		  if ( *mp->b_rptr&FLUSHR )
		   {
		    flushq(RD(q),FLUSHDATA);
		    /* Fuer den STREAM-Head */
		    *mp->b_rptr &= ~FLUSHW;
		    qreply(q,mp);
		   }
		  else
		   /* Fertig */
		   freemsg(mp);
		  return;
   case M_DATA  : /* Werden einfach durchgereicht */
#if LOGLEVEL&LOGWDATA
  		  strlog(50,msgdsize(mp),0,SL_ERROR,"uwput data");
#endif
     		  if ( pc->tcp.l_qbot && (pc->state == PS_CONNECTED) )
		   {
#if LOGLEVEL&LOGWDATA
  		    strlog(50,0,0,SL_ERROR,"uwput data processed");
#endif
		    putdown(pc,q,mp);
		    return;
		   }
   default      : /* Unbekannte STREAM-Codes oder illegale Verbindung */
#if LOGLEVEL&LOGCTRL
  		  strlog(50,0,0,SL_ERROR,"uwput illegal message");
#endif
     		  freemsg(mp);
		  putctl1(RD(q)->q_next,M_ERROR,EINVAL);
  }

#if LOGLEVEL&LOGPUT
 strlog(50,0,0,SL_ERROR,"uwput done");
#endif
}

/*
  Daten vom Netzwerk (lower) entgegennehmen
*/
static int parallellrput(q,mp)
queue_t *q;
mblk_t *mp;
{
 struct parallel_connect *src,*dst;
 union T_primitives *tp;
 mblk_t *fp = mp,*mb;
 struct parconn *pc;
 struct iocblk *ip;
 int ans = 0;
 long *opts;

#if LOGLEVEL&LOGPUT
 strlog(50,0,0,SL_ERROR,"rput started");
#endif

 /* Verbindungskanal ermitteln */
 pc = (struct parconn *)q->q_ptr;

 /* Nachrichtentyp ermitteln */
 switch (mp->b_datap->db_type)
  {
   case M_FLUSH   : /* Aktionen eines STREAMS-Heads ausfuehren */
#if LOGLEVEL&LOGFLUSH
 		    strlog(50,0,0,SL_ERROR,"rput flush");
#endif
     		    if ( *mp->b_rptr&FLUSHR ) flushq(q,FLUSHDATA);
		    if ( *mp->b_rptr&FLUSHW )
		     {
		      *mp->b_rptr &= ~FLUSHR;
		      qreply(q,mp);
		     }
		    else
		     freemsg(mp);
		    /* Fertig */
		    return;
   case M_ERROR   :
   case M_HANGUP  : /* Kanal darf nicht mehr benutzt werden */
#if LOGLEVEL&LOGCTRL
 		    strlog(50,0,0,SL_ERROR,"rput error");
#endif
     		    if ( pc->tcp.l_qbot ) pc->state = PS_ERROR;
		    /* Nachricht freigeben */
		    freemsg(mp);
		    return;
   case M_IOCACK  :
   case M_IOCNAK  : /* Nach oben durchgeben */
     		    if ( pc->qptr )
		     {
#if LOGLEVEL&LOGIOCTL
     		      strlog(50,0,0,SL_ERROR,"rput ioctl");
#endif
		      putnext(pc->qptr,mp);
		      return;
		     }
   default        : /* Nachricht ignorieren */
#if LOGLEVEL&LOGCTRL
 		    strlog(50,0,0,SL_ERROR,"rput illegal message");
#endif
     		    freemsg(mp);
		    return;
   case M_PROTO   :
   case M_PCPROTO : break;
  }
 
 /* Kontrollfeld auswerten */
 tp = (union T_primitives *)mp->b_rptr;

 /* Je nach Zustand verfahren */
 if ( pc->tcp.l_qbot )
  switch (pc->state)
   {
    case PS_CONNECTING    : /* Bestaetigung des Befehls abwarten */
      			    if ( tp->type == T_OK_ACK )
			     pc->state = PS_CONFIRMING;
			    else if ( tp->type == T_ERROR_ACK )
			     ans = 1;
			    break;
    case PS_CONFIRMING    : /* Bestaetigung der Verbindung abwarten */
      			    if ( tp->type == T_DISCON_IND )
 			     ans = 1;
			    else if ( tp->type == T_CONN_CON )
			     {
			      pc->state = PS_CONNECTED;
			      if ( !pc->qptr ) 
			       par_disconnect(pc,0);
			      else if ( pc->iow ) 
			       {
			        ans = 2;
				pc->iow->b_datap->db_type = M_IOCACK;
				/* Eigene TCP-Adresse ermitteln */
				if ( (tp->conn_con.OPT_length >= 3*sizeof(opts[0])) &&
				     (tp->conn_con.OPT_offset >= sizeof(tp->conn_con)) )
				 {
				  /* IOCTL berechnen */
				  ip = (struct iocblk *)pc->iow->b_rptr;
				  /* Optionen ermitteln */
				  opts = (long *)(((char *)tp)+tp->conn_con.OPT_offset);
				  /* Optionen auswerten */
				  if ( (*opts++ == (TCP_LOCAL_ADDRESS_SIZE/sizeof(long))) &&
				       (*opts++ == TCP_LOCAL_ADDRESS) &&
				       (*opts++ == sizeof(struct parallel_connect)) )
				   {
				    /* Daten uebertragen */
				    src = (struct parallel_connect *)opts;
				    dst = (struct parallel_connect *)pc->iow->b_cont->b_rptr;
				    *dst = *src;
				    /* Laenge setzen */
				    ip->ioc_count = sizeof(*dst);
				   }
				 }
			       }
			     }
			    break;
    case PS_CONNECTED     : /* Daten oder Verbindungszusammenbruch abwarten */
      			    if ( tp->type == T_DATA_IND )
			     {
#if LOGLEVEL&LOGRDATA
 		    	      strlog(50,0,0,SL_ERROR,"rput data");
#endif
			      /* Verbindung muss bestehen */
			      if ( !pc->qptr ) break;
			      /* Kontrollblock und Nulldatenbloecke entfernen */
			      do
			       {
			        mp = fp->b_cont;
			        freeb(fp);
			       }
			      while ( (fp = mp) && (fp->b_rptr == fp->b_wptr) );
			      /* Daten nach oben durchreichen */
			      if ( mp )
			       {
#if LOGLEVEL&LOGRDATA
 		    	        strlog(50,msgdsize(mp),0,SL_ERROR,"rput data processed");
#endif
				putup(pc,q,mp);
			       }
			      /* Fertig */
			      return;
			     }
			    /* Verbindung sauber beenden */
			    if ( (tp->type == T_ORDREL_IND) || (tp->type == T_DISCON_IND) )
			     {
#if LOGLEVEL&LOGCTRL
 		    	      strlog(50,0,0,SL_ERROR,"rput ordrel_ind");
#endif
			      /* Nulldatenblock als EOF-Kennung nach oben schicken */
			      if ( pc->qptr && (mb = allocb(0,BPRI_MED)) )
			       {
				mb->b_cont = NULL;
				putup(pc,q,mb);
			       }
			      /* Verbindung aufbrechen */
			      par_disconnect(pc,0);
			     }
			    break;
   }

 /* IOCTL beantworten, falls noetig */
 if ( ans )
  {
   /* IOCTL verschicken */
   if ( pc->qptr && pc->iow )
    {
#if LOGLEVEL&LOGCTRL
     strlog(50,0,0,SL_ERROR,"rput ioctl");
#endif
     /* Beantworten */
     putnext(pc->qptr,pc->iow);
     /* Speicher freigeben */
     pc->iow = NULL;
    }
   /* Zustand aktualisieren */
   if ( ans == 1 ) pc->state = PS_IDLE;
  }  

 /* Nachricht freigeben */
#if LOGLEVEL&LOGPUT
 strlog(50,0,0,SL_ERROR,"rput done");
#endif
 freemsg(mp);
}

/*
  Flowcontrol fuer das Netzwerk (lower) bearbeiten
*/
static int parallellwsrv(q)
queue_t *q;
{
 mblk_t *mp = (mblk_t *)-1;
 struct parconn *pc;
 queue_t *nq;

#if LOGLEVEL&LOGSRV
 strlog(50,0,0,SL_ERROR,"wsrv started");
#endif

 /* Verbindung errmitteln */
 pc = (struct parconn *)q->q_ptr;

 /* Nachrichten weitergeben, solange wie moeglich */
 while ( (nq = pc->qptr) && canput(q->q_next) && (mp = getq(WR(nq))) )
  {
#if LOGLEVEL&LOGDDATA
   strlog(50,msgdsize(mp),0,SL_ERROR,"wsrv data");
#endif
   putnext(q,mp);    
  }

 /* Eventuell Synchronisationsnachricht geben */
 if ( nq && !mp && pc->sync )
  {
#if LOGLEVEL&LOGCTRL
   strlog(50,0,0,SL_ERROR,"wsrv ioctl sync");
#endif
   /* Beantworten */
   putnext(nq,pc->sync);
   /* Nur einmal */
   pc->sync = NULL;
  }
#if LOGLEVEL&LOGSRV
 strlog(50,0,0,SL_ERROR,"wsrv done");
#endif
}

/*
  Flowcontrol fuer den Benutzer (upper) bearbeiten
*/
static int parallelursrv(q)
queue_t *q;
{
 struct parconn *pc;
 queue_t *nq;
 mblk_t *mp;

#if LOGLEVEL&LOGSRV
 strlog(50,0,0,SL_ERROR,"rsrv started");
#endif

 /* Verbindung errmitteln */
 pc = (struct parconn *)q->q_ptr;

 /* Nachrichten weitergeben, solange wie moeglich */
 while ( canput(q->q_next) && (nq = pc->tcp.l_qbot) && (mp = getq(RD(nq))) )
  {
#if LOGLEVEL&LOGUDATA
   strlog(50,msgdsize(mp),0,SL_ERROR,"rsrv data");
#endif
   putnext(q,mp); 
  }

#if LOGLEVEL&LOGSRV
 strlog(50,0,0,SL_ERROR,"rsrv done");
#endif
}

/*
  TCP Kanal schliessen, falls noetig
*/
static int par_disconnect(pc,force)
struct parconn *pc;
int force;
{
 union T_primitives *tp;
 mblk_t *mb;

#if LOGLEVEL&LOGCTRL
 strlog(50,force,0,SL_ERROR,"par_disconnect started");
#endif

 /* Geht nur mit Verbindung und in gewissen Zustaenden */
 if ( !pc->tcp.l_qbot ||
      ((pc->state != PS_CONNECTED) && 
      (!force || ((pc->state != PS_CONNECTING) && (pc->state != PS_CONFIRMING)))) )
  return 0;

 /* Zustand aufsetzen */
 pc->state = PS_IDLE;

 /* ORDREL_REQ aufsetzen und absenden */
 if ( pc->qptr )
  {
   /* Nachrichtenblock reservieren */
   if ( !(mb = allocb(usizeof(tp->ordrel_req),BPRI_MED)) )
    {
     pc->state = PS_ERROR;
     return 0;
    }
#if LOGLEVEL&LOGCTRL
   strlog(50,0,0,SL_ERROR,"par_disconnect ordrel_req");
#endif
   mb->b_datap->db_type = M_PROTO;
   tp = (union T_primitives *)mb->b_wptr;
   tp->type = T_ORDREL_REQ;
   mb->b_wptr += usizeof(tp->ordrel_req);
   mb->b_cont = 0;
   putdown(pc,WR(pc->qptr),mb);
  }

 /* Alles in Ordung */
#if LOGLEVEL&LOGCTRL
 strlog(50,0,0,SL_ERROR,"par_disconnect done");
#endif
 return 1;
}

/*
  Verbindung aufbauen
*/
static int par_connect(pc,mp,ip,par_req)
struct parconn *pc;
mblk_t *mp;
struct iocblk *ip;
struct parallel_connect *par_req;
{
 struct parallel_connect *dst;
 union T_primitives *tp;
 mblk_t *mb;
 
#if LOGLEVEL&LOGCTRL
 strlog(50,0,0,SL_ERROR,"par_connect started");
#endif

 /* Geht nur mit Verbindung */
 if ( !pc->tcp.l_qbot || !pc->qptr ) return 0;

 /* Nur in einem Zustand erlaubt */
 if ( pc->state != PS_IDLE ) 
  {
   /* Fehlercode setzen */
   switch (pc->state)
    {
     case PS_CONNECTING    : ip->ioc_error = EALREADY;
			     break;
     case PS_CONFIRMING    : ip->ioc_error = EBUSY;
			     break;
     case PS_CONNECTED     : ip->ioc_error = EEXIST;
			     break;
     case PS_ERROR         : ip->ioc_error = EIO;
			     break;
    }
   /* Fehler melden */
   return 0;
  }

 /* Nachrichtenblock reservieren */
 if ( !(mb = allocb(usizeof(tp->conn_req)+usizeof(*par_req),BPRI_MED)) )
  {
   pc->state = PS_ERROR;
   return 0;
  }
 
 /* IOCTL vermerken */
 if ( pc->iow ) freemsg(pc->iow);
 pc->iow = mp;

 /* Neuen Zustand setzen */
 pc->state = PS_CONNECTING;

 /* CONN_REQ aufsetzen und absenden */
#if LOGLEVEL&LOGCTRL
 strlog(50,0,0,SL_ERROR,"par_connect conn_req");
#endif
 mb->b_datap->db_type = M_PROTO;
 tp = (union T_primitives *)mb->b_wptr;
 tp->type = T_CONN_REQ;
 tp->conn_req.DEST_length = usizeof(*par_req);
 tp->conn_req.DEST_offset = usizeof(tp->conn_req);
 tp->conn_req.OPT_length = tp->conn_req.OPT_offset = 0;
 mb->b_wptr += usizeof(tp->conn_req);
 dst = (struct parallel_connect *)mb->b_wptr;
 *dst = *par_req;
 mb->b_wptr += usizeof(*par_req);
 mb->b_cont = 0;
 putdown(pc,WR(pc->qptr),mb);

 /* Erfolg melden */
#if LOGLEVEL&LOGCTRL
 strlog(50,0,0,SL_ERROR,"par_connect done");
#endif
 return 1;
}
