#include <grp.h>
#include <pwd.h>
#include <time.h>
#include <stdio.h>
#include <varargs.h>

#include <rpc/rpc.h>

#include <sys/param.h>
#include <sys/dg_process_info.h>

#include <saphir/lnm/lnm.h>

#define SYSDIRNAME      "LNM$SYSTEM_DIRECTORY"
#define PIDDIRNAME      "LNM$PROCESS_DIRECTORY"
#define SYSNAME         "LNM$SYSTEM_TABLE"
#define PIDNAME         "LNM$PROCESS_TABLE"

#define MAXINTDEFLEN    10
#define NOARG           ((char *)0)

/*
  Interne Authorisierungsstruktur.
*/
struct auth_lnm
       {
	int     uid;    /* Benuzter           */
	int	gid;	/* Gruppe             */
	int	pid;	/* Prozess            */
	int 	sysnam; /* SYSNAM oder 'root' */
     	int 	grpnam; /* GRPNAM	      */
       };

static table *sysdir = 0,*curpid,**piddir = 0,**tbuf[2] = { 0, 0 },*in,*lnmtab;
static char myself[MAXHOSTNAMELEN] = "",jobname[30],gidname[30];
static int ntab[2] = { 0, 0 },atab[2] = { 0, 0 },marked,lntres,lnmid = 0;
static struct auth_lnm auth_lnm;
static int apid = 0,npid = 0;

extern char *malloc(),*strdup();
extern logname *bsearch();

table *create_table(),**find_table(),*addlnt();
logname *findlnm(),*trnlnm(),*getlnm();

/******
             MACROs

   MALLOC              Speicherallokation
   TABLE	       Tabellenadresse
   WERT                Werte eines logischen Namens
   WLEN                Anzahl der Werte eines logischen Namens
   NAME                Name eines Wertes
   FLAGS               Attribute eines Wertes
   NAMES               Logische Namen einer Tabelle
   NSLEN               Anzahl der logischen Namen einer Tabelle
   LNAME               Ein logischer Name einer Tabelle

   GID                 Gruppenkennung ermitteln
   UID                 Benutzerkennung ermitteln
   SYSNAM              SYSNAM Privileg ermitteln
   GRPNAM              GRPNAM Privileg ermitteln

   PROALL              Alle Protectionbits einer Klasse
   PROANY              Alle Protectionbits aller Klassen

   verify_protection   Protection verifizieren

******/

#define MALLOC(t,n)          ((t *)malloc((n)*sizeof(t)))

#define TABLE		     info.loginfo_u.tptr

#define WERT(l)              ((l)->info.loginfo_u.wert.wert_val)
#define WLEN(l)              ((l)->info.loginfo_u.wert.wert_len)
#define NAME(l,i)            (WERT(l)[i].name)
#define FLAGS(l,i)           (WERT(l)[i].flags)

#define NAMES(t)             ((t)->names.names_val)
#define NSLEN(t)             ((t)->names.names_len)
#define LNAME(t,i)           (NAMES(t)[i])

#define GID(a)               ((a) ? (a)->gid : 0)
#define UID(a)               ((a) ? (a)->uid : 0)
#define SYSNAM(a)            ((a) ? (a)->sysnam : 1)
#define GRPNAM(a)            ((a) ? (a)->grpnam : 1)

#define PROALL               (PRORD|PROWR|PRODE)
#define PROANY               (PROSYSTEM(PROALL)|PROGROUP(PROALL)|\
			      PROOWNER(PROALL)|PROWORLD(PROALL))

#define verify_protection(p) ((p == -1) || !(p&PROANY))

/******
              SCHNITTSTELLEN

    crelnm_1  erzeuge einen logischen Namen
    crelnt_1  erzeuge eine Tabelle logischer Namen
    dellnm_1  loesche einen logischen Namen oder eine Tabelle logischer Namen
    dumplnt_1 lese eine Tabelle logischer Namen aus
    spawn_1   Prozesstabellen erzeugen
    trnlnm_1  suche logischen Namen

******/

static logname tmpres = { 0 };

/* 
  Erzeugen eines logischen Namens nach Vorgabe durch den Benutzer in einer Definitions-
  struktur 'cl'. Nachdem die Zugriffsberechtigung des Benutzers ueberprueft worden ist,
  wird die Tabelle ermittelt, in die der logische Name eingetragen werden soll. Der
  Benutzer muss auf diese Tabelle Schreiberlaubnis haben. Ist der Zugriff auf die
  Tabelle verifiziert, muss vor Eintrag des Namens festgestellt werden, ob bereits
  ein derartiger Name auf dem gleichen Zugrrifsmode existiert, der dann geloescht
  werden muss. Dabei muessen natuerlich das LNM$M_NO_ALIAS Flag sowie, im Falle einer
  zu loeschenden Tabelle, auch das LNM$M_NO_DELETE Flag beachtet werden.
*/
logname *crelnm_1(cl,msg)
crelnm_t *cl;
struct svc_req *msg;
{
 /* Allgemeine Zugriffsberechtigung ueberpruefen */
 if ( tmpres.acmode = -verify_auth(msg,cl->pid,0) ) return &tmpres;
 /* Parameter ueberpruefen */
 if ( tmpres.acmode = -verify_acmode(cl->acmode,1) ) return &tmpres;
 cl->flags &= (LNM$M_CONFINE|LNM$M_NO_ALIAS);
 /* Vatertabelle suchen und auf Schreibberechtigung pruefen */
 if ( tmpres.acmode = -translate_table(cl->parent,1) ) return &tmpres;
 if ( !ntab[1] )
  {
   tmpres.acmode = -SS$_NOLOGTAB;
   return &tmpres;
  }
 /* Name anlegen */
 if ( tmpres.acmode = -addlnm(tbuf[1][0],(crelnt_t *)0,cl,&auth_lnm) ) return &tmpres;
 /* Ergebnis melden */
 return getlnm();
}

/*
  Erzeuge eine Tabelle logischer Namen. In wesentlichen Zuegen arbeitet 'crelnt_1'
  genau wie 'crelnm_1', so dass hier auf eine nochmalige Erlaeuterung verzichtet
  werden kann. Ein kleines Problem sind lediglich die Defaultnamen und die Option
  LNM$M_CREATE_IF, die ein paar besondere Operationen erfordern.
*/
logname *crelnt_1(cl,msg)
crelnt_t *cl;
struct svc_req *msg;
{
 static int unique = 0;
 char namebuf[20],*scan,ch; 
 table *parent,*cur;
 crelnt_t mcl;
 logname *lnt;
 int nlen;

 /* Defaultwert fuer den Namen ermitteln */
 mcl = *cl;
 if ( !mcl.name || !*mcl.name )
  {
   sprintf(namebuf,"LNM$%08X",++unique);
   mcl.name = namebuf;
  }
 /* Allgemeine Zugriffsberechtigung ueberpruefen */
 if ( tmpres.acmode = -verify_auth(msg,mcl.pid,0) ) return &tmpres;
 /* Parameter ueberpruefen */
 if ( tmpres.acmode = -verify_acmode(mcl.acmode,1) ) return &tmpres;
 if ( !verify_protection(mcl.prot) )
  {
   tmpres.acmode = -SS$_BADPARAM;
   return &tmpres;
  }
 mcl.flags &= LNM$M_CONFINE|LNM$M_NO_ALIAS|LNM$M_CREATE_IF;
 /* Syntax des Namens verifizieren: 1 bis 31 Zeichen aus A-Za-z0-9_$ */
 nlen = 0;
 scan = mcl.name;
 while ( (ch = *scan++) && (nlen++ < 31) &&
	 ((ch == '$') || (ch == '_') || ((ch >= '0') && (ch <= '9')) ||
         ((ch >= 'A') && (ch <= 'Z')) || ((ch >= 'a') && (ch <= 'z'))) );
 /* Bei Fehler aufhoeren */
 if ( !nlen || (nlen > 31) || ch )
  {
   tmpres.acmode = -SS$_IVLOGTAB;
   return &tmpres;
  }
 /* Vatertabelle suchen und Zugriffsrecht ueberpruefen */
 if ( tmpres.acmode = -translate_table(mcl.parent,1) ) return &tmpres;
 if ( !ntab[1] )
  {
   tmpres.acmode = -SS$_NOLOGTAB;
   return &tmpres;
  }
 /* Je nach Vatertabelle wird die neue Tabelle in einer der Defaultdirectories platziert */
 parent = tbuf[1][0]->insysdir ? sysdir : curpid;
 /* Nachsehen, ob diese Tabelle schon definiert ist */
 if ( (lnt = findlnm(parent,mcl.name,0,mcl.acmode)) && (lnt->acmode == mcl.acmode) )
  /* LNM$M_CREATE_IF fuehrt dann zu einer Warnung */
  if ( mcl.flags&LNM$M_CREATE_IF )
   {
    tmpres = *lnt;
    tmpres.flags |= LNM$M_EXISTS;
    return &tmpres;
   }
  /* Ansonsten darf die Vatertabelle nicht angetastet werden */
  else if ( lnt->info.table == TRUE )
   /* Alle Tabellen durchgehen */
   for ( cur = tbuf[1][0] ; ; cur = (table *)cur->partab )
    {
     /* Vatertabelle gefunden */
     if ( (long)cur == lnt->TABLE )
      {
       tmpres.acmode = -SS$_PARENT_DEL;
       return &tmpres;
      }
     /* Weitersuchen, wenn moeglich */
     if ( (long)cur == cur->partab ) break;
    }
 /* Tabelle in der Vatertabelle anlegen */
 if ( !addlnt(&mcl,&auth_lnm,parent) )
  {
   tmpres.acmode = -lntres;
   return &tmpres;
  }
 /* Vatertabelle vermerken */
 lnt = getlnm();
 cur = (table *)lnt->TABLE;
 cur->parent = tbuf[1][0]->name;
 cur->partab = (long)tbuf[1][0];
 /* Ergebnis melden */
 return lnt;
}

/*
  Elemination eines logischen Namens. Nach dem Ermitteln der Vatertabelle mit Schreib-
  berechtigung wird der Name einfach mit 'verify_del' geloescht, falls ein Name ange-
  geben wurde. Zudem werden alle gleichlautenden logischen Namen mit niedrigeren 
  Zugriffsmodes ebenfalls geloescht. Dazu bedient sich 'dellnm_1' eines kleinen
  Tricks und ruft 'verify_add' auf. Ohne Angabe eines Namens werden alle logischen
  Namen mit einem Zugriffsmode kleiner als der angegebene in der ersten Tabelle ge-
  loescht.
*/
int *dellnm_1(del,msg)
dellnm_t *del;
struct svc_req *msg;
{
 static int res;
 int flags = LNM$M_NO_ALIAS|LNM$M_NO_DELETE,ix;
 logname *lnm;

 /* Allgemeine Zugriffsberechtigung ueberpruefen */
 if ( res = verify_auth(msg,del->pid,0) ) return &res;
 /* Parameter ueberpruefen */
 if ( res = verify_acmode(del->acmode,1) ) return &res;
 /* Vatertabelle suchen */
 if ( res = translate_table(del->parent,1) ) return &res;
 if ( !ntab[1] )
  {
   res = SS$_NOLOGTAB;
   return &res;
  }
 /* Nachsehen, was der Aufrufer will */
 if ( del->name && *del->name )
  {
   /* Logischen Namen oder Tabelle suchen */
   if ( !(lnm = trnlnm(del->name,tbuf[1],ntab[1],0,del->acmode)) ||
        (lnm->acmode != del->acmode) )
    {
     res = SS$_NOLOGNAM;
     return &res;
    }
   /* Loeschen vorbereiten */
   if ( res = verify_add(in,del->name,del->acmode,&flags,0,&auth_lnm) ) return &res;
   /* Loeschen des Namens */
   lnmtab = 0;
   verify_add(in,del->name,del->acmode,&flags,1,&auth_lnm);
  }
 else
  {
   /* Mehrere logische Namen loeschen */
   for ( ix = NSLEN(tbuf[1][0]) ; ix-- ; )
    if ( LNAME(tbuf[1][0],ix).acmode >= del->acmode )
     verify_del(tbuf[1][0],&LNAME(tbuf[1][0],ix),1,&auth_lnm);
   /* Kindtabellen vernichten */
   delcld(tbuf[1][0]);
  }
 /* Ergebnis melden */
 return &res;
}

/*
  Suchen einer Tabelle logischer Namen und melden ihres gesamten Inhalts, d.h. aller
  darin definierten logischen Namen und Tabellen.
*/
table *dumplnt_1(proc,msg)
dumplnt_t *proc;
struct svc_req *msg;
{
 static table res;

 /* Temporaren Buffer initialisieren */
 res.name = res.parent = "";
 NSLEN(&res) = 0;
 NAMES(&res) = 0;
 /* Allgemeines Zugriffsrecht validieren */
 if ( res.nalloc = -verify_auth(msg,proc->pid,0) ) return &res;
 /* Vatertabelle suchen */
 if ( res.nalloc = -translate_table(proc->parent,0) ) return &res;
 if ( !ntab[1] )
  {
   res.nalloc = -SS$_NOLOGTAB;
   return &res;
  }
 /* Tabelle suchen */
 ntab[0] = 0;
 if ( res.nalloc = -trnlnt(proc->name,tbuf[1],ntab[1],LNM$C_MAXDEPTH,0,0) ) return &res;
 if ( !ntab[0] )
  {
   res.nalloc = -SS$_NOLOGTAB;
   return &res;
  }
 /* Anfrage beantworten */
 return tbuf[0][0];
}

/*
  Erzeuge die Defaulttabellen eines neuen Prozesses, d.h. simulieren ein LIB$SPAWN.
*/
int *spawn_1(pid,msg)
int *pid;
struct svc_req *msg;
{
 static int res;

 /* Allgemeine Zugriffsberechtigung ueberpruefen */
 res = verify_auth(msg,*pid,1);
 /* Ergebnis melden */
 return &res;
}

/*
  Suchen eine logischen Name in einer vorgegebenen Tabelle. Nach den ueblichen
  Ueberpruefungen werden die Vatertabellen ermitteln und darin nach dem logischen
  Namen gesucht.
*/
logname *trnlnm_1(trn,msg)
trnlnm_t *trn;
struct svc_req *msg;
{
 logname *lnm;

 /* Allgemeine Zugriffsberechtigung ueberpruefen */
 if ( tmpres.acmode = -verify_auth(msg,trn->pid,0) ) return &tmpres;
 /* Parameter ueberpruefen */
 if ( tmpres.acmode = -verify_acmode(trn->acmode,0) ) return &tmpres;
 /* Vatertabelle suchen */
 if ( tmpres.acmode = -translate_table(trn->parent,0) ) return &tmpres;
 if ( !ntab[1] )
  {
   tmpres.acmode = -SS$_NOLOGTAB;
   return &tmpres;
  }
 /* Logischen Namen suchen und zurueckgeben */
 if ( !(lnm = trnlnm(trn->name,tbuf[1],ntab[1],trn->flags&LNM$M_CASE_BLIND,trn->acmode)) )
  {
   tmpres.acmode = -SS$_NOLOGNAM;
   return &tmpres;
  }
 /* Ergebnis melden */
 return lnm;
}

/******

              AUTHORISIERUNG

    verify_access  Zugriffsrecht eines Benutzers auf eine Tabelle ueberpruefen
    verify_acmode  Zugriffsmode ueberprurefen (MACRO)
    verify_add     Eintrag eines logischen Namens ueberpruefen
    verify_auth    Ueberpruefung der Authosisierung eines RPC-Clients
    verify_clean   Loesche Tabellen bereits verstorbener Prozesse
    verify_del     Logischen Namen oder Tabelle loeschen, falls erlaubt

    cleanlnt       Tabelle loschen, falls der zugehoerige Prozess nicht existiert

******/

/*
  Stelle mit Hilfe der Authorisierungsstruktur 'auth' fest, ob der Benutzer das
  verlangte Zugriffsrecht 'mode' - PRORD, PROWR oder PRODE - auf die angegebene
  Tabelle hat.
*/
verify_access(lnt,mode,auth)
logname *lnt;
int mode;
struct auth_lnm *auth;
{
 table *tab = (table *)lnt->TABLE;
 int prot = tab->protection;

 return (prot == -1) ||
        (prot&PROWORLD(mode)) ||
	((GID(auth) == tab->gid) && (GRPNAM(auth) || (prot&PROGROUP(mode)))) ||
	((UID(auth) == tab->uid) && (prot&PROOWNER(mode))) ||
        (SYSNAM(auth) && (prot&PROSYSTEM(mode)));
}

/*
  Zugriffsmode ueberpruefen.
*/
verify_acmode(a,needpriv)
int a,needpriv;
{
 /* Normale Modes fuer jeden Benutzer */
 if ( (a == PSL$C_USER) || (a == PSL$C_SUPER) ) return 0;
 /* Modes mit SYSNAM Privileg */
 if ( (a != PSL$C_EXEC) && (a != PSL$C_KERNEL) ) return SS$_BADPARAM;
 if ( needpriv && !auth_lnm.sysnam ) return SS$_NOPRIV;
 /* Alles klar */
 return 0;
}

/*
  Ueberpruefe, ob der logische Name 'name' im Zugriffsmode 'acmode' definiert werden
  kann. Dabei ist zu beachten, dass eine etwaige definierte Tabelle mit LNM$M_NO_DELETE
  nicht geloescht werden kann und gewisse Vorkehrungen bei Benutzung von LNM$M_NO_ALIAS.
*/
verify_add(tab,name,acmode,flags,exec,auth)
table *tab;
char *name;
int acmode,*flags,exec;
struct auth_lnm *auth;
{
 int res = -1;
 logname *up;

 /* Erst einmal nachsehen, ob ueberhaupt ein logischer Name 'name' da ist */
 if ( !(up = findlnm(tab,name,0,PSL$C_USER+1)) ) return 0;
 /* Alle Zugriffsmodes des logischen Namens durchgehen */
 do
  /* Weniger privilegierte Zugriffsmodes */
  if ( acmode <= up->acmode )
   {
    /* Muss geloescht werden */
    if ( acmode == up->acmode )
     {
      *flags |= LNM$M_SUPERSEDE;
      if ( !exec || (up != getlnm()) ) res = verify_del(tab,up,exec,auth);
     }
    else if ( *flags&LNM$M_NO_ALIAS )
     res = verify_del(tab,up,exec,auth);
    /* Fehler bearbeiten und Zeiger korrigieren */
    switch (res)
     {
      case  0 : if ( exec ) up--;
	        res = -1;	
      case -1 : break;
      default : return res;
     }
   }
  /* Hoeher privilegierte Zugriffsmodes */
  else if ( !(*flags&LNM$M_NO_DELETE) && (up->flags&LNM$M_NO_ALIAS) )
   return SS$_DUPLNAM; 
 while ( (++up < &LNAME(tab,NSLEN(tab))) && !strcmp(name,up->name) );
 /* Kindtabellen vernichten */
 delcld(tab);
 /* Alles in Ordung */
 return 0;
}

/*
  Diese Routine hat verschiedene Aufgaben. Zum einen werden alle zum Betrieb des LNM
  noetigen globalen Groessen auf Defaultwerte gesetzt. Zudem wird die Authorisierung
  des Clients ueberprueft. Dann werden alle notwendigen Tabellen logischer Namen fuer
  die Prozesses des durch 'pid' bezeichneten Stammbaums erzeugt, falls noch nicht vor-
  handen. 'verify_auth' uebertraegt saemtliche Ergebnisse seiner Arbeit in globale
  Variablen, wo sie dann von den anderen Teilen des Programms gefunden werden koennen.
*/
static verify_auth(msg,pid,neg)
struct svc_req *msg;
int pid,neg;
{
 struct passwd *pw,*getpwnam();
 struct group *gr,*getgrnam();
 struct authunix_parms *au;
 crelnt_t cresysdir;
 table *systab;
 int gcnt;

 /* Globalen Rueckgabebuffer initialisieren */
 tmpres.name = tmpres.parent = "";
 tmpres.info.table = TRUE;
 /* 'pid' validieren */
 if ( pid <= 0 ) return SS$_BADPARAM;
 curpid = 0;
 /* Verwaltungsgroessen initialisieren */
 marked = 0;
 /* Authorisierungsstruktur suchen */
 if ( (msg->rq_cred.oa_flavor != AUTH_UNIX) ||
      !(au = (struct authunix_parms *)msg->rq_clntcred) )
  return SS$_NOPRIV;
 /* Name der lokalen Maschine */
 if ( !myself[0] && gethostname(myself,sizeof(myself)) ) return SS$_INSFMEM;
 /* Zugriff sind nur innerhalb einer Maschine gestattet */
 if ( !au->aup_machname || strcmp(myself,au->aup_machname) ) return SS$_NOPRIV;
 /* Alle Authorisierungsinformationen auswerten und in globale Variablen uebertragen */
 auth_lnm.pid = pid;
 auth_lnm.uid = au->aup_uid;
 auth_lnm.gid = au->aup_gid;
 auth_lnm.sysnam = auth_lnm.grpnam = 0;
 if ( !au->aup_uid || 
      has_privilege(au->aup_uid,"SYSNAM") || 
      has_privilege(au->aup_uid,"OPER") )
  auth_lnm.sysnam = auth_lnm.grpnam = 1;
 else if ( has_privilege(au->aup_uid,"GRPNAM") )
  auth_lnm.grpnam = 1;
 /* Arbeitsvariablen initialisieren */
 curpid = 0;
 /* LNM$SYSTEM_DIRECTORY erzeugen, falls noch nicht vorhanden */
 if ( !sysdir )
  {
   /* Definitionsblock aufsetzen */
   cresysdir.name = SYSDIRNAME;
   cresysdir.pid = getpid();
   cresysdir.flags = LNM$M_NO_ALIAS;
   cresysdir.quota = -1;
   cresysdir.acmode = PSL$C_KERNEL;
   cresysdir.prot = PROSYSTEM(PRORD|PROWR)|PROOWNER(PRORD|PROWR)|
                    PROGROUP(PRORD)|PROWORLD(PRORD);
   /* Tabelle logischer Namen anlegen */
   if ( !(sysdir = addlnt(&cresysdir,(struct auth_lnm *)0,(table *)0)) ) return SS$_INSFMEM;
   /* Das selbe fuer LNM$SYSTEM_TABLE */
   cresysdir.name = SYSNAME;
   if ( !(systab = addlnt(&cresysdir,(struct auth_lnm *)0,sysdir)) )
    {
     /* Bei Fehler aufraeumen */
     dellnt(sysdir);
     return SS$_INSFMEM;
    }
   /* Systemtabellen mit Defaultwerten versehen */
   preset_system(sysdir);
  }
 /* Gruppentabelle erzeugen, falls noch nicht vorhanden */
 if ( !create_gid(auth_lnm.gid) ) return SS$_INSFMEM;
 /* Alte Tabellen loeschen, falls nicht mehr benoetigt */
 verify_clean();
 /* Prozessstammbaum durchgehen und alle noetigen Tabellen und logischen Namen anlegen */
 if ( auth_lnm.uid != create_all(neg ? -pid : pid,-1) ) return SS$_INSFMEM;
 /* Fertig, keine Fehler aufgetreten */
 return 0;
}

/*
  Loesche alle Tabellen, deren Benutzer verstorben sind. Alle 10 Sekunden wird die
  Routine aktiv und loescht dann entweder allse nicht mehr benoetigten Tabellen aus
  der 'piddir' oder alle Jobtabellen aus der 'sysdir'.
*/
verify_clean()
{
 static time_t last = 0;
 static int state = 1;
 time_t cur;
 int ix;

 /* Ermitteln, ob die Zeit abgelaufen ist */
 time(&cur);
 if ( last && ((cur-last) < 10) ) return;
 last = cur;
 /* Arbeit aufnehmen */
 if ( (state = 1-state) == 1 )
  /* Prozessdirectories bereinigen */
  for ( ix = npid ; ix-- ; )
   cleanlnt((logname *)0,piddir[ix]); 
 else
  /* Jobtabellen bereinigen */
  for ( ix = NSLEN(sysdir) ; ix-- ; )
   if ( (strlen(LNAME(sysdir,ix).name) == 24) &&
        !memcmp(LNAME(sysdir,ix).name,"LNM$JOB_",8) &&
        (LNAME(sysdir,ix).flags&LNM$M_NO_DELETE) )
    cleanlnt(&LNAME(sysdir,ix),sysdir);
}

/*
  Entferne den logischen Namen 'lnm' aus der Tabelle 'tab', falls dies erlaubt ist.
  Ist 'exec' 0, so wird nur eine Ueberpruefung vorgenommen.
*/
verify_del(tab,lnm,exec,auth)
table *tab;
logname *lnm;
int exec;
struct auth_lnm *auth;
{
 table *tbl = 0;
 int lix,freed;

 /* Eine Tabelle mit LNM$M_NO_DELETE darf niemals geloescht werden */
 if ( lnm->info.table == TRUE )
  {
   if ( (lnm->flags&LNM$M_NO_DELETE) || !verify_access(lnm,PRODE,auth) ) return SS$_NOPRIV;
   /* Tabelle vermerken */
   tbl = (table *)lnm->TABLE;
  }
 /* Falls 'exec' 0 ist, sind wir schon fertig */
 if ( !exec ) return 0;
 /* Logischen Namen oder Tabelle loeschen */
 if ( ((freed = dellnm(lnm)) > 0) && (tab->free >= 0) ) tab->free += freed;
 /* Tabelle aufraeumen */
 remlnm(tab,lnm);
 /* Tabellen zum Loeschen vorbereiten */
 if ( tbl ) marklnt(tab,tbl);
 /* Erfolg melden */
 return 0;
}

/*
  Loesche eine Tabelle logischer Namen, falls der erzeugende Prozess bereits verstorben
  ist. Je nach Wunsch wird eine reine Tabelle - 'lnt' ist 0 - oder ein mit einer Tabelle
  assoziierte logische Name geloescht.
*/ 
cleanlnt(lnt,tab)
logname *lnt;
table *tab;
{
 int key = DG_PROCESS_INFO_INITIAL_KEY;
 struct dg_process_info pi;
 table *deltab = tab;

 /* Parameter auslesen */
 if ( lnt ) deltab = (table *)lnt->TABLE;
 /* Nachsehen, ob der Prozess noch existiert */
 if ( dg_process_info(DG_PROCESS_INFO_SELECTOR_PID,deltab->pid,
		      DG_PROCESS_INFO_CMD_NAME_NULL,&key,&pi,
		      DG_PROCESS_INFO_VERSION_0) )
  /* Remark: may be done by kill */
  return;
 /* Tabelle oder logischen Namen loeschen */
 if ( lnt )
  {
   lnt->flags &= ~LNM$M_NO_DELETE;
   verify_del(tab,lnt,1,(struct auth_lnm *)0);
   delcld(tab);
  }
 else
  dellnt(deltab);
}

/******

              LOGISCHE NAMEN

    addlnm    logischen Namen oder Tabelle in eine Tabelle eintragen
    copylnm   dupliziere einen logischen Namen
    deflnm    definiere einen per Default gesetzten logischen Namen 
    dellnm    loesche einen logischen Namen
    findlnm   logischen Namen in einer Tabelle suchen
    getlnm    ermittele zuletzt eingetragenen logischen Namen
    remlnm    Logischen Namen aus einer Tabelle eleminieren
    trnlnm    uebersetze einen logischen Namen

    resort    Tabelle logischer Namen alphabetisch und nach Zugriffsmode sortieren
    tabcmp    Zwei logische Namen vergleichen
    tabcmpi   Zwei logische Namen ohne Ruecksicht auf Gross- und Kleinschreibung vergleichen

******/

/*
  Diese Routine erzeugt einen logischen Namen - falls 'islnm' nicht 0 ist - bzw.
  eine Tabelle logischer Namen - dann ist 'islnm' 0 und 'islnt' darf nicht 0 sein -
  und traegt die neu erzeugte Struktur in die bereits vorhandene Tabelle 'where'
  ein. Bei Erzeugen einer Tabelle sind gewisse Authorisierungsinformationen aus
  'auth' wichtig. Zudem muss dann natuerlich auch die Verwaltungsstruktur der
  zur neuen Tabelle gehoerigen logischen Namen bekannt sein, die in diesem Fall
  'lnt' entnommen wird.
*/
addlnm(where,islnt,islnm,auth,lnt)
table *where,*lnt;
crelnt_t *islnt;
crelnm_t *islnm;
struct auth_lnm *auth;
{
 int i,bytes = -1,acmode,flags,res;
 logname *nl,*narr;
 char *name;

 /* Generelle Parameter auslesen */
 if ( islnm )
  {
   name = islnm->name;
   acmode = islnm->acmode;
   flags = islnm->flags;
  }
 else
  {
   name = islnt->name;
   acmode = islnt->acmode;
   flags = islnt->flags;
  }  
 /* Logische Namen zum Loeschen vorbereiten */
 if ( res = verify_add(where,name,acmode,&flags,0,auth) ) return res;
 /* Bei der Definition eines logischen Namens wird die Tabellenquote beruecksichtigt */
 if ( islnm && (where->free >= 0) )
  {
   /* Groesse der Struktur, des Namens und der Zeiger auf die Werte des logischen Namens */
   bytes = sizeof(logname)+strlen(name)+1+(i = islnm->names.names_len)*sizeof(equiv_t);
   /* Alle Werte des logischen Namens */
   while ( i-- ) bytes += strlen(islnm->names.names_val[i].name)+1;
   /* Quote ueberpruefen */
   if ( bytes > where->free ) return SS$_EXLNMQUOTA;
  } 
 /* Freies Feld fuer den logischen Namen in der Tabelle 'where' reservieren */
 if ( where->nalloc == NSLEN(where) )
  {
   /* Feld muss neu angelegt werden */
   if ( !(narr = MALLOC(logname,where->nalloc+10)) ) return SS$_INSFMEM;
   /* Alte Werte eventuell kopieren und Verwaltungsstruktur freigeben */
   if ( where->nalloc )
    {
     memcpy((char *)narr,(char *)NAMES(where),where->nalloc*sizeof(LNAME(where,0)));
     free(NAMES(where));
    }
   /* Neue Verwaltungsstruktur aufsetzen */
   where->nalloc += 10;
   NAMES(where) = narr; 
  } 
 /* Zeiger auf den reservierten Prototypen ermitteln und mit einer Kennung versehen */
 nl = &LNAME(where,NSLEN(where));
 nl->id = ++lnmid;
 lnmtab = where;
 /* Name der Vatertabelle vermerken */
 nl->parent = where->name;
 /* Fuer logische Namen das Zeigerfeld fuer die Werte reservieren */
 if ( islnm && !(WERT(nl) = MALLOC(equiv_t,islnm->names.names_len)) )
  return SS$_INSFMEM;
 /* Allgemeine Parameter eintragen */
 nl->info.table = FALSE;
 WLEN(nl) = 0;
 if ( !(nl->name = strdup(name)) )
  {
   /* Logischen Namen oder Tabelle soweit definiert loeschen und Fehler melden */
   dellnm(nl);
   return SS$_INSFMEM;
  }
 nl->acmode = acmode;
 nl->flags = flags;
 /* Die Initialisierung erfolgt je nach Typ der zu definierenden Struktur */
 if ( islnm )
  /* Speicherplatz fuer alle Werte reservieren */
  for ( i = 0 ; i < islnm->names.names_len ; i++ )
   /* Einen Wert anlegen */
   if ( !(NAME(nl,i) = strdup(islnm->names.names_val[i].name)) )
    {
     /* Kein Speicher mehr zu bekommen */
     dellnm(nl);
     return SS$_INSFMEM;
    }
   else
    {
     /* Optionen eintragen */
     FLAGS(nl,i) = islnm->names.names_val[i].flags&(LNM$M_CONCEALED|LNM$M_TERMINAL);
     /* Zaehler fuer die Werte aktualisieren */
     WLEN(nl)++;
    }
 else
  {
   /* Logische Tabelle anlegen */
   nl->info.table = TRUE;
   /* Felder aus der Definitionsstruktur uebernehmen */
   lnt->quota = islnt->quota;
   lnt->protection = islnt->prot;
   /* Felder aus der Authorisierungsstruktur uebernehmen */
   lnt->uid = UID(auth);
   lnt->gid = GID(auth);
   /* Verwaltungsstruktur fuer die logischen Namen anlegen */
   nl->TABLE = (long)lnt;
  }
 /* Beruecksichtigung der Quoten */
 if ( bytes > 0 ) where->free -= bytes;
 /* Logischen Namen oder Tabelle eintragen und Tabelle sortieren */
 NSLEN(where)++;
 resort(where);
 /* Tabelle aufraeumen */
 verify_add(where,name,acmode,&flags,1,auth);
 /* Erfolg melden */
 return 0;
}

/*
  Fertige eine Kopie des logischen Namens 'lnm' an und trage diese in die Tabelle
  'dir' logischer Namen an. Handelt es sich bei dem logischen Namen in Wirklichkeit
  um eine Tabelle logischer Namen, so wird die gesamte Tabelle dupliziert, d.h. alle
  darin definierte logische Namen. Der Kontrollparameter 'pdir' ist ein Schutz gegen
  die nicht endende Rekursion, die dadurch entstehen, dass die Defaultdirectories
  LNM$PROCESS_DIRECTORY und LNM$SYSTEM_DIRECTORY in sich selbst definiert sind. 
  Weiterhin koennen Tabellen logischer Namen ausschliesslich in diesen Directories
  angelegt werden, so dass ansonsten 'copylnm' hoechstens zweimal aktiv sein wird.
*/
copylnm(dir,lnm,pdir)
table *dir,*pdir;
logname *lnm;
{
 struct auth_lnm auth;
 table *tab,*ntab;
 crelnt_t nlnt;
 crelnm_t nlnm;
 logname *nl;
 int ix;

 /* Es werden nur logischen Namen kopiert, die nicht CONFINE sind */
 if ( lnm->flags&LNM$M_CONFINE ) return;
 /* Logische Namen und Tabellen muessen separat behandelt werden */
 if ( lnm->info.table == FALSE )
  {
   /* Aufsetzen des Definitionsblocks aus den vorhandenen Informationen */
   nlnm.name = lnm->name;
   nlnm.flags = lnm->flags;
   nlnm.acmode = lnm->acmode;
   nlnm.names.names_len = WLEN(lnm);
   nlnm.names.names_val = WERT(lnm);
   /* Logischen Namen definieren */
   addlnm(dir,(crelnt_t *)0,&nlnm,(struct auth_lnm *)0);
  }
 else
  {
   /* Tabellenstruktur zum weiteren Bearbeiten auslesen */
   tab = (table *)lnm->TABLE;
   /* Defininitionsstruktur aufsetzen */
   nlnt.name = lnm->name;
   nlnt.pid = tab->pid;
   nlnt.flags = lnm->flags;
   nlnt.acmode = lnm->acmode;
   nlnt.quota = tab->quota;
   nlnt.prot = tab->protection;
   /* Authorisierungsstruktur zum Eintragen anlegen */
   auth.uid = tab->uid;
   auth.gid = tab->gid;
   auth.sysnam = auth.grpnam = 1;
   /* Neue Tabelle erzeugen und in die aktuelle Tabelle eintragen */
   if ( !(ntab = addlnt(&nlnt,&auth,dir)) ) return;
   /* Rekursionssicherung */
   if ( tab == pdir )
    {
     /* Diese Tabelle wird in sich selbst definiert */
     nl = getlnm();
     nl->TABLE = (long)dir;
     /* Name kann aus der uebergeordneten Tabelle entnommen werden */
     free(nl->name);
     nl->name = nl->parent;
     /* Diese Tabelle kann nicht vom Benutzer geloescht werden */
     nl->flags |= LNM$M_NO_DELETE;
     /* Freigabe der bereits reservierten Verwaltungsstrukturen */
     free(ntab);
    }
   else
    /* Alle logischen Namen dieser Tabelle durchgehen */
    for ( ix = 0 ; ix < NSLEN(tab) ; ix++ )
     copylnm(ntab,&LNAME(tab,ix),tab);
  }
}

/*
  Die folgende Routine definiert einen logischen Namen und wird nur intern aufgerufen,
  typischerweise von den 'preset_' Routinen. Die variable Anzahl der Parameter beschreibt
  die einzelnen Werte des logischen Namens. Nach Aufsetzen der Definitionsstruktur wird
  einfach 'addlnm' aufgerufen.
*/
deflnm(va_alist)
va_dcl
{
 equiv_t equivs[MAXINTDEFLEN];
 crelnm_t crelnm;
 table *parent;
 va_list args;
 int i,flags;

 /* Einlesen variabler Parameter vorbereiten */
 va_start(args);
 /* Feste Parameter einlesen und Definitionsstruktur fuellen */
 parent = va_arg(args,table *);
 crelnm.name = va_arg(args,char *);
 crelnm.flags = (flags = va_arg(args,int))&~LNM$M_TERMINAL;
 crelnm.acmode = va_arg(args,int);
 /* Liste der variablen Parameter als Werte des logischen Namens interpretieren */
 for ( i = 0 ; (i < MAXINTDEFLEN) && (equivs[i].name = va_arg(args,char *)) ; i++ )
  equivs[i].flags = flags&LNM$M_TERMINAL;
 crelnm.names.names_len = i;
 crelnm.names.names_val = equivs;
 /* Bearbeiten der variablen Liste beenden */
 va_end(args);
 /* Eintragen des logischen Namens in die angegebene Tabelle */
 return addlnm(parent,(crelnt_t *)0,&crelnm,(struct auth_lnm *)0);
}

/*
  Freigabe eines logischen Namens oder einer Tabelle. Die Struktur selbst ist Teil einer
  Tabelle und braucht nicht freigegeben zu werden. Bei einem logischen Namen muessen der
  Name selbst, alle Werte sowie das Wertfeld freigegeben werden. 'dellnm' liefert in diesem
  Fall die Zahl der freigegebenen Bytes. Bei einer Tabelle muessen alle Eintrage dieser
  Tabelle eleminiert werden, wozu 'dellnt' aufgerufen werden. Dazu kommt natuerlich 
  wieder der Name der Tabelle. Bei Tabellen liefert 'dellnm' stets -1.
*/
dellnm(lnm)
logname *lnm;
{
 int i,freed = -1;

 /* Entscheiden, ob eine Tabelle oder nur ein logischer Name vorliegt */
 if ( lnm->info.table == TRUE )
  /* Alle Eintraege der Tabelle loeschen */
  dellnt(lnm->TABLE);
 else
  {
   /* Groesse der Struktur selbst sowie des Wertfeldes */
   freed = sizeof(logname)+(i = WLEN(lnm))*sizeof(equiv_t);
   /* Alle Werte dieses logischen Namens durchgehen */
   while ( i-- )
    {
     /* Speicherplatz des Wertes */
     freed += strlen(NAME(lnm,i))+1;
     /* Speicher des Wertes freigeben */
     free(NAME(lnm,i));
    }
   /* Speicher des Wertfeldes freigeben, falls vorhanden */
   if ( WERT(lnm) ) free(WERT(lnm));
  }
 /* Name des logischen Namens oder der Tabelle freigeben */
 if ( lnm->name )
  {
   /* Anzahl der Bytes nur bei logischen Namen berechnen */
   if ( freed > 0 ) freed += strlen(lnm->name)+1;
   /* Speicher freigeben */
   free(lnm->name);
  }
 /* Meldung der freigemachten Bytes */
 return freed;
}

/*
  Beim Suchen eines logischen Namens in einer Tabelle ueber die eingebaute Binaersuchroutine
  'bsearch' wird zuerst nur auf den logischen Namen unabhaengig von seinem Zugriffsmode
  verglichen. Dann werden alle gleichen logischen Namen mit verschiedenen Zugriffsmodi
  untersucht, bis ein Zugriffsmode gefunden ist, der mindestens so privilegiert ist wie
  der mit 'acmode' geforderte.
*/
logname *findlnm(tab,name,insens,acmode)
table *tab;
char *name;
int insens,acmode;
{
 int tabcmp(),tabcmpi(),(*tabcmpx)();
 logname look,*found;

 /* Bei einer leeren Tabelle kann man die Suche hier abkuerzen */
 if ( !NSLEN(tab) ) return 0;
 /* Ja nach Wert von 'insens' muessen die Textvergleiche ausgefuehrt werden */
 tabcmpx = insens ? tabcmpi : tabcmp;
 /* Suchmuster aufsetzen und logischen Namen suchen */
 look.name = name;
 look.acmode = -1;
 found = bsearch(&look,NAMES(tab),NSLEN(tab),sizeof(LNAME(tab,0)),tabcmpx);
 /* Wenn nicht gefunden, sofort zurueck */
 if ( !found ) return 0;
 /* Niedrigstprivilegierten Zugriffsmode dieses logischen Namens ermitteln */
 while ( (found > NAMES(tab)) && !(*tabcmpx)(found-1,&look) ) found--;
 /* Ersten relativ zu 'acmode' geeigneten Zugriffsmode ermitteln */
 do
  /* Zugriffsmode muss mindestens so privilegiert sein wie 'acmode' */
  if ( acmode >= found->acmode ) return found;
 /* Naechsten Zugriffsmode vornehmen */
 while ( (++found != (&LNAME(tab,NSLEN(tab)))) && !(*tabcmpx)(found,&look) );
 /* Logischer Namen existiert zwar, aber nicht mit dem gewuenschten Zugriffsmode */
 return 0;
}

/*
  Suche den in 'addlnm' eingetragenen letzten logischen Namen. Dazu wurde der Name
  mit einer 32-Bit Kennung versehen, die auch bei laengerem Betrieb eindeutig sein
  sollte.
*/
logname *getlnm()
{
 int i;

 /* Alle logischen Namen durchgehen */
 if ( lnmtab )
  for ( i = NSLEN(lnmtab) ; i-- ; )
   if ( LNAME(lnmtab,i).id == lnmid )
    return &LNAME(lnmtab,i);
 /* Nicht gefunden, sollte niemals vorkommen */
 return 0;
}

/*
  Elemniniere einen logischen Namen aus einer Tabelle.
*/
remlnm(tab,lnm)
table *tab;
logname *lnm;
{
 int i = lnm-NAMES(tab);

 memcpy((char *)&LNAME(tab,i),(char *)&LNAME(tab,i+1),(NSLEN(tab)-i-1)*sizeof(LNAME(tab,0)));
 NSLEN(tab)--;
}

/*
  Logischen Namen in verschiedenen Tabellen mittels 'findlnm' suchen und den ersten
  gefundenen logischen Namen melden.
*/
logname *trnlnm(name,tabset,ntabs,insens,acmode)
char *name;
table **tabset;
int ntabs,insens,acmode;
{
 logname *found;

 /* Bearbeitung aller angegebenen Tabellen */
 while ( ntabs-- )
  /* Suchen des logischen Namens mit den gewaehlten Attributen */
  if ( found = findlnm(*tabset++,name,insens,acmode) )
   {
    /* Melden des gefundenen Namens und der Tabelle */
    in = tabset[-1];
    return found;
   }
 /* Keinen Namen gefunden */
 return 0;
}

/*
  Sortieren einer Tabelle logischer Namen mit der eingebauten Quicksort Routine 'qsort'.
  Die Namen werden dabei primaer alphabetisch und sekundaer nach dem Zugriffsmode sor-
  tiert. Gross- und Kleinschreibung wird unterschieden.
*/
resort(tab)
table *tab;
{
 int tabcmp();

 qsort(NAMES(tab),NSLEN(tab),sizeof(LNAME(tab,0)),tabcmp);
}

/*
  Vergleich zweier logischer Namen nach obigen Kriterien. Neben dem Sortieren benoetigt
  'findlnm' eine Suchoperation, die nicht auf den Zugriffsmode achtet. Dazu wird bei
  dem zu suchenden Element einfach der Zugriffsmode auf einen illegalen Wert (-1) gesetzt.
  Dieser Wert wird dann beruecksichtigt und verhindert eventuell einen vollstaendigen
  Vergleich. 'tabcmp' vergleicht die Namen mit Ruecksicht auf Gross- und Kleinschreibung.
*/
tabcmp(ln1,ln2)
logname *ln1,*ln2;
{
 int del;

 /* Namen vergleichen */
 if ( del = strcmp(ln1->name,ln2->name) ) return del;
 /* Suchmodus gesondert behandeln */
 if ( (ln1->acmode == -1) || (ln2->acmode == -1) ) return 0;
 /* Vollstaendig ueberpruefen */
 return ln2->acmode-ln1->acmode;
}

/*
  Vergleiche wie unter 'tabcmp', allerdings unter Ignorierung von Gross- und Klein-
  schreibung, so dass 'a' und 'A' als gleich betrachtet werden.
*/
tabcmpi(ln1,ln2)
logname *ln1,*ln2;
{
 int del;

 /* Namen vergleichen */
 if ( del = strcmpi(ln1->name,ln2->name) ) return del;
 /* Suchmodus gesondert behandeln */
 if ( (ln1->acmode == -1) || (ln2->acmode == -1) ) return 0;
 /* Vollstaendig ueberpruefen */
 return ln2->acmode-ln1->acmode;
}

/******

              TABELLEN LOGISCHER NAMEN

    addlnt    Tabelle logischer Namen erzeugen
    delcld    Kindtabellen loeschen
    dellnt    Loeschen einer Tabelle logischer Namen
    marklnt   Kindtabellen zum Loeschen vorbereiten
    trnlnt    Uebersetzen eines logischen Namens in einen Satz von Tabellen

******/

/*
  Erzeuge eine Tabelle logischer Namen in einer anderen Tabelle. Es werden alle
  notwendigen Strukturen erzeugt und der zugehoerige logische Name in der Vater-
  tabelle 'parent' angelegt, wobei die Definitionsstruktur 'cl' und die Autho-
  risierungsstruktur 'auth' verwendet werden. Ist 'parent' 0, wird die Tabelle
  in sich selbst angelegt.
*/
table *addlnt(cl,auth,parent)
crelnt_t *cl;
struct auth_lnm *auth;
table *parent;
{
 table *pt,*partab;
 logname *nl;

 /* Verwaltungsstruktur fuer logische Namen erzeugen */
 if ( !(pt = create_table(cl)) )
  {
   lntres = SS$_INSFMEM;
   return 0;
  }
 /* Vatertabelle ermitteln */
 partab = parent ? parent : pt;
 /* Logischen Namen erzeugen und eintragen */
 if ( !(lntres = addlnm(partab,cl,(crelnm_t *)0,auth,pt)) )
  {
   /* Defaulttabellen gegen Loeschen schuetzen */
   nl = getlnm();
   if ( parent )
    nl->flags &= ~LNM$M_NO_DELETE;
   else
    {
     nl->flags |= LNM$M_NO_DELETE;
     /* Name der Vatertabelle bei Defaulttabellen korrigieren */
     nl->parent = nl->name;
    }
   /* Tabellenname eintragen und weitere Felder initialisieren */
   pt->name = nl->name;
   pt->insysdir = (parent == sysdir);
   pt->parent = partab->name;
   pt->partab = (long)partab;
   /* Tabellen sind als logische Namen immer LNM$M_TERMINAL */
   nl->flags |= LNM$M_TERMINAL;
   /* Ergebnis melden */
   return pt;
  }
 /* Erzeugte Tabelle wieder zerstoeren */
 dellnt(pt);
 return 0;
}

/*
  Alle markierten Kindtabellen in einer Tabelle logischer Namen loeschen.
*/
delcld(tab)
table *tab;
{
 logname *lnt;
 int i;

 /* Markierungsmaske aktualisieren */
 if ( !marked ) return;
 marked = 0;
 /* Tabelle durchgehen und loeschen */
 for ( i = NSLEN(tab) ; i-- ; )
  {
   /* Festellen, ob es sich um eine markierte Tabelle handelt */
   lnt = &LNAME(tab,i);
   if ( lnt->info.table == FALSE ) continue;
   if ( ((table *)lnt->TABLE)->partab != -1 ) continue;
   /* Tabelle loeschen und aus der Liste entfernen */
   dellnm(lnt);
   remlnm(tab,lnt);
  }
}

/*
  Loeschen einer Verwaltungsstruktur logischer Namen. Alle logischen Namen werden wie
  gehabt mit 'dellnm' freigegeben, dann alle allokatierten Strukturen. Da gewisse
  Tabellen in sich selbst definiert sind, wird ein besonderes Sperrbit zur Vermeidung
  von Rekursionen gesetzt.
*/
dellnt(lnt)
table *lnt;
{
 int i;

 /* Rekursionssperre ueberpruefen und setzen */
 if ( lnt->pid == -2 ) return;
 lnt->pid = -2;
 /* Alle logischen Namen freigeben */
 for ( i = NSLEN(lnt) ; i-- ; ) dellnm(&LNAME(lnt,i));
 /* Verwaltungsstruktur der logischen Namen freigeben, falls vorhanden */
 if ( NAMES(lnt) ) free(NAMES(lnt));
 /* Tabelle freigeben */
 free(lnt);
 /* Globale Variablen aktualisieren */
 if ( lnt == sysdir )
  /* LNM$SYSTEM_DIRECTORY ist einmalig */
  sysdir = 0;
 /* Falls schon LNM$PROCESS_DIRECTORY vorhanden sind */
 else if ( piddir )
  {
   /* Tabelle dort suchen */
   for ( i = npid ; i-- ; )
    /* Zeiger vergleichen */
    if ( lnt == piddir[i] )
     {
      /* Liste der LNM$PROCESS_DIRECTORY komprimieren */
      memcpy((char *)(piddir+i),(char *)(piddir+i+1),(npid-i-1)*sizeof(piddir[0]));
      npid--;
      break;
     }
  } 
}

/*
  Markiere in einer Tabelle logischer Namen alle Tabellen logischer Namen die Kind
  einer bestimmten anderen Tabelle sind.
*/
marklnt(tab,partab)
table *tab,*partab;
{
 logname *lnt;
 table *cur;
 int i;

 /* Alle logischen Namen durchgehen */
 for ( i = NSLEN(tab) ; i-- ; )
  {
   /* Logischer Name muss eine Tabelle sein */
   lnt = &LNAME(tab,i);
   if ( lnt->info.table == FALSE ) continue;
   /* Tabelle muss Kind der angegebenen Vatertabelle sein */
   cur = (table *)lnt->TABLE;
   if ( cur->partab != (long)partab ) continue;
   /* Tabelle markieren und deren Kinder feststellen */
   cur->partab = -1;
   marklnt(tab,cur); 
  }
 /* Vorgang wurde abgeschlossen */
 marked = 1;
}

/*
  Der logischen Name 'name' wird mit Hilfe der Tabellen 'tabset' in einen neuen Satz
  von Tabellen in 'atab', 'ntab' und 'tbuf' mit index 'tx' uebersetzt, der dann von
  'trnlnm' oder aehnlichen Routinen zum Suchen logischer Namen verwendet werden kann.
  Die Zugriffsberechtigung des Benutzer wird dabei staendig ueberprueft.
*/
trnlnt(name,tabset,ntabs,lev,tx,wrt)
char *name;
table **tabset;
int ntabs,lev,tx,wrt;
{
 int ix,res,nl;
 logname *lnm;
 table **nt;

 /* Suchen des logischen Namens in den Tabellen */
 if ( !(lnm = trnlnm(name,tabset,ntabs,1,PSL$C_USER)) ) return 0;
 /* Handelt es sich um eine Tabelle, wird diese vermerkt */
 if ( lnm->info.table )
  {
   /* Berechtigung des Benutzers ueberpruefen */
   if ( !verify_access(lnm,wrt ? PROWR : PRORD,&auth_lnm) ) return SS$_NOPRIV;
   /* Platz fuer die Tabelle in der neuen Tabellenmenge reservieren */
   if ( ntab[tx] == atab[tx] )
    {
     /* Feld muss neu angelegt werden */
     if ( !(nt = MALLOC(table *,atab[tx]+10)) ) return SS$_INSFMEM;
     /* Alte Menge kopieren, falls vorhanden */
     if ( atab[tx] )
      {
       /* Tabellenadressen kopieren */
       memcpy((char *)nt,(char *)(tbuf[tx]),atab[tx]*sizeof(tbuf[tx][0]));
       /* Altes Feld freigeben */
       free(tbuf[tx]);
      }
     /* Verwaltungsdaten aktualisieren */
     atab[tx] += 10;
     tbuf[tx] = nt;
    }
   /* Tabelle in den Verwaltungsdaten vermerken */
   tbuf[tx][ntab[tx]++] = (table *)lnm->TABLE;
   return 0;
  }
 /* Logischen Namen weiter untersuchen, falls Tiefe noch nicht erreicht */
 if ( !--lev ) return SS$_NOLOGTAB;
 /* Alle Werte durchgehen */
 for ( ix = 0 ; ix < WLEN(lnm) ; ix++ )
  {
   /* Falls der Name TERMINAL ist, darf nur noch eine weitere Ebene untersucht werden */
   nl = (FLAGS(lnm,ix)&LNM$M_TERMINAL) ? 0 : lev;
   if ( res = trnlnt(NAME(lnm,ix),tabset,ntabs,nl,tx,wrt) )
    /* Fehler nach oben durchgeben */
    return res;
  }
 /* Alles in Ordnung */
 return 0;
}

/******

              VERWALTUNGSSTRUKTUREN LOGISCHER NAMEN

    create_table	erzeuge Verwaltungsstruktur
    find_table		suche Tabelle logischer Namen
    translate_table	ermittele Tabellen logischer Namen
    
    tblsort		sortiere Tabellen logischer Namen
    tblcmp		vergleiche tabellen logischer Namen

******/

/*
  Erzeuge eine Verwaltungsstruktur fuer logische Namen. Eine Definitionsstruktur
  dient als Vorgabe gewisser Felder.
*/
table *create_table(cre)
crelnt_t *cre;
{
 table *nt;
 int dir;

 /* Speicher fuer die Tabelle der logischen Namen reservieren */
 if ( !(nt = MALLOC(table,1)) ) return 0;
 /* Tabelle leer initialisieren */
 nt->pid = cre->pid;
 nt->free = (cre->quota <= 0) ? -1 : cre->quota;
 nt->acmode = cre->acmode;
 nt->nalloc = NSLEN(nt) = 0;
 NAMES(nt) = 0;
 /* Ergebnis melden */
 return nt;
}

/*
  Suchen einer Verwaltungsstruktur logischer Namen ueber den zugeordneten Prozess.
  Die LNM$PROCESS_DIRECTORY sind in 'piddir' abgelegt und wurden mit 'tblsort' in
  der Reihenfolge der Prozesskennungen 'pid' sortiert.
*/
table **find_table(pid,uid)
int pid,uid;
{
 table form,*fptr = &form;
 int tblcmp();

 /* Ohne LNM$PROCESS_DIRECTORY ist keine Suche noetig */
 if ( !piddir ) return 0;
 /* Muster fuer die zu suchende Struktur aufsetzen */
 form.pid = pid;
 form.uid = uid;
 /* Binaere Suchroutine aufrufen */
 return (table **)bsearch(&fptr,piddir,npid,sizeof(piddir[0]),tblcmp);
}

/*
  Nach Vorgabe eines Namens einer Tabelle logischer Namen wird dieser in der aus
  LNM$DIRECTORIES ermittelten Menge von Tabellen ermittelt. Daraus koennen unter
  Umstaenden wieder mehrere Tabellen entstehen. Das Ergebnis wird in den globalen
  Variablen 'atab[1]', 'ntab[1]' und 'tbuf[1]' abgelegt. Der 'wrt' Parameter gibt
  an, ob der Aufrufer Schreibberechtigung benoetigt.
*/
translate_table(name,wrt)
char *name;
int wrt;
{
 table *look[2];
 int res;

 /* Defaultwert fuer die Tabelle einsetzen */
 if ( !name || !*name ) name = "LNM$DCL_LOGICAL";
 /* Globale Variablen initialisieren */
 ntab[0] = ntab[1] = 0;
 /* Im ersten Schritt LNM$DIRECTORIES ueber LNM$PROCESS_ und LNM$SYSTEM_DIRECTORY suchen */
 look[0] = curpid;
 look[1] = sysdir;
 if ( res = trnlnt("LNM$DIRECTORIES",look,2,LNM$C_MAXDEPTH,0,0) ) return res;
 /* Name in Tabellenmenge umsetzen, ohne LNM$DIRECTORIES nach Defaulttabellen des Prozesses */
 if ( ntab[0] )
  res = trnlnt(name,tbuf[0],ntab[0],LNM$C_MAXDEPTH,1,wrt);
 else
  res = trnlnt(name,look,2,LNM$C_MAXDEPTH,1,wrt);
 /* Ergebnis melden */
 return res;
}

/*
  Prozesstabellen LNM$PROCESS_DIRECTORY in 'piddir' mit der Quicksortroutine 'qsort'
  sortieren.
*/
tblsort()
{
 int tblcmp();

 qsort(piddir,npid,sizeof(piddir[0]),tblcmp);
}

/*
  Die Prozesstabellen werden einfach nach ihrer Prozesskennung aufsteigend sortiert.
*/
tblcmp(pt1,pt2)
table **pt1,**pt2;
{
 int del;

 if ( del = ((*pt1)->pid-(*pt2)->pid) ) return del;
 return (*pt1)->uid-(*pt2)->uid;
}

/******

              DEFAULTTABELLEN ERZEUGEN
  
    create_all		Prozessstammbaum durchsuchen und Defaulttabellen erzeugen	
    create_gid		Gruppentabelle LNM$GID_ erzeugen
    create_job		Jobtabelle LNM$JOB_ erzeugen
    create_pid  	Prozesstabellen LNM$PROCESS_DIRECTORY und LNM$PROCESS_TABLE erzeugen

    expand_piddir	Platz fuer LNM$PROCESS_DIRECTORY schaffen

******/

/* 
  Diese Routine geht den Stammbaum eines Prozesses durch und erzeugt alle notwendigen
  Defaulttabellen LNM$JOB_, LNM$PROCESS_DIRECTORY und LNM$PROCESS_TABLE.
*/
create_all(pid,uid)
int uid;
{
 int key = DG_PROCESS_INFO_INITIAL_KEY,done = 0,mpid = (pid < 0) ? -pid : pid;
 struct dg_process_info pi;

 /* Informationen ueber den angegebenen Prozess auslesen */
 if ( dg_process_info(DG_PROCESS_INFO_SELECTOR_PID,mpid,
		      DG_PROCESS_INFO_CMD_NAME_ONLY,&key,&pi,
		      DG_PROCESS_INFO_VERSION_0) != 1 )
  return -1;
 /* Bei dem ersten Prozess der Kette ist 'uid' -1 */
 if ( uid == -1 )
  {
   /* 'uid' uebernehmen */
   uid = pi.user_id;
   /* Muss mit dem 'uid' des Aufrufers uebereinstimmen */
   if ( uid != auth_lnm.uid ) return -2;
  }
 /* Bei den Ahnen des Prozesses muss 'uid' nur noch uebereinstimmen */
 else if ( uid != pi.user_id )
  return -1;
 /* Eine LOGIN-Shell beendet den Stammbaum */
 if ( pi.cmd[0] == '-' )
  /* Zustaendig fuer die Jobtabelle */
  done = 1;
 /* Schutz vor dem ersten UNIX-Prozess, der sein eigener Vater ist */ 
 else if ( pi.parent_process_id == mpid ) 
  done = 1;
 else
  /* Die Suche nach dem Urahn geht weiter */
  switch (create_all(pi.parent_process_id,uid))
   {
    /* Der Vaterprozess gehoert nicht zum Stammbaum, also ist der aktuelle 'pid' der Urahn */
    case -1 : done = 1;
              break;
    /* Abbruch bei schweren Fehlern */
    case -2 : return -2;
   }
 /* Der Urahn erzeugt die Jobtabelle */
 if ( done && !create_job(mpid,uid) ) return -2;
 /* Alle Prozesse erzeugen ihre Defaulttabellen */
 if ( !create_pid(done ? -mpid : pid,uid) ) return -2;
 /* Erfolg durch den gefundenen immer positiven 'uid' melden */
 return uid;
}

/*
  Falls die Gruppentabelle LNM$GID_ooooooooooo noch nicht in LNM$SYSTEM_DIRECTORY
  existiert, wird sie leer erzeugt.
*/
create_gid(gid)
int gid;
{
 struct auth_lnm auth;
 crelnt_t cregidtab;

 /* Erst mal den vollen Namen in einer globalen Variablen ablegen */
 sprintf(gidname,"LNM$GID_%011o",gid);
 /* Name in LNM$SYSTEM_DIRECTORY suchen */
 if ( findlnm(sysdir,gidname,0,PSL$C_KERNEL) ) return 1;
 /* Tabelle neu erzeugen */
 cregidtab.name = gidname;
 cregidtab.pid = getpid();
 cregidtab.flags = LNM$M_NO_ALIAS;
 cregidtab.quota = -1;
 cregidtab.acmode = PSL$C_KERNEL;
 cregidtab.prot = PROSYSTEM(PRORD|PROWR|PRODE)|PROOWNER(PRORD)|PROGROUP(PRORD);
 /* Authorisierungsstruktur aufsetzen */
 auth.uid = 0;
 auth.gid = gid;
 auth.grpnam = auth.sysnam = 1;
 /* Tabelle in LNM$SYSTEM_DIRECTORY erzeugen */
 if ( !addlnt(&cregidtab,&auth,sysdir) ) return 0;
 /* Erfolg melden */
 return 1;
}

/*
  Die Jobtabelle LNM$JOB_XXXXXXXXYYYYYYYY des Urahns erzeugen.
*/
create_job(pid,uid)
int pid,uid;
{
 crelnt_t crejobtab;

 /* Voller Name der Tabelle in einer globalen Variablen ermitteln */
 sprintf(jobname,"LNM$JOB_%08X%08X",pid,uid);
 /* Name in LNM$SYSTEM_DIRECTORY suchen */
 if ( findlnm(sysdir,jobname,0,PSL$C_KERNEL) ) return 1;
 /* Tabelle neu erzeugen */
 crejobtab.name = jobname;
 crejobtab.pid = pid;
 crejobtab.flags = LNM$M_NO_ALIAS;
 crejobtab.quota = 2048;
 crejobtab.acmode = PSL$C_KERNEL;
 crejobtab.prot = PROSYSTEM(PRORD|PROWR|PRODE)|PROOWNER(PRORD|PROWR|PRODE);
 /* Tabelle nach Authorisierungsvorgabe des Aufrufers in LNM$SYSTEM_DIRECTORY erzeugen */
 if ( !addlnt(&crejobtab,&auth_lnm,sysdir) ) return 0;
 /* Tabelle gegen Loeschen durch den Benutzer schuetzen */
 getlnm()->flags |= LNM$M_NO_DELETE;
 /* Erfolg melden */
 return 1;
}

/*
  Falls LNM$PROCESS_DIRECTORY noch nicht definert ist, wird es neu angelegt und in
  'piddir' eingetragen. Sollte ein Vorfahr dieses Prozesses 'pid' existieren, so
  werden alle Informationen, also auch die LNM$PROCESS_TABLE, aus diesem kopiert.
  Darin eingeschlossen sind natuerlich auch alle vom Benutzer nicht mit CONFINE
  angelegte Variablen. Ansonsten erhaelt LNM$PROCESS_DIRECTORY gewisse Defaultwerte,
  genau wie die darin neu erzeugte Tabelle LNM$PROCESS_TABLE. Die Synchronisations
  der verschiedenen Stammbaumebenen uebernimmt die globale Variable 'curpid'.
*/
create_pid(xpid,uid)
int xpid,uid;
{
 table **parr,*oldpid = curpid,*ctab,tmp;
 int pid = (xpid < 0) ? -xpid : xpid;
 crelnt_t crepiddir;
 logname *ptab;
 
 /* Nachsehen, ob LNM$PROCESS_DIRECTORY schon vorhanden ist */
 if ( !(parr = find_table(pid,uid)) )
  {
   /* Nur erzeugen, falls der Benutzer es explizit wuenscht */
   if ( xpid >= 0 ) return 1;
   /* Platz schaffen fuer eine weitere Tabelle */
   if ( !expand_piddir() ) return 0;
   /* Nachsehen, ob der Vater des Prozesses eine Tabelle angelegt hat */
   if ( oldpid )
    /* Zugehoerigen logischen Namen suchen und verifizieren */
    if ( ptab = findlnm(oldpid,PIDDIRNAME,0,PSL$C_KERNEL) )
     {
      /* Leere temporaere Speicherstruktur anlegen */
      tmp.free = tmp.pid = -1;
      tmp.nalloc = NSLEN(&tmp) = 0;
      NAMES(&tmp) = 0;
      /* Tabelle in den temporaeren Bereich kopieren */
      copylnm(&tmp,ptab,0);
      /* Ueberpruefen, ob das funktioniert hat */
      if ( !NSLEN(&tmp) ) return 0;
      /* Adresse der neuen Tabelle auslesen und Prozesskennung aktualisieren */
      curpid = (table *)LNAME(&tmp,0).TABLE;
      curpid->parent = curpid->name;
      curpid->pid = pid;
      /* Unnuetze Informationen freigeben */
      free(NAMES(&tmp));
     }
    else
     /* Logischer Name zur Tabelle muss existieren */
     return 0;
   else
    {
     /* Voellig neue Tabelle in der eigenen Verwaltungsstruktur erzeugen */
     crepiddir.name = PIDDIRNAME;
     crepiddir.pid = pid;
     crepiddir.flags = LNM$M_NO_ALIAS;
     crepiddir.quota = -1;
     crepiddir.acmode = PSL$C_KERNEL;
     crepiddir.prot = -1;
     /* Neu erzeugte Tabelle mit 'curpid' assoziieren */
     if ( !(curpid = addlnt(&crepiddir,&auth_lnm,(table *)0)) ) return 0;
     /* Defaultwerte einsetzen */
     preset_user(curpid);
    }
   /* Tabelle in die globale Liste eintragen und Liste sortieren */
   piddir[npid++] = curpid;
   tblsort();
  }
 else
  /* Bereits vorhandene Tabelle in 'curpid' vermerken */
  curpid = *parr;
 /* Logischen Namen der Gruppentabelle eintragen */
 deflnm(curpid,"LNM$GROUP",LNM$M_TERMINAL,PSL$C_KERNEL,gidname,NOARG);
 /* Suche nach LNM$PROCESS_TABLE in LNM$PROCESS_DIRECTORY */
 if ( ptab = findlnm(curpid,PIDNAME,0,PSL$C_KERNEL) ) return 1;
 /* Definitionsstruktur fuer die neue Tabelle aufsetzen */
 crepiddir.name = PIDNAME;
 crepiddir.pid = pid;
 crepiddir.flags = LNM$M_NO_ALIAS;
 crepiddir.quota = -1;
 crepiddir.acmode = PSL$C_KERNEL;
 crepiddir.prot = -1;
 /* Tabelle in LNM$PROCESS_DIRECTORY erzeugen */
 if ( !(ctab = addlnt(&crepiddir,&auth_lnm,curpid)) ) return 0;
 /* Tabelle mit den Defaultnamen belegen */
 preset_usertable(ctab,pid);
 /* Erfolg melden */
 return 1;
}

/*
  Schaffe Platz in 'piddir', so dass zumindest noch eine LNM$PROCESS_DIRECTORY
  eingetragen werden kann.
*/
expand_piddir()
{
 table **parr;

 /* Wenn noch Platz vorhanden ist, sind wir fertig */
 if ( npid != apid ) return 1;
 /* Vergroessertes Feld reservieren */
 if ( !(parr = MALLOC(table *,apid+10)) ) return 0;
 /* Schon vorhandene Informationen beibehalten */
 if ( apid )
  {
   /* Alte Tabellenadressen kopieren */
   memcpy((char *)parr,(char *)piddir,apid*sizeof(piddir[0]));
   /* Vorheriges Feld freigeben */
   free(piddir);
  }
 /* Globale Variablen aktualisieren */
 apid += 10;
 piddir = parr;
}

/******
              DEFAULTNAMEN ERZEUGEN

    preset_system	Logische Namen in LNM$SYSTEM_DIRECTORY
    preset_user		Logische Namen in LNM$PROCESS_DIRECTORY
    preset_usertable	Logische Namen in LNM$PROCESS_TABLE

******/

/*
  Erzeuge logische Namen in einer frisch erzeugten LNM$PROCESS_DIRECTORY.
*/
preset_user(pdir)
table *pdir;
{
 /* Logische Namen zum Auffinden der Defaulttabellen */
 deflnm(pdir,"LNM$PROCESS",LNM$M_TERMINAL,PSL$C_KERNEL,PIDNAME,NOARG);
 deflnm(pdir,"LNM$JOB",LNM$M_TERMINAL,PSL$C_KERNEL,jobname,NOARG);
}

/*
  Erzeuge logische Namen in einer neuen LNM$PROCESS_TABLE;
*/
preset_usertable(ptab,pid)
table *ptab;
int pid;
{
}

/*
  Erzeuge logische Namen in LNM$SYSTEM_DIRECTORY bzw. LNM$SYSTEM_TABLE. Diese Namen
  stehen allen Benutzern zur Verfuegung. Die Tabellen werden als 'sdir' respektive
  'stab' der Routine uebergeben.
*/
preset_system(sdir)
table *sdir;
{
 /* LNM$SYSTEM_DIRECTORY*/
 deflnm(sdir,"LNM$FILE_DEV",0,PSL$C_EXEC,"LNM$SYSTEM",NOARG);
 deflnm(sdir,"LNM$FILE_DEV",0,PSL$C_SUPER,
	     "LNM$PROCESS","LNM$JOB","LNM$GROUP","LNM$SYSTEM",NOARG);
 deflnm(sdir,"LNM$DCL_LOGICAL",0,PSL$C_SUPER,"LNM$FILE_DEV",NOARG);
 deflnm(sdir,"LNM$DIRECTORIES",LNM$M_NO_ALIAS|LNM$M_TERMINAL,PSL$C_KERNEL,
	     PIDDIRNAME,SYSDIRNAME,NOARG);
 deflnm(sdir,"LNM$SYSTEM",LNM$M_NO_ALIAS|LNM$M_TERMINAL,PSL$C_KERNEL,SYSNAME,NOARG);
}

/******

              HILFSROUTINEN

    strcmpi   Vergleich zweier Strings mit Beruecksichtigung von Gross- und Kleinschreibung

******/

/* 
  Vergleiche zwei Strings, wobei 'a' und 'A' als identisch betrachtet werden.
*/
strcmpi(s1,s2)
unsigned char *s1,*s2;
{
 unsigned char c1,c2;
 int del;

 do
  {
   /* Naechste Zeichen ermitteln */
   c1 = *s1++;
   c2 = *s2++;
   /* Konversion in Grossbuchstaben */
   if ( (c1 >= 'a') && (c1 <= 'z') ) c1 += 'A'-'a';
   if ( (c2 >= 'a') && (c1 <= 'z') ) c2 += 'A'-'a';
   /* Vergleich der Zeichen */
   if ( del = c1-c2 ) return del;
  }
 /* Bis zum Ende des Strings */
 while ( c1 );
 /* Strings sind gleich */
 return 0;
}

