/*
  Dieser Prozess wird vom DCL aus zur Verwaltung der Symbole gestartet. Er nimmt
  ueber eine UNIX named pipe Befehle entgegen, bearbeitet sie, und gibt Antworten.
  Die Symbolverwaltung ist wie auf der VAX organisiert.
*/
#include <errno.h>

#include <sys/un.h>
#include <sys/uio.h>
#include <sys/signal.h>
#include <sys/socket.h>

#include <saphir/vmsdef.h>

#include <saphir/DCL/symbols.h>

/* 
  Verwaltungsstrukturen
*/
typedef struct symbol_t
        {
	 int  flags;
	 int  abbr;
	 char *name;
	 char *value;
	 int  vallen;
        } symbol_t;

typedef struct table_t
        {
	 int      flags;
	 int	  nsym;
	 int	  asym;
	 symbol_t *symbol;
        } table_t;

/*
  Variablen
*/
#define NOSTR			((char *)0)
#define NOINT			((int *)0)

static int ntab = 0,atab = 0,len1 = 0,len2 = 0,symfull,dpid = -1;
static struct sockaddr_un pipesock;
static struct symbol_request req;
static char *buf1 = 0,*buf2 = 0;
static table_t *table = 0;

static int symcmp(),cleanup(),stop_symbol_server();
static symbol_t *findsymbol();
static char *getpara();

extern char *calloc(),*bsearch(),*strchr();
extern int match_wildcard(),strcmp();

#define MALLOC(t,n)		((t *)calloc(n,sizeof(t)))
#define SALLOC(t)		MALLOC(t,1)

#define ok(fd)			{ reply(fd,0,NOSTR,0); return; }
#define error(fd,err)		{ reply(fd,0,NOSTR,-err); return; }

#define SYM$MASK_VERB   	(SYM$FLAG_VLOCAL|SYM$FLAG_VGLOBAL)
#define SYM$MASK_GENERAL	(SYM$FLAG_GLOCAL|SYM$FLAG_GGLOBAL)

#define swrite(fd,item)		(write(fd,&(item),sizeof(item)) == sizeof(item))

/*
  Prozess starten.
*/
start_symbol_server()
{
 static char env[sizeof("DCLSYMBOLS=")+sizeof(pipesock.sun_path)-1];
 struct sockaddr_un rem;
 char *name,*value;
 int nfd,len,pfd;
 
 /* Kanal oeffnen */
 if ( (pfd = socket(AF_UNIX,SOCK_STREAM,0)) == -1 ) return -1;
 /* Temporaere Datei erzeugen und Verbindung aufbauen */
 pipesock.sun_family = AF_UNIX;
 strcpy(pipesock.sun_path,"/tmp/DCLsymXXXXXX");
 if ( !mktemp(pipesock.sun_path) ||
      (bind(pfd,&pipesock,sizeof(pipesock.sun_family)+strlen(pipesock.sun_path)) == -1) )  
  {
   close(pfd);
   return -1;
  }
 /* Kanal aktivieren */
 if ( listen(pfd,3) == -1 )
  {
   cleanup();
   close(pfd);
   return -1;
  }
 /* Prozesskennung merken */
 dpid = getpid();
 /* Prozess erzeugen */
 switch (fork())
  {
   case -1 : cleanup();
     	     close(pfd);
	     return -1;
   case  0 : break;
   default : /* Environment aufbauen */
     	     sprintf(env,"DCLSYMBOLS=%s",pipesock.sun_path);
     	     putenv(env);
     	     /* Exithandler installieren */
     	     atexit(stop_symbol_server);
	     /* Fertig */
	     close(pfd);
     	     return 0;
  }
 /* Exithandler und Exceptionhandler installieren */
 atexit(cleanup);
 signal(SIGINT,cleanup);
 /* Tabelle initialisieren */
 startfile(-1);
 startfile(-1);
 /* Befehle abwarten */
 for ( len = sizeof(rem) ; (nfd = accept(pfd,&rem,&len)) != -1 ; len = sizeof(rem) )
  /* Daten einlesen */
  if ( read(nfd,&req,sizeof(req)) != sizeof(req) )
   close(nfd);
  else
   {
    /* Parameter einlesen */
    if ( (name = getpara(nfd,req.namelen,&buf1,&len1)) == (char *)-1 ) continue;
    if ( (value = getpara(nfd,req.vallen,&buf2,&len2)) == (char *)-1 ) continue;
    /* Auswerten */
    switch (req.op)
     {
      case SYM$OP_SET       : setsymbol(nfd,name,value);
      			      break;
      case SYM$OP_GET       : getsymbol(nfd,name);
      			      break;
      case SYM$OP_DELETE    : deletesymbol(nfd,name);
			      break;
      case SYM$OP_STARTFILE : startfile(nfd);
			      break;
      case SYM$OP_ENDFILE   : endfile(nfd);
	 		      break;
      case SYM$OP_SETOPTION : setoptions(nfd);
      			      break;
      case SYM$OP_TERMINATE : reply(nfd,0,NOSTR,0);
     			      exit(0);
      case SYM$OP_READOUT   : readout(nfd,name,value);
      			      break;
      default	            : reply(nfd,0,NOSTR,-LIB$_FATERRLIB);
     }
   }
 /* Das war wohl nichts */
 if ( errno == EINTR ) exit(0);
 perror("accept");
 exit(1);
}

/*
  Anfrage beantworten und Kanal schliessen.
*/
static reply(fd,flags,data,len)
int fd,flags,len;
char *data;
{
 /* Optionen und Laenge, dann die Daten */
 if ( swrite(fd,flags) && swrite(fd,len) && data && (len > 0) ) write(fd,data,len);
 /* Fertig */
 close(fd); 
}

/*
  Prozess beenden.
*/
static stop_symbol_server()
{
 /* Nur wenn ich es selbst bin */
 if ( getpid() != dpid ) return;
 /* Libraryfunktion aufrufen */
 symbol_operation(SYM$OP_TERMINATE,NOINT,NOSTR,0,NOSTR,0,NOSTR,0,NOINT);
}

/*
  Parameterstring ermitteln.
*/
static char *getpara(fd,len,bufp,lenp)
int fd,len,*lenp;
char **bufp;
{
 /* Kein Parameter */
 if ( len < 0 ) return NOSTR;
 /* Platz reservieren */
 if ( len+1 > *lenp )
  {
   /* Alten Buffer freigeben */
   if ( *lenp ) free(*bufp);
   /* Etwas mehr reservieren */
   *lenp = len+32;
   /* Neuen Buffer reservieren */
   if ( !(*bufp = MALLOC(char,*lenp)) )
    {
     /* Aufraeumen */
     *lenp = 0;
     close(fd);
     /* Fehler melden */
     return (char *)-1;
    }
  }
 /* Daten kopieren */
 if ( len && (read(fd,*bufp,len) != len) )
  {
   close(fd);
   return (char *)-1;
  }
 /* Abschliessen und Ergebnis melden */
 (*bufp)[len] = '\0';
 return *bufp;
}

/*
  Neue Symbolebene eroeffnen.
*/
static startfile(fd)
int fd;
{
 table_t *nt;

 /* Speicher allokatieren */
 if ( ntab == atab )
  {
   /* Neues Feld */
   if ( !(nt = MALLOC(table_t,atab+1)) )
    if ( fd == -1 )
     {
      errno = ENOMEM;
      perror("calloc");
      exit(1);
     }
    else
     error(fd,LIB$_INSCLIMEM);
   /* Altes Feld kopieren */
   if ( atab )
    {
     memmove(nt,table,atab*sizeof(nt[0]));
     free(table);
    }
   /* Werte aktualisieren */
   atab++;
   table = nt;
  }
 /* Struktur initialiseren */
 nt = table+ntab++;
 nt->flags = SYM$MASK_VERB|SYM$MASK_GENERAL;
 nt->nsym = 0;
 /* Fertig */
 if ( fd != -1 ) ok(fd);
}

/*
  Symbolebene schliessen.
*/
static endfile(fd)
int fd;
{
 symbol_t *cur;

 /* Gibt es ueberhaupt eine (lokale) Ebene */
 if ( ntab < 3 ) error(fd,LIB$_UNECLIERR);
 /* Symbole freigeben */
 for ( cur = table[--ntab].symbol ; table[ntab].nsym-- ; cur++ ) freesymbol(cur);
 /* Fertig */
 ok(fd);
}

/*
  Speicher fuer ein Symbol freigeben.
*/
static freesymbol(sym)
symbol_t *sym;
{
 /* Name */
 free(sym->name);
 /* Wert */
 free(sym->value);
}

/*
  Optionen setzen.
*/
static setoptions(fd)
int fd;
{
 table_t *top = table+(ntab-1);

 /* SET SYMBOL/VERB */
 if ( req.flags&SYM$FLAG_VERB ) 
  if ( req.flags&SYM$FLAG_LOCAL ) 
   top->flags = (top->flags&~SYM$FLAG_VLOCAL)|(req.flags&SYM$FLAG_VLOCAL);
  else
   top->flags = (top->flags&~SYM$FLAG_VGLOBAL)|(req.flags&SYM$FLAG_VGLOBAL);
 /* SET SYMBOL/GENERAL */
 if ( req.flags&SYM$FLAG_GENERAL )
  if ( req.flags&SYM$FLAG_LOCAL ) 
   top->flags = (top->flags&~SYM$FLAG_GLOCAL)|(req.flags&SYM$FLAG_GLOCAL);
  else
   top->flags = (top->flags&~SYM$FLAG_GGLOBAL)|(req.flags&SYM$FLAG_GGLOBAL);
 /* Fertig */
 ok(fd);
}

/*
  Symbol erzeugen.
*/
static setsymbol(fd,name,value)
int fd;
char *name,*value;
{
 table_t *use = table+((req.flags&SYM$FLAG_LOCAL) ? (ntab-1) : 0);
 int abbr,slen,clen;
 symbol_t *ns,*cur;

 /* Parameter verifizieren */
 if ( !value ) error(fd,LIB$_UNECLIERR);
 if ( (abbr = verify_name(name)) < 0 ) error(fd,LIB$_INVSYMNAM);
 if ( req.vallen > 1023 ) error(fd,LIB$_INVARG);
 /* Neues Symbol anlegen */
 if ( use->nsym == use->asym )
  {
   /* Speicher erweitern */
   if ( !(ns = MALLOC(symbol_t,use->asym+20)) ) error(fd,LIB$_INSCLIMEM);
   /* Speicher kopieren */
   if ( use->asym )
    {
     memmove(ns,use->symbol,use->asym*sizeof(ns[0]));
     free(use->symbol);
    }
   /* Speicher anlegen */
   use->asym += 20;
   use->symbol = ns;
  }
 /* Neues Symbol initialisieren */
 ns = use->symbol+use->nsym;
 ns->flags = req.flags&(SYM$FLAG_STRING|SYM$FLAG_LOCAL);
 ns->abbr = abbr;
 ns->name = name;
 ns->value = value;
 ns->vallen = (req.flags&SYM$FLAG_STRING) ? req.vallen : sizeof(long);
 /* Sicherstellen, dass noch keine Definition existiert */
 /* - Vergleich mit allen abgekuerzten Symbolen. Man beachte, dass es dabei keine */
 /* - exakt definierte Ordnungsreihenfolge gibt. A*X ist zum Beispiel 'groesser'  */
 /* - als AB*X, passt aber trotzem auf A*A, das seinerseits kleiner als AB*X ist. */
 for ( symfull = 1, cur = ns, slen = strlen(ns->name) ; --cur >= use->symbol ; )
  if ( cur->abbr && !symcmp(ns,cur) )
   if ( abbr && (slen >= (clen = strlen(cur->name))) && !strncmp(ns->name,cur->name,clen) )
    {
     /* Symbol entfernen */
     freesymbol(cur);
     /* Tabelle aufrauemen */
     memmove(cur,cur+1,(ns-cur)*sizeof(cur[0]));
     use->nsym--;
     ns--;
    }
   else
    error(fd,LIB$_AMBSYMDEF);
 /* - Alle nicht abgekuerzten Symbole gleichen Namens entfernen */
 for ( cur = ns ; --cur >= use->symbol ; ) 
  if ( !cur->abbr && !symcmp(cur,ns) )
   {
    /* Symbol entfernen */
    freesymbol(cur);
    /* Tabelle aufrauemen */
    memmove(cur,cur+1,(ns-cur)*sizeof(cur[0]));
    use->nsym--;
    ns--;
   }
 /* Tabelle aktualisieren */
 len1 = len2 = 0;
 use->nsym++;
 sortsymbol(use);
 /* Erfolg melden */
 ok(fd);
}

/*
  Symbol suchen.
*/
static getsymbol(fd,name)
int fd;
char *name;
{
 table_t *use = table+ntab;
 symbol_t *cur = 0;
 int test;

 /* Name verifizieren */
 if ( verify_name(name) ) error(fd,LIB$_INVSYMNAM);
 /* Lokale Tabellen durchsuchen */
 test = (req.flags&SYM$FLAG_VERB) ? SYM$FLAG_VLOCAL : SYM$FLAG_GLOCAL;
 while ( (--use > table) && !(cur = findsymbol(name,use)) && (use->flags&test) );
 if ( !cur )
  {
   /* Globale Tabelle durchsuchen */
   test = (req.flags&SYM$FLAG_VERB) ? SYM$FLAG_VGLOBAL : SYM$FLAG_GGLOBAL;
   for ( use = table+ntab ; --use > table ; )
    if ( !(use->flags&test) )
     error(fd,LIB$_NOSUCHSYM);
   if ( !(cur = findsymbol(name,table)) ) error(fd,LIB$_NOSUCHSYM);
  }
 /* Ergebnis weitergeben */
 reply(fd,cur->flags,cur->value,cur->vallen);
}

/*
  Symbol entfernen.
*/
static deletesymbol(fd,name)
int fd;
char *name;
{
 table_t *use = table+((req.flags&SYM$FLAG_LOCAL) ? (ntab-1) : 0);
 symbol_t *cur;

 /* /ALL */
 if ( req.flags&SYM$FLAG_GENERAL )
  {
   /* Tabelle aufraeumen */
   while ( use->nsym-- ) freesymbol(use->symbol+use->nsym);
   use->nsym = 0;
  } 
 else
  {
   /* Name verifizieren */
   if ( verify_name(name) ) error(fd,LIB$_INVSYMNAM);
   /* Symbol in der angegebenen Tabelle suchen */
   if ( !(cur = findsymbol(name,use)) ) error(fd,LIB$_NOSUCHSYM);
   /* Symbol entfernen */
   freesymbol(cur);
   /* Tabelle aufraeumen */
   memmove(cur,cur+1,(use->nsym-(cur-use->symbol)-1)*sizeof(cur[0]));
   use->nsym--;
  }
 /* Erfolg melden */
 ok(fd);
}

/*
  Symbole auslesen.
*/
static readout(fd,name,value)
int fd;
char *name,*value;
{
 table_t *loc = table+(ntab-1),*glob = table,*tab;
 int lix,gix,lenn,rlen,mflags;
 symbol_t sym,*res = 0;
 
 /* Parameter ueberpruefen */
 if ( !name || !value || (req.vallen != (sizeof(lix)+sizeof(gix))) ) error(fd,LIB$_UNECLIERR);
 /* Indizes auslesen */
 memmove(&lix,value+0,sizeof(lix));
 memmove(&gix,value+sizeof(lix),sizeof(gix));
 if ( (lix < -1) || (gix < -1) ) error(fd,LIB$_UNECLIERR);
 /* Eventuell globale Symbole abschalten */
 if ( (gix < glob->nsym) && (req.flags&SYM$FLAG_LOCAL) )
  for ( tab = loc ; tab > table ; )
   if ( !((tab--)->flags&SYM$FLAG_GGLOBAL) )
    {
     /* Nicht verwenden */
     gix = 0x7ffffffe;
     break;
    }
 /* Naechsten Eintrag suchen */
 if ( strchr(name,'*') || strchr(name,'%') )
  {
   /* Patternsuche */
   while ( (++lix < loc->nsym) && !match_wildcard(loc->symbol[lix].name,name) );
   while ( (++gix < glob->nsym) && !match_wildcard(glob->symbol[gix].name,name) );
  }
 else
  {
   /* Erst mal richtig machen */
   if ( verify_name(name) ) error(fd,LIB$_INVSYMNAM);
   /* Namenssuche */
   sym.abbr = 0;
   sym.name = name;
   symfull = 1;
   /* Naechstes passendes lokales Symbol */
   while ( (++lix < loc->nsym) && symcmp(loc->symbol+lix,&sym) );
   /* Naechstes passendes globales Symbol */
   while ( (++gix < glob->nsym) && symcmp(glob->symbol+gix,&sym) );   
  }
 /* Ergebnis auswerten */
 switch ((lix >= loc->nsym)+2*(gix >= glob->nsym))
  {
   case 0 : /* beide gefunden */
     	    if ( strcmp(loc->symbol[lix].name,glob->symbol[gix].name) <= 0 )
	     {
	      /* Erst die lokalen Symbole */
	      gix--;
	      res = loc->symbol+lix;
	     }
	    else
	     {
	      /* Dann die globalen Symbole */
	      lix--;
	      res = glob->symbol+gix;
	     }
	    break;
   case 1 : /* lix ausserhalb der Liste */
            lix = 0x7ffffffe;
	    res = glob->symbol+gix;
	    break;
   case 2 : /* gix ausserhalb der Liste */
            gix = 0x7ffffffe;
	    res = loc->symbol+lix;
	    break;
   case 3 : /* gix und lix ausserhalb der Liste */
     	    lix = gix = -1;
	    break;
  }
 /* Informationen sammeln */
 rlen = sizeof(lix)+sizeof(gix);
 if ( res )
  {
   lenn = strlen(res->name);
   rlen += sizeof(res->abbr)+sizeof(lenn)+lenn+sizeof(res->vallen)+res->vallen;
   mflags = res->flags;
  }
 else 
  lenn = mflags = 0;
 /* Informationen uebertragen */
 if ( swrite(fd,mflags) && swrite(fd,rlen) && swrite(fd,lix) && swrite(fd,gix) && res )
  if ( swrite(fd,res->abbr) && 
       swrite(fd,lenn) && ((lenn <= 0) || (write(fd,res->name,lenn) == lenn)) && 
       swrite(fd,res->vallen) && (res->vallen > 0) )
   write(fd,res->value,res->vallen);
 /* Kanal schliessen */
 close(fd);
}

/*
  Symbolname verifizieren.
*/
static verify_name(name)
char *name;
{
 char *name0 = name+1,*tail,*abbr = 0,ch;

 /* Leerzeichen am Ende */
 if ( !(tail = name) ) return -1;
 for ( tail += strlen(name) ; (--tail >= name) && ((*tail == ' ') || (*tail == '\t')) ; );
 tail[1] = '\0';
 /* Laenge */
 if ( !*name || (strlen(name) > 255) ) return -1;
 /* [A-Za-z$_][A-Za-z$_0-9]* mit einem optionalen * */
 while ( ch = *name++ )
  if ( (ch >= 'a') && (ch <= 'z' ) )
   name[-1] += 'A'-'a';
  else if ( ((ch < 'A') || (ch > 'Z')) && (ch != '$') && (ch != '_') )
   if ( name == name0 )
    return -1;
   else if ( (ch < '0') || (ch > '9') )
    if ( (ch != '*') || abbr )
     return -1;
    else
     abbr = name;
 /* Abkuerzung ermitteln */
 if ( !abbr ) return 0;
 memmove(abbr-1,abbr,strlen(abbr)+1);
 return (abbr[-1] ? (abbr-name0) : 0);
}

/*
  Symbol in einer Tabelle suchen.
*/
static symbol_t *findsymbol(name,tab)
char *name;
table_t *tab;
{
 symbol_t sym;

 /* Leere Tabelle */
 if ( !tab->nsym ) return 0;
 /* Aufsetzen */
 sym.abbr = 0;
 sym.name = name;
 /* Suchen */
 symfull = 1;
 return (symbol_t *)bsearch(&sym,tab->symbol,tab->nsym,sizeof(sym),symcmp);
}

/*
  Vergleich zweier Symbole.
*/
static symcmp(s1,s2)
symbol_t *s1,*s2;
{
 /* Keine Beruecksichtigung von Abkuerzungen beim Sortieren */
 if ( !symfull ) return strcmp(s1->name,s2->name);
 /* Keine Abkuerzungen verwendet */
 if ( !s1->abbr )
  if ( !s2->abbr ) 
   return strcmp(s1->name,s2->name);
  else
 /* Abkuerzung bearbeiten */
   return -abbr1cmp(s2,s1->name);
 else if ( !s2->abbr ) 
  return abbr1cmp(s1,s2->name);
 /* Zwei abgekuerzte Symbole */
 else if ( s1->abbr >= s2->abbr ) 
  return abbr2cmp(s1,s2->name);
 else
  return -abbr2cmp(s2,s1->name);
}

/*
  Vergleich eines Namens mit einem abgekuerzten Symbol.
*/
static abbr1cmp(sym,name)
symbol_t *sym;
char *name;
{
 int len = strlen(name);
 
 /* Laenge */
 if ( (len < sym->abbr) || (len > strlen(sym->name)) ) return strcmp(sym->name,name);
 /* Text */
 return strncmp(sym->name,name,len);
}

/*
  Vergleich zweier abgekuerzter Symbole.
*/
static abbr2cmp(sym,name)
symbol_t *sym;
char *name;
{
 /* Laenge */
 if ( sym->abbr > strlen(name) ) return strcmp(sym->name,name);
 /* Echter Vergleich der Pattern */
 return strncmp(sym->name,name,sym->abbr);
}

/*
  Sortieren einer Symbolliste.
*/
static sortsymbol(tab)
table_t *tab;
{
 /* Leere Tabelle */
 if ( !tab->nsym ) return;
 /* Sortieren */
 symfull = 0;
 qsort(tab->symbol,tab->nsym,sizeof(tab->symbol[0]),symcmp);
}

/*
  Aufraeumen ist einfach.
*/
static cleanup()
{
 /* Kanal entfernen */
 unlink(pipesock.sun_path);
}

