#include "dcl.h"
#include "command.h"

#include <netinet/in.h>

#include <saphir/CLD.h>

#include <saphir/DCL/symbols.h>

#include <saphir/modules/NUMBER.h>
#include <saphir/modules/EXPRESSION.h>

/*
  Verwaltung eines Wertes auf dem Stack.
*/
typedef union t_value
	{
	 long	number;
	 char	*string;
	 int	(*routine)();
	} t_value;

typedef struct t_stack
	{
	 int		kind;
	 int		vallen;
	 t_value	value;
	} t_stack;

typedef struct t_map
	{
	 int		lexID;
	 int		(*routine)();
	} t_map;

#define v_number	value.number
#define v_string	value.string
#define v_routine	value.routine

/*
  DCL Funktionen.
*/
static int Ftype(),Finteger(),Fstring(),Flength(),Flocate(),Fextract();

static t_map funcs[] =
       {
	{ LEX_TYPE   , Ftype },
	{ LEX_INTEGER, Finteger },
	{ LEX_STRING , Fstring },
	{ LEX_LENGTH , Flength },
	{ LEX_LOCATE , Flocate },
	{ LEX_EXTRACT, Fextract }
       };

#define NFUNCS		(sizeof(funcs)/sizeof(funcs[0]))

/*
  Der Stack.
*/
static t_stack *stack = 0;
static int nstack,astack;

/*
  Hilfsprogramme.
*/
extern int (*EXPRESSIONuser)();
extern char *createValue();
static int doExpression();

char *dcl$replace_ampersand();

#define	ERROR(err,num)	return cleanError(pcb,num,err)
#define ERROR0(err)	ERROR(err,0)
#define ERROR1(err)	ERROR(err,1)

/*
  ['&']SYMBOL ['[' offset ',' size ']'] [':']'='['='] VALUE
*/
dcl$assign_symbol(pcb)
struct epcbCLD *pcb;
{
 static struct epcbEXPRESSION expcb;
 int flags,len,res,max,fact,blen,cnt,bit,carry,i;
 long offset,size,tmp[2],mask[2];
 char *data,*buf;
 t_stack *sym;

 /* Symbol komplettieren */
 if ( res = doAmpersand(&pcb->pcb_symbol,pcb->pcb_flags) ) ERROR1(res);
 /* Ersten Ausdruck auswerten */
 if ( pcb->pcb_flags&OFFSET )
  {
   /* Keine Symbole mit * erlaubt */
   if ( strchr(pcb->pcb_symbol.text,'*') ) ERROR1(CLI$_EXPSYN);
   /* Offset */
   if ( theCalculator(pcb->pcb_offset.text) || !doInteger(stack,0L) ) ERROR1(CLI$_EXPSYN);
   offset = stack->v_number;
   /* Size */
   if ( theCalculator(pcb->pcb_size.text) || !doInteger(stack,0L) ) ERROR1(CLI$_EXPSYN);
   size = stack->v_number;
  }
 /* Alles */
 if ( pcb->pcb_flags&STRING )
  {
   /* Stack aufraeumen */
   freepara(stack,nstack);
   nstack = 0;
   /* String uebernehmen */
   if ( doExpression(&expcb,PUSH_NUMBER) ) return CDU_NOMEM;
   stack->kind = PUSH_QUOTED;
   stack->v_string = pcb->pcb_rightside.text;
   stack->vallen = pcb->pcb_rightside.len;
   /* Aufraeumen */
   pcb->pcb_rightside.text = 0;
  }
 /* Ausdruck auswerten */
 else if ( theCalculator(pcb->pcb_rightside.text) || !doSymbol(stack) ) 
  ERROR1(CLI$_EXPSYN);
 /* Alten Wert ermitteln und einsetzen */
 if ( pcb->pcb_flags&OFFSET )
  {
   /* Stack erweitern */
   if ( doExpression(&expcb,PUSH_NULL) ) return CDU_NOMEM;
   sym = stack+1;
   if ( !doString(sym,pcb->pcb_symbol.text,0) ) ERROR1(CLI$_EXPSYN);
   /* Symbol umsetzen oder Defaultwert ermitteln */
   if ( !findSymbol(sym,!(pcb->pcb_flags&GLOBAL)) )
    {
     sym->kind = PUSH_QUOTED;
     *sym->v_string = '\0';
     sym->vallen = 0;
    }
   /* Zahl konvertieren */
   if ( (sym->kind == PUSH_NUMBER) && (pcb->pcb_flags&STRING) && !doString(sym,"",0) ) 
    ERROR1(CLI$_EXPSYN);
   /* Ganzzahlueberschreibung */
   if ( sym->kind == PUSH_NUMBER )
    if ( (offset < 0) || (offset > 32) || (size < 0) || (size > 32) || (offset+size > 32) )
     ERROR1(CLI$_EXPSYN);
    else
     data = (char *)&sym->v_number;
   else
    /* Textueberschreibung */
    data = sym->v_string;
   len = sym->vallen;
   /* Groesse des Ergebnisses ermitteln */
   fact = (pcb->pcb_flags&STRING) ? 1 : 8;
   if ( (offset < 0) || (offset > fact*768) || (size < 0) || (size > fact*768) ||
	(offset+size > fact*768) )
    ERROR1(CLI$_EXPSYN);
   /* Feld erweitern */
   if ( offset+size > fact*len )
    {
     /* Neue Groesse */
     sym->vallen = (offset+size+fact-1)/fact;
     /* Neuen Datenbereich */
     if ( !(buf = malloc(sym->vallen+1)) ) return CDU_NOMEM;
     memmove(buf,data,len);
     memset(buf+len,(pcb->pcb_flags&STRING) ? ' ' : 0,sym->vallen-len);
     buf[sym->vallen] = '\0';
     /* Uebernehmen */
     free(sym->v_string);
     data = sym->v_string = buf;
    }
   /* Laenge des Ergebniswertes */
   if ( !(pcb->pcb_flags&STRING) && (stack->kind != PUSH_NUMBER) ) ERROR1(CLI$_EXPSYN);
   /* Feld ueberschreiben */
   if ( size )
    if ( pcb->pcb_flags&STRING )
     {
      if ( (len = stack->vallen) > size ) len = size;
      memmove(data+offset,stack->v_string,len);
      memset(data+offset+len,' ',size-len);
     }
    else if ( sym->kind == PUSH_NUMBER )
     {
      mask[1] = ((1<<size)-1)<<offset;
      sym->v_number = (sym->v_number&~mask[1])|((stack->v_number<<offset)&mask[1]);
     }
    else
     {
      /* Zahlenwert richtig aufsetzen */
      if ( size < 32 )
       {
	mask[0] = 0;
	mask[1] = htonl(1<<size)-1;
       }
      else if ( size < 64 )
       {
	mask[0] = htonl(1<<(size%32))-1;
	mask[1] = -1;
       }
      else
       mask[0] = mask[1] = -1;
      tmp[0] = 0;
      tmp[1] = htonl(stack->v_number)&mask[1];
      /* Schieben */
      for ( cnt = offset%8 ; cnt-- ; )
       for ( i = 2 ; i-- ; )
	for ( buf = ((char *)(i ? tmp : mask))+(bit = sizeof(tmp)), carry = 0 ; bit-- ; )
	 {
	  carry += (unsigned char)*--buf;
	  carry += (unsigned char)*buf;
	  *buf = carry;
	  carry >>= 8;
	 }
      /* Ueberlagern */
      buf = data+(offset+size-1)/8;
      data += offset/8;
      for ( i = sizeof(tmp) ; data <= buf ; data++ )
       if ( i-- > 0 )
	*data = (*data&~((char *)mask)[i])|((char *)tmp)[i];
       else if ( data == buf )
	*data &= ~((1<<(((offset+size-1)%8)+1))-1);
       else
	*data = 0;
     }
   /* Wert uebernehmen */
   freestack(stack);
   *stack = *sym;
   sym->kind = PUSH_NUMBER;
   /* Stack aufraeumen */
   freestack(sym);
   nstack--;
  }
 /* Parameter ermitteln */
 flags = 0;
 if ( !(pcb->pcb_flags&GLOBAL) ) flags |= SYM$FLAG_LOCAL;
 if ( stack->kind == PUSH_QUOTED )
  {
   flags |= SYM$FLAG_STRING;
   data = stack->v_string;
  }
 else
  data = (char *)&stack->v_number;
 len = stack->vallen;
 /* Wert definieren */
 res = symbol_operation(SYM$OP_SET,&flags,
			pcb->pcb_symbol.text,(int)pcb->pcb_symbol.len,data,len,
			NOSTR,0,NOINT);
 /* Aufraeumen fuer cli$dcl_parse_ */
 cleanCommand(pcb,1);
 pcb->pcb_symbol.text = 0;
 /* Speicher freigeben */
 return (res&1) ? parse_exit : res;
}

/*
  Speicher aufraeumen.
*/
cleanCommand(pcb,sym)
struct epcbCLD *pcb;
int sym;
{
 if ( sym ) cleanItem(&pcb->pcb_symbol);
 cleanItem(&pcb->pcb_offset);
 cleanItem(&pcb->pcb_size);
 cleanItem(&pcb->pcb_rightside);
 cleanItem(&pcb->pcb_filename);
 cleanItem(&pcb->pcb_p1);
 cleanItem(&pcb->pcb_p2);
 cleanItem(&pcb->pcb_p3);
 cleanItem(&pcb->pcb_p4);
 cleanItem(&pcb->pcb_p5);
 cleanItem(&pcb->pcb_p6);
 cleanItem(&pcb->pcb_p7);
 cleanItem(&pcb->pcb_p8);
}

/*
  Speicher aufraeumen und Fehler melden.
*/
static cleanError(pcb,sym,err)
struct epcbCLD *pcb;
int sym,err;
{
 cleanCommand(pcb,sym);
 return err;
}

/*
  Speicher fuer ein Textelement freigeben.
*/
static cleanItem(des)
struct descriptor *des;
{
 if ( !des->text ) return;
 free(des->text);
 des->text = 0;
 des->len = 0;
}

/*
  Symbol ersetzen, falls moeglich.
*/
dcl$replace_symbol(pcb)
struct epcbCLD *pcb;
{
 static char value[1024];
 char *pos = pcb->pcb_symbol.text,*sym;
 int res,len = pcb->pcb_symbol.len,rest = strlen(pos+len),vlen,clen,mlen;
 int done = pos-(char *)pcb->pcb_start;

 /* Waren wir schon einmal hier */
 if ( !(pcb->pcb_flags&RESETDONE) )
  {
   /* Symbol ermitteln */
   if ( res = doAmpersand(&pcb->pcb_symbol,pcb->pcb_flags) ) ERROR1(res);
   /* Zaehler korrigieren */
   if ( pcb->pcb_flags&AMPER )
    {
     pos--;
     len++;
     done--;
    }
   /* Vorbereiten fuer cli$dcl_parse */
   sym = pcb->pcb_symbol.text;
   pcb->pcb_symbol.text = pos;
   pcb->sys_pcb.pcb_line = pos+pcb->pcb_symbol.len;
   /* Laenge ueberpruefen */
   if ( done+pcb->pcb_symbol.len+rest >= COMMAND_MAXLENGTH ) 
    {
     /* Aufrauemen */
     free(sym);
     ERROR0(CLI$_EXPSYN);
    }
   /* Symbol uebernehmen */
   overwrite(pos,len,rest+1,sym,(int)pcb->pcb_symbol.len);
   free(sym);
   len = pcb->pcb_symbol.len;
   /* Symbol umsetzen */
   if ( amprepl(pos,len,value,sizeof(value),&vlen)&1 )
    {
     /* Einfuegen */
     if ( done+vlen+rest >= COMMAND_MAXLENGTH ) ERROR0(CLI$_EXPSYN);
     overwrite(pos,len,rest+1,value,vlen);
     /* & auch im Symbol ersetzen */
     rest += vlen;
     len = 0;
     /* $ Befehl */
     for ( sym = pos ; (vlen >= 0) && ((*sym == ' ') || (*sym == '\t')) ; sym++, vlen-- );
     if ( (vlen >= 0) && (*sym == '$') )
      {
       pcb->pcb_symbol.text = 0;
       pcb->sys_pcb.pcb_line = sym;
       pcb->pcb_flags |= FOREIGN;
      }
     else
      /* Nochmal probieren */
      pcb->pcb_flags |= RESET;
    }
   else if ( pcb->pcb_flags&AMPER )
    pcb->pcb_flags |= RESET;
   /* Restliche Ampersands umsetzen */
   if ( !(pcb->pcb_flags&FOREIGN) )
    {
     mlen = COMMAND_MAXLENGTH-(done+len)-1;
     if ( !dcl$replace_ampersand(pos+len,&rest,mlen) ) ERROR0(CLI$_EXPSYN);
     pos[len+rest] = '\0';
     pcb->pcb_flags |= AMPERDONE;
    }
   /* Aufraeumen */
   pcb->pcb_flags &= ~AMPER;
  }
 /* Aufraeumen fuer cli$dcl_parse_ */
 return (pcb->pcb_flags&(RESET|AMPER)) ? parse_fail : parse_exit;
}

/*
  Textbereich ueberschreiben.
      OOOOOORRRRRR -> NNNNNNRRRRRR
      <olen><rest>    <nlen><rest>
      ostr	      nstr
*/
static overwrite(ostr,olen,rest,nstr,nlen)
char *ostr,*nstr;
int olen,rest,nlen;
{
 /* Platz schaffen */
 if ( olen != nlen ) memmove(ostr+nlen,ostr+olen,rest);
 /* Neuen Wert uebernehmen */
 memmove(ostr,nstr,nlen);
}

/*
  Symbole mit & ersetzen.
*/
char *dcl$replace_ampersand(str,curlen,maxlen)
char *str;
int *curlen,maxlen;
{
 static char value[1024];
 int wlen,ix,mode,pos,vlen,len;
 char *nstr,ch,lst;

 /* Neuen String anlegen */
 if ( curlen )
  {
   nstr = str;
   wlen = *curlen;
  }
 else if ( nstr = malloc(maxlen) ) 
  wlen = strlen(str);
 else
  return 0;
 /* Bearbeiten */
 memmove(nstr,str,wlen);
 for ( mode = ix = 0, ch = lst ; ix < wlen ; lst = ch )
  if ( (ch = nstr[ix++]) == '"' )
   mode = !mode;
  else if ( !mode )
   if ( ch == '!' )
    break;
   else if ( (ch == '&') && !issymchar(lst) )
    {
     /* Symbol ermitteln */
     for ( pos = ix ; (ix < wlen) && issymchar(nstr[ix]) ; ix++ );
     if ( pos == ix ) return 0;
     /* Symbol umsetzen */
     len = ix-(pos-1);
     if ( !(amprepl(nstr+pos,len-1,value,sizeof(value),&vlen)&1) ) return 0;
     /* Ersetzen */
     if ( wlen+(vlen-len) >= maxlen ) return 0;
     overwrite(nstr+pos-1,len,wlen-ix,value,vlen);
     /* Zaehler aktualisieren */
     ix += vlen-len;
     wlen += vlen-len;
     if ( curlen ) *curlen = wlen;
     /* Sperren */
     ch = 'a';
    }
 /* Loeschen */
 if ( curlen ) 
  memset(nstr+wlen,' ',maxlen-wlen);
 else
  nstr[wlen] = '\0';
 /* Fertig */
 return nstr;
}

/*
  Symbol ersetzen und Verb beachten.
*/
static amprepl(sym,slen,val,vlen,len)
char *sym,*val;
int slen,vlen,*len;
{
 extern int isFIRSTsymbol;
 int flags,res;
 long lval;

 /* Das erste Symbol einer Zeile ist immer das VERB */
 flags = isFIRSTsymbol ? SYM$FLAG_VERB : SYM$FLAG_GENERAL;
 isFIRSTsymbol = 0;
 /* Ausfuehren */
 res = symbol_operation(SYM$OP_GET,&flags,sym,slen,NOSTR,0,val,vlen,len);
 if ( !(res&1) ) return res;
 /* Zahl */
 if ( !(flags&SYM$FLAG_STRING) )
  {
   memmove(&lval,val,sizeof(lval));
   sprintf(val,"%ld",lval);
   *len = strlen(val);
   memset(val+(*len),' ',vlen-(*len));
  }
 /* In Ordung */
 return res;
}

/*
  Ausrechnen eines Strings als Expression.
*/
static theCalculator(str)
char *str;
{
 static struct epcbEXPRESSION *expcb = 0;
 char *nstr;
 int result;

 /* Tabelle laden */
 if ( !expcb && !(expcb = (struct epcbEXPRESSION *)getpcb(pcbEXPRESSION_module)) )
  {
   fprintf(stderr,"out of memory\n");
   exit(2);
  }
 /* Stack aufraeumen */
 freepara(stack,nstack);
 nstack = 0;
 /* Ampersands ersetzen */
 if ( !(nstr = dcl$replace_ampersand(str,NOINT,1024)) ) return CDU_NOMEM;
 /* Ausdruck berechnen und aufraeumen */
 EXPRESSIONuser = doExpression;
 result = parse(expcb,str);
 free(expcb->pcb_tree);
 EXPRESSIONuser = 0;
 free(nstr);
 if ( result ) return result;
 /* Muss sauber sein */
 for ( str = expcb->sys_pcb.pcb_line ; *str ; str++ )
  if ( (*str != ' ') && (*str != '\t') )
   return parse_fail;
 /* Alles in Ordnung */
 return (nstack == 1) ? 0 : -1;
}

/*
  Ausdruck ausrechnen.
*/
dcl$solve_expression(expr,nump,charp)
char *expr,**charp;
int *nump;
{ 
 int res;

 /* Ausrechnen */
 if ( res = theCalculator(expr) ) return res;
 /* Umwandeln */
 if ( !doSymbol(stack) ) return CLI$_EXPSYN;
 if ( nump )
  {
   /* Zahl */
   if ( !doInteger(stack,0) ) return CLI$_EXPSYN;
   *nump = stack->v_number;
  }
 else
  {
   /* Zeichenkette */
   if ( !doString(stack,"",1) ) return CLI$_EXPSYN;
   *charp = stack->v_string;
   stack->kind = PUSH_NUMBER;
  }
 /* Alles in Ordung */
 return res;
}

/*
  Wird vom Modul EXPRESSION waerend der Auswertung aufgerufen.
*/
static doExpression(pcb,op)
struct epcbEXPRESSION *pcb;
long op;
{
 int len1,len2,cmp = 0,cres,n,result;
 t_stack *op1,*op2,*res;
 char *tmp;

 /* Stack erweitern */
 if ( op >= PUSH_NULL )
  {
   /* Platz schaffen */
   if ( nstack == astack )
    {
     /* Speicher reservieren */
     if ( !(res = (t_stack *)malloc((astack+10)*sizeof(res[0]))) ) return CDU_NOMEM;
     /* Alten Bereich kopieren */
     if ( astack )
      {
       memmove(res,stack,astack*sizeof(res[0]));
       free(stack);
      }
     /* Zeiger aufsetzen */
     astack += 10;
     stack = res;
    }
   /* Eintrag reservieren */
   res = stack+nstack++;
   /* Eintrag aufsetzen */
   switch (res->kind = op)
    {
     case PUSH_NUMBER   : res->v_number = pcb->pcb_value.len;
     case PUSH_NULL     : res->vallen = sizeof(res->v_number);
       			  break;
     case PUSH_QUOTED   : 
     case PUSH_STRING   : if ( !(res->v_string = createValue(pcb->pcb_value.text,
							     (int)pcb->pcb_value.len,3,0)) )
       			   return CDU_NOMEM;
       			  res->vallen = strlen(res->v_string);
       			  break;
     case PUSH_FUNCTION : for ( n = NFUNCS ; n-- && (pcb->pcb_value.len != funcs[n].lexID) ; );
   			  if ( n < 0 ) return parse_fail;
   			  res->v_routine = funcs[n].routine;
       			  res->vallen = 0;
       			  break;
    }
   /* Fertig */
   return parse_exit;
  }
 /* Aufruf einer Funktion */
 if ( op == EXPR_CALL ) 
  {
   /* Kopf suchen */
   for ( res = stack+nstack, n = 0 ; (--res)->kind != PUSH_FUNCTION ; n++ );
   /* Aufraeumen */
   res->kind = PUSH_NUMBER;
   res->vallen = sizeof(res->v_number);
   /* Aufrufen */
   result = (*res->v_routine)(res,res+1,n);
   /* Aufraeumen */
   freepara(res+1,n);
   nstack -= n;
   /* Ergebnis melden */
   return result;
  }
 /* Operatoren */
 op1 = stack+(nstack-2);
 op2 = stack+(nstack-1);
 res = op1;
 /* Konversionen */
 switch (op)
  {
   case EXPR_OR	   :
   case EXPR_AND   :
   case EXPR_NE	   :
   case EXPR_LT	   :
   case EXPR_LE	   :
   case EXPR_GT	   :
   case EXPR_GE	   :
   case EXPR_EQ	   :
   case EXPR_DIV   :
   case EXPR_MUL   : if ( !doInteger(op1,0L) || !doInteger(op2,0L) ) return parse_fail;
     		     break;
   case EXPR_NOT   :
   case EXPR_NEG   : if ( !doInteger(res = op2,0L) ) return parse_fail;
     		     break;
   case EXPR_ADD   : 
   case EXPR_SUB   : if ( (op1->kind == PUSH_NUMBER) || (op2->kind == PUSH_NUMBER) )
     		      {
		       if ( !doInteger(op1,0L) || !doInteger(op2,0L) ) return parse_fail;
     		       break;
		      }
     		     cmp--;
   case EXPR_NES   :
   case EXPR_LTS   :
   case EXPR_LES   :
   case EXPR_GTS   :
   case EXPR_GES   :
   case EXPR_EQS   : cmp++;
   case EXPR_MERGE : if ( !doString(op1,"",op != EXPR_MERGE) || 
			  !doString(op2,"",op != EXPR_MERGE) )
     		      return parse_fail;
     		     break;
  }
 /* Auswertung */
 switch (op)
  {
   case EXPR_OR	   : res->v_number |= op2->v_number;
		     nstack--;
     		     return parse_exit;
   case EXPR_AND   : res->v_number &= op2->v_number;
		     nstack--;
     		     return parse_exit;
   case EXPR_NE	   : res->v_number = (op1->v_number != op2->v_number);
		     nstack--;
     		     return parse_exit;
   case EXPR_LT	   : res->v_number = (op1->v_number < op2->v_number);
		     nstack--;
     		     return parse_exit;
   case EXPR_LE	   : res->v_number = (op1->v_number <= op2->v_number);
		     nstack--;
     		     return parse_exit;
   case EXPR_GT	   : res->v_number = (op1->v_number > op2->v_number);
		     nstack--;
     		     return parse_exit;
   case EXPR_GE	   : res->v_number = (op1->v_number >= op2->v_number);
		     nstack--;
     		     return parse_exit;
   case EXPR_EQ	   : res->v_number = (op1->v_number == op2->v_number);
		     nstack--;
     		     return parse_exit;
   case EXPR_DIV   : if ( op2->v_number )
     		      res->v_number /= op2->v_number;
     		     else if ( op1->v_number < 0 )
		      res->v_number = -0x7fffffff;
     		     else
		      res->v_number = 0x7fffffff;
		     nstack--;
     		     return parse_exit;
   case EXPR_MUL   : res->v_number *= op2->v_number;
		     nstack--;
     		     return parse_exit;
   case EXPR_NOT   : res->v_number = ~res->v_number;
     		     return parse_exit;
   case EXPR_NEG   : res->v_number = -res->v_number;
   default         : return parse_exit;
   case EXPR_ADD   : if ( res->kind == PUSH_NUMBER )
     		      {
		       res->v_number += op2->v_number;
		       nstack--;
		       return parse_exit;
		      }
   case EXPR_MERGE : len1 = op1->vallen;
     		     len2 = op2->vallen;
     		     res->vallen = len1+len2;
		     if ( !(tmp = malloc(res->vallen+1)) ) return parse_fail;
     		     memmove(tmp,op1->v_string,len1);
     		     memmove(tmp+len1,op2->v_string,len2+1);
     		     free(res->v_string);
		     res->v_string = tmp;
     		     if ( op == EXPR_MERGE ) res->kind = PUSH_STRING;
		     break;
   case EXPR_SUB   : if ( res->kind == PUSH_NUMBER )
     		      {
		       res->v_number -= op2->v_number;
		       nstack--;
		       return parse_exit;
		      }
     		     len1 = op1->vallen;
     		     len2 = op2->vallen;
     		     for ( n = 0 ; n <= (len1-len2) ; n++ )
		      if ( !memcmp(res->v_string+n,op2->v_string,len2) )
		       {
			memmove(res->v_string+n,res->v_string+n+len2,(len1-(n+len2)+1));
			res->vallen -= len2;
		       }
     		     break;
   case EXPR_NES   : cres = (strcmp(res->v_string,op2->v_string) != 0);
     		     break;
   case EXPR_LTS   : cres = (strcmp(res->v_string,op2->v_string) < 0);
     		     break;
   case EXPR_LES   : cres = (strcmp(res->v_string,op2->v_string) <= 0);
     		     break;
   case EXPR_GTS   : cres = (strcmp(res->v_string,op2->v_string) > 0);
     		     break;
   case EXPR_GES   : cres = (strcmp(res->v_string,op2->v_string) >= 0);
     		     break;
   case EXPR_EQS   : cres = (strcmp(res->v_string,op2->v_string) == 0);
     		     break;
  }
 /* Aufraeumen */
 if ( cmp )
  {
   free(res->v_string);
   res->kind = PUSH_NUMBER;
   res->v_number = cres;
   res->vallen = sizeof(res->v_number);
  }
 free(op2->v_string);
 nstack--;
 /* Erfolg melden */
 return parse_exit;
}

/* 
  Konvertiere einen Wert auf dem Stack in eine Zahl
*/
static doInteger(sv,def)
t_stack *sv;
long def;
{
 static struct epcbNUMBER *numpcb = 0;
 char *str,*str2,c;

 /* Tabelle laden */
 if ( !numpcb && !(numpcb = (struct epcbNUMBER *)getpcb(pcbNUMBER_module)) )
  {
   fprintf(stderr,"out of memory\n");
   exit(2);
  } 
 /* Symbol ermitteln */
 if ( !doSymbol(sv) ) return 0;
 /* Wert aendern */
 switch (sv->kind)
  {
   case PUSH_NULL   : /* Defaultwert einsetzen */
		      sv->v_number = def;
   case PUSH_NUMBER : break;
   case PUSH_STRING : 
   case PUSH_QUOTED : /* Leerzeichen entfernen */
		      for ( str = sv->v_string ; (*str == ' ') || (*str == '\t') ; str++ );
		      /* Auswerten */
     		      if ( !parse(numpcb,str) )
		       {
			/* Speicher freigeben */
			free(numpcb->pcb_tree);
			/* Duerfen nur noch Leerzeichen folgen */
		        str2 = numpcb->sys_pcb.pcb_line;
			while ( (*str2 == ' ') || (*str2 == '\t') ) str2++;
			/* Erlaubter Wert ? */
			if ( !*str2 )
			 {
			  free(sv->v_string);
			  sv->v_number = numpcb->pcb_value[1];
			  break;
			 }
		       }
		      /* Das erste Zeichen entscheidet */
		      c = *str;
		      free(sv->v_string);
		      sv->v_number = ((c == 'T') || (c == 't') || (c == 'Y') || (c == 'y'));
  }
 /* Ist jetzt eine Zahl */
 sv->kind = PUSH_NUMBER;
 sv->vallen = sizeof(sv->v_number);
 /* Alles in Ordung */
 return 1;
}

/* 
  Konvertiere einen Wert auf dem Stack in eine Zahl
*/
static doString(sv,def,getsym)
t_stack *sv;
char *def;
int getsym;
{
 static char buf[20];

 /* Symbol ermitteln */
 if ( getsym && !doSymbol(sv) ) return 0;
 /* Mal sehen, was wir haben */
 switch (sv->kind)
  {
   case PUSH_NUMBER : sprintf(def = buf,"%ld",sv->v_number);
   case PUSH_NULL   : if ( !(sv->v_string = strdup(def)) ) return 0;
     		      sv->vallen = strlen(sv->v_string);
   case PUSH_STRING : 
   case PUSH_QUOTED : break;
  }
 /* Ist jetzt ein String */
 sv->kind = PUSH_QUOTED;
 /* Alles in Ordung */
 return 1;
}

/*
  Teste, ob ein Symbol eingesetzt werden kann.
*/
static doSymbol(sv)
t_stack *sv;
{
 static char repl[1024];
 int rlen,flags = SYM$FLAG_GENERAL;
 char *new;

 /* Nur bei Strings */
 if ( sv->kind != PUSH_STRING ) return 1;
 /* Versuchen */
 if ( symbol_operation(SYM$OP_GET,&flags,
		       sv->v_string,sv->vallen,NOSTR,0,repl,sizeof(repl)-1,&rlen)&1 ) 
  {
   /* Wert ermitteln */
   if ( !(flags&SYM$FLAG_STRING) )
    {
     /* Zahlenwert */
     free(sv->v_string);
     sv->kind = PUSH_NUMBER;
     memmove(&sv->v_number,repl,sv->vallen = sizeof(sv->v_number));
     return 1;
    }
   else if ( new = malloc(rlen+1) ) 
    {
     memmove(new,repl,rlen);
     free(sv->v_string);
     /* Zeichenkette */
     sv->kind = PUSH_QUOTED;
     (sv->v_string = new)[sv->vallen = rlen] = '\0';
     return 1;
    }
  }
 /* Fehler */
 return 0;
}

/*
  Symbol umwandeln und eventuell & bearbeiten.
*/
static doAmpersand(des,flags)
struct descriptor *des;
long flags;
{
 static char value[1024];
 int len;

 /* Symboltext umsetzen */
 if ( !(des->text = createValue(des->text,(int)des->len,3,0)) ) return CDU_NOMEM;
 des->len = strlen(des->text); 
 /* Eventuell einmal uebersetzen */
 if ( flags&AMPER )
  if ( amprepl(des->text,(int)des->len,value,sizeof(value)-1,&len)&1 )
   {
    /* Als Symbol uebernehmen */
    free(des->text);
    value[len] = '\0';
    if ( !(des->text = strdup(value)) ) return CDU_NOMEM;
    des->len = strlen(des->text);
   }
  else
   return LIB$_NOSUCHSYM;
 /* Alles in Ordnung */
 return 0;
}

/*
  Symbol in einer bestimmten Tabelle suchen.
*/
static findSymbol(sv,local)
t_stack *sv;
int local;
{
 static char answer[2048];
 int flags = 0,rlen,ix[2];
 char *start,*new;;

 /* Indizes aufsetzen */
 ix[0] = local ? -1 : 0x7ffffffe;
 ix[1] = local ? 0x7ffffffe : -1;
 /* Aufrufen */
 if ( !(symbol_operation(SYM$OP_READOUT,&flags,
		         sv->v_string,sv->vallen,ix,sizeof(ix),
		         answer,sizeof(answer)-1,&rlen)&1) ) 
  return 0;
 /* Symbol gefunden ? */
 if ( rlen == sizeof(ix) ) return 0;
 /* Wert ermitteln */
 memmove(&rlen,answer+sizeof(ix)+sizeof(int),sizeof(rlen));
 start = answer+sizeof(ix)+sizeof(int)+sizeof(rlen)+rlen;
 memmove(&rlen,start,sizeof(rlen));
 start += 4;
 /* Symbol uebernehmen */
 if ( !(flags&SYM$FLAG_STRING) )
  {
   /* Zahlenwert */
   free(sv->v_string);
   sv->kind = PUSH_NUMBER;
   memmove(&sv->v_number,start,sv->vallen = sizeof(sv->v_number));
   return 1;
  }
 if ( !(new = malloc(rlen+1)) ) return 0;
 memmove(new,start,rlen);
 free(sv->v_string);
 /* Zeichenkette */
 sv->kind = PUSH_QUOTED;
 (sv->v_string = new)[sv->vallen = rlen] = '\0';
 /* Fertig */
 return 1;
}

/*
  DCL Funktionen.
*/
static Ftype(sv,par,npar)
t_stack *sv,*par;
int npar;
{
 char *value = "";

 /* Parameter muss vorhanden sein */
 if ( (npar != 1) || (par->kind == PUSH_NULL) ) return parse_fail;
 /* Eventuell suchen */
 if ( (par->kind == PUSH_STRING) && doSymbol(par) )
  value = (par->kind == PUSH_NUMBER) ? "INTEGER" : "STRING";
 /* Fertigmachen */
 sv->kind = PUSH_NULL;
 if ( !doString(sv,value,0) ) return parse_fail;
 return parse_exit;
}

static Finteger(sv,par,npar)
t_stack *sv,*par;
int npar;
{
 /* Parameter muss vorhanden sein */
 if ( (npar != 1) || (par->kind == PUSH_NULL) ) return parse_fail;
 /* Umwandeln */
 if ( !doInteger(par,0L) ) return parse_fail;
 /* Alles in Ordung */
 sv->v_number = par->v_number;
 return parse_exit;
}

static Fstring(sv,par,npar)
t_stack *sv,*par;
int npar;
{
 /* Parameter muss vorhanden sein */
 if ( (npar != 1) || (par->kind == PUSH_NULL) ) return parse_fail;
 /* Umwandeln */
 if ( !doString(par,"",1) ) return parse_fail;
 /* Alles in Ordung */
 *sv = *par;
 par->kind = PUSH_NUMBER;
 return parse_exit;
}

static Flength(sv,par,npar)
t_stack *sv,*par;
int npar;
{
 /* Parameter muss vorhanden sein */
 if ( (npar != 1) || (par->kind == PUSH_NULL) ) return parse_fail;
 /* Umwandeln */
 if ( !doString(par,"",1) ) return parse_fail;
 /* Alles in Ordung */
 sv->v_number = par->vallen;
 return parse_exit;
}

static Flocate(sv,par,npar)
t_stack *sv,*par;
int npar;
{ 
 char *str1,*str2;
 int len1,len2,n;

 /* Parameter muss vorhanden sein */
 if ( (npar != 2) || (par[0].kind == PUSH_NULL) || (par[1].kind == PUSH_NULL) ) 
  return parse_fail;
 /* Umwandeln */
 if ( !doString(par+0,"",1) || !doString(par+1,"",1) ) return parse_fail;
 /* Alles in Ordung */
 str1 = par[0].v_string;
 len1 = par[0].vallen;
 str2 = par[1].v_string;
 len2 = par[1].vallen;
 for ( n = 0 ; (n <= (len2-len1)) && memcmp(str2++,str1,len1) ; n++ );
 sv->v_number = (n > (len2-len1)) ? len2 : n;
 return parse_exit;
}

static Fextract(sv,par,npar)
t_stack *sv,*par;
int npar;
{
 int off,size;
 
 /* Parameter muss vorhanden sein */
 if ( (npar != 3) || 
      (par[0].kind == PUSH_NULL) || (par[1].kind == PUSH_NULL) || (par[2].kind == PUSH_NULL) ) 
  return parse_fail;
 /* Umwandeln */
 if ( !doInteger(par+0,0) || !doInteger(par+1,0) || !doString(par+2,"",1) ) return parse_fail;
 /* Alles in Ordung */
 off = par[0].v_number;
 size = par[1].v_number;
 *sv = par[2];
 par[2].kind = PUSH_NUMBER;
 /* Ausschneiden */
 if ( (off < 0) || (off >= sv->vallen) )
  size = 0;
 else
  {
   /* Echte Groesse */
   if ( (size < 0) || (off+size > sv->vallen) ) size = sv->vallen-off;
   memmove(sv->v_string,sv->v_string+off,size);
  }
 /* Fertig */
 sv->v_string[sv->vallen = size] = '\0';
 return parse_exit;
}

/*
  Speicher fuer Parameterstrings freigeben.
*/
static freepara(par,npar)
t_stack *par;
int npar;
{
 /* Aufraeumen */
 while ( npar-- ) freestack(par++);
}

/* 
  Stackelement aufraeumen.
*/
static freestack(sv)
t_stack *sv;
{
 /* Stringspeicher freigeben */
 switch (sv->kind)
  {
   case PUSH_STRING   :
   case PUSH_QUOTED   : if ( sv->v_string ) free(sv->v_string);
  }
}

/*
  SHOW SYMBOL p2
  	/LOCAL
	/GLOBAL
	/ALL
	/LOG
*/
CLIshowsymbl()
{
 static char symbol[256],answer[2048];
 int gix = -1,lix = -1,slen,rlen,nlen,vlen,abbr,nsym = 0,ix[2],flags,res,gglob,gloc;
 char *name,*value,keep;
 long val;

 /* /NOLOG */
 get_nolog();
 /* /LOCAL und /GLOBAL global als Qualifier zu SYMBOL */
 if ( VALUE("OPTION",answer,&rlen)&1 )
  {
   gglob = PRESENT("GLOBAL")&1;
   gloc = PRESENT("LOCAL")&1;
  }
 else
  gglob = gloc = 0;
 /* Symbol */
 res = VALUE("P2",symbol,&slen);
 if ( res&1 )
  {
   /* /GLOBAL */
   if ( gglob || (PRESENT("GLOBAL")&1) )
    lix = 0x7ffffffe;
   else if ( gloc || (PRESENT("LOCAL")&1) )
    gix = 0x7ffffffe;
  }
 else
  {
   /* /ALL */
   strcpy(symbol,"*");
   slen = 1;
   /* /GLOBAL */
   if ( gglob )
    lix = 0x7ffffffe;
   else
    gix = 0x7ffffffe;
  }
 /* Tabelle der Symbole auslesen */
 do
  {
   /* Befehl abschicken */
   flags = SYM$FLAG_LOCAL;
   ix[0] = lix;
   ix[1] = gix;
   res = symbol_operation(SYM$OP_READOUT,&flags,symbol,slen,ix,sizeof(ix),
			  			answer,sizeof(answer)-1,&rlen);
   if ( !(res&1) )
    {
     Status = res;
     return;
    }
   /* Ergebnis ermitteln */
   if ( rlen == sizeof(ix) ) break;
   memmove(&lix,answer+0,sizeof(lix));
   memmove(&gix,answer+sizeof(lix),sizeof(gix));
   memmove(&abbr,answer+sizeof(ix),sizeof(abbr));
   memmove(&nlen,answer+sizeof(ix)+sizeof(abbr),sizeof(nlen));
   name = answer+sizeof(ix)+sizeof(abbr)+sizeof(nlen);
   memmove(&vlen,name+nlen,sizeof(vlen));
   value = name+nlen+sizeof(vlen);
   name[nlen] = value[vlen] = '\0';
   /* Ergebnis anzeigen */
   printf("  ");
   if ( !abbr )
    printf("%s =",name);
   else
    {
     keep = name[abbr];
     name[abbr] = '\0';
     printf("%s*%c%s =",name,keep,name+abbr+1);
    }
   if ( !(flags&SYM$FLAG_LOCAL) ) printf("=");
   if ( flags&SYM$FLAG_STRING )
    {
     for ( name = value ; vlen-- > 0 ; name++ )
      if ( (*name < ' ') || (*name > '~') )
       *name = '.';
     printf(" \"%s\"\n",value);
    }
   else
    {
     memmove(&val,value,sizeof(val));
     printf(" %ld   Hex = %08lX  Octal = %011lo\n",val,val,val);
    }
   /* Zaehlen */
   nsym++;
  }
 while ( (lix != -1) || (gix != -1) );
 /* Fehler */
 if ( nsym )
  NewStatus = 0;
 else
  Status = LIB$_NOSUCHSYM;
 /* Alles in Ordnung */
 return;
}

/*
  SET SYMBOL
  	/ALL
	/GENERAL
	/VERB
	/SCOPE=([NO]LOCAL,[NO]GLOBAL)
*/
CLIsetsymbol()
{
 int dflags,mask1,mask2;

 /* /GENERAL */
 if ( PRESENT("GENERAL")&1 )
  {
   dflags = SYM$FLAG_GENERAL;
   mask1 = SYM$FLAG_GLOCAL;
   mask2 = SYM$FLAG_GGLOBAL;
  }
 /* /VERB */
 else if ( PRESENT("VERB")&1 )
  {
   dflags = SYM$FLAG_VERB;
   mask1 = SYM$FLAG_VLOCAL;
   mask2 = SYM$FLAG_VGLOBAL;
  }
 /* /ALL */
 else
  {
   dflags = SYM$FLAG_VERB|SYM$FLAG_GENERAL;
   mask1 = SYM$FLAG_VLOCAL|SYM$FLAG_GLOCAL;
   mask2 = SYM$FLAG_VGLOBAL|SYM$FLAG_GGLOBAL;
  }
 /* /SCOPE = [NO]LOCAL */
 setsymbol(dflags|SYM$FLAG_LOCAL,PRESENT("LOCAL"),mask1);
 /* /SCOPE = [NO]GLOBAL */
 setsymbol(dflags,PRESENT("GLOBAL"),mask2);
}

/*
  Operation ausfuehren.
*/
static setsymbol(flags,present,lflags)
int flags,present,lflags;
{
 int res;

 /* XXXX */
 if ( present == CLI$_PRESENT )
  flags |= lflags;
 /* NOXXXX */
 else if ( present != CLI$_NEGATED )
  return;
 /* Ausfuehren */
 res = symbol_operation(SYM$OP_SETOPTION,&flags,NOSTR,0,NOSTR,0,NOSTR,0,NOINT);
 if ( !(res&1) ) Status = res;
}

/*
  DELETE/SYMBOL [P1]
  	/LOCAL
	/GLOBAL
	/ALL
	/LOG
*/
CLIdelsym()
{
 static char symbol[256];
 int flags = 0,slen;

 /* /NOLOG */
 get_nolog();
 /* /GLOBAL und /LOCAL */
 if ( !(PRESENT("GLOBAL")&1) ) flags = SYM$FLAG_LOCAL;
 /* /ALL */
 if ( !(VALUE("P1",symbol,&slen)&1) ) 
  {
   slen = 0;
   flags |= SYM$FLAG_GENERAL;
  }
 /* Ausfuehren */
 Status = symbol_operation(SYM$OP_DELETE,&flags,symbol,slen,NOSTR,0,NOSTR,0,NOINT);
}
