#include "CDU.h"

/*
  Vordefinierte Typen und deren aequivalente. Die Typen muessen alphabetisch
  sortiert sein, da sonst Inkonsistenzen mit der Parsertabelle moeglich
  sind. Zu jedem Modul kann ein Parameter uebergeben werden, der eventuell
  zu kopierende nicht textuelle Daten anzeigt.
*/
char *predef[] =
     {
      "$ACL",
      "$DATETIME",
      ".DEFAULT",
      "$DELTATIME",
      "$DEVICE",
      "$DIRECTORY",
      "$EXPRESSION",
      "$FILE",
      "$LOGICAL",
      "$NODE",
      "$NUMBER",
      "$PROCESS",
      "$QUOTED_STRING",
      "$REST_OF_LINE",
      "$SYMBOL",
      "$UIC"
     };

static char *maps[][2] =
       {
        { "$OUTFILE", "$FILE"    },
        { "$INFILE" , "$FILE"    },
        { "$OUTLOG" , "$LOGICAL" },
        { "$INLOG"  , "$LOGICAL" },
        { "$INSYM"  , "$SYMBOL"  }
       };

#define PREDEF		(sizeof(predef)/sizeof(predef[0]))
#define MAPS		(sizeof(maps)/sizeof(maps[0]))

/*
  Optimierungstabellen fuer Werte.
*/
static struct Value *optval[256];

/*
  Hauptroutine fuer Phase 2 dieses Programms. In diesem Schritt wird
  versucht alle Referenzen aufzuloesen, d.h. Namen in die Adressen
  gewisser Strukturen umzusetzten. Es werden nur die Referenzen
  aufgeloest, die ausgehend von den VERBs wirklich benoetigt werden.
*/
Resolve()
{
 REGISTER struct Verb *ds;
 REGISTER int i;
 
 for ( i = 256 ; i-- ; )
  for ( curverb = verb[i] ; curverb ; curverb = curverb->next )
   {
    bzero(optval,sizeof(optval));
    curverb->syntax = 0;
    curverb->nsyntax = 0;
    ResolveVerb(curverb);
    for ( ds = curverb ; ds ; ds = ds->syntax )
     ResolveExpression(ds,ds->disallow);
    if ( !errors ) GenerateVerb(curverb);
   }
}

/*
  Aufloesung der Referenzen eines VERBs oder einer SYNTAX.
*/
static ResolveVerb(v)
REGISTER struct Verb *v;
{
 REGISTER struct Parameter *p;

 if ( v->verb == curverb ) return;
 v->verb = curverb;
 v->iaddr = 0;
 if ( v != curverb )
  {
   v->syntax = curverb->syntax;
   curverb->syntax = v;
  }
 curverb->nsyntax++;
 for ( p = v->parameter ; p ; p = p->next ) ResolveParameter(p);
 for ( p = v->qualifier ; p ; p = p->next ) ResolveParameter(p);
}

/*
  Werte optimieren, so dass zwei identische Codeteile nur einmal in der
  Parsertabelle vorkommen.
*/
static struct Value *optimizeValue(v)
REGISTER struct Value *v;
{
 REGISTER struct Value **list,*cur;
 REGISTER long hash;

 hash = (long)v->flags ^(long)v->defval ^(long)v->type^(long)v->pflags+
	(long)v->predef^(long)v->psyntax^(long)v->ispar;
 hash ^= hash>>16;
 list = optval+((hash^(hash>>8))&0xff);
 for ( ; cur = *list ; list = &cur->next )
  if ( (v->flags  == cur->flags ) && (v->defval  == cur->defval ) &&
       (v->type   == cur->type  ) && (v->predef  == cur->predef ) &&
       (v->ispar  == cur->ispar ) && (v->psyntax == cur->psyntax) &&
       (v->pflags == cur->pflags) )
   return cur;
 v->next = 0;
 return (*list = v);
}

/*
  Referenzen eines Parameters, Qualifiers oder Schluesselwortes
  aufloesen. 
*/
static ResolveParameter(p)
REGISTER struct Parameter *p;
{
 REGISTER struct Verb *s = 0;
 REGISTER struct Type *t;

 if ( p->synname )
  if ( (s = getSyntax(p->synname)) && !strcmp(s->name,p->synname) )
   ResolveVerb(p->syntax = s);
  else
   error("SYNTAX '%s' ist nicht definiert",p->synname);
 if ( !(p->optval = p->value) ) return;
 ResolveValue(p->value);
 if ( errors ) return;
 p->value->psyntax = s;
 p->optval = optimizeValue(p->value);
 if ( ((p->flags&idfDEF) != idfDEF) && ((p->flags&idfBAT) != idfBAT) )
  return;
 if ( (t = p->value->type) && !t->ndef && !t->ndval )
  error("TYPE '%s' hat keinen DEFAULT Wert",t->name);
}

/*
  Alle Referenzen in einem Ausdurck durch die Adresse des jeweiligen
  Parameters, Qualifiers oder Schluesselwortes aufloesen.
*/
static ResolveExpression(syn,e)
struct Verb *syn;
REGISTER struct Expression *e;
{
 for ( ; e ; e = e->next )
  switch (e->kind)
   {
    case EK_ANY2 :
    case EK_LEAF :
    case EK_NEG  : ResolveReference(syn,e->ereference);
    		   break;
    case EK_AND  :
    case EK_OR   : ResolveExpression(syn,e->eleft);
    case EK_NOT  : 
    case EK_SUB  : ResolveExpression(syn,e->eright);
    		   break;
   }
}

/*
  Referenzblock aufloesen durch Suchen der zugehoerigen Defintion.
*/
static ResolveReference(syn,r)
struct Verb *syn;
REGISTER struct Reference *r;
{
 REGISTER struct Verb *v;
 REGISTER struct Parameter *p,*l;
 struct Parameter *res;
 struct Verb *s;
 char **np;
 int **ap;

 for ( ; r ; r = r->next )
  {
   if ( !r->def )
    v = syn;
   else
    {
     v = getVerb(r->def);
     s = getSyntax(r->def);
     if ( !v )
      if ( !s )
       {
        error("Definition '%s' nicht gefunden",r->def);
        continue;
       }
      else
       v = s;
     else if ( s )
      {
       error("Definitionsreferenz '%s' ist nicht eindeutig",r->def);
       continue;
      }
     if ( v->verb != curverb )
      {
       error("Referenz '%s' gehoert nicht zum aktuellem VERB",v->name);
       continue;
      }
    }
   r->base = v;
   p = res = 0;
   for ( np = r->names, ap = r->addrs ; *np ; v = 0, res = 0 )
    {
     if ( !p )
      {
       for ( l = v->parameter ; l ; l = l->next )
        if ( !strcmp(l->label ? l->label : l->name,*np) )
	 break;
       if ( !l )
	for ( l = v->qualifier ; l ; l = l->next )
         if ( !strcmp(l->label ? l->label : l->name,*np) )
 	  break;
      }
     else if ( !p->value || !p->value->type )
      l = 0;
     else
      for ( l = p->value->type->keyword ; l ; l = l->next )
       if ( !strcmp(l->label ? l->label : l->name,*np) )
	break;       
     if ( l )
      {
       np++;
       p = l;
      }
     else
      {
       if ( !findReference(*np++,v,p,&res) ) break;
       if ( !(p = res) )
        {
         error("Referenz '%s' nicht gefunden",np[-1]);
         break;
        }
      }
     *ap++ = (p->label ? &p->laddr : &p->naddr);
    }
  }
}

/*
  Reference bezueglich einem Bezugspunkt aufloesen und auf Eindeutigkeit
  ueberpruefen. In der Codeerzeugungsphase laeuft eine aehnliche Analyse
  der Zusammenhaenge nochmals ab.
*/
static findReference(n,v,p,res)
char *n;
REGISTER struct Verb *v;
struct Parameter *p,**res;
{
 REGISTER struct Type *t;
 int ok = 1;

 if ( v )
  {
   if ( !findList(n,v->parameter,res) ||
        !findList(n,v->qualifier,res) )
    return 0;
  }
 else if ( p->value && (t = p->value->type) && (t->verb == curverb) )
  {
   t->verb = 0;
   ok = findList(n,t->keyword,res);
   t->verb = curverb;
  }
 return ok;
}

/*
  Eine Liste von Parametern durchsuchen.
*/
static findList(n,l,res)
char *n;
REGISTER struct Parameter *l,**res;
{
 for ( ; l ; l = l->next )
  {
   if ( !strcmp(n,l->label ? l->label : l->name) )
    {
     if ( *res )
      {
       error("Referenz '%s' ist nicht eindeutig",n);
       return 0;
      }
     *res = l;
    }
   if ( !findReference(n,(struct Verb *)0,l,res) ) return 0;
  }
 return 1;
}

/*
  Suche den zu dem Wert gehoerigen Typ. Dazu wird zuerst die Liste des
  Benutzers und dann die interne durchsucht. Dadurch ist es moeglich,
  die vordefinierten Typen bei Bedarf zu ueberschreiben. Ist der Typ
  durch den CDU-Quelltext gegeben, so ist die Angabe eines Default-
  wertes nicht gestattet.
*/
static ResolveValue(v)
REGISTER struct Value *v;
{
 REGISTER struct Type *t;
 REGISTER char *look;
 REGISTER int i;

 v->caddr = 0;
 v->psyntax = 0;
 if ( !v->typename ) v->typename = ".DEFAULT";
 if ( (t = getType(v->typename)) && !strcmp(t->name,v->typename) )
  {
   ResolveType(v->type = t);
   if ( v->defval )
    error("DEFAULT und TYPE='%s' gleichzeitig verwendet",v->typename);
  }
 else
  {
   look = v->typename;
   for ( i = MAPS ; i-- ; )
    if ( !strcmp(maps[i][0],look) )
     {
      look = maps[i][1];
      break;
     }
   for ( i = PREDEF ; i-- ; )
    if ( !strcmp(predef[i],look) )
     {
      v->typename = predef[v->predef = i];
      return;
     }
   error("TYPE '%s' ist nicht definiert",v->typename);
  } 
}

/*
  Alle Schluesselworte eines Types durchsuchen.
*/
static ResolveType(t)
REGISTER struct Type *t;
{
 REGISTER struct Parameter *k;

 if ( t->verb == curverb ) return;
 t->verb = curverb;
 t->kaddr = t->caddr = 0;
 for ( k = t->keyword ; k ; k = k->next ) ResolveParameter(k);
}

