#include "CDU.h"

extern char *predef[];

/*
  Version der erzeugten CDU-Parsertabelle.
*/
#define CDUVERSION	0x4d02

/*
  Aktionsroutinen fuer die Parsertabelle. Die hier aufgelisteten Routinen
  muessen in alphabethischer Reihenfolge angeordnet sein, da der Parser
  ansonsten nicht in der Lage ist, korrekt zu arbeiten.
*/
static char *routines[] =
       {
	"AddSeparator",
        "AddValue",
	"DefaultKeyword",
	"DefaultParameter",
	"DefaultQualifier",
        "EndValue",
	"Initialize",
	"SetError",
        "StartValue",
	"SwitchParameter",
	"SwitchQualifier",
	"SwitchSyntax"
       };

#define NOSTR		((char *)0)
#define ADDS		routines[ 0]
#define ADDV		routines[ 1]
#define DEFK		routines[ 2]
#define DEFP		routines[ 3]
#define DEFQ		routines[ 4]
#define ENDV		routines[ 5]
#define INIT		routines[ 6]
#define SETE		routines[ 7]
#define STARTV		routines[ 8]
#define SWTP		routines[ 9]
#define SWTQ		routines[10]
#define SWTS		routines[11]

#define ROUTINES	(sizeof(routines)/sizeof(routines[0]))

/*
  CURSIZE	Aktuelle Groesse der Tabelle eines VERBs
  FILLED	Bereits hinausgeschriebene Datenbytes aller Tabellen
  VERBCNT	Anzahl der VERBs in der Tabelle
  PARDEF        Selektion nach DEFAULT Parametern
  QUALPOS	Selektion nach aktueller Qualifiergruppe
  VALTAB	Tabelle der Werte fuer die 'arg' Aufrufe
  AVAL		Allokatierte Groesse des Feldes VALTAB
  NVAL		Benutzte Eintraege in VALTAB
*/
static int cursize,filled = 0,verbcnt = 0,qualpos,pardef,aval = 0,nval = 0;
static short *valtab;

#define BYTESPERLINE	16

#define VALSIZE(n)	(3*sizeof(valtab[0])*(n))
#define VALTAB(n)	(valtab+3*(n))

/*
  Konstanten fuer die Parsertabelle.
*/
#define SIZE_MIN	3
#define SIZE_SUB	4
#define SIZE_STD	6
#define SIZE_EXT	7

/*
  Einige MACROs, die anstelle von Routinen eingesetzt werden koennen. Die
  ersten fuenf verschluesseln drei Parameter in ein einzelnes Argument fuer
  eine Aktionsroutine des Parsers.
*/
#define argt(i)			((long)(i))

#define arg0			arg(0,0,0)
#define arg2(i)			arg(0,i,0)

#define codeChar(c,t,a)		gc(c,fm_onekey,t,0,a,arg0)
#define codeCmp(t,a,i)  	gc(LAMBDA,0,t,0,a,argt(i))
#define codeErr(e)     	 	gc(LAMBDA,0,state_exit,0,SETE,argt(e))
#define codeGoto(t,a,i) 	gc(LAMBDA,0,t,0,a,argt(i))
#define codeKey(t,k,i)   	gc(KEY,0,t,k,STARTV,argt(i))
#define codeSep(s,t)		gc(s,fm_onekey,t,0,ADDS,argt(s))
#define codeSub(t,s)		gc(SUB,0,t,s,NOSTR,arg0)
#define codeText(s)		gt(s,1)
#define codePref(s)		gt(s,0)
#define codeNO()		codePref("NO")
#define codeWord(i)		gt(NOSTR,i)

/*
  Struktur fuer Routinen. Sind Routinen vorhanden, so wird eine C-Datei
  erzeugt, die zum Programm, das die CDU-Tabelle nutzen moechte, hinzu-
  gelinkt werden muss.
*/
struct Routine
       {
	struct Routine *next;
	char	       *name;
       };

static struct Routine *rnames = 0;

/*
  Speicherbereich fuer die Zusammenstellung der Parsertabelle.
*/
#define BLKSIZE		0x01000
#define MAXSIZE		0x40000
#define MAXBLK		(MAXSIZE/BLKSIZE)

static int talloc = 0,tmax = 0;
static char *mem[MAXBLK];

/*
  Konstanten fuer die Parsertabelle.
*/
#define fm_action	0x01
#define fm_parameter	0x02
#define fm_onekey	0x80

#define state_exit	-1
#define state_fail	-3

#define abbr_unique	-1

#define KEY		9
#define LAMBDA		10
#define EOS		11
#define SUB		12
#define MODULE		14

struct table_header
       {
	short          th_magic;
	long           th_total;
	long           th_debug;
	unsigned char  th_version;
	unsigned char  th_flags;
	unsigned char  th_abbr;
	unsigned char  th_tabs;
	unsigned short th_size;	
	unsigned short th_clear;	
       };

static char ctab[6][32] =
       {
	{
	 0xfe,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
	},
	{
	 0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x03,
	 0xfe,0xff,0xff,0x07,0xfe,0xff,0xff,0x07,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	{
	 0x00,0x00,0x00,0x00,0x00,0x00,0xff,0x03,
	 0xfe,0xff,0xff,0x07,0xfe,0xff,0xff,0x07,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	{
	 0x00,0x00,0x00,0x00,0x10,0x00,0x00,0x00,
	 0xfe,0xff,0xff,0x07,0xfe,0xff,0xff,0x07,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	{
	 0x00,0x00,0x00,0x00,0x10,0x00,0xff,0x03,
	 0xfe,0xff,0xff,0x87,0xfe,0xff,0xff,0x07,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	},
	{
	 0xfe,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	 0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
	}
       };

static struct table_header head =
       {
	0x4a02,
	0,
	0,
	0x10,
	0,
	abbr_unique,
	sizeof(ctab)/sizeof(ctab[0]),
	sizeof(struct epcbCDU)-sizeof(struct pcb),
	0
       };

/*
  Erzeuge Bytes fuer die Parsertabelle.
*/
static gc(op,onekey,target,sub,routine,args)
int op,onekey,target,sub;
REGISTER char *routine;
long args;
{
 static short buf[SIZE_EXT];
 REGISTER char *code = (char *)buf;
 REGISTER short *bp = buf+1;
 REGISTER int ix;
 short *lp;

 code[0] = op;
 code[1] = onekey;
 switch (op)
  {
   case KEY    : 
   case SUB    : *bp++ = sub-(cursize+1);
		 break;
   case MODULE : *bp++ = sub;
    		 break;
  }
 lp = bp++;
 if ( (target == state_exit) || (target == state_fail) )
  *bp = target;
 else
  *bp = target-(cursize+(bp-buf));
 bp++;
 if ( routine )
  {
   for ( ix = ROUTINES ; ix-- && (routine != routines[ix]) ; ) ;
   *bp++ = ix;
   *bp++ = args>>16;
   *bp++ = args;
   code[1] |= (fm_action|fm_parameter);
  }
 *lp = bp-(buf+(lp-buf));
 if ( onekey )
  if ( op )
   {
    code[0] = code[1];
    code[1] = op;
   }
  else
   {
    code[0] = EOS;
    code[1] &= ~fm_onekey;
   }
 writeBytes(buf,(char *)bp-(char *)buf);
}

/*
  Textausgabe erzeugen.
*/
static gt(s,pad)
REGISTER char *s;
REGISTER int pad;
{
 REGISTER int l;
 short val;

 if ( s )
  {
   writeBytes(s,l = strlen(s));
   if ( !((l+pad+1)&1) ) writeBytes("\000",2);
  }
 else
  {
   val = pad;
   writeBytes(&val,2);
  }
}

/*
  Argument in die Wertetabelle eintragen und Position in dieser melden.
*/
arg(i1,i2,i3)
int i1,i2,i3;
{
 short *nt;
 int ix;

 /* Wert suchen */
 for ( ix = nval ; ix-- ; )
  if ( (VALTAB(ix)[0] == i1) && (VALTAB(ix)[1] == i2) && (VALTAB(ix)[2] == i3) ) 
   return 3*ix;
 /* Speicher fuer den neuen Wert reservieren */
 if ( nval == aval )
  {
   if ( !(nt = MALLOC(short,VALSIZE(aval+16))) ) nomem();
   if ( aval )
    {
     memcpy((char *)nt,(char *)valtab,VALSIZE(aval));
     free(valtab);
    }
   aval += 16;
   valtab = nt;
  }
 /* Wert in die Tabelle eintragen */
 VALTAB(nval)[0] = i1;
 VALTAB(nval)[1] = i2;
 VALTAB(nval)[2] = i3;
 /* Ergebnis melden */
 return 3*nval++;
}

/*
  Daten in den Speicherbereich schreiben.
*/
static writeBytes(addr,bytes)
char *addr;
int bytes;
{
 int pos = cursize*2,end = pos+(bytes = (bytes+1)&~1);
 int offset,len;
 char **bptr;

 if ( end > MAXSIZE )
  {
   error("Tabelle wird groesser als %ld Bytes",(long)MAXSIZE);
   exit(2);
  }
 while ( talloc < end )
  {
   if ( !(mem[talloc/BLKSIZE] = MALLOC(char,BLKSIZE)) ) nomem();
   talloc += BLKSIZE;
  }
 if ( end > tmax ) tmax = end;
 bptr = mem+(pos/BLKSIZE);
 offset = pos%BLKSIZE;
 while ( bytes )
  {
   if ( (len = (BLKSIZE-offset)) > bytes ) len = bytes;
   bcopy(addr,(*bptr)+offset,len);
   bptr++;
   offset = 0;
   addr += len;
   bytes -= len;
  } 
 cursize = end/2;
}

/*
  Positionsattribut eines Qualifiers oder Schluesselwortes feststellen.
*/
codePos(p,val)
REGISTER struct Parameter *p;
REGISTER int val;
{
 if ( (p->flags&idfLOC) == idfLOC ) return val|CDU_LOCAL;
 if ( (p->flags&idfPOS) == idfPOS ) return val;
 return val|CDU_GLOBAL;
}

/*
  VERB durchgehen und Code intern erzeugen. Die Daten werden in dem globalem
  MEM-Feld aufgebaut und dann in die TAB-Datei im Parsertabellenformat
  geschrieben.
*/
GenerateVerb(v)
REGISTER struct Verb *v;
{
 REGISTER int size1,size2,size3,size4,vtab;
 REGISTER struct Verb *s;

 tmax = nval = 0;
 size1 = SIZE_STD+3*SIZE_MIN+SIZE_STD;
 size2 = SIZE_SUB+SIZE_STD;
 size3 = 0;
 size4 = SIZE_MIN;
 for ( s = v ; s ; s = s->syntax )
  {
   if ( s->npar ) size1 += SIZE_STD;
   if ( s->nqual ) size2 += SIZE_STD;
   if ( s->ndef || s->ndval ) size3 += SIZE_STD;
   if ( s->pdef ) size4 += SIZE_STD;
  }
 cursize = size1+size2+size3+size4;
 qualpos = size1+SIZE_SUB;
 pardef = size1+size2+size3;
 textVerb(v,1);
 textVerb(v,0);
 for ( s = v ; s ; s = s->syntax ) codeSyntax(s);
 vtab = cursize;
 cursize = 0;
 codeGoto(SIZE_STD,INIT,argt((vtab<<16)|v->iaddr));
 codeChar('/',size1,NOSTR);
 codeChar('!',size1+size2,NOSTR);
 codeChar(0,size1+size2,NOSTR);
 for ( s = v ; s ; s = s->syntax )
  if ( s->npar )
   codeCmp(s->paddr,SWTS,arg(CDU_START,s->iaddr,0));
 codeErr(CDU_NOPAR);
 codeSub(SIZE_STD,qualpos);
 for ( s = v ; s ; s = s->syntax )
  if ( s->nqual )
   codeCmp(s->caddr,SWTQ,arg(CDU_START,s->iaddr,0));
 codeErr(CDU_NOQUAL);
 for ( s = v ; s ; s = s->syntax )
  if ( s->ndef || s->ndval )
   codeCmp(s->dqaddr,SWTQ,arg(CDU_NOSTART,s->iaddr,0));
 for ( s = v ; s ; s = s->syntax )
  if ( s->pdef )
   codeCmp(s->dpaddr,SWTS,arg(CDU_NOSTART,s->iaddr,0));
 codeGoto(state_exit,NOSTR,arg0); 
 if ( nval )
  {
   cursize = vtab;
   writeBytes(VALTAB(0),VALSIZE(nval));
  }
 flushVerb(); 
}

/*
  Texte eines VERBs einbauen. In einem ersten Schritt werden die legale
  Schluesselworte eingetragen. Damit kann dann anhand der Tabelle ermittelt
  werden, welche Schluesselwortketten legal sind und welche nicht.
*/
static textVerb(v,keys)
REGISTER struct Verb *v;
REGISTER int keys;
{
 REGISTER struct Parameter *p;

 for ( ; v ; v = v->syntax )
  { 
   if ( keys ) v->saddrP = cursize;
   for ( p = v->parameter ; p ; p = p->next ) textParameter(p,keys);
   if ( keys )
    {
     codeWord(0);
     v->saddrQ = cursize;
    } 
   for ( p = v->qualifier ; p ; p = p->next ) textParameter(p,keys);
   if ( keys ) codeWord(0);
  }
}

/*
  Namen eines Parameters, Qualifiers oder Schluesselwortes eintragen. Im
  ersten Durchlauf werden nur die bei Suchlisten benutzten Kennungen heraus-
  geschrieben.
*/
static textParameter(p,keys)
REGISTER struct Parameter *p;
REGISTER int keys;
{
 REGISTER struct Type *t;
 REGISTER char *s;

 if ( keys )
  {
   p->xaddr = cursize;
   codeNO();
   p->laddr = cursize;
   if ( s = p->label )
    p->naddr = 0;
   else
    {
     p->naddr = cursize;
     s = p->name;
    }
   codeText(s);
  }  
 else if ( !p->naddr )
  {
   if ( (p->flags&idfNEG) == idfNEG )
    {
     p->xaddr = cursize;
     codeNO();
    }
   p->naddr = cursize;
   codeText(p->name);
  }
 if ( p->optval ) textValue(p->optval,keys);
 if ( keys ) codeWord(0);
}

/*
  Texte eines Wertes eintragen. Der Defaultstring wird in der zweiten
  Phase vermerkt. Ansonsten dient ein Wert lediglich dazu, alle definierten
  TYPEs zu finden.
*/
static textValue(v,keys)
REGISTER struct Value *v;
REGISTER int keys;
{
 if ( !keys && v->defval ) 
  {
   v->daddr = cursize;
   codeText(v->defval);
  }
 if ( v->type ) textType(v->type,keys);
}

/*
  Texte eines TYPEs bearbeiten:
  a) Erste Definition:
     0xff00	TYPE
     Schluesselworte
     0x0000     implizit durch das uebergeordnete Schluesselwort
  b) Referenz
     0xffff	TYPE Referenz
     0x____	Adresse des ersten Schluesselwortes
*/
static textType(t,keys)
REGISTER struct Type *t;
REGISTER int keys;
{
 REGISTER struct Parameter *p;

 if ( keys )
  {
   if ( t->kaddr )
    {
     codeWord(0xffff);
     codeWord(t->kaddr);
     return;
    }
   t->kaddr = cursize;
   codeWord(0xff00);
  }
 else if ( t->verb != curverb )
  return;
 t->verb = 0;
 for ( p = t->keyword ; p ; p = p->next ) textParameter(p,keys);
 t->verb = curverb;
}

/*
  Code fuer ein negierbares Schluesselwort erzeugen. Um ein sauberes Arbeiten
  des Parser zu garantieren, muessen alle Schluesselworte alphabetisch
  sortiert sein. Dazu gehoeren natuerlich auch die Worte, denen ein NO vor-
  angesetzt wird.
*/
static struct Parameter *codeNegated(neg,p)
REGISTER struct Parameter *neg,*p;
{
 if ( !p || (strcmp(p->name,"NO") >= 0) )
  if ( p && (p->name[0] == 'N') && (p->name[1] == 'O') )
   {
    for ( ; neg ; neg = neg->next )
     if ( (neg->flags&idfNEG) == idfNEG )
      {
       if ( strcmp(p->name+2,neg->name) < 0 ) break;
       codeKey(state_exit,neg->xaddr,
	       arg(neg->laddr,neg->syntax ? neg->syntax->iaddr : 0,codePos(neg,CDU_NEG)));
      }
   }
  else
   for ( ; neg ; neg = neg->next )
    if ( (neg->flags&idfNEG) == idfNEG )
     codeKey(state_exit,neg->xaddr,
	     arg(neg->laddr,neg->syntax ? neg->syntax->iaddr : 0,codePos(neg,CDU_NEG)));
 return neg;
}

/*
  Code fuer eine Syntax erzeugen. Dazu werden im Prinzip drei Tabellenbereich
  erzeugt:
  1. Verteilung zwischen Qualifiern und Parametern
  2. Defaultqualifier am Ende setzen
  3. Qualifier durchsuchen
*/
static codeSyntax(v)
REGISTER struct Verb *v;
{
 REGISTER struct Parameter *p;
 REGISTER struct Synonym *s;
 REGISTER int ix;
 int start,sep,keep;

 if ( v->iaddr ) return;
 v->iaddr = cursize;
 codeWord(((v->flags&vfNQU) == vfNQU) ? -1 : v->nqual);
 codeWord(((v->flags&vfNPA) == vfNPA) ? -1 : v->npar);
 codeWord(((v->flags&vfCFL) == vfCFL) ? (int)v->CLIflags : -1);
 if ( v->image )
  {
   if ( (v->flags&vfROU) == vfROU )
    {
     addRoutine(v->image);
     codePref("RO");
    }
   else if ( (v->flags&vfCLI) == vfCLI )
    codePref("CR");
   else
    codePref("IM");
   codeText(v->image);
  }
 else
  codeWord(0);
 if ( v->prefix )
  codeText(v->prefix);
 else
  codeWord(0);
 if ( (v->flags&vfNDI) == vfNDI )
  codeWord(0xffff);
 else
  codeDisallows(v->disallow);
 for ( p = v->parameter, sep = 0 ; p ; p = p->next ) 
  {
   if ( !sep && ((p->optval->flags&vcfREQ) != vcfREQ) )
    {
     sep = 1;
     codeWord(0); 
    }
   if ( p->prompt )
    codeText(p->prompt);
   else
    {
     codeWord(0xffff);
     codeWord(p->label ? p->laddr : p->naddr);
    }
  }
 if ( !sep ) codeWord(0);
 codeWord(0);
 codeWord(v->saddrP);
 codeWord(v->saddrQ);
 if ( v->isverb )
  {
   codeText(v->name);
   for ( s = v->synonym ; s ; s = s->next, verbcnt++ ) codeText(s->text);
   codeWord(0);
  }
 start = cursize;
 if ( v->npar ) cursize += (v->npar+1)*SIZE_STD;
 if ( v->nqual ) cursize += (v->nqual+v->nneg)*SIZE_EXT+SIZE_STD;
 if ( v->ndef || v->ndval ) cursize += v->ndef*SIZE_STD+v->ndval*(SIZE_STD+SIZE_SUB)+SIZE_MIN;
 if ( v->pdef ) cursize += v->pdef*(SIZE_STD+SIZE_SUB)+SIZE_MIN;
 if ( cursize == start ) return;
 for ( p = v->parameter ; p ; p = p->next ) codeParameter(p);
 for ( p = v->qualifier ; p ; p = p->next ) codeParameter(p);
 keep = cursize;
 cursize = start;
 if ( v->nqual )
  {
   v->caddr = cursize;
   codeList(v->qualifier,CDU_ILLQUAL);
  }
 v->dqaddr = codeDefault(v->qualifier,DEFQ,pardef);
 if ( v->npar )
  {
   v->paddr = cursize;
   for ( p = v->parameter, ix = 0 ; p ; p = p->next )
    codeCmp(p->optval->caddr,SWTP,arg(p->laddr,ix++,0));
   codeErr(CDU_GARB);
  }
 if ( v->pdef )
  {
   v->dpaddr = cursize;
   for ( p = v->parameter ; p ; p = p->next ) 
    if ( (p->flags&idfDEF) == idfDEF )
     {
      codeCmp(cursize+SIZE_STD,DEFP,arg(p->laddr,CDU_START,CDU_DEF));
      codeSub(cursize+SIZE_SUB,p->optval->laddr);
     }
   codeGoto(state_exit,NOSTR,arg0);
  }
 cursize = keep;
}

/*
  Disallows rausschreiben. Die Referenzen werden dabei in die Adresse der
  Namen oder Labels in der Parsertabelle uebersetzt.
*/
static codeDisallows(e)
REGISTER struct Expression *e;
{
 for ( ; e ; e = e->next )
  {
   codeExpression(e);
   codeWord(0);
  }
 codeWord(0xff00);
}

/*
  Eine Ausdruck herausschreiben.
*/
static codeExpression(e)
REGISTER struct Expression *e;
{
 codeWord(e->kind);
 switch (e->kind)
  {
   case EK_ANY2 :
   case EK_LEAF :
   case EK_NEG  : codeReference(e->ereference);
		  if ( e->kind == EK_ANY2 ) codeWord(0);
   		  break;
   case EK_AND  :
   case EK_OR   : codeExpression(e->eleft);
   case EK_NOT  : 
   case EK_SUB  : codeExpression(e->eright);
   		  break;
  }
}

/*
  Referenzliste kodieren.
*/
static codeReference(r)
REGISTER struct Reference *r;
{
 REGISTER char **np;
 REGISTER int **ap;

 for ( ; r ; r = r->next )
  {
   for ( np = r->names, ap = r->addrs ; *np++ ; ap++ ) codeWord(**ap);
   codeWord(0);
  }
}

/*
  Erzeuge den Code fuer eine Liste von Qualifiern oder Schluesselworte.
  Die invertierten Namen muessen korrekt einsortiert werden.
*/
static codeList(p,err)
REGISTER struct Parameter *p;
int err;
{
 REGISTER struct Parameter *neg = p;
 REGISTER int syntax;

 for ( ; neg && ((neg->flags&idfNEG) != idfNEG) ; neg = neg->next ) ;
 for ( ; p ; p = p->next ) 
  {
   syntax = (p->syntax ? p->syntax->iaddr : 0);
   if ( neg ) neg = codeNegated(neg,p);
   if ( p->optval )
    codeKey(p->optval->caddr,p->naddr,arg(p->laddr,syntax,codePos(p,0)));
   else
    codeKey(state_exit,p->naddr,arg(p->laddr,syntax,codePos(p,CDU_END)));
  }
 if ( neg ) codeNegated(neg,p);
 codeErr(err);
}

/*
  Erzeuge Code, der Defaultwerte fuer Qualifier oder Schluesselworte
  einsetzt.
*/
codeDefault(p,act,pos)
REGISTER struct Parameter *p;
REGISTER char *act;
int pos;
{
 REGISTER int val;
 int start,done = 0;

 start = cursize;
 for ( ; p ; p = p->next ) 
  {
   if ( (p->flags&idfDEF) == idfDEF )
    val = CDU_DEF;
   else if ( (p->flags&idfBAT) == idfBAT )
    val = CDU_BAT;
   else
    continue;
   done = 1;
   if ( p->optval )
    {
     codeCmp(cursize+SIZE_STD+SIZE_SUB,act,arg(p->laddr,CDU_START,codePos(p,val)));
     codeSub(cursize+SIZE_SUB,p->optval->laddr);
    }
   else
    codeCmp(cursize+SIZE_STD,act,arg(p->laddr,CDU_NOSTART,codePos(p,val)));
  }
 if ( !done ) return 0;
 codeGoto(pos,NOSTR,arg0);
 return start;
}

/*
  Erzeuge Code fuer einen Parameter. Das einzige, was zu tun ist ist Code
  fuer einen etwaigen Wert zu erzeugen. Eine etwaige Syntaxaenderung wird
  zuvor bearbeitet.
*/
static codeParameter(p)
REGISTER struct Parameter *p;
{
 if ( p->syntax ) codeSyntax(p->syntax);
 if ( p->optval ) codeValue(p->optval,p);
}

/*
  Erzeuge Code fuer einen VALUE. Ein Wert fuer einen Parameter ist syntaktisch
  anders als fuer einen Qualifier oder ein Schluesselwort, daher werden die
  beiden Faelle trotz einiger weniger Gemeinsamkeiten getrennt behandelt.
*/
static codeValue(v,parent)
REGISTER struct Value *v;
REGISTER struct Parameter *parent;
{
 REGISTER int size,pos,list;
 int conc,noconc,endup = state_exit;
 char *act = ENDV;

 if ( v->caddr ) return;
 if ( v->type ) codeType(v->type);
 v->caddr = pos = cursize;
 list = ((v->flags&vcfLIS) == vcfLIS);
 if ( v->ispar )
  {
   conc = ((v->flags&vcfCON) == vcfCON);
   noconc = ((v->flags&vcfNCO) == vcfNCO);
   if ( list || conc ) act = ADDV;
   codeAnalyze(v,pos+SIZE_EXT+SIZE_STD+SIZE_SUB,act,CDU_ILLPAR);
   codeSub(cursize+SIZE_SUB,qualpos);
   codeChar('/',cursize-SIZE_SUB,NOSTR);
   if ( list || conc )
    {
     if ( !list || !noconc ) codeSep('+',pos);
     if ( list ) codeSep(',',pos);
     codeGoto(SIZE_STD,ENDV,arg0);
    }
   else
    codeGoto(SIZE_STD,NOSTR,arg0); 
   v->laddr = cursize;
   if ( (parent->flags&idfDEF) == idfDEF )
    if ( (v->flags&vcfDEF) == vcfDEF )
     codeGoto(state_exit,ENDV,arg(v->daddr,0,CDU_DEF));
    else if ( v->type )
     codeGoto(v->type->daddr,NOSTR,arg0);
    else
     codeGoto(state_exit,ENDV,arg0);
  }
 else
  {
   size = 2*SIZE_MIN;
   if ( (v->flags&vcfREQ) == vcfREQ ) size += SIZE_STD;
   if ( ((v->flags&vcfDEF) != vcfDEF) && v->type && ((parent->flags&idfDEF) == idfDEF) )
    size += SIZE_MIN;
   else
    size += SIZE_STD;
   codeChar('=',pos+size,NOSTR);
   codeChar(':',pos+size,NOSTR);
   if ( (v->flags&vcfREQ) == vcfREQ ) codeErr(CDU_NOVAL);
   v->laddr = cursize;
   if ( (v->flags&vcfDEF) == vcfDEF )
    codeGoto(state_exit,ENDV,arg(v->daddr,0,CDU_DEF));
   else if ( v->type && ((parent->flags&idfDEF) == idfDEF) )
    {
     endup = v->type->daddr;
     act = ADDV;
     codeGoto(v->type->daddr,NOSTR,arg0);
    }
   else
    codeGoto(state_exit,ENDV,arg0);
   if ( list ) codeChar('(',cursize+SIZE_MIN+SIZE_EXT+SIZE_STD,NOSTR);
   codeAnalyze(v,endup,act,CDU_ILLVAL);
   if ( !list ) return;
   pos = cursize;
   codeAnalyze(v,cursize+SIZE_EXT+SIZE_STD,ADDV,CDU_ILLVAL);
   codeSep(',',pos);
   codeChar(')',endup,act);   
   codeErr(CDU_NOSEP);
  }
}

/*
  Code zur Analyse eines Wertes in einer der eingebauten Repraesentationen
  oder als Schluesselwortliste.
*/
static codeAnalyze(v,target,routine,errcode)
REGISTER struct Value *v;
int target,errcode;
char *routine;
{
 if ( v->type )
  gc(SUB,0,target,v->type->caddr,routine,arg0);
 else
  gc(MODULE,0,target,v->predef,routine,arg(0,0,CDU_COPY));
 codeErr(errcode);
}

/*
  Code fuer einen TYPE erzeugen. Der Mechanismus ist sehr aehnlich zu
  den Qualifiern eines VERBs oder einer SYNTAX.
*/
static codeType(t)
REGISTER struct Type *t;
{
 REGISTER struct Parameter *p;
 int keep;

 if ( t->caddr ) return;
 t->caddr = cursize;
 cursize += (t->nkey+t->nneg)*SIZE_EXT+SIZE_STD;
 if ( t->ndef || t->ndval )
  cursize += t->ndef*SIZE_STD+t->ndval*(SIZE_STD+SIZE_SUB)+SIZE_MIN;
 for ( p = t->keyword ; p ; p = p->next ) codeParameter(p);
 keep = cursize;
 cursize = t->caddr;
 codeList(t->keyword,CDU_ILLKEY);
 t->daddr = codeDefault(t->keyword,DEFK,state_exit);
 cursize = keep;
}

/*
  Routine definieren.
*/
static addRoutine(name)
REGISTER char *name;
{
 REGISTER struct Routine *nr;

 if ( !(nr = MALLOC(struct Routine,1)) ) nomem();
 nr->next = rnames;
 nr->name = name;
 rnames = nr;
}

/*
  Parsertabelle rausschreiben.
*/
flushVerb()
{
 REGISTER int out,len;
 REGISTER char **curb;

 verbcnt++;
 if ( (tmax/2) > 0x7fff )
  {
   error("Tabelle ist groesser als %ld Worte\n",0x7fff);
   exit(1);
  }
 head.th_total = head.th_debug = sizeof(head)+sizeof(ctab)+tmax;
 bwrite(&head,sizeof(head));
 bwrite(ctab,sizeof(ctab));
 for ( out = tmax, curb = mem ; out > 0 ; out -= len )
  {
   if ( (len = out) > BLKSIZE ) len = BLKSIZE;
   bwrite(*curb++,len);
  }
}

/*
  Bytes hexadezimal in einen C-Quelltext schreiben.
*/
bwrite(pos,len)
REGISTER unsigned char *pos;
int len;
{
 REGISTER int rest = len;

 if ( !filled ) fprintf(obj,"static char TAB[] = \"");
 while ( rest-- )
  {
   if ( !(filled++%BYTESPERLINE) ) fprintf(obj,"\\\n");
   fprintf(obj,"\\%o",*pos++);
  }
}

/*
  Codegenerierung abschliessen, C-Datei mit Routinen erzeugen. Diese 
  Datei wird dann automatisch uebersetzt und geloescht.
*/
Generate(cfile,ofile)
char *cfile,*ofile;
{
 REGISTER struct Routine *rn;
 REGISTER char *lower,cvt;
 int state = 0,rcnt = 0;
#ifndef MCH_AMIGA 
 int child;
#endif

 if ( !module || nomodule ) module = defmodule;
 for ( lower = module ; cvt = *lower++ ; )
  if ( (cvt >= 'A') && (cvt <= 'Z') )
   lower[-1] += 'a'-'A';
 if ( filled )
  fprintf(obj,"\\0\\\n\";\n\n");
 else
  fprintf(obj,"static char TAB[] = \"<none>\";\n\n");
 if ( rn = rnames )
  {
   for ( rn = rnames ; rn ; rn = rn->next, rcnt++ )
    fprintf(obj,"extern int %s_();\n",rn->name);
   fprintf(obj,"\nstatic struct MAP\n\t{\n");
   fprintf(obj,"\t char *name;\n\t int (*entry)();\n\t} MAP[] =\n\t{\n");
   for ( rn = rnames ; rn ; )
    {
     fprintf(obj,"\t { \"%s\",%s_ }",rn->name,rn->name);
     if ( rn = rn->next ) fprintf(obj,",\n");
    }
   fprintf(obj,"\n\t};\n");
  }
 else
  fprintf(obj,"struct MAP { int noop; };\n"); 
 fprintf(obj,"\nstruct\n\t{\n\t int version;\n\t int verbs;");
 fprintf(obj,"\n\t char *table;\n\t int tsize;");
 fprintf(obj,"\n\t struct MAP *map;\n\t int msize;\n\t char *admin;\n\t} ");
 fprintf(obj,"%s_ = { 0x%04x,%d,TAB,sizeof(TAB)-2,",module,CDUVERSION,verbcnt);
 fprintf(obj,"%s,%d,0 };\n\n",rcnt ? "MAP" : "0",rcnt);
 fclose(obj);
 if ( !noobj )
  {
#ifdef MCH_AMIGA
   state = (fexecl("C:cc","cc","+B","+p",cfile,NOSTR) || wait());
#else
   if ( !(child = fork()) )
    {
     execl("/usr/bin/cc","cc","-O","-c",cfile,NOSTR);
     exit(1);
    } 
   wait(&state);
#endif
  }
 if ( !keep ) unlink(cfile);
 if ( state )
  {
   error("Tabelle '%s' kann nicht erzeugt werden",ofile);
   exit(1);
  }
}
