/*
	Options:

	-rN	Set internal resolution to N (should be FIRST arg if given)
	-afile	Read action file
	-dN	Turn on debugging flag N (N missing = all flags)
	-mXX	Read macro package file XX

	troffcvt - troff-to-XXX converter

	Reading the default action file is delayed as long as possible,
	because the -r option might be used to reset the resolution.  If
	-a is given to read another action file, or -m to read a macro
	package, the default action file is read first.  If -r is given
	*after* the action file has been read a panic occurs.

	02 Apr 92	Paul DuBois	dubois@primate.wisc.edu

	02 Apr 92 V1.00.  Created.
*/

# include	<stdio.h>
# include	<ctype.h>
# ifdef THINK_C
# include	<console.h>
# include	<time.h>
# else	/* !THINK_C */
# include	<sys/types.h>		/* for size_t */
# include	<sys/time.h>
# endif /* THINK_C */

# include	"etm.h"
# include	"memmgr.h"

# include	"troffcvt.h"


# ifndef	LIBDIR
# define	LIBDIR	"/usr/lib/troffcvt"
# endif
# ifndef	MACLIBDIR
# define	MACLIBDIR	"/usr/lib/tmac"
# endif

# define	defActionFile	"actions"


static void	ReadDefActionFile ();
static void	SetupDefaults ();
static void	WriteDefaults ();
static void	ReadFile ();

static char	*libDir = LIBDIR;

int	troff = 1;			/* 0 = nroff, !0 = troff */

Param	resolution = 432;		/* units/inch */

int	allowInput = 1;

/* input chars */
int	escChar = defEscChar;		/* escape char */
int	doEscapes = 1;			/* whether to process escape char */

int	ctrlChar = defCtrlChar;		/* request control character */
int	nbCtrlChar = defNbCtrlChar;	/* non-break control character */
int	curCtrlChar;			/* ctrl char from current request */
int	fieldDelimChar = '\0';		/* field delim char (null = none) */
int	fieldPadChar = '\0';		/* field padding char */
int	fieldDelimCount = 0;		/* fdc's seen on current line */
int	optHyphenChar;		/* optional hyphen char */

/* output chars */
int	tabChar = '\0';			/* tab char (null = motion) */
int	leaderChar = '\0';		/* leader char (null = motion) */


NameDef	*nameList = (NameDef *) NULL;	/* request, macro, string, register list */
SpChar	*spCharList = (SpChar *) NULL;	/* special characters */

static char	*actionFile = defActionFile;


int	needSpace = 0;
int	inContinuation = 0;

int	ifLevel = 0;
int	ifResult = 1;	/* set TRUE so any wayward .el will be skipped */

char	fontTab[maxFonts][maxFontNameLen];
char	curFont[maxFontNameLen];
char	prevFont[maxFontNameLen];

UChar	*itMacro = (UChar *) NULL;
Param	itCount = 0;
UChar	*endMacro = (UChar *) NULL;

Param	centerCount = 0;
Param	ulCount = 0;
Param	cUlCount = 0;

Param	fillMode = fill;
Param	adjMode = adjFull;
Param	adjust = 1;

Param	curSize;
Param	prevSize;
Param	curSpaceSize;

Param	curVSize;
Param	prevVSize;
Param	curLineSpacing;
Param	prevLineSpacing;

Param	curIndent;
Param	prevIndent;
Param	curTempIndent;
Param	curLineLen;
Param	prevLineLen;

Param	curPageLen;
Param	curPageNum;
Param	curOffset;
Param	prevOffset;

Param	curHyphenMode;

Param	pageNumChar;
Param	curTitleLen;
Param	prevTitleLen;

Param	curTabCount;
Param	tabPos[maxTabStops];
char	tabType[maxTabStops];



long	debug = 0;


int main (argc, argv)
int	argc;
char	**argv;
{
char	buf[bufSiz];
char	*p;
int	inc;

	ETMInit (NULL);

# ifdef	THINK_C
	argc = ccommand (&argv);
# endif /* THINK_C */

	InitTransliterate ();
	(void) AllowOutput (0);
	SetupDefaults ();

	/* process arguments */

	--argc;
	++argv;
	while (argc > 0 && argv[0][0] == '-' && argv[0][1] != '\0')
	{
		inc = 1;
		if (strncmp (argv[0], "-d", (size_t) 2) == 0)
		{
			if (argv[0][2] != '\0')	
				debug |= (1 << Atoi (&argv[0][2]));
			else
				debug = 0xffffffff;
			/* don't want output order mixed up by buffering */
			setbuf (stdout, (char *) NULL);
			setbuf (stderr, (char *) NULL);
		}
		else if (strncmp (argv[0], "-a", (size_t) 2) == 0)
		{
			if (argv[0][2] != '\0')
				p = &argv[0][2];
			else if (argc > 1)
			{
				p = argv[1];
				inc = 2;
			}
			else
				ETMPanic ("missing filename after -a");

			ReadDefActionFile ();
			if (!ReadActionFile ((char *) NULL, p))
			{
				if (!ReadActionFile (libDir, p))
					ETMPanic ("can't file action file <%s>",
								p);
			}
		}
		else if (strncmp (argv[0], "-m", (size_t) 2) == 0)
		{
			if (argv[0][2] == '\0')
				ETMPanic ("empty macro package specifier");
			ReadDefActionFile ();
			sprintf (buf, "tmac.%s", &argv[0][2]);
			if (PushFile (buf))
				ReadFile ();
			else
			{
				sprintf (buf, "%s/tmac.%s", MACLIBDIR, &argv[0][2]);
				if (PushFile (buf))
					ReadFile ();
				else
					ETMMsg ("cannot read %s", buf);
			}
		}
		else if (strncmp (argv[0], "-r", (size_t) 2) == 0)
		{
			if (actionFile == (char *) NULL)
				ETMPanic ("-r must precede -a and -m options");
			if (argv[0][2] != '\0')
				resolution = Atol (&argv[0][2]);
			else if (argc > 1)
			{
				resolution = Atol (argv[1]);
				inc = 2;
			}
			else
				ETMPanic ("missing resolution after -r");
			if (resolution < 1)
				ETMPanic ("nonsensical resolution: %ld",
								resolution);
		}
		else if (strcmp (argv[0], "-troff") == 0)
			troff = 1;
		else if (strcmp (argv[0], "-nroff") == 0)
			troff = 0;
		else
			ETMPanic ("Unknown option: %s", argv[0]);
		argc -= inc;
		argv += inc;
	}

	/*
		Read default action file if still necessary.
		Turn output back on and force out initial state.
	*/

	ReadDefActionFile ();
	(void) AllowOutput (1);
	WriteDefaults ();

	if (argc == 0)		/* stdin */
	{
		ETMMsg ("Reading from standard input...");
		CommentOut ("processing input (stdin)");
		if (!PushFile ((char *) NULL))
			ETMPanic ("Cannot read stdin");
		ReadFile ();
	}
	else while (argc > 0)
	{
	char	buf[bufSiz];

		/* allow -a file anywhere in argument list */
		if (strncmp (argv[0], "-a", (size_t) 2) == 0)
		{
			inc = 1;
			if (argv[0][2] != '\0')
				p = &argv[0][2];
			else if (argc > 1)
			{
				p = argv[1];
				inc = 2;
			}
			else
				ETMPanic ("missing filename after -a");

			if (!ReadActionFile ((char *) NULL, p))
			{
				if (!ReadActionFile (libDir, p))
					ETMPanic ("can't file action file <%s>",
								p);
			}
			argc -= inc;
			argv += inc;
			continue;
		}

		sprintf (buf, "processing input file %s", argv[0]);
		CommentOut (buf);
		if (strcmp (argv[0], "-") == 0)
		{
			if (!PushFile ((char *) NULL))
				ETMMsg ("Cannot read stdin");
			else
				ReadFile ();
		}
		else if (!PushFile (argv[0]))
			ETMMsg ("Cannot open: %s", argv[0]);
		else
			ReadFile ();
		--argc;
		++argv;
	}

	/* do .em here if defined (no arguments) */

	if (endMacro != (UChar *) NULL)
	{
		PushMacro (endMacro, 0, (char **) NULL);
		ReadFile ();
	}

	Break ();	/* just in case still in paragraph... */

	ETMEnd ();
	return (0);
}


static void ReadDefActionFile ()
{
	if (actionFile != (char *) NULL)
	{
		if (!ReadActionFile ((char *) NULL, actionFile))
		{
			if (!ReadActionFile (libDir, actionFile))
				ETMPanic ("can't file action file <%s>",
							actionFile);
		}
		/* reset these here - its known that resolution won't change */
		(void) NewRegister (".H", resolution, (Param) 0, "1", 1);
		(void) NewRegister (".V", resolution, (Param) 0, "1", 1);
		(void) NewRegister (".w", Units (1, 'n'), (Param) 0, "1", 1);

		InitEnvironments ();

		actionFile = (char *) NULL;
	}
}


/*
	Set up defaults.

	troff manual says 26/27i for initial default offset, but that
	was because of a 1/27i paper margin.  Use 1i.

	Values are set to those in the Ossanna manual.
*/

static void SetupDefaults ()
{
struct tm	*tm;
time_t	t;
int	i;

	/*
		General registers
	*/

	(void) time (&t);
	tm = localtime (&t);
	(void) NewRegister ("dw", (Param) tm->tm_wday+1, (Param) 0, "1", 0);
	(void) NewRegister ("dy", (Param) tm->tm_mday, (Param) 0, "1", 0);
	(void) NewRegister ("mo", (Param) tm->tm_mon+1, (Param) 0, "1", 0);
	(void) NewRegister ("yr", (Param) tm->tm_year, (Param) 0, "1", 0);

	(void) NewRegister ("%", (Param) 0, (Param) 0, "1", 0);
	(void) NewRegister (".b", (Param) 0, (Param) 0, "1", 0);
	(void) NewRegister ("c.", (Param) 0, (Param) 0, "1", 0);
	(void) NewRegister (".R", (Param) 10000, (Param) 0, "1", 0);

	/*
		Read-only registers.  Most are created with value zero,
		but are modified by SetXXX() functions in setget.c.
		Members of the first group below are actually supported,
		members of the second group are created just so that
		attempts to modify them can be noticed.
	*/

	(void) NewRegister ("$$", (Param) GetPid (), (Param) 0, "1", 1);
	(void) NewRegister (".$", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".H", resolution, (Param) 0, "1", 1);
	(void) NewRegister (".V", resolution, (Param) 0, "1", 1);
	(void) NewRegister (".c", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".f", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".i", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".j", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".l", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".L", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".o", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".p", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".s", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".u", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".v", (Param) 0, (Param) 0, "1", 1);

	(void) NewRegister (".A", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".T", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".a", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".d", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".h", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".k", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".n", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".P", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".t", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".w", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".x", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".y", (Param) 0, (Param) 0, "1", 1);
	(void) NewRegister (".z", (Param) 0, (Param) 0, "1", 1);

	/* position zero is reserved */

	curFont[0] = '\0';
	prevFont[0] = '\0';
	SetFontNameBySlot ((Param) 0, " ");
	SetFontNameBySlot ((Param) 1, "R");
	SetFontNameBySlot ((Param) 2, "I");
	SetFontNameBySlot ((Param) 3, "B");
	for (i = 4; i < maxFonts; i++)
		SetFontNameBySlot ((Param) i, "S");

	SetFont ("R");		/* do twice to get current and previous */
	SetFont ("R");		/* both set to same value */

	SetCFA ((Param) 0, (Param) 1, (Param) 1, (Param) adjFull);
	SetUnderline ((Param) 0, (Param) 0);

	curPageLen = Units (11, 'i');
	curSize = prevSize = Units (10, 'x');
	curSpaceSize = Units (12./36., 'm');
	curVSize = prevVSize = Units (12, 'p');
	curLineSpacing = prevLineSpacing = Units (1, 'v');
	curIndent = prevIndent = Units (0, 'i');
	curTempIndent = Units (0, 'i');
	curLineLen = prevLineLen = Units (6.5, 'i');
	curOffset = prevOffset = Units (0, 'i');
	curTitleLen = prevTitleLen = Units (6.5, 'i');

	SetTabStops ((Param) 0, tabPos, tabType);

	/*
		Need curPageNum so can refer to "page-number" in
		parse-absrel-num action
	*/

	curPageNum = GetRegisterValue ("%");

	pageNumChar = '%';

	curHyphenMode = 1;
	optHyphenChar = ToEsc ('%');
}


/*
	Since most of the SetXXX() functions only write out values
	when they *change* from the current, a trick is used to force
	out current values:  change the relevant parameter externally
	to the SetXXX() function, then pass the real value back in to
	both set it and write it out.  Also then sync "previous" value
	of revertibles to current value.

	Ugly tricks are also used to get the CFA, underlining and font
	values written and synced.
*/

static void WriteDefaults ()
{
char	buf[bufSiz];

	ControlOut ("begin-setup");

	sprintf (buf, "resolution %ld", resolution);
	ControlOut (buf);
	SetPageLength (++curPageLen - 1);

	SetSize (++curSize - 1);
	prevSize = curSize;
	SetSpaceSize (++curSpaceSize - 1);
	SetSpacing (++curVSize - 1);
	prevVSize = curVSize;
	SetLineSpacing (++curLineSpacing - 1);
	prevLineSpacing = curLineSpacing;
	SetOffset (++curOffset - 1);
	prevOffset = curOffset;
	SetLineLength (++curLineLen - 1);
	prevLineLen = curLineLen;
	SetIndent (++curIndent - 1);
	prevIndent = curIndent;
	SetTitleLength (++curTitleLen - 1);
	prevTitleLen = curTitleLen;
	SetHyphenation (++curHyphenMode - 1);

	/* more than a little ugly */

	(void) AllowOutput (0);
	SetCFA (centerCount, fillMode, (Param) !adjust, adjMode);
	(void) AllowOutput (1);
	SetCFA (centerCount, fillMode, (Param) !adjust, adjMode);

	(void) AllowOutput (0);
	SetUnderline (ulCount + 1, cUlCount);
	(void) AllowOutput (1);
	SetUnderline (ulCount - 1, cUlCount);

	(void) AllowOutput (0);
	(void) strcpy (buf, curFont);
	SetFont (" ");		/* real fonts will never be this */
	(void) AllowOutput (1);
	SetFont (buf);		/* will write out font */
	SetFont (buf);		/* won't write, but syncs prevFont */

	SetPageNumber (++curPageNum - 1);

	ControlOut ("end-setup");

	SetTempIndent (++curTempIndent - 1);

	SetTabStops (++curTabCount - 1, tabPos, tabType);
}


static void ReadFile ()
{
	while (ProcessLine ())
	{
		/* */
	}
}


int ProcessLine ()
{
int	c;

	if ((c = ChPeek ()) == EOF)
		return (0);
	if (c == ctrlChar || c == nbCtrlChar)
		ProcessRequest ();
	else if (c == ToEsc ('!'))	/* transparent mode */
	{
		/* this isn't really correct since */
		/* \n, \$, \*, \w are still processed */
		SkipToEol ();
	}
	else
		ProcessText ();
	return (1);
}


int ProcessActionList (ap)
Action	*ap;
{
UChar	*argv[maxActArgs];
int	stat;
int	i;

	while (ap != (Action *) NULL)
	{
		if (Bug (bugActionProcess))
			ETMMsg ("processing %s %d", ap->actName, ap->actArgc);
		for (i = 0; i < ap->actArgc; i++)
			argv[i] = UStrAlloc (InterpretActionArg (ap->actArgv[i]));
		stat = (*ap->actFunc) (ap->actArgc, argv);
		for (i = 0; i < ap->actArgc; i++)
			UFree (argv[i]);
		if (!stat)
			return (0);	/* action failed, bail out */
		ap = ap->actNext;
	}
	return (1);
}
