/*
	Process input text line.
*/

# include	<stdio.h>

# include	"etm.h"
# include	"memmgr.h"

# include	"troffcvt.h"


/*
	Structure for mapping simple escape sequences to their result.
	If result is NULL, then the sequence is recognized but ignored.
*/

typedef	struct EscMap	EscMap;

struct EscMap
{
	char	emChar;
	char	*emStr;
};



static void	Text2 ();
static void	CheckInputTrap ();
static void	EscapedText ();
static void	SkipDelimString ();


static EscMap	escMap[] =
{
	{ '\\',	"backslash" },
	{ '&',	"zero-space" },
	{ '^',	"twelfth-space" },
	{ '|',	"sixth-space" },
	{ '0',	"digit-space" },
	{ ' ',	"hard-space" },
	{ '-',	"minus" },
	{ '\'',	"acute-accent" },
	{ '`',	"grave-accent" },
	{ 'a',	"leader" },
	{ 't',	"tab" },
	{ '\0',	(char *) NULL }
};


/*
	This has to take some care not to overrun the end of the line...

	Process a single text line.  After it has been processed and flushed,
	determine whether to issue a break, whether to decrement counters
	(centered lines, underlined lines, etc.) and whether a space will be
	necessary in front of the next text line.  None of these things will
	be necessary if continuation is in effect (\c found in current line)
	since the next line should be considered logically part of this one.

	Actually, \c doesn't seem to have that effect for .ul/.cu or .it
	input line counting, only for .ce - so that's what's done below.
	I don't understand why, but that's what troff versions I've looked
	at do...
*/

void ProcessText ()
{
	inContinuation = 0;
	Text2 ();
	if (inContinuation)
	{
		SetUnderline (ulCount - 1, cUlCount - 1);
		CheckInputTrap ();
		return;
	}

	/* break if in nofill more or currently centering lines */

	if (fillMode == nofill || centerCount > 0)
		Break ();

	/*
		Need a space at beginning of next text line if in fill
		mode and mode isn't temporarily overridden (centering on).
	*/

	if (fillMode == fill && centerCount == 0)
		needSpace = 1;

	/*
		Turn centering off if it was on, but this was the last
		line to be centered.  Filling/adjustment setting in effect
		at time centering was turned on now go back into effect.
	*/

	SetCFA (centerCount - 1, fillMode, adjust, adjMode);
	SetUnderline (ulCount - 1, cUlCount - 1);

	CheckInputTrap ();
}


/*
	Do the following things, besides dumping out text.
	- If line begins with space, do a break.
	- If the line is *completely* blank (empty or all spaces), it's
	equivalent to .sp 1.
	- If a space is needed between this line and the previous one,
	put out a space before the first character from this line is
	written.
*/

static void Text2 ()
{
int	c, sCount, blank = 1;
SpChar	*sp;
char	buf[bufSiz];

	if (ChPeek () == ' ')	/* if line begins with space, do break */
		Break ();


	/*
		While reading, keep track of sequences of spaces.  If they
		occur at the end of the line, forget about them.  If the
		line is completely blank, that's equivalent to .sp 1.
	*/

	for (;;)
	{
		sCount = 0;
		while ((c = ChIn ()) == ' ')
			++sCount;
		if (Eol (c))
			break;
		while (sCount-- > 0)		/* dump spaces if any */
			TextChOut (' ');	/* occurred before c */

		blank = 0;			/* known not to be blank now */

		if (needSpace)
		{
			TextChOut (' ');
			needSpace = 0;
		}

		/*
			Check character to see if it's one of several special 
			characters.  Check *before* escape processing or
			\(xx special characters since these characters can
			themselves be \(xx or escaped characters.
		*/
		if (c == fieldDelimChar)
		{
			sprintf (buf, "%s-field",
				(fieldDelimCount % 2) == 0 ? "begin" : "end");
			SpecialTextOut (buf);
			++fieldDelimCount;
		}
		else if (c == fieldPadChar)	/* recognize, but only if */
		{				/* currently inside field */
			if (fieldDelimCount % 2)
				SpecialTextOut ("field-pad");
		}
		else if (c == optHyphenChar)
			SpecialTextOut ("opt-hyphen");
		else if (Special (c))
		{
			if ((sp = LookupSpCharByCode (c)) != (SpChar *) NULL)
			{
				if (sp->spValue[0] == '@')
					SpecialTextOut (sp->spValue + 1);
				else
					TextStrOut (sp->spValue);
			}
		}
		else if (Esc (c))
			EscapedText (FromEsc (c));
		else if (c == '`')
		{
			if (ChPeek () == '`')
			{
				(void) ChIn ();
				SpecialTextOut ("ldblquote");
			}
			else
				SpecialTextOut ("lsglquote");
		}
		else if (c == '\'')
		{
			if (ChPeek () == '\'')
			{
				(void) ChIn ();
				SpecialTextOut ("rdblquote");
			}
			else
				SpecialTextOut ("rsglquote");
		}
		else	/* just plain text */
			TextChOut (c);
	}

	if (blank)		/* was blank line; acts like .sp 1 */
	{
		sprintf (buf, "%csp 1\n", ctrlChar);
		PushAnonString (buf);
	}
	else
		FlushOText ();
}


/*
	If input line trap should be sprung, push it onto the
	input stack and remove the trap.
*/

static void CheckInputTrap ()
{
	if (itMacro != (UChar *) NULL && --itCount <= 0)
	{
		PushMacro (itMacro, 0, (char **) NULL);
		UFree (itMacro);
		itMacro = (UChar *) NULL;
	}
}


static void EscapedText (ec)
int	ec;
{
EscMap	*em;
UChar	*p;
char	*cp;
Param	val;
int	delim, unit, c, repChar;
char	buf[bufSiz];

	/* see if this is a simple single-char escape */

	for (em = escMap; em->emChar != '\0'; em++)
	{
		if (em->emChar == ec)
		{
			/*if (em->emStr != (char *) NULL)*/
			SpecialTextOut (em->emStr);
			return;
		}
	}

	/*
		Anything left requires extra stuff following the escape
		character.  Exceptions are \n, \$, \*, which should all have
		been recognized in ChIn() and had input switched accordingly
		(thus making the escape sequence disappear).  \w will have
		been recognized when not in copy mode, and Text() isn't called
		in copy mode, so the \w should not be seen here.
	*/

	switch (ec)
	{
	default:
		TextChOut (ec);
		break;
	case 'n':
	case '$':
	case '*':
	case '}':
	case '(':
	case 'w':
	case '{':
		ETMMsg ("EscapedText: logic error, wayward <\\%c>", ec);
		break;
	case '.':
		TextChOut ('.');
		break;
	case 'b':				/* \b'stuff' - just skip it */
	case 'o':				/* \o'stuff' - just skip it */
		if (Eol (delim = ChIn ()))	/* malformed */
		{
			UnChIn (delim);
			break;
		}
		ControlOut (ec == 'b' ? "begin-bracket" : "begin-overstrike");
		while ((c = ChIn ()) != delim)
		{
			if (Eol (c))
			{
				UnChIn (c);
				break;
			}
			cp = CharToStr (c);
			if (*cp == '@')
				SpecialTextOut (cp+1);
			else
				TextStrOut (cp);
		}
		ControlOut (ec == 'b' ? "end-bracket" : "end-overstrike");
		break;
	case 'c':
		/*
			\c is recognized anywhere in line, although troff
			manual only indicates what to do when it's at the end
			(sec. 4.2).  Recognizing anywhere matches behavior
			observed in the troff's to which I have access.
		*/
		inContinuation = 1;
		break;
	case 'd':
		sprintf (buf, "motion %ld v", Units (.5, 'm'));
		ControlOut (buf);
		break;
	case 'r':
		sprintf (buf, "motion %ld v", Units (-1, 'm'));
		ControlOut (buf);
		break;
	case 'u':
		sprintf (buf, "motion %ld v", Units (-.5, 'm'));
		ControlOut (buf);
		break;
	case 'e':
		TextChOut (escChar);	/* print current escape char */
		break;
	case 'f':
		if ((p = ParseNameRef ()) != (UChar *) NULL)
		{
			SetFont (p);
			UFree (p);
		}
		break;
	case 'h':				/* \h'expr' */
	case 'v':				/* \v'expr' */
	case 'x':				/* \x'expr' */
		if (Eol (delim = ChIn ()))	/* malformed */
		{
			UnChIn (delim);
			break;
		}
		unit = (ec == 'h' ? 'm' : 'v');
		if (!ParseNumber (unit, 0, (Param) 0, &val))
			val = 0;
		while ((c = ChIn ()) != delim)
		{
			if (Eol (c))
			{
				UnChIn (c);
				break;
			}
		}
		switch (ec)
		{
		case 'h':
		case 'v':
			sprintf (buf, "motion %ld %c", val, ec);
			break;
		case 'x':
			sprintf (buf, "extra-space %ld", val);
			break;
		}
		ControlOut (buf);
		break;
	case 'k':		/* gobble mark char, ignore sequence */
		if (!Eol (ChPeek ()))
			(void) ChIn ();
		break;
	case 'l':
	case 'L':
		/*
			Get delim, parse expr, parse possible \&c, get
			delim, and write \line.
		*/
		if (Eol (delim = ChIn ()))	/* malformed */
		{
			UnChIn (delim);
			break;
		}
		unit = (ec == 'l' ? 'm' : 'v');
		repChar = '\0';
		if (!ParseNumber (unit, 0, (Param) 0, &val))
			val = 0;
		if (ChPeek () == Esc ('&'))
			(void) ChIn ();
		if (!Eol (c = ChPeek ()) && c != delim)
			repChar = ChIn ();
		while ((c = ChIn ()) != delim)
		{
			if (Eol (c))
			{
				UnChIn (c);
				break;
			}
		}
		/* note that the repChar isn't written... */
		c = (ec == 'l' ? 'h' : 'v');
		if (repChar == '\0')
			sprintf (buf, "line %ld %c", val, c);
		else
			sprintf (buf, "line %ld %c %s", val, c,
							CharToStr (repChar));
		ControlOut (buf);
		break;
	case 'p':
		ControlOut ("break-spread");
		break;
	case 's':
		/* 0, n, +/-n, or nothing */
		SetSize (ParseSizeRef ());
		break;
	case 'z':
		if (!Eol (ChPeek ()))
		{
			sprintf (buf, "zero-width %s", CharToStr (ChIn ()));
			ControlOut (buf);
		}
		break;
	}
}


/* NEVER USED? */

static void SkipDelimString ()
{
int	delim, c;

	if (Eol (delim = ChIn ()))	/* malformed */
		UnChIn (delim);
	else while ((c = ChIn ()) != delim)
	{
		if (Eol (c))		/* malformed */
		{
			UnChIn (c);
			break;
		}
	}
}


/*
	Convert a character code to a printable string.  Escaped
	characters are converted to unescaped characters, if they're
	not known in the escape map (which is possibly wrong in some cases).
	Specials are converted to @-form.

	Fails for things like space, \e, quotes, some others.
	Oh, well.
*/

char *CharToStr (c)
int	c;
{
SpChar	*sp;
EscMap	*em;
static char	buf[bufSiz];

	if (Esc (c))
	{
		c = FromEsc (c);
		/* see if this is a simple single-char escape */

		for (em = escMap; em->emChar != '\0'; em++)
		{
			if (em->emChar == c)
			{
				/*if (em->emStr != (char *) NULL)*/
				sprintf (buf, "@%s", em->emStr);
				break;
			}
		}
		if (em->emChar == '\0')	/* not in map */
		{
			buf[0] = c;
			buf[1] = '\0';
		}
	}
	else if (!Special (c))
	{
		buf[0] = c;
		buf[1] = '\0';
	}
	else if ((sp = LookupSpCharByCode (c)) != (SpChar *) NULL)
		(void) strcpy (buf, sp->spValue);
	else
		ETMPanic ("CharToStr: woof");
	return (buf);
}
