/*
	There needs to be a kludge option to turn off some stuff, like
	WordPerfect's failure to understand \sl.

	tc2rtf - read troffcvt output, convert to RTF

	Paragraph format values when \end-setup is seen are saved and
	made the default style.  During further processing, paragraph
	format changes are made to an internally cached state.  When it
	comes time to write the first char of paragraph text, and differences
	between the internal state and last written state are flushed out
	to sync the internal and written states.

	(This implicitly means that para format changes WITHIN a paragraph
	don't take effect until the next paragraph).

	Some readers are dumb and don't understand \lquote, \lsglquote,
	etc., so the hex equivalents for the appropriate character set
	are written instead.

	20 Apr 92	Paul DuBois	dubois@primate.wisc.edu

	29 Apr 92 V1.00.  Created.
*/

# ifdef	THINK_C
# include	<console.h>
# endif /* THINK_C */
# include	<stdio.h>
# include	<ctype.h>

# include	"etm.h"
# include	"memmgr.h"

# include	"tcr.internal.h"
# include	"tcr.h"

# include	"tc2rtf.h"


# ifndef	LIBDIR
# define	LIBDIR	"/usr/lib/troffcvt"
# endif

# define	twips	1440

# define	Twips(n)	((long) (((double) (n) * twips) / (resolution)))


static void	ReadFile ();
static void	ControlLine ();
static void	Prolog ();
static void	WriteParFmt ();
static void	SetFont ();
static void	SetPointSize ();
static void	SetTempIndent ();
static void	SetMargins ();
static void	SetSpacing ();
static void	PlainText ();
static void	SpecialText ();
static void	Para ();
static void	DrawLine ();
static void	Motion ();
static void	ResetTabs ();
static void	SetTabStop ();
static long	StrToTwips ();
static long	StrToHalfPoints ();
static void	ControlOut ();
static void	TextStrOut ();
static void	TextChOut ();
static void	StrOut ();
static void	ChOut ();
static int	AllowOutput ();


static char	*libDir = LIBDIR;

static int	oAllow = 1;	/* zero if output is throttled */


static int	pCount = 0;	/* text chars written to current paragraph */
static int	lCount = 0;	/* text chars written to current output line */

int	charSet = macCharSet;	/* default character set number */
char	*charSetName = "mac";	/* default character set name */


static long	resolution = 1440;

/* troff values */
static long	offset = 1 * twips;
static long	indent = 0;
/*static long	tempIndent = 0;*/
static long	lineLen = 6.5 * twips;
static long	spacing = 12 * 2;	/* half-points */
static long	lineSpacing = 1;
static long    spaceSize = 12;	/* 12/36 em */

/* document format properties */

static long	margl = 0;	/* left/right margins fixed in Prolog() */
static long	margr = 0;
static long	pageLen = 11 * twips;
static long	pageWid = 8.5 * twips;

/* character format properties */
/*static long	fontNum = 1;*/
static long	pointSize = 10 * 2;	/* size is in half-points */

static TCRFont	*curFInfo = (TCRFont *) NULL;
static char	curFont[30] = "R";
static long	charUpDn = 0;	/* super/subscript value, in half points */



FTabInfo	*fTabList = (FTabInfo *) NULL;
SpChar	*spList = (SpChar *) NULL;


/*
	Four paragraph format states are maintained.

	rtfParFmt	The standard RTF defaults.  Reflects the para
			state after \pard is done.  Never modified.
	intParFmt	Internal para state.  Para state changes are cached
			between paras and flushed when the first text character
			for a para needs to be written.
	wrtParFmt	Para state as last written to the output.
*/

static ParFmt	rtfParFmt =
{
	(long)	tcrAdjLeft,		/* justification */
	(long)	0,			/* left indent */
	(long)	0,			/* right indent */
	(long)	0,			/* first indent */
	(long)	0,			/* space before paragraph */
	(long)	0,			/* space after paragraph */
	(long)	1000,			/* space between lines (automatic) */
	(int)	0,			/* number of tabstops */
	{ (long) 0 },			/* tabstop positions */
	{ (char) 'l' }			/* tabstop types */
};


static ParFmt	intParFmt;
static ParFmt	wrtParFmt;


/*
	Convenience pointers into state structures
*/

ParFmt	*rpf = &rtfParFmt;
ParFmt	*ipf = &intParFmt;
ParFmt	*wpf = &wrtParFmt;

int	debug = 0;
int	echo = 0;


int main (argc, argv)
int	argc;
char	**argv;
{
char	buf[bufSiz];
int	inc;

	ETMInit (NULL);

# ifdef	THINK_C
	argc = ccommand (&argv);
# endif /* THINK_C */

	--argc;
	++argv;
	while (argc > 0 && argv[0][0] == '-')
	{
		inc = 1;
		if (strcmp (argv[0], "-d") == 0)
			debug = 0xffff;
		else if (strcmp (argv[0], "-e") == 0)
			echo = 1;
		else if (strcmp (argv[0], "-cs") == 0)
		{
			if (argc < 2)
				ETMPanic ("Missing argument after -cs");
			inc = 2;
			if (strcmp (argv[1], "ansi") == 0)
			{
				charSetName = &argv[1][1];
				charSet = ansiCharSet;
			}
			else if (strcmp (argv[1], "mac") == 0)
			{
				charSetName = &argv[1][1];
				charSet = macCharSet;
			}
			else if (strcmp (argv[1], "pc") == 0)
			{
				charSetName = &argv[1][1];
				charSet = pcCharSet;
			}
			else if (strcmp (argv[1], "pca") == 0)
			{
				charSetName = &argv[1][1];
				charSet = pcaCharSet;
			}
			else
				ETMPanic ("Unknown character set: %s", argv[1]);
		}
		else
			ETMPanic ("Unknown option: %s", argv[0]);
		argc -= inc;
		argv += inc;
	}

	if (!TCRReadFonts ((char *) NULL, "tcr-fonts"))
	{
		if (!TCRReadFonts (libDir, "tcr-fonts"))
			ETMPanic ("cannot find tcr-fonts file");
	}
	if (!ReadFontInfo ((char *) NULL, "rtf-fonts"))
	{
		if (!ReadFontInfo (libDir, "rtf-fonts"))
			ETMPanic ("cannot find rtf-fonts file");
	}
	sprintf (buf, "rtf-spec-%s", charSetName);
	if (!ReadSpecials ((char *) NULL, buf))
	{
		if (!ReadSpecials (libDir, buf))
			ETMPanic ("cannot find %s file", buf);
	}

	(void) AllowOutput (0);

	*wpf = *ipf = *rpf;	/* sync all states to RTF defaults */

	TCRInit ();

	if (argc == 0)		/* stdin */
		ReadFile ();
	else while (argc > 0)
	{
		if (freopen (argv[0], "r", stdin) == (FILE *) NULL)
			ETMMsg ("Cannot open: %s", argv[0]);
		else
			ReadFile ();
		--argc;
		++argv;
	}

	StrOut ("}\n");

	ETMEnd ();
	exit (0);
	/*NOTREACHED*/
}


static void ReadFile ()
{
int	i;

	while (TCRGetToken () != tcrEOF)
	{
		if (echo)
		{
			ETMMsg ("class %d maj %d min %d <%s>",
				tcrClass, tcrMajor, tcrMinor, tcrArgv[0]);
			if (tcrClass == tcrControl || tcrClass == tcrSText)
			{
				for (i = 1; i < tcrArgc; i++)
					ETMMsg ("\t<%s>", tcrArgv[i]);
			}
		}
		switch (tcrClass)
		{
		case tcrControl:	ControlLine (); break;
		case tcrText:		PlainText (); break;
		case tcrSText:		SpecialText (); break;
		default:	ETMPanic ("ReadFile: unknown token %d <%s>",
							tcrClass, tcrArgv[0]);
		}
	}
}



static void ControlLine ()
{
char	*argv1 = tcrArgv[1];
char	buf[bufSiz];

	switch (tcrMajor)
	{
	default:
		ETMMsg ("ControlLine: bad control major code: %d <%s>",
						tcrMajor, tcrArgv[0]);
		break;
	case tcrCUnknown:
		ETMMsg ("ControlLine: unknown control token: <%s>",
						tcrArgv[0]);
		break;
	case tcrComment:
		break;
	case tcrBeginSetup:
		break;
	case tcrResolution:
		resolution = TCRStrToNum (argv1);
		break;
	case tcrEndSetup:
		(void) AllowOutput (1);
		Prolog ();
		break;
	case tcrBreak:
		Para ();
		break;
	case tcrCFA:
		ipf->pJustify = tcrMinor;
		break;
	case tcrPageLength:
		break;
	case tcrPageNumber:
		break;
	case tcrFont:
		if (tcrArgc < 2)
			ETMMsg ("missing font on \\font line");
		else
			SetFont (tcrArgv[1]);
		break;
	case tcrPointSize:
		SetPointSize (StrToHalfPoints (argv1));
		break;
	case tcrSpacing:
		SetSpacing (StrToTwips (argv1), lineSpacing);
		break;
	case tcrLineSpacing:
		SetSpacing (spacing, TCRStrToNum (argv1));
		break;
	case tcrOffset:
		SetMargins (StrToTwips (argv1), indent, lineLen);
		break;
	case tcrIndent:
		SetMargins (offset, StrToTwips (argv1), lineLen);
		break;
	case tcrTempIndent:
		SetTempIndent (StrToTwips (argv1));
		break;
	case tcrLineLength:
		SetMargins (offset, indent, StrToTwips (argv1));
		break;
	case tcrTitleLength:
		break;
	case tcrBeginTitle:
		break;
	case tcrEndTitle:
		break;
	case tcrSpace:
		/*
			Begin new group and space to avoid changing params.
			Negative param means exact spacing (positive means
			"at least" spacing.
		*/
		if (pCount > 0)
			Para ();
		sprintf (buf, "{\\sl%ld\\sa0\\sb0\\par}\n", - StrToTwips (argv1));
		StrOut (buf);
		break;
	case tcrUnderline:
		break;
	case tcrCUnderline:
		break;
	case tcrNoUnderline:
		break;
	case tcrULineFont:
		break;
	case tcrBeginBracket:
		TextStrOut ("<BRACKET<");	/* ugly but makes it visible */
		break;
	case tcrEndBracket:
		TextStrOut (">BRACKET>");
		break;
	case tcrBreakSpread:
		break;
	case tcrExtraSpace:
		break;
	case tcrLine:
		DrawLine (StrToTwips (argv1), tcrArgv[2][0]);
		break;
	case tcrMark:
		break;
	case tcrMotion:
		Motion (StrToTwips (argv1), tcrArgv[2][0]);
		break;
	case tcrBeginOverstrike:
		TextStrOut ("<OVERSTRIKE<");	/* ugly but makes it visible */
		break;
	case tcrEndOverstrike:
		TextStrOut (">OVERSTRIKE>");
		break;
	case tcrBeginPage:
		Para ();			/* necessary? */
		ControlOut ("page\n");
		break;
	case tcrZeroWidth:
		break;
	case tcrSpaceSize:
		spaceSize = TCRStrToNum (argv1);
		break;
	case tcrConstantWidth:
		break;
	case tcrNeed:
		break;
	case tcrEmbolden:
		break;
	case tcrSEmbolden:
		break;
	case tcrResetTabs:
		ResetTabs ();
		break;
	case tcrFirstTab:
		ResetTabs ();
		/* fall through... */
	case tcrNextTab:
		SetTabStop (StrToTwips (tcrArgv[1]), tcrArgv[2][0]);
		break;
	case tcrHyphenate:
		break;
	case tcrBegDiversion:
		break;
	case tcrAppDiversion:
		break;
	case tcrEndDiversion:
		break;
	case tcrTabChar:
		break;
	case tcrLeaderChar:
		break;
	}
}


/*
	The troffcvt setup section has now been seen, can write RTF prolog.

	margl and margr become the left and right margin values.  Those
	can only be set once (since they're document properties).  They
	become the zero-indent values.  Changes to offset, indent or
	line length afterwards map onto changes in \li or \ri (which
	are paragraph properties).
*/

static void Prolog ()
{
FTabInfo	*fp;
char	buf[bufSiz];

	margl = offset;
	margr = pageWid - offset - lineLen;
	SetMargins (offset, indent, lineLen);

	sprintf (buf, "{\\rtf1\\%s\\deff1\n", charSetName);
	StrOut (buf);
	StrOut ("{\\fonttbl\n");
	for (fp = fTabList; fp != (FTabInfo *) NULL; fp = fp->nextFTabInfo)
	{
		sprintf (buf, "{\\f%d\\fnil %s;}\n",
				fp->tcrInfo->tcrFontNum,
				fp->rtfName);
		StrOut (buf);
	}
	StrOut ("}\n");

	sprintf (buf, "\\paperh%ld \\paperw%ld\n", pageLen, pageWid);
	StrOut (buf);
	/*StrOut ("\\sectd\\sbknone\\linemod0\\linex0\\cols1\\endnhere\n");*/
	sprintf (buf, "margl%ld ", margl);
	ControlOut (buf);
	sprintf (buf, "margr%ld ", margr);
	ControlOut (buf);
	ControlOut ("pard ");
	/*WriteParFmt (rpf, ipf);*/
	*wpf = *rpf;

	/* flush out initial character format values */

	SetFont (curFont);
	SetPointSize (pointSize);
}


/*
	Draw a line (horizontal only).  The implicit assumption is that
	- is as wide as a space (all guesses are arbitrary since RTF has
	no font metrics).
*/

static void DrawLine (length, direction)
long	length;
int	direction;
{
long	ssTwips;		/* current space size in twips */

	if (direction != 'h')	/* ignore vertical lines */
		return;
	ssTwips = (spaceSize * (pointSize * 10)) / 36;
	while (length > 0)
	{
		TextStrOut ("-");
		length -= ssTwips;
	}
}


/*
	Perform motion.  For horizontal motion, the implicit assumption
	is that \~ and backspace are as wide as a space (all guesses are
	arbitrary since RTF has no font metrics).

	Vertical motion is in half-points.  If there is any outstanding undone
	vertical motion, it must be undone first to balance; some RTF readers
	gag otherwise.
*/

static void Motion (motion, direction)
long	motion;
int	direction;
{
char	buf[bufSiz];
long	ssTwips;	/* current space size in twips */

	if (direction == 'h')
	{
		ssTwips = (spaceSize * (pointSize * 10)) / 36;
		if (motion > 0)
		{
			while (motion > 0)
			{
				ControlOut ("~");
				motion -= ssTwips;
			}
		}
		else
		{
			while (motion < 0)
			{
				TextStrOut ("\b");
				motion += ssTwips;
			}
		}
	}
	else if (direction == 'v')
	{
		/* convert value to half-points (20 twips/point)*/
		motion /= 10;

		/* check for previous undone motion */

		if (charUpDn != 0)
		{
			if (charUpDn < 0)
				ControlOut ("up0 ");
			else
				ControlOut ("dn0 ");
			/*
				Factor out just-undone motion from request.
				If this leaves motion=0, nothing more need
				be done.
			*/

			motion += charUpDn;
			charUpDn = 0;
		}
		if (motion == 0)
			return;
		else if (motion < 0)
			sprintf (buf, "up%ld ", -motion);
		else
			sprintf (buf, "dn%ld ", motion);
		ControlOut (buf);
		charUpDn = motion;
	}
}


static void ResetTabs ()
{
	ipf->pTabCount = 0;
}


static void SetTabStop (pos, type)
long	pos;
char	type;
{
	if (ipf->pTabCount < maxTabStops)
	{
		/* convert position to be relative to left indent */
		pos += (offset + indent) - margl;
		ipf->pTabPos[ipf->pTabCount] = pos;
		ipf->pTabType[ipf->pTabCount] = type;
		++ipf->pTabCount;
	}
}


/*
	Map troff combination of offset, indent and linelength onto the
	RTF equivalents of left and right indent (which are themselves
	relative to the left and right margin).
*/

static void SetMargins (nOffset, nIndent, nLineLen)
long	nOffset, nIndent, nLineLen;
{
	offset = nOffset;
	indent = nIndent;
	lineLen = nLineLen;
	ipf->pLIndent = (offset + indent) - margl;
	ipf->pRIndent = (pageWid - offset - lineLen) - margr;
}


/*
	Map troff temporary indent onto RTF first-line indent.  Also,
	whereas troffcvt \temp-indent is an absolute value, RTF \fi
	is relative to \li.
*/

static void SetTempIndent (val)
long	val;
{
	ipf->pFIndent = val - indent;	/* ??? correct ??? */
}


/*
	Set line spacing, which is a combination of the \spacing and
	\line-spacing values.  This translates into RTF space-between
	and space-after.
*/

static void SetSpacing (nSpacing, nLineSpacing)
long	nSpacing, nLineSpacing;
{
	spacing = nSpacing;
	lineSpacing = nLineSpacing;
	ipf->pSpaceBetween = ipf->pSpaceAfter = spacing * lineSpacing;
}


/*
	This generates more output than it needs to, but it
	basically works, so too bad.
*/

static void SetFont (name)
char	*name;
{
TCRFont	*fp;
char	buf[bufSiz];

	if ((fp = TCRLookupFont (name)) == (TCRFont *) NULL)
	{
		ETMMsg ("no info for font <%s>", name);
		ControlOut ("f1\n");	/* default font */
		return;
	}
	sprintf (buf, "\\f%d\\i0\\b0\n", fp->tcrFontNum);
	StrOut (buf);
	if (fp->tcrAtts & tcrItal)
		ControlOut ("i ");
	if (fp->tcrAtts & tcrBold)
		ControlOut ("b ");
	curFInfo = fp;
	(void) strcpy (curFont, name);
}


/*
	Size is maintained in half-points.
*/

static void SetPointSize (size)
long	size;
{
char	buf[bufSiz];

	pointSize = size;
	sprintf (buf, "fs%ld ", pointSize);
	ControlOut (buf);
}


static void PlainText ()
{
	if (tcrMajor != '\n')
	{
		if (tcrMajor == '{' || tcrMajor == '}')
			TextChOut ('\\');
		TextChOut (tcrMajor);
	}
}


static void SpecialText ()
{
SpChar	*sp;
char	buf[bufSiz];
int	setFont = 0;
char	font[bufSiz];
TCRFont	*fntInfo;

	if ((sp = LookupSpecial (&tcrArgv[0][1])) == (SpChar *) NULL
		|| sp->spRtf == (char *) NULL)
	{
		sprintf (buf, "[[%s]]", &tcrArgv[0][1]);
		TextStrOut (buf);
	}
	else
	{
		setFont = 0;
		if (sp->spFont != NULL && strcmp (sp->spFont, curFont) != 0)
			++setFont;

		if (setFont)
		{
			StrOut ("{");
			/* save info so can restore manually */
			(void) strcpy (font, curFont);
			fntInfo = curFInfo;
			SetFont (sp->spFont);
		}
		TextStrOut (sp->spRtf);
		if (setFont)
		{
			StrOut ("}");
			(void) strcpy (curFont, font);
			curFInfo = fntInfo;
		}
	}
}


static void Para ()
{
	ControlOut ("par\n");
	SetTempIndent (indent);
	pCount = 0;
	lCount = 0;
}


/*
	Write out paragraph format properties that differ from some
	baseline.  bpf = base paragraph format, npf = new paragraph
	format.
*/

static void WriteParFmt (bpf, npf)
ParFmt	*bpf;
ParFmt	*npf;
{
char	buf[bufSiz];
char	*p;
int	c, i;

	if (bpf->pJustify != npf->pJustify)
	{
		p = (char *) NULL;

		switch (npf->pJustify)
		{
		default:
			ETMPanic ("WriteParFmt: bad justify code %d",
							npf->pJustify);
		case tcrCenter:
		case tcrAdjCenter:
			p = "qc ";
			break;
		case tcrNofill:
		case tcrAdjLeft:
			p = "ql ";
			break;
		case tcrAdjFull:
			p = "qj ";
			break;
		case tcrAdjRight:
			p = "qr ";
			break;
		}
		ControlOut (p);
	}
	if (bpf->pLIndent != npf->pLIndent)
	{
		sprintf (buf, "li%ld ", npf->pLIndent);
		ControlOut (buf);
	}
	if (bpf->pRIndent != npf->pRIndent)
	{
		sprintf (buf, "ri%ld ", npf->pRIndent);
		ControlOut (buf);
	}
	if (bpf->pFIndent != npf->pFIndent)
	{
		sprintf (buf, "fi%ld ", npf->pFIndent);
		ControlOut (buf);
	}
	if (bpf->pSpaceBetween != npf->pSpaceBetween)
	{
		sprintf (buf, "sl%ld ", npf->pSpaceBetween);
		ControlOut (buf);
	}
/*
	Output looks better without this...

	if (bpf->pSpaceAfter != npf->pSpaceAfter)
	{
		sprintf (buf, "sa%ld ", npf->pSpaceAfter);
		ControlOut (buf);
	}
*/

	/* if there are no explicit tabs, this writes nothing */
	if (bpf->pTabCount != npf->pTabCount)
	{
		if (npf->pTabCount > 0)
			ChOut ('\n');
		for (i = 0; i < npf->pTabCount; i++)
		{
			sprintf (buf, "tx%ld ", npf->pTabPos[i]);
			ControlOut (buf);
			if ((c = npf->pTabType[i]) == 'l')	/* default */
				continue;
			sprintf (buf, "tq%c ", c == 'c' ? 'c' : 'r');
			ControlOut (buf);
		}
		if (npf->pTabCount > 0)
			ChOut ('\n');
	}
}


/*
	basic output routines

	ControlOut() - write control word.
	TextStrOut() - write out a string of paragraph text.
	TextChOut() - write out a character of paragraph text.
	StrOut() - write out a string.
	ChOut() - write out a character.  ALL output (except ETM messages)
		comes through this routine.
	AllowOutput () - allow or throttle output.
*/


static void ControlOut (p)
char	*p;
{
	if (p != (char *) NULL)
	{
		if (lCount > 0)
		{
			ChOut ('\n');
			lCount = 0;
		}
		ChOut ('\\');
		StrOut (p);
	}
}



static void TextStrOut (p)
char	*p;
{
	if (p != (char *) NULL)
	{
		while (*p != '\0')
			TextChOut (*p++);
	}
}


/*
	Write a character of paragraph text.  If this is the first character
	and the internally maintained paragraph format state is different
	than the last one written out, put them in sync:  write \pard to
	reset to RTF defaults.  If the internal state is then different
	from that, write output to put it into effect.
*/
	
static void TextChOut (c)
int	c;
{
	if (pCount == 0 && !ParFmtEq (wpf, ipf))
	{
		ControlOut ("pard ");
		*wpf = *rpf;
		if (!ParFmtEq (wpf, ipf))
		{
			WriteParFmt (wpf, ipf);
			*wpf = *ipf;
		}
	}
	ChOut (c);
	++pCount;
	/* break lines every now and then */
	if (++lCount > 50 && tcrMajor == ' ')
	{
		ChOut ('\n');
		lCount = 0;
	}
}



static void StrOut (p)
char	*p;
{
	if (p != (char *) NULL)
	{
		while (*p != '\0')
			ChOut (*p++);
	}
}


static void ChOut (c)
char	c;
{
	if (oAllow && putc (c, stdout) == EOF)
		ETMPanic ("Write error, cannot continue");
}


static int AllowOutput (yesno)
int	yesno;
{
int	prev = oAllow;

	oAllow = yesno;
	return (prev);
}


/*
	Convert a string representing basic units to twips
*/

static long StrToTwips (s)
char	*s;
{
	return (Twips (TCRStrToNum (s)));
}


/*
	Convert a string representing points to half-points.
*/

static long StrToHalfPoints (s)
char	*s;
{
	return (TCRStrToNum (s) * 2);
}
