/*
	Request action processors.

	The Axxx() routines all take the liberty of assuming that they're
	called with the correct number of arguments (as defined in action.c),
	although they check when need be to see whether arguments are empty
	or numeric.

	BUG: Escaped characters can get into titles.  What then.
*/

# include	<stdio.h>

# include	"etm.h"
# include	"memmgr.h"

# include	"troffcvt.h"


# define	maxDiversions	10


static void	AddMacChar ();
static void	ScanMacroBody ();
static void	PushDiversion ();
static void	PopDiversion ();
static void	DumpChar ();


/* not UChar; diversion names must be plain characters */

static char	*diStack[maxDiversions];
static int	nDiversions = 0;


/* ------------------------------------------------------------- */

/*
	Font and character size control
*/

int APointSize (argc, argv)
int	argc;
UChar	**argv;
{
	SetSize (InterpretParam (argv[0], prevSize));
	return (1);
}



/*
	Set space size to N/36 em.  Note that what is written out
	is simply N; since the current space size is expressed in
	terms of ems (and hence depends on current point size), any
	postprocessor needs to tie actual space size to instantaneous
	value of 1m.
*/

int ASpaceSize (argc, argv)
int	argc;
UChar	**argv;
{
Param	val;

	if (EmptyArg (argv[0]))		/* ignore if no argument */
		return (0);
	val = InterpretParam (argv[0], (Param) 0);
	if (val < 0)
		val = 0;
	/*val = Units (val, 'm') / 36;*/
	SetSpaceSize (val);
	return (1);
}



/*
	This only writes out the font that should be constant-spaced.
*/

int AConstantWidth (argc, argv)
int	argc;
UChar	**argv;
{
char	buf[bufSiz];
char	*font;

	if ((font = GetFontName (argv[0])) == (char *) NULL)
		return (0);
	/* missing arguments are non-numeric, so this test catches those */
	if (!NumericArg (argv[1]))
		sprintf (buf, "noconstant-width %s", font);
	else
		sprintf (buf, "constant-width %s", font);
	ControlOut (buf);
	return (1);
}


/*
	If argv[0] is empty, it's ".bd F N", otherwise ".bd S F N".
	The contents of argv[0] are otherwise ignored.

	If N is missing, turn emboldening off.
*/

int AEmbolden (argc, argv)
int	argc;
UChar	**argv;
{
char	buf[bufSiz];
char	*font, *special = "";
Param	smear;

	if ((font = GetFontName (argv[1])) == (char *) NULL)
		return (0);
	if (EmptyArg (argv[2]))
		smear = 0;
	smear = InterpretParam (argv[2], (Param) 0);
	if (!EmptyArg (argv[0]))
		special= "-special";
	sprintf (buf, "embolden%s %s %ld", special, font, smear);
	ControlOut (buf);
	SetRegisterValue (".b", smear);
	return (1);
}


int AFont (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))
		SetFont (prevFont);
	else
		SetFont ((char *) argv[0]);
	return (1);
}


int AFontPosition (argc, argv)
int	argc;
UChar	**argv;
{
	if (!NumericArg (argv[0]) || EmptyArg (argv[1]))
		return (0);
	SetFontNameBySlot ((Param) Atol (argv[0]), argv[1]);
	return (1);
}


/* ------------------------------------------------------------- */

/*
	Page control
*/


int APageLength (argc, argv)
int	argc;
UChar	**argv;
{
	SetPageLength (InterpretParam (argv[0], Units (11, 'i')));
	return (1);
}


/*
	begin-page and page-number are broken, because relative changes
	aren't properly done - there isn't any way to know the current
	page number!

	(A lame excuse; this should write out information the postprocessor
	could use to determine that the change is relative.)
*/

int ABeginPage (argc, argv)
int	argc;
UChar	**argv;
{
char	buf[bufSiz];

	if (!NumericArg (argv[0]))
		ControlOut ("begin-page");
	else
	{
		SetPageNumber ((Param) Atol (argv[0]));
		sprintf (buf, "begin-page %s", argv[0]);
		ControlOut (buf);
	}
	return (1);
}


int APageNumber (argc, argv)
int	argc;
UChar	**argv;
{
	if (NumericArg (argv[0]))
		SetPageNumber ((Param) Atol (argv[0]));
	return (1);
}


int AOffset (argc, argv)
int	argc;
UChar	**argv;
{
	SetOffset (InterpretParam (argv[0], prevOffset));
	return (1);
}


int ANeed (argc, argv)
int	argc;
UChar	**argv;
{
Param	val;
char	buf[bufSiz];

	val = InterpretParam (argv[0], Units (1, 'v'));
	sprintf (buf, "need %ld", val);
	ControlOut (buf);
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Text filling, adjusting and centering
*/

/*
	Issue a break -- maybe.  This checks whether a break should be
	issued or not by checking whether the request was given with the
	normal or no-break control character.

	"break" should be specified in the action list for any request
	that normally causes a break.
*/

int ABreak (argc, argv)
int	argc;
UChar	**argv;
{
	if (curCtrlChar == ctrlChar)
		Break ();
	return (1);
}


/*
	Turn filling on or off
*/

int AFill (argc, argv)
int	argc;
UChar	**argv;
{
	SetCFA (centerCount, (Param) fill, adjust, adjMode);
	return (1);
}


int ANofill (argc, argv)
int	argc;
UChar	**argv;
{
	SetCFA (centerCount, (Param) nofill, adjust, adjMode);
	return (1);
}



/*
	Set adjustment type.  If no type is given, use whatever the
	value was before.  Thus, ".ad" after ".na" will restore adjustment
	and set it to what it was before being suppressed with ".na".

	Use of the .j register value as an argument is supported, but
	its correct operation depends on adj{Center,Left,Right,Full}
	values being single-digit and non-negative, since the .ad request
	argument is probably parsed with the "parse-char" action.
*/

int AAdjust (argc, argv)
int	argc;
UChar	**argv;
{
Param	mode = adjMode, adj = 1;

	if (Digit (argv[0][0]))	/* probably had ".ad \n(.j" */
	{
		adj = 0;
		if ((mode = (Param) (argv[0][0] - '0')) > 4)
		{
			adj = 1;
			mode -= 4;
		}
	}
	else switch (argv[0][0])
	{
	case 'c': mode = adjCenter; break;
	case 'l': mode = adjLeft; break;
	case 'r': mode = adjRight; break;
	case 'b':
	case 'n': mode = adjFull; break;
	}
	SetCFA (centerCount, fillMode, adj, mode);
	return (1);
}


int ANoAdjust (argc, argv)
int	argc;
UChar	**argv;
{
	SetCFA (centerCount, fillMode, (Param) 0, adjMode);
	return (1);
}


/*
	Somewhat similar to .ad c but causes a break, takes a line count,
	suppresses filling and adjusting, and turns off after a certain
	number of input lines.

	If the count is missing, use 1.  If the count is 0 (or negative!)
	turn off centering, if it isn't already off.  If count is positive,
	turn centering on, if it isn't already on.
*/

int ACenter (argc, argv)
int	argc;
UChar	**argv;
{
Param	count;

	if (!NumericArg (argv[0]))
		count = 1;
	else
		count = Atol (argv[0]);
	if (count < 0)
		count = 0;
	SetCFA (count, fillMode, adjust, adjMode);
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Vertical spacing
*/


int ASpacing (argc, argv)
int	argc;
UChar	**argv;
{
	SetSpacing (InterpretParam (argv[0], prevVSize));
	return (1);
}


/*
	Fractional values (e.g., .ls 1.5) are processed as just that,
	even though troff truncates to integer values.
*/

int ALineSpacing (argc, argv)
int	argc;
UChar	**argv;
{
	SetLineSpacing (InterpretParam (argv[0], prevLineSpacing));
	return (1);
}


int ASpace (argc, argv)
int	argc;
UChar	**argv;
{
Param	val;
char	buf[bufSiz];

	val = InterpretParam (argv[0], Units (1, 'v'));
	sprintf (buf, "space %ld", val);
	ControlOut (buf);
	return (1);
}


/*
	Argument is 'y if no-space mode should be turned on, 'n' if it
	should be turned off.

	Actually: not implemented.
*/

int ANoSpace (argc, argv)
int	argc;
UChar	**argv;
{
	return (0);
}


/* ------------------------------------------------------------- */


/*
	Line length and indenting
*/


int ALineLength (argc, argv)
int	argc;
UChar	**argv;
{
	SetLineLength (InterpretParam (argv[0], prevLineLen));
	return (1);
}


int AIndent (argc, argv)
int	argc;
UChar	**argv;
{
	SetIndent (InterpretParam (argv[0], prevIndent));
	return (1);
}


int ATempIndent (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))		/* ignore if no argument */	
		return (0);
	SetTempIndent (InterpretParam (argv[0], (Param) 0));
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Macro and string definitions
*/


/*
	Define macro.  Currently just skip to the end of the definition
	and ignore it.  Put a \n back on the input in case definition
	is empty, otherwise we'll miss the end marker.
*/

int ADefineMacro (argc, argv)
int	argc;
UChar	**argv;
{
Macro	*mp;

	if (EmptyArg (argv[0]))		/* no macro name */
		return (0);

	if (Bug (bugMacroDef))
		ETMMsg ("New macro %s", UStrToStr (argv[0]));
	mp = NewMacro (argv[0]);
	mp->macBuf = Alloc (512);	/* may need more later... */
	mp->macMaxSiz = 512;
	mp->macSiz = 0;
	ScanMacroBody (mp, (char *) argv[1]);

	return (1);
}


/*
	Append to a macro (create it if it doesn't exist).
*/

int AAppendMacro (argc, argv)
int	argc;
UChar	**argv;
{
Macro	*mp;

	if (EmptyArg (argv[0]))		/* no macro name */
		return (0);
	if ((mp = LookupMacro (argv[0])) == (Macro *) NULL)
		return (ADefineMacro (argc, argv));
	ScanMacroBody (mp, (char *) argv[1]);
	return (1);
}


/*
	Scan current macro body.  This is also used by .ig, in which
	case mp is NULL.  endMarker is non-empty if an explicit endMarker
	was given, and is taken to be the name of a macro to invoke
	after collecting the macro body.

	Note the ambiguity of ".de XX ." as opposed to ".de XX"; in both
	cases the ending line is "..", but in the first case the macro "."
	should be invoked after collecting the macro.  (Wonder if that
	was Ossanna's intent?)
*/

static void ScanMacroBody (mp, endMarker)
Macro	*mp;
char	*endMarker;
{
char	*endMacro = (char *) NULL, *ep, *p;
int	c, cPrev;

	if (EmptyArg ((UChar *) endMarker))		/* no explicit end marker */
		endMarker = ".";
	else
		endMacro = endMarker;
	if (Bug (bugMacroDef))
	{
		p = StrAlloc (UStrToStr (mp->macParent->ndName));
		ETMMsg ("ScanMacroBody(%s): until <%s>",
				p, UStrToStr ((UChar *) endMarker));
		Free (p);
	}
	cPrev = '\n';
	if (mp != (Macro *) NULL)
		p = mp->macBuf;
	CopyMode (1);
	for (;;)
	{
		if ((c = ChIn ()) == EOF)
			ETMPanic ("EOF while scanning macro body");
		AddMacChar (mp, c);
		if (c == ctrlChar && Eol (cPrev))
		{
			/*if (Bug (bugMacroDef))
				ETMMsg ("SMB: look for end marker");*/
			if (mp != (Macro *) NULL)
				p = &mp->macBuf[mp->macSiz-1];
			for (ep = endMarker; *ep != '\0'; ep++)
			{
				if (Bug (bugMacroDef))
					ETMMsg ("SMB: look for %c", *ep);
				if ((c = ChIn ()) == EOF)
					ETMPanic ("EOF while scanning macro body");
				AddMacChar (mp, c);
				if (*ep != c)
					break;
			}
			if (*ep == '\0' && Eol (ChPeek ()))	/* found match */
			{
				if (mp != (Macro *) NULL)
					mp->macSiz = (p - mp->macBuf);
				break;
			}
		}
		cPrev = c;
	}
	CopyMode (0);
	SkipToEol ();
	if (Bug (bugMacroDef))
	{
		p = StrAlloc (UStrToStr (mp->macParent->ndName));
		ETMMsg ("ScanMacroBody(%s) : found <%s>",
				p, UStrToStr ((UChar *) endMarker));
		Free (p);
	}

	if (endMacro != (char *) NULL)
		PushMacro (endMacro, 0, (char **) NULL);
}


/*
	Add a character to a macro body.  If it won't fit, reallocate
	to get more space.
*/

static void AddMacChar (mp, c)
Macro	*mp;
int	c;
{
	if (mp != (Macro *) NULL)
	{
		if (mp->macSiz >= mp->macMaxSiz)
		{
			mp->macBuf = Realloc (mp->macBuf, mp->macMaxSiz+512);
			mp->macMaxSiz += 512;
		}
		mp->macBuf[mp->macSiz++] = c;
	}
}


/*
	Define a string.  The first and second arguments are the name and value
	of the string.
*/

int ADefineString (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))
		return (0);
	(void) NewString (argv[0], argv[1]);
	return (1);
}


/*
	Append to an existing string.
*/

int AAppendString (argc, argv)
int	argc;
UChar	**argv;
{
String	*sp;
UChar	*p;

	if ((sp = LookupString (argv[0])) == (String *) NULL)
		return (ADefineString (argc, argv));
	p = (UChar *) Alloc (strlen (sp->strValue) + strlen ((char *) argv[1]) + 1);
	(void) strcpy (p, sp->strValue);
	(void) strcat (p, argv[1]);
	Free (sp->strValue);
	sp->strValue = (char *) p;
	return (1);
}


/*
	Rename request, macro or string name
*/

int ARenameName (argc, argv)
int	argc;
UChar	**argv;
{
NameDef	*np;

	if (EmptyArg (argv[0]) || EmptyArg (argv[1]))
		return (0);
	if ((np = LookupName (argv[0], reqDef|macDef|strDef))
					== (NameDef *) NULL)
		return (0);
	RenameName (np, argv[1]);
	return (1);
}


int AEndMacro (argc, argv)
int	argc;
UChar	**argv;
{
	UFree (endMacro);
	endMacro = UStrAlloc (argv[0]);
	return (1);
}


int AInputTrap (argc, argv)
int	argc;
UChar	**argv;
{
	UFree (itMacro);
	itCount = 0;
	if (NumericArg (argv[0]) && !EmptyArg (argv[1]))
	{
		itMacro = UStrAlloc (argv[1]);
		itCount = InterpretParam (argv[0], (Param) 0);
	}
	return (1);
}


int ADiversion (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))		/* terminate current diversion */
		PopDiversion ();
	else				/* begin new diversion */
		PushDiversion (argv[0], 0);
	return (1);
}


int AAppendDiversion (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))		/* terminate current diversion */
		PopDiversion ();
	else				/* append to existing diversion */
		PushDiversion (argv[0], 1);
	return (1);
}


char *CurrentDiversion ()
{
	if (nDiversions < 1)
		return ("");
	return (diStack[nDiversions-1]);
}


/*
	Copy diversion name into stack and indicate start (or continuation)
	of diversion.
*/

static void PushDiversion (name, append)
UChar	*name;
int	append;
{
char	buf[bufSiz];

	if (!PlainStr (name))
	{
		ETMMsg ("disallowed diversion name <%s>", UStrToStr (name));
		return;
	}
	if (nDiversions >= maxDiversions)
		ETMPanic ("PushDiversion: diversion stack overflow");
	diStack[nDiversions++] = StrAlloc ((char *) name);
	sprintf (buf, "%s-diversion %s", append ? "append" : "begin", name);
	ControlOut (buf);
}


/*
	Indicate end of diversion, free storage for name.
*/

static void PopDiversion ()
{
char	buf[bufSiz];

	if (nDiversions > 0)
	{
		sprintf (buf, "end-diversion %s", diStack[--nDiversions]);
		ControlOut (buf);
		Free (diStack[nDiversions]);
	}
}


/* ------------------------------------------------------------- */


/*
	Define a number register.  If the register existed before,
	use previous increment and format if corresponding arguments
	are empty.  (Actually, old format is ignored, since formats
	are pretty well broken anyway.)
*/

int ADefineRegister (argc, argv)
int	argc;
UChar	**argv;
{
Register	*rp;
Param	value = 0, oldValue = 0, oldIncr = 0;
Param	increment = 0;

	if (EmptyArg (argv[0]))
		return (0);
	if ((rp = LookupRegister (argv[0])) != (Register *) NULL)
	{
		oldValue = rp->regValue;
		oldIncr = rp->regIncrement;
		if (rp->regReadOnly)
		{
			ETMMsg ("attempt to set readonly register <%s> ignored",
							UStrToStr (argv[0]));
			return (0);
		}
	}
	value = InterpretParam (argv[1], oldValue);
	increment = InterpretParam (argv[2], oldIncr);
	rp = NewRegister (argv[0], value, increment, "1", 0);
	return (1);
}


int ARegisterFormat (argc, argv)
int	argc;
UChar	**argv;
{
Register	*rp;

	if (EmptyArg (argv[1]))
		return (0);
	if ((rp = LookupRegister (argv[0])) == (Register *) NULL)
		return (0);
	SetRegisterFormat (rp, argv[1]);
	return (1);
}


/* ------------------------------------------------------------- */


int ASetTabChar (argc, argv)
int	argc;
UChar	**argv;
{
	tabChar = InterpretParamChar (argv[0], '\0');
	DumpChar ("tab-char", tabChar);
	return (1);
}


int ASetLeaderChar (argc, argv)
int	argc;
UChar	**argv;
{
	leaderChar = InterpretParamChar (argv[0], '\0');
	DumpChar ("leader-char", leaderChar);
	return (1);
}


static void DumpChar (name, c)
char	*name;
int	c;
{
char	buf[bufSiz];

	if (c == '\0')
		ControlOut (name);
	else if (c == '"' || c == '\\')
	{
		sprintf (buf, "%s \"\\%c\"", name, c);
		ControlOut (buf);
	}
	else
	{
		sprintf (buf, "%s %s", name, CharToStr (c));
		ControlOut (buf);
	}
}


int ASetFieldChars (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]) && EmptyArg (argv[1]))
	{
		fieldDelimChar = '\0';	/* turn field mechanism off */
		fieldPadChar = '\0';
	}
	else
	{
		fieldDelimChar = InterpretParamChar (argv[0], '\0');
		fieldPadChar = InterpretParamChar (argv[1], ' ');
	}
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Escape character cannot be an escaped character \x or special
	character \(xx since then it would be recursive.
*/

int ASetEscape (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))
		escChar = defEscChar;
	else if (PlainStr (argv[0]))
		escChar = argv[0][0];
	else
		ETMMsg ("illegal escape character: <%s>", UStrToStr (argv[0]));
	doEscapes = 1;
	return (1);
}


int ANoEscape (argc, argv)
int	argc;
UChar	**argv;
{
	doEscapes = 0;
	return (1);
}


/* ------------------------------------------------------------- */


int AUnderline (argc, argv)
int	argc;
UChar	**argv;
{
Param	count;

	count = InterpretParam (argv[0], (Param) 1);
	if (count < 0)
		count = 0;

	SetUnderline (count, 0);

	return (1);
}


int ACUnderline (argc, argv)
int	argc;
UChar	**argv;
{
Param	count;

	count = InterpretParam (argv[0], (Param) 1);
	if (count < 0)
		count = 0;

	SetUnderline (0, count);

	return (1);
}


int AUnderlineFont (argc, argv)
int	argc;
UChar	**argv;
{
char	buf[bufSiz];
char	*font;

	if ((font = GetFontName (argv[0])) == (char *) NULL)
		return (0);
	sprintf (buf, "underline-font %s", font);
	ControlOut (buf);
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Control character operations
*/


int ASetControl (argc, argv)
int	argc;
UChar	**argv;
{
	ctrlChar = InterpretParamChar (argv[0], defCtrlChar);
	return (1);
}


int ASetControl2 (argc, argv)
int	argc;
UChar	**argv;
{
	ctrlChar = InterpretParamChar (argv[0], defNbCtrlChar);
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Transliteration
*/



int ATransliterate (argc, argv)
int	argc;
UChar	**argv;
{
UChar	*p = (UChar *) argv[0];
int	c1, c2;

	while ((c1 = *p) != '\0')
	{
		if (*++p != '\0')
			c2 = *p++;
		else
			c2 = ' ';
		Transliterate (c1, c2);
	}

	return (1);
}

/* ------------------------------------------------------------- */

/*
	Hyphenation
*/


/*
	Turn hyphenation on or off.  0 = off, empty defaults to 1.
	Values are as described in troff manual.
*/

int AHyphenation (argc, argv)
int	argc;
UChar	**argv;
{
	SetHyphenation (InterpretParam (argv[0], (Param) 1));
	return (1);
}



/*
	Set the optional-hyphenation character.
*/

int ASetHyphenChar (argc, argv)
int	argc;
UChar	**argv;
{
	optHyphenChar = InterpretParamChar (argv[0], ToEsc ('%'));
	return (1);
}


/* ------------------------------------------------------------- */

/*
	Titles
*/


/*
	Three part titles aren't implemented.
*/

int ATitle (argc, argv)
int	argc;
UChar	**argv;
{
	return (0);
}


/*
	Set, or remove, page number character
*/

int APageNumChar (argc, argv)
int	argc;
UChar	**argv;
{
	pageNumChar = InterpretParamChar (argv[0], '\0');
	return (1);
}


/*
	Set title length
*/

int ATitleLength (argc, argv)
int	argc;
UChar	**argv;
{
	SetTitleLength (InterpretParam (argv[0], prevTitleLen));
	return (1);
}


/* ------------------------------------------------------------- */

/*
	Environments
*/

int AEnvironment (argc, argv)
int	argc;
UChar	**argv;
{
	if (!NumericArg (argv[0]))	/* revert to previous */
		PopEnvironment ();
	else				/* enter new environment */
		PushEnvironment ((int) Atoi (argv[0]));
	return (1);
}


/* ------------------------------------------------------------- */



int AExit (argc, argv)
int	argc;
UChar	**argv;
{
	allowInput = 0;
	return (1);
}



int AAbort (argc, argv)
int	argc;
UChar	**argv;
{
UChar	*p = (UChar *) "User abort";

	if (!EmptyArg (argv[0]))
		p = argv[0];
	ETMPanic ("%s", UStrToStr (p));
	return (1);
}



int APushFile (argc, argv)
int	argc;
UChar	**argv;
{
	/* on error, say something but continue */
	if (EmptyArg (argv[0]))
		ETMMsg ("Missing filename for push-file");
	else if (!PlainStr (argv[0]))
		ETMMsg ("Disallowed filename for push-file <%s>",
						UStrToStr (argv[0]));
	else if (!PushFile (argv[0]))
		ETMMsg ("cannot open \"%s\" for push-file", argv[0]);
	else
		return (1);
	return (0);
}


int ASwitchFile (argc, argv)
int	argc;
UChar	**argv;
{
	/* on error, say something and exit */
	if (EmptyArg (argv[0]))
		ETMPanic ("Missing filename for switch-file");
	if (!PlainStr (argv[0]))
		ETMMsg ("Disallowed filename for switch-file <%s>",
						UStrToStr (argv[0]));
	else if (!SwitchFile (argv[0]))
		ETMPanic ("cannot open \"%s\" for switch-file", argv[0]);
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Read rest of request line and echo to stderr.  Used for .tm,
	but can be useful for debugging argument parsing, too.
*/

int AEcho (argc, argv)
int	argc;
UChar	**argv;
{
	ETMMsg ("%s", UStrToStr (argv[0]));
	return (1);
}


/*
	Ignore input for a while
*/

int AIgnore (argc, argv)
int	argc;
UChar	**argv;
{
	ScanMacroBody ((Macro *) NULL, (char *) argv[0]);
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Miscellaneous actions that don't really correspond to
	any particular request.
*/

/*
	Define a special character.  This isn't a troff request, but
	it allows the list of known special characters to be constructed
	at run time, to allow for the differences in troff versions.
	First arg is the special char name, second is the string to
	output when character is referenced.

	If the character is already defined, just replace the value.
	Otherwise add it to the list.
*/

int ASpecialChar (argc, argv)
int	argc;
UChar	**argv;
{
	if (EmptyArg (argv[0]))
		return (0);
	NewSpChar (argv[0], argv[1]);
	return (1);
}



/*
	Push a string onto the input stack to interpolate it into
	the input stream.
*/

int APushStr (argc, argv)
int	argc;
UChar	**argv;
{
	PushAnonString (argv[0]);
	return (1);
}


/*
	Skip to the end of the current line
*/

int AEol (argc, argv)
int	argc;
UChar	**argv;
{
	SkipToEol ();
	return (1);
}


/*
	AWrite{Control,Text,Special} should not contain escaped or
	special characters...
*/

int AWriteControl (argc, argv)
int	argc;
UChar	**argv;
{
	ControlOut ((char *) argv[0]);
	return (1);
}


int AWriteText (argc, argv)
int	argc;
UChar	**argv;
{
UChar	*p = argv[0];

	while (*p != '\0');
		TextChOut (*p);
	return (1);
}


int AWriteSpecial (argc, argv)
int	argc;
UChar	**argv;
{
	SpecialTextOut (argv[0]);
	return (1);
}


int ADebugFlag (argc, argv)
int	argc;
UChar	**argv;
{
Param	val;

	/* missing arguments are non-numeric, so this test catches those */
	if (!NumericArg (argv[0]))
		debug = 0xffffffff;
	else if ((val = InterpretParam (argv[0], (Param) 0)) < 0)
		debug = 0;
	else
		debug |= 1L<<val;
	ETMMsg ("debugging flags now %#lx", debug);
	return (1);
}


int ADumpMacro (argc, argv)
int	argc;
UChar	**argv;
{
Macro	*mp;
char	*p;

	if (EmptyArg (argv[0]))
		return (0);
	if ((mp = LookupMacro (argv[0])) == (Macro *) NULL)
		ETMMsg ("Macro <%s>: not found", UStrToStr (argv[0]));
	else
	{
		ETMMsg ("Macro <%s> body:", UStrToStr (argv[0]));
		if ((p = mp->macBuf) != (char *) NULL)
			fwrite (p, 1, mp->macSiz, stderr);
		ETMMsg ("-------");
	}
	return (1);
}


/* ------------------------------------------------------------- */


/*
	Look at a string, return its value.  String should be empty or
	represent a number.  If empty, return default value.  If represents
	a number, look at last char to see if it's a units indicator.  If
	it is, use it to convert number to basic units.
*/

Param InterpretParam (s, defValue)
UChar	*s;
Param	defValue;
{
int	unit;
int	len;

	/* missing arguments are non-numeric, so this test catches those */
	if (!NumericArg (s))
		return (defValue);
	if ((len = strlen ((char *) s)) > 0 && UnitChar (unit = s[len-1]))
		s[len-1] = '\0';
	else
		unit = 'u';
	return (Units (Atof (s), unit));
}


int InterpretParamChar (s, defValue)
UChar	*s;
int	defValue;
{
	if (EmptyArg (s))
		return (defValue);
	if (s[0] == 0x80)		/* two-byte special character */
		return (scMask | s[1]);
	return (s[0]);
}
