/*
	Should arg parsers take argument indicating whether not to fail
	if no argument present?  Or should they ALWAYS return 1, even if
	the thing they're looking for isn't present or is malformed?

	parse.c

	Routines to parse request name from request line, several
	different styles of request-argument parsing, and escape
	sequence parsing.

	Argument parsers should skip whitespace before trying to parse
	the argument(s).

	The action list for a request should specify the "eol" action
	after all arguments have been parsed, in order to skip to the
	end of the input line.

	All argument parsing routines leave their result rargc/rargv.
	However, the argument initialization stuff is done in
	ParseRequestName(), since that must always be called before
	any argument parsing is done anyway.  Better to call it once
	than have to remember it in each parsing routine.

	Escape sequence parsers are called after the escape and the
	character following have been read.

	All parsers must be careful not to run past the end of the input
	line, and have to push back the linefeed if they read it.
*/

# include	<stdio.h>

# include	"etm.h"
# include	"memmgr.h"

# include	"troffcvt.h"

# define	maxReqArgs	9

# define	opEq		1
# define	opNe		2
# define	opLess		3
# define	opLessEq	4
# define	opGreater	5
# define	opGreaterEq	6
# define	opPlus		7
# define	opMinus		8
# define	opMultiply	9
# define	opDivide	10
# define	opMod		11
# define	opAnd		12
# define	opOr		13


typedef	struct ParamMap	ParamMap;

struct ParamMap
{
	char	*pName;
	Param	*pAddr;
};


static int	ParseRequestName ();
static void	InitArgs ();
static void	AddArgChar ();
static void	EndArg ();
static void	ParseNumberArg ();
static Param	ParseExpr ();
static Param	ParseExprTerm ();
static int	LooksLikeExpr ();
static int	ParseCondition ();
static int	ParseStrCondition ();
static void	ProcessCondition ();
static void	SkipCondition ();



/*
	Request name, arguments and argument count.  The arguments are
	pointers into a single buffer and should NOT be written into by
	anything but InitArg()/AddArgChar()/EndArg().

	InitArg() is called by ParseRequestName() and the other two should be
	called only b argument parsing routines.
*/

static UChar	req[5];
static int	rargc = 0;
static UChar	*rargv[maxReqArgs];
static int	raover = 0;



static UChar	rargBuf[bufSiz];
static UChar	*rpCur;
static UChar	*rpStart;


/*
	These are the parameters which can be set relatively, and for
	which there must be a way to refer to them symbolically.
*/

ParamMap	paramMap [] =
{
	"point-size",		&curSize,
	"spacing",		&curVSize,
	"line-spacing",		&curLineSpacing,
	"indent",		&curIndent,
	"line-length",		&curLineLen,
	"page-length",		&curPageLen,
	"title-length",		&curTitleLen,
	"offset",		&curOffset,
	"page-number",		&curPageNum,
	(char *) NULL,		(Param *) NULL
};



/*
	Initialize rargc to zero and set up pointers for stuffing
	arguments in the argument buffer.
*/

static void InitArgs ()
{
	rargc = 0;
	rpCur = rpStart = rargBuf;
	raover = 0;
}


/*
	Add a character to the end of the current argument.  This
	call does nothing if the maximum number of arguments has already
	been parsed, or if there isn't enough room to add the character
	and a trailing null byte.

	Special character codes are unpacked into a 0x80 byte + sc index
	byte.
*/

static void AddArgChar (c)
int	c;
{
int	need = 1;

	if (Special (c))
		need = 2;
	if (rargc < maxReqArgs
		&& rpCur + need < rargBuf + (int) sizeof (rargBuf))
	{
		if (Special (c))
		{
			*rpCur++ = 0x80;
			*rpCur++ = (c & 0xff);
		}
		else
			*rpCur++ = c;
	}
	else if (raover++ == 0)
		ETMMsg ("request <%s>, arg buffer overflow, excess ignored",
							UStrToStr (req));
}


/*
	Call this function when the end of an argument has been seen.  It
	bumps the arg count rargc and initializes the pointer to it in the
	rargv array.
*/

static void EndArg ()
{
	if (!raover)
	{
		*rpCur++ = '\0';
		rargv[rargc++] = rpStart;
		rpStart = rpCur;
	}
}


/*
	Read request line and parse request name and arguments
*/

void ProcessRequest ()
{
Request	*rp;
int	i;

	curCtrlChar = ChIn ();
	if (!ParseRequestName ())
	{
		SkipToEol ();
		return;
	}

	/*
		First see if name's a macro.
	*/

	if (ProcessMacro (req))
		return;
	if (Bug (bugReqProcess))
		ETMMsg ("Request: <%s>", UStrToStr (req));
	if ((rp = LookupRequest (req)) == (Request *) NULL)
	{
		/* request not found; gobble rest of line and ignore */
		if (Bug (bugReqProcess))
			ETMMsg ("request <%s> not found", UStrToStr (req));
		SkipToEol ();
		return;
	}
	if (Bug (bugReqProcess))
		ETMMsg ("processing <%s>", UStrToStr (req));

	/*
		Perform parsing actions and skip to the end of the line.
		If parsing worked, process rest of actions.
	*/
	i = ProcessActionList (rp->reqPActions);
	SkipToEol ();
	if (i)
		(void) ProcessActionList (rp->reqActions);
}


int ProcessMacro (name)
UChar	*name;
{
int	result = 0;

	if (LookupMacro (name) != (Macro *) NULL)
	{
		AParseMacroArgs (0, (UChar **) NULL);
		SkipToEol ();
		PushMacro (name, rargc, rargv);
		result = 1;
	}
	return (result);
}



/*
	Skip whitespace up to the request (or macro) name, and read one or two
	characters to get the name.  Also initialize the argument buffer.

	The linefeed at the end of the line will NOT be eaten by this
	function, no matter what the return value.

	Request name can be up to four characters, if both request characters
	are special chars (each is unpacked into two bytes).
*/

static int ParseRequestName ()
{
int	c, i = 0;

	InitArgs ();
	SkipWhite ();			/* ctrl char might have white after */
	if (Eol (ChPeek ()))
		return (0);		/* no request named */
	if (Special (c = ChIn ()))
	{
		req[i++] = 0x80;
		req[i++] = c & 0xff;
	}
	else
		req[i++] = c;
	if (!WhiteOrEol (ChPeek ()))	/* 2-character request */
	{
		if (Special (c = ChIn ()))
		{
			req[i++] = 0x80;
			req[i++] = c & 0xff;
		}
		else
			req[i++] = c;
	}
	req[i] = '\0';
	return (1);
}


/*
	Parse macro arguments, each consisting of strings of non-white
	characters.  Double quotes may be used to surround arguments
	containing whitespace.  Quote may be doubled within quoted section
	to included double-quote.  Terminated by end of line.
*/

int AParseMacroArgs (argc, argv)
int	argc;
UChar	**argv;
{
int	quote;
int	c;

	CopyMode (1);			/* turn on copy mode */
	for (;;)
	{
		quote = 0;
		SkipWhite ();
		if (Eol (ChPeek ()))
			break;
		if (ChPeek () == '"')		/* quoted argument */
		{
			quote = 1;
			(void) ChIn ();
		}
		while (!Eol (ChPeek ()))
		{
			c = ChIn ();
			if (quote && c == '"')
			{
				if (ChPeek () != '"')	/* matching end quote */
					break;
				c = ChIn ();		/* doubled quote */
			}
			else if (!quote && White (c))	/* found word break */
				break;
			AddArgChar (c);
		}
		EndArg ();
	}
	CopyMode (0);			/* turn off copy mode */
	return (1);
}


/*
	Look for a filename argument: skip any whitespace, then find
	a sequence of non-white characters.  Reads through the end of
	line character.
*/

int AParseFileName (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	if (!Eol (ChPeek ()))
	{
		while (!WhiteOrEol (ChPeek ()))
			AddArgChar (ChIn ());
	}
	EndArg ();
	return (1);
}


/*
	Look for a (register, string, macro) name (x or xx).
*/

int AParseName (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	if (!Eol (ChPeek ()))
	{
		AddArgChar (ChIn ());
		if (!WhiteOrEol (ChPeek ()))
			AddArgChar (ChIn ());
	}
	EndArg ();
	/*if (!WhiteOrEol (ChPeek ()))*/
		/*return (0);*/	/* string name too long */
	return (1);
}


/*
	Look for a list of names and remove them.  If argv[0][0] is 'y',
	remove number registers, otherwise remove string, request, macro
	names.

	Used for .rm and .rr.  troff manual doesn't say it, but these
	requests can be given multiple arguments...
*/

int AParseRemoveList (argc, argv)
int	argc;
UChar	**argv;
{
Register	*rp;
int	i;
int	numReg;

	numReg = ((char) argv[0][0] == 'y');	/* 1 if number, 0 if not */
	for (;;)
	{
		SkipWhite ();
		if (Eol (ChPeek ()))
			break;
		AddArgChar (ChIn ());
		if (!WhiteOrEol (ChPeek ()))
			AddArgChar (ChIn ());
		EndArg ();
		if (numReg && (rp = LookupRegister (rargv[0]))
							!= (Register *) NULL)
		{
			if (rp->regReadOnly)
			{
				ETMMsg ("cannot remove readonly register <%s>",
							UStrToStr (rargv[0]));
				continue;
			}
		}
		RemoveName (rargv[0], numReg);
		/* reset arg mechanism to allow an arbitrary number of names */
		InitArgs ();
	}
	for (i = 0; i < rargc; i++)
	{
	}
	return (1);
}


/*
	Skip whitespace, then parse a string to the end of the input
	line in copy mode.  The argument is 'y' if a leading double
	quote should be recognized and stripped (as for .ds) or not
	(as for .tm and .ab).
*/

int AParseStringValue (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	if (argv[0][0] == 'y' && ChPeek () == '"')
		(void) ChIn ();
	CopyMode (1);			/* turn on copy mode */
	while (!Eol (ChPeek ()))
		AddArgChar (ChIn ());
	EndArg ();
	CopyMode (0);			/* turn off copy mode */
	return (1);
}


/*
	Parse one of those three-part title things: three title parts,
	delimited by the first non-white character following the request
	name.  Any part may be empty/missing.  Stop on end of line.  The
	three parts go into rargv as usual.  rargc indicates the number
	of parts actually found (0..3).
*/

int AParseTitle (argc, argv)
int	argc;
UChar	**argv;
{
int	i, delim;

	SkipWhite ();
	if (!Eol (ChPeek ()))
	{
		delim = ChIn ();
		for (i = 0; i < 3; i++)
		{
			while (ChPeek () != delim)
			{
				if (Eol (ChPeek ()))
					break;
				AddArgChar (ChIn ());
			}
			EndArg ();
			if (ChPeek () == delim)
				(void) ChIn ();
			if (Eol (ChPeek ()))
				break;
		}
	}
	return (1);
}


/*
	Parse a single-character argument.
*/

int AParseChar (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	if (!Eol (ChPeek ()))
	{
		AddArgChar (ChIn ());
		EndArg ();
	}
	return (1);
}


/*
	Parse number as an absolute quantity and convert back to string
	form for placement in the argument vector.

	argv[0] is default units indicator.
*/

int AParseNumber (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	ParseNumberArg ((int) argv[0][0], 0, (Param) 0);
	return (1);
}


/*
	Parse number as an absolute or relative quantity and convert back
	to string form for placement in the argument vector.

	argv[0] is default units indicator, argv[1] is the value against
	which relative values should be taken (zero if they must be absolute).
	argv[1] may be a symbolic name to represent the current setting
	for some parameter, a literal number, or, if it begins with \,
	the name of a number register (use the register value).  The latter
	is for use with .nr, where the initial value argument can be relative
	to the register's current value.
*/

int AParseARNumber (argc, argv)
int	argc;
UChar	**argv;
{
ParamMap	*pm;
Param	current;

	if (argv[1][0] == '\\')
		current = GetRegisterValue (&argv[1][1]);
	else
	{
		for (pm = paramMap; pm->pName != (char *) NULL; pm++)
		{
			if (UStrCmp (pm->pName, argv[1]) == 0)
				break;
		}
		if (pm->pName == (char *) NULL)
			current = (Param) Atol (argv[1]);
		else
			current = *(pm->pAddr);
	}
	SkipWhite ();
	ParseNumberArg ((int) argv[0][0], 1, current);
	return (1);
}


/*
	Parse number and put it into the current argument, in basic units
	if need be.  First argument is the default units, the second is
	non-zero if the number can be a relative quantity, zero if it must
	be absolute.  If it can be relative, the third argument is the
	current value, which is adjusted by the value of the number parsed.

	If the input doesn't look like it could be a number, the argument
	is left empty.
*/

static void ParseNumberArg (defUnit, relAllowed, curVal)
int	defUnit;
int	relAllowed;
Param	curVal;
{
char	buf[bufSiz], *p;
Param	val;

	if (ParseNumber (defUnit, relAllowed, curVal, &val))
	{
		sprintf (buf, "%ld", val);
		for (p = buf; *p != '\0'; p++)
			AddArgChar (*p);
	}
	EndArg ();
}


/*
	Parse number and return it, in basic units if need be.  Return value
	is zero if no number is found, else non-zero and the number value is
	placed in val.

	First argument is the default units, the second is non-zero if the
	number can be a relative quantity, zero if it must be absolute.
	If it can be relative, the third argument is the current value,
	which is adjusted by the value of the number parsed.

	Expressions that specify distance to an absolute position ("|N")
	evaluate to zero, since the current position is unknown.
*/

int ParseNumber (defUnit, relAllowed, curVal, val)
int	defUnit;
int	relAllowed;
Param	curVal;
Param	*val;
{
int	inc = 0;
int	dist = 0;

	if (!LooksLikeExpr ())
		return (0);

	if (relAllowed && Sign (ChPeek ()))
		inc = ChIn ();
	else if (ChPeek () == '|')
	{
		(void) ChIn ();
		dist = 1;
	}

	*val = ParseExpr (defUnit);

	/* inc is 0 if !relAllowed... */
	if (inc == '+')
		*val = curVal + *val;
	else if (inc == '-')
		*val = curVal - *val;

	if (dist)
		*val = 0;

	return (1);
}


/*
	Parse a numeric expression: term { op term { op term ... } }.
	Evaluation is left to right, except as overridden by parens.
*/

static Param ParseExpr (defUnit)
int	defUnit;
{
Param	val, val2;
int	op;
int	c;

	val = ParseExprTerm (defUnit);

	/*
		Look for op expr now
	*/

	for (;;)
	{
		switch (c = ChIn ())
		{
		default:		/* no operator; end of expr */
			UnChIn (c);
			return (val);
		case '+':
			op = opPlus;
			break;
		case '-':
			op = opMinus;
			break;
		case '/':
			op = opDivide;
			break;
		case '*':
			op = opMultiply;
			break;
		case '%':
			op = opMod;
			break;
		case '&':
			op = opAnd;
			break;
		case ':':
			op = opOr;
			break;
		case '<':			/* < or <= */
			if (ChPeek () != '=')
				op = opLess;
			else
			{
				(void) ChIn ();
				op = opLessEq;
			}
			break;
		case '>':			/* > or >= */
			if (ChPeek () != '=')
				op = opGreater;
			else
			{
				(void) ChIn ();
				op = opGreaterEq;
			}
			break;
		case '=':			/* = or == */
			if (ChPeek () == '=')
				(void) ChIn ();
			op = opEq;
			break;
		}
		val2 = ParseExprTerm (defUnit);
		switch (op)
		{
		default:
			ETMPanic ("ParseExpr: logic error, op = %d", op);
		case opPlus:
			val += val2;
			break;
		case opMinus:
			val -= val2;
			break;
		case opDivide:
			if (val2 == 0)
			{
				ETMMsg ("divide by zero detected");
				val = 0;
			}
			else
				val /= val2;
			break;
		case opMultiply:
			val *= val2;
			break;
		case opMod:
			val %= val2;
			break;
		case opAnd:
			val = (val && val2);
			break;
		case opOr:
			val = (val || val2);
			break;
		case opLess:
			val = (val < val2);
			break;
		case opLessEq:
			val = (val <= val2);
			break;
		case opGreater:
			val = (val > val2);
			break;
		case opGreaterEq:
			val = (val >= val2);
			break;
		case opEq:
			val = (val == val2);
			break;
		}
	}
}


/*
	Parse a single term of an expression, either a number or "(expr)"
*/

static Param ParseExprTerm (defUnit)
int	defUnit;
{
double	val = 0.0, mult;
int	sign = 1;
int	unit;

	if (ChPeek () == '(')		/* ( expr ) - recurse */
	{
		(void) ChIn ();
		val = ParseExpr (defUnit);
		if (ChPeek () == ')')	/* ')' should be there, but */
			(void) ChIn ();	/* don't be too fussy :-) */
		return (Units (val, 'u'));
	}

	while (Sign (ChPeek ()))
		sign *= (ChIn () == '+' ? 1 : -1);

	while (Digit (ChPeek ()))
		val = val * 10 + ChIn () - '0';

	if (ChPeek () == '.')
	{
		(void) ChIn ();
		mult = .1;
		while (Digit (ChPeek ()))
		{
			val += (ChIn () - '0') * mult;
			mult /= 10.0;
		}
	}

	unit = (UnitChar (ChPeek ()) ? ChIn () : defUnit);
	if (defUnit == 'x')
		unit = 'x';

	return (Units (val, unit) * sign);
}


/*
	Look at the next character to see whether it looks like it
	could begin a numeric expression.
*/

static int LooksLikeExpr ()
{
int	c = ChPeek ();

	return (Digit (c) || Sign (c) || c == '.' || c == '(' || c == '|');
}


/*
	Parse tab stops.  This writes out the results as it parses;
	the stops are not maintained internally at all.
*/

int AParseTabStops (argc, argv)
int	argc;
UChar	**argv;
{
Param	pos[maxTabStops];
char	type[maxTabStops];
int	i;
Param	prev = 0;

	for (i = 0; i < maxTabStops; i++)
	{
		SkipWhite ();
		if (!ParseNumber ('m', 1, prev, &pos[i]))
			break;
		type[i] = 'l';
		switch (ChPeek ())
		{
		case 'C':
			type[i] = 'c';
			(void) ChIn ();
			break;
		case 'R':
			type[i] = 'r';
			(void) ChIn ();
			break;
		}
		prev = pos[i];
	}
	SetTabStops ((Param) i, pos, type);

	return (1);
}


/*
	Parse "if" condition.  argv[0][0] is 'y' if an .el should
	be expected afterward.
*/

int AParseCondition (argc, argv)
int	argc;
UChar	**argv;
{
int	result = 0;

	if ((result = ParseCondition ()) <= 0)	/* malformed or false */
		SkipCondition ();
	else if (result > 0)			/* condition true */
		ProcessCondition ();

	/*
		It's necessary to put a linefeed back into the input
		since the request processor is going to look for a linefeed
		after calling the argument processing actions for the current
		.if/.ie request.  Without this, the next input line would be
		lost.  A subterfuge.
	*/

	PushAnonString ((UChar *) "\n");

	/*
		Save result for following .el if this was .ie request.
		result < 0 and result > 0 are both considered successes.
		This causes the .el to be skipped both if the test was
		malformed, or true.  If this was an .if (not .ie), then
		set the result true to any (illegal) following .el gets
		skipped.
	*/

	if (argv[0][0] == 'n')		/* was .if, not .ie */
		ifResult = 1;
	else if (result != 0)		/* .ie succeeded */
		ifResult = 1;
	else				/* .ie failed */
		ifResult = 0;
	return (1);
}


/*
	Parse the condition in preparation for processing the
	stuff contingent on the condition.  Some tests are impossible
	to do, e.g., whether the current page is even or odd.  In
	such cases, assume true.

	Right now, this only really interprets t, n, !t and !n correctly.
*/

static int ParseCondition ()
{
int	reverse = 0;
int	result = 0;
int	c;

	SkipWhite ();
	if (ChPeek () == '!')
	{
		(void) ChIn ();
		reverse = 1;
	}
	if (WhiteOrEol (ChPeek ()))
		return (-1);
	c = ChIn ();
	if (c == 't')
		result = troff;
	else if (c == 'n')
		result = !troff;
	else if (c == 'e' || c == 'o')	/* can't tell - yucko */
		result = 1;
	else
	{
		UnChIn (c);
		if (LooksLikeExpr ())
			result = (ParseExpr ('u') > 0);
		else
			result = ParseStrCondition ();
	}
	if (result < 0)
		return (-1);
	return (reverse ? !result : result);
}


static int ParseStrCondition ()
{
int	delim, c;
UChar	buf[bufSiz], *p;

	delim = ChIn ();
	p = buf;
	while (!Eol (c = ChIn ()))	/* find first string */
	{
		if (c == delim)
			break;
		*p++ = c;
	}
	if (Eol (c))		/* no terminating delim - malformed */
	{
		UnChIn (c);
		return (-1);
	}
	*p = '\0';
	p = buf;
	while (!Eol (c = ChIn ()))	/* find second string */
	{
		if (c == delim)			/* end of second */
			return (*p == '\0');	/* end of first? */
		if (c != *p++)		/* mismatch */
			break;
	}
	/*
		At this point, have either seen end of line without
		terminating delimiter (malformed condition) or a
		mismatch in the strings and need to look for closing
		delimiter of second string.
	*/
	if (Eol (c))		/* no terminating delim - malformed */
	{
		UnChIn (c);
		return (-1);
	}
	while (!Eol (ChPeek ()))
	{
		if (ChIn () == delim)
			return (0);
	}
	return (-1);
}


/*
	This is used for .el clauses.  Look at the result of the last
	.ie to see whether to process or skip the clause.  If it's
	processed, set ifResult so that any further .el's without an
	intervening .ie (shouldn't happen, but you never know) are
	skipped.
*/

int AProcessCondition (argc, argv)
int	argc;
UChar	**argv;
{
	if (Bug (bugConditional))
		ETMMsg ("el: %s condition", ifResult ? "skip" : "process");
	if (ifResult)			/* .ie succeeded, skip .el */
		SkipCondition ();
	else				/* .ie failed, accept .el */
	{
		ProcessCondition ();
		ifResult = 1;
	}
	PushAnonString ((UChar *) "\n");
	return (1);
}


static void ProcessCondition ()
{
int	level;

	if (Bug (bugConditional))
		ETMMsg ("process condition");
	SkipWhite ();
	if (ChPeek () != ToEsc ('{'))	/* single line */
	{
		if (Bug (bugConditional))
			ETMMsg ("process single-line condition");
		(void) ProcessLine ();
	}
	else				/* \{ ... \} */
	{
		if (Bug (bugConditional))
			ETMMsg ("process multiple-line condition");
		(void) ChIn ();
		level = ifLevel++;
		while (level < ifLevel)
		{
			if (Bug (bugConditional))
				ETMMsg ("current level %d, seeking %d",
							ifLevel, level);
			if (!ProcessLine ())
				break; /* oops...EOF in middle of condition */
		}
		if (Bug (bugConditional))
			ETMMsg ("found level %d", level);
	}
	if (Bug (bugConditional))
		ETMMsg ("done processing");
}


static void SkipCondition ()
{
int	ifBeg = ToEsc ('{');
int	level;
int	c;

	if (Bug (bugConditional))
		ETMMsg ("skip condition");
	while (!Eol (c = ChIn ()) && c != ifBeg)
	{
		/* spin wheels */
	}
	if (Eol (c))
	{
		if (Bug (bugConditional))
			ETMMsg ("skip single-line condition");
		return;
	}
	if (Bug (bugConditional))
		ETMMsg ("skip multiple-line condition");
	level = ifLevel++;
	while (level < ifLevel)
	{
		if (Bug (bugConditional) && Eol (c))
			ETMMsg ("current level %d, seeking %d",
							ifLevel, level);
		if ((c = ChIn ()) == EOF)
			ETMPanic ("EOF looking for conditional end");
		if (c == ifBeg)
			++ifLevel;
	}
	if (Bug (bugConditional))
		ETMMsg ("found level %d", level);
	while (!Eol (c))
		c = ChIn ();
	if (Bug (bugConditional))
		ETMMsg ("done skipping");
}


/*
	This is a hack for .bd request.  Look and see if there's an "S"
	argument.  If so, eat it.  Else make the request argument empty.

	S has to be followed by whitespace, since .bs Sx N is also legal.
	Note that .bs S N isn't handled properly.  Too bad.
*/

int AParseS (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	if (ChPeek () == 'S')
	{
		(void) ChIn ();
		if (White (ChPeek ()))
			AddArgChar ('S');
		else
			UnChIn ('S');
	}
	EndArg ();
	return (1);
}


/*
	Parse transliteration list.
*/

int AParseTransList (argc, argv)
int	argc;
UChar	**argv;
{
	SkipWhite ();
	while (!Eol (ChPeek ()))
		AddArgChar (ChIn ());
	EndArg ();
	return (1);
}


/*
	Interpret an action argument.  Returns a pointer into a static
	area, which should be copied out if it's to be held for a while.

	Instances of $n (n=1..9) are replaced by the appropriate request
	argument.  $$ is replaced by the number of request arguments.  \n
	and \t are replaced by linefeed and tab, respectively.  Other
	backslash sequences just have the initial backslash stripped.
	(This means that to include a backslash, double it.)
*/

UChar *InterpretActionArg (p)
char	*p;
{
static UChar	buf[bufSiz];
UChar	*bp = buf;
int	c, i;

	if (p == NULL)
		ETMPanic ("InterpretActionArg: logic error");
	while ((c = *p++) != '\0')
	{
		if (c == '$')		/* substitute request argument */
		{
			if ((c = *p++) == '\0')
				break;		/* malformed */
			if (c == '$')
			{
				sprintf ((char *) bp, "%d", rargc);
				bp += strlen ((char *) bp);
			}
			else if (Digit (c))
			{
				if ((i = c - '0') > 0 && i <= rargc)
				{
					strcpy (bp, rargv[i-1]);
					bp += strlen ((char *) bp);
				}
			}
			continue;
		}
		if (c == '\\')		/* strip escape, process next char */
		{
			if ((c = *p++) == '\0')
				break;		/* malformed */
			switch (c)
			{
			case 'n': c = '\n'; break;
			case 't': c = '\t'; break;
			/* otherwise leave alone */
			}
		}
		*bp++ = c;
	}
	*bp = '\0';
	return (buf);
}


/*	--------------------------------------------------	*/
/*	Escape sequence interpreters				*/
/*	--------------------------------------------------	*/



/*
	Look for sequence following \s.  Anything illegal is treated
	like \s0 (e.g., \sx)

	\s0		revert to previous
	\s-[1-9]	decrease current by 1-9
	\s+[1-9]	increase current by 1-9
	\s[1-9]		set to 1-9
	\s[1-2][0-9]	set to 10..29
	\s3[0-6]	set to 30..36

	Those last two lines require comment:
	\s can be followed by a two-digit number <= 36; the number is
	interpreted as an absolute change.  (Original C/A/T phototypesetter
	had sizes up to 36.  Original troff would take \s36 as .ps 36 and
	\s37 as .ps 3 followed by "7".  What a pain.)
*/

Param ParseSizeRef ()
{
int	c;
int	inc = 0;	/* 0 = abs, -1 = rel. dec., +1 = rel. inc. */
Param	val;

	c = ChPeek ();
	if (c == '0')			/* restore previous */
	{
		(void) ChIn ();
		return (prevSize);
	}
	if (Sign (c))			/* relative change */
	{
		(void) ChIn ();
		inc = (c == '+' ? 1 : -1);
		if (!Digit (ChPeek ()))	/* malformed */
			return (prevSize);
		return (curSize + Units (((ChIn () - '0') * inc), 'x'));
	}

	/* one- or two-digit absolute setting */

	if (!Digit (c))			/* malformed */
		return (prevSize);
	val = ChIn () - '0';
	if (!Digit (c = ChPeek ()))		/* one-digit */
		return (Units (val, 'x'));
	/* two-digit, but don't eat second unless <= 36 */
	c -= '0';
	if (val < 3 || (val == 3 && c <= 6))
	{
		val = val * 10 + c;
		(void) ChIn ();
	}
	return (Units (val, 'x'));
}


/*
	Parse a register reference following \n.  Return pointer to
	register structure if it was found, NULL on malformed reference.

	\nx
	\n(xx
	\n+x,\n+(xx
	\n-x,\n-(xx
*/

Register *ParseRegisterRef ()
{
Register	*rp;
UChar	*p;
int	inc = 0;
int	c;

	c = ChPeek ();
	if (Eol (c))
		return ((Register *) NULL);
	if (Sign (c))			/* auto inc/dec */
	{
		(void) ChIn ();		/* skip inc/dec char */
		inc = (c == '+' ? 1 : -1);
	}

	/* now positioned at x or (xx */

	if ((p = ParseNameRef ()) == (UChar *) NULL)
		return ((Register *) NULL);

	if ((rp = LookupRegister (p)) != (Register *) NULL)
	{
		/* auto inc/dec if necessary */
		if (inc && rp->regReadOnly)
			ETMMsg ("attempt to set readonly register <%s> ignored",
								UStrToStr (p));
		else
			rp->regValue += (inc * rp->regIncrement);
		UFree (p);
		return (rp);
	}
	UFree (p);
	return (rp);
}



/*
	Look for a single or double character reference.  Used for references
	to fonts \fX or \f(XX, strings \*X or \*(XX}, number registers \nX
	or \n(XX and special characters \(XX.

	This HAS to allocate memory and can't use a static buffer because
	it can be called recursively (e.g., .nr \(ts\(ts ....).
*/

UChar *ParseNameRef ()
{
UChar	regName[5];
int	c, i = 0;

	if (WhiteOrEol (ChPeek ()))		/* malformed */
		return ((UChar *) NULL);
	if ((c = ChIn ()) != '(')		/* single-char name */
	{
		if (Special (c))
		{
			regName[i++] = 0x80;
			regName[i++] = c & 0xff;
		}
		else
			regName[i++] = c;

		regName[i] = '\0';
		return (UStrAlloc (regName));
	}

	/* double-char sequence - skip '(', read two more */

	if (WhiteOrEol (ChPeek ()))		/* malformed */
		return ((UChar *) NULL);
	if (Special (c = ChIn ()))
	{
		regName[i++] = 0x80;
		regName[i++] = c & 0xff;
	}
	else
		regName[i++] = c;
	if (Eol (ChPeek ()))			/* malformed */
		return ((UChar *) NULL);
	if (Special (c = ChIn ()))
	{
		regName[i++] = 0x80;
		regName[i++] = c & 0xff;
	}
	else
		regName[i++] = c;
	regName[i] = '\0';
	return (UStrAlloc (regName));
}


/*
	Parse a sequence of the form \w'string', calculate the width
	of the string, convert that number to string form and switch
	the input to it.  The \w will already have been seen.

	The width is crudely approximated as the number of characters
	in the string times the width of an en.

	Size and font changes are recognized but skipped.

	The width string is put into the input pushback to avoid problems
	with the order in which any trailing peeked-at character and the
	width string will be returned later.
*/

void ParseWidth ()
{
char	buf[bufSiz], *p;
int	c, delim;
Param	wid = 0;

	if (Eol (delim = ChIn ()))		/* malformed */
		UnChIn (delim);
	else while ((c = ChIn ()) != delim)
	{
		if (Eol (c))			/* malformed */
		{
			UnChIn (c);
			break;
		}
		if (c == ToEsc ('s'))		/* size change */
			(void) ParseSizeRef ();
		else if (c == ToEsc ('f'))	/* font change */
			UFree (ParseNameRef ());
		else
			++wid;
	}

	/* figure width in ens, convert to string, push back onto input */
	sprintf (buf, "%ld", Units (wid, 'n'));
	p = buf + strlen (buf);
	while (p > buf)
		UnChIn (*--p);
}
