/*
	Name list operations
*/

# include	<stdio.h>

# include	"etm.h"
# include	"memmgr.h"

# include	"troffcvt.h"


static int	Hash ();



/*
	Create a new name structure, initialize the name and type field, and
	create a structure corresponding to the type.  The latter structure
	must be initialized later.  The name structure is added to the list
	of known names.

	If the name is currently known, remove it first.
*/

static NameDef *NewName (name, type)
UChar	*name;
int	type;
{
NameDef	*np;

	RemoveName (name, type == regDef);
	np = New (NameDef);
	np->ndName = UStrAlloc (name);
	np->ndHash = Hash (name);
	switch (np->ndType = type)
	{
	case reqDef:
		np->ndRequest = New (Request);
		break;
	case macDef:
		np->ndMacro = New (Macro);
		np->ndMacro->macParent = np;
		break;
	case strDef:
		np->ndString = New (String);
		break;
	case regDef:
		np->ndRegister = New (Register);
		np->ndRegister->regParent = np;
		break;
	default:
		ETMPanic ("NewName: logic error");
	}
	np->ndNext = nameList;
	nameList = np;
	return (np);
}


/*
	Allocate a new request.  The action list isn't initialized because
	it might not be known.
*/

Request *NewRequest (name)
UChar	*name;
{
Request	*rp = NewName (name, reqDef)->ndRequest;

	rp->reqPActions = (Action *) NULL;
	rp->reqActions = (Action *) NULL;
	return (rp);
}


/*
	Allocate a new macro.  The in-use and remove flags are used
	for the case that a macro is redefined while it's executing.
	It can't be removed while it's still active, but it should
	be when it finishes.
*/

Macro *NewMacro (name)
UChar	*name;
{
Macro	*mp = NewName (name, macDef)->ndMacro;

	mp->macBuf = (char *) NULL;
	mp->macSiz = mp->macMaxSiz = 0;
	mp->macInUse = mp->macRemove = 0;
	return (mp);
}


/*
	Allocate a new string and initialize its value (shouldn't be NULL).
*/

String *NewString (name, value)
UChar	*name;
char	*value;
{
String	*sp = NewName (name, strDef)->ndString;

	sp->strValue = StrAlloc (value);
	sp->strInUse = 0;
	return (sp);
}


/*
	Allocate a new register and initialize its values.  The hard part
	is determining the format type.  The default is numeric, with width
	zero (which means "as wide as necessary").  Width is only consulted
	for numeric format.  If the format isn't recognized, numeric is used.
*/

Register *NewRegister (name, initial, increment, format, readonly)
UChar	*name;
Param	initial;
Param	increment;
char	*format;
int	readonly;
{
Register	*rp = NewName (name, regDef)->ndRegister;

	rp->regValue = initial;
	rp->regIncrement = increment;
	rp->regReadOnly = readonly;
	SetRegisterFormat (rp, format);
	return (rp);
}


/*
	Look up a name of the given type or types.  "type" is a bitmap of all
	the types that will be accepted (anyDef for all).
*/

NameDef *LookupName (name, type)
UChar	*name;
int	type;
{
NameDef	*nd;
int	hash = Hash (name);

	for (nd = nameList; nd != (NameDef *) NULL; nd = nd->ndNext)
	{
		if ((nd->ndType & type)
			&& nd->ndHash == hash
			&& UStrCmp (nd->ndName, name) == 0)
			return (nd);
	}
	return ((NameDef *) NULL);
}


Request *LookupRequest (name)
UChar	*name;
{
NameDef	*nd;

	if ((nd = LookupName (name, reqDef)) == (NameDef *) NULL)
		return ((Request *) NULL);
	return (nd->ndRequest);
}


Macro *LookupMacro (name)
UChar	*name;
{
NameDef	*nd;

	if ((nd = LookupName (name, macDef)) == (NameDef *) NULL)
		return ((Macro *) NULL);
	return (nd->ndMacro);
}


String *LookupString (name)
UChar	*name;
{
NameDef	*nd;

	if ((nd = LookupName (name, strDef)) == (NameDef *) NULL)
		return ((String *) NULL);
	return (nd->ndString);
}


Register *LookupRegister (name)
UChar	*name;
{
NameDef	*nd;

	if ((nd = LookupName (name, regDef)) == (NameDef *) NULL)
		return ((Register *) NULL);
	return (nd->ndRegister);
}


/*
	Remove a name from the name list.  numReg is true if the name to
	be removed is a number register, false if it is a request, macro
	or string.  It is used because requests, registers, macros and
	strings are all maintained on the same list, but registers are
	considered to live in a different name space.
*/

void RemoveName (name, numReg)
UChar	*name;
int	numReg;
{
NameDef	*nd;
Macro	*mp;
int	hash = Hash (name);

	for (nd = nameList; nd != (NameDef *) NULL; nd = nd->ndNext)
	{
		if (((numReg && nd->ndType == regDef)
			|| (!numReg && nd->ndType != regDef))
			&& nd->ndHash == hash
			&& UStrCmp (nd->ndName, name) == 0)
			break;
	}
	if (nd == (NameDef *) NULL)
		return;
	switch (nd->ndType)	/* deallocate name-type specific stuff */
	{
	case reqDef:
		FreeActions (nd->ndRequest->reqPActions);
		FreeActions (nd->ndRequest->reqActions);
		Free ((char *) nd->ndRequest);
		break;
	case macDef:
		mp = nd->ndMacro;
		if (mp->macInUse)	/* if in use, defer removal */
		{
			mp->macRemove = 1;
			return;
		}
		/* don't free macParent 'cause it points into nd struct */
		Free (mp->macBuf);	/* remove macro body */
		Free ((char *) nd->ndMacro);
		break;
	case strDef:
		Free (nd->ndString->strValue);
		Free ((char *) nd->ndString);
		break;
	case regDef:
		Free ((char *) nd->ndRegister);
		break;
	default:
		ETMPanic ("RemoveName: logic error");
	}
	RemoveNameDef (nd);
}


void RemoveNameDef (nd)
NameDef	*nd;
{
NameDef	*nd2;

	/*
		Remove nd from name list and free it
	*/

	if (nameList == nd)		/* first element in list */
		nameList = nd->ndNext;	/* skip to next */
	else				/* remove from middle of list */
	{
		nd2 = nameList;
		while (nd2->ndNext != (NameDef *) NULL)
		{
			if (nd2->ndNext == nd)
			{
				nd2->ndNext = nd->ndNext;
				break;
			}
			nd2 = nd2->ndNext;
		}
	}
	UFree (nd->ndName);
	Free ((char *) nd);
}


/*
	Rename a name definition, replacing both the name and the
	hash value.
*/

void RenameName (np, name)
NameDef	*np;
char	*name;
{
	UFree (np->ndName);
	np->ndName = UStrAlloc (name);
	np->ndHash = Hash ((UChar *) name);
}


/*
	Define a special character.  If the character is already defined,
	just replace the value.  Otherwise add it to the list.
	Codes start at 1 because otherwise special char codes in strings
	end up as null, and strlen/strcmp doesn't work.
*/

void NewSpChar (name, value)
char	*name, *value;
{
SpChar	*sp;
static int	code = 1;

	if ((sp = LookupSpChar (name)) != (SpChar *) NULL)
	{
		Free (sp->spValue);
		sp->spValue = StrAlloc (value);
	}
	else
	{
		if (code > 255)
			ETMPanic ("too many special characters defined");
		sp = New (SpChar);
		sp->spName = StrAlloc (name);
		sp->spCode = (scMask | code++);
		sp->spHash = Hash ((UChar *) name);
		sp->spValue = StrAlloc (value);
		sp->spNext = spCharList;
		spCharList = sp;
	}
}


/*
	Look up a special character definition
*/

SpChar *LookupSpChar (name)
char	*name;
{
SpChar	*sp;
int	hash = Hash ((UChar *) name);

	for (sp = spCharList; sp != (SpChar *) NULL; sp = sp->spNext)
	{
		if (sp->spHash == hash && strcmp (sp->spName, name) == 0)
			break;
	}
	return (sp);	/* NULL if not found */
}


/*
	Look up a special character definition by code
*/

SpChar *LookupSpCharByCode (code)
int	code;
{
SpChar	*sp;

	for (sp = spCharList; sp != (SpChar *) NULL; sp = sp->spNext)
	{
		if (sp->spCode == code)
			break;
	}
	return (sp);	/* NULL if not found */
}


/*
	Compute hash value of string
*/

static int Hash (s)
UChar	*s;
{
int	c;
int	val = 0;

	while ((c = *s++) != '\0')
		val += (int) c;
	return (val);
}
