/* 
 *
 *  soview.c++
 *
 *  Forms Library object that 
 *  displays Inventor scene graphs
 *
 *  Written by Allard Siemelink
 *  Version 1.0
 *  Date Apr 13, 1993
 *
 */


#include <gl/gl.h>
#include <gl/device.h>
#include <Inventor/SoGL.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/SoSensor.h>
#include <Inventor/events/SoEvents.h>
#include <Inventor/actions/SoRayPickAction.h>
#include "malloc.h"
#include "forms.h"
#include "soview.h"


#define SPEC(obj)	((struct soviewspec *)obj->spec)
#define MODIFIERNO	6		// number of modifier keys


struct soviewspec {
	SoNode		*scene;		// root of scene graph
	SoSceneManager	*sceneManager;	// redraws scene, handles events
	SoDataSensor	*dataSensor; 	// triggers redraws
	SoRayPickAction *pickAction;	// set by user, applied in pickmode
	int		 pickMode;	// pickmode active?
	short		 modval[MODIFIERNO];  // state of the modifier keys
};


static SoMouseButtonEvent mouseButtonEvent; // reports mouse clicks
static SoLocation2Event loc2Event;	    // reports mouse movement
static SoKeyboardEvent keyboardEvent; 	    // reports key presses


#define ske		SoKeyboardEvent
#define any		SoKeyboardEvent::ANY

static Device moddev[MODIFIERNO] = {
	LEFTSHIFTKEY, RIGHTSHIFTKEY,	// Shift keys
	LEFTALTKEY,   RIGHTALTKEY,	// Alt keys
	LEFTCTRLKEY,  RIGHTCTRLKEY	// Control keys
};

static SoKeyboardKey devtrans[MODIFIERNO] = {
	ske::LEFT_SHIFT,   ske::RIGHT_SHIFT,	// Inventor modifier
	ske::LEFT_ALT,     ske::RIGHT_ALT,	// key definitions
	ske::LEFT_CONTROL, ske::RIGHT_CONTROL
};


// conversion table to convert ascii
// key codes to Inventor key codes
static SoKeyboardKey keytable[128] = {
  any,
  ske::LEFT_ARROW, ske::RIGHT_ARROW, ske::UP_ARROW, ske::DOWN_ARROW,	// 1-4
  any, any, any, ske::BACKSPACE, ske::TAB,			     	// 5-9
  ske::RETURN, any, any, any, any, any, any, any, any, any,	     	// 10-19
  any, any, any, any, any, any, any, any, any, any,		     	// 20-29
  any, any, ske::SPACE, any, any, any, any, any, any, ske::APOSTROPHE,	// 30-39
  any, any, any, any, 
  ske::COMMA, ske::MINUS, ske::PERIOD, ske::SLASH,
  ske::NUMBER_0, ske::NUMBER_1, ske::NUMBER_2, ske::NUMBER_3,		// 48-51
  ske::NUMBER_4, ske::NUMBER_5, ske::NUMBER_6, ske::NUMBER_7,		// 52-55
  ske::NUMBER_8, ske::NUMBER_9, any, ske::SEMICOLON,			// 56-59
  any, ske::EQUAL, any, any, any,					// 60-64
  ske::A, ske::B, ske::C, ske::D, ske::E, ske::F, ske::G, 		// 65-
  ske::H, ske::I, ske::J, ske::K, ske::L, ske::M, ske::N,
  ske::O, ske::P, ske::Q, ske::R, ske::S, ske::T, ske::U,
  ske::V, ske::W, ske::X, ske::Y, ske::Z,		  		//   -90
  ske::BRACKETLEFT, ske::BACKSLASH, ske::BRACKETRIGHT, any, any, ske::GRAVE,
  ske::A, ske::B, ske::C, ske::D, ske::E, ske::F, ske::G, 		// 97-
  ske::H, ske::I, ske::J, ske::K, ske::L, ske::M, ske::N,
  ske::O, ske::P, ske::Q, ske::R, ske::S, ske::T, ske::U,
  ske::V, ske::W, ske::X, ske::Y, ske::Z,
  any, any, any, any							//   -127
};
#undef any
#undef ske
 

// sets the state of the modifier
// keys for a given event
static void
setModifierKeys(SoEvent *event, short *modval)
{
  event->setShiftDown(modval[0]||modval[1]);
  event->setCtrlDown(modval[2]||modval[3]);
  event->setAltDown(modval[4]||modval[5]);
}


// If the state of a modifier key is changed, create
// keyboard event and report change to Inventor
static void
updateModifierKeys(FL_OBJECT *obj, short *modval, short mx, short my)
{
  int i;
  short newmodval[MODIFIERNO];

  getdev(MODIFIERNO, moddev, newmodval);
  for (i=0; i<MODIFIERNO; i++) {
    if (newmodval[i] != modval[i]) {	// modifier key changed state
      modval[i] = newmodval[i];
      keyboardEvent.setKey(devtrans[i]);
      if (modval[i]) {
        keyboardEvent.setState(SoButtonEvent::DOWN);
      }
      else {
        keyboardEvent.setState(SoButtonEvent::UP);
      }
      keyboardEvent.setTime(SbTime::getTimeOfDay());
      keyboardEvent.setPosition(SbVec2s(mx, my));
      setModifierKeys(&keyboardEvent, modval);

      SPEC(obj)->sceneManager->processEvent(&keyboardEvent);
    }
  }
}


// translate ascii key code to Inventor keyboard event
static SoEvent *
translateKeyboardEvent(short *modval, short mx, short my, int key)
{
  keyboardEvent.setKey(keytable[key]);
  keyboardEvent.setState(SoButtonEvent::UNKNOWN);
  
  keyboardEvent.setTime(SbTime::getTimeOfDay());
  keyboardEvent.setPosition(SbVec2s(mx, my));

  setModifierKeys(&keyboardEvent, modval);
  return (SoEvent *)&keyboardEvent;
}


// translate mouse location to Inventor mouse event
static SoEvent *
translateMouseEvent(short *modval, short mx, short my)
{
  loc2Event.setTime(SbTime::getTimeOfDay());
  loc2Event.setPosition(SbVec2s(mx, my));
  setModifierKeys(&loc2Event, modval);

  return (SoEvent *)&loc2Event;
}


// mouse button released, translate to Inventor event
static SoEvent *
translateReleaseEvent(short *modval, short mx, short my, int button)
{
  switch (button) {
    case 1: mouseButtonEvent.setButton(SoMouseButtonEvent::BUTTON3);
	break;
    case 2: mouseButtonEvent.setButton(SoMouseButtonEvent::BUTTON2);
	break;
    case 3: mouseButtonEvent.setButton(SoMouseButtonEvent::BUTTON1);
	break;
  }

  mouseButtonEvent.setState(SoButtonEvent::UP);
  mouseButtonEvent.setTime(SbTime::getTimeOfDay());
  mouseButtonEvent.setPosition(SbVec2s(mx, my));

  setModifierKeys(&mouseButtonEvent, modval);
  return (SoEvent *)&mouseButtonEvent;
}


// mouse button pressed, translate to Inventor event
static SoEvent *
translatePushEvent(short *modval, short mx, short my, int button)
{
  switch (button) {
    case 1: mouseButtonEvent.setButton(SoMouseButtonEvent::BUTTON3);
	break;
    case 2: mouseButtonEvent.setButton(SoMouseButtonEvent::BUTTON2);
	break;
    case 3: mouseButtonEvent.setButton(SoMouseButtonEvent::BUTTON1);
	break;
  }

  mouseButtonEvent.setState(SoButtonEvent::DOWN);
  mouseButtonEvent.setTime(SbTime::getTimeOfDay());
  mouseButtonEvent.setPosition(SbVec2s(mx, my));

  setModifierKeys(&mouseButtonEvent, modval);
  return (SoEvent *)&mouseButtonEvent;
}


// handler function for Forms Library Events
// This function is called by the Forms Library kernel
// whenever the soview object needs to handle an event
static int
handle_soview(FL_OBJECT *obj, int event, float mx, float my, char key)
{
  struct soviewspec *spec = (struct soviewspec *)obj->spec;

  switch (event) {
    case FL_DRAW:
	fl_drw_box(obj->boxtype, obj->x, obj->y, obj->w, obj->h,
		   obj->col1, FL_BOUND_WIDTH);
	fl_drw_text_beside(obj->align, obj->x, obj->y, obj->w, obj->h,
			   obj->lcol,  obj->lsize, obj->lstyle, obj->label);

	if (spec->scene) {
	  // save drawing attributes
	  pushattributes();

	  spec->sceneManager->reinitialize();
	  spec->sceneManager->setOrigin(SbVec2s((short)(obj->x+FL_BOUND_WIDTH),
					        (short)(obj->y+FL_BOUND_WIDTH)));
	  spec->sceneManager->resize   (SbVec2s((short)(obj->w-2*FL_BOUND_WIDTH),
					        (short)(obj->h-2*FL_BOUND_WIDTH)));
	  SoGL::setWindowOrigin(SbVec2s((short)(obj->form->x+FL_BOUND_WIDTH),
				        (short)(obj->form->y+FL_BOUND_WIDTH)));
	  spec->sceneManager->render(0);	/* No clear before rendering */

	  // Restore drawing attributes
	  popattributes();
	  reshapeviewport();
	  ortho2(-0.5, obj->form->w-0.5, -0.5, obj->form->h-0.5);

	  zbuffer(FALSE);	// Not restored by Inventor!
	  shademodel(GOURAUD);
	  depthcue(0);
	}

	break;
    case FL_MOUSE:
	// Pass mouse motion event to scenegraph
	if (spec->scene) {
	  spec->sceneManager->processEvent(translateMouseEvent(
					   spec->modval, (short)(mx), (short)(my)));
	}
	break;
    case FL_PUSH:
	if (spec->scene) {
	  if (spec->pickMode && spec->pickAction) {
	    // in pickmode, apply pick action
	    spec->pickAction->setWindowSize(
			  SbVec2s((short)obj->w, (short)obj->h));
	    spec->pickAction->setWindowOrigin(
			  SbVec2s((short)obj->x, (short)obj->y));
	    spec->pickAction->setPoint(
			  SbVec2s((short)mx, (short)my));
	    spec->pickAction->apply(spec->scene);

	    return 1;	// notify application
	  }
	  else {
	    // not in pickmode, pass push event to scene graph
	    spec->sceneManager->processEvent(translatePushEvent(
					     spec->modval, (short)mx, (short)my, key));
	  }
	}
	break;
    case FL_RELEASE:
	// pass mouse button release event to scene graph
	if (spec->scene) {
	  spec->sceneManager->processEvent(translateReleaseEvent(
					   spec->modval, (short)mx, (short)my, key));
	}
	break;
    case FL_KEYBOARD:
	// pass keyboard event to scene graph
	if (spec->scene) {
	  spec->sceneManager->processEvent(translateKeyboardEvent(
					   spec->modval, (short)mx, (short)my, key));
	}
	break;
    case FL_STEP:
	// keep Inventor running
      	if (SoGL::isTimerSensorPending()) SoGL::processTimerSensors();
      	if (SoGL::isIdleSensorPending()) SoGL::processIdleSensors();
	if (spec->scene) {
	  updateModifierKeys(obj, spec->modval, (short)mx, (short)my);
	}
	break;
    case FL_FREEMEM:
	fl_soview_setSceneGraph(obj, 0);	// unrefs() scene graph
	delete spec->sceneManager;
	spec->dataSensor->unref();
	free(spec);
	break;
  }

  return 0;
} 


// the scene graph has changed,
// tell soview object to redraw
static void
sceneChanged_cb(void *udata, SoSensor *)
{
  fl_redraw_object((FL_OBJECT *)udata);
}


/////////////////////////////////////////////////////
////		Public Functions		/////
/////////////////////////////////////////////////////


FL_OBJECT *
fl_create_soview(int type, float x, float y, float w, float h, char *label)
{
  FL_OBJECT *obj;
  struct soviewspec *spec;
  int i;

  obj = fl_make_object(FL_SOVIEW, type, x, y, w, h, label, handle_soview);
  if (obj) {
    obj->boxtype = FL_SOVIEW_BOXTYPE;
    obj->col1	 = FL_SOVIEW_COL1;
    obj->col2	 = FL_SOVIEW_COL2;
    obj->lcol	 = FL_SOVIEW_LCOL;
    obj->align   = FL_SOVIEW_ALIGN;
    obj->spec	 = (int *)malloc(sizeof(struct soviewspec));
    spec	 = (struct soviewspec *)obj->spec;
    obj->active  = 1;
    obj->input   = 1;
    obj->automatic = 1;			// receive FL_STEP events

    // initialize spec
    spec->scene	       = 0;
    spec->pickAction   = 0;
    spec->pickMode     = 0;
    spec->sceneManager = new SoSceneManager(SbVec2s(0, 0));
    spec->dataSensor   = new SoDataSensor;
    spec->dataSensor->ref();
    spec->dataSensor->setFunction(&sceneChanged_cb);
    spec->dataSensor->setData((void *)obj);	// obj reported to callback
    SbTime time(0, 20000);			// 1/50th of a second
    spec->dataSensor->setDelayTimeOut(1, time);	// delay redraw after change
    for (i=0; i<MODIFIERNO; i++) spec->modval[i] = 0;
  }

  return obj;
}


FL_OBJECT *
fl_add_soview(int type, float x, float y, float w, float h, char *label)
{
  FL_OBJECT *obj;

  obj = fl_create_soview(type, x, y, w, h, label);
  if (obj) {
    fl_add_object(fl_current_form, obj);
  }

  return obj;
}


SoNode *
fl_soview_getSceneGraph(FL_OBJECT *obj)
{
  return SPEC(obj)->scene;
}


// diplay a new scene, be aware
// that node may be NULL
void
fl_soview_setSceneGraph(FL_OBJECT *obj, SoNode *node)
{
  if (node) {
    node->ref();		// reference node
  }
  if (SPEC(obj)->scene) {	// current scene
    SPEC(obj)->dataSensor->detach(SPEC(obj)->scene);
    SPEC(obj)->scene->unref();	// release scene
  }
  SPEC(obj)->sceneManager->setSceneGraph(node);
  SPEC(obj)->scene = node;
  if (node) {
    SPEC(obj)->dataSensor->attach(node);
  }
}


void
fl_soview_setTransparencyType(FL_OBJECT *obj, SoTransparencyType type)
{
  SPEC(obj)->sceneManager->setTransparencyType(type);
}


SoTransparencyType
fl_soview_getTransparencyType(FL_OBJECT *obj)
{
  return SPEC(obj)->sceneManager->getTransparencyType();
}


void
fl_soview_setAntialiasing(FL_OBJECT *obj, SbBool smoothing, int numPasses)
{
  SPEC(obj)->sceneManager->setAntialiasing(smoothing, numPasses);
}


void
fl_soview_getAntialiasing(FL_OBJECT *obj, SbBool &smoothing, int &numPasses)
{
  SPEC(obj)->sceneManager->getAntialiasing(smoothing, numPasses);
}


void
fl_soview_setPickAction(FL_OBJECT *obj, SoRayPickAction *pa)
{
  SPEC(obj)->pickAction = pa;
}


SoRayPickAction *
fl_soview_getPickAction(FL_OBJECT *obj)
{
  return SPEC(obj)->pickAction;
}


void
fl_soview_setPickMode(FL_OBJECT *obj, int on)
{
  SPEC(obj)->pickMode = on;
}


int
fl_soview_getPickMode(FL_OBJECT *obj)
{
  return SPEC(obj)->pickMode;
}


void
fl_soview_setDelay(FL_OBJECT *obj, SbBool on)
{
  SPEC(obj)->dataSensor->setDelay(on);
}


SbBool
fl_soview_isDelayed(FL_OBJECT *obj)
{
  return SPEC(obj)->dataSensor->isDelayed();
}


void
fl_soview_setDelayTimeOut(FL_OBJECT *obj, SbBool on, SbTime timeOut)
{
  SPEC(obj)->dataSensor->setDelayTimeOut(on, timeOut);
}


void
fl_soview_getDelayTimeOut(FL_OBJECT *obj, SbBool &t, SbTime &timeOut)
{
  SPEC(obj)->dataSensor->getDelayTimeOut(t, timeOut);
}
