/*  
 *  colorsel.c, version 1.2
 *  24-bit Color Selector Pop-Up Form
 *
 *  by Dan Wallach <dwallach@nas.nasa.gov>
 *  
 *  Based on an original concept by Dave Navas.
 *  
 *  This belongs with the other "goodies" in the Forms library.
 *  Simply call fl_show_color_selector(&red, &green, &blue) and
 *  the red, green, and blue integers you pass will be changed
 *  by the user to reflect the his/her's preferences.  The RGB
 *  values must have sane values beforehand, or results will be
 *  undetermined.
 *
 *  Various improvements: double buffering now works, but single
 *  buffering is still available.  Indicators alongs the sides
 *  seem to help people figure out what they're doing.
 *
 *  Only two routines in here are for "external" use:
 *  
 *  fl_show_color_selector(int *red, int *green, int *blue);
 *  fl_show_color_selector_sbuffer(int *red, int *green, int *blue);
 *  
 *  The former is double buffered, but doesn't look as nice on some
 *  Personal Irises (and others) which only give you 12 bits instead
 *  of the usual 24.  The color values generated are independent of
 *  how the Iris decides to render the colors.
 */

#include <stdio.h>
#include "forms.h"


#define SCHEME_RED_BLUE		0
#define SCHEME_RED_GREEN	1
#define SCHEME_GREEN_BLUE	2

/* set this to fprintf(stderr, (x)) to see debugging output */
#define DBG(x)

static int current_color_scheme;
static int new_red, new_green, new_blue;

static FL_FORM *color_selector_form;

static FL_OBJECT *button_cancel, *button_okay,
		 *button_red_blue, *button_red_green,
		 *button_green_blue, *current_color_box,
		 *color_selector_box;

static void button_color_callback(FL_OBJECT * dont_care, long which)
{
    DBG("Button color callback");
    current_color_scheme = which;
    fl_redraw_object(color_selector_box);
}

float tmp_vector[2];

#define DRAW_VBAR(foo) { \
		bgnpolygon(); \
		cpack(0xff000000); \
		tmp_vector[0] = (color_selector_box)->x + 2.0; \
		tmp_vector[1] = (ystart); \
		v2f(tmp_vector); \
		tmp_vector[0] += 8.0; \
		v2f(tmp_vector); \
		cpack(foo); \
		tmp_vector[1] = (yend)+1.0; \
		v2f(tmp_vector); \
		tmp_vector[0] = (color_selector_box)->x + 2.0; \
		v2f(tmp_vector); \
		endpolygon(); }

#define DRAW_HBAR(foo) { \
		bgnpolygon(); \
		cpack(0xff000000); \
		tmp_vector[0] = (xstart); \
		tmp_vector[1] = (color_selector_box)->y + 2.0; \
		v2f(tmp_vector); \
		tmp_vector[1] += 8.0; \
		v2f(tmp_vector); \
		cpack(foo); \
		tmp_vector[0] = (xend)+1.0; \
		v2f(tmp_vector); \
		tmp_vector[1] = (color_selector_box)->y + 2.0; \
		v2f(tmp_vector); \
		endpolygon(); }

static int current_color_handler(FL_OBJECT *obj, int event, float mx, float my, char key)
{
    int horiz_val, vert_val;

    int xstart, ystart, xend, yend;

    int old_shade_model = getsm();
    shademodel(GOURAUD);

    DBG("Current color handler");
    /*  
     *  first things, first, we put the current color in the appropriate
     *  box...
     */
    RGBcolor(new_red, new_green, new_blue);
    rectf(obj->x, obj->y, obj->x+obj->w, obj->y+obj->h);

    /*
     * these variables used in the DRAW_VBAR, and DRAW_HBAR macros
     */
    xstart = color_selector_box->x + 12;
    ystart = color_selector_box->y + 12;
    xend = color_selector_box->x + color_selector_box->w - 2;
    yend = color_selector_box->y + color_selector_box->h - 2;


    switch(current_color_scheme)
	{
	    case SCHEME_RED_BLUE:
		vert_val = new_blue;
		DRAW_VBAR(0xffff0000);  /* blue */
		horiz_val = new_red;
		DRAW_HBAR(0xff0000ff);  /* red */

		break;
	    case SCHEME_RED_GREEN:
		vert_val = new_green;
		DRAW_VBAR(0xff00ff00);  /* green */
		horiz_val = new_red;
		DRAW_HBAR(0xff0000ff);  /* red */

		break;
	    case SCHEME_GREEN_BLUE:
		vert_val = new_blue;
		DRAW_VBAR(0xffff0000);  /* blue */
		horiz_val = new_green;
		DRAW_HBAR(0xff00ff00);  /* green */

		break;
	}
    /*  
     *  now, we draw the new current color markers
     */
    ystart -= 12;

    if(horiz_val < 1) horiz_val = 1;
    if(vert_val < 1) vert_val = 1;

    RGBcolor(0, 0, 0);
    rectf(xstart+horiz_val-1, ystart + 2.5, xstart+horiz_val+1, ystart + 9);
    RGBcolor(255, 255, 255);
    move2(xstart+horiz_val, ystart + 2.5);
    draw2(xstart+horiz_val, ystart + 9);

    xstart -= 12;
    ystart += 12;
    RGBcolor(0, 0, 0);
    rectf(xstart + 2.5, ystart+vert_val-1, xstart + 9, ystart+vert_val+1);
    RGBcolor(255, 255, 255);
    move2(xstart + 2.5, ystart+vert_val);
    draw2(xstart + 9, ystart+vert_val);

    shademodel(old_shade_model);
    return FALSE;  /* the status of this object never really changes */
}

#define RED_INC		0x000001
#define GREEN_INC	0x000100
#define BLUE_INC	0x010000

static int color_selector_handler(FL_OBJECT *obj, int event, float mx, float my, char key)
{
    int xstart, xend, ystart, yend, i;
    static int mouse_down = FALSE;

    long color_nw = 0xff000000,
	 color_ne = 0xff000000,
	 color_sw = 0xff000000,
	 color_se = 0xff000000;

    static float corners[4][2]; static long first_time = TRUE;

    int old_shade_model = getsm();
    shademodel(GOURAUD);

    DBG("Color selector handler");
    switch(event)
    {
      case FL_DRAW:
	DBG("  Drawing...");
	xstart = obj->x + 12;
	ystart = obj->y + 12;
	xend = obj->x + obj->w - 2;
	yend = obj->y + obj->h - 2;

	/*
	 *  Draw all the boxes... not 100% efficient, but works with
	 *  existing code better -- all are in color 0 (but we don't
	 *  really care) and have two-pixel borders (about which we
	 *  DO care)
	 */
	
	fl_drw_box(FL_FRAME_BOX, obj->x, obj->y, obj->w, obj->h, 0, 2.0);
	fl_drw_box(FL_FRAME_BOX, obj->x, obj->y, 11.0, obj->h, 0, 2.0);
	fl_drw_box(FL_FRAME_BOX, obj->x, obj->y, obj->w, 11.0, 0, 2.0);
	fl_drw_box(FL_FRAME_BOX, obj->x, obj->y, 11.0, 11.0, 0, 2.0);

	/*
	 * first, do some general stuff, and draw the vertical stripes
	 */
	switch(current_color_scheme)
	{
	    case SCHEME_RED_BLUE:
		color_nw += new_green * GREEN_INC + 0xff * BLUE_INC;
		color_ne += new_green * GREEN_INC + 0xff * RED_INC + 0xff * BLUE_INC;
		color_sw += new_green * GREEN_INC;
		color_se += new_green * GREEN_INC + 0xff * RED_INC;


		/* fill box in the corner */
		RGBcolor(0, new_green, 0);
		rectf(obj->x + 2, obj->y + 2, obj->x + 9, obj->y + 9);

		break;
	    case SCHEME_RED_GREEN:
		color_nw += new_blue * BLUE_INC + 0xff * GREEN_INC;
		color_ne += new_blue * BLUE_INC + 0xff * RED_INC + 0xff * GREEN_INC;;
		color_sw += new_blue * BLUE_INC;
		color_se += new_blue * BLUE_INC + 0xff * RED_INC;

		/* fill box in the corner */
		RGBcolor(0, 0, new_blue);
		rectf(obj->x + 2, obj->y + 2, obj->x + 9, obj->y + 9);

		break;
	    case SCHEME_GREEN_BLUE:
		color_nw += new_red * RED_INC + 0xff * BLUE_INC;
		color_ne += new_red * RED_INC + 0xff * BLUE_INC + 0xff * GREEN_INC;;
		color_sw += new_red * RED_INC;
		color_se += new_red * RED_INC + 0xff * GREEN_INC;;

		/* fill box in the corner */
		RGBcolor(new_red, 0, 0);
		rectf(obj->x + 2, obj->y + 2, obj->x + 9, obj->y + 9);

		break;
	}

	if(first_time)
	{
	    first_time = FALSE;
	    corners[0][0] = xstart;   corners[0][1] = ystart;
	    corners[1][0] = xend+1.0; corners[1][1] = ystart;
	    corners[2][0] = xstart;   corners[2][1] = yend+1.0;
	    corners[3][0] = xend+1.0; corners[3][1] = yend+1.0;
	}
	bgntmesh();
	cpack(color_sw);
	v2f(corners[0]);
	cpack(color_se);
	v2f(corners[1]);
	cpack(color_nw);
	v2f(corners[2]);
	cpack(color_ne);
	v2f(corners[3]);
	endtmesh();

#ifdef DOESN_T_DO_QUITE_WHAT_WE_WANT
	fl_redraw_object(current_color_box);  /* need to get those
						 color handles right... */
#else
	current_color_handler(current_color_box, FL_DRAW, 0.0, 0.0, 0);
#endif

	break;
      case FL_PUSH:
	mouse_down = TRUE;  /* now fall through to the code to deal with it */
      case FL_MOVE:
      case FL_MOUSE:
	DBG("  Mousing...");
	if (!mouse_down) break;
	DBG("  Changing color...");
	switch(current_color_scheme)
	{
	  case SCHEME_RED_BLUE:
	    new_red = mx - obj->x - 12;
	    new_blue = my - obj->y - 12;
	    break;
	  case SCHEME_RED_GREEN:
	    new_red = mx - obj->x - 12;
	    new_green = my - obj->y - 12;
	    break;
	  case SCHEME_GREEN_BLUE:
	    new_green = mx - obj->x - 12;
	    new_blue = my - obj->y - 12;
	    break;
	  default:  /* shouldn't happen */
	    ;
	}

	/*  
	 *  If the mouse moves outside the color selector, we can
	 *  still "do the right thing."
	 */
	if(new_red < 0) new_red = 0;
	if(new_green < 0) new_green = 0;
	if(new_blue < 0) new_blue = 0;

	if(new_red > 255) new_red = 255;
	if(new_green > 255) new_green = 255;
	if(new_blue > 255) new_blue = 255;

	fl_redraw_object(current_color_box);
	break;

      case FL_RELEASE:
	mouse_down = FALSE;
	break;
      case FL_ENTER:
      case FL_LEAVE:
      default:        /* we don't care */
	;
    }
    shademodel(old_shade_model);
    return FALSE;
}

static void fl_color_selector_create(int dbuffer)
{
    static int created_already = 0;

    FL_OBJECT *obj;

    if (created_already)
    {
	color_selector_form->doublebuf = dbuffer;
	return;
    }
    created_already = TRUE;

    color_selector_form = fl_bgn_form(FL_NO_BOX, 470.0, 350.0);
    color_selector_form->doublebuf = dbuffer;

    obj = fl_add_box(FL_UP_BOX, 0.0, 0.0, 470.0, 350.0, "24-bit Color Selector");
    fl_set_object_align(obj, FL_ALIGN_TOP);
    obj = fl_add_text(FL_NORMAL_TEXT, 20.0, 290.0, 140.0, 40.0, "24-bit Color Selector");
    fl_set_object_boxtype(obj, FL_SHADOW_BOX);
    fl_set_object_align(obj, FL_ALIGN_CENTER);

    obj = fl_add_box(FL_FRAME_BOX, 180.0, 300.0, 270.0, 30.0, "");
    current_color_box = fl_add_free(FL_SLEEPING_FREE,
				    182.0, 302.0, 266.0, 26.0,
				    "",
				    current_color_handler);

    color_selector_box = fl_add_free(FL_NORMAL_FREE,
				    180.0, 20.0, 270.0, 270.0,
				    "",
				    color_selector_handler);

    button_cancel = obj = fl_add_button(FL_NORMAL_BUTTON, 20.0, 70.0, 140.0, 40.0, "Cancel");
    fl_set_object_color(obj, 47, 45);

    button_okay = obj = fl_add_button(FL_RETURN_BUTTON, 20.0, 20.0, 140.0, 40.0, "Okay");
    fl_set_object_color(obj, 47, 45);

    fl_bgn_group();
    button_red_blue = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 20.0, 230.0, 140.0, 40.0, "Red Vs. Blue");
    fl_set_object_boxtype(obj, FL_FRAME_BOX);
    fl_set_call_back(obj, button_color_callback, SCHEME_RED_BLUE);
    button_red_green = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 20.0, 180.0, 140.0, 40.0, "Red Vs. Green");
    fl_set_object_boxtype(obj, FL_FRAME_BOX);
    fl_set_call_back(obj, button_color_callback, SCHEME_RED_GREEN);
    button_green_blue = obj = fl_add_lightbutton(FL_RADIO_BUTTON, 20.0, 130.0, 140.0, 40.0, "Green Vs. Blue");
    fl_set_object_boxtype(obj, FL_FRAME_BOX);
    fl_set_call_back(obj, button_color_callback, SCHEME_GREEN_BLUE);
    fl_end_group();

    fl_end_form();
}

static void fl_show_color_selector_internal(int *red, int *green, int *blue, int dbuffer)
{
    FL_OBJECT *somebody;

    new_red = *red;
    new_green = *green;
    new_blue = *blue;

    fl_color_selector_create(dbuffer);
    fl_deactivate_all_forms();

    /*
     * press the red_blue button, first
     */
    fl_set_button(button_red_blue, TRUE);
    fl_set_button(button_red_green, FALSE);
    fl_set_button(button_green_blue, FALSE);
    button_color_callback(button_red_blue, SCHEME_RED_BLUE);

    fl_show_form(color_selector_form, FL_PLACE_MOUSE, FALSE, "24-bit Color Selector");
    for(;;)
    {
	somebody = fl_do_forms();
	if(somebody == button_okay)
	{
	    *red = new_red;
	    *green = new_green;
	    *blue = new_blue;
	    break;
	}
	if(somebody == button_cancel) break;


	/* otherwise, we don't care... */
    }

    fl_hide_form(color_selector_form);
    fl_activate_all_forms();
}

/*  
 *  shows a fancy 24-bit color selector, original concept by Dave Navas
 */
void fl_show_color_selector_sbuffer(int *red, int *green, int *blue)
{
    fl_show_color_selector_internal(red, green, blue, FALSE);
}

void fl_show_color_selector(int *red, int *green, int *blue)
{
    fl_show_color_selector_internal(red, green, blue, TRUE);
}

