/*

sshserver.h

  Authors:
        Tatu Ylnen <ylo@ssh.fi>
        Markku-Juhani Saarinen <mjos@ssh.fi>
        Timo J. Rinne <tri@ssh.fi>
        Sami Lehtinen <sjl@ssh.fi>

  Copyright (C) 1997-1998 SSH Communications Security Oy, Espoo, Finland
  All rights reserved.

SSH server functionality for processing a connection.

*/

#ifndef SSHSERVER_H
#define SSHSERVER_H

#include "sshcrypt.h"
#include "sshconfig.h"
#include "sshcommon.h"
#include "sshauth.h"
#include "sshconn.h"
#include "sshtrans.h"

/* Data type for an SSH server object.  The server object processes one
   connection (potentially multiple sessions/channels). */

struct SshServerRec
{
  /* Connection data for both client and server. */
  SshCommon common;

  /* Configuration options that apply to the server. */
  SshConfig config;
  
  /* Authentication methods. */
  SshAuthServerMethod *methods;
};

typedef struct SshServerRec *SshServer;

/* Takes a stream, and creates an SSH server for processing that
   connection.  This closes the stream and returns NULL (without
   calling the destroy function) if an error occurs.  This does not
   free the given server key.  The random state is required to stay
   valid until the server has been destroyed.  ``config'' must remain
   valid until the server is destroyed; it is not automatically freed.
     `stream'        the connection stream
     `config'        configuration data (not freed, must remain valid)
     `random_state'  random number generator state
     `private_server_key'   private key that changes every hour or NULL
     `disconnect'    function to call on disconnect
     `debug'         function to call on debug message (may be NULL)
     `version_check' version check callback (may be NULL)
     `context'       context to pass to the callbacks
   The object should be destroyed from the ``disconnect'' callback. */
SshServer ssh_server_wrap(SshStream stream, SshConfig config,
                          SshPrivateKey private_server_key,
                          SshConnDisconnectProc disconnect,
                          SshConnDebugProc debug,
                          SshVersionCallback version_check,
                          SshAuthPolicyProc auth_policy_proc,
                          SshCommonAuthenticationNotify authentication_notify,
                          void *context);

/* Forcibly destroys the given connection. */
void ssh_server_destroy(SshServer server);

#endif /* SSHSERVER_H */
