/*____________________________________________________________________________
	Copyright (C) 1996-1998 Network Associates, Inc. and its affiliates.
	All rights reserved.
	
	$Id: CBlindWriteQueue.cp,v 1.6.8.1 1998/11/12 03:05:33 heller Exp $
____________________________________________________________________________*/

#if BLIND_WRITES	// [

#include "MacMemory.h"
#include "MacFiles.h"

#include "CBlindWriteQueue.h"

#include "A4Stuff.h"
#include "SetupA4.h"



typedef union XIOPB
	{
	ParamBlockRec	pb;
	XIOParam		xpb;
	} XIOPB;

typedef struct RequestHeader
	{
#if PGP_DEBUG
	ulong				magic;
	enum { kMagic = 'BLWQ' };
#endif
	RequestHeader *		next;
	XIOPB				ioPB;
	CBlindWriteQueue *	thisObject;
	ulong				numBytes;
	} RequestHeader;
	
const ulong	kMinRequestSize	= sizeof( RequestHeader ) + 512;

	inline ulong
TotalSize( const RequestHeader *request )
	{
	return( sizeof( RequestHeader ) + request->numBytes );
	}
	
	inline RequestHeader *
FollowingRequest( const RequestHeader *request )
	{
	return( (RequestHeader *) ( ((Ptr) request ) + TotalSize( request ) ) );
	}
	
	inline Ptr
RequestBytes( RequestHeader *request )
	{
	return( (Ptr) request + sizeof( RequestHeader ) );
	}
	
#if PGP_DEBUG
	inline void
SetMagic( RequestHeader *request, ulong magic)
	{
	request->magic = magic;
	}
#else
#define SetMagic( r, m )
#endif
	

#if ! TARGET_RT_MAC_CFM
 
#pragma parameter __D0 InterruptsOff( )
extern pascal short	InterruptsOff( void )
		THREEWORDINLINE( 0x40C0, 0x007C, 0x0700 );
	
	
#pragma parameter RestoreInterrupts( __D0 )
extern pascal void		RestoreInterrupts( short sr )
		ONEWORDINLINE( 0x46C0 );

#endif

#define MAX( a, b ) 	( (a) >= (b) ? (a) : (b) )

	
	

	
CBlindWriteQueue::CBlindWriteQueue( ulong bufferSize )
	{
	const ulong	kMinBufferSize	= 1 * ( sizeof( RequestHeader ) +
			kDiskBlockSize );
	
	mGoingAway	= false;
	
	pgpClearMemory( &mFreeList, sizeof( mFreeList ) );
	
	mBufferSize	= MAX( bufferSize, kMinBufferSize );
	
	mBuffer		= pgpNewPtrMost( mBufferSize, kMinBufferSize,
					kMacMemory_UseSystemHeap, &mBufferSize );
	pgpAssert( IsntNull( mBuffer ) );
	
	if ( IsntNull( mBuffer ) )
		{
		CreateFreeList( );
		}
	}


CBlindWriteQueue::~CBlindWriteQueue(  )
	{
	mGoingAway	= TRUE;
	
	WaitIODone();
	
	if ( IsntNull( mBuffer ) )
		{
		pgpFreeMac( (Ptr)mBuffer );
		mBuffer	= nil;
		}
	}


	void
CBlindWriteQueue::CreateFreeList( )
	{
	RequestHeader *	request;
	
	pgpAssert( IsntNull( mBuffer ) );
	pgpAssert( mBufferSize >= sizeof( RequestHeader ) + kDiskBlockSize );
	
	// initially, we have one free item, with all the space in it
	request				= (RequestHeader *)mBuffer;
	request->numBytes	= mBufferSize - sizeof( RequestHeader );
	request->next		= nil;
	SetMagic( request, RequestHeader::kMagic );
	
	mFreeList.head			= request;
	mFreeList.maxFreeBytes	= request->numBytes;
	mFreeList.curFreeBytes	= request->numBytes;
	}
	
	
	RequestHeader *
CBlindWriteQueue::Allocate( ulong	size )
	{
	RequestHeader *	result	= nil;
	
	pgpAssert( mFreeList.curFreeBytes <= mFreeList.maxFreeBytes );
	
	if ( size <= mBufferSize - sizeof( RequestHeader ) )
		{
		RequestHeader *	curRequest	= nil;
		
		short saveSR	= InterruptsOff();
			
		curRequest	= mFreeList.head;
		if ( IsntNull( curRequest ) )
			{
			RequestHeader *	last	= nil;
			
			// NOTE: this loop should be fast...interrupts are off!
			while ( IsntNull( curRequest ) )
				{
				if ( curRequest->numBytes >= size )
					{
					ulong			leftover;
					RequestHeader *	newFreeItem = nil;
			
					// carve this request up if there's enough remaining
					leftover	= curRequest->numBytes - size;
					if ( leftover >= kMinRequestSize )
						{
						ulong	totalTaken	= size + sizeof( RequestHeader);
						
						pgpAssert( mFreeList.curFreeBytes >= totalTaken );
						
						newFreeItem	= (RequestHeader *) ( ( (Ptr)(curRequest +
											1 )) + size );
						newFreeItem->numBytes = curRequest->numBytes -
								totalTaken;
						newFreeItem->next		= curRequest->next;
						SetMagic( newFreeItem, RequestHeader::kMagic );
						
						mFreeList.curFreeBytes	-= totalTaken;
						curRequest->numBytes	= size;
						
						pgpAssert( newFreeItem->numBytes <=
								mFreeList.curFreeBytes );
						}
					else
						{
						newFreeItem				= curRequest->next;
						pgpAssert( mFreeList.curFreeBytes >=
								curRequest->numBytes );
						mFreeList.curFreeBytes	-= curRequest->numBytes;
						}
					
					
					result				= curRequest;
					
					if ( IsNull( last ) )
						mFreeList.head	= newFreeItem;
					else
						last->next		= newFreeItem;
					break;
					}
					
				last		= curRequest;
				curRequest	= curRequest->next;
				}
			}
			
		pgpAssert( mFreeList.curFreeBytes < mFreeList.maxFreeBytes );
	
		RestoreInterrupts( saveSR );
		}
	
	return( result );
	}
	

	void
CBlindWriteQueue::MergeWithNext( RequestHeader *	request)
	{
	RequestHeader *next;
		
	while ( (next = request->next ) == FollowingRequest( request ) )
		{
		// bytes themselves are already accounted for
		mFreeList.curFreeBytes	+= sizeof( RequestHeader );
		request->numBytes		+= sizeof( RequestHeader ) + next->numBytes;
		request->next			= next->next;
		SetMagic( next, 0 );
		}
	}
	
	
	void
CBlindWriteQueue::Deallocate( RequestHeader *request)
	{
	pgpAssertAddrValid( request, RequestHeader );
	pgpAssert( request->magic == RequestHeader::kMagic );
	pgpAssert( mFreeList.curFreeBytes < mFreeList.maxFreeBytes );
	
	// the request next in memory (may or may not exist if 'request' is last )
	request->next	= nil;
		
	short saveSR	= InterruptsOff();
					
	if ( IsNull( mFreeList.head ) )
		{
		// this is the only free item
		mFreeList.head			= request;
		mFreeList.curFreeBytes	= request->numBytes;
		}
	else
		{
		RequestHeader *	curRequest	= mFreeList.head;
		RequestHeader *	last	= nil;
		Boolean			didLinkIn	= false;
	
		mFreeList.curFreeBytes	+= request->numBytes;
				
		while ( IsntNull( curRequest ) )
			{
			if ( (long)request < (long)curRequest )
				{
				request->next			= curRequest;
				
				if ( IsntNull( last ) )
					{
					last->next	= request;
					MergeWithNext( last );
					}
				else
					{
					mFreeList.head	= request;
					MergeWithNext( request );
					}
				didLinkIn	= TRUE;
				break;
				}
				
			last		= curRequest;
			curRequest	= curRequest->next;
			}
			
		if ( ! didLinkIn )
			{
			// very last block in list 
			pgpAssert( IsntNull( last ) );
			last->next	= request;
			MergeWithNext( last );
			}
		}
	
	RestoreInterrupts( saveSR );
	}
	
	
/*_________________________________________________________________________
	An I/O request has been completed.
	This hook proc is called by Device Manager
_________________________________________________________________________*/
	pascal void
CBlindWriteQueue::sCompletionProc( void )
	{
	EnterCallback();
	
	XIOPB *			pb;
	RequestHeader *	request;
	
	pb	= (XIOPB *)GetA0();
	
	request	= (RequestHeader *)(((char *)pb) - offsetof( RequestHeader, ioPB ));
	pgpAssertAddrValid( request, RequestHeader );
	pgpAssert( request->magic == RequestHeader::kMagic );
	
	request->thisObject->Deallocate( request );
	
	ExitCallback();
	}



	CBlindWriteQueue::RequestResult
CBlindWriteQueue::HandleRequest( const ParamBlockRec * pb )
	{
	RequestResult	result	= kRequestNotHandled;
		
	if ( ! mGoingAway )
		{
		RequestHeader *	request;
		const ulong		requestCount	= pb->ioParam.ioReqCount;
		
		request	= Allocate( requestCount );
		if ( IsntNull( request ) )
			{
			pgpAssert( requestCount <= request->numBytes );
			
			// copy the pb and the bytes
			request->ioPB		= *(const XIOPB *)pb;
			request->thisObject	= this;
			request->ioPB.pb.ioParam.ioCompletion	= sCompletionProc;
			request->ioPB.pb.ioParam.ioBuffer		= RequestBytes( request );
			BlockMoveData( pb->ioParam.ioBuffer, RequestBytes( request ),
					requestCount );
			
			(void)PBWriteAsync( &request->ioPB.pb );
			
			result	= kRequestHandled;
			}
		}
	
	return( result );
	}
	
	

	void
CBlindWriteQueue::WaitIODone( void )
	{
	Boolean	notAllFree	= TRUE;
	
	while ( mFreeList.curFreeBytes != mFreeList.maxFreeBytes )
		{
		// no need to turn off interrupts
		// wait
		}
		
	}


	OSErr
CBlindWriteQueue::Create(
	CBlindWriteQueue ** writerPtr,
	ulong				size)
	{
	OSErr				err	= memFullErr;
	CBlindWriteQueue *	queue;
	
	*writerPtr	= nil;
	
	queue	= new CBlindWriteQueue( size );
	if ( IsntNull( queue ) )
		{
		if ( IsNull( queue->mBuffer ) )
			{
			delete queue;
			queue	= nil;
			}
		else
			{
			err			= noErr;
			*writerPtr	= queue;
			}
		}
	
	pgpAssert( err == noErr );
	
	return( err );
	}

#if PGP_DEBUG	// [

/*_________________________________________________________________________
	test allocation followed by deallocating every other even item followed by
	deallocating every other odd item
_________________________________________________________________________*/
	void
CBlindWriteQueue::TestOddEven( void )
	{
	const ushort		kNumRequests	= 500;
	RequestHeader *		requests[ kNumRequests ];
	CBlindWriteQueue	test;

	pgpAssert( IsntNull( test.mFreeList.head ) );
	pgpAssert( IsNull( test.mFreeList.head->next ) );
	pgpAssert( test.mFreeList.head->numBytes == test.mFreeList.maxFreeBytes );
	
	ulong	startFree	= test.mFreeList.curFreeBytes;
	
	for ( ushort index = 0; index < kNumRequests; ++index )
		{
		requests[ index ]	= test.Allocate( 512 );
		}

	// deallocate every even item
	for ( ushort index = 0; index < kNumRequests; index += 2 )
		{
		if ( IsntNull( requests[ index ] ) )
			{
			test.Deallocate( requests[ index ] );
			requests[ index ]	= nil;
			}
		}
		
	// deallocate every odd item
	for ( ushort index = 1; index < kNumRequests; index += 2 )
		{
		if ( IsntNull( requests[ index ] ) )
			{
			test.Deallocate( requests[ index ] );
			requests[ index ]	= nil;
			}
		}

	pgpAssert( IsntNull( test.mFreeList.head ) );
	pgpAssert( IsNull( test.mFreeList.head->next ) );
	pgpAssert( test.mFreeList.head->numBytes == test.mFreeList.maxFreeBytes );
	pgpAssert( startFree == test.mFreeList.curFreeBytes );
	}


/*_________________________________________________________________________
	test allocation followed by all possible orders of deallocation
_________________________________________________________________________*/
	void
CBlindWriteQueue::TestDeallocateCombinations( void )
	{
	CBlindWriteQueue	test( 3 * (sizeof( RequestHeader ) + 512 ) );
	ulong				startFree	= test.mFreeList.curFreeBytes;
	
	pgpAssert( IsntNull( test.mFreeList.head ) );
	pgpAssert( IsNull( test.mFreeList.head->next ) );
	pgpAssert( test.mFreeList.head->numBytes == test.mFreeList.maxFreeBytes );
	
	
	RequestHeader *request1;
	RequestHeader *request2;
	RequestHeader *request3;
	
	
	request1	= test.Allocate( 512 );	pgpAssert( IsntNull( request1 ) );
	request2	= test.Allocate( 512 );	pgpAssert( IsntNull( request2 ) );
	request3	= test.Allocate( 512 );	pgpAssert( IsntNull( request3 ) );
	test.Deallocate( request1 );
	test.Deallocate( request2 );
	test.Deallocate( request3 );
	
	
	request1	= test.Allocate( 512 );
	request2	= test.Allocate( 512 );
	request3	= test.Allocate( 512 );
	test.Deallocate( request1 );
	test.Deallocate( request3 );
	test.Deallocate( request2 );
	
	
	request1	= test.Allocate( 512 );
	request2	= test.Allocate( 512 );
	request3	= test.Allocate( 512 );
	test.Deallocate( request2 );
	test.Deallocate( request1 );
	test.Deallocate( request3 );
	
	
	request1	= test.Allocate( 512 );
	request2	= test.Allocate( 512 );
	request3	= test.Allocate( 512 );
	test.Deallocate( request2 );
	test.Deallocate( request3 );
	test.Deallocate( request1 );
	
	
	request1	= test.Allocate( 512 );
	request2	= test.Allocate( 512 );
	request3	= test.Allocate( 512 );
	test.Deallocate( request3 );
	test.Deallocate( request1 );
	test.Deallocate( request2 );
	
	
	request1	= test.Allocate( 512 );
	request2	= test.Allocate( 512 );
	request3	= test.Allocate( 512 );
	test.Deallocate( request3 );
	test.Deallocate( request2 );
	test.Deallocate( request1 );


	pgpAssert( IsntNull( test.mFreeList.head ) );
	pgpAssert( IsNull( test.mFreeList.head->next ) );
	pgpAssert( test.mFreeList.head->numBytes == test.mFreeList.maxFreeBytes );
	pgpAssert( startFree == test.mFreeList.curFreeBytes );
	}
	
	
	void
CBlindWriteQueue::Test( void )
	{
	TestOddEven();
	TestDeallocateCombinations();
	}
	
#endif	// ]
	
	
	

#endif	// ] BLIND_WRITES




















