// Base64InputStream.java

import java.io.*;

/**
 * Do RFCxxxx style conversion of ascii to binary data.
 * Based on base64.cpp in Wei Dai's Crypto10 library.
 * @see Base64OutputStream
 */
public class Base64InputStream extends FilterInputStream {
static final int CONV_WHITE = -1;
static final int CONV_PAD = -2;
static final int CONV_OTHER = -3;

/**
 * Underlying input stream.  Needs to have pushback capability.
 */
InputStream is;

/**
 * Read a checksum afterwards, for compatibility with PGP
 */
boolean checksum;

/**
 * Accumulated checksum, if requested
 */
PGPBase64CRC cksum;

/**
 * Input data buffer
 */
byte inBuf[];

/**
 * Bytes in input data buffer
 */
int inBufSize;

/**
 * Output data buffer
 */
byte outBuf[];

/**
 * Byte index into output data buffer
 */
int outBufSize;

/**
 * Total bytes in output data buffer, always 3 except for last chunk
 */
int outBufMax;

/**
 * Flag for having read all the formatted data
 */
boolean closed;

/** 
 * Constructore for Base64InputStream
 * @param is underlying InputStream.  If it is a PushBackInputStream then when this
 * stream returns EOF, characters can be read from is immediately after the end of the
 * base64 portion.  Otherwise one character beyond the base64 portion will be consumed.
 * (Whitespace beyond the base64 portion will also be consumed in either case).
 * @param checksum true to look for and check a PGP-style checksum immediately after the
 * base64 portion.
 */
public Base64InputStream (InputStream is, boolean checksum)
{
    super(is);
    this.is = is;
    this.checksum = checksum;
    inBuf = new byte[4];
    outBuf = new byte[3];
    outBufMax = 3;
    inBufSize = 0;
    outBufSize = 0;
    closed = false;
    if (checksum)
	cksum = new PGPBase64CRC();
}

/* Parse input in fours, producing three bytes to outBuf.  Should only see whitespace
 * or non-legal characters on four-character boundaries.  If checksum is set, when we
 * see "=" as the first character of a four set, we read the next four and compare with
 * our checksum.  If get EOF error at a bad spot, or other bad input, throw IOException.
 * When get first non-legal character (on a four byte boundary), push it back if possible
 * and return EOF.
 */
public int read() throws IOException
{
    if (outBufSize == 0) {
        // Must read another set of four bytes.  Check first to see if we are at end.
        if (closed)
            return -1;
        int inByte = 0;
        int n = CONV_WHITE;
        while (n==CONV_WHITE) { // skip whitespace
            inByte = is.read();
            if (inByte < 0)
                return -1;
            n = ConvToNumber(inByte);
        }
        boolean reading_checksum = false;
        if (n<0) {  // checksum character or other
            if (!checksum || n==CONV_OTHER) {
                if (is instanceof PushbackInputStream)
                    ((PushbackInputStream)is).unread(inByte);
                return -1;
            }
            reading_checksum = true;
            inByte = is.read(); 
            if (inByte < 0)
                throw(new IOException("Premature EOF"));
            n = ConvToNumber(inByte);
        }
        for (int i=0; i<4; ++i) {
            if (n==CONV_PAD) {
                if (i<2)
                    throw(new IOException("Unexpected character in input: "+inByte));
            } else if (n<0) {
                throw(new IOException("Unexpected character in input: "+inByte));
            } else {
                inBuf[inBufSize++]=(byte) n;
            }
            if (i!=3) {
                inByte = is.read();
                if (inByte < 0)
                    throw(new IOException("Premature EOF"));
                n = ConvToNumber(inByte);
            }
        }
        DecodeQuantum();
        if (reading_checksum) {
            closed = true;
            int cksfile = ((outBuf[0]<<16)&0xff0000) | ((outBuf[1]<<8)&0xff00) |
                                ((int)(outBuf[2])&0xff);
            if (cksfile != cksum.getCRC() || outBufMax!=3)
                throw(new IOException("Checksum incorrect"));
            else
                return -1;
        }
    }
    byte b = outBuf[outBufSize++];
    if (outBufSize==outBufMax)
        outBufSize = 0;
    if (checksum)
	cksum.CRCByte((int)b&0xff);
    return (int)b & 0xff;
}

public int read(byte[] b) throws IOException
{
    return (read(b, 0, b.length));
}

public int read(byte[] b, int off, int len) throws IOException
{
    for (int i=0; i<len; ++i) {
        int c = read();
        if (c < 0) {
            if (i==0)
                return -1;
            else
                return i;
        }
        b[off+i] = (byte) c;
    }
    return len;
}

public boolean markSupported()
{
    return false;
}

/**
 * It is not practical to predict how many bytes will be available, since we remove
 * whitespace.  So always return 0.
 */
public int available() throws IOException
{
    return 0;
}

public void close() throws IOException
{
    closed = true;
    outBufSize = 0;
    super.close();
}

protected void DecodeQuantum()
{
    byte out;

    outBufMax = 0;

    out = (byte)((inBuf[0] << 2) | (inBuf[1] >> 4));
    outBuf[outBufMax++] = out;

    out = (byte)((inBuf[1] << 4) | (inBuf[2] >> 2));
    if (inBufSize > 2)
        outBuf[outBufMax++] = out;

    out = (byte)((inBuf[2] << 6) | inBuf[3]);
    if (inBufSize > 3)
        outBuf[outBufMax++] = out;

    inBufSize=0;
}

protected final int ConvToNumber(int inByte)
{
    if (inByte >= 'A' && inByte <= 'Z')
        return (inByte - 'A');

    if (inByte >= 'a' && inByte <= 'z')
        return (inByte - 'a' + 26);

    if (inByte >= '0' && inByte <= '9')
        return (inByte - '0' + 52);

    if (inByte == '+')
        return (62);

    if (inByte == '/')
        return (63);
    
    if (inByte == '=')
        return CONV_PAD;
    
    if (inByte=='\n' || inByte=='\r' || inByte==' ' || inByte=='\t')
        return CONV_WHITE;
    
    return CONV_OTHER;
}


}

