/*
 * Copyright (c) 1998 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Kungliga Tekniska
 *      Hgskolan and its contributors.
 * 
 * 4. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#include "milko_locl.h"

RCSID("$Id: volcommon.c,v 1.2 1998/10/07 22:11:48 lha Exp $");

static Hashtab *fshashid = NULL;
static Hashtab *fshashname = NULL;

static int 
fshashidcmp(void *a, void *b)
{
    struct mvolume *v1 = (struct mvolume *) a;
    struct mvolume *v2 = (struct mvolume *) b;
    return v1->volumeid != v2->volumeid;
}

static int 
fshashnamecmp(void *a, void *b)
{
    struct mvolume *v1 = (struct mvolume *) a;
    struct mvolume *v2 = (struct mvolume *) b;
    return !strcmp(v1->name, v2->name);
}

static unsigned int 
fshashidhash(void *a)
{
    struct mvolume *v1 = (struct mvolume *) a;
    return v1->volumeid;
}

static unsigned int 
fshashnamehash(void *a)
{
    struct mvolume *v1 = (struct mvolume *) a;
    return hashadd(v1->name);
}


int
volser_init(int hashsize) 
{
    fshashid = hashtabnew(100, fshashidcmp, fshashidhash);
    if (fshashid == NULL) 
	errx(1, "main: hashtabnew: fshashid");

    fshashname = hashtabnew(100, fshashnamecmp, fshashnamehash);
    if (fshashname == NULL) 
	errx(1, "main: hashtabnew: fshashname");


    /* CONFIG: config file would be nice */

#if 0
    volser_addVolume(&rom_filesystem, 1 /*volid */ , "root.cell",  "/usr/arla/db/obsdcdrom", "obsd cdrom");
#endif
#if 0
    volser_addVolume(&ufs_filesystem, 1 /*volid */ , "root.cell",  "/var", "my-var");
#endif
#if 1
    volser_addVolume(&ontop_filesystem, 1 /*volid */ , "root.cell",  "/vicepa:1", "root.cell");
#endif

    return 0;
}

struct mvolume *volser_getByName(char *name)
{
    struct mvolume vol;
    struct mvolume *vol2;

    assert(name);

    vol.name = name;
    vol2 = hashtabsearch(fshashname, &vol);
    if (vol2)
	vol2->refcount++;
    return vol2;
}

struct mvolume *volser_getById(int32_t id)
{
    struct mvolume vol;
    struct mvolume *vol2;

    vol.volumeid = id;
    vol2 = hashtabsearch(fshashid, &vol);
    if (vol2)
	vol2->refcount++;

    return vol2;
}

int volser_addVolume(struct filesystem *fs, int32_t id, 
		     char *name, char *conf, char *comment)
{
    struct mvolume *vol = NULL;
    int error;

    vol = calloc(sizeof(struct mvolume), 1);
    if (vol == NULL)
	goto error;

    vol->fs = fs ;
    vol->name = strdup(name);
    if (vol->name == NULL)
	goto error;
    
    vol->conf = strdup(conf);
    if (vol->conf == NULL)
	goto error;

    vol->comment = strdup(comment);
    if (vol->comment == NULL)
	goto error;

    vol->volumeid = id;


    error = vol->fs->fs_init(vol->conf, &vol->data);
    if (error) {
	fprintf(stderr, "Error when initing volume %s\n", vol->name);
	free(vol);
	return EINVAL;
    }

    vol->name = strdup(name);
    vol->conf = strdup(conf);
    vol->volumeid = id;
    hashtabadd(fshashname, vol);
    hashtabadd(fshashid, vol);
    return 0;
error:
    if (vol) {
	if (vol->name)
	    free(vol->name);
	if (vol->conf)
	    free(vol->conf);
	if (vol->comment)
	    free(vol->comment);
	free(vol);
    }

    return errno;
}

static void
zappme(struct mvolume *vol)
{
    assert(vol);
    assert(vol->refcount == 0);

    if (vol->name)
	free(vol->name);
    if (vol->conf)
	free(vol->conf);
    if (vol->fs && vol->data)
	vol->fs->fs_close(vol, 0);
    hashtabdel(fshashname, vol);
    hashtabdel(fshashid, vol);
    free(vol);
}


int volser_removeVolume(struct mvolume *volume)
{
    assert(volume);

    if (volume->refcount != 0) {
	volume->killme = 1;
	return 0;
    }

    zappme(volume);
    return 0;
	
}

void
volser_release(struct mvolume *vol)
{
    assert(vol);

    assert(vol->refcount);
    vol->refcount--;

    if (vol->killme)
	zappme(vol);
}
