/*
This code is copyright (C) 1998 Robert O'Callahan.
This code is free for non-commercial use. Modification in all forms is permitted.
This license continues to apply to any modified versions. This license text must be
reproduced and distributed with any modified versions.
As a matter of courtesy I (Robert O'Callahan) would like to be informed of
any potentially useful modifications.
*/

#include "ttssh.h"
#include "util.h"

#include <io.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <rsa.h>

#include "resource.h"

static void destroy_malloced_string(char FAR * * str) {
  if (*str != NULL) {
    memset(*str, 0, strlen(*str));
    free(*str);
    *str = NULL;
  }
}

void AUTH_init(struct _TInstVar FAR * pvar) {
  pvar->auth_state.user = NULL;
  pvar->auth_state.cur_cred.method = SSH_AUTH_NONE;
  pvar->auth_state.failed_method = SSH_AUTH_NONE;
  pvar->auth_state.auth_dialog = NULL;
}

static char ID_string[] = "SSH PRIVATE KEY FILE FORMAT 1.1\n";

static BIGNUM FAR * get_bignum(unsigned char FAR * bytes) {
  int bits = get_ushort16_MSBfirst(bytes);

  return BN_bin2bn(bytes + 2, (bits + 7)/8, NULL);
}

static RSA FAR * read_private_key(struct _TInstVar FAR * pvar, char FAR * relative_name,
                                  char FAR * passphrase, BOOL FAR * invalid_passphrase) {
  char filename[2048];
  int fd;
  unsigned int length, amount_read;
  unsigned char * keyfile_data;
  unsigned int index;
  int cipher;
  RSA FAR * key;
  unsigned int E_index, N_index, D_index, U_index, P_index, Q_index = 0;

  *invalid_passphrase = FALSE;

  get_teraterm_dir_relative_name(filename, sizeof(filename), relative_name);

  fd = _open(filename, _O_RDONLY | _O_SEQUENTIAL | _O_BINARY);
  if (fd == -1) {
    if (errno == ENOENT) {
      notify_nonfatal_error(pvar, "An error occurred while trying to read the key file.\n"
        "The specified filename does not exist.");
    } else {
      notify_nonfatal_error(pvar, "An error occurred while trying to read the key file.");
    }
    return NULL;
  }
  
  length = (int)_lseek(fd, 0, SEEK_END);
  _lseek(fd, 0, SEEK_SET);

  if (length >= 0 && length < 0x7FFFFFFF) {
    keyfile_data = malloc(length + 1);
    if (keyfile_data == NULL) {
      notify_nonfatal_error(pvar, "Memory ran out while trying to allocate space to read the key file.");
      _close(fd);
      return NULL;
    }
  } else {
    notify_nonfatal_error(pvar, "An error occurred while trying to read the key file.");
    _close(fd);
    return NULL;
  }

  amount_read = _read(fd, keyfile_data, length);
  /* terminate it with 0 so that the strncmp below is guaranteed not to
     crash */
  keyfile_data[length] = 0;
      
  _close(fd);

  if (amount_read != length) {
    notify_nonfatal_error(pvar, "An error occurred while trying to read the key file.");
    free(keyfile_data);
    return NULL;
  }

  if (strcmp(keyfile_data, ID_string) != 0) {
    notify_nonfatal_error(pvar, "The specified key file does not contain an SSH private key.");
    free(keyfile_data);
    return NULL;
  }

  index = NUM_ELEM(ID_string);

  if (length < index + 9) {
    notify_nonfatal_error(pvar, "The specified key file has been truncated and does not contain a valid private key.");
    free(keyfile_data);
    return NULL;
  }

  cipher = keyfile_data[index];
  /* skip reserved int32, public key bits int32 */
  index += 9;
  /* skip public key e and n mp_ints */
  if (length < index + 2) {
    notify_nonfatal_error(pvar, "The specified key file has been truncated and does not contain a valid private key.");
    free(keyfile_data);
    return NULL;
  }
  N_index = index;
  index += (get_ushort16_MSBfirst(keyfile_data + index) + 7)/8 + 2;
  if (length < index + 2) {
    notify_nonfatal_error(pvar, "The specified key file has been truncated and does not contain a valid private key.");
    free(keyfile_data);
    return NULL;
  }
  E_index = index;
  index += (get_ushort16_MSBfirst(keyfile_data + index) + 7)/8 + 2;
  /* skip comment */
  if (length < index + 4) {
    notify_nonfatal_error(pvar, "The specified key file has been truncated and does not contain a valid private key.");
    free(keyfile_data);
    return NULL;
  }
  index += get_uint32_MSBfirst(keyfile_data + index) + 4;

  if (length < index + 6) {
    notify_nonfatal_error(pvar, "The specified key file has been truncated and does not contain a valid private key.");
    free(keyfile_data);
    return NULL;
  }
  if (cipher != SSH_CIPHER_NONE) {
    if ((length - index) % 8 != 0) {
      notify_nonfatal_error(pvar, "The specified key file cannot be decrypted using the passphrase.\n"
        "The file does not have the correct length.");
      free(keyfile_data);
      return NULL;
    }
    if (!CRYPT_passphrase_decrypt(cipher, passphrase, keyfile_data + index, length - index)) {
      notify_nonfatal_error(pvar, "The specified key file cannot be decrypted using the passphrase.\n"
        "The cipher type used to encrypt the file is not supported by TTSSH for this purpose.");
      free(keyfile_data);
      return NULL;
    }
  }

  if (keyfile_data[index] != keyfile_data[index + 2]
    || keyfile_data[index + 1] != keyfile_data[index + 3]) {
    *invalid_passphrase = TRUE;
    notify_nonfatal_error(pvar, "The specified key file cannot be decrypted using the passphrase.\n"
      "The passphrase is incorrect.");
    memset(keyfile_data, 0, length);
    free(keyfile_data);
    return NULL;
  }
  index += 4;

  D_index = index;
  if (length >= D_index + 2) {
    U_index = D_index + (get_ushort16_MSBfirst(keyfile_data + D_index) + 7)/8 + 2;
    if (length >= U_index + 2) {
      P_index = U_index + (get_ushort16_MSBfirst(keyfile_data + U_index) + 7)/8 + 2;
      if (length >= P_index + 2) {
        Q_index = P_index + (get_ushort16_MSBfirst(keyfile_data + P_index) + 7)/8 + 2;
      }
    }
  }
  if (Q_index == 0 || length < Q_index + (get_ushort16_MSBfirst(keyfile_data + Q_index) + 7)/8 + 2) {
    notify_nonfatal_error(pvar, "The specified key file has been truncated and does not contain a valid private key.");
    memset(keyfile_data, 0, length);
    free(keyfile_data);
    return NULL;
  }

  key = RSA_new();
  key->n = get_bignum(keyfile_data + N_index);
  key->e = get_bignum(keyfile_data + E_index);
  key->d = get_bignum(keyfile_data + D_index);
  key->p = get_bignum(keyfile_data + P_index);
  key->q = get_bignum(keyfile_data + Q_index);

  memset(keyfile_data, 0, length);
  free(keyfile_data);
  return key;
}

static LRESULT CALLBACK password_wnd_proc(HWND control, UINT msg, WPARAM wParam, LPARAM lParam) {
  char chars[2];

  switch (msg) {
  case WM_CHAR:
    if ((GetKeyState(VK_CONTROL) & 0x8000) != 0) {
      chars[0] = (char)wParam;
      chars[1] = 0;
      SendMessage(control, EM_REPLACESEL, (WPARAM)TRUE, (LPARAM)(char FAR *)chars);
      return 0;
    }
  }

  return CallWindowProc((WNDPROC)GetWindowLong(control, GWL_USERDATA),
    control, msg, wParam, lParam);
}

static void set_auth_options_status(HWND dlg, WORD item) {
  BOOL RSA_enabled = item == IDC_SSHUSERSA;
  BOOL rhosts_enabled = item == IDC_SSHUSERHOSTS;
  int i;

  CheckRadioButton(dlg, IDC_SSHUSEPASSWORD, IDC_SSHUSERHOSTS, item);

  for (i = IDC_CHOOSERSAFILE; i <= IDC_RSAFILENAME; i++) {
    EnableWindow(GetDlgItem(dlg, i), RSA_enabled);
  }

  for (i = IDC_LOCALUSERNAMELABEL; i <= IDC_HOSTRSAFILENAME; i++) {
    EnableWindow(GetDlgItem(dlg, i), rhosts_enabled);
  }
}

static void init_auth_dlg(struct _TInstVar FAR * pvar, HWND dlg) {
  char buf[1024] = "Logging in to ";
  HWND passwordControl = GetDlgItem(dlg, IDC_SSHPASSWORD);
  int default_method = pvar->session_settings.DefaultAuthMethod;

  if (pvar->auth_state.failed_method != SSH_AUTH_NONE) {
    /* must be retrying a failed attempt */
    SetDlgItemText(dlg, IDC_SSHAUTHBANNER2, "Authentication failed. Please retry.");
    SetWindowText(dlg, "Retrying SSH Authentication");
    default_method = pvar->auth_state.failed_method;
  }

  switch (default_method) {
  case SSH_AUTH_RSA:
    set_auth_options_status(dlg, IDC_SSHUSERSA);
    break;
  case SSH_AUTH_RHOSTS:
  case SSH_AUTH_RHOSTS_RSA:
    set_auth_options_status(dlg, IDC_SSHUSERHOSTS);
    break;
  case SSH_AUTH_PASSWORD:
  default:
    set_auth_options_status(dlg, IDC_SSHUSEPASSWORD);
  }

  if (pvar->auth_state.user != NULL) {
    SetDlgItemText(dlg, IDC_SSHUSERNAME, pvar->auth_state.user);
    SetFocus(passwordControl);
    EnableWindow(GetDlgItem(dlg, IDC_SSHUSERNAME), FALSE);
    EnableWindow(GetDlgItem(dlg, IDC_SSHUSERNAMELABEL), FALSE);
  } else if (pvar->session_settings.DefaultUserName[0] != 0) {
    SetDlgItemText(dlg, IDC_SSHUSERNAME, pvar->session_settings.DefaultUserName);
    SetFocus(passwordControl);
  } else {
    SetFocus(GetDlgItem(dlg, IDC_SSHUSERNAME));
  }

  SetDlgItemText(dlg, IDC_RSAFILENAME, pvar->session_settings.DefaultRSAPrivateKeyFile);
  SetDlgItemText(dlg, IDC_HOSTRSAFILENAME, pvar->session_settings.DefaultRhostsHostPrivateKeyFile);
  SetDlgItemText(dlg, IDC_LOCALUSERNAME, pvar->session_settings.DefaultRhostsLocalUserName);
  
  if (strlen(buf) + strlen(SSH_get_host_name(pvar)) < sizeof(buf) - 2) {
    strcat(buf, SSH_get_host_name(pvar));
  }
  SetDlgItemText(dlg, IDC_SSHAUTHBANNER, buf);

  SetWindowLong(passwordControl, GWL_USERDATA,
    SetWindowLong(passwordControl, GWL_WNDPROC, (LONG)password_wnd_proc));
}

static char FAR * alloc_control_text(HWND ctl) {
  int len = GetWindowTextLength(ctl);
  char FAR * result = malloc(len + 1);

  if (result != NULL) {
    GetWindowText(ctl, result, len + 1);
    result[len] = 0;
  }

  return result;
}

static int get_key_file_name(HWND parent, char FAR * buf, int bufsize) {
#ifdef TERATERM32
  OPENFILENAME params;
  char fullname_buf[2048] = "identity";

  params.lStructSize = sizeof(OPENFILENAME);
  params.hwndOwner = parent;
  params.lpstrFilter = NULL;
  params.lpstrCustomFilter = NULL;
  params.nFilterIndex = 0;
  buf[0] = 0;
  params.lpstrFile = fullname_buf;
  params.nMaxFile = sizeof(fullname_buf);
  params.lpstrFileTitle = NULL;
  params.lpstrInitialDir = NULL;
  params.lpstrTitle = "Choose a file with the RSA private key";
  params.Flags = OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST | OFN_HIDEREADONLY;
  params.lpstrDefExt = NULL;

  if (GetOpenFileName(&params) != 0) {
    copy_teraterm_dir_relative_path(buf, bufsize, fullname_buf);
    return 1;
  } else {
    return 0;
  }
#else
  return 0;
#endif
}

static void choose_RSA_key_file(HWND dlg) {
  char buf[1024];

  if (get_key_file_name(dlg, buf, sizeof(buf))) {
    SetDlgItemText(dlg, IDC_RSAFILENAME, buf);
  }
}

static void choose_host_RSA_key_file(HWND dlg) {
  char buf[1024];

  if (get_key_file_name(dlg, buf, sizeof(buf))) {
    SetDlgItemText(dlg, IDC_HOSTRSAFILENAME, buf);
  }
}

static BOOL end_auth_dlg(struct _TInstVar FAR * pvar, HWND dlg) {
  int method = SSH_AUTH_PASSWORD;
  char FAR * password = alloc_control_text(GetDlgItem(dlg, IDC_SSHPASSWORD));
  RSA FAR * key;

  if (IsDlgButtonChecked(dlg, IDC_SSHUSERSA)) {
    method = SSH_AUTH_RSA;
  } else if (IsDlgButtonChecked(dlg, IDC_SSHUSERHOSTS)) {
    if (GetWindowTextLength(GetDlgItem(dlg, IDC_HOSTRSAFILENAME)) > 0) {
      method = SSH_AUTH_RHOSTS_RSA;
    } else {
      method = SSH_AUTH_RHOSTS;
    }
  }
 
  if (method == SSH_AUTH_RSA || method == SSH_AUTH_RHOSTS_RSA) {
    char buf[2048];
    int file_ctl_ID = method == SSH_AUTH_RSA ? IDC_RSAFILENAME : IDC_HOSTRSAFILENAME;

    buf[0] == 0;
    GetDlgItemText(dlg, file_ctl_ID, buf, sizeof(buf));
    if (buf[0] == 0) {
      notify_nonfatal_error(pvar, "You must specify a file containing the RSA private key.");
      SetFocus(GetDlgItem(dlg, file_ctl_ID));
      free(password);
      return FALSE;
    } else {
      BOOL invalid_passphrase;

      key = read_private_key(pvar, buf, password, &invalid_passphrase);

      if (key == NULL) {
        if (invalid_passphrase) {
          SetFocus(GetDlgItem(dlg, IDC_SSHPASSWORD));
        } else {
          SetFocus(GetDlgItem(dlg, file_ctl_ID));
        }
        free(password);
        return FALSE;
      }
    }
  }

  /* from here on, we cannot fail, so just munge cur_cred in place */
  pvar->auth_state.cur_cred.method = method;
  if (method == SSH_AUTH_RSA || method == SSH_AUTH_RHOSTS_RSA) {
    pvar->auth_state.cur_cred.RSA_key = key;
  }
  /* we can't change the user name once it's set. It may already have
     been sent to the server, and it can only be sent once. */
  if (pvar->auth_state.user == NULL) {
    pvar->auth_state.user = alloc_control_text(GetDlgItem(dlg, IDC_SSHUSERNAME));
  }
  if (method == SSH_AUTH_PASSWORD) {
    pvar->auth_state.cur_cred.password = password;
  } else {
    memset(password, 0, strlen(password));
    free(password);
  }
  if (method == SSH_AUTH_RHOSTS || method == SSH_AUTH_RHOSTS_RSA) {
    pvar->auth_state.cur_cred.rhosts_client_user =
      alloc_control_text(GetDlgItem(dlg, IDC_LOCALUSERNAME));
  }
  pvar->auth_state.auth_dialog = NULL;

  GetDlgItemText(dlg, IDC_RSAFILENAME, pvar->session_settings.DefaultRSAPrivateKeyFile, sizeof(pvar->session_settings.DefaultRSAPrivateKeyFile));
  GetDlgItemText(dlg, IDC_HOSTRSAFILENAME, pvar->session_settings.DefaultRhostsHostPrivateKeyFile, sizeof(pvar->session_settings.DefaultRhostsHostPrivateKeyFile));
  GetDlgItemText(dlg, IDC_LOCALUSERNAME, pvar->session_settings.DefaultRhostsLocalUserName, sizeof(pvar->session_settings.DefaultRhostsLocalUserName));

  SSH_notify_user_name(pvar);
  SSH_notify_cred(pvar);
  
  EndDialog(dlg, 1);
  return TRUE;
}

static BOOL CALLBACK auth_dlg_proc(HWND dlg, UINT msg, WPARAM wParam, LPARAM lParam) {
  struct _TInstVar FAR * pvar;

  switch (msg) {
  case WM_INITDIALOG:
    pvar = (struct _TInstVar FAR *)lParam;
    pvar->auth_state.auth_dialog = dlg;
    SetWindowLong(dlg, DWL_USER, lParam);

    init_auth_dlg(pvar, dlg);
    return FALSE; /* because we set the focus */

  case WM_COMMAND:
    pvar = (struct _TInstVar FAR *)GetWindowLong(dlg, DWL_USER);

    switch (LOWORD(wParam)) {
    case IDOK:
      return end_auth_dlg(pvar, dlg);
      
    case IDCANCEL: /* kill the connection */
      pvar->auth_state.auth_dialog = NULL;
      notify_closed_connection(pvar);
      EndDialog(dlg, 0);
      return TRUE;

    case IDC_SSHUSEPASSWORD:
    case IDC_SSHUSERSA:
    case IDC_SSHUSERHOSTS:
      set_auth_options_status(dlg, LOWORD(wParam));
      return TRUE;

    case IDC_CHOOSERSAFILE:
      choose_RSA_key_file(dlg);
      return TRUE;

    case IDC_CHOOSEHOSTRSAFILE:
      choose_host_RSA_key_file(dlg);
      return TRUE;

    default:
      return FALSE;
    }

  default:
    return FALSE;
  }
}

char FAR * AUTH_get_user_name(struct _TInstVar FAR * pvar) {
  return pvar->auth_state.user;
}

int AUTH_set_supported_auth_types(struct _TInstVar FAR * pvar, int types) {
  types &= (SSH_AUTH_PASSWORD << 1);
  pvar->auth_state.supported_types = types;

  if (types == 0) {
    notify_fatal_error(pvar, "Server does not support password authentication.\n"
      "This connection will now close.");
    return 0;
  } else {
    return 1;
  }
}

void AUTH_advance_to_next_cred(struct _TInstVar FAR * pvar) {
  PostMessage(pvar->NotificationWindow, WM_COMMAND, (WPARAM)ID_SSHAUTH, (LPARAM)NULL);
  pvar->auth_state.failed_method = pvar->auth_state.cur_cred.method;
  pvar->auth_state.cur_cred.method = SSH_AUTH_NONE;
}

void AUTH_do_cred_dialog(struct _TInstVar FAR * pvar) {
  if (pvar->auth_state.auth_dialog == NULL) {
    HWND cur_active = GetActiveWindow();

    if (DialogBoxParam(hInst, MAKEINTRESOURCE(IDD_SSHAUTH),
      cur_active != NULL ? cur_active : pvar->NotificationWindow, auth_dlg_proc, (LPARAM)pvar) == -1) {
      notify_fatal_error(pvar, "Unable to display authentication dialog box.\n"
        "Connection terminated.");
    }
  }
}

static void init_default_auth_dlg(struct _TInstVar FAR * pvar, HWND dlg) {
  switch (pvar->settings.DefaultAuthMethod) {
  case SSH_AUTH_RSA:
    CheckRadioButton(dlg, IDC_SSHUSEPASSWORD, IDC_SSHUSERHOSTS, IDC_SSHUSERSA);
    break;
  case SSH_AUTH_RHOSTS:
  case SSH_AUTH_RHOSTS_RSA:
    CheckRadioButton(dlg, IDC_SSHUSEPASSWORD, IDC_SSHUSERHOSTS, IDC_SSHUSERHOSTS);
    break;
  case SSH_AUTH_PASSWORD:
  default:
    CheckRadioButton(dlg, IDC_SSHUSEPASSWORD, IDC_SSHUSERHOSTS, IDC_SSHUSEPASSWORD);
  }

  SetDlgItemText(dlg, IDC_SSHUSERNAME, pvar->settings.DefaultUserName);
  SetDlgItemText(dlg, IDC_RSAFILENAME, pvar->settings.DefaultRSAPrivateKeyFile);
  SetDlgItemText(dlg, IDC_HOSTRSAFILENAME, pvar->settings.DefaultRhostsHostPrivateKeyFile);
  SetDlgItemText(dlg, IDC_LOCALUSERNAME, pvar->settings.DefaultRhostsLocalUserName);
}

static BOOL end_default_auth_dlg(struct _TInstVar FAR * pvar, HWND dlg) {
  if (IsDlgButtonChecked(dlg, IDC_SSHUSERSA)) {
    pvar->settings.DefaultAuthMethod = SSH_AUTH_RSA;
  } else if (IsDlgButtonChecked(dlg, IDC_SSHUSERHOSTS)) {
    if (GetWindowTextLength(GetDlgItem(dlg, IDC_HOSTRSAFILENAME)) > 0) {
      pvar->settings.DefaultAuthMethod = SSH_AUTH_RHOSTS_RSA;
    } else {
      pvar->settings.DefaultAuthMethod = SSH_AUTH_RHOSTS;
    }
  } else {
    pvar->settings.DefaultAuthMethod = SSH_AUTH_PASSWORD;
  }

  GetDlgItemText(dlg, IDC_SSHUSERNAME, pvar->settings.DefaultUserName, sizeof(pvar->settings.DefaultUserName));
  GetDlgItemText(dlg, IDC_RSAFILENAME, pvar->settings.DefaultRSAPrivateKeyFile, sizeof(pvar->settings.DefaultRSAPrivateKeyFile));
  GetDlgItemText(dlg, IDC_HOSTRSAFILENAME, pvar->settings.DefaultRhostsHostPrivateKeyFile, sizeof(pvar->settings.DefaultRhostsHostPrivateKeyFile));
  GetDlgItemText(dlg, IDC_LOCALUSERNAME, pvar->settings.DefaultRhostsLocalUserName, sizeof(pvar->settings.DefaultRhostsLocalUserName));
  
  EndDialog(dlg, 1);
  return TRUE;
}

static BOOL CALLBACK default_auth_dlg_proc(HWND dlg, UINT msg, WPARAM wParam, LPARAM lParam) {
  struct _TInstVar FAR * pvar;

  switch (msg) {
  case WM_INITDIALOG:
    pvar = (struct _TInstVar FAR *)lParam;
    SetWindowLong(dlg, DWL_USER, lParam);

    init_default_auth_dlg(pvar, dlg);
    return TRUE; /* because we do not set the focus */

  case WM_COMMAND:
    pvar = (struct _TInstVar FAR *)GetWindowLong(dlg, DWL_USER);

    switch (LOWORD(wParam)) {
    case IDOK:
      return end_default_auth_dlg(pvar, dlg);
      
    case IDCANCEL:
      EndDialog(dlg, 0);
      return TRUE;

    case IDC_CHOOSERSAFILE:
      choose_RSA_key_file(dlg);
      return TRUE;

    case IDC_CHOOSEHOSTRSAFILE:
      choose_host_RSA_key_file(dlg);
      return TRUE;

    default:
      return FALSE;
    }

  default:
    return FALSE;
  }
}

void AUTH_do_default_cred_dialog(struct _TInstVar FAR * pvar) {
  HWND cur_active = GetActiveWindow();
  
  if (DialogBoxParam(hInst, MAKEINTRESOURCE(IDD_SSHAUTHSETUP),
    cur_active != NULL ? cur_active : pvar->NotificationWindow, default_auth_dlg_proc, (LPARAM)pvar) == -1) {
    notify_fatal_error(pvar, "Unable to display authentication setup dialog box.");
  }
}

void AUTH_destroy_cur_cred(struct _TInstVar FAR * pvar) {
  destroy_malloced_string(&pvar->auth_state.cur_cred.password);
  destroy_malloced_string(&pvar->auth_state.cur_cred.rhosts_client_user);
  RSA_free(pvar->auth_state.cur_cred.RSA_key);
  pvar->auth_state.cur_cred.RSA_key = NULL;
}

static char FAR * get_auth_method_name(SSHAuthMethod auth) {
  switch (auth) {
  case SSH_AUTH_PASSWORD: return "password";
  case SSH_AUTH_RSA: return "RSA";
  case SSH_AUTH_RHOSTS: return "rhosts";
  case SSH_AUTH_RHOSTS_RSA: return "rhosts with RSA";
  default:
    return "unknown method";
  }
}

void AUTH_get_auth_info(struct _TInstVar FAR * pvar, char FAR * dest, int len) {
  if (pvar->auth_state.user == NULL) {
    strncpy(dest, "None", len);
  } else if (pvar->auth_state.cur_cred.method != SSH_AUTH_NONE) {
    _snprintf(dest, len, "User '%s', using %s", pvar->auth_state.user,
      get_auth_method_name(pvar->auth_state.cur_cred.method));
  } else {
    _snprintf(dest, len, "User '%s', using %s", pvar->auth_state.user,
      get_auth_method_name(pvar->auth_state.failed_method));
  }
}

void AUTH_notify_disconnecting(struct _TInstVar FAR * pvar) {
  if (pvar->auth_state.auth_dialog != NULL) {
    PostMessage(pvar->auth_state.auth_dialog, WM_COMMAND, IDCANCEL, 0);
      /* the main window might not go away if it's not enabled. (see vtwin.cpp) */
    EnableWindow(pvar->NotificationWindow, TRUE);
  }
}

void AUTH_end(struct _TInstVar FAR * pvar) {
  destroy_malloced_string(&pvar->auth_state.user);

  AUTH_destroy_cur_cred(pvar);
}
