/*
SKIP Source Code License Statement:
------------------------------------------------------------------
  Copyright
  Sun Microsystems, Inc.


  Copyright (C) 1994, 1995, 1996 Sun Microsystems, Inc.  All Rights
  Reserved.

  Permission is hereby granted, free of charge, to any person
  obtaining a copy of this software and associated documentation
  files (the "Software"), to deal in the Software without
  restriction, including without limitation the rights to use,
  copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software or derivatives of the Software, and to 
  permit persons to whom the Software or its derivatives is furnished 
  to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be
  included in all copies or substantial portions of the Software.

  The Software must not be transferred to persons who are not US
  citizens or permanent residents of the US or exported outside
  the US (except Canada) in any form (including by electronic
  transmission) without prior written approval from the US
  Government. Non-compliance with these restrictions constitutes
  a violation of the U.S. Export Control Laws.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
  NONINFRINGEMENT.  IN NO EVENT SHALL SUN MICROSYSTEMS, INC., BE LIABLE
  FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
  OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
  CONNECTION WITH THE SOFTWARE OR DERIVATES OF THIS SOFTWARE OR 
  THE USE OR OTHER DEALINGS IN THE SOFTWARE.

  Except as contained in this notice, the name of Sun Microsystems, Inc.
  shall not be used in advertising or otherwise to promote
  the sale, use or other dealings in this Software or its derivatives 
  without prior written authorization from Sun Microsystems, Inc.
*/

#pragma ident "@(#)audio_entropy.c	1.4 96/10/08 Sun Microsystems"

/*
 * audio_entropy.c
 *
 * use /dev/audio as a source of entropy
 *
 * The output is not entirely random. Specifically the high order bits of the
 * sound channels are usually all 0's or all 1's depending on it the value
 * is positive or negative. For 16 bit precision the low order 6 bits seem
 * to be reasonably random. The audio output is passed through a filter
 * to reduce the redundancy of the output, and to unbias (or deskew) the 
 * output.
 *
 * For 8 bit precision with ulaw at 8000Hz (the usual Sun default) it will
 * take much longer to collect a good sized sample. For example it will take
 * about 1.5 seconds to collect 10k bytes of data.
 *
 * NOTE #1: The amount of entropy produced will be much less than the total
 * amount of data output. Entropy will be at most 1/3 the size of the output.
 *
 * NOTE #2: The is no telling what entropy will be produced on a system with
 * a microphone attached.
 */

#include <fcntl.h>
#include <unistd.h>
#include <memory.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>
#include <stropts.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/conf.h>
#include <sys/time.h>
#include <stdlib.h>

#ifdef SUNOS
#include <sun/audioio.h>
#else
#include <sys/audioio.h>
#endif

#define TRUE          1
#define FALSE         0

#define BUFF_SIZE     256
#define MIN_BYTES     BUFF_SIZE

#define HOWMUCH       800      /* default number of bytes to output */

/* the following are the names of audio devices we look for by default */
char   *audio_devs[] = { "/dev/audio",
			 "/dev/sound/0",
			 "/dev/sound/1", 
			 NULL };

char            *program_name;
int             block = FALSE;         /* flag to block on io       */
int             use_default = FALSE;   /* /dev/audio defaults ?     */
unsigned char   out_buff[BUFF_SIZE];   /* deskewed output buffer    */
unsigned char   current;               /* current byte being filled */
int             buff_pos = 0;          /* possition in the buffer   */
int             byte_pos = 0;          /* possition in the byte     */

void cat_bit(char bit)
{

    current = current << 1;
    if (bit) 
        current++;
    
    byte_pos++;
    if (byte_pos == 8) {
        out_buff[buff_pos++] = current;
	byte_pos = 0;
	current = 0;
    }

}

void usage() {
fprintf(stderr,"%s [-z size] [-d] [-b]\n",program_name);
fprintf(stderr,"    -a   set the name of the audio device to use\n");
fprintf(stderr,"    -z   the size of the data collected in bytes\n");
fprintf(stderr,"    -s   do not de-skew the output (raw audio)\n");
fprintf(stderr,"    -d   use the default settings for the audio device\n");
fprintf(stderr,"    -b   block on read from the audio device\n");
exit(0);
}

#ifdef DEBUG
void print_status(audio_info_t *ai, char *ad) 
{
fprintf(stderr,"Recording parameters for %s:\n", ad);
fprintf(stderr,"\tencoding=%d\n",    ai->record.encoding);
fprintf(stderr,"\tprecision=%d\n",   ai->record.precision);
fprintf(stderr,"\tbalance=%d\n",     ai->record.balance);
fprintf(stderr,"\tsample_rate=%d\n", ai->record.sample_rate);
fprintf(stderr,"\tchannels=%d\n",    ai->record.channels);
fprintf(stderr,"\tvolume=%d\n",      ai->record.gain);
fprintf(stderr,"\tbuffer=%d\n",      ai->record.buffer_size);
fprintf(stderr,"\tdata collection rate=%d (bytes per second)\n", 
(ai->record.sample_rate * ai->record.precision * ai->record.channels) / 8);
}
#endif

void warn(char *msg)
{
      fprintf(stderr,"WARNING: %s\n",msg);
      if (errno != 0)
	perror(program_name);
      return;
}

void error(char *msg)
{
      fprintf(stderr,"ERROR: %s\n",msg);
      if (errno != 0)
	perror(program_name);
      exit(1);
}

int open_audioctl(char *ac)
{
    int cfd;
    audio_device_t  ad_t;    /* data struct for /dev/audio ioctl */
    audio_info_t    ai_t;    /* data struct for /dev/audio ioctl */

    /* this will always succeed since we're using it readonly */
    cfd = open(ac, O_RDONLY | O_NDELAY);
    if ((cfd  == -1))
         return 0;

    /* check the audio chip name */
    if (ioctl(cfd, AUDIO_GETDEV, &ad_t) == -1) {
        warn("can't get device settings from audio device");
	close(cfd);
	return 0;
    }

#ifdef DEBUG
    if (strcmp(ad_t.name,"SUNW,CS4231") && strcmp(ad_t.name,"SUNW,dbri")) {
        char *msg;
	msg = malloc(strlen(ad_t.name) + 32); 
	sprintf(msg,"Unknown audio device type: %s", ad_t.name);
	warn(msg);
    }
#endif

    /* check for read & write access */
    if (ioctl(cfd, AUDIO_GETINFO, &ai_t) == -1) {
        warn("can't get info from audio device");
	close(cfd);
	return 0;
    }

    if (ai_t.record.open || ai_t.record.waiting || ai_t.record.active) {
        /* someone else already has the device open */
        warn("Device already open.");
	close(cfd);
	return 0;
    }

    return cfd;
}

int open_audio(char *a)
{
    int afd;

    if (block)
        afd = open(a, O_RDONLY);
    else
        afd = open(a, O_RDONLY | O_NDELAY);
    if ((afd < 0))
        return 0;

    return afd;
}


int main(int argc,char **argv) 
{
#ifdef DEBUG
	struct timeval  start;        /* start time of program            */
	struct timeval  stop;         /* stop time of program             */  
        long            sec, usec;    /* total time of program (real time)*/
#endif
	int             deskew = TRUE;/* flag to deskew the output        */
	int             i,j;          /* counters                         */
        int             afd;          /* file descriptor for /dev/audio   */
	fd_set          afd_set;      /* select data struct for file desc.*/
        int             cfd;          /* file descriptor for /dev/audio   */
	struct timeval  timeout;      /* stop time of program             */  
	int             bytes_of_entropy = HOWMUCH ;
	int             bytes;        /* bytes read from audio            */
	int             bytes_out;    /* bytes written so far             */
	int             audio_bytes;  /* total audio bytes read           */
	unsigned char   buff[BUFF_SIZE];         /* the read buffer       */
	audio_info_t    ai_t;         /* data struct for /dev/audio ioctl */
	audio_info_t    ai_old;       /* data struct for /dev/audio ioctl */
	char            *audiodev;    /* audio device name                */
	char            *audioctl;    /* audio device controler name      */
	char            c;            /* temp for parsing command line    */
	unsigned int    mask, bp;     /* bit masks for deskewing          */
	int             rc;           /* return code for select call      */

#ifdef DEBUG
	gettimeofday(&start, 0);
#endif
	
	/* parse the command line arguments */
	program_name = argv[0];
	audiodev = NULL;

        while ((c = getopt(argc, argv, "a:bdhsz:")) != EOF) {
          switch(c) {
	  case 'a':    /* name of audio device to use */
	    audiodev = optarg;
	    break;
          case 'b':    /* block */
	    block = TRUE;
	    break;
	  case 'd':    /* use default settings */
	    use_default = TRUE;
	    break;
	  case 'h':    /* help! */
	    usage();
	    break;
	  case 's':    /* leave the output skewed */
	    deskew = FALSE;
	    break;
	  case 'z':    /* size of rand data */
	    bytes_of_entropy = atoi(optarg);
	    if (bytes_of_entropy < MIN_BYTES)
	        bytes_of_entropy = MIN_BYTES;
	    break;
	  default:
	    usage();
	    break;
	  }
	}

        if (optind != argc)
            usage();	

	/*----------------------------------------------------------
	 * Find an available audio device.
	 */
	afd = 0;
	if (audiodev) {
	    audioctl = malloc(strlen(audiodev) + 4);
	    sprintf(audioctl,"%sctl",audiodev);
	    fprintf(stderr,"Attempting to open: <%s>\n", audiodev);
	    if (cfd = open_audioctl(audioctl)) 
	       afd = open_audio(audiodev);

	    if (!afd) {
	      if (cfd)
		  close(cfd);
	      free(audioctl);
	    }
	}

	i = 0;
	while ((!afd) && (audio_devs[i])) {
	    audiodev = audio_devs[i];
	    audioctl = malloc(strlen(audiodev) + 4);
	    sprintf(audioctl,"%sctl",audiodev);
	    if (cfd = open_audioctl(audioctl)) 
	       afd = open_audio(audiodev);

	    if (!afd) {
	      if (cfd)
		  close(cfd);
	      free(audioctl);
	      i++;
	    }
	}

	if (!afd) 
	    error("could not find available audio device.");

	/* save old status and store it. */
	AUDIO_INITINFO(&ai_old);
	if (ioctl(afd, AUDIO_GETINFO, &ai_old) == -1) 
	    warn("can't get current audio settings");	

	/* pause and flush audio */
 	if (use_default == FALSE) {
	    AUDIO_INITINFO(&ai_t);
	    ai_t.record.pause  = TRUE;
	    if (ioctl(afd, AUDIO_SETINFO, &ai_t) == -1) 
	        warn("can't pause audio for switching parameters");

	    if (ioctl(afd, I_FLUSH, FLUSHW) == -1) 
	        warn("can't flush audio device");	    
	}

	/* ALWAYS set the max record volume (gain) */
	AUDIO_INITINFO(&ai_t);
	ai_t.record.gain    = AUDIO_MAX_GAIN;
	ai_t.record.balance = AUDIO_MID_BALANCE;
 	ai_t.record.port    = AUDIO_MICROPHONE; 
	ai_t.record.pause   = FALSE;
	ai_t.record.buffer_size = BUFF_SIZE; 

	if (use_default == FALSE) {
	    /* this configuration produces about 10 bits of randomness
	     * per sample (32 bits, 16 bits for left channel + 16 bits 
	     * for the right channel). The random bits are the low order
	     * 5 bits of the left and right channel.
	     * Theoretical data rate = 32000*16*2 bits per minute
	     */
	    ai_t.record.encoding    = AUDIO_ENCODING_LINEAR; 
	    ai_t.record.sample_rate = 32000;
	    ai_t.record.precision   = 16;
	    ai_t.record.channels    = 2;
	}
	
	if (ioctl(afd, AUDIO_SETINFO, &ai_t) == -1) 
  	    warn("can't set audio parameters (try using defaults)");

	if (ioctl(afd, I_SRDOPT, RMSGN) == -1) 
  	    warn("can't set Message-nondiscard mode.");
	
#ifdef DEBUG
	if (ioctl(afd, AUDIO_GETINFO, &ai_t) == -1)
	    warn("can't get audio settings");
	print_status(&ai_t, audiodev);
#endif

	/* set the timeout for a single read from audio device */
	timeout.tv_sec  = 1;
	timeout.tv_usec = 0;

	bytes_out = 0;
	audio_bytes = 0;
	while (bytes_out < bytes_of_entropy) {
	    FD_ZERO(&afd_set);
	    FD_SET(afd, &afd_set);
	    rc = select(8,&afd_set,NULL,NULL,& timeout);
	    if (rc > 0){
	        bytes = read(afd,buff,BUFF_SIZE); 
		if (bytes <= 0) 
		    error("problem reading from /dev/audio");
	    }
	    else if (rc = 0)
	        error("Read from audio timed out.\n");
	    else
	        error("Select failed");

	    audio_bytes += bytes;

	    /* to deskew we will use the following map:
	     * 00 => <nothing>
	     * 01 =>     0
	     * 10 =>     1	
	     * 11 => <nothing>
	     */
	    if (deskew == TRUE) {
	        for (i=0; i < bytes; i++) {
		    mask = 0xc0;
		    for (j=6; j >= 0 ; j = j - 2) {
		        bp = (buff[i] & mask) >> j;
			if (bp == 1)
			    cat_bit(0);
			else if (bp == 2)
			    cat_bit(1);
			mask = mask >> 2;
		    }
		}
		bytes = write(1,out_buff,buff_pos);
		buff_pos = 0;
	    }
	    else {
	        bytes = write(1,buff,bytes);
	    }
	      
	    if (bytes < 0)
	        error("problem writing out data");
	    bytes_out += bytes;
	}

	/* restore audio settings */
	if (ioctl(afd, AUDIO_SETINFO, &ai_old) == -1) 
	    warn("can't restore audio settings");

	/* close files */
        close(afd);
	close(cfd);

#ifdef DEBUG
	gettimeofday(&stop, 0);
	sec = stop.tv_sec - start.tv_sec;
	usec = stop.tv_usec - start.tv_usec;
	if (usec < 0) {
	    sec--;
	    usec = 1000000 - start.tv_usec +  stop.tv_usec;
	}
	fprintf(stderr,"elapsed sec = %d micosecs = %d \n", sec, usec);
	fprintf(stderr,"total output bytes = %d \n", bytes_out );
	fprintf(stderr,"total audio bytes  = %d \n", audio_bytes );
	fprintf(stderr,"percent audio used as entropy = %2.2f \n", 
		((float)bytes_out / (float)audio_bytes) * 100.0);
#endif

	return(0);
        
}

