/* Copyright (c) 1986 Massachusetts Institute of Technology

   Permission to use, copy, modify, and distribute this program
   for any purpose and without fee is hereby granted, provided
   that this copyright and permission notice appear on all copies
   and supporting documentation, the name of M.I.T. not be used
   in advertising or publicity pertaining to distribution of the
   program without specific prior permission, and notice be given
   in supporting documentation that copying and distribution is
   by permission of M.I.T.  M.I.T. makes no representations about
   the suitability of this software for any purpose.  It is pro-
   vided "as is" without express or implied warranty.*/
#ifndef lint
static char *copyright = "Copyright (c) 1986, Massachusetts Institute of Technology";
#endif lint
/*
 *	$Source: /afs/net.mit.edu/project/krb4/src/appl/knetd/RCS/knetd.c,v $
 *	$Header: /afs/net.mit.edu/project/krb4/src/appl/knetd/RCS/knetd.c,v 4.5 92/11/10 21:24:03 tytso Exp $
 */

/*
 * knetd.c: multiplex network servers over one "well-known" Internet port.
 * 
 * for TCP connections, accept the connection, read a longword (4 8-bit bytes)
 * in network order which indicates how many characters follow.
 * Read in those characters as ASCII and lookup the resulting name in a
 * service table.  If found, fork() and exec() that server with the socket
 * connected to file descriptors 0,1, and 2.  If not found, shutdown and
 * close connection.
 *
 *
 * On a SIGHUP, re-read the configuration file to establish new server
 * name/image mappings
 *
 * see the file "parse.c" for a description of the format of the configuration
 * file.
 */

#ifndef lint
static char *rcsid_knetd_c = "$Header: /afs/net.mit.edu/project/krb4/src/appl/knetd/RCS/knetd.c,v 4.5 92/11/10 21:24:03 tytso Exp $";
#endif lint

#include <sys/types.h>
#include <netdb.h>
#include <syslog.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>
#include <sys/param.h>
#include "knetd.h"


#define max(x,y) ((x > y) ? x : y)
#define EVER (;;)

#ifndef FD_SET
#define fd_set int
#define FD_SET(n,p) (*(p) |= (1<<(n)))
#define FD_ISSET(n,p) (*(p) & (1<<(n)))
#define FD_ZERO(p) bzero(p, sizeof(*(p)))
#endif FD_SET

extern int errno;
struct protoent *proto;			/* protocol descriptor */
struct knet_entry *services;		/* services structure table */
char *config;				/* filename of config file */
int debug = 0;				/* debug flag */
int num_entries;			/* number of entries in the table */
long tcp_proto = -1;			/* protocol number for tcp */
int configure(), configure_it(), reapchild(), closedown();
char *calloc(), *realloc();
struct knet_entry *find_entry();

main(argc,argv)
int argc;
char **argv;
{
    struct servent *mytcp;
    int forkstat, selectstat, on = 1;
    int tcp_socket, new_fd = -1, cc;
    int namelen, nlnet;
    char servname[512];
    struct sockaddr_in tcp_saddr;
    fd_set sockets, readsockets;
    struct knet_entry *entry;

    config = CONFIG;
    if (argc > 1) {
	if (!strcmp(argv[1],"-d")) {
	    debug = 1;
	    argv++; argc--;
	}
	if (argc > 1) config = argv[1];
    }	

    if (!debug) {
	if (forkstat = fork()) exit(forkstat < 0 ? forkstat : 0);
	{
	    int s;
	    for(s = getdtablesize(); --s >= 0; )
	      (void) close(s);
	}
	(void) open ("/", O_RDONLY);
	(void) dup2(0, 1);
	(void) dup2(0, 2);
	{
	    int tty = open("/dev/tty", O_RDWR);
	    if (tty > 0) {
		(void) ioctl(tty, (int)TIOCNOTTY, (char *) NULL);
		(void) close(tty);
	    }
	}
    }
#ifdef LOG_DAEMON  /* BSD 4.3 Syslog */
    openlog("knetd", LOG_PID, LOG_DAEMON);
#else !LOG_DAEMON
    openlog("knetd", LOG_PID);
#endif LOG_DAEMON
    if ((proto = getprotobyname("tcp")) != NULL)
      tcp_proto = proto->p_proto;

    if (mytcp = getservbyname("knetd","tcp"))
      tcp_saddr.sin_port = mytcp->s_port;

    if (mytcp == NULL) {
	if (debug)
	  printf("unknown service: knetd/tcp");
	syslog(LOG_WARNING, "unknown service: knetd/tcp");
	exit(1);
    }

    num_entries = configure();

    (void) signal(SIGHUP, configure_it);
    (void) signal(SIGCHLD, reapchild);
    (void) signal(SIGTERM, closedown);

    tcp_socket = socket(AF_INET, SOCK_STREAM, 0);

    if (tcp_socket != 3) {
	(void) dup2(tcp_socket, 3);
	(void) close(tcp_socket);
	tcp_socket = 3;
    }

    if (setsockopt(tcp_socket, SOL_SOCKET, SO_REUSEADDR,
		   (char *) &on, sizeof(on)))
      syslog(LOG_WARNING, "tcp setsockopt SO_REUSEADDR: %m");


    if (bind(tcp_socket, &tcp_saddr, sizeof(tcp_saddr))) {
	if (debug)
	  perror("bind tcp socket");
	syslog(LOG_WARNING, "cannot bind tcp socket: %m");
	exit(1);
    }

    /* set up to get connections */

    FD_ZERO(&sockets);
    FD_SET(tcp_socket, &sockets);

    (void) listen(tcp_socket, 5);

    
    for EVER {
	readsockets = sockets;
	if ((selectstat = select(tcp_socket + 1, &readsockets,
				(fd_set *) 0, (fd_set *) 0,
				(struct timeval *) 0)) <= 0) {
				    if (selectstat < 0 && errno != EINTR)
				      syslog(LOG_WARNING, "select: %m");
				    sleep(1);
				    continue;
				}
	/* got something: figure out who wants what */

	if (FD_ISSET(tcp_socket, &readsockets)) {
	    /* tcp connection */

	    if ((new_fd =
		accept(tcp_socket, (struct sockaddr *) 0, (int *) 0)) <0) {
		    if (debug)
		      perror("accept");
		    syslog(LOG_WARNING, "accept: %m");
		    continue;
		}
	} else {
	    syslog(LOG_WARNING, "select not ready on tcp???: %m");
	    continue;
	}

	cc = read(new_fd, (char *) &nlnet, sizeof(nlnet));

	entry = NULL;

	if (cc == sizeof(nlnet)) {
	    namelen = ntohl(nlnet);
	    if (namelen < 0 || namelen > BUFSIZ)
		syslog(LOG_WARNING,"bogus byte count %d",namelen);
	    else {
		cc = read(new_fd, servname, namelen);

		servname[namelen] = '\0';
		if (debug)
		  syslog(LOG_WARNING,"someone wants %s",servname);

		entry = find_entry(servname);
	    }
	} else {		/* didn't get a longword of byte count */
	    if (debug)
	      printf("byte count mismatch: %d != %d\n",cc,sizeof(nlnet));
	    syslog(LOG_WARNING, "byte count mismatch: %d", cc);
	}

	if (entry == NULL) {
	    (void) shutdown(new_fd, 2); /* disallow receives and sends */
	    (void) close(new_fd);
	    continue;		/* for EVER */
	}

	if (fork()) {
	    (void) close(new_fd);
	} else {
	    /* child: clean up and exec */
	    if (debug) {
		/* detach from terminal if necessary */
		int s;

		syslog(LOG_WARNING,"%d execl %s\n",getpid(), entry->kn_pathname);

		{
		    int tty = open("/dev/tty", O_RDWR);
		    if (tty > 0) {
			(void) ioctl(tty, (int)TIOCNOTTY, (char *) NULL);
			(void) close(tty);
		    }
		}
		for (s = getdtablesize(); --s >= 0; )
		  if (s != new_fd) (void) close(s);
	    }

	    close(tcp_socket);		/* don't listen anymore */
	    /* set up socket as file descriptors 0,1,2 */
	    if (new_fd != 0) {
		(void) dup2(new_fd, 0);
		(void) close(new_fd);
	    }		
	    (void) dup2(0, 1);
	    (void) dup2(0, 2);

	    (void) setuid(entry->kn_uid);

	    (void) execl(entry->kn_pathname, entry->kn_arg0, 0);

	    /* fall through: exec failed */

	    (void) shutdown(0, 2);		/* clean up connection */

	    syslog(LOG_WARNING, "execl %s: %m", entry->kn_name);
	    _exit(1);
	}
    }
}


/* return pointer to the entry for server "name", protocol proto, or NULL
   if there is none */
struct knet_entry *find_entry(name)
char *name;
{
    register int i;
    int sigs;

    /* don't want to reconfigure in the middle of looking something up */
    sigs = sigblock((1 << SIGHUP));

    for (i = 0; i < num_entries; i++)
      if (!strcmp(services[i].kn_name, name))
	return(&services[i]);

    /* okay to reconfigure now */
    (void) sigsetmask(sigs);

    return((struct knet_entry *)NULL);
}

#ifdef DEBUG
dump_entry(entry)
struct knet_entry *entry;
{
    printf("name=%s\ntype=%d\nuid=%d\npathname=%s\narg0=%s\n\n",
	   entry->kn_name,
	   entry->kn_type,
	   entry->kn_uid,
	   entry->kn_pathname,
	   entry->kn_arg0);
}
#endif DEBUG

int configured = 0;

/* (re)initialize the configuration table */

int configure()
{
    register int i;
    struct knet_entry *entry;

    setconfig(config);

/* Two-part scheme here:  we automatically allocate room for at least
 * MAX_ENTRIES entries in the table.  If we overflow by some chance,
 * we realloc() more space for each one over MAX_ENTRIES.  Less efficient,
 * but easier to arrange the structures in a block this way.
 */

/* 2nd time around: free up storage */
    if (configured) {
	syslog(LOG_WARNING,"knetd reconfiguring");
	free_up(services);
    }

    services = (struct knet_entry *)calloc(MAX_ENTRIES, KNET_SIZE);
    configured = 1;

    /* Strings are malloc()'ed within getconfigent() before being added to the
       structure returned to us, so we can just copy the structure directly.*/

    for (i = 0; i < MAX_ENTRIES; i++) {
	if ((entry = getconfigent()) == NULL) break;
	if (entry->kn_type != tcp_proto) {
	    if (debug)
	      printf("protocol number %d not supported\n",entry->kn_type);
	    syslog(LOG_WARNING, "protocol number %d not supported",
		   entry->kn_type);
	    continue;
	}
	bcopy(entry, &services[i], KNET_SIZE);
    }

    if (entry != NULL)
      for (;(entry  = getconfigent()) != NULL; i++) {
	  if (entry->kn_type != tcp_proto) {
	      if (debug)
		printf("protocol number %d not supported\n",entry->kn_type);
	      syslog(LOG_WARNING, "protocol number %d not supported",
		     entry->kn_type);
	      continue;
	  }
	  services = (struct knet_entry *) realloc((char *)services,
						   (unsigned) (KNET_SIZE * i));
	  bcopy(entry, &services[i], KNET_SIZE);
      }

    endconfig();
    return(i);
}

free_up(stbl)
struct knet_entry *stbl;
{
    register int i;

    /* we need to free up all the strings we allocated, plus the structure
       table itself. */

    for (i = 0; i < num_entries; i++) {
	free(stbl[i].kn_name);
	free(stbl[i].kn_pathname);
	free(stbl[i].kn_arg0);
    }
    free((char *)stbl);
}

/* SIGHUP handler: hangup causes reinitialization of the table */
int configure_it()
{
#ifdef DEBUG
    register int i;
#endif DEBUG
    
    num_entries = configure();
#ifdef DEBUG
    if (debug) {
	printf("you have %d entries\n",num_entries);
	for (i = 0; i < num_entries; i++)
	  dump_entry(&services[i]);
    }
#endif DEBUG
}

/* SIGCHLD handler */
int reapchild()
{
    (void) wait((union wait *)0);
}

/* SIGTERM handler */
int closedown()
{
    syslog(LOG_WARNING, "shutting down");
    exit(0);
}
