/*
 *  $Id: inode.c,v 1.1 1998/12/02 23:37:07 ezk Exp $
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif /* HAVE_CONFIG_H */
#include <fist.h>
#include <cryptfs.h>


// cryptfs_create(struct inode *dir, const char *name, int namelen, int mode, struct inode **result)
STATIC int
cryptfs_create(inode_t *dir, dentry_t *dentry, int mode)
{
    int err = -EACCES;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();
    ASSERT(hidden_dentry != NULL);

    fist_checkinode(dir, "cryptfs_create");

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;
    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->create) {
	unlock_dir(hidden_dir_dentry);
	goto out;
    }
    err = hidden_dir_dentry->d_inode->i_op->create(hidden_dir_dentry->d_inode,
						   hidden_dentry, mode);
    unlock_dir(hidden_dir_dentry);

    if (err)
	goto out;

    err = cryptfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (!err) {
	fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);
    }
    fist_checkinode(dir, "post cryptfs_create");

 out:
    print_exit_status(err);
    return err;
}


// cryptfs_lookup(struct inode *dir, const char *name, int namelen, struct inode **inode)
STATIC int
cryptfs_lookup(inode_t *dir, dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dir_dentry = cryptfs_hidden_dentry(dentry->d_parent);
    dentry_t *hidden_dentry;
    const char *name = dentry->d_name.name;
    unsigned int namelen = dentry->d_name.len;
    char *encoded_name;
    unsigned int encoded_namelen;
    void *key;

    print_entry_location();
    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }

    fist_checkinode(dir, "cryptfs_lookup");

    fist_print_dentry("LOOKUP: dentry IN", dentry);
    fist_print_dentry("LOOKUP: dentry->d_parent IN", dentry->d_parent);
    fist_print_dentry("LOOKUP: hidden_dir_dentry IN", hidden_dir_dentry);
    fist_print_inode("LOOKUP: dir IN", dir);

    if (hidden_dir_dentry->d_inode)
	fist_print_inode("LOOKUP: hidden_dir_dentry->d_inode",
			 hidden_dir_dentry->d_inode);

    /* must initialize dentryu operations */
    dentry->d_op = &cryptfs_dops;

    /* increase refcount of base dentry (lookup_dentry will decrement) */
    // THIS IS RIGHT! (don't "fix" it)
    dget(hidden_dir_dentry);

    encoded_namelen = cryptfs_encode_filename(name,
					      namelen,
					      &encoded_name,
					      key,
					      SKIP_DOTS);

    /* will allocate a new hidden dentry if needed */
    hidden_dentry = lookup_dentry(encoded_name, hidden_dir_dentry, 0);
    kfree_s(encoded_name, encoded_namelen);
    if (IS_ERR(hidden_dentry)) {
	printk("ERR from hidden_dentry!!!\n");
	err = PTR_ERR(hidden_dentry);
	goto out;
    }

    /* update parent directory's atime */
    fist_copy_attr_atime(dir, hidden_dir_dentry->d_inode);

    /* lookup is special: it needs to handle negative dentries */
    if (!hidden_dentry->d_inode) {
	dtohd(dentry) = hidden_dentry;
	d_add(dentry, NULL);
	fist_print_dentry("lookup hidden", hidden_dentry);
	fist_print_dentry("lookup dentry", dentry);
	// ION: not needed b/c d_inode is already null here
	//	dentry->d_inode = NULL;
	// dput(hidden_dentry);
	goto out;
    }

    fist_dprint(6, "lookup \"%s\" -> inode %d\n", name, hidden_dentry->d_inode->i_ino);
    err = cryptfs_interpose(hidden_dentry, dentry, dir->i_sb, 1);
#if 0
    /*
     * interpose always increments the hidden inode's refcount by one
     * but in this function it was already done by the lookup_dentry
     * above, which eventually calls iget() on hidden_inode, thus
     * increasing the refcount once too many.
     */
    if (!err)
	iput(hidden_dentry->d_inode);
#endif

    fist_checkinode(dentry->d_inode, "cryptfs_lookup OUT: dentry->d_inode:");
    fist_checkinode(dir, "cryptfs_lookup OUT: dir:");

    fist_print_dentry(__FUNCTION__ " OUT hidden_dentry", hidden_dentry);
    fist_print_dentry(__FUNCTION__ " OUT hidden_dir_dentry", hidden_dir_dentry);
out:
    fist_print_dentry("LOOKUP: dentry->d_parent OUT", dentry->d_parent);
    print_exit_status(err);
    return err;
}


// cryptfs_link(inode_t *oldinode, inode_t *dir, const char *name, int namelen)
STATIC int
cryptfs_link(dentry_t *old_dentry, inode_t *dir, dentry_t *new_dentry)
{
    int err = -EPERM;
    dentry_t *hidden_old_dentry = cryptfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = cryptfs_hidden_dentry(new_dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    fist_checkinode(dir, "cryptfs_link-dir");
    fist_checkinode(old_dentry->d_inode, "cryptfs_link-oldinode");

    hidden_dir_dentry = lock_parent(hidden_new_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    err = -EXDEV;
    if (hidden_dir_dentry->d_inode->i_dev != hidden_old_dentry->d_inode->i_dev)
	goto out_lock;

    err = -EPERM;
    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->link)
	goto out_lock;

    // XXX: DQUOT_INIT(dir->d_inode); // do we need that
    err = hidden_dir_dentry->d_inode->i_op->link(hidden_old_dentry,
						 hidden_dir_dentry->d_inode,
						 hidden_new_dentry);

    if (err)
	goto out_lock;
    err = cryptfs_interpose(hidden_new_dentry, new_dentry, dir->i_sb, 0);
    /*
     * XXX: we have to do a cryptfs_copy_inode (or subset thereof) in
     * *ALL* functions, all file systems, and all OSs!!!
     */
    if (!err)
	fist_copy_attr_timesizes(dir, hidden_new_dentry->d_inode);

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    print_exit_status(err);
    return err;
}


// cryptfs_unlink(inode_t *dir, const char *name, int namelen)
STATIC int
cryptfs_unlink(inode_t *dir, dentry_t *dentry)
{
    int err = -EPERM;
    inode_t *hidden_dir = itohi(dir);
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);

    print_entry_location();
    ASSERT(hidden_dentry != NULL);

    fist_checkinode(dir, "cryptfs_unlink-dir");
    if (!hidden_dir->i_op || !hidden_dir->i_op->unlink)
	goto out;

    err = hidden_dir->i_op->unlink(hidden_dir, hidden_dentry);
    if (!err)
        fist_copy_attr_times(dir, hidden_dir);
    fist_checkinode(dir, "post cryptfs_unlink-dir");

 out:
    print_exit_status(err);
    return err;
}


// cryptfs_symlink(inode_t *dir, const char *name, int namelen, const char *oldname)
STATIC int
cryptfs_symlink(inode_t *dir, dentry_t *dentry, const char *symname)
{
    int err = -EPERM;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    char *encoded_symname;
    unsigned int encoded_symlen;
    dentry_t *hidden_dir_dentry;
    void *key;

    print_entry_location();

    if ((key = fist_get_userpass(dir->i_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    fist_checkinode(dir, "cryptfs_symlink-dir");
    if (!hidden_dir_dentry->d_inode->i_op ||
	!hidden_dir_dentry->d_inode->i_op->symlink) {
	err = -EPERM;
	goto out_lock;
    }

    encoded_symlen = cryptfs_encode_filename(symname,
					     strlen(symname),
					     &encoded_symname,
					     key,
					     DO_DOTS);
    err = hidden_dir_dentry->d_inode->i_op->symlink(hidden_dir_dentry->d_inode,
						    hidden_dentry,
						    encoded_symname);
    kfree_s(encoded_symname, encoded_symlen);
    if (err)
	goto out_lock;
    err = cryptfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (!err)
        fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

    fist_checkinode(dir, "post cryptfs_symlink-dir");

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    print_exit_status(err);
    return err;
}


// cryptfs_mkdir(inode_t *dir, const char *name, int namelen, int mode)
STATIC int
cryptfs_mkdir(inode_t *dir, dentry_t *dentry, int mode)
{
    int err = -EPERM;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    fist_checkinode(dir, "cryptfs_mkdir-dir");
    if (!hidden_dir_dentry->d_inode->i_op ||
	!hidden_dir_dentry->d_inode->i_op->mkdir) {
	err = -EPERM;
	goto out_lock;
    }

    err = hidden_dir_dentry->d_inode->i_op->mkdir(hidden_dir_dentry->d_inode,
						  hidden_dentry,
						  mode);
    if (err)
	goto out_lock;
    err = cryptfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    if (!err)
        fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

    fist_checkinode(dir, "post cryptfs_mkdir-dir");

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    print_exit_status(err);
    return err;
}


// cryptfs_rmdir(inode_t *dir, const char *name, int namelen)
STATIC int
cryptfs_rmdir(inode_t *dir, dentry_t *dentry)
{
    int err = -EPERM;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    fist_checkinode(dir, "cryptfs_rmdir-dir");
    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->rmdir) {
	err = -EPERM;
	goto out_lock;
    }

    err = hidden_dir_dentry->d_inode->i_op->rmdir(hidden_dir_dentry->d_inode,
						  hidden_dentry);
    fist_checkinode(dir, "post cryptfs_rmdir-dir");
    if (!err)
        fist_copy_attr_times(dir, hidden_dir_dentry->d_inode);

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    print_exit_status(err);
    return err;
}


// cryptfs_mknod(inode_t *dir, const char *name, int namelen, int mode, int dev)
STATIC int
cryptfs_mknod(inode_t *dir, dentry_t *dentry, int mode, int dev)
{
    int err = -EPERM;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    dentry_t *hidden_dir_dentry;

    print_entry_location();

    hidden_dir_dentry = lock_parent(hidden_dentry);
    err = PTR_ERR(hidden_dir_dentry);
    if (IS_ERR(hidden_dir_dentry))
	goto out;

    fist_checkinode(dir, "cryptfs_mknod-dir");
    if (!hidden_dir_dentry->d_inode->i_op || !hidden_dir_dentry->d_inode->i_op->mknod) {
	err = -EPERM;
	goto out_lock;
    }

    err = hidden_dir_dentry->d_inode->i_op->mknod(hidden_dir_dentry->d_inode,
						  hidden_dentry,
						  mode,
						  dev);
    if (err)
	goto out_lock;
    err = cryptfs_interpose(hidden_dentry, dentry, dir->i_sb, 0);
    fist_checkinode(dir, "post cryptfs_mknod-dir");
    if (!err)
        fist_copy_attr_timesizes(dir, hidden_dir_dentry->d_inode);

 out_lock:
    unlock_dir(hidden_dir_dentry);
 out:
    print_exit_status(err);
    return err;
}


// cryptfs_rename(inode_t *old_dir, const char *old_name, int old_len,
//		 inode_t *new_dir, const char *new_name, int new_len, int must_be_dir)
STATIC int
cryptfs_rename(inode_t *old_dir, dentry_t *old_dentry,
	      inode_t *new_dir, dentry_t *new_dentry)
{
    int err = -EPERM;
    dentry_t *hidden_old_dentry = cryptfs_hidden_dentry(old_dentry);
    dentry_t *hidden_new_dentry = cryptfs_hidden_dentry(new_dentry);
    dentry_t *hidden_old_dir_dentry;
    dentry_t *hidden_new_dir_dentry;
    int (*rename_fxn) (inode_t *, dentry_t *, inode_t *, dentry_t *);

    print_entry_location();

    hidden_old_dir_dentry = get_parent(hidden_old_dentry);
    hidden_new_dir_dentry = get_parent(hidden_new_dentry);
    err = PTR_ERR(hidden_old_dir_dentry);
    if (IS_ERR(hidden_old_dir_dentry))
	goto out;
    err = PTR_ERR(hidden_new_dir_dentry);
    if (IS_ERR(hidden_new_dir_dentry))
	goto out;
    double_lock(hidden_old_dir_dentry, hidden_new_dir_dentry);

    fist_checkinode(old_dir, "cryptfs_rename-old_dir");
    fist_checkinode(new_dir, "cryptfs_rename-new_dir");

    if (!hidden_old_dir_dentry->d_inode->i_op ||
	!(rename_fxn = hidden_old_dir_dentry->d_inode->i_op->rename)) {
	err = -EPERM;
	goto out_lock;
    }

    err = rename_fxn(hidden_old_dir_dentry->d_inode, hidden_old_dentry,
		     hidden_new_dir_dentry->d_inode, hidden_new_dentry);
    if (err)
	goto out_lock;
    //    ASSERT(new_dentry->d_inode == NULL);
    //    new_dentry->d_inode = old_dentry->d_inode;
    //    err = cryptfs_interpose(hidden_new_dentry, new_dentry, new_dir->i_sb, 0);
    d_move(old_dentry, new_dentry);

    if (!err) {
        fist_copy_attr_all(new_dir, hidden_new_dir_dentry->d_inode);
	if (new_dir != old_dir)
	    fist_copy_attr_times(old_dir, hidden_old_dir_dentry->d_inode);
    }

    fist_checkinode(new_dir, "post cryptfs_mknod-new_dir");

 out_lock:
    double_unlock(hidden_old_dir_dentry, hidden_new_dir_dentry);
 out:
    print_exit_status(err);
    return err;
}


// cryptfs_readlink(inode_t *dir, char *buf, int bufsiz)
STATIC int
cryptfs_readlink(dentry_t *dentry, char *buf, int bufsiz)
{
    int err;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    char *decoded_name, *hidden_buf;
    mm_segment_t old_fs;
    void *key;

    print_entry_location();
    if ((key = fist_get_userpass(dentry->d_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }
    fist_print_dentry("cryptfs_readlink IN", dentry);

    if (!hidden_dentry->d_inode->i_op || !hidden_dentry->d_inode->i_op->readlink) {
	err = -EPERM;
	goto out;
    }

    hidden_buf = kmalloc(bufsiz, GFP_KERNEL);
    if (hidden_buf == NULL) {
	printk("Out of memory.\n");
	err = -ENOMEM;
	goto out;
    }

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = hidden_dentry->d_inode->i_op->readlink(hidden_dentry,
						 hidden_buf,
						 bufsiz);
    set_fs(old_fs);
    if (err >= 0) {
	fist_dprint(7, "READLINK: link \"%s\", length %d\n", hidden_buf, err);
	err = cryptfs_decode_filename(hidden_buf, err,
				      &decoded_name, key, DO_DOTS);
	fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
	if (err > 0) {
	    if (copy_to_user(buf, decoded_name, err))
		err = -EFAULT;
	    kfree_s(decoded_name, err);
	}
    }

    kfree_s(hidden_buf, bufsiz);
 out:
    print_exit_status(err);
    return err;
}


/* inode is from cryptfs, dir could be anything */
// int
// cryptfs_follow_link(inode_t *dir, inode_t *inode, int flag, int mode, inode_t **result)
STATIC dentry_t *
cryptfs_follow_link(dentry_t *dentry, dentry_t *base, unsigned int follow)
{
    char *buf;
    int len = PAGE_SIZE, err;
    dentry_t *ret_dentry = dentry;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    mm_segment_t old_fs;

    print_entry_location();
    fist_print_dentry("cryptfs_follow_link dentry IN", dentry);
    fist_print_dentry("cryptfs_follow_link base IN", base);

    if (!hidden_dentry->d_inode->i_op ||
	!hidden_dentry->d_inode->i_op->readlink ||
	!hidden_dentry->d_inode->i_op->follow_link) {
	/*
	 * do_follow_link will do a dput() on the dentry passed to us
	 * and if that is the same as ret_dentry, and we want it kept,
	 * then we must increment the dentry's refcount to avoid having
	 * it deallocate/destroyed.
	 */
	dget(ret_dentry);
	/*
	 * Looking at do_follow_link, if we implement follow_link, then
	 * we have to dput(base).  Normally that's done by lookup_dentry
	 * but here we don't run that code.
	 */
	dput(base);
	goto out;
    }

    buf = kmalloc(len, GFP_KERNEL);
    if (!buf) {
	ret_dentry = ERR_PTR(-ENOMEM);
	dput(base); // for the same reason as above
	goto out;
    }
    /* read the hidden symlink, and then we will follow it */
    old_fs = get_fs();
    set_fs(KERNEL_DS);
    err = dentry->d_inode->i_op->readlink(dentry, buf, len);
    set_fs(old_fs);
    if (err < 0) {
	ret_dentry = ERR_PTR(err);
	dput(base); // for the same reason as above
	goto out_free;
    }
    fist_copy_attr_atime(dentry->d_inode, hidden_dentry->d_inode);
    buf[err] = 0;	// terminate the buffer for lookup_dentry's sake

#if 0
    /* run dget b/c lookup_dentry expects the caller to do so */
    dget(base);
#endif
    /*
     * We do not dget(base) b/c the way we were called here,
     * the base (passed to us) was already dget'ed.
     */
    // XXX: FIXME w/ cryptfs_encode_filename()
    ret_dentry = lookup_dentry(buf, base, follow);
    fist_print_dentry("cryptfs_follow_link base OUT", base);

 out_free:
    kfree_s(buf, len);
 out:
    print_exit_location();
    return ret_dentry;
}


#if 0
STATIC void
fist_page_mkclean(page_t *page)
{
    unsigned long address;
    pmd_t *pmd;
    pgd_t *pgd;
    pte_t *ppte;

    ASSERT(page != NULL);
    ASSERT(page->inode != NULL);
    ASSERT(page->inode->i_mmap != NULL);
    ASSERT(page->inode->i_mmap->vm_mm != NULL);
    ASSERT(page->inode->i_mmap->vm_mm->pgd != NULL);

    address = page->offset;
    pgd = page->inode->i_mmap->vm_mm->pgd;
    pmd = pmd_offset(pgd, address);
    ppte = pte_offset(pmd, address);
    set_pte(ppte, pte_mkclean(*ppte));
    ASSERT(!pte_dirty(*ppte));
}

STATIC void
fist_page_assertclean(page_t *page)
{
    unsigned long address;
    pmd_t *pmd;
    pgd_t *pgd;
    pte_t *ppte;

    ASSERT(page != NULL);
    ASSERT(page->inode != NULL);
    ASSERT(page->inode->i_mmap != NULL);
    ASSERT(page->inode->i_mmap->vm_mm != NULL);
    ASSERT(page->inode->i_mmap->vm_mm->pgd != NULL);

    address = page->offset;
    pgd = page->inode->i_mmap->vm_mm->pgd;
    pmd = pmd_offset(pgd, address);
    ppte = pte_offset(pmd, address);
    //    set_pte(ppte, pte_mkclean(*ppte));
    ASSERT(!pte_dirty(*ppte));
}
#endif


STATIC int
cryptfs_readpage(file_t *file, page_t *page)
{
    int err = 0;
    inode_t *inode = file->f_dentry->d_inode;
    inode_t *hidden_inode = itohi(inode);
    file_t *hidden_file = ftohf(file);
    page_t *hidden_page, **hash;
    unsigned long page_cache = 0, page_data, hidden_page_data;
    void *key;

    print_entry_location();
    if ((key = fist_get_userpass(file->f_dentry->d_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }

    ASSERT(hidden_inode->i_op != NULL);
    fist_dprint(6, "READPAGE: f_pos:0x%x inode:0x%x i_size:0x%x hidden_inode:0x%x page->inode:0x%x page->offset:%lu\n",
		(int) file->f_pos, (int) inode, inode->i_size, (int) hidden_inode,
		(int) page->inode, page->offset);

    /* create new hidden_page (look for it in cache?) */
    hash = page_hash(hidden_inode, page->offset & PAGE_MASK);

restart_page:
    hidden_page = __find_page(hidden_inode, page->offset & PAGE_MASK, *hash);
    if (!hidden_page) {
	if (!page_cache) {		/* allocate new page */
	    page_cache = __get_free_page(GFP_KERNEL);
	    if (!page_cache) {
		err = -ENOMEM;
		goto out;		/* no need to free anything */
	    }
	    goto restart_page;
	}
	hidden_page = mem_map + MAP_NR(page_cache);
	page_cache = 0;
	add_to_page_cache(hidden_page, hidden_inode,
			  page->offset & PAGE_MASK, hash);
    }
    //    set_bit(PG_locked, &page->flags);
    //    set_bit(PG_locked, &hidden_page->flags);

    /* now we have a valid hidden page */

    fist_dprint(6, "READPAGE: hidden_page %x\n", (int) hidden_page);

    if (hidden_inode->i_op->readpage) {
	/*
	 * create new hidden_page (look for it in cache?)
	 * call lower level readpage on hidden_inode and hidden_page
	 */
	// ION, I add this. needed?
	file->f_pos = hidden_file->f_pos = page->offset;

#if 0
	hidden_file->f_reada = file->f_reada;
	hidden_file->f_ramax = file->f_ramax;
	hidden_file->f_raend = file->f_raend;
	hidden_file->f_ralen = file->f_ralen;
	hidden_file->f_rawin = file->f_rawin;
#endif

	err = hidden_inode->i_op->readpage(hidden_file, hidden_page);
	if (err) {
	    printk("READPAGE: error reading hidden page: %d\n", err);
	    goto out_free;
	}

	/* XXX: may need to wait_on_page(hidden_page) */
	wait_on_page(hidden_page);
	if (!PageUptodate(hidden_page) || PageError(hidden_page)) {
	    printk("hidden page not up-to-date or error!\n");
	    err = -EIO;
	    goto out_free;
	}
    }

    /*
     * decode hidden_page data onto page
     */
    page_data = page_address(page);
    hidden_page_data = page_address(hidden_page);

    cryptfs_decode_block((char *) hidden_page_data, (char *) page_data, PAGE_SIZE, key);

    /*
     * adjust flags and wake up processes waiting on the page
     * code shamelessly stolen from generic_readpage
     */
    // ION, why? we never set the locked bit?
    clear_bit(PG_locked, &page->flags);
    set_bit(PG_uptodate, &page->flags);
    wake_up(&page->wait);

    after_unlock_page(page);

out_free:
    if (page_cache) {
	printk("READPAGE: page cache is not null: 0x%x\n", (int) page_cache);
	free_page(page_cache);
    }
    // ION, do we need to __free_page?
    __free_page(hidden_page);	/* release_page() in mm/filemap.c */
out:
    fist_print_page_flags("READPAGE flags (exit)", page);
    print_exit_status(err);
    return err;
}


/* XXX: the VFS never calls writepage */
STATIC int
cryptfs_writepage(file_t *file, page_t *page)
{
    int err = -EIO;
    file_t *hidden_file = ftohf(file);
    inode_t *hidden_inode = hidden_file->f_dentry->d_inode;

    panic("ARE YOU SURE YOU WANT TO DEFINE WRITEPAGE???\n");
    print_entry_location();

    if (hidden_inode->i_op && hidden_inode->i_op->writepage) {
	fist_dprint(2, "non-generic writepage\n");
	err = hidden_inode->i_op->writepage(hidden_file, page);
    }

    print_exit_status(err);
    return err;
}


STATIC int
cryptfs_bmap(inode_t *inode, int block)
{
    int err = -EINVAL;
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();

    if (hidden_inode->i_op && hidden_inode->i_op->bmap)
	err = hidden_inode->i_op->bmap(hidden_inode, block);

    print_exit_status(err);
    return err;
}


/* kludgey - replace when up/down problem fixed */
STATIC void
cryptfs_truncate(inode_t *inode)
{
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();

    fist_checkinode(inode, "cryptfs_truncate");
    if (!hidden_inode->i_op || !hidden_inode->i_op->truncate)
	goto out;

    down(&hidden_inode->i_sem);
    hidden_inode->i_op->truncate(hidden_inode);
    up(&hidden_inode->i_sem);

    fist_copy_attr_timesizes(inode, hidden_inode);

 out:
    print_exit_location();
}


STATIC int
cryptfs_permission(inode_t *inode, int mask)
{
    inode_t *hidden_inode = itohi(inode);
    int mode = hidden_inode->i_mode;
    int err;

    print_entry_location();
    //    err = -ENOSPC;goto out;
    if (hidden_inode->i_op && hidden_inode->i_op->permission) {
	err = hidden_inode->i_op->permission(hidden_inode, mask);
	goto out;
    }

    if ((mask & S_IWOTH) && IS_RDONLY(hidden_inode) &&
	(S_ISREG(mode) || S_ISDIR(mode) || S_ISLNK(mode))) {
	err = -EROFS; /* Nobody gets write access to a read-only fs */
	goto out;
    }
    if ((mask & S_IWOTH) && IS_IMMUTABLE(hidden_inode)) {
	err = -EACCES; /* Nobody gets write access to an immutable file */
	goto out;
    }
    if (current->fsuid == hidden_inode->i_uid)
	mode >>= 6;
    else if (in_group_p(hidden_inode->i_gid))
	mode >>= 3;

    if (((mode & mask & S_IRWXO) == mask) || capable(CAP_DAC_OVERRIDE)) {
	err = 0;
	goto out;
    }

    /* read and search access */
    if ((mask == S_IROTH) ||
	(S_ISDIR(mode)  && !(mask & ~(S_IROTH | S_IXOTH))))
	if (capable(CAP_DAC_READ_SEARCH)) {
	    err = 0;
	    goto out;
	}
    err = -EACCES;

 out:
    print_exit_status(err);
    return err;
}


STATIC int
cryptfs_smap(inode_t *inode, int block)
{
    int err = -EINVAL;
    inode_t *hidden_inode = itohi(inode);

    print_entry_location();

    if (hidden_inode->i_op && hidden_inode->i_op->smap)
	err = hidden_inode->i_op->smap(hidden_inode, block);
    else
	err = bmap(hidden_inode, block);

    print_exit_status(err);
    return err;
}


/*
 * We don't need this because we will implement file_write, which will
 * call the hidden file_write, and the latter, if set to generic_file_write,
 * will call hidden's update_page (if implemented by lower file system).
 */
STATIC int
cryptfs_updatepage(file_t *file, page_t *page, const char *buf,
		  unsigned long offset, unsigned int count, int sync)
{
    int err = -EIO;
    file_t *hidden_file = ftohf(file);
    char *hidden_buf, *page_data;
    mm_segment_t old_fs;
    loff_t hidden_file_offset = file->f_pos;
    void *key;

    print_entry_location();
    if ((key = fist_get_userpass(file->f_dentry->d_sb)) == NULL) {
	err = -EACCES;
	goto out;
    }

    fist_dprint(6, "UPDATEPAGE1: buf[0]=%d, offset=0x%x, count=%d, sync=%d\n",
		(int) buf[0], (int) offset, count, sync);

    if (!hidden_file->f_op || !hidden_file->f_op->write)
	goto out;

    fist_print_file("Updatepage BEFORE file", file);
    fist_print_file("Updatepage BEFORE hidden_file", hidden_file);

    hidden_buf = kmalloc(count, GFP_KERNEL);
    cryptfs_encode_block(buf, hidden_buf, count, key);

    old_fs = get_fs();
    set_fs(KERNEL_DS);
    hidden_file->f_pos = file->f_pos;
    err = hidden_file->f_op->write(hidden_file,
				   hidden_buf,
				   count,
				   &hidden_file_offset);
    set_fs(old_fs);
    fist_print_file("Updatepage AFTER file", file);
    fist_print_file("Updatepage AFTER hidden_file", hidden_file);
    if (err >= 0) {
	//	file->f_pos = hidden_file_offset;
	page_data = (char *) page_address(page);
	memcpy(page_data, buf, count);
	//	file->f_pos = hidden_file->f_pos;
    }
    kfree_s(hidden_buf, count);

 out:
    print_exit_status(err);
    return err;
}


STATIC int
cryptfs_inode_revalidate(dentry_t *dentry)
{
    int err = 0;
    dentry_t *hidden_dentry = cryptfs_hidden_dentry(dentry);
    inode_t *hidden_inode = hidden_dentry->d_inode;

    print_entry_location();
    fist_print_dentry("inode_revalidate IN", dentry);

    if (hidden_inode->i_op && hidden_inode->i_op->revalidate) {
	err = hidden_inode->i_op->revalidate(hidden_dentry);
	if (!err)
	    fist_copy_attr_all(dentry->d_inode, hidden_inode);
    }

    fist_print_dentry("inode_revalidate OUT", dentry);
    print_exit_status(err);
    return err;
}


struct inode_operations cryptfs_iops =
{
    &cryptfs_fops,		/* default file operations */
    cryptfs_create,		/* create */
    cryptfs_lookup,		/* lookup */
    cryptfs_link,		/* link */
    cryptfs_unlink,		/* unlink */
    cryptfs_symlink,		/* symlink */
    cryptfs_mkdir,		/* mkdir */
    cryptfs_rmdir,		/* rmdir */
    cryptfs_mknod,		/* mknod */
    cryptfs_rename,		/* rename */
    cryptfs_readlink,		/* readlink */
#if 1
    cryptfs_follow_link,		/* follow_link */
#else
    NULL,			/* follow link */
#endif
    cryptfs_readpage,		/* readpage */
    cryptfs_writepage,		/* writepage */
    cryptfs_bmap,		/* bmap */
    cryptfs_truncate,		/* truncate */
    cryptfs_permission,		/* permission */
    cryptfs_smap,		/* smap */
#if 0
    cryptfs_updatepage,		/* update page (XXX: do we need this?) */
#else
    NULL,			/* update page (XXX: do we need this?) */
#endif
    cryptfs_inode_revalidate	/* revalidate */
};

/*
 * Local variables:
 * c-basic-offset: 4
 * End:
 */
