/*
 * java.net.PlainDatagramSocketImpl.c
 *
 * Copyright (c) 1996,97 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.co.uk>
 **/

/*** CHANGELOG ***
 *
 * 12.1.1998   Olli-Pekka Auvinen   Rewrote datagramSocketCreate, bind,
 *                                  send and close.
 * 13.1.1998   Olli-Pekka Auvinen   Removed #include <netinet/in.h>,
 *                                  #include <sys/types.h> and.
 *                                  #include <sys/socket.h>.
 *
 * 16.1.1998   Olli-Pekka Auvinen   Completed bind.
 *                                  Rewrote receive, socketSetOptions and
 *                                  socketGetOptions.
 *                                  Added seek to start of ctl file.
 *                                  Rewrote multicast functions as
 *                                  unimplemented in plan9.
 *
 * 10.2.1998   Olli-Pekka Auvinen   Added the dokumatic documentation. 
 *
 */

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "config-io.h"
#include <native.h>
#include "../../APIcore/java.lang.stubs/Integer.h"
#include "../../APIcore/java.io.stubs/FileDescriptor.h"
#include "../java.net.stubs/DatagramPacket.h"
#include "../java.net.stubs/PlainDatagramSocketImpl.h"
#include "../java.net.stubs/InetAddress.h"
#include "../java.net.stubs/SocketOptions.h"
#include "nets.h"
#include "kthread.h"
#include "support.h"

/**
  @title java_net_PlainDatagramSocketImpl
  @desc Instances of this class represent plain datagram (UDP) sockets.
  @funcidx
  @end
  */

/**
  @function    java_net_PlainDatagramSocketImpl_datagramSocketCreate.
  @description Create a new datagram socket.
  @parameter   Reference to the current (this) object.
  @end
  */

void java_net_PlainDatagramSocketImpl_datagramSocketCreate(struct Hjava_net_PlainDatagramSocketImpl* this)
{
  int ctlFileFD;

  /* Creating a datagram socket in Plan9 is implemented with a udp device
     which is controlled though its ctl file. Socket's file descriptor is
     emulated with this ctl file's file descriptor. */
  
  ctlFileFD = open("/net/udp/clone", ORDWR);
  
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file opening failed");
  }
  
  unhand(unhand(this)->fd)->fd = ctlFileFD;
}

/**
  @function    java_net_PlainDatagramSocketImpl_bind
  @description Bind a port to the socket (address is ignored in Plan9).
  @parameter   Reference to the current (this) object.
  @parameter   The port number to bind to.
  @parameter   The local address to bind to.
  @end
  */

void java_net_PlainDatagramSocketImpl_bind(struct Hjava_net_PlainDatagramSocketImpl* this, jint lport, struct Hjava_net_InetAddress* laddr)
{
  char bindString[MAX_PATH_LEN], connNumber[MAX_CNUM_LEN];
  char *portString;
  int ctlFileFD, localFileFD, r;
  
  USED(laddr);

  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }

  /* Construct the bind call string and write it to the ctl file of the udp
     device. */
  
  sprint(bindString, "bind %d", lport);
  
  r = strlen(bindString);
  if (write(ctlFileFD, bindString, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Ctl file writing failed");
  }
  
  /* Open the local file from the protocol directory and parse the local
     port number from there. The local port number is then set to a
     member variable. This way the correct port number is got in case
     the bind did not succeed, otherwise the argument port number could
     have been used. */
  
  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  r = read(ctlFileFD, connNumber, MAX_CNUM_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  connNumber[r] = '\0';
  
  sprint(bindString, "/net/tcp/%d/local", atoi(connNumber));
  
  localFileFD = open(bindString, OREAD);
  if (localFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file opening failed");
  }
  
  r = read(localFileFD, bindString, MAX_PATH_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Local file reading failed");
  }
  bindString[r] = '\0';
  
  portString = strstr(bindString, "!");
  portString++;
  
  /* Set the member variables according to the new state of the udp device. */
  
  unhand(this)->localPort = atoi(portString);
  
  if (close(localFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file closing failed");
  }
}

/**
  @function    java_net_PlainDatagramSocketImpl_send
  @description Send a datagram to the socket.
  @parameter   Reference to the current (this) object.
  @parameter   The datagram packet object to be sent.
  @end
  */

void java_net_PlainDatagramSocketImpl_send(struct Hjava_net_PlainDatagramSocketImpl* this, struct Hjava_net_DatagramPacket* pkt)
{
  char sendString[MAX_PATH_LEN], connNumber[MAX_CNUM_LEN];
  int ctlFileFD, dataFileFD, dport;
  int ipAddress, ip1, ip2, ip3, ip4, r;

  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }
  
  /* Construct the connection call string and write it to the ctl file of the
     udp device. */
  
  ipAddress = unhand(unhand(pkt)->address)->address;
  dport = unhand(pkt)->port;
  
  ip1 = (ipAddress & 0xff000000) >> 24;
  ip2 = (ipAddress & 0x00ff0000) >> 16;
  ip3 = (ipAddress & 0x0000ff00) >> 8;
  ip4 = (ipAddress & 0x000000ff);
  sprint(sendString, "connect %d.%d.%d.%d!%d", ip1, ip2, ip3, ip4, dport);
  
  r = strlen(sendString);
  if (write(ctlFileFD, sendString, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Connecting failed");
  }
  
  /* Open the data file of the udp device and write the contents of the
     requested packet to the file. */
  
  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  if (read(ctlFileFD, connNumber, MAX_CNUM_LEN) < 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  sprint(sendString, "/net/udp/%d/data", atoi(connNumber));
  
  dataFileFD = open(sendString, OWRITE);
  if (dataFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Data file opening failed");
  }
  
  r = unhand(pkt)->length;
  if (write(dataFileFD, unhand(unhand(pkt)->buf)->body, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Data file writing failed");
  }
  
  if (close(dataFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException", "Data file closing failed");
  }
  
  /* Clear the remote address of the UDP connection. */
  
  sprint(sendString, "disconnect");
  
  r = strlen(sendString);
  if (write(ctlFileFD, sendString, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Disconnecting failed");
  }
}

jint java_net_PlainDatagramSocketImpl_peek(struct Hjava_net_PlainDatagramSocketImpl* this, struct Hjava_net_InetAddress* addr)
{
  USED(this);
  USED(addr);
  unimp("java.net.PlainDatagramSocketImpl:peek unimplemented");
  return -1;
}


/**
  @function    java_net_PlainDatagramSocketImpl_receive
  @description Receive a datagram from the socket.
  @parameter   Reference to the current (this) object.
  @parameter   The datagram packet object to be received.
  @end
  */

void java_net_PlainDatagramSocketImpl_receive(struct Hjava_net_PlainDatagramSocketImpl* this, struct Hjava_net_DatagramPacket* pkt)
{
  int ctlFileFD, dataFileFD, remoteFileFD, daddr, r;
  char receiveString[MAX_PATH_LEN], connNumber[MAX_CNUM_LEN];
  char *portString, *ip1, *ip2, *ip3, *ip4;
  struct Hjava_net_InetAddress **fromaddr;
  
  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }
  
  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  r = read(ctlFileFD, connNumber, MAX_CNUM_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  connNumber[r] = '\0';
  
  /* Open the data file of the udp device and read requested amount of
     bytes from the file. */
  
  sprint(receiveString, "/net/udp/%d/data", atoi(connNumber));
  
  dataFileFD = open(receiveString, OREAD);
  
  if (dataFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Data file opening failed");
  }
  
  r = read(dataFileFD, unhand(unhand(pkt)->buf)->body, unhand(pkt)->length);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Data file reading failed");
  }
  unhand(pkt)->length = r;
  
  /* Open the remote file from the protocol directory and parse the name
     and port of the peer protocol entity from there. */
  
  sprint(receiveString, "/net/tcp/%d/remote", atoi(connNumber));
  
  remoteFileFD = open(receiveString, OREAD);
  if (remoteFileFD <0) {
    SignalError(NULL, "java.net.SocketException",
		"Remote file opening failed");
  }
  
  r = read(remoteFileFD, receiveString, MAX_PATH_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException",
		"Remote file reading failed");
  }
  receiveString[r] = '\0';
  
  portString = strstr(receiveString, "!");
  *portString = '\0';
  portString++;
  
  ip1 = receiveString;  
  ip2 = strstr(ip1, ".");
  *ip2 = '\0';
  ip2++;

  ip3 = strstr(ip2, ".");
  *ip3 = '\0';
  ip3++;
  
  ip4 = strstr(ip3, ".");
  *ip4 = '\0';
  ip4++;

  daddr = (atoi(ip1) << 24) + (atoi(ip2) << 16) + (atoi(ip3) << 8) + atoi(ip4);
  
  /* Set the member variables according to the new state of the udp device. */
  
  unhand(pkt)->port = atoi(portString);

  fromaddr = &unhand(pkt)->address;
  if (*fromaddr == 0) {
    *fromaddr =
      (struct Hjava_net_InetAddress *) AllocObject("java/net/InetAddress");
  }
  unhand(*fromaddr)->address = daddr;
  
  if (close(remoteFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException",
		"Remote file closing failed");
  }
  
  if (close(dataFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException",
		"Data file closing failed");
  }
}

/**
  @function    java_net_PlainDatagramSocketImpl_datagramSocketClose
  @description Closes the datagram socket.
  @parameter   Reference to the current (this) object.
  @end
  */

void java_net_PlainDatagramSocketImpl_datagramSocketClose(struct Hjava_net_PlainDatagramSocketImpl* this)
{
  int ctlFileFD, r;
  
  /* Simply close the ctl file and set the file descriptor instance variable
     to -1. */
  
  ctlFileFD = unhand(unhand(this)->fd)->fd;
  
  if (ctlFileFD != -1) {
    
    r = close(ctlFileFD);
    unhand(unhand(this)->fd)->fd = -1;
    
    if (r < 0) {
      SignalError(NULL, "java.net.SocketException", "Ctl file closing failed");
    }
  }
}

void java_net_PlainDatagramSocketImpl_socketSetOption(struct Hjava_net_PlainDatagramSocketImpl* this, jint v1, struct Hjava_lang_Object* v2)
{
  USED(v2);
  USED(this);
  
  switch(v1) {
  
    /* SO_BINDADDR, SO_REUSEADDR and IP_MULTICAST_IF are currently implemented
       in kaffe, but due to incompatibility between the plan9 and linux
       tcp/ip implementations they are not supported anymore in plan9 */
    
  case java_net_SocketOptions_SO_BINDADDR:
  case java_net_SocketOptions_SO_REUSEADDR:
  case java_net_SocketOptions_IP_MULTICAST_IF:
  case java_net_SocketOptions_SO_TIMEOUT: /* JAVA takes care **/

  default:
    
    SignalError(NULL, "java.net.SocketException",
		"Unimplemented socket option");     
  }
}

jint java_net_PlainDatagramSocketImpl_socketGetOption(struct Hjava_net_PlainDatagramSocketImpl* this, jint val)
{
  int r;
  USED(this);
  SET(r);

  switch(val) {
    
  case java_net_SocketOptions_SO_BINDADDR:

    r = INADDR_ANY;
    break;
    
    /* SO_REUSEADDR and IP_MULTICAST_IF are currently implemented
       in kaffe, but due to incompatibility between the plan9 and linux
       tcp/ip implementations they are not supported anymore in plan9 */
    
  case java_net_SocketOptions_SO_REUSEADDR:
  case java_net_SocketOptions_IP_MULTICAST_IF:
  case java_net_SocketOptions_SO_TIMEOUT: /* JAVA takes care **/
    
  default:
    
    SignalError(NULL, "java.net.SocketException",
		"Unimplemented socket option");    
  }
  return (r);
}

void java_net_PlainDatagramSocketImpl_join(struct Hjava_net_PlainDatagramSocketImpl* this, struct Hjava_net_InetAddress* laddr)
{
  USED(this);
  USED(laddr);
  unimp("java.net.PlainDatagramSocketImpl:join unimplemented");
}

void java_net_PlainDatagramSocketImpl_leave(struct Hjava_net_PlainDatagramSocketImpl* this, struct Hjava_net_InetAddress* laddr)
{
  USED(this);
  USED(laddr);
  unimp("java.net.PlainDatagramSocketImpl:leave unimplemented");
}

void java_net_PlainDatagramSocketImpl_setTTL(struct Hjava_net_PlainDatagramSocketImpl* this, jbool ttl)
{
  USED(this);
  USED(ttl);
  unimp("java.net.PlainDatagramSocketImpl:setTTL unimplemented");
}

jbyte java_net_PlainDatagramSocketImpl_getTTL(struct Hjava_net_PlainDatagramSocketImpl* this)
{
  USED(this);
  unimp("java.net.PlainDatagramSocketImpl:getTTL unimplemented");
  return -1;
}




