/*
 * java.lang.Class.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 */

/*** CHANGELOG ***
 *
 * 21.1.1998   Vesa Karpijoki   Added #includes <u.h> and <libc.h>.
 *
 * 18.3.1998   Vesa Karpijoki   Added Dokumatic documentation.
 *                              Changed #include:s.
 *
 */

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "../../../src/gtypes.h"
#include "../../../src/access.h"
#include "../../../src/constants.h"
#include "../../../src/object.h"
#include "../../../src/classMethod.h"
#include "../../../src/itypes.h"
#include "../../../src/support.h"
#include "../../../src/soft.h"
#include "../java.io.stubs/InputStream.h"
#include "../java.io.stubs/PrintStream.h"
#include "../java.lang.stubs/System.h"
#include "../java.lang.reflect.stubs/Field.h"
#include <native.h>
#include "defs.h"

/**
  @title java_lang_Class
  @desc Native methods of the Java API class java.lang.Class.
  @funcidx
  @end
  */

/**
  @function java_lang_Class_forName
  @description Converts string name to class object.
  @parameter Reference to the string to be converted.
  @rvalue Reference to the converted class.
  @end
  */
struct Hjava_lang_Class* java_lang_Class_forName(struct Hjava_lang_String* str)
{
	Hjava_lang_Class* clazz;
	char buf[MAXNAMELEN];

	/* Get string and convert '.' to '/' */
	javaString2CString(str, buf, sizeof(buf));
	classname2pathname(buf, buf);

	clazz = loadClass(makeUtf8Const (buf, strlen(buf)), 0);
	processClass(clazz, CSTATE_OK);

	return (clazz);
}

/**
  @function java_lang_Class_getName
  @description Converts class to string name.
  @parameter Reference to the class to be converted.
  @rvalue Reference to the converted string.
  @end
  */
struct Hjava_lang_String* java_lang_Class_getName(struct Hjava_lang_Class* c)
{
	char buffer[100];
	struct Hjava_lang_String* str;
	int len;
	char* name;
	int i;
	char* ptr;
	char ch;

	len = c->name->length;
	name = len > 100 ? (char*)gc_malloc_fixed(len) : buffer;
	ptr = c->name->data;
	for (i = 0; i < len; i++) {
		ch = *ptr++;
		if (ch == '/') {
			ch = '.';
		}
		name[i] = ch;
	}
	str = makeJavaString(name, len);
	if (name != buffer) {
		gc_free (name);
	}
	return (str);
}

/**
  @function java_lang_Class_newInstance
  @description Creates a new instance of the derived class.
  @parameter Reference to the current class (this).
  @rvalue Reference to the new instance. 
  @end
  */
struct Hjava_lang_Object* java_lang_Class_newInstance(struct Hjava_lang_Class* this)
{
	return (execute_java_constructor(0, 0, this, "()V"));
}

/**
  @function java_lang_Class_newInstance
  @description Returns the super class of the current class.
  @parameter Reference to the current class (this).
  @rvalue Reference to the super class.
  @end
  */
struct Hjava_lang_Class* java_lang_Class_getSuperclass(struct Hjava_lang_Class* this)
{
	return (this->superclass);
}

/**
  @function java_lang_Class_getInterfaces
  @description Returns the interfaces implemented by this class.
  @parameter Reference to the current class (this).
  @rvalue Reference to the array of the interfaces.
  @end
  */
HArrayOfObject* java_lang_Class_getInterfaces(struct Hjava_lang_Class* this)
{
	HArrayOfObject* obj;
	struct Hjava_lang_Class** ifaces;
	int i;

	obj = (HArrayOfObject*)AllocObjectArray(this->interface_len, "Ljava/lang/Class");
	ifaces = (struct Hjava_lang_Class**)unhand(obj)->body;
	for (i = 0; i < this->interface_len; i++) {
		ifaces[i] = this->interfaces[i];
	}

	return (obj);
}

/**
  @function java_lang_ClassLoader_getClassLoader
  @description Returns the class loader which loaded me.
  @parameter Reference to the current class (this).
  @rvalue Reference to the class loader.
  @end
  */
struct Hjava_lang_ClassLoader* java_lang_Class_getClassLoader(struct Hjava_lang_Class* this)
{
	return (this->loader);
}

/**
  @function java_lang_Class_isInterface
  @description Decides whether the current class is an interface.
  @parameter Reference to the current class (this).
  @rvalue Boolean that has value <i>true</i> if the class is a interface, <i>false</i>
  otherwise.
  @end
  */
jbool java_lang_Class_isInterface(struct Hjava_lang_Class* this)
{
	return ((this->accflags & ACC_INTERFACE) ? 1 : 0);
}

/**
  @function java_lang_Class_isPrimitive
  @description Decides whether the current class is a primitive.
  @parameter Reference to the current class (this).
  @rvalue Boolean that has value <i>true</i> if the class is a primitive, <i>false</i>
  otherwise.
  @end
  */
jbool java_lang_Class_isPrimitive(struct Hjava_lang_Class* this)
{
	return (CLASS_IS_PRIMITIVE(this));
}

/**
  @function java_lang_Class_isArray
  @description Decides whether the current class is an array.
  @parameter Reference to the current class (this).
  @rvalue Boolean that has value <i>true</i> if the class is an array, <i>false</i>
  otherwise.
  @end
  */
jbool java_lang_Class_isArray(struct Hjava_lang_Class* this)
{
	return (CLASS_IS_ARRAY(this));
}

/**
  @function java_lang_Class_getComponentType
  @description Decides whether the current class is an array.
  @parameter Reference to the current class (this).
  @rvalue Reference to an element if (this) is an array, otherwise 0 is returned.
  @end
  */
Hjava_lang_Class* java_lang_Class_getComponentType(struct Hjava_lang_Class* this)
{
    /*	Hjava_lang_Class* type; */

	if (CLASS_IS_ARRAY(this)) {
		return (CLASS_ELEMENT_TYPE(this));
	}
	else {
		return ((Hjava_lang_Class*)0);
	}
}

/**
  @function java_lang_Class_isAssignableFrom
  @description Decides whether the current class is assignable from another class.
  @parameter Reference to the current class (this).
  @parameter Reference to the other class.
  @rvalue Boolean that has value <i>true</i> if (this) is assignable from the other class,
  <i>false</i> otherwise.
  @end
  */
jbool java_lang_Class_isAssignableFrom(struct Hjava_lang_Class* this, struct Hjava_lang_Class* cls)
{
	return (instanceof(this, cls));
}

/**
  @function java_lang_Class_getPrimitiveClass
  @description Gets primitive class from class name (JDK 1.1).
  @parameter Reference to the class name.
  @rvalue Reference to the primitive class if found, otherwise NULL.
  @end
  */
struct Hjava_lang_Class* java_lang_Class_getPrimitiveClass(struct Hjava_lang_String* name)
{
	jchar* chrs;

	chrs = &unhand(unhand(name)->value)->body[unhand(name)->offset];
	switch (chrs[0]) {
	case 'b':
		if (chrs[1] == 'y') {
			return (&byteClass);
		}
		if (chrs[1] == 'o') {
			return (&booleanClass);
		}
		break;
	case 'c':
		return (&charClass);
	case 'd':
		return (&doubleClass);
	case 'f':
		return (&floatClass);
	case 'i':
		return (&intClass);
	case 'l':
		return (&longClass);
	case 's':
		return (&shortClass);
	case 'v':
		return (&voidClass);
	}
	return(NULL);
}

/**
  @function java_lang_Class_isInstance
  @description Checks whether an object is an instance of this class.
  @parameter Reference to the current (this) class.
  @parameter Reference to the object to be checked.
  @rvalue Boolean value <i>true</i> if the object is an instance of the class, <i>false</i>
  otherwise.
  @end
  */
jbool java_lang_Class_isInstance(struct Hjava_lang_Class* this, struct Hjava_lang_Object* obj)
{
	return (soft_instanceof(this, obj));
}

/**
  @function java_lang_Class_getModifiers
  @description Gets the modifiers of the current (this) class.
  @parameter Reference to the current (this) class.
  @rvalue The modifiers of the class.
  @end
  */
jint java_lang_Class_getModifiers(struct Hjava_lang_Class* this)
{
	return (this->accflags);
}

/**
  @function java_lang_Class_getSigners
  @description Unimplemented!
  @rvalue -1
  @end
  */
jint java_lang_Class_getSigners()
{
	unimp("java.lang.Class:getSigners unimplemented");
	return -1;
}

/**
  @function java_lang_Class_setSigners
  @description Unimplemented!
  @rvalue -1
  @end
  */
jint java_lang_Class_setSigners()
{
	unimp("java.lang.Class:setSigners unimplemented");
	return -1;
}

/**
  @function java_lang_Class_getMethods0
  @description Unimplemented!
  @rvalue -1
  @end
  */
jint java_lang_Class_getMethods0()
{
	unimp("java.lang.Class:getMethods0 unimplemented");
	return -1;
}

/**
  @function java_lang_Class_getConstructors0
  @description Unimplemented!
  @rvalue -1
  @end
  */
jint java_lang_Class_getConstructors0()
{
	unimp("java.lang.Class:getConstructors0 unimplemented");
	return -1;
}

/**
  @function java_lang_Class_getMethod0
  @description Unimplemented!
  @rvalue -1
  @end
  */
jint java_lang_Class_getMethod0()
{
	unimp("java.lang.Class:getMethod0 unimplemented");
	return -1;
}

/**
  @function java_lang_Class_getConstructor0
  @description Unimplemented
  @rvalue -1
  @end
  */
jint java_lang_Class_getConstructor0()
{
	unimp("java.lang.Class:getConstructor0 unimplemented");
	return -1;
}

/**
  @function makeField
  @description Makes a new field to a given class.
  @parameter Reference to a class.
  @parameter Slot where the field is put.
  @rvalue Reference to the new Field object.
  @end
  */
static Hjava_lang_reflect_Field* makeField(struct Hjava_lang_Class* clazz, int slot)
{
	Hjava_lang_reflect_Field* field;
	Field* fld;

	fld = CLASS_FIELDS((Hjava_lang_Class*)clazz) + slot;
	field = (Hjava_lang_reflect_Field*)AllocObject("java/lang/reflect/Field");
	unhand(field)->clazz = (struct Hjava_lang_Class*) clazz;
	unhand(field)->slot = slot;
	unhand(field)->type = (struct Hjava_lang_Class*) fld->type;
	unhand(field)->name = Utf8Const2JavaString(fld->name);
	return (field);
}

/**
  @function java_lang_Class_getFields0
  @description Gets the public fields of the given class.
  <b>NOTE:</b> This method does not work properly!!
  @parameter Reference to a class.
  @parameter Check whether the class is declared.
  @rvalue Reference to the fields array.
  @end
  */
HArrayOfObject* java_lang_Class_getFields0(struct Hjava_lang_Class* clazz, int declared)
{
	int count;
	Hjava_lang_Class* clas;
	Field* fld;
	Hjava_lang_reflect_Field** ptr;
	HArrayOfObject* array;
	int i;

	if (declared) {
		count = CLASS_NFIELDS((Hjava_lang_Class*)clazz);
	}
	else {
		count = 0;
		clas = (Hjava_lang_Class*) clazz;
		for (; clas != NULL; clas = clas->superclass) {
			fld = CLASS_FIELDS(clas);
			i = CLASS_NFIELDS(clas);
			for ( ; --i >= 0;  ++fld) {
				if (fld->accflags & ACC_PUBLIC) {
					count++;
				}
			}
		}
	}
	array = (HArrayOfObject*)AllocObjectArray(count, "Ljava/lang/reflect/Field;");
	ptr = (Hjava_lang_reflect_Field**) ARRAY_DATA(array) + count;
	clas = (Hjava_lang_Class*) clazz;
	do {
		fld = CLASS_FIELDS(clas);
		i = CLASS_NFIELDS(clas);
		for ( ; --i >= 0;  ++fld) {
			if (! (fld->accflags & ACC_PUBLIC) && ! declared) {
				continue;
			}
			*--ptr = makeField(clas, i);
		}
		clas = clas->superclass;
	} while (!declared && clas != NULL);

	return (array);
}


/**
  @function java_lang_Class_getField0
  @description Gets a certain field of the given class.
  @parameter Reference to a class.
  @parameter Reference to the field.
  @parameter Check whether the class is declared.
  @rvalue Reference to the field if found, otherwise NULL.
  @end
  */
Hjava_lang_reflect_Field* java_lang_Class_getField0(struct Hjava_lang_Class* clazz, struct Hjava_lang_String* name, int declared)
{
	Hjava_lang_Class* clas;

	clas = (Hjava_lang_Class*) clazz;
	do {
		Field* fld = CLASS_FIELDS(clas);
		int n = CLASS_NFIELDS(clas);
		int i;
		for (i = 0;  i < n;  ++fld, ++i) {
			if (((fld->accflags & ACC_PUBLIC) || declared)
			    && equalUtf8JavaStrings (fld->name, name)) {
				return makeField(clas, i);
			}
		}
		clas = clas->superclass;
	} while (!declared && clas != NULL);
	SignalError(0, "java.lang.NoSuchFieldException", ""); /* FIXME */
	return NULL;
}
