/*
 * java.io.FileInputStream.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 **/

/*** CHANGELOG ***
 *
 *  1.2.1998   Arttu Kuukankorpi          changed call to open
 *                                        changed off_t:s to jlong:s
 *                                        rewrote FileInputStream_available
 *                                        included u.h and libc.h
 *
 *  2.2.1998   Vesa Karpijoki             fixed some bugs (lseek -> seek)
 *
 * 12.2.1998   Teemu Ikonen               fixed some fatal bugs, (java-compiler 
 *                                        killer bugs)
 *
 * 15.2.1998   Arttu Kuukankorpi          added dokumatic-dokumentation
 *
 * 16.3.1998   Arttu Kuukankorpi          changed include-directories of
 *                                         FileInputStream.h and
 *                                         FileDescriptor.h
 */


#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-mem.h"
#include "config-io.h"
#include "files.h"
#include "defs.h"
#include "../java.io.stubs/FileInputStream.h"
#include "../java.io.stubs/FileDescriptor.h"
#include "kthread.h"

/**
  @title java_io_FileInputStream
  @desc A file input stream is an input stream for reading data from a File or from a FileDescriptor.
  @funcidx
  @end
  */

/**
  @function java_io_FileInputStream_open
  @description Open a file for input.
  @parameter Reference to the current (this) object.
  @parameter Name of the file to be opened.
  @end
  */

void
java_io_FileInputStream_open(struct Hjava_io_FileInputStream* this, struct Hjava_lang_String* name)
{
	char str[MAXPATHLEN];
	int fd;

	javaString2CString(name, str, sizeof(str));
	fd = open(str, OREAD);
	unhand(unhand(this)->fd)->fd = fd;
	if (fd < 0) {
		SignalError(0, "java.io.IOException", SYS_ERROR);
	}
}

/**
  @function java_io_FileInputStream_close
  @description Close file.
  @parameter Reference to the current (this) object.
  @end
  */

void
java_io_FileInputStream_close(struct Hjava_io_FileInputStream* this)
{
	int r;

	if (unhand(unhand(this)->fd)->fd >= 0) {
		r = close(unhand(unhand(this)->fd)->fd);
		unhand(unhand(this)->fd)->fd = -1;
		if (r < 0) {
			SignalError(NULL, "java.io.IOException", SYS_ERROR);
		}
	}
}

/**
  @function java_io_FileInputStream_readBytes
  @description Read in bytes.
  @parameter Stream to read from.
  @parameter Array of bytes to be filled with read data.
  @parameter Offset.
  @parameter Length.
  @rvalue Number of bytes read, -1 in case of error.
  @end
  */

jint
java_io_FileInputStream_readBytes(struct Hjava_io_FileInputStream* fh, HArrayOfByte* bytes, jint off, jint len)
{
	jint ret;

	ret = read(unhand(unhand(fh)->fd)->fd, &unhand(bytes)->body[off], len);
	if (ret < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
	return (ret > 0 ? ret : -1);
}

/**
  @function java_io_FileInputStream_read
  @description Read a single byte.
  @parameter Stream to read from.
  @rvalue Byte read, -1 in case of error.
  @end
  */

jint
java_io_FileInputStream_read(struct Hjava_io_FileInputStream* fh)
{
	jint ret;
	/* unsigned */ char byte;

	ret = read(unhand(unhand(fh)->fd)->fd, &byte, 1);
	if (ret < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
	return (ret > 0 ? byte : -1);
}

/**
  @function java_io_FileInputStream_skip
  @description Skip forward in stream.
  @parameter Stream to skip forward in.
  @parameter Offset.
  @rvalue Number of bytes read skipped.
  @end
  */

jlong
java_io_FileInputStream_skip(struct Hjava_io_FileInputStream* fh, jlong off)
{
	jlong orig;
	jlong ret;
	char buffer[100];
	int count;

	orig = seek(unhand(unhand(fh)->fd)->fd, (jlong)0, 1);
	ret = seek(unhand(unhand(fh)->fd)->fd, off, 1);
	if (ret >= 0) {
		return (ret-orig);
	}

	/* Not seekable - try just reading. **/
	ret = 0;
	while (off > 0) {
		count = read(unhand(unhand(fh)->fd)->fd, buffer, 100);
		if (count == 0) {	/* Reached EOF. **/
			break;
		}
		if (count < 0) {
			SignalError(NULL, "java.io.IOException", SYS_ERROR);
		}
		ret += count;
		off -= count;
	}
	return (ret);
}

/**
  @function java_io_FileInputStream_available
  @description Return the number of bytes available to read without blocking.
  @parameter Stream to read from.
  @rvalue Number of bytes available to read without blocking.
  @end
  */

jint
java_io_FileInputStream_available(struct Hjava_io_FileInputStream* fh)
{
    /*
	int nr;
	int r;

	r = ioctl(unhand(unhand(fh)->fd)->fd, FIONREAD, &nr);
	if (r < 0) {
		nr = 0;
	}


	return (nr);
     */

    jint cur, end, pos;

    cur = seek(unhand(unhand(fh)->fd)->fd, 0L, 1);
    end = seek(unhand(unhand(fh)->fd)->fd, 0L, 2);
    pos = seek(unhand(unhand(fh)->fd)->fd, cur, 0);

    if( pos < 0 || cur < 0 || end < 0 )
	return 0;

    return (end - cur);
}
