# lib/Software/License/GPL_3/or_later.pm #

#pod =encoding UTF-8
#pod
#pod =head1 COPYRIGHT AND LICENSE
#pod
#pod Copyright © 2015 Van de Bugger
#pod
#pod This file is part of Software-License-OrLaterPack.
#pod
#pod Software-License-OrLaterPack is free software: you can redistribute it and/or modify it under the
#pod terms of the GNU General Public License as published by the Free Software Foundation, either
#pod version 3 of the License, or (at your option) any later version.
#pod
#pod Software-License-OrLaterPack is distributed in the hope that it will be useful, but WITHOUT ANY
#pod WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#pod See the GNU General Public License for more details.
#pod
#pod You should have received a copy of the GNU General Public License along with
#pod Software-License-OrLaterPack. If not, see <http://www.gnu.org/licenses/>.
#pod
#pod =cut

#pod =head1 DESCRIPTION
#pod
#pod C<Software::License::GPL_3::or_later> is a subclass of L<Software::License::GPL_3>. It overrides
#pod few parent methods:
#pod
#pod =for :list
#pod = C<name>
#pod appends ", or any later" clause to the parent license name;
#pod = C<version>
#pod appends "+" to the parent license version;
#pod = C<notice>
#pod returns copyright statement and standard GNU 3-paragraph license notice.
#pod
#pod See documentation on L<Software::License> for a general description of class interface.
#pod
#pod =cut

package Software::License::GPL_3::or_later;

# ABSTRACT: GPLv3+ license for Software::License

use strict;
use warnings;

our $VERSION = '0.006_003'; # TRIAL VERSION

use parent 'Software::License::GPL_3';
use Text::Wrap;

#pod =method _name
#pod
#pod Returns the name of license, with "the" article in front of name (in lower case). It does include
#pod neither license version nor "or any later" clause.
#pod
#pod =cut

sub _name { 'the GNU General Public License'; };

#pod =method _version
#pod
#pod Returns version of the license, just a number, no date, no plus sign at the end.
#pod
#pod =cut

sub _version { '3'; };

#pod =method name
#pod
#pod This method, in contrast to L<_name>, meets <Software::License> specification: Definitive article
#pod is capitalized ("The"), name includes version and "or any later" clause.
#pod
#pod =cut

sub name {
    sprintf( "%s version %s, or any later", ucfirst( $_[ 0 ]->_name ), $_[ 0 ]->_version );
}; # sub name

#pod =method version
#pod
#pod Returns version with trailing plus sign.
#pod
#pod =cut

sub version { $_[ 0 ]->_version . '+'; };

#pod =method notice
#pod
#pod This method overrides L<Software::License>'s C<notice>. Differences are:
#pod
#pod =for :list
#pod * The method has optional parameter — program name (default name is "this program"). It is used in
#pod notice text.
#pod * It returns copyright statement followed by standard GNU 3-paragraph license notice.
#pod * Result is formatted with L<Text::Wrap::fill>.
#pod
#pod B<Note:> Program name is an experimental feature.
#pod
#pod =cut

sub notice {
    my ( $self, $program ) = @_;
    #   Documentation on `fill` says it deletes all trailing whitespace, but it looks like it
    #   may leave one space. Let us make sure notice ends with one newline.
    $self->{ program } = defined( $program ) ? $program : 'this program';
    ( my $notice = Text::Wrap::fill( '', '', $self->SUPER::notice() ) ) =~ s{\s*\z}{\n};
    delete( $self->{ program } );
    return $notice;
}; # sub notice

1;

=pod

=encoding UTF-8

=head1 NAME

Software::License::GPL_3::or_later - GPLv3+ license for Software::License

=head1 VERSION

Version 0.006_003.
This is a B<trial release>.

=head1 DESCRIPTION

C<Software::License::GPL_3::or_later> is a subclass of L<Software::License::GPL_3>. It overrides
few parent methods:

=over 4

=item C<name>

appends ", or any later" clause to the parent license name;

=item C<version>

appends "+" to the parent license version;

=item C<notice>

returns copyright statement and standard GNU 3-paragraph license notice.

=back

See documentation on L<Software::License> for a general description of class interface.

=head1 METHODS

=head2 _name

Returns the name of license, with "the" article in front of name (in lower case). It does include
neither license version nor "or any later" clause.

=head2 _version

Returns version of the license, just a number, no date, no plus sign at the end.

=head2 name

This method, in contrast to L<_name>, meets <Software::License> specification: Definitive article
is capitalized ("The"), name includes version and "or any later" clause.

=head2 version

Returns version with trailing plus sign.

=head2 notice

This method overrides L<Software::License>'s C<notice>. Differences are:

=over 4

=item *

The method has optional parameter — program name (default name is "this program"). It is used in

notice text.

=item *

It returns copyright statement followed by standard GNU 3-paragraph license notice.

=item *

Result is formatted with L<Text::Wrap::fill>.

=back

B<Note:> Program name is an experimental feature.

=head1 AUTHOR

Van de Bugger <van.de.bugger@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright © 2015 Van de Bugger

This file is part of Software-License-OrLaterPack.

Software-License-OrLaterPack is free software: you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software Foundation, either
version 3 of the License, or (at your option) any later version.

Software-License-OrLaterPack is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with
Software-License-OrLaterPack. If not, see <http://www.gnu.org/licenses/>.

=cut

__DATA__

__[ BEGIN BLOCK ]__
# BEGIN_BLOCK: Notice

__NOTICE__
Copyright (C) {{ $self->year }} {{ $self->holder }}

{{ ucfirst( $self->{ program } ) }} is free software: you can redistribute it and/or modify it
under the terms of {{ $self->_name }} as published by the Free Software Foundation, either version
{{ $self->_version }} of the License, or (at your option) any later version.

{{ ucfirst( $self->{ program } ) }} is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE. See {{ $self->_name }} for more details.

You should have received a copy of {{ $self->_name }} along with {{ $self->{ program } }}. If not,
see <http://www.gnu.org/licenses/>.

__[ END BLOCK ]__
# END_BLOCK: Notice

# end of file #
