use warnings;
use strict;

use inc::Module::Install '1.01';

use 5.006002;

use File::Spec;
my $IS_SMOKER = (
  $ENV{AUTOMATED_TESTING}
    and
  ! $ENV{PERL5_CPANM_IS_RUNNING}
    and
  ! $ENV{RELEASE_TESTING}
    and
  ! -d '.git'
    and
  -d 'inc'
    and
  ! -e File::Spec->catfile('inc', $^O eq 'VMS' ? '_author' : '.author')
);


all_from 'lib/OS/ABI/Constants.pm';

requires 'Sys::Info' => '0.78';

if (my $cc = can_cc) {
  # this is a horrendous hack to force an UNKNOWN report as opposed to N/A
  # because of the destructiveness we go to great lengths above to make sure
  # we are in fact a smoker
  local $SIG{__DIE__} = sub {
    my $err = shift;

    require File::Path;
    File::Path::rmtree('t');

    die "$err\n";
  } if $IS_SMOKER;

  gen_my_abi($cc);
}
### FIXME - temporary, for the duration of the survey
else {
  warn "The survey stage requires a compiler\n";
  exit 1;
}

# these are also temporary
configure_requires 'MIME::Base64' => 0;
configure_requires 'IO::Compress::Bzip2' => 0;
configure_requires 'Storable' => 0;

resources 'repository'  => 'http://github.com/ribasushi/OS-ABI-Constants';

WriteAll;

sub gen_my_abi {
  my $cc = shift;

  warn "C compiler `$cc` detected - gathering ABI constants from your system\n";
  my $abi = examine_abi($cc);

  my $fn = 'lib/OS/ABI/MyConstants.pm';
  unlink $fn; # don't care if there or not

  if (grep { keys %$_ } values %$abi) {
    warn "Writing constants to $fn\n";

    open (my $fh, '>', $fn) or die "Unable to open $fn: $!\n";

    require Data::Dumper;
    my $cdump = Data::Dumper->new([$abi])->Indent(1)->Useqq(1)->Terse(1)->Quotekeys(0)->Sortkeys(1)->Pad('  ')->Dump;
    my $ver = time();
    my $ver_comment = gmtime;

    print $fh <<EOC;
#############################################################
###                                                       ###
###    FILE AUTOGENERATED AT THE TIME OF INSTALLATION     ###
###                    DO NOT EDIT!!!                     ###
###                                                       ###
### If your system has changed drastically since - simply ###
### reinstall OS::ABI::Constants                          ###
###                                                       ###
#############################################################

package # not the package you are looking for
  OS::ABI::MyConstants;

use warnings;
use strict;

our \$VERSION = $ver; # $ver_comment

sub constants { return
$cdump}

1;

EOC

  }
  else {
    warn "No interesting constants found during ABI examination - skipping creation of $fn\n";
    # FIXME - temporary
    warn "No reason to continue during survey stage - exiting\n";
    exit 1;
  }
}

sub examine_abi {
  my $cc = shift;

  require IO::Handle;
  require IPC::Open3;
  require B;
  require Config;

  my $examine = {
    time => {
      headers => {
        generic => [ $Config::Config{timeincl} ? split / /, $Config::Config{timeincl} : 'time.h' ],
        linux => [qw| linux/time.h time.h |],
        MSWin32 => undef, # win32 pretty much does not support *ANY* of the POSIX time stuff,
                          # Time::HiRes ends up emulating everything :(
      },
      defs => qr/
        ITIMER_\w+ | CLOCK_\w+
      /x,
    },
    # not everything implements syscall()
    ! $Config::Config{d_syscall} ? () : ( syscall => {
      headers => {
        generic => 'syscall.h',
      },
      defs => qr/
        SYS_[a-z]\w+
      /x,
    }),
    # others...?
  };

  my ($my_abi, $errors);

  for my $type (keys %$examine) {
    $my_abi->{$type} = {};

    my $h = $examine->{$type}{headers};
    $h = ( exists $h->{$^O} ? $h->{$^O} : $h->{generic} );

    next unless defined $h;

    my $qr = sprintf '^ \#define \s+ ( %s ) \s+ \S', $examine->{$type}{defs};
    $qr = qr/$qr/x;

    my $c_src;
    for (ref $h eq 'ARRAY' ? @$h : $h) {
      $c_src .= sprintf "#include %s\n", $_ =~ m|^/| ? qq|"$_"| : "<$_>";
    }

    my ($out, $err) = _run_cc (
      [ $cc, qw/-E -dD/ ],
      $c_src,
    );
    if ($err) {
      warn "Execution of `$err->{cmd}` failed ($err->{exit})\n";
      $errors->{$type} = $err;
      next;
    }

    my @consts = map { $_ =~ $qr ? $1 : () } split /\n+/, $out
      or next;

    $my_abi->{$type}{$_} = undef for @consts;

    $c_src .= "int main (void)\n{\n";
    $c_src .= "'$_' => \n'__VALUE__START__'\n$_\n'__VALUE__END__'\n,\n" for @consts;
    $c_src .= "}\n";

    ($out, $err) = _run_cc (
      [ $cc, '-E' ],
      $c_src,
    );
    if ($err) {
      warn "Execution of `$err->{cmd}` failed ($err->{exit})\n";
      $errors->{$type} = $err;
      next;
    }

    $out =~ s/ \n'__VALUE__START__'\n (.+?) \n'__VALUE__END__'\n /B::perlstring($1)/egxms;
    if ( my ($resolved) = $out =~ /^int \s main \s \(void\) \n ( \{ .+? \} ) $/xms ) {
      $my_abi->{$type} = eval $resolved or die "Unable to eval preprocessed pseudo-source:\n$@";
    }

  }

  # shit hit the fan - die if a smoker, otherwise just warn (the error struct is rather small)
  if ($errors) {
    unshift @INC, 'lib';
    if (
      eval { require OS::ABI::Constants }
        and
      my $enc = OS::ABI::Constants::__encode_struct ($errors)
    ) {
      $errors =
"
=== BEGIN ABI Survey Errors
$enc
=== END ABI Survey Errors
"
      ;
    }
    else {
      my $oac_encode_error = $@ || 'UNKNOWN';
      chomp $oac_encode_error;
      require Data::Dumper;
      $errors =
          "Error (encoding failed due to $oac_encode_error)\n\n"
        . Data::Dumper->new([ $errors ])->Indent(1)->Terse(1)->Quotekeys(0)->Sortkeys(1)->Pad('  ')->Dump
      ;
    }

    die $errors if $IS_SMOKER;

    warn $errors;
  }

  $my_abi;
}

# use this in lieu of IPC::Run to avoid configure_requires hell, and to not shove
# the dep down everyone's throats (only Win32 can't do open3 IO reliably)
sub _run_cc {
  my ($cmd, $stdin) = @_;

  my ($infh, $infn);
  if ($^O eq 'MSWin32') {
    require File::Temp;
    $infn = File::Temp->new( UNLINK => 1, SUFFIX => '.c' );
    $infn->print ($stdin);
    $infn->close;
    push @$cmd, "$infn";
  }
  else {
    $infh = IO::Handle->new;
    push @$cmd, '-';
  }

  my $outfh = IO::Handle->new;

  my $pid = IPC::Open3::open3 (
    $infh,
    $outfh,
    undef,  # splitting out/err will hang on win32
    @$cmd,
  ) or die "open3 of '@$cmd' failed: $!\n";

  if ($infh) {
    $infh->print($stdin);
    $infh->close;
  }

  my $out = do { local $/; $outfh->getline };

  wait;

  if (my $ex = $?) {
    return (undef, {
      cmd => ( join ' ', @$cmd ),
      exit => $ex >> 8,
      signal => $ex & 127,
      coredump => ($ex & 128) ? 'Y' : 'N',
      stdin => $stdin,
      stdouterr => $out,
    });
  }
  else {
    return $out;
  }
}
