# Perlbug bug record handler
# (C) 1999 Richard Foley RFI perlbug@rfi.net
# $Id: Template.pm,v 1.8 2001/10/19 12:40:21 richardf Exp $
#

=head1 NAME

Perlbug::Object::Template - Template class

=cut

package Perlbug::Object::Template;
use strict;
use Text::Wrap;
use vars qw($VERSION @ISA);
$VERSION = do { my @r = (q$Revision: 1.8 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r }; 
$|=1;

use Data::Dumper;
use HTML::Entities;
use Perlbug::Object;
@ISA = qw(Perlbug::Object); 
my $o_Perlbug_Base = undef;


=head1 DESCRIPTION

Applies a template to the data generated by L<Perlbug::Format>.

Each B<user> may apply a B<template> to each B<object> | B<type>. 

Defaults for a particular type of object (eg; mail, flag, ...) may be 
applied by enabling the type column instead of the object one.

For inherited methods, see L<Perlbug::Object>

=cut

=head1 SYNOPSIS

	use Perlbug::Object::Template;

	my $o_template = Perlbug::Object:;Template->new();

	print $o_template->read('7')->format('h');

	print $o_template->object('bug')->read('19870502.007')->template('display', 'h');

=cut

=head1 ARGUMENTS

=over 4

=item format

See L<Perlbug::Format>

=back

=cut


=head1 PLACEHOLDERS 

Placeholders in templates look like this: B<<{datacol}>> and B<<{rel_ids}>> etc.

Example using bug object:

  Column names:

	bugid:   <{bugid}>
	created: <{created}>
	author:  <{sourceaddr}>
	subject: <{subject}>

  Relationships:

	message count(<{message_count}>)

	messageids: <{message_ids}>
	patch ids:  <{patch_ids}>
	admins:     <{user_names}>
	CC list:    <{address_names}>
	status:     <{status_names}>

=back

=cut


=head1 METHODS

=over 4

=item new

Create new Template object:

	my $o_template = Perlbug::Object::Template->new();

=cut

sub new {
    my $proto = shift;
    my $class = ref($proto) || $proto; 
	$o_Perlbug_Base = (ref($_[0])) ? shift : Perlbug::Base->new();

	my $self = Perlbug::Object->new( $o_Perlbug_Base,
		'name'		=> 'Template',
		'from'		=> [qw(user)],
		'to'		=> [qw()],
	);

	bless($self, $class);
}


=item _template 

Return B<catchall> object data laid out against format(B<a>).

Long lines will be wrapped - if you want a better format, define a template :-)

	my $str = $o_tmp->_template($h_data, $h_rels);

	my $str = $o_tmp->_template($h_data, $h_rels, [$fmt, [$i_withhdr]]]);

=cut

sub _template {
	my $self   = shift;
	my $h_data = shift;
	my $h_rels = shift;
	my $fmt    = shift || $self->base->current('format');
	my $withhdr= shift || 0;
	my $str    = '';

	if (ref($h_data) ne 'HASH' or ref($h_rels) ne 'HASH') {
		$self->error("non-valid required args: data_href($h_data) and relations_href($h_rels)!");		
	} else {
		# my $h_dat = $self->format_fields({%{$h_data}, %{$h_rels}}, $fmt);
		# my $br   = ($fmt =~ /[hHIL]/) ? "<br>\n" : "\n";

		my $br   = "\n";
		my ($dmax) = reverse sort {$a <=> $b} map { length($_) } keys %{$h_data};
		my ($rmax) = reverse sort {$a <=> $b} map { length($_) } keys %{$h_rels};

		$str .= "DATA:$br";
		DATA:
		foreach my $xdata (keys %{$h_data}) {
			$^W = 0;
			my $data = ' '.sprintf('%-'.$dmax.'s', $xdata).' = '.$$h_data{$xdata}.$br;
			$^W = 1;
			$str .= wrap('', '', $data);
		}

		$str .= "RELATIONS:$br";
		RELS:
		foreach my $rel (keys %{$h_rels}) {
			$^W = 0;
			my $data = '  '.sprintf('%-'.$rmax.'s', $rel).' ids:   '.join("', '", @{$$h_rels{$rel}{'ids'}}).$br 
				if @{$$h_rels{$rel}{'ids'}} >= 1;
			$data = '  '.sprintf('%-'.$rmax.'s', $rel).' names: '.join("', '", @{$$h_rels{$rel}{'names'}}).$br 
				if @{$$h_rels{$rel}{'names'}} >= 1;
			$^W = 1;
			$str .= wrap('', '', $data);
		}

		# if ($self->base->current('context') eq 'http' && $fmt !~ /[hHIL]/) {
		if ($self->base->current('context') eq 'http') {
			$str = encode_entities($str);
			$str = '<pre>'.$str.'</pre>';
		}

		$self->debug(3, "str($str)") if $Perlbug::DEBUG;
	}

	return $str;
}



=item template 

Return object in template layout according to format(B<a>).

See also L<_template()>

	my $str = $o_tmp->template($h_data, $h_rels);

	my $str = $o_tmp->_template($h_data, $h_rels, [$fmt, [$i_withhdr]]);

=cut

sub template {
	my $self   = shift;
	my $h_data = shift;
	my $h_rels = shift;
	my $fmt    = shift || $self->base->current('format');
	my $withhdr= shift || 0;
	my $str    = '';

	if (ref($h_data) ne 'HASH' or ref($h_rels) ne 'HASH') {
		$self->error("non-valid required args: data_href($h_data) and relations_href($h_rels)!");		
	} else {
		my $h_dat = $self->format_fields({%{$h_data}, %{$h_rels}}, $fmt);
		my $br    = ($fmt =~ /[hHIL]/) ? "<br>\n" : "\n";
		my $hdr   = $self->data('header'); 
		$str      = $self->data('body'); 

		# $^W = 0;
		DATA:
		foreach my $data (keys %{$h_dat}) {
			if (ref($data) eq 'HASH') {
				redo DATA;
			} elsif (ref($$h_dat{$data}) eq 'ARRAY') {
				my $replace = join(', ', @{$$h_dat{$data}});
				$str =~ s/\<\{$data\}\>/$replace/gmsi;
				$hdr =~ s/\<\{$data\}\>/$replace/gmsi if $withhdr;
			} else {
				$str =~ s/\<\{$data\}\>/$$h_dat{$data}/gmsi;
				$hdr =~ s/\<\{$data\}\>/$$h_dat{$data}/gmsi if $withhdr;
			}
		}
		# $^W = 1;

		# Wrap
		if ($self->data('wrap') =~ /^[1-9]/) {
			$str = wrap('', '', $str);
			if ($withhdr) {
				$hdr = wrap('', '', $hdr);
				$str = $hdr.$br.$str;
			}
		}

		# if ($self->base->current('context') eq 'http' && $fmt !~ /[hHIL]/) {
		#	$str = encode_entities($str);
		#	$str = '<pre>'.$str.'</pre>';
		#}
	}
	$self->debug(3, "str($str)") if $Perlbug::DEBUG;

	return $str;
}


=pod

=back

=head1 AUTHOR

Richard Foley perlbug@rfi.net 2001

=cut


1;

