package # hide from PAUSE
App::YTDL::Merge;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( search_and_merge );

use File::Copy            qw( move );
use File::Spec::Functions qw( catfile catdir );
use Time::Local           qw( timelocal );

use File::Touch         qw();
use File::Which         qw( which );
use IPC::System::Simple qw( capture system );
use Term::Choose        qw( choose );
use Try::Tiny           qw( try catch );

use if $^O eq 'MSWin32', 'Win32';

use App::YTDL::Helper qw( get_filename timestamp_to_upload_date encode_fs );



sub search_and_merge {
    my ( $opt, $info ) = @_;
    if ( ! merge_requirements_ok( $opt ) ) {
        return;
    }
    for my $ex ( keys %$info ) {
        for my $video_id ( keys %{$info->{$ex}} ) {
            if ( keys %{$info->{$ex}{$video_id}{file_name}} == 2 ) {
                try {
                    my ( $video, $audio );
                    for my $fmt ( keys %{$info->{$ex}{$video_id}{file_name}} ) {
                        my $file = $info->{$ex}{$video_id}{file_name}{$fmt};
                        my $stream_info = _get_stream_info( $opt, $file );
                        if ( $stream_info =~ /^\s*codec_type=audio\s*$/m && $stream_info !~ /^\s*codec_type=video\s*$/m ) {
                            $audio = $file;
                        }
                        elsif ( $stream_info =~ /^\s*codec_type=video\s*$/m ) {
                            $video = $file;
                        }
                    }
                    if ( ! $audio ) {
                        die "No file with audio-only stream!";
                    }
                    if ( ! $video ) {
                        die "No file with video stream!";
                    }
                    my $video_dir = $info->{$ex}{$video_id}{video_dir};
                    my $file_name = get_filename( $opt, $info->{$ex}{$video_id}{title}, $opt->{merge_format} );
                    my $out = catfile $video_dir, $file_name;
                    if ( -f $out && ! $opt->{merge_overwrite} ) {
                        my $overwrite_ok = choose(
                            [ 'NO merge', 'Merge and overwrite' ],
                            { prompt => $out . ' already exists:', index => 1 }
                        );
                        return if ! $overwrite_ok;
                    }
                    _merge_video_with_audio( $opt, $video, $audio, $out );
                    say "Merged: $out";
                    if ( $opt->{merged_in_files} == 1 ) {
                        my $dir_orig_files = catdir $video_dir, $opt->{dir_stream_files};
                        if ( ! -d $dir_orig_files ) {
                            mkdir $dir_orig_files or die $!;
                        }
                        move $video, $dir_orig_files or die $!;
                        move $audio, $dir_orig_files or die $!;
                    }
                    elsif ( $opt->{merged_in_files} == 2 ) {
                        unlink $video or die $!;
                        unlink $audio or die $!;
                    }
                    if ( $opt->{upload_date_is_file_date} ) {
                        timestamp_to_upload_date( $info, $ex, $video_id, $out );
                    }
                }
                catch {
                    say "$video_id - Merge: $_";
                    push @{$opt->{error_merge}}, $video_id;
                };
            }
        }
    }
}


sub _merge_video_with_audio {
    my ( $opt, $video, $audio, $out ) = @_;
    my ( $fs_video, $fs_audio, $fs_out );
    if ( $^O eq 'MSWin32' ) {
        $fs_video = Win32::GetShortPathName( encode_fs( $video ) );
        $fs_audio = Win32::GetShortPathName( encode_fs( $audio ) );
        open my $fh, '>', encode_fs( $out ) or die $!;                  # overwrites $out
        close $fh;
        $fs_out = Win32::GetShortPathName( encode_fs( $out ) );
    }
    else {
        $fs_video = encode_fs( $video );
        $fs_audio = encode_fs( $audio );
        $fs_out   = encode_fs( $out );
    }
    my @cmd = ( $opt->{ffmpeg} );
    push @cmd, '-loglevel', $opt->{merge_loglevel};
    push @cmd, '-y';
    push @cmd, '-i', $fs_video;
    push @cmd, '-i', $fs_audio;
    push @cmd, '-codec', 'copy';
    push @cmd, '-map', '0:v:0';
    push @cmd, '-map', '1:a:0';
    push @cmd, $fs_out;
    system( @cmd );
}


sub _get_stream_info {
    my ( $opt, $file ) = @_;
    my $fs_file;
    if ( $^O eq 'MSWin32' ) {
        $fs_file = Win32::GetShortPathName( encode_fs( $file ) );
    }
    else {
        $fs_file   = encode_fs( $file );
    }
    my @cmd = ( $opt->{ffprobe} );
    push @cmd, '-loglevel', 'warning';
    push @cmd, '-show_streams';
    push @cmd, $fs_file;
    my $stream_info = capture( @cmd );
    return $stream_info;
}


sub merge_requirements_ok {
    my ( $opt ) = @_;
    try {
        $opt->{ffmpeg} = $^O eq 'MSWin32'
            ? Win32::GetShortPathName( which( 'ffmpeg'  ) )
            :                          which( 'ffmpeg' );
        my $capture = capture( $opt->{ffmpeg}, '-version' );
    }
    catch {
        say "Could not find 'ffmpeg' - No merge.";
        return;
    };
    try {
        $opt->{ffprobe} = $^O eq 'MSWin32'
            ? Win32::GetShortPathName( which( 'ffprobe' ) )
            :                          which( 'ffprobe' );
        my $capture = capture( $opt->{ffprobe}, '-version' );
    }
    catch {
        say "Could not find 'ffprobe' - No merge.";
        return;
    };
    return 1;
}


1;


__END__
