#!perl

use strict;
use warnings;
$|=1;
use Test::More tests => 1;
use DBI;
#use DBD::SQLite;
use File::Spec;
use File::Path;
use File::Basename;

# rebuild cpanstats db

my $f = File::Spec->catfile('t','_DBDIR','test.db');
unlink $f if -f $f;
mkpath( dirname($f) );

my $dbh = DBI->connect("dbi:SQLite:dbname=$f", '', '', {AutoCommit=>1});
$dbh->do(q{
  CREATE TABLE cpanstats (
                          id            INTEGER PRIMARY KEY,
                          state         TEXT,
                          postdate      TEXT,
                          tester        TEXT,
                          dist          TEXT,
                          version       TEXT,
                          platform      TEXT,
                          perl          TEXT,
                          osname        TEXT,
                          osvers        TEXT,
                          fulldate      TEXT
  )
});

while(<DATA>){
  chomp;
  $dbh->do('INSERT INTO cpanstats ( id, state, postdate, tester, dist, version, platform, perl, osname, osvers, fulldate ) VALUES ( ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ? )', {}, split(/\|/,$_) );
}

$dbh->do(q{ CREATE INDEX distverstate ON cpanstats (dist, version, state) });
$dbh->do(q{ CREATE INDEX ixdate ON cpanstats (postdate) });
$dbh->do(q{ CREATE INDEX ixperl ON cpanstats (perl) });
$dbh->do(q{ CREATE INDEX ixplat ON cpanstats (platform) });

my ($ct) = $dbh->selectrow_array('select count(*) from cpanstats');

$dbh->disconnect;

is($ct, 13796, "row count for cpanstats");

#select * from cpanstats where state='cpan' and dist in ('AEAE', 'AI-NeuralNet-BackProp', 'AI-NeuralNet-Mesh', 'AI-NeuralNet-SOM', 'AOL-TOC', 'Abstract-Meta-Class', 'Acme', 'Acme-Anything', 'Acme-BOPE', 'Acme-Brainfuck', 'Acme-Buffy', 'Acme-CPANAuthors-Canadian', 'Acme-CPANAuthors-CodeRepos', 'Acme-CPANAuthors-French', 'Acme-CPANAuthors-Japanese');
# sqlite> select * from cpanstats where postdate=200901 order by dist limit 20;
# id|state|postdate|tester|dist|version|platform|perl|osname|osvers|date
__DATA__
4772144|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.8.8|solaris|2.10|200907251628
4769404|pass|200907|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250820
4769374|pass|200907|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250813
4769310|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.10.1|linux|2.6.28-13-generic|200907250304
4769245|pass|200907|bingos@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250745
4769240|pass|200907|bingos@cpan.org|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250744
4769230|pass|200907|bingos@cpan.org|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250743
4769229|pass|200907|bingos@cpan.org|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250742
4768975|pass|200907|bingos@cpan.org|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250624
4768963|pass|200907|bingos@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250622
4768953|pass|200907|bingos@cpan.org|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250621
4768945|pass|200907|bingos@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250620
4768725|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200907250641
4768479|pass|200907|bingos@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250453
4768477|pass|200907|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250453
4768471|pass|200907|bingos@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250452
4768469|pass|200907|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250452
4768447|pass|200907|bingos@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250450
4768326|pass|200907|bingos@cpan.org|Net-Random|2.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250423
4768209|pass|200907|bingos@cpan.org|NestedMap|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250354
4767716|pass|200907|bingos@cpan.org|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250158
4767670|pass|200907|bingos@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250152
4767658|pass|200907|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250150
4767455|pass|200907|bingos@cpan.org|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250110
4767321|pass|200907|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250033
4767308|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200907250127
4767284|pass|200907|bingos@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250020
4767259|pass|200907|bingos@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250016
4767253|pass|200907|bingos@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250014
4767247|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.8|solaris|2.10|200907250015
4767242|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.8.8|solaris|2.10|200907250014
4767222|pass|200907|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250007
4767216|pass|200907|bingos@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907250006
4767090|pass|200907|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907242342
4767080|pass|200907|bingos@cpan.org|Class-CanBeA|1.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907242340
4766918|pass|200907|cpan.dcollins.5100rc1@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241806
4766906|pass|200907|cpan.dcollins.5100rc1@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907241806
4766195|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907241747
4766048|fail|200907|bingos@cpan.org|Bryar|3.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907242241
4765508|pass|200907|cpan.dcollins.589@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907241727
4765368|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200907242323
4765197|pass|200907|cpan.dcollins.smoker8@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907241719
4765186|pass|200907|cpan.dcollins.smoker8@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907241718
4764585|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200907242301
4764563|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200907242300
4764410|pass|200907|cpan.dcollins.smoker6@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241657
4764207|pass|200907|cpan.dcollins.smoker9@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241653
4764204|pass|200907|cpan.dcollins.smoker9@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241653
4763679|pass|200907|cpan.dcollins.smoker9@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241638
4763409|pass|200907|cpan.dcollins.589@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241630
4762699|pass|200907|cpan.dcollins.smoker1@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907241611
4762057|pass|200907|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907242059
4762027|pass|200907|bingos@cpan.org|Acme-Pony|1.1.2|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907242058
4761962|pass|200907|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907242057
4761907|pass|200907|cpan.dcollins.smoker4@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907241558
4761641|pass|200907|cpan.dcollins.5100rc1@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907241553
4761089|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200907241927
4759796|pass|200907|martin.evans@easysoft.com|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.24-24-generic|200907241418
4758816|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200907241207
4757596|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200907240637
4753687|pass|200907|bingos@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907232225
4753484|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200907232226
4752187|pass|200907|martin.evans@easysoft.com|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.24-24-generic|200907231741
4751010|pass|200907|bingos@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907231449
4750433|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200907231350
4750418|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.6.2|netbsd|3.1|200907231350
4749367|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200907231139
4749357|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200907231137
4749275|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Class-CanBeA|1.2|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200907231117
4747480|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907230257
4747402|pass|200907|cpan.dcollins.5100rc1@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907230255
4746636|pass|200907|cpan.dcollins.smoker8@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907230241
4744386|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.7|solaris|2.10|200907222028
4743087|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i86pc-solaris|5.8.7|solaris|2.10|200907221650
4743085|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.8.7|solaris|2.10|200907221650
4742565|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907221611
4742432|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907221553
4742430|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907221553
4741849|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.8.7|solaris|2.10|200907221406
4741830|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200907221429
4741479|pass|200907|cpan.dcollins.smoker2@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907220915
4741469|pass|200907|cpan.dcollins.smoker6@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907220915
4741246|pass|200907|cpan.dcollins.smoker3@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907220905
4741094|pass|200907|cpan.dcollins.smoker1@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907220858
4740997|pass|200907|cpan.dcollins.5100rc1@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907220854
4740953|pass|200907|lists.perl.cpan-testers@csjewell.fastmail.us ("Curtis Jewell")|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.1|200907211527
4740653|pass|200907|cpan.dcollins.smoker1@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907220840
4740469|pass|200907|cpan.dcollins.smoker8@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907220831
4739306|pass|200907|lists.perl.cpan-testers@csjewell.fastmail.us ("Curtis Jewell")|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200907211532
4738758|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200907221022
4738756|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200907221021
4737751|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200907220637
4737561|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907220007
4736767|pass|200907|cpan.dcollins.smoker6@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907220353
4736492|pass|200907|cpan.dcollins.smoker6@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907220348
4736259|pass|200907|cpan.dcollins.589@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907220344
4736223|pass|200907|cpan.dcollins.smoker3@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907220343
4736094|pass|200907|cpan.dcollins.smoker6@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907220341
4735049|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200907220259
4735046|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200907220258
4734326|pass|200907|cpan.dcollins.smoker4@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211926
4733867|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211917
4733721|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi|5.10.1|linux|2.6.28-13-generic|200907211915
4733717|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.10.1|linux|2.6.28-13-generic|200907211915
4733281|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211904
4732595|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211848
4732472|pass|200907|cpan.dcollins.smoker5@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211844
4732353|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211838
4732316|pass|200907|cpan.dcollins.smoker9@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211837
4732155|pass|200907|cpan.dcollins.smoker5@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211832
4732050|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211826
4731685|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211812
4731603|fail|200907|cpan.dcollins.smoker9@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211809
4731484|pass|200907|cpan.dcollins.589@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211805
4731476|pass|200907|cpan.dcollins.smoker4@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211805
4731335|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200907220001
4731095|pass|200907|cpan.dcollins.smoker8@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211753
4731080|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211752
4730847|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211746
4730799|pass|200907|cpan.dcollins.5100rc1@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211745
4730584|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200907212339
4730251|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211730
4730085|fail|200907|cpan.dcollins.smoker8@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211726
4729942|pass|200907|cpan.dcollins.smoker8@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211721
4729857|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211719
4729813|pass|200907|cpan.dcollins.smoker8@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211711
4729798|pass|200907|cpan.dcollins.smoker3@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907211717
4729664|pass|200907|cpan.dcollins.589@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211713
4729264|fail|200907|cpan.dcollins.smoker8@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211703
4728996|pass|200907|cpan.dcollins.589@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907211656
4728842|pass|200907|cpan.dcollins.smoker8@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907211652
4727777|pass|200907|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907211740
4727205|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200907211641
4726820|pass|200907|martin.evans@easysoft.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.24-24-generic|200907211436
4724466|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907210115
4724408|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907210106
4724353|unknown|200907|cpan@greerga.m-l.org ("George Greer")|Acme-Pony|1.1.2|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907210058
4724307|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907210041
4724306|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907210041
4724039|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200907210542
4723047|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.6|solaris|2.10|200907210047
4723011|pass|200907|jeff@zeroclue.com|Devel-CheckOS|1.61|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907210018
4723007|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.6|solaris|2.10|200907210047
4722831|fail|200907|cpan@greerga.m-l.org ("George Greer")|DBIx-Class-SingletonRows|0.11|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907201938
4722586|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200907202212
4722383|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200907210015
4722318|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200907210005
4722317|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200907210005
4721837|pass|200907|cpan.dcollins.smoker4@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907201744
4721277|pass|200907|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907201731
4720131|pass|200907|cpan.dcollins.smoker9@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907201703
4719752|pass|200907|cpan.dcollins.smoker2@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907201653
4718957|pass|200907|cpan.dcollins.smoker4@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907201634
4718756|pass|200907|cpan.dcollins.smoker5@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907201629
4717610|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907201558
4717571|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907201556
4716993|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907201540
4716432|pass|200907|cpan.dcollins.smoker1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907201527
4716136|pass|200907|justin.d.hunter@gmail.com|XML-Tiny-DOM|1.1|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200907201206
4715948|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.8.6|solaris|2.10|200907201914
4715947|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i86pc-solaris|5.8.6|solaris|2.10|200907201914
4715601|pass|200907|smoretti@unil.ch|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.8|linux|2.6.18|200907201502
4715231|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200907201614
4715230|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200907201614
4715034|pass|200907|smoretti@unil.ch|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.8|linux|2.6.22|200907201456
4714791|pass|200907|smoretti@unil.ch|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.8|linux|2.6.22|200907201511
4714785|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.8.6|solaris|2.10|200907201628
4714723|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200907201744
4714625|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200907201734
4714607|unknown|200907|cpan@greerga.m-l.org ("George Greer")|Acme-Licence|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907201132
4714326|pass|200907|cpan@greerga.m-l.org ("George Greer")|Sort-MultipleFields|1.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907201106
4714157|pass|200907|smoretti@unil.ch|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.8|linux|2.6.18|200907201503
4714122|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200907201409
4713510|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200907201528
4713053|pass|200907|jeff@zeroclue.com|XML-Tiny-DOM|1.0|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907200519
4713051|pass|200907|jeff@zeroclue.com|XML-Tiny|2.02|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907200519
4713036|pass|200907|jeff@zeroclue.com|Palm-ProjectGutenberg|1.0|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907200517
4712297|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200907201213
4711887|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200907201118
4711638|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200907201049
4710934|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200907200853
4710706|pass|200907|cpan@greerga.m-l.org ("George Greer")|NestedMap|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907200221
4710422|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Class-CanBeA|1.2|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200907200732
4710308|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200907200710
4710203|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200907200652
4710017|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200907200624
4709942|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200907200614
4709804|pass|200907|jeff@zeroclue.com|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907200352
4709294|pass|200907|cpan.dcollins.smoker6@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907200336
4709124|pass|200907|cpan.dcollins.589@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907200333
4708923|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907200329
4708867|pass|200907|cpan.dcollins.smoker5@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907200328
4708755|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907200327
4708730|pass|200907|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907200326
4708494|pass|200907|cpan.dcollins.smoker9@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907200322
4708476|pass|200907|cpan.dcollins.smoker4@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907200322
4708183|pass|200907|cpan.dcollins.smoker5@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907200317
4708033|pass|200907|cpan.dcollins.smoker2@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907200315
4707625|pass|200907|cpan.dcollins.smoker4@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907200308
4707453|pass|200907|jeff@zeroclue.com|Palm-Treo680MessagesDB|1.01|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907200247
4707437|pass|200907|jeff@zeroclue.com|File-Overwrite|1.1|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907200243
4707357|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200907200430
4707303|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907200223
4707166|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200907200404
4706513|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200907200225
4706314|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200907200156
4705553|pass|200907|cpan@greerga.m-l.org ("George Greer")|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907191751
4705550|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907191750
4705526|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.5|solaris|2.10|200907192228
4705297|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Class-DBI-ClassGenerator|1.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907200716
4705243|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|File-Find-Rule-Permissions|2.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907200704
4705002|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.1|linux|2.6.28-13-generic|200907191638
4705001|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.1|linux|2.6.28-13-generic|200907191638
4704419|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200907192050
4703769|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|WWW-Facebook-Go-SGF|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907200249
4703708|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200907191837
4702871|pass|200907|cpan.dcollins.smoker4@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907191015
4702819|pass|200907|cpan.dcollins.smoker6@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907191018
4702499|fail|200907|cpan.dcollins.smoker4@gmail.com|Bryar|3.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907191012
4701711|pass|200907|cpan.dcollins.smoker1@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907191001
4701675|pass|200907|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907191000
4701657|pass|200907|cpan.dcollins.smoker2@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907191000
4701504|pass|200907|cpan.dcollins.smoker6@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190958
4701471|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200907191557
4701203|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190953
4701167|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907190952
4700870|pass|200907|cpan.dcollins.smoker6@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190948
4700454|pass|200907|cpan.dcollins.589@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190941
4700385|pass|200907|cpan.dcollins.smoker1@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190940
4700076|pass|200907|cpan.dcollins.smoker4@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190936
4700004|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.10.0|linux|2.6.18-92.el5|200907191535
4699879|pass|200907|cpan.dcollins.smoker8@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190933
4699875|pass|200907|cpan.dcollins.589@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190933
4699810|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907190932
4699628|pass|200907|cpan.dcollins.smoker6@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907190923
4699232|pass|200907|cpan.dcollins.smoker3@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907190924
4699137|pass|200907|cpan.dcollins.smoker3@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190922
4698863|pass|200907|cpan.dcollins.smoker1@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190918
4698533|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200907191513
4698512|pass|200907|cpan.dcollins.smoker4@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190913
4698043|pass|200907|cpan.dcollins.589@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907190906
4697654|pass|200907|cpan.dcollins.smoker5@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190859
4697558|pass|200907|cpan.dcollins.smoker5@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907190857
4697442|pass|200907|cpan.dcollins.smoker6@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907190855
4697271|pass|200907|cpan.dcollins.smoker1@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907190852
4697025|pass|200907|jeff@zeroclue.com|Number-Phone|1.7002|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907190531
4696871|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Tie-STDOUT|1.0401|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907192209
4696598|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907190729
4694572|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200907190755
4694428|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200907190726
4694346|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200907190714
4694248|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200907190653
4694233|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200907190650
4694031|pass|200907|cpan.dcollins.589@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190041
4693973|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190040
4693385|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190027
4693368|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190027
4693332|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190026
4693163|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200907190622
4692801|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200907190611
4692751|pass|200907|cpan.dcollins.smoker4@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190010
4692617|pass|200907|cpan.dcollins.589@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190006
4692559|pass|200907|cpan.dcollins.smoker2@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190004
4691880|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200907190544
4691591|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200907190536
4691475|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190333
4691463|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190332
4691116|pass|200907|cpan.dcollins.smoker2@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190322
4690343|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190300
4690327|pass|200907|cpan.dcollins.smoker5@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190259
4690209|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190256
4690135|pass|200907|cpan.dcollins.5100rc1@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190254
4689539|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.1|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200907190437
4689354|pass|200907|cpan.dcollins.smoker2@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190232
4689093|fail|200907|cpan.dcollins.smoker4@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190225
4689003|pass|200907|cpan@greerga.m-l.org ("George Greer")|Sub-WrapPackages|1.2|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907190222
4688258|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907190202
4687806|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907190148
4686216|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200907182221
4685289|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i86pc-solaris|5.8.5|solaris|2.10|200907182056
4685287|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.8.5|solaris|2.10|200907182056
4684817|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200907181904
4684816|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200907181904
4684801|pass|200907|cpan@greerga.m-l.org ("George Greer")|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907181501
4684198|pass|200907|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200907181909
4683794|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200907181617
4683638|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.8.5|solaris|2.10|200907181645
4681097|pass|200907|jeff@zeroclue.com|Devel-CheckLib|0.6|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907180052
4680459|pass|200907|cpan@greerga.m-l.org ("George Greer")|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907180202
4680012|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907180126
4680011|pass|200907|cpan.dcollins.smoker5@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907180126
4680003|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180126
4679661|pass|200907|cpan.dcollins.smoker6@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907180114
4679505|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180117
4679221|pass|200907|cpan.dcollins.smoker6@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907180112
4679104|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180111
4678463|pass|200907|cpan.dcollins.smoker8@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907180100
4678300|pass|200907|cpan.dcollins.smoker2@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180057
4678294|pass|200907|cpan.dcollins.smoker2@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180057
4678039|pass|200907|cpan.dcollins.smoker7@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180052
4677839|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180049
4677826|pass|200907|cpan.dcollins.smoker6@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180049
4677709|pass|200907|cpan.dcollins.smoker9@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180047
4677657|pass|200907|cpan.dcollins.5100@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907180046
4677604|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907180045
4677491|pass|200907|cpan.dcollins.smoker7@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907180043
4677109|fail|200907|cpan.dcollins.smoker7@gmail.com|Bryar|3.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180036
4677024|pass|200907|cpan.dcollins.5100@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180035
4676767|pass|200907|cpan.dcollins.smoker2@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180031
4676443|pass|200907|cpan.dcollins.smoker6@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180025
4676432|pass|200907|cpan.dcollins.smoker4@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180025
4676337|pass|200907|cpan.dcollins.smoker8@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180023
4676270|pass|200907|cpan.dcollins.smoker5@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180022
4675805|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180014
4675636|pass|200907|cpan.dcollins.smoker2@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180011
4675431|pass|200907|cpan.dcollins.smoker3@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180008
4675124|pass|200907|cpan.dcollins.5100@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180003
4674767|pass|200907|cpan.dcollins.smoker7@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180357
4674539|pass|200907|cpan.dcollins.smoker3@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180353
4674434|pass|200907|cpan.dcollins.smoker7@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180351
4674411|pass|200907|cpan.dcollins.smoker2@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907180350
4674327|fail|200907|cpan.dcollins.smoker8@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907180349
4672619|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd|5.8.9|openbsd|4.5|200907180031
4671999|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.4|solaris|2.10|200907172251
4671927|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200907172350
4671648|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907171716
4671283|pass|200907|cpan@greerga.m-l.org ("George Greer")|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907171633
4670334|pass|200907|jeff@zeroclue.com|Data-Compare|1.2101|darwin-2level|5.10.0|darwin|9.2.0|200907171125
4669741|pass|200907|cpan@greerga.m-l.org ("George Greer")|Net-Random|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907171303
4669047|pass|200907|imacat@mail.imacat.idv.tw|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200907172349
4669009|pass|200907|mw487.yahoo.com@gmail.com|XML-Tiny-DOM|1.1|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200907171046
4668869|pass|200907|martin.evans@easysoft.com|XML-Tiny-DOM|1.1|i686-linux|5.10.0|linux|2.6.24-24-generic|200907171632
4668443|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200907171645
4667875|pass|200907|cpan.dcollins.smoker8@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907171011
4667385|pass|200907|cpan.dcollins.smoker6@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907171005
4667384|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907170958
4667249|pass|200907|cpan.dcollins.5100@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907171001
4667124|pass|200907|martin.evans@easysoft.com|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200907171458
4666852|pass|200907|cpan.dcollins.5100@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907170952
4666307|pass|200907|cpan.dcollins.smoker4@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907170939
4666228|pass|200907|cpan.dcollins.smoker6@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907170938
4665870|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907170929
4665754|pass|200907|cpan.dcollins.smoker6@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907170926
4665388|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200907171517
4665344|pass|200907|cpan.dcollins.smoker9@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907170917
4664802|unknown|200907|cpan.dcollins.smoker4@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170904
4664278|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200907171454
4664180|pass|200907|cpan.dcollins.smoker4@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907170853
4663720|pass|200907|cpan.dcollins.smoker9@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907170844
4663621|pass|200907|cpan.dcollins.smoker3@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907170843
4663313|pass|200907|cpan.dcollins.5100@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907170837
4662804|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|alpha-netbsd|5.10.0|netbsd|4.0.1|200907171229
4662192|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200907170718
4661958|pass|200907|cpan.dcollins.smoker9@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907170814
4661564|pass|200907|cpan.dcollins.5100@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907170808
4661533|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|IP35-irix-64int|5.8.8|irix|6.5|200907171407
4661464|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200907171300
4661431|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200907171257
4661415|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200907171255
4661401|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200907171253
4661398|pass|200907|david@cantrell.org.uk|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200907171253
4660329|pass|200907|mw487.yahoo.com@gmail.com|XML-Tiny-DOM|1.1|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200907170430
4660328|pass|200907|mw487.yahoo.com@gmail.com|XML-Tiny|2.02|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200907170430
4658482|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200907170719
4657775|unknown|200907|cpan.dcollins.smoker3@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170314
4657302|pass|200907|cpan.dcollins.smoker1@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170306
4657261|pass|200907|cpan.dcollins.smoker1@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907170305
4657149|pass|200907|cpan.dcollins.smoker5@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907170303
4656980|pass|200907|cpan.dcollins.smoker4@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170300
4656529|pass|200907|cpan.dcollins.smoker1@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907170253
4655918|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907170242
4655837|pass|200907|cpan.dcollins.smoker1@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170241
4655618|unknown|200907|cpan.dcollins.smoker6@gmail.com|XML-Tiny-DOM|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170237
4655268|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907170231
4654844|pass|200907|cpan.dcollins.5100@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907170224
4654658|pass|200907|cpan.dcollins.smoker4@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907170221
4653083|pass|200907|cpan@greerga.m-l.org ("George Greer")|Class-CanBeA|1.2|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907170154
4652873|pass|200907|cpan.dcollins.smoker5@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907170151
4652268|pass|200907|cpan.dcollins.smoker5@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907170140
4651954|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907170132
4651682|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Palm-ProjectGutenberg|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907171043
4650105|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|CPAN-ParseDistribution|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170713
4648795|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170416
4648774|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Class-DBI-ClassGenerator|1.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170414
4648712|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|File-Find-Rule-Permissions|2.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170406
4648543|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Palm-Treo680MessagesDB|1.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170337
4648541|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170337
4648524|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|File-Overwrite|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907170335
4648200|pass|200907|martin.evans@easysoft.com|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.24-24-generic|200907161803
4648068|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|i86pc-solaris|5.8.4|solaris|2.10|200907161740
4648067|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.8.4|solaris|2.10|200907161740
4647484|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|OpenBSD.i386-openbsd|5.8.9|openbsd|4.5|200907161648
4647482|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd|5.8.9|openbsd|4.5|200907161648
4646032|pass|200907|bingos@cpan.org|XML-Tiny-DOM|1.1|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200907161354
4646029|pass|200907|bingos@cpan.org|XML-Tiny|2.02|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200907161354
4645977|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.8.4|solaris|2.10|200907161337
4645687|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd|5.8.9|openbsd|4.5|200907161314
4645305|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny-DOM|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907162128
4645303|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny|2.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907162128
4644963|pass|200907|cpan@greerga.m-l.org ("George Greer")|Sub-WrapPackages|1.2|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907160651
4644442|pass|200907|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny-DOM|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200907161148
4643788|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|CPAN-FindDependencies|2.32|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907161842
4643740|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-SMS|0.03|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907160437
4643181|pass|200907|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny-DOM|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200907160940
4643156|pass|200907|rbo@cpan.org|XML-Tiny-DOM|1.1|i86pc-solaris|5.10.0|solaris|2.11|200907160938
4643151|pass|200907|rbo@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.10.0|solaris|2.11|200907160938
4642889|pass|200907|srezic@cpan.org|XML-Tiny-DOM|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200907160819
4642769|pass|200907|cpan@greerga.m-l.org ("George Greer")|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907160244
4642584|pass|200907|srezic@cpan.org|XML-Tiny-DOM|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200907160819
4642113|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Class-CanBeA|1.2|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200907160701
4641964|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny-DOM|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907161432
4641896|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907160024
4641695|pass|200907|cpan.dcollins.smoker9@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160020
4641432|unknown|200907|cpan.dcollins.5100rc1@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160415
4641365|pass|200907|cpan.dcollins.smoker5@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160013
4641254|unknown|200907|cpan.dcollins.589@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160011
4641105|pass|200907|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160007
4640761|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907160001
4639837|pass|200907|cpan.dcollins.smoker3@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160344
4639780|unknown|200907|cpan.dcollins.smoker9@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160343
4639697|pass|200907|cpan.dcollins.smoker4@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907160341
4639286|pass|200907|cpan.dcollins.smoker5@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160334
4638928|pass|200907|cpan.dcollins.smoker5@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907160327
4638893|pass|200907|cpan.dcollins.5100@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907160326
4638511|pass|200907|cpan.dcollins.smoker8@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160319
4638490|pass|200907|cpan.dcollins.5100rc1@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160319
4638165|pass|200907|cpan.dcollins.smoker5@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907160311
4638138|unknown|200907|cpan.dcollins.smoker5@gmail.com|Net-Random|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160311
4638074|pass|200907|cpan.dcollins.5100@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160309
4637559|unknown|200907|cpan.dcollins.smoker6@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160256
4637370|pass|200907|cpan.dcollins.smoker5@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160252
4637232|pass|200907|cpan.dcollins.smoker3@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160248
4637219|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160248
4636978|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160243
4636816|pass|200907|cpan.dcollins.smoker7@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160239
4636504|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160232
4636498|pass|200907|cpan.dcollins.smoker4@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160231
4636491|pass|200907|cpan.dcollins.smoker9@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160231
4636486|pass|200907|cpan.dcollins.smoker4@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160231
4636463|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160230
4636370|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160229
4636328|pass|200907|cpan.dcollins.smoker1@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160227
4636248|pass|200907|cpan.dcollins.smoker4@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160225
4636206|pass|200907|cpan.dcollins.smoker2@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907160224
4635870|pass|200907|cpan.dcollins.smoker1@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907160216
4634842|pass|200907|bob@randomness.org.uk ("bob walker")|XML-Tiny-DOM|1.1|i86pc-solaris|5.10.0|solaris|2.10|200907160049
4634841|pass|200907|bob@randomness.org.uk ("bob walker")|XML-Tiny|2.02|i86pc-solaris|5.10.0|solaris|2.10|200907160049
4634824|pass|200907|d@0x64.nl|XML-Tiny|2.02|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.18-xen-3.1-1-amd64|200907160148
4634429|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200907160049
4634425|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200907160049
4634422|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|x86_64-linux-thread-multi-ld|5.8.7|linux|2.6.26-1-amd64|200907160049
4634419|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200907160049
4634416|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|x86_64-linux-thread-multi-ld|5.8.7|linux|2.6.26-1-amd64|200907160049
4634415|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|x86_64-linux|5.8.9|linux|2.6.26-1-amd64|200907160049
4634337|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200907160050
4634331|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200907160050
4634327|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|x86_64-linux|5.8.9|linux|2.6.26-1-amd64|200907160049
4634323|pass|200907|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.1|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200907160049
4634312|cpan|200907|DCANTRELL|XML-Tiny-DOM|1.1||0|||200907160048
4633617|pass|200907|jeff@zeroclue.com|Data-Compare|1.2101|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200907151433
4632093|pass|200907|cpan.dcollins.smoker1@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907151505
4631460|pass|200907|cpan.dcollins.smoker2@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907151449
4630984|pass|200907|cpan.dcollins.smoker9@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907151438
4630823|pass|200907|cpan.dcollins.smoker5@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907151434
4629707|pass|200907|cpan.dcollins.smoker9@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907151407
4628889|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907151348
4628794|pass|200907|cpan.dcollins.smoker9@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907151346
4628688|pass|200907|cpan.dcollins.smoker9@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907151343
4628665|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907151342
4628526|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907151339
4628368|unknown|200907|cpan.dcollins.5100rc1@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907151335
4627715|fail|200907|cpan.dcollins.5100@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907151317
4625969|fail|200907|cpan.dcollins.smoker9@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907151223
4625837|pass|200907|cpan.dcollins.5100@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907151218
4625368|pass|200907|cpan.dcollins.smoker8@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907151203
4624872|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris|5.10.0|solaris|2.10|200907151641
4624249|pass|200907|cpan@greerga.m-l.org ("George Greer")|Net-Random|2.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907151055
4621671|pass|200907|rlane10@wi.rr.com|Tie-STDOUT|1.0401|darwin-2level-ld|5.10.0|darwin|8.11.0|200907150313
4621302|pass|200907|rlane10@wi.rr.com|CPAN-FindDependencies|2.32|darwin-2level-ld|5.10.0|darwin|8.11.0|200907150419
4619590|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907150211
4619512|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Number-Phone|1.7002|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907151559
4619260|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd|5.10.0|openbsd|4.5|200907150624
4618835|pass|200907|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907150014
4618423|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907150327
4618251|unknown|200907|cpan.dcollins.smoker8@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907150320
4617769|pass|200907|cpan.dcollins.5100rc1@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150301
4617761|pass|200907|cpan.dcollins.smoker2@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907150301
4617508|pass|200907|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907150252
4617048|pass|200907|cpan.dcollins.smoker2@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907150237
4616912|pass|200907|cpan.dcollins.smoker5@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150234
4616803|pass|200907|cpan.dcollins.5100@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907150231
4615833|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|CPAN-FindDependencies|2.32|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907151208
4615657|pass|200907|cpan.dcollins.smoker8@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150203
4614822|pass|200907|cpan.dcollins.5100rc1@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907150147
4614510|unknown|200907|cpan.dcollins.smoker5@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907150139
4614428|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.10.0|solaris|2.10|200907150229
4614373|pass|200907|cpan.dcollins.5100rc1@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150135
4614220|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150130
4614138|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150129
4613819|pass|200907|cpan.dcollins.5100@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907150123
4613112|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.10.0|solaris|2.10|200907150041
4610683|pass|200907|cpan.dcollins.smoker7@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907141704
4610487|unknown|200907|cpan.dcollins.5100rc1@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907141651
4609992|unknown|200907|cpan.dcollins.smoker8@gmail.com|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907141626
4609870|pass|200907|cpan.dcollins.smoker5@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907141618
4609794|pass|200907|cpan.dcollins.smoker6@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907141613
4608570|fail|200907|cpan.dcollins.smoker2@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907141521
4608133|pass|200907|cpan.dcollins.smoker3@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907141508
4607720|pass|200907|cpan.dcollins.589@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907141457
4607683|unknown|200907|cpan.dcollins.smoker1@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907141456
4607619|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|WWW-Facebook-Go-SGF|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907150455
4607329|pass|200907|cpan.dcollins.smoker2@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907141447
4607004|unknown|200907|cpan.dcollins.smoker5@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907141440
4604872|pass|200907|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907141101
4604012|pass|200907|rlane10@wi.rr.com|Data-Compare|1.2101|darwin-2level-ld|5.10.0|darwin|8.11.0|200907140017
4602940|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd|5.10.0|openbsd|4.5|200907141207
4602561|pass|200907|rlane10@wi.rr.com|XML-Tiny|2.02|darwin-2level-ld|5.10.0|darwin|8.11.0|200907131744
4601456|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd|5.10.0|openbsd|4.5|200907140949
4598708|pass|200907|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.6.24-16-generic|200907140134
4598012|unknown|200907|cpan.dcollins.smoker1@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140115
4597698|unknown|200907|cpan.dcollins.smoker5@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140102
4596367|pass|200907|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200907140016
4594982|pass|200907|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux|5.10.0|linux|2.6.24-16-generic|200907140232
4594872|unknown|200907|cpan.dcollins.smoker5@gmail.com|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140228
4594643|unknown|200907|cpan.dcollins.5100rc1@gmail.com|NestedMap|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140224
4594236|pass|200907|cpan.dcollins.smoker5@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907140214
4593018|unknown|200907|cpan.dcollins.smoker2@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140108
4592516|pass|200907|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.24-16-generic|200907140058
4592512|pass|200907|cpan.dcollins.smoker2@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140058
4592306|unknown|200907|cpan.dcollins.smoker9@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140054
4592037|unknown|200907|cpan.dcollins.smoker8@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907140049
4587509|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907131930
4586880|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907131912
4585974|pass|200907|cpan.dcollins.smoker7@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907131849
4584863|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200907132227
4583981|pass|200907|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.6.24-16-generic|200907131612
4583857|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907131601
4583851|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907131601
4582863|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Transactional|1.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907131456
4582802|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907131452
4582800|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907131452
4582771|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907131451
4582258|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200907131908
4581695|pass|200907|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux|5.10.0|linux|2.6.24-16-generic|200907131334
4579815|pass|200907|cpan@greerga.m-l.org ("George Greer")|DBIx-Class-SingletonRows|0.11|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907131057
4577632|pass|200907|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200907130827
4577180|pass|200907|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.24-16-generic|200907130750
4576087|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907130623
4574481|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907130219
4574480|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907130218
4574405|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Tie-STDOUT|1.0401|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131604
4574285|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Transactional|1.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907130132
4574125|pass|200907|cpan@greerga.m-l.org ("George Greer")|Sort-MultipleFields|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907130055
4573806|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny-DOM|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131401
4573805|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny|2.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131401
4573800|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Palm-ProjectGutenberg|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131400
4573726|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131353
4573697|pass|200907|cpan.dcollins.smoker9@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907130351
4573561|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907130341
4573407|pass|200907|cpan.dcollins.smoker4@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907130330
4573393|pass|200907|cpan@greerga.m-l.org ("George Greer")|DBIx-Class-SingletonRows|0.11|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907130329
4573207|pass|200907|cpan.dcollins.smoker4@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907130321
4573064|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907130315
4572422|pass|200907|cpan.dcollins.smoker7@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907130249
4572248|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907130246
4572034|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907130240
4572029|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907130239
4571742|pass|200907|cpan.dcollins.smoker1@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907130228
4571562|pass|200907|cpan.dcollins.smoker6@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907130222
4571219|pass|200907|cpan.dcollins.smoker9@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907130210
4570999|pass|200907|cpan.dcollins.smoker8@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907130203
4570858|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|CPAN-ParseDistribution|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131158
4570637|pass|200907|cpan.dcollins.smoker6@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907130151
4570630|pass|200907|cpan.dcollins.smoker9@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907130151
4570520|pass|200907|cpan.dcollins.smoker4@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907130147
4570369|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907130142
4570245|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.5|200907130243
4569997|pass|200907|bob@randomness.org.uk ("bob walker")|Devel-CheckOS|1.61|i86pc-solaris|5.10.0|solaris|2.10|200907130136
4569996|pass|200907|bob@randomness.org.uk ("bob walker")|Data-Compare|1.2101|i86pc-solaris|5.10.0|solaris|2.10|200907130136
4569865|fail|200907|ivan.wills@gmail.com ("Ivan Wills")|Bryar|3.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907131013
4569849|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907130011
4569580|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907130931
4569469|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Palm-Treo680MessagesDB|1.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907130911
4569466|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907130911
4569459|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|File-Overwrite|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907130909
4569213|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907121826
4569206|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907121825
4569163|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907121820
4568986|pass|200907|cpan@greerga.m-l.org ("George Greer")|Sort-MultipleFields|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907121749
4568963|fail|200907|rlane10@wi.rr.com|CPAN-FindDependencies|2.32|darwin-2level-ld|5.10.0|darwin|8.11.0|200907121540
4568961|pass|200907|rlane10@wi.rr.com|Devel-CheckOS|1.61|darwin-2level-ld|5.10.0|darwin|8.11.0|200907121520
4568057|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Number-Phone|1.7002|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907130511
4566493|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907120730
4564920|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.5|200907120526
4564797|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907120313
4564518|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200907120300
4563690|pass|200907|cpan.dcollins.smoker1@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907120127
4563421|fail|200907|cpan.dcollins.smoker5@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907120119
4562847|pass|200907|cpan.dcollins.smoker6@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907120103
4562775|fail|200907|cpan.dcollins.5100@gmail.com|Bryar|3.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907120101
4562425|fail|200907|cpan.dcollins.5100@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907120050
4560221|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.5|200907120121
4559561|pass|200907|cpan.dcollins.smoker1@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907111941
4559559|fail|200907|cpan.dcollins.589@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907111941
4559554|pass|200907|cpan.dcollins.smoker1@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907111941
4557312|pass|200907|andy.situ@verizon.net|CPAN-FindDependencies|2.32|i686-linux-multi-64int|5.10.0|linux|2.6.24-24-generic|200907111452
4557311|pass|200907|andy.situ@verizon.net|Devel-CheckOS|1.61|i686-linux-multi-64int|5.10.0|linux|2.6.24-24-generic|200907111450
4557310|pass|200907|andy.situ@verizon.net|Data-Compare|1.2101|i686-linux-multi-64int|5.10.0|linux|2.6.24-24-generic|200907111450
4555617|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-SMS|0.03|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907111212
4553695|pass|200907|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110626
4553648|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110618
4553390|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907111953
4553175|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200907111006
4552622|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-SMS|0.03|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907110440
4552329|pass|200907|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110334
4552169|fail|200907|ivan.wills@gmail.com ("Ivan Wills")|Bryar|3.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907111633
4551871|pass|200907|cpan.dcollins.smoker7@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907110213
4551657|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907110206
4550796|pass|200907|cpan.dcollins.5100@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907110141
4550566|pass|200907|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907110135
4550155|pass|200907|cpan.dcollins.smoker7@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907110126
4550044|pass|200907|cpan.dcollins.5100@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907110123
4549236|pass|200907|cpan.dcollins.5100rc1@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907110055
4549215|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907110107
4549069|pass|200907|cpan.dcollins.smoker6@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907110104
4548527|pass|200907|cpan.dcollins.5100rc1@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907110055
4547708|pass|200907|cpan.dcollins.smoker7@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907110033
4547578|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907110029
4546583|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110258
4546364|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110219
4546363|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110219
4546359|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110219
4546199|pass|200907|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907110142
4546170|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907110133
4546098|fail|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907110113
4545902|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101955
4545857|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101944
4545855|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101943
4545854|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101943
4545446|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101816
4545423|pass|200907|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101810
4545258|fail|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101735
4545181|fail|200907|bingos@cpan.org|Bryar|3.1|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907102222
4543559|pass|200907|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907101936
4543480|pass|200907|bingos@cpan.org|Acme-Pony|1.1.2|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907101934
4543404|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101431
4543386|pass|200907|cpan.dcollins.589@gmail.com|Net-Random|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907101430
4543358|pass|200907|bingos@cpan.org|Acme-Licence|1.0|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907101931
4542219|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101330
4542218|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101330
4542215|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101330
4542079|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101302
4541484|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101154
4541436|fail|200907|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907101149
4540515|pass|200907|cpan.dcollins.smoker7@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907101114
4539762|pass|200907|stro@cpan.org|DBIx-Class-SingletonRows|0.11|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200907101756
4539748|pass|200907|stro@cpan.org|DBIx-Class-SingletonRows|0.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200907101756
4539556|pass|200907|cpan.dcollins.smoker3@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907101051
4538757|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101030
4538061|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101015
4538036|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101014
4537994|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907101014
4537957|pass|200907|cpan.dcollins.5100@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907101013
4535957|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907100714
4535117|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907100556
4533438|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.5|200907100801
4531593|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907100232
4531093|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907100131
4531091|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907100131
4530269|pass|200907|cpan.dcollins.smoker3@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907100040
4529945|pass|200907|cpan.dcollins.smoker7@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907100034
4528724|pass|200907|cpan.dcollins.589@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907100013
4528213|pass|200907|cpan.dcollins.5100@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907100003
4527882|pass|200907|cpan.dcollins.589@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907091957
4526603|pass|200907|stro@cpan.org|Palm-Treo680MessagesDB|1.01|armv5tejl-linux|5.10.0|linux|2.6.17.14|200907100229
4526017|pass|200907|cpan.dcollins.589@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907091918
4525801|pass|200907|cpan.dcollins.smoker5@gmail.com|Net-Random|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907091913
4525669|pass|200907|stro@cpan.org|File-Overwrite|1.1|armv5tejl-linux|5.10.0|linux|2.6.17.14|200907100211
4524114|pass|200907|cpan@tenzobox.net ("Axel Bourgeois")|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907091837
4523690|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907100437
4523041|pass|200907|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200907091837
4522945|pass|200907|andy@hexten.net|Data-Compare|1.2101|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200907091837
4522363|pass|200907|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200907091739
4522360|pass|200907|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200907091738
4518220|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907090943
4517749|pass|200907|cpan.dcollins.5100rc1@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907090932
4516898|pass|200907|cpan.dcollins.smoker2@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907090910
4516862|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907092310
4516516|pass|200907|cpan.dcollins.smoker2@gmail.com|Net-Random|2.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907090859
4514318|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200907091211
4514313|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200907091210
4514274|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.5|200907091246
4514270|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.5|200907091245
4512323|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-SMS|0.03|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907090210
4511135|pass|200907|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907090146
4511108|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907090141
4510454|pass|200907|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081940
4510040|pass|200907|bingos@cpan.org|Devel-CheckLib|0.6|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907082302
4509824|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081715
4509604|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081645
4509603|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081645
4509599|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081645
4509326|fail|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081556
4509182|pass|200907|cpan.dcollins.smoker6@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907081547
4508840|pass|200907|cpan.dcollins.smoker4@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907081538
4508318|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907081523
4508152|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081515
4508099|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081508
4508098|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081508
4508097|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081507
4507538|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081340
4507268|pass|200907|cpan.dcollins.smoker3@gmail.com|Net-Random|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907081323
4506417|pass|200907|cpan.dcollins.smoker7@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907081258
4506408|fail|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907081258
4506273|pass|200907|cpan.dcollins.smoker4@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907081254
4503611|unknown|200907|cpan.dcollins.smoker5@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907081148
4503359|pass|200907|cpan.dcollins.5100@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907081140
4502950|pass|200907|cpan.dcollins.smoker6@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907081129
4502199|pass|200907|cpan.dcollins.smoker1@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907081109
4501519|pass|200907|cpan.dcollins.smoker6@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907081054
4500395|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907080837
4499852|fail|200907|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907080724
4498497|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907080447
4498496|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907080446
4495987|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200907080107
4494802|pass|200907|bingos@cpan.org|Number-Phone|1.7002|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907071923
4494266|pass|200907|cpan.dcollins.smoker4@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907071311
4494147|pass|200907|cpan.dcollins.smoker8@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907071307
4494100|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907071305
4493945|pass|200907|cpan.dcollins.5100@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907071302
4492945|pass|200907|cpan.dcollins.smoker8@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907071230
4491970|pass|200907|cpan.dcollins.smoker3@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907071209
4491648|pass|200907|ss5@renormalist.net|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.18-5-k7|200907071801
4491338|pass|200907|cpan.dcollins.5100@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907071153
4490722|pass|200907|cpan.dcollins.589@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907071134
4490519|unknown|200907|cpan.dcollins.smoker9@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907071128
4489123|pass|200907|martin.evans@easysoft.com|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.24-24-generic|200907071446
4489122|pass|200907|martin.evans@easysoft.com|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200907071446
4488987|pass|200907|martin.evans@easysoft.com|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.24-24-generic|200907071423
4488554|pass|200907|martin.evans@easysoft.com|DBIx-Class-SingletonRows|0.11|i686-linux|5.10.0|linux|2.6.24-24-generic|200907071340
4488334|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907072218
4488330|pass|200907|ivan.wills@gmail.com ("Ivan Wills")|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-23-server|200907072218
4487631|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200907071220
4487043|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200907071123
4484480|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907070210
4484477|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907070210
4484469|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907070209
4484465|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907070209
4483987|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907070133
4483985|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907070132
4481331|pass|200907|cpan.dcollins.589@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907061907
4481096|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-64int|5.10.0 patch GitLive-maint-5.10-1501-gba4fcaa|netbsd|5.0|200907062358
4480991|pass|200907|cpan.dcollins.smoker2@gmail.com|NestedMap|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907061857
4480818|pass|200907|cpan.dcollins.smoker7@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907061852
4480703|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061849
4480609|pass|200907|cpan.dcollins.smoker4@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907061845
4480314|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907061835
4478620|pass|200907|cpan.dcollins.smoker6@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907061748
4477829|pass|200907|andy@hexten.net|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-686|200907062058
4477826|pass|200907|andy@hexten.net|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-686|200907062057
4476302|pass|200907|cpan.dcollins.smoker8@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061325
4475730|pass|200907|cpan.dcollins.5100@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061310
4475690|pass|200907|cpan.dcollins.smoker8@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907061309
4475685|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907061811
4475142|pass|200907|cpan.dcollins.smoker3@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061255
4474876|pass|200907|cpan.dcollins.smoker2@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907061248
4474702|pass|200907|cpan.dcollins.589@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061243
4474656|pass|200907|cpan.dcollins.smoker4@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061236
4474653|pass|200907|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.11.0 patch GitLive-blead-1530-g9ef4351|darwin|8.11.0|200907061744
4474377|pass|200907|cpan.dcollins.smoker7@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907061236
4474347|pass|200907|cpan.dcollins.smoker2@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061236
4474028|pass|200907|cpan.dcollins.smoker7@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061227
4473890|pass|200907|cpan.dcollins.5100@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061223
4473727|pass|200907|cpan.dcollins.smoker3@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907061219
4473415|pass|200907|cpan.dcollins.smoker5@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061211
4473161|pass|200907|cpan.dcollins.smoker3@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061205
4473025|pass|200907|cpan.dcollins.smoker7@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061201
4472707|pass|200907|cpan.dcollins.smoker4@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061154
4472273|pass|200907|cpan.dcollins.smoker1@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907061145
4472031|pass|200907|martin.evans@easysoft.com|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200907061640
4471341|pass|200907|cpan.dcollins.smoker2@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061125
4470995|pass|200907|cpan.dcollins.smoker8@gmail.com|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907061117
4470476|pass|200907|cpan.dcollins.5100rc1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907061106
4470450|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-64int|5.10.0 patch GitLive-maint-5.10-1501-gba4fcaa|netbsd|5.0|200907061604
4470351|pass|200907|cpan.dcollins.smoker2@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907061102
4469966|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907061051
4469874|pass|200907|cpan.dcollins.smoker5@gmail.com|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907061049
4469467|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-64int|5.10.0 patch GitLive-maint-5.10-1501-gba4fcaa|netbsd|5.0|200907061534
4469127|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907061010
4469126|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907061010
4469070|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907061003
4469069|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907061003
4468263|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907060915
4468253|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-13-generic|200907060915
4464346|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200907060547
4464302|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200907060515
4463291|pass|200907|cpan.dcollins.smoker3@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907060125
4463158|pass|200907|cpan.dcollins.smoker5@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907060122
4462637|pass|200907|cpan.dcollins.smoker8@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907060113
4462608|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060113
4462295|pass|200907|cpan.dcollins.smoker7@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060106
4461834|pass|200907|cpan.dcollins.smoker5@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060057
4461809|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907060056
4461543|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060052
4461399|pass|200907|cpan.dcollins.smoker1@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907060049
4460587|pass|200907|cpan.dcollins.5100@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907060029
4460017|pass|200907|cpan.dcollins.smoker2@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060016
4459339|pass|200907|cpan.dcollins.smoker7@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060005
4459235|pass|200907|cpan.dcollins.smoker6@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907060003
4459198|pass|200907|cpan.dcollins.smoker2@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907060002
4459093|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200907060055
4458760|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051905
4458758|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051905
4458753|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051905
4458518|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907051827
4457970|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051721
4457503|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051603
4457420|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051551
4457415|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051550
4457410|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051550
4456733|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051330
4456258|pass|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907051153
4455713|pass|200907|cpan.dcollins.smoker4@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050914
4455672|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050946
4455472|pass|200907|cpan.dcollins.smoker2@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907050850
4455351|pass|200907|cpan.dcollins.smoker5@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050933
4453930|pass|200907|cpan.dcollins.smoker4@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050922
4453746|pass|200907|cpan.dcollins.smoker2@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050919
4453169|fail|200907|cpan.dcollins.5100rc1@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907050908
4452166|pass|200907|cpan.dcollins.smoker4@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050851
4452134|pass|200907|cpan.dcollins.smoker1@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907050851
4452009|pass|200907|cpan.dcollins.smoker6@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907050849
4450797|pass|200907|cpan.dcollins.smoker5@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907050829
4450795|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907050829
4450490|pass|200907|cpan.dcollins.smoker7@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907050823
4450367|pass|200907|cpan.dcollins.smoker6@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907050821
4449593|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200907051123
4449491|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200907051055
4449314|pass|200907|g.grigelionis@computer.org|Statistics-SerialCorrelation|1.1|i586-linux-thread-multi|5.8.8|linux|2.6.16|200907051101
4449242|pass|200907|g.grigelionis@computer.org|Statistics-ChiSquare|0.5|i586-linux-thread-multi|5.8.8|linux|2.6.16|200907051047
4448992|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907050419
4448827|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-SMS|0.03|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907050340
4448386|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907050232
4447713|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.11.0|netbsd|5.0|200907050513
4447570|pass|200907|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907050345
4447563|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907050339
4446902|pass|200907|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907050121
4445660|pass|200907|cpan.dcollins.smoker9@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907041750
4445659|pass|200907|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041807
4445360|pass|200907|cpan.dcollins.smoker7@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041757
4445092|pass|200907|cpan.dcollins.smoker7@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041752
4445011|pass|200907|cpan.dcollins.5100@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041648
4444450|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907041740
4444372|pass|200907|cpan.dcollins.smoker7@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041739
4444062|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907041733
4443914|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041731
4443801|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041729
4443798|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041729
4443782|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041728
4443691|pass|200907|cpan.dcollins.smoker9@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041710
4443644|unknown|200907|cpan.dcollins.smoker3@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907041725
4443350|pass|200907|cpan.dcollins.smoker2@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041719
4443318|fail|200907|cpan.dcollins.589@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041701
4443279|pass|200907|cpan.dcollins.smoker1@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907041717
4442730|fail|200907|cpan.dcollins.smoker3@gmail.com|Bryar|3.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041708
4441847|pass|200907|cpan.dcollins.smoker4@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041651
4441773|pass|200907|cpan.dcollins.5100rc1@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907041650
4441769|pass|200907|cpan.dcollins.smoker7@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907041650
4441764|pass|200907|cpan.dcollins.smoker7@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907041650
4441686|pass|200907|cpan.dcollins.5100@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041648
4441593|pass|200907|cpan.dcollins.smoker2@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041647
4441532|pass|200907|cpan.dcollins.smoker9@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041645
4440938|fail|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041635
4440764|pass|200907|cpan.dcollins.smoker6@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041632
4440681|pass|200907|cpan.dcollins.smoker1@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907041631
4440590|pass|200907|cpan.dcollins.589@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041628
4439907|pass|200907|cpan.dcollins.smoker5@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041612
4439750|pass|200907|cpan.dcollins.smoker2@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041609
4439187|pass|200907|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041545
4439130|pass|200907|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041536
4439128|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041536
4439125|pass|200907|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041535
4438574|pass|200907|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907041408
4438379|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.11.0|netbsd|5.0|200907041822
4438203|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.11.0|netbsd|5.0|200907041740
4437524|pass|200907|cpan.dcollins.smoker4@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907041137
4436679|pass|200907|cpan.dcollins.smoker1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041135
4436039|pass|200907|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200907041624
4436030|pass|200907|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200907041624
4435915|pass|200907|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907041122
4435781|pass|200907|cpan.dcollins.smoker9@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041120
4435178|pass|200907|cpan.dcollins.smoker6@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907041108
4435053|pass|200907|cpan.dcollins.smoker6@gmail.com|Net-Random|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907041105
4434913|pass|200907|cpan.dcollins.smoker1@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907041100
4434830|pass|200907|cpan.dcollins.smoker8@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041057
4434822|pass|200907|cpan.dcollins.smoker2@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907041057
4434431|pass|200907|cpan.dcollins.5100@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907041047
4434377|pass|200907|cpan.dcollins.smoker6@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041046
4434299|pass|200907|cpan.dcollins.smoker1@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041044
4433817|pass|200907|cpan.dcollins.smoker4@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041033
4433416|fail|200907|cpan.dcollins.smoker8@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907041021
4433267|pass|200907|cpan.dcollins.smoker5@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907041017
4432711|pass|200907|cpan.dcollins.5100@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907041002
4432507|pass|200907|cpan.dcollins.smoker7@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907040954
4432357|pass|200907|cpan.dcollins.5100rc1@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907040935
4431488|pass|200907|cpan.dcollins.smoker7@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907040920
4431380|pass|200907|cpan.dcollins.smoker4@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907040916
4430488|fail|200907|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907040435
4429095|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907040338
4428667|fail|200907|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200907040152
4428458|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200907040217
4425633|pass|200907|cpan.dcollins.smoker3@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031449
4425531|pass|200907|cpan.dcollins.smoker2@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031447
4425156|pass|200907|cpan.dcollins.smoker6@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907031440
4424754|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907031433
4424134|pass|200907|cpan.dcollins.smoker6@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907031419
4423788|pass|200907|cpan.dcollins.smoker1@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907031354
4423782|pass|200907|cpan.dcollins.smoker1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907031412
4423312|pass|200907|cpan.dcollins.smoker4@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031403
4422176|pass|200907|cpan.dcollins.5100rc1@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031339
4421628|pass|200907|cpan.dcollins.smoker8@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031327
4421418|pass|200907|cpan.dcollins.smoker6@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031322
4421365|pass|200907|cpan.dcollins.smoker3@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031320
4420069|pass|200907|cpan.dcollins.smoker8@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031243
4420028|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907031242
4418889|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200907031508
4418646|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200907031405
4417843|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200907031004
4417709|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200907030937
4416235|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907030116
4416234|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-13-generic|200907030116
4415917|pass|200907|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907030038
4415915|pass|200907|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-13-generic|200907030038
4412843|pass|200907|cpan.dcollins.5100@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907021925
4412744|pass|200907|cpan.dcollins.5100@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907021922
4412634|pass|200907|cpan.dcollins.5100@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907021920
4412245|pass|200907|cpan.dcollins.smoker4@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907021910
4411907|pass|200907|cpan.dcollins.smoker9@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907021900
4411768|pass|200907|cpan.dcollins.smoker1@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907021857
4410643|pass|200907|cpan.dcollins.smoker9@gmail.com|NestedMap|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907021828
4410628|pass|200907|cpan.dcollins.589@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907021827
4410317|unknown|200907|cpan.dcollins.smoker8@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907021819
4409879|pass|200907|cpan.dcollins.smoker8@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907021809
4406914|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd-64int|5.10.0 patch GitLive-maint-5.10-1462-g178839f|midnightbsd|0.1.1-release|200907021521
4404657|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|darwin-64int-2level|5.8.6|darwin|8.11.0|200907020957
4404515|pass|200907|bingos@cpan.org|Data-Compare|1.2101|darwin-64int-2level|5.8.6|darwin|8.11.0|200907020934
4404374|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200907020958
4403753|pass|200907|cpan.dcollins.smoker5@gmail.com|Net-Random|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907020043
4403637|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd-64int|5.10.0 patch GitLive-maint-5.10-1462-g178839f|midnightbsd|0.1.1-release|200907020442
4402923|pass|200907|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd-64int|5.10.0 patch GitLive-maint-5.10-1462-g178839f|midnightbsd|0.1.1-release|200907020407
4402806|unknown|200907|cpan.dcollins.smoker9@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907020040
4402662|pass|200907|cpan.dcollins.smoker7@gmail.com|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907020037
4402563|pass|200907|cpan.dcollins.5100rc1@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907020036
4402079|pass|200907|cpan.dcollins.smoker2@gmail.com|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907020027
4401703|pass|200907|cpan.dcollins.5100@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907020018
4401123|pass|200907|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200907020500
4398109|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200907020012
4398048|pass|200907|cpan.dcollins.smoker3@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907011801
4397871|pass|200907|cpan.dcollins.smoker4@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907011754
4397629|pass|200907|cpan.dcollins.smoker6@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907011743
4397365|pass|200907|cpan.dcollins.smoker4@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907011731
4396870|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200907012307
4396209|pass|200907|cpan.dcollins.smoker8@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907011634
4393696|pass|200907|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200907011825
4393466|pass|200907|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200907011756
4393000|pass|200907|cpan.dcollins.589@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907011140
4392370|pass|200907|cpan.dcollins.smoker3@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200907011104
4390559|pass|200907|cpan.dcollins.smoker4@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907010946
4389471|pass|200907|cpan.dcollins.smoker8@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907010904
4389466|pass|200907|cpan.dcollins.smoker1@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907010904
4389365|pass|200907|cpan.dcollins.smoker7@gmail.com|NestedMap|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907010901
4388882|pass|200907|cpan.dcollins.589@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200907010847
4386726|pass|200907|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200907011232
4383139|pass|200907|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd-64int|5.11.0 patch GitLive-blead-1493-g9015f10|midnightbsd|0.1.1-release|200907010213
4381647|pass|200907|cpan.dcollins.smoker9@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200907010025
4381565|pass|200907|cpan.dcollins.smoker9@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907010022
4381480|pass|200907|cpan.dcollins.5100rc1@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907010020
4381405|pass|200907|cpan.dcollins.5100rc1@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200907010018
4380734|pass|200907|cpan.dcollins.smoker7@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200907010006
4380054|pass|200906|cpan.dcollins.smoker8@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906301958
4379510|pass|200906|cpan.dcollins.smoker8@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906301950
4379397|pass|200906|cpan.dcollins.589@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906301934
4378756|pass|200906|cpan.dcollins.589@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906301941
4378278|pass|200906|cpan.dcollins.smoker3@gmail.com|NestedMap|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906301933
4377586|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200906301923
4377580|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.9 patch 35104|linux|2.6.28-11-generic|200906301923
4376962|pass|200906|cpan.dcollins.smoker4@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906301855
4376353|unknown|200906|cpan.dcollins.589@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906301837
4376265|pass|200906|cpan.dcollins.5100rc1@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906301835
4374719|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200906302243
4373520|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd-64int|5.11.0 patch GitLive-blead-1493-g9015f10|midnightbsd|0.1.1-release|200906301546
4373312|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd-64int|5.11.0 patch GitLive-blead-1493-g9015f10|midnightbsd|0.1.1-release|200906301509
4372181|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-11-generic|200906300902
4372180|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.11.0|linux|2.6.28-11-generic|200906300902
4369545|pass|200906|autarch@urth.org|Devel-CheckOS|1.61|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906291109
4369373|pass|200906|autarch@urth.org|CPAN-FindDependencies|2.32|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906291109
4367833|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-11-generic|200906300319
4367830|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.28-11-generic|200906300319
4367793|pass|200906|cpan.dcollins.smoker9@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906300316
4367508|pass|200906|cpan.dcollins.589@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906300257
4367057|pass|200906|cpan.dcollins.smoker5@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906300236
4366891|pass|200906|cpan.dcollins.smoker1@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906300227
4366244|pass|200906|cpan.dcollins.smoker4@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906300209
4365888|pass|200906|cpan.dcollins.smoker1@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906300202
4365787|pass|200906|cpan.dcollins.5100@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906300201
4365685|pass|200906|cpan.dcollins.5100rc1@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906300159
4365433|unknown|200906|cpan.dcollins.smoker9@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906300155
4365178|pass|200906|cpan.dcollins.smoker6@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906300152
4364718|pass|200906|cpan.dcollins.smoker6@gmail.com|Net-Random|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906300145
4364456|pass|200906|cpan.dcollins.smoker6@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906300141
4363024|pass|200906|cpan.dcollins.5100rc1@gmail.com|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906300117
4362961|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200906300315
4362292|pass|200906|cpan.dcollins.smoker2@gmail.com|Net-Random|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906300105
4362054|pass|200906|cpan.dcollins.smoker1@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906300101
4360983|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200906300154
4358128|pass|200906|cpan.dcollins.5100@gmail.com|Sub-WrapPackages|1.2|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906291510
4357835|pass|200906|cpan.dcollins.smoker8@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906291502
4355022|pass|200906|cpan.dcollins.smoker5@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906291338
4354548|pass|200906|cpan.dcollins.smoker6@gmail.com|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906291340
4354360|pass|200906|cpan.dcollins.5100@gmail.com|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906291335
4353903|pass|200906|cpan.dcollins.smoker4@gmail.com|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906291325
4353274|pass|200906|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200906291243
4352831|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200906291737
4352487|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd|5.8.9|midnightbsd|0.1.1-release|200906291430
4350411|pass|200906|stro@rail.org.ua|Devel-CheckOS|1.61|i386-freebsd|5.8.9|freebsd|7.2-release|200906291031
4349745|pass|200906|stro@rail.org.ua|Data-Compare|1.2101|i386-freebsd|5.8.9|freebsd|7.2-release|200906291011
4349525|pass|200906|andy.situ@verizon.net|Devel-CheckOS|1.61|i686-linux-multi-64int-ld|5.10.0|linux|2.6.22.1.emp6|200906281759
4349522|pass|200906|andy.situ@verizon.net|Data-Compare|1.2101|i686-linux-multi-64int-ld|5.10.0|linux|2.6.22.1.emp6|200906281759
4349521|pass|200906|andy.situ@verizon.net|CPAN-FindDependencies|2.32|i686-linux-multi-64int-ld|5.10.0|linux|2.6.22.1.emp6|200906281800
4346878|pass|200906|cpan.dcollins.smoker3@gmail.com|Statistics-ChiSquare|0.5|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906290334
4346423|pass|200906|cpan.dcollins.smoker3@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906290326
4346179|pass|200906|cpan.dcollins.smoker4@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906290320
4346014|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200906290511
4345739|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200906290403
4345375|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200906290204
4344689|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906281941
4344688|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906281941
4342973|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906281643
4342971|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906281643
4342896|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd|5.8.9|midnightbsd|0.1.1-release|200906282027
4342508|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd|5.8.9|midnightbsd|0.1.1-release|200906281948
4340255|pass|200906|cpan.dcollins.smoker4@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906281002
4339162|pass|200906|cpan.dcollins.5100rc1@gmail.com|Net-Random|2.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906280952
4339083|pass|200906|cpan.dcollins.5100@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906280951
4338141|pass|200906|cpan.dcollins.smoker9@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906280932
4338126|pass|200906|cpan.dcollins.smoker5@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906280913
4337848|pass|200906|cpan.dcollins.smoker2@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906280911
4336958|pass|200906|cpan.dcollins.smoker5@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906280909
4336660|pass|200906|cpan.dcollins.smoker1@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906280851
4336541|pass|200906|cpan.dcollins.589@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906280901
4335328|pass|200906|cpan.dcollins.smoker9@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906280837
4333266|pass|200906|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200906281024
4331561|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200906280801
4331311|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200906280620
4331145|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200906280529
4329967|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906280159
4329964|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906280159
4329773|pass|200906|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.6|i686-linux|5.10.0|linux|2.6.24-16-generic|200906280143
4328923|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906280043
4328919|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906280042
4328370|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906271939
4328369|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906271939
4327446|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906272347
4327445|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906272346
4325329|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-SerialCorrelation|1.1|darwin-2level|5.10.0|darwin|8.11.0|200906271927
4325282|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-SerialCorrelation|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906271924
4325267|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-ChiSquare|0.5|darwin-2level|5.10.0|darwin|8.11.0|200906271923
4325233|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-ChiSquare|0.5|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906271921
4323458|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200906271305
4323337|pass|200906|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200906271228
4323204|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200906271321
4322043|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd|5.10.0|midnightbsd|0.1.1-release|200906270659
4320925|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200906270531
4318893|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Devel-CheckOS|1.61|i686-linux-thread-multi|5.11.0|linux|2.6.29.1.20090403|200906262341
4318890|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Data-Compare|1.2101|i686-linux-thread-multi|5.11.0|linux|2.6.29.1.20090403|200906262341
4318649|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-SerialCorrelation|1.1|darwin-2level|5.8.8|darwin|8.11.0|200906262314
4318612|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-ChiSquare|0.5|darwin-2level|5.8.8|darwin|8.11.0|200906262308
4318204|pass|200906|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200906262202
4318103|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd|5.10.0|midnightbsd|0.1.1-release|200906262011
4317767|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd|5.10.0|midnightbsd|0.1.1-release|200906261938
4317411|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.11.0|linux|2.6.29.1.20090403|200906262111
4317316|pass|200906|ipenburg@xs4all.nl ("Roland van Ipenburg")|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.11.0|linux|2.6.29.1.20090403|200906262105
4313231|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200906261215
4313102|pass|200906|martin.evans@easysoft.com|Palm-SMS|0.03|i686-linux|5.10.0|linux|2.6.24-24-generic|200906261100
4312988|pass|200906|justin.d.hunter@gmail.com|DBIx-Class-SingletonRows|0.11|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200906260244
4312414|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200906260955
4312187|pass|200906|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200906260858
4312103|pass|200906|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200906260858
4310902|pass|200906|justin.d.hunter@gmail.com|XML-Tiny|2.02|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200906260329
4310162|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200906260355
4309712|pass|200906|cpan.dcollins.smoker3@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260126
4309688|pass|200906|cpan.dcollins.smoker3@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906260125
4309588|pass|200906|cpan.dcollins.smoker4@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906260120
4309506|pass|200906|cpan.dcollins.5100rc1@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906260116
4309484|pass|200906|cpan.dcollins.5100@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906260116
4309385|pass|200906|cpan.dcollins.smoker3@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906260113
4309291|pass|200906|cpan.dcollins.smoker6@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260111
4309070|pass|200906|cpan.dcollins.5100rc1@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260106
4309027|pass|200906|cpan.dcollins.5100rc1@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260105
4308825|pass|200906|cpan.dcollins.smoker2@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260101
4308756|pass|200906|cpan.dcollins.smoker8@gmail.com|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260059
4308752|pass|200906|cpan.dcollins.smoker3@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906260059
4308703|pass|200906|cpan.dcollins.smoker8@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906260058
4308628|pass|200906|cpan.dcollins.smoker3@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260056
4308598|pass|200906|cpan.dcollins.smoker9@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906260055
4308574|pass|200906|cpan.dcollins.smoker7@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906260055
4308559|pass|200906|cpan.dcollins.smoker3@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906260054
4308517|pass|200906|cpan.dcollins.smoker3@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906260053
4307049|pass|200906|cpan.dcollins.smoker9@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251757
4306961|pass|200906|cpan.dcollins.smoker6@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906251754
4306889|pass|200906|cpan.dcollins.5100rc1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251752
4306305|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200906252136
4306299|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200906252136
4306212|pass|200906|cpan.dcollins.smoker7@gmail.com|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906251734
4306186|pass|200906|cpan.dcollins.smoker4@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906251734
4306034|pass|200906|cpan.dcollins.smoker2@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251729
4305830|pass|200906|cpan.dcollins.smoker8@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251724
4305765|pass|200906|cpan.dcollins.smoker8@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251721
4303725|pass|200906|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200906251910
4303660|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-2level|5.8.7|darwin|8.11.0|200906251928
4302977|fail|200906|cpan.dcollins.smoker1@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906251214
4301842|pass|200906|cpan.dcollins.smoker5@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251240
4301056|fail|200906|cpan.dcollins.smoker4@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251211
4299798|pass|200906|cpan.dcollins.smoker1@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906251058
4299120|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-2level|5.8.7|darwin|8.11.0|200906251643
4299025|pass|200906|cpan.dcollins.smoker4@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906251140
4298592|pass|200906|cpan.dcollins.smoker6@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906251129
4298474|pass|200906|cpan.dcollins.smoker6@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906251109
4297772|pass|200906|cpan.dcollins.smoker4@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906251111
4297443|pass|200906|cpan.dcollins.smoker5@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906251104
4297111|pass|200906|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906251053
4297041|pass|200906|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906251038
4297031|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906251036
4296912|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906251023
4296881|pass|200906|cpan@greerga.m-l.org ("George Greer")|Class-DBI-ClassGenerator|1.02|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906251018
4296795|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Find-Rule-Permissions|2.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906251006
4296701|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200906251355
4293122|pass|200906|cpan.dcollins.smoker7@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906250020
4291491|unknown|200906|cpan.dcollins.5100rc1@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906250028
4291126|pass|200906|cpan.dcollins.smoker9@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906250019
4290999|pass|200906|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906250016
4289835|pass|200906|cpan.dcollins.smoker7@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906250352
4289441|pass|200906|cpan.dcollins.smoker6@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906250343
4288058|pass|200906|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906250312
4288027|pass|200906|cpan.dcollins.smoker9@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906250311
4287740|pass|200906|cpan.dcollins.smoker8@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906250305
4287492|pass|200906|cpan.dcollins.smoker8@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1437-g8bca5c8|linux|2.6.30-020630rc7-generic|200906250259
4286319|pass|200906|doug@peaches.acomp.usf.edu|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.1|200906250040
4286216|pass|200906|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906250019
4286189|pass|200906|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906250013
4285516|pass|200906|cpan@greerga.m-l.org ("George Greer")|WWW-Facebook-Go-SGF|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906241652
4284099|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200906241636
4282398|pass|200906|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906240907
4282396|pass|200906|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906240906
4282381|pass|200906|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200906241349
4281791|pass|200906|cpan.dcollins.smoker3@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906240837
4281761|pass|200906|cpan.dcollins.smoker6@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906240837
4280769|pass|200906|cpan.dcollins.smoker7@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906240828
4280366|pass|200906|cpan.dcollins.smoker6@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906240818
4278977|pass|200906|chris@bingosnet.co.uk|Devel-CheckLib|0.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200906241237
4278732|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200906241207
4278731|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200906241207
4278487|pass|200906|martin.evans@easysoft.com|Class-DBI-ClassGenerator|1.02|i686-linux|5.10.0|linux|2.6.24-24-generic|200906241121
4278381|pass|200906|martin.evans@easysoft.com|File-Find-Rule-Permissions|2.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200906241058
4277792|pass|200906|cpan.dcollins.smoker5@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906231358
4277742|pass|200906|cpan.dcollins.smoker1@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906231053
4276759|pass|200906|cpan.dcollins.smoker5@gmail.com|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906240401
4276661|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|cygwin-thread-multi-64int|5.8.9|cygwin|1.5.25(0.15642)|200906240456
4276588|pass|200906|cpan.dcollins.smoker6@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906240354
4276002|pass|200906|cpan.dcollins.5100rc1@gmail.com|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906240335
4275950|pass|200906|cpan.dcollins.5100@gmail.com|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906240334
4274966|pass|200906|cpan.dcollins.589@gmail.com|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906240308
4274783|pass|200906|cpan.dcollins.5100rc1@gmail.com|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906240304
4273942|pass|200906|cpan.dcollins.smoker2@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906240242
4273577|pass|200906|cpan@greerga.m-l.org ("George Greer")|Tie-STDOUT|1.0401|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906240225
4273573|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200906240422
4273278|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906240123
4273274|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906240123
4273258|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906240123
4272938|pass|200906|cpan.dcollins.smoker5@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906231400
4272402|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906240031
4272401|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906240031
4272389|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906240030
4272350|pass|200906|cpan.dcollins.smoker8@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906231053
4271912|pass|200906|bingos@cpan.org|Data-Compare|1.2101|cygwin-thread-multi-64int|5.8.9|cygwin|1.5.25(0.15642)|200906240102
4271786|pass|200906|cpan.dcollins.smoker6@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906231336
4270714|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906231853
4270712|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906231853
4270705|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906231852
4270668|pass|200906|cpan.dcollins.smoker8@gmail.com|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906231030
4270206|pass|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906231753
4270035|pass|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906231732
4269764|fail|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906231644
4268907|pass|200906|cpan.dcollins.smoker2@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906231357
4268846|pass|200906|cpan.dcollins.589@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906231401
4267652|pass|200906|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906231048
4267328|pass|200906|cpan.dcollins.smoker7@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906231025
4267129|pass|200906|cpan.dcollins.smoker7@gmail.com|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906231046
4266730|pass|200906|cpan.dcollins.5100rc1@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906231048
4266635|pass|200906|cpan.dcollins.smoker7@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906231029
4266560|pass|200906|cpan.dcollins.5100rc1@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906231042
4266376|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906231033
4266032|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906230943
4266029|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906230943
4266018|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906230941
4265284|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906230803
4265231|pass|200906|cpan.dcollins.smoker6@gmail.com|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221751
4264901|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906230719
4264898|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906230719
4264887|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906230717
4264650|pass|200906|cpan.dcollins.smoker8@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221826
4264557|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906230650
4264452|pass|200906|cpan.dcollins.smoker7@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906221722
4264204|pass|200906|cpan.dcollins.smoker4@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221703
4264063|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906230615
4264061|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906230615
4264057|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906230613
4263546|pass|200906|cpan.dcollins.smoker9@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906221749
4263433|pass|200906|cpan.dcollins.smoker8@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221826
4263202|pass|200906|cpan.dcollins.smoker9@gmail.com|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906221741
4263139|pass|200906|cpan.dcollins.smoker5@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906221709
4263018|pass|200906|cpan.dcollins.smoker9@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906221724
4262837|pass|200906|cpan.dcollins.smoker9@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906221758
4262814|pass|200906|cpan.dcollins.smoker9@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906221758
4262780|pass|200906|cpan.dcollins.smoker4@gmail.com|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906221710
4262108|pass|200906|cpan.dcollins.smoker4@gmail.com|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221703
4261723|pass|200906|cpan.dcollins.smoker9@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221824
4260840|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200906230924
4260754|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200906230909
4260121|pass|200906|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906230029
4259828|pass|200906|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906230346
4259706|pass|200906|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906230323
4259126|pass|200906|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200906230208
4258677|pass|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906230017
4258510|pass|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906221945
4258090|pass|200906|cpan.dcollins.5100@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906221508
4257522|pass|200906|cpan.dcollins.smoker1@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906221523
4257445|pass|200906|cpan.dcollins.smoker1@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906221509
4257241|pass|200906|trey+cpanplus@lopsa.org|Data-Compare|1.2101|darwin-2level|5.10.0|darwin|9.6.0|200906220946
4257060|pass|200906|cpan.dcollins.smoker3@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906221400
4256563|pass|200906|cpan.dcollins.smoker1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906221508
4254403|pass|200906|cpan.dcollins.5100@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906221502
4253639|pass|200906|cpan.dcollins.5100@gmail.com|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906221356
4253035|pass|200906|martin.evans@easysoft.com|WWW-Facebook-Go-SGF|1.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200906221800
4251867|pass|200906|bejuryu@gmail.com|CPAN-FindDependencies|2.32|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906221835
4251848|pass|200906|bejuryu@gmail.com|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906221834
4251743|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi|5.8.9|netbsd|5.0|200906221446
4251240|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200906221453
4248084|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200906220033
4248013|pass|200906|cpan.dcollins.5100@gmail.com|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906211934
4247753|pass|200906|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906211927
4247684|pass|200906|cpan.dcollins.5100rc1@gmail.com|Number-Phone|1.7002|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906211925
4247092|pass|200906|cpan.dcollins.5100@gmail.com|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906211910
4244731|pass|200906|bingos@cpan.org|Data-Compare|1.2101|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200906212009
4244449|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi|5.8.9|netbsd|5.0|200906211910
4243789|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi|5.8.9|netbsd|5.0|200906211707
4240345|fail|200906|dagolden@cpan.org (DAGOLDEN)|Bryar|3.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200906210342
4240160|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906210329
4239172|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906210236
4238136|pass|200906|justin.d.hunter@gmail.com|Devel-CheckOS|1.61|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200906210547
4236953|pass|200906|justin.d.hunter@gmail.com|Data-Compare|1.2101|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200906210502
4236261|pass|200906|l.mai@web.de|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.22-gentoo-r9|200906202116
4236105|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906210028
4231331|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi|5.10.0|netbsd|5.0|200906210158
4230930|fail|200906|cpan.dcollins.589@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906210051
4230926|fail|200906|cpan.dcollins.589@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906210051
4230924|fail|200906|cpan.dcollins.589@gmail.com|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906210051
4230922|fail|200906|cpan.dcollins.589@gmail.com|Bryar|3.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906210051
4229584|fail|200906|ivan.wills@gmail.com ("Ivan Wills")|Bryar|3.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200906210704
4227643|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200906201752
4226873|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.8.4|linux|2.6.29.4-167.fc11.i686.pae|200906201732
4226872|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.8.4|linux|2.6.29.4-167.fc11.i686.pae|200906201732
4226071|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i686-linux|5.8.3|linux|2.6.29.4-167.fc11.i686.pae|200906201540
4225852|pass|200906|andy@hexten.net|Data-Compare|1.2101|i686-linux|5.8.3|linux|2.6.29.4-167.fc11.i686.pae|200906201540
4225380|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.8.1|linux|2.6.24-19-server|200906201358
4225377|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.8.1|linux|2.6.24-19-server|200906201358
4225340|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.8.9 patch 35104|darwin|9.7.0|200906201353
4225339|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|darwin-thread-multi-2level|5.8.9 patch 35104|darwin|9.7.0|200906201353
4225171|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.8.3|linux|2.6.29.4-167.fc11.i686.pae|200906201441
4225170|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.8.3|linux|2.6.29.4-167.fc11.i686.pae|200906201441
4224764|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i686-linux|5.8.2|linux|2.6.29.4-167.fc11.i686.pae|200906201345
4224762|pass|200906|andy@hexten.net|Data-Compare|1.2101|i686-linux|5.8.2|linux|2.6.29.4-167.fc11.i686.pae|200906201345
4224528|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi|5.10.0|netbsd|5.0|200906201152
4223752|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi|5.10.0|netbsd|5.0|200906200953
4222674|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200906200611
4222506|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200906200541
4222308|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.8.7|darwin|8.11.0|200906200407
4221929|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200906200358
4221876|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Sub-WrapPackages|1.2|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200906200345
4221730|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200906200203
4221697|fail|200906|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906200109
4221487|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200906200238
4221040|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200906200010
4220718|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200906200000
4220310|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200906192200
4219362|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.8.2|linux|2.6.29.4-167.fc11.i686.pae|200906191901
4219360|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.8.2|linux|2.6.29.4-167.fc11.i686.pae|200906191901
4219214|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i686-linux|5.8.1|linux|2.6.29.4-167.fc11.i686.pae|200906191843
4219213|pass|200906|andy@hexten.net|Data-Compare|1.2101|i686-linux|5.8.1|linux|2.6.29.4-167.fc11.i686.pae|200906191842
4218686|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.8.1|linux|2.6.29.4-167.fc11.i686.pae|200906191726
4218685|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.8.1|linux|2.6.29.4-167.fc11.i686.pae|200906191726
4218588|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i686-linux|5.6.2|linux|2.6.29.4-167.fc11.i686.pae|200906191712
4218587|pass|200906|andy@hexten.net|Data-Compare|1.2101|i686-linux|5.6.2|linux|2.6.29.4-167.fc11.i686.pae|200906191712
4217825|fail|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.6.2|linux|2.6.29.4-167.fc11.i686.pae|200906191534
4217824|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.6.2|linux|2.6.29.4-167.fc11.i686.pae|200906191534
4216887|pass|200906|stro@cpan.org|CPAN-FindDependencies|2.22|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200906191444
4216871|pass|200906|stro@cpan.org|CPAN-FindDependencies|2.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200906191442
4216790|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd|5.8.9|netbsd|5.0|200906191232
4216500|fail|200906|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906190702
4215435|pass|200906|martin.evans@easysoft.com|Tie-STDOUT|1.0401|i686-linux|5.10.0|linux|2.6.24-24-generic|200906190910
4215243|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200906190829
4214392|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.8.7|darwin|8.11.0|200906190539
4213783|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd|5.8.9|netbsd|5.0|200906190335
4213195|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.8.7|darwin|8.11.0|200906190151
4212534|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200906190108
4211979|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200906182139
4211772|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200906182153
4211768|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200906182152
4211659|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200906182130
4211656|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200906182130
4211385|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd|5.8.9|netbsd|5.0|200906181939
4210731|pass|200906|mithaldu@yahoo.de ("Christian Walde")|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200906181307
4210565|pass|200906|mithaldu@yahoo.de ("Christian Walde")|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200906181256
4210156|fail|200906|mithaldu@yahoo.de ("Christian Walde")|CPAN-FindDependencies|2.32|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200906181229
4209545|pass|200906|martin.evans@easysoft.com|Palm-ProjectGutenberg|1.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200906181403
4209543|pass|200906|martin.evans@easysoft.com|XML-Tiny-DOM|1.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200906181403
4209542|pass|200906|martin.evans@easysoft.com|XML-Tiny|2.02|i686-linux|5.10.0|linux|2.6.24-24-generic|200906181403
4209008|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i386-linux-thread-multi|5.10.0|linux|2.6.29.4-167.fc11.i686.pae|200906181411
4209007|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i386-linux-thread-multi|5.10.0|linux|2.6.29.4-167.fc11.i686.pae|200906181411
4207025|pass|200906|martin.evans@easysoft.com|CPAN-ParseDistribution|1.1|i686-linux|5.10.0|linux|2.6.24-24-generic|200906180902
4205084|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-64int|5.10.0|netbsd|5.0|200906180329
4204623|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-64int-2level|5.8.9|darwin|8.11.0|200906180223
4203964|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200906180051
4203892|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906171949
4202826|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i686-linux|5.6.2|linux|2.6.20-16-generic|200906172052
4202822|pass|200906|andy@hexten.net|Data-Compare|1.2101|i686-linux|5.6.2|linux|2.6.20-16-generic|200906172052
4202508|fail|200906|andy@hexten.net|Devel-CheckLib|0.6|i686-linux|5.6.2|linux|2.6.20-16-generic|200906172010
4202506|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i686-linux|5.6.2|linux|2.6.20-16-generic|200906172010
4200974|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200906171707
4200944|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-64int|5.10.0|netbsd|5.0|200906171656
4200571|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906171111
4200321|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906171043
4199443|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906170914
4199315|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906170905
4199185|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906170856
4198848|pass|200906|martin.evans@easysoft.com|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-24-generic|200906171337
4198265|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200906171347
4197989|pass|200906|brian@minton.name|CPAN-FindDependencies|2.32|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.29.4-dsa-dl380-oldxeon|200906161528
4197987|pass|200906|brian@minton.name|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.29.4-dsa-dl380-oldxeon|200906161527
4197984|pass|200906|brian@minton.name|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.29.4-dsa-dl380-oldxeon|200906161527
4197647|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-64int|5.10.0|netbsd|5.0|200906171123
4196352|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-64int-2level|5.8.9|darwin|8.11.0|200906170636
4196349|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-64int-2level|5.8.9|darwin|8.11.0|200906170635
4194696|pass|200906|chris@bingosnet.co.uk|Devel-CheckLib|0.6|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200906170044
4194641|pass|200906|chris@bingosnet.co.uk|Devel-CheckOS|1.61|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200906170041
4194633|pass|200906|chris@bingosnet.co.uk|Data-Compare|1.2101|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200906170040
4194549|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200906170030
4191794|pass|200906|martin.evans@easysoft.com|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.6.24-24-generic|200906161744
4191793|pass|200906|martin.evans@easysoft.com|Palm-Treo680MessagesDB|1.01|i686-linux|5.10.0|linux|2.6.24-24-generic|200906161745
4191789|pass|200906|martin.evans@easysoft.com|File-Overwrite|1.1|i686-linux|5.10.0|linux|2.6.24-24-generic|200906161744
4191562|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd|5.8.9|netbsd|5.0|200906161701
4189524|pass|200906|martin.evans@easysoft.com|Number-Phone|1.7002|i686-linux|5.10.0|linux|2.6.24-24-generic|200906161036
4189092|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906160950
4189089|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906160950
4188769|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200906160909
4188764|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200906160908
4186841|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd|5.8.9|netbsd|5.0|200906160408
4186840|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd|5.8.9|netbsd|5.0|200906160407
4186711|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200906160453
4186273|pass|200906|justin.d.hunter@gmail.com|Devel-CheckOS|1.61|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200906160145
4186269|pass|200906|justin.d.hunter@gmail.com|Data-Compare|1.2101|i386-linux-thread-multi|5.8.8|linux|2.6.18-128.1.1.el5|200906160145
4184288|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.9|linux|2.6.28-11-generic|200906152323
4184285|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.9|linux|2.6.28-11-generic|200906152323
4184267|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.8|linux|2.6.28-11-generic|200906152312
4184195|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.8|linux|2.6.28-11-generic|200906152312
4182740|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-64int-2level|5.8.8|darwin|8.11.0|200906151940
4182364|pass|200906|gms8994@dp.cx|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200906150905
4181806|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.8|linux|2.6.28-11-generic|200906151742
4181803|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.8|linux|2.6.28-11-generic|200906151741
4181758|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.7|linux|2.6.28-11-generic|200906151731
4181712|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.7|linux|2.6.28-11-generic|200906151731
4180982|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.7|linux|2.6.28-11-generic|200906151602
4180979|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.7|linux|2.6.28-11-generic|200906151602
4179588|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.6|linux|2.6.28-11-generic|200906151315
4179587|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.6|linux|2.6.28-11-generic|200906151315
4177049|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd|5.10.0|netbsd|5.0|200906150541
4175751|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200906150113
4175728|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-64int-2level|5.8.8|darwin|8.11.0|200906150113
4175727|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-64int-2level|5.8.8|darwin|8.11.0|200906150113
4174140|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.6|linux|2.6.28-11-generic|200906142158
4174138|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.6|linux|2.6.28-11-generic|200906142158
4174108|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.5|linux|2.6.28-11-generic|200906142155
4174106|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.5|linux|2.6.28-11-generic|200906142155
4173887|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.5|linux|2.6.28-11-generic|200906142135
4173885|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.5|linux|2.6.28-11-generic|200906142135
4173843|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.4|linux|2.6.28-11-generic|200906142131
4173841|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.4|linux|2.6.28-11-generic|200906142131
4173720|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.4|linux|2.6.28-11-generic|200906142059
4173699|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.3|linux|2.6.28-11-generic|200906142053
4173550|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.4|linux|2.6.28-11-generic|200906142059
4173510|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.3|linux|2.6.28-11-generic|200906142053
4173436|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.1|linux|2.6.28-11-generic|200906142000
4173343|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux|5.8.2|linux|2.6.28-11-generic|200906142025
4173275|fail|200906|stro@cpan.org|CPAN-FindDependencies|2.32|armv5tejl-linux|5.10.0|linux|2.6.17.14|200906142244
4173191|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd|5.10.0|netbsd|5.0|200906142033
4173179|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.3|linux|2.6.28-11-generic|200906142029
4173177|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd|5.10.0|netbsd|5.0|200906142032
4173175|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.3|linux|2.6.28-11-generic|200906142029
4173152|pass|200906|stro@cpan.org|Devel-CheckOS|1.61|armv5tejl-linux|5.10.0|linux|2.6.17.14|200906142234
4173110|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.2|linux|2.6.28-11-generic|200906142024
4172769|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux|5.8.2|linux|2.6.28-11-generic|200906142003
4172767|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux|5.8.2|linux|2.6.28-11-generic|200906142003
4172719|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux|5.8.1|linux|2.6.28-11-generic|200906142000
4171848|pass|200906|bernt.guldbrandtsen@agrsci.dk|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5|200906132206
4171633|pass|200906|bernt.guldbrandtsen@agrsci.dk|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5|200906132206
4170748|pass|200906|justin.d.hunter@gmail.com|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906131432
4170746|pass|200906|justin.d.hunter@gmail.com|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906131432
4170393|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200906132113
4170391|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200906132113
4165939|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200906130512
4165501|pass|200906|justin.d.hunter@gmail.com|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906130241
4165500|pass|200906|justin.d.hunter@gmail.com|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906130241
4164574|pass|200906|sscaffidi@cpan.org ("Stephen R. Scaffidi")|Devel-CheckOS|1.61|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906121503
4163792|pass|200906|sscaffidi@cpan.org ("Stephen R. Scaffidi")|CPAN-FindDependencies|2.32|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906121504
4163791|pass|200906|sscaffidi@cpan.org ("Stephen R. Scaffidi")|Data-Compare|1.2101|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200906121503
4163607|pass|200906|justin.d.hunter@gmail.com|DBIx-Class-SingletonRows|0.11|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906120211
4163504|pass|200906|justin.d.hunter@gmail.com|DBIx-Class-SingletonRows|0.11|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906111658
4163284|pass|200906|justin.d.hunter@gmail.com|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906110133
4162195|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200906121807
4162193|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200906121807
4161531|pass|200906|martin.evans@easysoft.com|CPAN-FindDependencies|2.32|i686-linux|5.10.0|linux|2.6.24-24-generic|200906121600
4161253|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200906121512
4160654|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200906121419
4159052|unknown|200906|stro@cpan.org|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906120700
4157546|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200906120255
4157545|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200906120255
4157429|pass|200906|stro@cpan.org|CPAN-FindDependencies|2.32|i686-linux|5.8.8|linux|2.4.27-2-386|200906120425
4156958|pass|200906|angryxpeh@gmail.com|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906120028
4153926|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200906111355
4152749|pass|200906|james@mastros.biz|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906110155
4152494|pass|200906|justin.d.hunter@gmail.com|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906110013
4151782|unknown|200906|stro@cpan.org|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.6.26-2-686|200906111236
4151723|pass|200906|stro@cpan.org|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906111231
4151430|pass|200906|stro@cpan.org|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906111127
4151238|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200906110554
4151237|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200906110554
4150797|pass|200906|stro@cpan.org|Data-Compare|1.2101|armv5tejl-linux|5.10.0|linux|2.6.17.14|200906110553
4149969|pass|200906|stro@rail.org.ua|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906110654
4149099|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200906110004
4146931|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-64int-2level|5.8.7|darwin|8.11.0|200906101644
4144065|pass|200906|stro@rail.org.ua|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.26-2-686|200906101046
4143371|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200906101014
4143368|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200906101014
4142536|pass|200906|angryxpeh@gmail.com|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.26-2-686|200906100822
4142260|pass|200906|angryxpeh@gmail.com|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906100755
4139866|pass|200906|ivan.wills@gmail.com ("Ivan Wills")|Sort-MultipleFields|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200906101123
4138539|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-64int-2level|5.8.7|darwin|8.11.0|200906100034
4138534|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-64int-2level|5.8.7|darwin|8.11.0|200906100034
4134936|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200906091246
4134928|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200906091245
4134173|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200906090949
4133554|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200906090743
4133350|pass|200906|stro@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.26-2-686|200906090720
4132738|pass|200906|stro@cpan.org|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906090455
4129916|pass|200906|bobtfisg@bobtfish.net|Data-Compare|1.2101|darwin-thread-multi-2level|5.8.6|darwin|8.0|200906071349
4129535|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-64int-2level|5.8.6|darwin|8.11.0|200906081842
4126379|pass|200906|angryxpeh@gmail.com|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906081931
4125947|pass|200906|angryxpeh@gmail.com|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906081908
4125945|pass|200906|angryxpeh@gmail.com|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.26-2-686|200906081908
4125821|pass|200906|martin.evans@easysoft.com|Devel-CheckLib|0.6|i686-linux|5.10.0|linux|2.6.24-24-generic|200906081405
4125580|pass|200906|martin.evans@easysoft.com|Devel-CheckOS|1.61|i686-linux|5.10.0|linux|2.6.24-24-generic|200906081352
4124974|pass|200906|stro@rail.org.ua|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906081810
4124973|pass|200906|stro@rail.org.ua|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.26-2-686|200906081810
4124934|pass|200906|stro@rail.org.ua|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.26-2-686|200906081806
4124786|pass|200906|stro@rail.org.ua|DBIx-Class-SingletonRows|0.11|i686-linux|5.10.0|linux|2.6.26-2-686|200906081751
4123425|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200906080709
4123367|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd-64int|5.8.9|midnightbsd|0.1.1-release|200906080459
4122125|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200906080251
4122034|pass|200906|stro@cpan.org|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906080700
4120629|pass|200906|rafl@debian.org|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.28.7|200906072154
4120363|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200906071951
4118608|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200906071549
4116550|pass|200906|stro@rail.org.ua|Palm-SMS|0.03|i686-linux|5.10.0|linux|2.6.26-2-686|200906071530
4116423|fail|200906|stro@rail.org.ua|Bryar|3.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906071520
4115758|pass|200906|angryxpeh@gmail.com|Class-DBI-ClassGenerator|1.02|i686-linux|5.10.0|linux|2.6.26-2-686|200906071434
4115696|pass|200906|angryxpeh@gmail.com|File-Find-Rule-Permissions|2.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906071428
4115341|pass|200906|angryxpeh@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906071344
4114381|pass|200906|stro@cpan.org|Tie-STDOUT|1.0401|i686-linux|5.10.0|linux|2.6.26-2-686|200906071139
4114361|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200906070754
4113884|pass|200906|stro@cpan.org|XML-Tiny-DOM|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906071008
4113882|pass|200906|stro@cpan.org|XML-Tiny|2.02|i686-linux|5.10.0|linux|2.6.26-2-686|200906071008
4113880|pass|200906|stro@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906071008
4113208|pass|200906|stro@cpan.org|CPAN-ParseDistribution|1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906070750
4112872|pass|200906|stro@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.26-2-686|200906070633
4112838|pass|200906|stro@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux|5.10.0|linux|2.6.26-2-686|200906070631
4112835|pass|200906|stro@cpan.org|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.6.26-2-686|200906070631
4112832|pass|200906|stro@cpan.org|File-Overwrite|1.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906070630
4112697|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-64int-2level|5.10.0|darwin|8.11.0|200906070221
4111202|pass|200906|stro@cpan.org|Number-Phone|1.7002|i686-linux|5.10.0|linux|2.6.26-2-686|200906070246
4110635|pass|200906|stro@cpan.org|CPAN-FindDependencies|2.32|i686-linux|5.10.0|linux|2.6.26-2-686|200906070126
4110126|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-64int-2level|5.10.0|darwin|8.11.0|200906061933
4109422|pass|200906|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906060833
4109420|pass|200906|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906060833
4108975|pass|200906|DCOLLINS@cpan.org|Net-Random|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906060830
4107842|pass|200906|stro@rail.org.ua|Devel-CheckLib|0.6|i686-linux|5.10.0|linux|2.6.26-2-686|200906062206
4107804|pass|200906|stro@rail.org.ua|Devel-CheckOS|1.61|i686-linux|5.10.0|linux|2.6.26-2-686|200906062205
4107433|pass|200906|stro@rail.org.ua|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.26-2-686|200906062144
4106719|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200906061043
4106121|fail|200906|stro@rail.org.ua|Bryar|3.1|i686-linux|5.10.0|linux|2.6.26-2-686|200906062026
4104755|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200906061401
4104632|pass|200906|andy@hexten.net|Data-Compare|1.2101|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200906061401
4103771|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200906061428
4103574|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200906061419
4103536|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-amd64|200906061417
4102520|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200906061251
4101912|pass|200906|angryxpeh@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200906061613
4101861|pass|200906|angryxpeh@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux|5.8.9|linux|2.6.26-2-686|200906061608
4101805|unknown|200906|angryxpeh@gmail.com|Acme-Pony|1.1.2|i686-linux|5.8.9|linux|2.6.26-2-686|200906061603
4101107|pass|200906|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060849
4100846|pass|200906|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060816
4100598|pass|200906|DCOLLINS@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906060750
4100537|pass|200906|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060743
4100288|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906060714
4099886|pass|200906|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060635
4099516|pass|200906|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060556
4099462|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200906060749
4099403|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200906060743
4099037|pass|200906|wiml@hhhh.org ("Wim Lewis")|Number-Phone|1.7002|powerpc-netbsd-thread-multi|5.10.0|netbsd|5.0_rc3|200906060448
4098588|pass|200906|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060357
4097751|pass|200906|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060230
4096721|pass|200906|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060052
4096130|pass|200906|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906060016
4096126|fail|200906|bingos@cpan.org|Bryar|3.1|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200906060113
4095378|unknown|200906|stro@rail.org.ua|Acme-Licence|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200906060527
4093529|pass|200906|stro@rail.org.ua|NestedMap|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200906060335
4092650|fail|200906|bingos@cpan.org|Bryar|3.1|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200906051644
4092419|pass|200906|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906051416
4092166|pass|200906|stro@rail.org.ua|Statistics-ChiSquare|0.5|i686-linux|5.8.9|linux|2.6.26-2-686|200906060217
4092162|pass|200906|stro@rail.org.ua|Tie-Hash-Longest|1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200906060217
4091090|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906051407
4091064|pass|200906|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906051407
4091042|pass|200906|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906051407
4090985|fail|200906|bingos@cpan.org|Bryar|3.1|darwin-64int-2level|5.10.0|darwin|8.11.0|200906052002
4090371|fail|200906|vpit@cpan.org|Bryar|3.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200906052021
4090056|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200906052004
4089807|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200906051842
4089438|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-midnightbsd-64int|5.10.0|midnightbsd|0.1.1-release|200906051250
4088634|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-64int-2level|5.10.0|darwin|8.11.0|200906051506
4088541|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200906051551
4087283|fail|200906|martin.evans@easysoft.com|Bryar|3.1|i686-linux|5.10.0|linux|2.6.24-24-generic|200906050919
4086975|pass|200906|stro@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200906050708
4086354|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200906050649
4085891|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny-DOM|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906050343
4085890|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-Tiny|2.02|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906050343
4085888|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-ProjectGutenberg|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906050343
4085850|pass|200906|DCOLLINS@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906050339
4085845|pass|200906|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906050339
4084891|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200906050336
4084690|pass|200906|DCOLLINS@cpan.org|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906050116
4084661|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906050114
4084374|fail|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-ParseDistribution|1.1|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906050048
4083492|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906041929
4083445|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200906050125
4083393|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906041403
4083377|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906041403
4082901|pass|200906|bingos@cpan.org|Bryar|3.1|darwin-2level|5.8.9|darwin|8.11.0|200906042354
4082647|pass|200906|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906042241
4080727|pass|200906|angryxpeh@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200906042045
4080298|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906041405
4080268|pass|200906|DCOLLINS@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906041405
4080265|pass|200906|DCOLLINS@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906041405
4080225|pass|200906|DCOLLINS@cpan.org|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906041404
4080151|pass|200906|DCOLLINS@cpan.org|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906041953
4080044|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906041403
4080042|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906041403
4080041|pass|200906|DCOLLINS@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906041402
4079528|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200906042054
4079101|pass|200906|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906040716
4079026|pass|200906|srezic@cpan.org|Bryar|3.1|amd64-freebsd|5.6.2|freebsd|7.0-release|200906042009
4078890|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200906041958
4076907|pass|200906|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906040716
4076906|pass|200906|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906040716
4076839|pass|200906|martin.evans@easysoft.com|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.24-24-generic|200906041547
4076817|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906041047
4076813|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906041046
4076787|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906041044
4076574|fail|200906|srezic@cpan.org|Devel-CheckLib|0.6|amd64-freebsd|5.6.2|freebsd|7.0-release|200906041624
4075624|pass|200906|bingos@cpan.org|Bryar|3.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200906041417
4074785|pass|200906|cpan@greerga.m-l.org ("George Greer")|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906040813
4074681|pass|200906|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1257-g2b4f847|linux|2.6.30-020630rc7-generic|200906040645
4074668|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-64int|5.8.8|linux|2.6.28-11-generic|200906041305
4074628|pass|200906|DCOLLINS@cpan.org|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906040645
4074158|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200906041324
4074002|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200906041313
4072917|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int|5.8.8|linux|2.6.28-11-generic|200906041032
4071950|pass|200906|cpan@greerga.m-l.org ("George Greer")|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906040312
4071947|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Hexdumper|2.01|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906040311
4071944|pass|200906|cpan@greerga.m-l.org ("George Greer")|File-Overwrite|1.1|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906040310
4071435|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200906040758
4071178|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200906040705
4071032|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200906040634
4070241|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200906040236
4070081|pass|200906|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906040114
4070061|pass|200906|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906040110
4069762|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200906040227
4069722|pass|200906|cpan@greerga.m-l.org ("George Greer")|Number-Phone|1.7002|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906040023
4068240|pass|200906|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906032234
4067174|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200906032307
4066595|pass|200906|bingos@cpan.org|Bryar|3.1|darwin-2level|5.8.8|darwin|8.11.0|200906032107
4066473|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200906032149
4066091|pass|200906|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906031901
4065744|fail|200906|bingos@cpan.org|Bryar|3.1|i686-linux-thread-multi-64int|5.8.8|linux|2.6.28-11-generic|200906031928
4065594|pass|200906|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030642
4065150|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200906031940
4064360|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200906031828
4063658|pass|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906031132
4063532|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int|5.8.8|linux|2.6.28-11-generic|200906031623
4063069|pass|200906|bminton@integ.com (Brian Minton)|CPAN-ParseDistribution|1.1|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200906031051
4062046|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200906031532
4061140|fail|200906|cpan@greerga.m-l.org ("George Greer")|CPAN-FindDependencies|2.32|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200906030820
4060950|fail|200906|bingos@cpan.org|Bryar|3.1|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200906031307
4060816|fail|200906|stsc@refcnt.org ("Steven Schubiger")|Bryar|3.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200906031359
4058877|pass|200906|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906030939
4058853|pass|200906|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906030937
4058848|pass|200906|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906030937
4058071|pass|200906|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906030840
4057732|pass|200906|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030815
4057256|pass|200906|bingos@cpan.org|Data-Compare|1.2101|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200906030846
4056092|pass|200906|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906030633
4055782|pass|200906|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906030612
4054862|pass|200906|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030505
4054528|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906030033
4054486|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-64int|5.8.9|linux|2.6.28-11-generic|200906030530
4054443|pass|200906|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906030426
4054379|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906030020
4052705|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200906030406
4052611|pass|200906|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030159
4052574|fail|200906|stro@cpan.org|Bryar|3.1|armv5tejl-linux|5.10.0|linux|2.6.17.14|200906030456
4052550|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200906030354
4052539|pass|200906|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906030153
4052442|pass|200906|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906030146
4052398|pass|200906|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906030143
4052221|pass|200906|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030131
4052180|pass|200906|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030127
4051840|pass|200906|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906030104
4051813|pass|200906|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030102
4051751|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int|5.8.9|linux|2.6.28-11-generic|200906030159
4051725|pass|200906|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906030058
4051694|pass|200906|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906030055
4051133|pass|200906|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906030017
4050996|pass|200906|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906030008
4050880|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906022358
4050439|pass|200906|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906021425
4050276|pass|200906|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906021425
4050204|pass|200906|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906021402
4050196|pass|200906|stro@rail.org.ua|Class-CanBeA|1.2|i686-linux|5.8.9|linux|2.6.26-2-686|200906021245
4049989|pass|200906|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906021407
4049532|pass|200906|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906021422
4049386|pass|200906|bingos@cpan.org|Bryar|3.1|darwin-2level|5.8.6|darwin|8.11.0|200906022347
4048937|pass|200906|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906021423
4048934|pass|200906|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906021423
4048820|pass|200906|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906021422
4048819|pass|200906|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906021422
4048772|fail|200906|srezic@cpan.org|Bryar|3.1|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200906022339
4047991|pass|200906|schwern@pobox.com|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.10.0|darwin|8.11.1|200906021057
4047985|pass|200906|schwern@pobox.com|Data-Compare|1.2101|darwin-thread-multi-2level|5.10.0|darwin|8.11.1|200906021052
4047073|pass|200906|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906021407
4047069|pass|200906|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906021407
4046771|pass|200906|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906021404
4046528|pass|200906|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906021402
4046525|pass|200906|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906021402
4046395|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200906022048
4045378|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020627
4045160|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200906021913
4045066|pass|200906|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020645
4045049|pass|200906|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020645
4045039|pass|200906|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020645
4045037|pass|200906|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906020645
4045029|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200906021900
4044969|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200906021856
4044891|pass|200906|srezic@cpan.org|Devel-CheckLib|0.6|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200906021845
4044861|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200906021846
4044676|pass|200906|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020638
4044666|pass|200906|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020638
4044564|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200906021821
4044562|pass|200906|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906020636
4044558|pass|200906|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020636
4043772|pass|200906|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020625
4043521|pass|200906|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020621
4043513|pass|200906|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020621
4043511|pass|200906|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020621
4043001|pass|200906|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906020645
4043000|pass|200906|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020645
4042999|pass|200906|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020645
4042981|pass|200906|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020645
4042977|pass|200906|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020645
4042619|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200906021607
4042573|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200906021605
4042410|fail|200906|bingos@cpan.org|Bryar|3.1|i686-linux-thread-multi-64int|5.8.9|linux|2.6.28-11-generic|200906021456
4042337|fail|200906|bingos@cpan.org|Bryar|3.1|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200906021450
4042251|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200906021544
4042018|pass|200906|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020638
4042015|pass|200906|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906020638
4041728|pass|200906|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020636
4041724|pass|200906|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020636
4040616|pass|200906|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020629
4040612|pass|200906|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020628
4040315|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020627
4040314|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020627
4040313|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906020627
4040188|pass|200906|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906020625
4040187|pass|200906|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020625
4039672|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int|5.8.9|linux|2.6.28-11-generic|200906021204
4039593|pass|200906|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020621
4039591|pass|200906|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906020621
4039570|pass|200906|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200906020621
4038411|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.10.0|linux|2.6.18-92.el5|200906021106
4038406|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200906021105
4038098|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200906021009
4037766|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200906020904
4036866|fail|200906|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906020106
4036502|pass|200906|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200906020008
4035010|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200906020242
4034978|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200906020341
4034683|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906020120
4034656|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906020119
4033841|pass|200906|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200906020120
4033822|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200906020219
4033613|fail|200906|DCOLLINS@cpan.org|Bryar|3.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906020005
4033246|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200906020138
4033048|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200906020123
4033019|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200906020121
4032457|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200906020038
4032230|fail|200906|DCOLLINS@cpan.org|Bryar|3.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906011807
4032104|pass|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200906020014
4030818|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200906012308
4030498|pass|200906|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906011455
4030253|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200906012221
4029830|pass|200906|angryxpeh@gmail.com|Sub-WrapPackages|1.2|i686-linux|5.8.9|linux|2.6.26-2-686|200906012252
4029816|fail|200906|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.1|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200906012151
4029140|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200906011457
4029138|pass|200906|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906011457
4029091|pass|200906|angryxpeh@gmail.com|Games-Dice-Advanced|1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200906012212
4028841|pass|200906|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200906011455
4028316|fail|200906|DCOLLINS@cpan.org|Bryar|3.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906011344
4027797|fail|200906|DCOLLINS@cpan.org|Bryar|3.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906011344
4027417|fail|200906|bingos@cpan.org|Bryar|3.1|darwin-2level|5.10.0|darwin|8.11.0|200906011750
4026013|fail|200906|bingos@cpan.org|Bryar|3.1|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200906011553
4025799|pass|200906|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906010949
4025790|pass|200906|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200906010949
4025448|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-686|200906011521
4025434|pass|200906|andy@hexten.net|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-2-686|200906011520
4025027|pass|200906|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906010949
4025018|pass|200906|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200906010949
4024227|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200906011339
4023836|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200906011254
4023828|pass|200906|andy@hexten.net|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200906011254
4023447|pass|200906|andy@hexten.net|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906011141
4023444|pass|200906|andy@hexten.net|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200906011140
4023338|pass|200906|bingos@cpan.org|Devel-CheckLib|0.6|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200906011125
4023333|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200906011124
4023320|pass|200906|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int|5.10.0|linux|2.6.28-11-generic|200906011123
4022258|pass|200906|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200906010825
4022257|pass|200906|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200906010824
4020468|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906010011
4020384|pass|200906|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200906010001
4020116|pass|200906|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200906010020
4019975|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905311818
4019880|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905311818
4019613|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905311818
4019599|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905311818
4019337|pass|200906|ivan.wills@gmail.com ("Ivan Wills")|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200906010755
4018635|pass|200905|andy@hexten.net|Devel-CheckOS|1.61|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200905312118
4018625|pass|200905|andy@hexten.net|Devel-CheckLib|0.6|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200905312118
4018415|pass|200905|andy@hexten.net|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200905312042
4018412|pass|200905|andy@hexten.net|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200905312041
4016104|pass|200905|stro@rail.org.ua|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.9|linux|2.6.26-2-686|200905311841
4015847|pass|200905|angryxpeh@gmail.com|Net-Random|2.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905311817
4015704|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200905311414
4015699|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200905311413
4014604|fail|200905|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200905310547
4014390|pass|200905|stro@cpan.org|Palm-SMS|0.03|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905310201
4014385|pass|200905|stro@cpan.org|Palm-SMS|0.03|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905310201
4014333|pass|200905|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200905310441
4013828|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd|5.10.0|freebsd|6.4-release|200905310832
4013821|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd|5.10.0|freebsd|6.4-release|200905310831
4013417|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905311643
4013380|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905311629
4012956|pass|200905|rbo@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.10.0|solaris|2.11|200905310735
4012729|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905310448
4012260|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd|5.10.0|freebsd|6.4-release|200905310442
4010784|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905310158
4010371|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905311033
4009714|pass|200905|stro@cpan.org|Class-DBI-ClassGenerator|1.02|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905301622
4009704|pass|200905|stro@cpan.org|Class-DBI-ClassGenerator|1.02|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905301622
4009385|pass|200905|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200905301830
4009351|pass|200905|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200905301827
4009342|pass|200905|cpan@greerga.m-l.org ("George Greer")|Devel-CheckLib|0.6|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200905301827
4009317|pass|200905|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200905301824
4009296|pass|200905|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200905301822
4009291|pass|200905|stro@cpan.org|File-Find-Rule-Permissions|2.0|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905301521
4009283|pass|200905|cpan@greerga.m-l.org ("George Greer")|Devel-CheckOS|1.61|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200905301821
4009276|pass|200905|stro@cpan.org|File-Find-Rule-Permissions|2.0|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905301521
4008247|fail|200905|mw487.yahoo.com@gmail.com|Bryar|3.1|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200905301541
4006119|fail|200905|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200905301333
4005957|pass|200905|stro@cpan.org|WWW-Facebook-Go-SGF|1.0|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905301042
4005950|pass|200905|stro@cpan.org|WWW-Facebook-Go-SGF|1.0|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905301042
4005881|fail|200905|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200905301340
4005777|fail|200905|cpan@greerga.m-l.org ("George Greer")|Bryar|3.1|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200905301333
4005304|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Tie-STDOUT|1.0401|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905310250
4003268|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905310011
4003265|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905310011
4003256|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905310010
4002899|fail|200905|srezic@cpan.org|Bryar|3.1|amd64-freebsd|5.10.0|freebsd|7.0-release|200905301546
4002805|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905302344
4001649|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Number-Phone|1.7002|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905302148
4001125|pass|200905|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.11.0|linux|2.6.28-11-generic|200905300137
4001072|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|CPAN-FindDependencies|2.32|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905301917
4000627|pass|200905|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.28-11-generic|200905300226
4000121|pass|200905|cpan@greerga.m-l.org ("George Greer")|Data-Compare|1.2101|x86_64-linux|5.8.9 patch 35104|linux|2.6.28-11-generic|200905300232
3999037|pass|200905|bingos@cpan.org|Bryar|3.1|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200905300555
3998904|fail|200905|ivan.wills@gmail.com ("Ivan Wills")|Bryar|3.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905301428
3998783|fail|200905|ivan.wills@gmail.com ("Ivan Wills")|Bryar|3.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905301407
3997908|fail|200905|bingos@cpan.org|Bryar|3.1|i386-freebsd|5.10.0|freebsd|6.4-release|200905300219
3997184|fail|200905|david@cantrell.org.uk|Bryar|3.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200905291712
3997101|fail|200905|david@cantrell.org.uk|Bryar|3.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200905292253
3997099|fail|200905|david@cantrell.org.uk|Bryar|3.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200905292253
3997096|fail|200905|david@cantrell.org.uk|Bryar|3.1|alpha-netbsd|5.10.0|netbsd|4.0.1|200905292153
3997084|fail|200905|david@cantrell.org.uk|Bryar|3.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200905292253
3996596|fail|200905|david@cantrell.org.uk|Bryar|3.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200905292119
3996341|pass|200905|bingos@cpan.org|Bryar|3.1|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200905292019
3996165|fail|200905|david@cantrell.org.uk|Bryar|3.1|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200905291955
3996155|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905300447
3995917|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905300350
3995915|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905300349
3995908|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905300347
3994987|fail|200905|imacat@mail.imacat.idv.tw|Bryar|3.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905292159
3994722|fail|200905|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200905291506
3994595|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200905291356
3994590|pass|200905|stro@rail.org.ua|CPU-Emulator-Z80|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905291349
3994589|pass|200905|stro@rail.org.ua|CPU-Emulator-Memory|1.1001|i686-linux|5.8.9|linux|2.6.26-2-686|200905291349
3994539|pass|200905|stro@rail.org.ua|Data-Transactional|1.02|i686-linux|5.8.9|linux|2.6.26-2-686|200905291345
3994449|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200905291345
3994359|pass|200905|stro@rail.org.ua|DBIx-Class-SingletonRows|0.11|i686-linux|5.8.9|linux|2.6.26-2-686|200905291332
3993797|pass|200905|angryxpeh@gmail.com|Sort-MultipleFields|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905291225
3993559|fail|200905|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200905291309
3992912|fail|200905|stro@cpan.org|Bryar|3.1|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905290241
3992757|pass|200905|stro@cpan.org|Bryar|3.1|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905290228
3991926|pass|200905|srezic@cpan.org|Bryar|3.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200905290756
3991834|pass|200905|srezic@cpan.org|Bryar|3.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200905290756
3991583|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200905290557
3991253|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|x86_64-linux-thread-multi-ld|5.8.7|linux|2.6.26-1-amd64|200905290549
3991246|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|x86_64-linux|5.8.9|linux|2.6.26-1-amd64|200905290549
3991126|fail|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200905290549
3991125|fail|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200905290549
3990807|fail|200905|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Bryar|3.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200905291154
3990082|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-2level|5.8.9|darwin|8.11.0|200905290057
3990031|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200905290037
3989574|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200905282356
3989572|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200905282356
3989571|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200905282355
3989568|fail|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200905282355
3989560|fail|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.1|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200905282354
3989559|cpan|200905|DCANTRELL|Bryar|3.1||0|||200905282352
3988417|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200905281918
3988264|pass|200905|stro@cpan.org|Devel-CheckLib|0.6|i686-linux|5.8.8|linux|2.4.27-2-386|200905282049
3987353|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200905281706
3985970|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Palm-SMS|0.03|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905282335
3985825|pass|200905|stro@rail.org.ua|Palm-SMS|0.03|i686-linux|5.8.9|linux|2.6.26-2-686|200905281851
3985574|fail|200905|stro@rail.org.ua|Bryar|3.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905281833
3984915|fail|200905|ivan.wills@gmail.com ("Ivan Wills")|Bryar|3.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905282232
3984562|pass|200905|stro@rail.org.ua|Class-DBI-ClassGenerator|1.02|i686-linux|5.8.9|linux|2.6.26-2-686|200905281742
3984548|pass|200905|stro@rail.org.ua|File-Find-Rule-Permissions|2.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905281742
3984149|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckLib|0.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905282153
3984043|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckOS|1.61|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905282148
3983946|pass|200905|angryxpeh@gmail.com|WWW-Facebook-Go-SGF|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905281707
3983422|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200905281215
3983344|pass|200905|angryxpeh@gmail.com|Tie-STDOUT|1.0401|i686-linux|5.8.9|linux|2.6.26-2-686|200905281634
3983205|pass|200905|angryxpeh@gmail.com|Number-Phone|1.7002|i686-linux|5.8.9|linux|2.6.26-2-686|200905281629
3983062|pass|200905|stro@cpan.org|Tie-STDOUT|1.0401|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905280359
3982598|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Data-Compare|1.2101|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.28-11-generic|200905282034
3981894|pass|200905|stro@cpan.org|Tie-STDOUT|1.0401|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905280049
3979873|pass|200905|stro@rail.org.ua|Devel-CheckOS|1.61|i686-linux|5.8.9|linux|2.6.26-2-686|200905271507
3979302|pass|200905|stro@rail.org.ua|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905271700
3979028|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-2level|5.8.8|darwin|8.11.0|200905272138
3979021|pass|200905|stro@rail.org.ua|XML-Tiny-DOM|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905271734
3979020|pass|200905|stro@rail.org.ua|XML-Tiny|2.02|i686-linux|5.8.9|linux|2.6.26-2-686|200905271734
3979017|pass|200905|stro@rail.org.ua|Palm-ProjectGutenberg|1.0|i686-linux|5.8.9|linux|2.6.26-2-686|200905271734
3978983|pass|200905|stro@rail.org.ua|Palm-Treo680MessagesDB|1.01|i686-linux|5.8.9|linux|2.6.26-2-686|200905271656
3978981|pass|200905|stro@rail.org.ua|Data-Hexdumper|2.01|i686-linux|5.8.9|linux|2.6.26-2-686|200905271656
3978979|pass|200905|stro@rail.org.ua|File-Overwrite|1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200905271655
3977908|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200905271951
3976401|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200905270931
3976253|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200905271657
3975848|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200905271536
3975401|pass|200905|stro@cpan.org|CPAN-ParseDistribution|1.1|i686-linux|5.8.9|linux|2.6.26-2-686|200905271024
3975400|pass|200905|stro@cpan.org|CPAN-FindDependencies|2.32|i686-linux|5.8.9|linux|2.6.26-2-686|200905271024
3975338|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200905271406
3975011|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200905271212
3974999|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200905271309
3974716|pass|200905|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905270628
3974512|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200905271220
3974511|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200905271220
3973221|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.8.8|darwin|8.11.0|200905270710
3972414|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200905270243
3971991|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200905270327
3971674|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.8.8|darwin|8.11.0|200905270115
3971076|pass|200905|stro@cpan.org|XML-Tiny-DOM|1.0|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261550
3971075|pass|200905|stro@cpan.org|XML-Tiny|2.02|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261550
3971071|pass|200905|stro@cpan.org|Palm-ProjectGutenberg|1.0|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261550
3970938|pass|200905|stro@cpan.org|XML-Tiny-DOM|1.0|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261541
3970936|pass|200905|stro@cpan.org|XML-Tiny|2.02|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261541
3970934|pass|200905|stro@cpan.org|Palm-ProjectGutenberg|1.0|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261541
3970785|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200905261800
3970201|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-2level|5.8.8|darwin|8.11.0|200905262216
3970073|pass|200905|andy@hexten.net|XML-Tiny-DOM|1.0|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200905262208
3969871|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200905261614
3968515|pass|200905|stro@cpan.org|Data-Compare|1.2101|i686-linux|5.8.9|linux|2.6.26-2-686|200905262307
3968092|pass|200905|stro@cpan.org|CPAN-ParseDistribution|1.1|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261025
3968082|pass|200905|stro@cpan.org|CPAN-ParseDistribution|1.1|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905261024
3966727|pass|200905|stro@cpan.org|Devel-CheckLib|0.6|i686-linux|5.8.9|linux|2.6.26-2-686|200905261824
3965875|pass|200905|srezic@cpan.org|Devel-CheckLib|0.6|amd64-freebsd|5.10.0|freebsd|7.0-release|200905261205
3962550|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-2level|5.8.6|darwin|8.11.0|200905252356
3960539|pass|200905|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905251513
3960390|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200905251912
3960221|pass|200905|vpit@cpan.org|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200905252113
3959198|pass|200905|andy@hexten.net|Tie-STDOUT|1.0401|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200905251349
3958767|pass|200905|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.6|i86pc-solaris-64int|5.8.9|solaris|2.11|200905251230
3958478|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200905251218
3958335|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200905251150
3958219|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200905251120
3958002|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200905251043
3957957|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.8.6|darwin|8.11.0|200905250930
3957939|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200905251031
3957715|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200905250823
3957520|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200905250916
3957235|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.10.0|linux|2.6.18-92.el5|200905250831
3956252|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200905250305
3956048|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.8.6|darwin|8.11.0|200905250225
3955933|pass|200905|stro@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905250101
3955909|pass|200905|stro@cpan.org|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905250057
3955907|pass|200905|stro@cpan.org|Data-Hexdumper|2.01|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905250057
3955905|pass|200905|stro@cpan.org|File-Overwrite|1.1|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905250057
3955619|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200905250031
3955185|pass|200905|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.6|i86pc-solaris-64int|5.10.0|solaris|2.11|200905250042
3954699|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-2level|5.8.6|darwin|8.11.0|200905242217
3954159|pass|200905|stro@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905241234
3954111|pass|200905|stro@cpan.org|Palm-Treo680MessagesDB|1.01|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905241228
3954103|pass|200905|stro@cpan.org|Data-Hexdumper|2.01|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905241228
3954098|pass|200905|stro@cpan.org|File-Overwrite|1.1|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905241227
3953938|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200905242105
3953754|pass|200905|stro@cpan.org|Number-Phone|1.7002|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905241138
3953722|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200905242032
3953672|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200905242024
3953486|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200905242001
3953266|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200905241909
3953078|pass|200905|stro@cpan.org|Devel-CheckLib|0.6|armv5tejl-linux|5.10.0|linux|2.6.17.14|200905241945
3952511|pass|200905|mw487.yahoo.com@gmail.com|Devel-CheckLib|0.6|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200905240944
3952453|pass|200905|stro@cpan.org|Devel-CheckOS|1.61|i686-linux|5.10.0|linux|2.4.27-2-386|200905240701
3952344|pass|200905|stro@cpan.org|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.4.27-2-386|200905240636
3952328|pass|200905|stro@cpan.org|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.4.27-2-386|200905240628
3952325|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200905241455
3951877|pass|200905|stro@cpan.org|Number-Phone|1.7002|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905240427
3950687|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200905240426
3950486|pass|200905|DCOLLINS@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905240223
3950261|pass|200905|DCOLLINS@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905240142
3950088|pass|200905|wiml@hhhh.org ("Wim Lewis")|Devel-CheckLib|0.6|powerpc-netbsd-thread-multi|5.10.0|netbsd|5.0_rc3|200905240126
3949982|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200905240207
3949841|pass|200905|DCOLLINS@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905240102
3948813|pass|200905|andy@hexten.net|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200905232227
3948791|pass|200905|andy@hexten.net|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200905232224
3948401|pass|200905|stro@cpan.org|Palm-SMS|0.03|i686-linux|5.8.8|linux|2.4.27-2-386|200905232253
3947767|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-2level|5.10.0|darwin|8.11.0|200905231819
3947614|pass|200905|stro@cpan.org|File-Overwrite|1.1|i686-linux|5.8.8|linux|2.4.27-2-386|200905231954
3947452|pass|200905|stro@cpan.org|Devel-CheckOS|1.61|i686-linux|5.8.8|linux|2.4.27-2-386|200905231922
3947217|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200905231618
3946896|pass|200905|rbo@cpan.org|Devel-CheckLib|0.6|i86pc-solaris|5.8.9|solaris|2.11|200905231646
3945825|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Class-DBI-ClassGenerator|1.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905232042
3945760|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|File-Find-Rule-Permissions|2.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905232013
3944815|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|WWW-Facebook-Go-SGF|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905231451
3944728|pass|200905|DCOLLINS@cpan.org|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905230433
3944102|pass|200905|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905230302
3943144|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.10.0|darwin|8.11.0|200905230040
3942342|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Tie-STDOUT|1.0401|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905230653
3942045|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Games-Dice-Advanced|1.1|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200905222137
3941779|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200905221923
3941642|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny-DOM|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905230358
3941641|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|XML-Tiny|2.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905230358
3941639|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Palm-ProjectGutenberg|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905230357
3941189|pass|200905|mw487.yahoo.com@gmail.com|Devel-CheckLib|0.6|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200905221145
3941123|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.10.0|darwin|8.11.0|200905221735
3940204|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|CPAN-ParseDistribution|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905230027
3939738|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200905221508
3939622|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200905221448
3939563|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-2level|5.10.0|darwin|8.11.0|200905221335
3939522|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200905221434
3939243|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200905221354
3939137|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200905221335
3938864|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200905221259
3938461|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|XML-DoubleEncodedEntities|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905222024
3938458|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200905221223
3938262|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Palm-Treo680MessagesDB|1.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905221946
3938256|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905221945
3938234|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|File-Overwrite|1.1|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905221943
3938115|fail|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.6|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200905221125
3935179|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|alpha-netbsd|5.10.0|netbsd|4.0.1|200905220006
3934778|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Number-Phone|1.7002|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905220810
3933643|pass|200905|andy@hexten.net|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200905211904
3933614|pass|200905|andy@hexten.net|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200905211900
3933523|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|CPAN-FindDependencies|2.32|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905220344
3932445|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200905210919
3932334|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|IP35-irix-64int|5.8.8|irix|6.5|200905211604
3931632|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200905211406
3931429|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200905211347
3931388|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200905211343
3931371|pass|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200905211342
3931297|fail|200905|david@cantrell.org.uk|Devel-CheckLib|0.6|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200905211335
3930716|pass|200905|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.6|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905211947
3927702|pass|200905|bingos@cpan.org|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905210009
3927655|pass|200905|chris@bingosnet.co.uk|Devel-CheckLib|0.6|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200905210001
3927602|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckLib|0.6|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905210845
3927537|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905210822
3927536|pass|200905|ivan.wills@gmail.com ("Ivan Wills")|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905210821
3927224|pass|200905|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckLib|0.6|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200905202240
3926349|fail|200905|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckLib|0.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200905210212
3926309|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|x86_64-linux-thread-multi-ld|5.8.7|linux|2.6.26-1-amd64|200905201855
3926301|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|x86_64-linux|5.8.9|linux|2.6.26-1-amd64|200905201855
3926260|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|x86_64-linux-thread-multi-ld|5.8.7|linux|2.6.26-1-amd64|200905201855
3926255|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|x86_64-linux|5.8.9|linux|2.6.26-1-amd64|200905201855
3926207|pass|200905|srezic@cpan.org|Devel-CheckLib|0.6|amd64-freebsd|5.8.9|freebsd|7.0-release|200905201843
3926195|pass|200905|srezic@cpan.org|Devel-CheckLib|0.6|amd64-freebsd|5.8.8|freebsd|7.0-release|200905201843
3926043|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200905201821
3925288|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200905201550
3925287|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200905201550
3925284|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200905201550
3925282|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200905201549
3925279|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200905201549
3925278|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|x86_64-linux|5.8.9|linux|2.6.26-1-amd64|200905201550
3925276|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|x86_64-linux|5.10.0|linux|2.6.26-1-amd64|200905201549
3925270|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200905201550
3925269|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|x86_64-linux-thread-multi-ld|5.8.7|linux|2.6.26-1-amd64|200905201550
3925267|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.6|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200905201549
3925252|cpan|200905|DCANTRELL|Devel-CheckLib|0.6||0|||200905201547
3923626|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200905201215
3922256|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200905200604
3918395|pass|200905|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905190739
3915712|pass|200905|greg@ie-entertainment.com|Number-Phone|1.7002|i386-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5|200905190111
3914278|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200905182139
3912993|pass|200905|srezic@cpan.org|Data-Compare|1.2101|amd64-freebsd|5.10.0|freebsd|7.0-release|200905181900
3912977|fail|200905|stro@cpan.org|CPAN-FindDependencies|2.32|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905181004
3912968|fail|200905|stro@cpan.org|CPAN-FindDependencies|2.32|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905181003
3912774|pass|200905|stro@cpan.org|Devel-CheckLib|0.5|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905180937
3912678|pass|200905|stro@cpan.org|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905180926
3911090|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200905181353
3910597|pass|200905|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905180742
3910154|fail|200905|DCOLLINS@cpan.org|Bryar|3.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905180739
3907796|pass|200905|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905180339
3906972|fail|200905|DCOLLINS@cpan.org|Bryar|3.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905180205
3906970|fail|200905|DCOLLINS@cpan.org|Bryar|3.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905180205
3906054|fail|200905|DCOLLINS@cpan.org|Bryar|3.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905171815
3905794|pass|200905|ipenburg@xs4all.nl ("Roland van Ipenburg")|Devel-CheckOS|1.61|darwin-2level|5.8.8|darwin|8.11.0|200905172325
3905792|pass|200905|ipenburg@xs4all.nl ("Roland van Ipenburg")|Data-Compare|1.2101|darwin-2level|5.8.8|darwin|8.11.0|200905172324
3905748|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-2level|5.8.8|darwin|8.11.0|200905172206
3903814|pass|200905|ipenburg@xs4all.nl ("Roland van Ipenburg")|Devel-CheckOS|1.61|darwin-2level|5.10.0|darwin|8.11.0|200905171913
3903810|pass|200905|ipenburg@xs4all.nl ("Roland van Ipenburg")|Data-Compare|1.2101|darwin-2level|5.10.0|darwin|8.11.0|200905171913
3902754|pass|200905|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905170853
3902510|pass|200905|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905170849
3901736|pass|200905|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905170850
3901568|pass|200905|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905170849
3899744|pass|200905|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905170533
3899616|pass|200905|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905170423
3899597|pass|200905|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905170418
3899385|pass|200905|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905170340
3899239|pass|200905|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905170323
3899197|pass|200905|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905170318
3896885|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-2level|5.8.8|darwin|8.11.0|200905161810
3896788|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200905161745
3895659|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-2level|5.8.8|darwin|8.11.0|200905161212
3895555|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200905161150
3893785|pass|200905|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905160411
3893393|pass|200905|ipenburg@xs4all.nl ("Roland van Ipenburg")|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.8.6|darwin|8.0|200905160516
3893392|pass|200905|ipenburg@xs4all.nl ("Roland van Ipenburg")|Data-Compare|1.2101|darwin-thread-multi-2level|5.8.6|darwin|8.0|200905160516
3893299|pass|200905|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905160258
3892187|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905152243
3892171|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905152241
3892169|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.24-19-server|200905152241
3890319|pass|200905|stro@cpan.org|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.24.5-xeon-aufs20081006-grsec|200905150826
3890023|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200905151545
3889160|pass|200905|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905150638
3888996|fail|200905|wiml@hhhh.org ("Wim Lewis")|CPAN-FindDependencies|2.32|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200905150235
3888600|pass|200905|wiml@hhhh.org ("Wim Lewis")|Devel-CheckLib|0.5|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200905150204
3888540|pass|200905|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905150638
3888062|pass|200905|DCOLLINS@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905150635
3887899|pass|200905|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905150638
3887855|pass|200905|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905150638
3887713|pass|200905|DCOLLINS@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905150637
3887461|pass|200905|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905150634
3885426|pass|200905|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905150423
3885227|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200905150541
3884008|pass|200905|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905150033
3883995|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-2level|5.10.0|darwin|8.11.0|200905150130
3883692|pass|200905|DCOLLINS@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905141852
3883539|pass|200905|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905141855
3883537|pass|200905|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905141855
3883535|pass|200905|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905141855
3883501|pass|200905|DCOLLINS@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905141855
3883482|pass|200905|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905141855
3883335|pass|200905|DCOLLINS@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905141852
3883140|pass|200905|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905141850
3882980|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200905150019
3882267|pass|200905|stro@cpan.org|Devel-CheckOS|1.61|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905141320
3882265|pass|200905|stro@cpan.org|Data-Compare|1.2101|x86_64-linux|5.8.9|linux|2.6.24.5-xeon-aufs20081006-grsec|200905141320
3881666|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200905142001
3880943|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200905141715
3880320|pass|200905|wiml@hhhh.org ("Wim Lewis")|Devel-CheckOS|1.61|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200905140409
3879203|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905141049
3878463|pass|200905|kamensky@cpan.org ("Moshe Kamensky")|Data-Compare|1.2101|sun4-solaris|5.10.0|solaris|2.8|200905131225
3877425|pass|200905|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905140442
3877416|pass|200905|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905140438
3877414|pass|200905|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905140438
3877362|pass|200905|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905140422
3877258|pass|200905|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905140411
3877257|pass|200905|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905140410
3877074|pass|200905|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905140352
3876981|pass|200905|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905140345
3876804|pass|200905|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905140331
3876779|pass|200905|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905140328
3876448|pass|200905|DCOLLINS@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905140250
3876210|pass|200905|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905140219
3876132|pass|200905|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905140210
3875271|pass|200905|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905131411
3874619|pass|200905|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905131408
3874469|pass|200905|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905131410
3873855|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200905131652
3873301|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200905131507
3872925|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200905131418
3872252|pass|200905|bingos@cpan.org|Data-Compare|1.2101|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200905131201
3871692|pass|200905|feldt@nhn.ou.edu|Data-Compare|1.2101|sun4-solaris-thread-multi|5.8.8|solaris|2.8|200905121226
3871670|pass|200905|carlos.arenas@gmail.com ("Carlos Arenas")|Data-Compare|1.2101|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200905121109
3871030|pass|200905|wiml@hhhh.org ("Wim Lewis")|Devel-CheckOS|1.61|powerpc-netbsd-thread-multi|5.10.0|netbsd|5.0_rc3|200905130330
3868567|pass|200905|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905121300
3868443|pass|200905|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905121243
3868361|pass|200905|DCOLLINS@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905121303
3868356|pass|200905|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905121303
3868034|pass|200905|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905121244
3867891|pass|200905|DCOLLINS@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905121303
3867833|pass|200905|DCOLLINS@cpan.org|Tie-STDOUT|1.0401|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905121302
3867688|pass|200905|DCOLLINS@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905121300
3867430|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200905121814
3867367|pass|200905|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905121244
3867298|pass|200905|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905121243
3866672|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200905121707
3866600|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200905121648
3866432|pass|200905|klaus+cpanplus@worstofall.com|XML-Tiny|2.02|x86_64-linux|5.10.0|linux|2.6.26-2-amd64|200905112307
3865970|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200905121503
3865931|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone-UK-DetailedLocations|1.3|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200905121445
3865415|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200905121106
3863142|pass|200905|DCOLLINS@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905120049
3862963|pass|200905|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905111703
3862352|pass|200905|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905111714
3862278|pass|200905|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905111712
3862234|pass|200905|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905111711
3862199|pass|200905|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905111710
3862002|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200905112201
3861384|pass|200905|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905111429
3861333|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200905112100
3861313|pass|200905|vpit@cpan.org|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905112059
3861309|pass|200905|DCOLLINS@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905111418
3860933|pass|200905|DCOLLINS@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905111419
3860902|pass|200905|DCOLLINS@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905111419
3860111|pass|200905|srezic@cpan.org|Devel-CheckLib|0.5|i686-linux|5.10.0 patch 32633|linux|2.6.18-4-486|200905111648
3860091|pass|200905|srezic@cpan.org|Data-Compare|1.2101|i686-linux|5.10.0 patch 32633|linux|2.6.18-4-486|200905111645
3858635|fail|200905|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905110649
3858048|pass|200905|wiml@hhhh.org ("Wim Lewis")|Data-Compare|1.2101|powerpc-netbsd-thread-multi|5.10.0|netbsd|5.0_rc3|200905110512
3855946|pass|200905|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905101904
3855879|pass|200905|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905101902
3855634|pass|200905|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905101856
3855569|pass|200905|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905101855
3855295|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-2level|5.8.6|darwin|8.11.0|200905102223
3854722|pass|200905|wiml@hhhh.org ("Wim Lewis")|Data-Compare|1.2101|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200905100830
3853671|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200905101237
3853665|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200905101236
3852262|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905100417
3852002|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905100338
3851774|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.11.0 patch GitLive-blead-1049-g5f1c709|linux|2.6.28-11-generic|200905100313
3851730|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi|5.9.5|linux|2.6.28-11-generic|200905100302
3850622|pass|200905|rafl@debian.org|Data-Compare|1.2101|i686-linux|5.10.0 patch 35117|linux|2.6.26|200905100223
3849637|pass|200905|chris@bingosnet.co.uk|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200905092050
3849492|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200905092012
3848749|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200905091809
3848606|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200905091744
3848503|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200905091727
3848116|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200905091624
3847962|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200905091613
3847897|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200905091607
3847642|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200905091531
3847567|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200905091522
3847214|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200905091431
3846786|pass|200905|cpan-testing@yorfeix.de ("yor_feix")|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905071553
3846660|pass|200905|rlane10@wi.rr.com|Data-Compare|1.2101|i386-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5xen|200905080024
3846601|fail|200905|rlane10@wi.rr.com|CPAN-FindDependencies|2.32|i386-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5xen|200905080144
3846475|pass|200905|pjacklam@cpan.org|Data-Compare|1.2101|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200905072052
3846231|pass|200905|Michael.Finken@rbs.com|Devel-CheckOS|1.61|sun4-solaris-thread-multi|5.8.9|solaris|2.8|200905071151
3846230|pass|200905|Michael.Finken@rbs.com|Data-Compare|1.2101|sun4-solaris-thread-multi|5.8.9|solaris|2.8|200905071143
3846132|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200905091301
3846042|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200905091245
3846040|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200905091244
3846010|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200905091236
3845972|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200905091229
3845937|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200905091221
3845896|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200905091211
3845871|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200905091203
3845692|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200905091127
3845653|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200905091119
3845610|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200905091108
3845155|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200905090843
3844392|pass|200905|DCOLLINS@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905090347
3844277|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200905090423
3844043|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-2level|5.8.6|darwin|8.11.0|200905090350
3844038|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-2level|5.8.6|darwin|8.11.0|200905090349
3842564|pass|200905|srezic@cpan.org|Data-Compare|1.2101|amd64-freebsd|5.6.2|freebsd|7.0-release|200905082323
3842518|pass|200905|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.6.2|freebsd|7.0-release|200905082312
3841304|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200905081846
3840803|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200905081721
3840307|pass|200905|perl@galumph.com|Data-Compare|1.2101|darwin-thread-multi-64int-ld-2level|5.10.0|darwin|9.4.0|200905080844
3839555|pass|200905|DCOLLINS@cpan.org|CPAN-FindDependencies|2.32|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905080756
3839415|pass|200905|DCOLLINS@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905080729
3837806|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905080432
3837805|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi|5.8.9|linux|2.6.28-11-generic|200905080432
3837613|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905080414
3837610|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi|5.8.8|linux|2.6.28-11-generic|200905080414
3837100|pass|200905|stro@cpan.org|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200905081029
3837093|pass|200905|rbo@cpan.org|Data-Compare|1.2101|i86pc-solaris|5.8.9|solaris|2.11|200905080929
3835855|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-2level|5.8.9|darwin|8.11.0|200905080609
3835831|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd|5.8.9|netbsd|5.0|200905080644
3835537|pass|200905|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.2101|i686-linux|5.10.0|linux|2.6.24-16-generic|200905080005
3834658|pass|200905|DCOLLINS@cpan.org|Devel-CheckOS|1.61|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905080151
3834657|pass|200905|DCOLLINS@cpan.org|Data-Compare|1.2101|i686-linux-thread-multi-ld|5.10.0|linux|2.6.28-11-generic|200905080150
3833917|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200905072351
3833142|pass|200905|imacat@mail.imacat.idv.tw|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080433
3833138|pass|200905|imacat@mail.imacat.idv.tw|XML-Tiny|2.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080433
3833134|pass|200905|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080432
3833131|pass|200905|imacat@mail.imacat.idv.tw|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080432
3833128|pass|200905|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080432
3833126|pass|200905|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.0401|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080431
3833120|pass|200905|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080431
3833115|pass|200905|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080430
3833111|pass|200905|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080430
3833106|pass|200905|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080430
3833101|pass|200905|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080429
3833099|pass|200905|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080429
3833087|pass|200905|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080429
3833062|pass|200905|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080427
3833049|pass|200905|imacat@mail.imacat.idv.tw|Palm-SMS|0.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080425
3833046|pass|200905|imacat@mail.imacat.idv.tw|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080424
3833043|pass|200905|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080424
3833038|pass|200905|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080423
3833026|pass|200905|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080421
3833024|pass|200905|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080421
3833022|pass|200905|imacat@mail.imacat.idv.tw|File-Overwrite|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080421
3833020|pass|200905|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080420
3833016|pass|200905|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080420
3833014|pass|200905|imacat@mail.imacat.idv.tw|Data-Hexdumper|2.01|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080419
3833012|pass|200905|imacat@mail.imacat.idv.tw|DBIx-Class-SingletonRows|0.11|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080419
3832998|pass|200905|imacat@mail.imacat.idv.tw|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080412
3832991|pass|200905|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080410
3832989|pass|200905|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080410
3832982|pass|200905|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080409
3832980|pass|200905|imacat@mail.imacat.idv.tw|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080408
3832977|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200905072207
3832976|fail|200905|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080408
3832972|pass|200905|imacat@mail.imacat.idv.tw|Data-Compare|1.2101|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080407
3832962|fail|200905|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080404
3832949|pass|200905|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080401
3832948|unknown|200905|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080400
3832947|unknown|200905|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905080400
3831525|pass|200905|andy@hexten.net|Data-Hexdumper|2.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905071459
3831510|pass|200905|andy@hexten.net|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905071456
3831250|unknown|200905|chris@bingosnet.co.uk|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905071349
3831175|pass|200905|chris@bingosnet.co.uk|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905071323
3831101|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd|5.8.9|netbsd|5.0|200905071346
3831100|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd|5.8.9|netbsd|5.0|200905071346
3830742|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200905071217
3830653|pass|200905|chris@bingosnet.co.uk|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905071034
3829982|pass|200905|srezic@cpan.org|CPAN-FindDependencies|2.32|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200905070854
3829152|pass|200905|mw487.yahoo.com@gmail.com|Data-Compare|1.2101|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200905070209
3828363|pass|200905|andy@hexten.net|Data-Compare|1.2101|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200905062244
3828353|pass|200905|andy@hexten.net|Data-Compare|1.2101|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200905062243
3828298|pass|200905|andy@hexten.net|Data-Compare|1.2101|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200905062236
3828188|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200905062306
3828148|pass|200905|stro@cpan.org|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200905070000
3828077|pass|200905|chris@bingosnet.co.uk|Data-Compare|1.2101|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200905062140
3828035|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-2level|5.8.9|darwin|8.11.0|200905062135
3828033|pass|200905|bingos@cpan.org|Data-Compare|1.2101|darwin-2level|5.8.9|darwin|8.11.0|200905062134
3827931|pass|200905|srezic@cpan.org|Data-Compare|1.2101|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200905062203
3827671|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200905062109
3827217|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|alpha-netbsd|5.10.0|netbsd|4.0.1|200905061724
3826563|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200905061023
3826546|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd|5.10.0|netbsd|5.0|200905061704
3826417|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200905061550
3826290|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200905061526
3826166|pass|200905|hohokus@gmail.com|Data-Compare|1.2101|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200905061004
3826152|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|IP35-irix-64int|5.8.8|irix|6.5|200905061602
3826074|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200905061449
3826072|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200905061449
3825962|pass|200905|vovkasm@gmail.com|Data-Compare|1.2101|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200905061552
3825562|pass|200905|g.grigelionis@computer.org|Data-Compare|1.2101|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200905061417
3825517|pass|200905|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.2101|i86pc-solaris-64int|5.8.9|solaris|2.11|200905061411
3825508|pass|200905|g.grigelionis@computer.org|Data-Compare|1.2101|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200905061410
3825486|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.10.0|linux|2.6.18-92.el5|200905061407
3825468|pass|200905|g.grigelionis@computer.org|Data-Compare|1.2101|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200905061406
3825446|pass|200905|g.grigelionis@computer.org|Data-Compare|1.2101|i586-linux-thread-multi|5.8.8|linux|2.6.16|200905061403
3825400|pass|200905|g.grigelionis@computer.org|Data-Compare|1.2101|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200905061358
3825318|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200905061344
3825074|pass|200905|chris@bingosnet.co.uk|Data-Compare|1.2101|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200905061128
3825050|pass|200905|david@cantrell.org.uk|Data-Compare|1.2101|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200905061123
3824621|pass|200905|bingos@cpan.org|Data-Compare|1.2101|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200905061002
3824544|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200905061050
3824283|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200905061003
3824202|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.2101|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200905060941
3824164|pass|200905|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.2101|i86pc-solaris-64int|5.10.0|solaris|2.11|200905060920
3823111|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-2level|5.10.0|darwin|8.11.0|200905060219
3822707|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.32|i386-netbsd|5.10.0|netbsd|5.0|200905060142
3822681|pass|200905|mw487.yahoo.com@gmail.com|Data-Compare|1.2101|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200905051856
3822495|pass|200905|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.2101|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905051922
3822281|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|i386-netbsd|5.10.0|netbsd|5.0|200905060019
3821991|pass|200905|dagolden@cpan.org (DAGOLDEN)|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905051747
3821935|pass|200905|dagolden@cpan.org (DAGOLDEN)|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905051735
3821934|pass|200905|dagolden@cpan.org (DAGOLDEN)|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905051735
3821796|pass|200905|bingos@cpan.org|Data-Compare|1.2101|i386-netbsd|5.10.0|netbsd|5.0|200905052259
3821794|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd|5.10.0|netbsd|5.0|200905052300
3821645|pass|200905|stsc@refcnt.org ("Steven Schubiger")|Data-Compare|1.2101|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200905052239
3821470|pass|200905|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.2101|i686-linux|5.8.9|linux|2.6.24-16-generic|200905051558
3821247|pass|200905|srezic@cpan.org|Data-Compare|1.2101|amd64-freebsd|5.8.9|freebsd|7.0-release|200905052035
3821101|pass|200905|srezic@cpan.org|Data-Compare|1.2101|amd64-freebsd|5.8.8|freebsd|7.0-release|200905052035
3820909|pass|200905|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.2101|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200905051408
3820784|pass|200905|andy@hexten.net|Data-Compare|1.2101|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200905051826
3820777|pass|200905|vpit@cpan.org|Data-Compare|1.2101|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200905051924
3820757|pass|200905|andy@hexten.net|Data-Compare|1.2101|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200905051819
3819856|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200905051644
3819854|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200905051644
3819852|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200905051644
3819850|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200905051644
3819849|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200905051644
3819847|pass|200905|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.2101|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200905051643
3819841|cpan|200905|DCANTRELL|Data-Compare|1.2101||0|||200905051643
3818121|pass|200905|srezic@cpan.org|Devel-CheckOS|1.61|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200905051139
3816446|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200905050746
3816200|pass|200905|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200905050122
3816199|pass|200905|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200905050121
3816195|pass|200905|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200905050121
3815894|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.32|darwin-2level|5.10.0|darwin|8.11.0|200905050518
3815752|pass|200905|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200905050349
3815377|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|darwin-2level|5.10.0|darwin|8.11.0|200905050255
3815133|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-2level|5.10.0|darwin|8.11.0|200905050034
3815130|pass|200905|bingos@cpan.org|Data-Compare|1.21|darwin-2level|5.10.0|darwin|8.11.0|200905050033
3814617|pass|200905|dagolden@cpan.org (DAGOLDEN)|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905041655
3814111|pass|200905|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905041433
3814105|pass|200905|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905041432
3813942|pass|200905|robin@smidsrod.no ("Robin Smidsrd")|Data-Compare|1.21|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.24-19-server|200905041419
3813585|pass|200905|robin@smidsrod.no ("Robin Smidsrd")|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.24-19-server|200905041419
3813276|pass|200905|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.32|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200905041126
3812829|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.32|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200905041555
3812299|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200905041430
3811869|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200905041319
3811866|pass|200905|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|5.0|200905041318
3811253|pass|200905|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.32|i686-linux|5.8.9|linux|2.6.24-16-generic|200905040500
3810091|pass|200905|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.32|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200905040320
3809639|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905040143
3809209|pass|200905|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905040631
3808378|pass|200905|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200905032119
3806704|pass|200905|srezic@cpan.org|Devel-CheckOS|1.61|amd64-freebsd|5.10.0|freebsd|7.0-release|200905031330
3805131|pass|200905|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.32|i686-linux|5.10.0|linux|2.6.24-16-generic|200905030120
3804824|pass|200905|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux|5.10.0|linux|2.6.24-16-generic|200905030022
3804419|pass|200905|imacat@mail.imacat.idv.tw|Number-Phone|1.7002|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905031103
3804343|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.32|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905030335
3804338|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.31|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905030332
3803912|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.32|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200905030249
3803909|pass|200905|bingos@cpan.org|CPAN-FindDependencies|2.31|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200905030248
3803632|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905030100
3803455|pass|200905|bingos@cpan.org|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200905030112
3803091|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905022257
3803088|pass|200905|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200905022257
3802702|pass|200905|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200905022301
3802701|pass|200905|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|5.0|200905022301
3801492|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200905021759
3801406|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200905021748
3800592|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200905021455
3800316|pass|200905|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.61|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905021954
3800315|pass|200905|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-2-amd64|200905021953
3799831|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200905021209
3799256|pass|200905|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200905021016
3798425|pass|200904|pjacklam@cpan.org|Devel-CheckOS|1.61|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904301235
3798266|pass|200905|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200905020231
3795838|pass|200905|mw487.yahoo.com@gmail.com|Number-Phone|1.7002|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200905010627
3793477|pass|200905|mw487.yahoo.com@gmail.com|Devel-CheckLib|0.5|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200905010115
3792562|pass|200904|bingos@cpan.org|Number-Phone|1.7002|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904302257
3792558|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.32|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904302256
3792552|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904302254
3792530|pass|200905|stro@cpan.org|Palm-Treo680MessagesDB|1.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200905010047
3792468|pass|200905|stro@cpan.org|Number-Phone|1.7002|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200905010036
3792434|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904302159
3792346|pass|200904|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200904302110
3792166|pass|200904|andy@hexten.net|Number-Phone|1.7002|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904302132
3792135|pass|200904|andy@hexten.net|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904302127
3792051|pass|200904|andy@hexten.net|CPAN-FindDependencies|2.32|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904302114
3791933|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904302047
3791369|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904301841
3791366|pass|200904|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904301840
3788533|pass|200904|bingos@cpan.org|Number-Phone|1.7002|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904300348
3788531|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904300347
3788530|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.32|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904300346
3788528|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904300343
3788379|pass|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.61|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200904300152
3788378|pass|200904|mw487.yahoo.com@gmail.com|Data-Compare|1.21|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200904300152
3788148|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904300127
3787668|pass|200904|stro@cpan.org|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904300100
3787390|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904292204
3787388|pass|200904|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904292204
3787129|pass|200904|srezic@cpan.org|Devel-CheckOS|1.55|amd64-freebsd|5.6.2|freebsd|7.0-release|200904292203
3787035|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291952
3787027|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291945
3787025|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.32|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291945
3787023|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291943
3786800|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291830
3786765|pass|200904|bingos@cpan.org|Number-Phone|1.7002|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291833
3786762|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291832
3786757|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.32|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291831
3786752|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291829
3786375|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291245
3786374|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904291245
3786349|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291658
3785781|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291414
3785780|pass|200904|bingos@cpan.org|Data-Compare|1.21|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904291413
3784584|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904290940
3784508|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904290913
3784470|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904290848
3784416|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904290824
3784356|pass|200904|bingos@cpan.org|Number-Phone|1.7002|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290704
3784354|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290702
3784352|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.32|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290701
3784349|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290659
3784212|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904290658
3784126|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904290619
3784125|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904290617
3784121|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290515
3783631|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904290338
3783616|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904290330
3783547|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904290304
3783465|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904290245
3783398|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904290230
3783357|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290121
3783353|pass|200904|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904290121
3782937|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904290101
3782820|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904290036
3782702|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904290017
3782385|pass|200904|andy@hexten.net|Data-Compare|1.21|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200904282237
3782329|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904282329
3782306|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904282326
3782208|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904282256
3782206|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904282256
3782193|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904282252
3782162|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904282244
3782039|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904282213
3781657|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904282013
3781635|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904282007
3781566|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904281946
3781501|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904281836
3781409|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904281918
3781112|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904281842
3781088|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904281838
3780929|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904281811
3780780|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904281753
3780767|pass|200904|andy@hexten.net|Data-Hexdumper|2.01|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200904281652
3780720|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904281745
3780350|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904281643
3780333|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904281640
3780258|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904281631
3780218|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.32|i86pc-solaris-64int|5.8.9|solaris|2.11|200904281623
3780093|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.32|alpha-netbsd|5.10.0|netbsd|4.0.1|200904281411
3779839|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904281536
3779737|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.32|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904280824
3779698|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.32|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904281518
3779519|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904281446
3779429|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200904281333
3779422|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200904281332
3779417|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200904281331
3779398|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200904281329
3779397|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904281429
3779265|pass|200904|carlos.arenas@gmail.com ("Carlos Arenas")|Devel-CheckOS|1.61|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200904270954
3779152|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904281358
3779045|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904281338
3779035|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904281335
3778995|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.32|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904281231
3778984|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.32|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904281230
3778979|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904281229
3778850|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904281305
3778845|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904281304
3778820|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904281258
3778728|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904281242
3778723|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904281241
3778665|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904281234
3778659|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904281232
3778597|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904281221
3778470|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904281053
3778434|pass|200904|david@cantrell.org.uk|Devel-CheckLib|0.5|alpha-netbsd|5.10.0|netbsd|4.0.1|200904280941
3778421|pass|200904|david@cantrell.org.uk|Devel-CheckLib|0.5|IP35-irix-64int|5.8.8|irix|6.5|200904281138
3778404|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904281133
3778393|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904281132
3778362|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904281126
3778307|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904281114
3778254|pass|200904|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.32|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904281706
3778066|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.32|i86pc-solaris-64int|5.10.0|solaris|2.11|200904281028
3777918|pass|200904|david@cantrell.org.uk|Devel-CheckLib|0.5|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904280904
3777894|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200904280902
3777893|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200904280902
3777442|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904280848
3777198|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.32|amd64-freebsd|5.8.8|freebsd|7.0-release|200904280733
3777058|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.32|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904280635
3777056|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.32|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904280634
3777053|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.32|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904280633
3777051|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.32|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904280631
3777050|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.32|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904280630
3776963|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904280605
3776447|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904272316
3776161|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.32|amd64-freebsd|5.8.9|freebsd|7.0-release|200904272252
3775998|pass|200904|srezic@cpan.org|Devel-CheckOS|1.61|amd64-freebsd|5.6.2|freebsd|7.0-release|200904272220
3775831|pass|200904|chris@bingosnet.co.uk|Devel-CheckLib|0.5|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200904272049
3775715|pass|200904|vpit@cpan.org|Number-Phone|1.7002|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904272130
3775679|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.32|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904272127
3775439|pass|200904|vpit@cpan.org|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904272049
3775330|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.32|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904272030
3775202|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904271953
3774891|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904271703
3774869|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904271655
3774866|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904271654
3774858|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904271652
3774732|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.32||0|||200904271731
3774457|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904271625
3774385|pass|200904|chris@bingosnet.co.uk|Devel-CheckOS|1.61|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904271504
3774340|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.31|i86pc-solaris-64int|5.10.0|solaris|2.11|200904271553
3774129|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904271417
3774128|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904271417
3774046|pass|200904|hohokus@gmail.com|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200904270857
3773940|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904271338
3773819|pass|200904|vovkasm@gmail.com|Devel-CheckOS|1.61|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200904271440
3773760|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904271405
3773627|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.31|i86pc-solaris-64int|5.8.9|solaris|2.11|200904271350
3773590|pass|200904|autarch@urth.org|Data-Compare|1.21|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.24-16-server|200904260010
3773445|pass|200904|rlane10@wi.rr.com|Devel-CheckOS|1.61|i386-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5xen|200904261836
3773291|pass|200904|rlane10@wi.rr.com|CPAN-FindDependencies|2.31|i386-linux-thread-multi|5.10.0|linux|2.6.18-128.1.6.el5xen|200904261329
3772807|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|alpha-netbsd|5.10.0|netbsd|4.0.1|200904270948
3772733|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.61|i86pc-solaris-64int|5.10.0|solaris|2.11|200904271124
3772667|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.61|i86pc-solaris-64int|5.8.9|solaris|2.11|200904271104
3771942|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904270552
3771937|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904270649
3771931|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904270645
3771823|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904270545
3771570|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904270359
3771562|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904270355
3771492|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904270336
3771314|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904270241
3771166|pass|200904|rbo@cpan.org|Devel-CheckLib|0.5|i86pc-solaris|5.8.9|solaris|2.11|200904270200
3771121|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904270146
3771093|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904270141
3770842|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904270029
3770695|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904262341
3770689|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904262338
3770532|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904262302
3770507|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904262258
3770422|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904262141
3770417|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904262140
3770413|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904262139
3770403|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904262137
3770224|pass|200904|rbo@cpan.org|Number-Phone|1.7002|i86pc-solaris|5.8.9|solaris|2.11|200904262149
3770222|pass|200904|rbo@cpan.org|Devel-CheckOS|1.6|i86pc-solaris|5.8.9|solaris|2.11|200904262149
3770031|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904262055
3769899|pass|200904|perl@galumph.com|Devel-CheckOS|1.61|darwin-thread-multi-64int-ld-2level|5.10.0|darwin|9.4.0|200904261324
3769824|pass|200904|rbo@cpan.org|Devel-CheckOS|1.61|i86pc-solaris|5.8.9|solaris|2.11|200904262006
3769389|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904261724
3769387|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904261724
3769380|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904261822
3769368|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904261820
3769345|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904261816
3768833|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904261609
3768809|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904261605
3768777|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904261440
3768683|pass|200904|mahony357@gmail.com|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200904261331
3768528|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904261438
3768507|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|x86_64-linux|5.10.0|linux|2.6.26-1-amd64|200904261411
3768319|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904261224
3768188|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|IP35-irix-64int|5.8.8|irix|6.5|200904261259
3768183|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904261158
3768175|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904261157
3768174|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904261156
3768134|pass|200904|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.31|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904261842
3768106|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904261137
3768036|pass|200904|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.61|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904261826
3767931|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.61|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200904261200
3767915|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904261157
3767886|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.61|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200904261149
3767871|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.61|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200904261147
3767860|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.61|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200904261145
3767815|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.61|i586-linux-thread-multi|5.8.8|linux|2.6.16|200904261135
3767632|pass|200904|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7002|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200904260444
3767505|pass|200904|srezic@cpan.org|Devel-CheckOS|1.61|amd64-freebsd|5.8.9|freebsd|7.0-release|200904261005
3767321|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904260824
3767283|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.31|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904260817
3767274|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.31|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200904260315
3767261|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.61|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904260912
3767249|pass|200904|stro@cpan.org|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904261010
3767169|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200904260247
3767135|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200904260239
3766926|pass|200904|rafl@debian.org|Devel-CheckOS|1.61|i686-linux|5.10.0 patch 35117|linux|2.6.26|200904260725
3766904|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux|5.8.9|linux|2.6.24-16-generic|200904260119
3766834|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904260605
3766574|pass|200904|dagolden@cpan.org (DAGOLDEN)|XML-Tiny-DOM|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200904260011
3766572|pass|200904|dagolden@cpan.org (DAGOLDEN)|Palm-ProjectGutenberg|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200904260011
3766148|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904260432
3766096|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-ParseDistribution|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200904260220
3765615|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904260245
3765490|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904260119
3765401|pass|200904|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.01|i686-linux|5.8.9|linux|2.6.24-16-generic|200904251956
3765398|pass|200904|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200904251955
3765271|pass|200904|andy@hexten.net|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200904260028
3765268|pass|200904|andy@hexten.net|Data-Compare|1.21|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200904260027
3765262|pass|200904|andy@hexten.net|Devel-CheckLib|0.5|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200904260027
3765227|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904260118
3765224|pass|200904|mw487.yahoo.com@gmail.com|CPAN-FindDependencies|2.31|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904251818
3765217|pass|200904|mw487.yahoo.com@gmail.com|CPAN-FindDependencies|2.3|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904251817
3765169|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.31|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904260009
3765147|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.61|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904260003
3764975|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904252332
3764942|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904252325
3764940|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904252324
3764898|pass|200904|srezic@cpan.org|Devel-CheckOS|1.61|amd64-freebsd|5.8.8|freebsd|7.0-release|200904260010
3764724|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904252333
3764683|pass|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.61|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904251627
3764412|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904252235
3764154|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904252053
3764122|pass|200904|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.61|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904252146
3764063|pass|200904|chris@bingosnet.co.uk|Devel-CheckOS|1.61|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200904252033
3764002|pass|200904|chris@bingosnet.co.uk|Devel-CheckOS|1.61|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200904252024
3763999|pass|200904|chris@bingosnet.co.uk|Data-Compare|1.21|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200904252024
3763884|fail|200904|wiml@hhhh.org ("Wim Lewis")|CPAN-FindDependencies|2.3|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904250925
3763826|pass|200904|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7002|i686-linux|5.8.9|linux|2.6.24-16-generic|200904251452
3763802|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904252047
3763711|pass|200904|bingos@cpan.org|Devel-CheckOS|1.61|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904251935
3763708|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904251934
3763586|pass|200904|andy@hexten.net|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.28-11-generic|200904251915
3763491|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904251902
3763251|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904251933
3763229|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.31|i686-linux|5.8.9|linux|2.6.24-16-generic|200904251330
3763224|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.6|i686-linux|5.8.9|linux|2.6.24-16-generic|200904251330
3763213|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904251928
3762920|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.61|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904251241
3762786|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904251809
3762744|pass|200904|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.0401|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904251158
3762639|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.31|alpha-netbsd|5.10.0|netbsd|4.0.1|200904251533
3762620|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904251727
3762331|pass|200904|dagolden@cpan.org (DAGOLDEN)|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904251029
3762330|pass|200904|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904251029
3762329|pass|200904|dagolden@cpan.org (DAGOLDEN)|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904251029
3762182|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904251605
3762132|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904251555
3762012|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904251535
3761967|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.61|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904252232
3761690|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.31|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904251340
3761689|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.31|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904251339
3761596|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.31|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904251320
3761534|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.31|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904251313
3761486|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904251405
3761437|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904250755
3761319|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904251331
3761286|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904251319
3761281|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904251320
3761278|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904251320
3761274|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904251319
3761271|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904251319
3761269|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.61|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904251319
3761267|cpan|200904|DCANTRELL|Devel-CheckOS|1.61||0|||200904251318
3761077|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904251232
3761056|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904251228
3761010|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904251119
3760964|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904251202
3760790|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904251116
3760696|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904251102
3760648|pass|200904|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904250450
3760645|pass|200904|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904250449
3760611|pass|200904|wiml@hhhh.org ("Wim Lewis")|Number-Phone|1.7002|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904250845
3760593|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.3|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904251042
3760372|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904250955
3760167|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904250915
3759967|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904250835
3759940|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904250831
3759804|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904250800
3759665|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904250732
3759542|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904250703
3759215|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904250452
3759204|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904250451
3758867|pass|200904|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7002|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904250240
3758652|pass|200904|wiml@hhhh.org ("Wim Lewis")|Devel-CheckOS|1.6|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904250154
3758626|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904250349
3758317|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.31|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904250101
3758311|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200904250100
3758131|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904250230
3758091|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904250223
3758015|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904250211
3758008|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904250210
3757882|pass|200904|stro@cpan.org|File-Overwrite|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904250244
3757821|pass|200904|stro@cpan.org|Devel-CheckOS|1.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904250236
3757816|pass|200904|stro@cpan.org|Devel-CheckOS|1.55|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904250235
3757811|pass|200904|stro@cpan.org|Devel-CheckOS|1.51|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904250235
3757769|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.31|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241928
3757672|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904250110
3757656|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904250008
3757653|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904250007
3757560|pass|200904|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.0401|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241850
3757515|pass|200904|stro@cpan.org|CPAN-FindDependencies|2.22|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904250143
3757288|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904242314
3757286|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904242314
3757072|pass|200904|dagolden@cpan.org (DAGOLDEN)|XML-Tiny-DOM|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241735
3757071|pass|200904|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241735
3757069|pass|200904|dagolden@cpan.org (DAGOLDEN)|Palm-ProjectGutenberg|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241735
3756935|pass|200904|graham@crookham.uklinux.net (grahamc)|Number-Phone|1.7002|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-server|200904242214
3756742|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904242249
3756734|pass|200904|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.6|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904242249
3756554|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.6|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200904242220
3756411|pass|200904|stsc@refcnt.org ("Steven Schubiger")|Number-Phone|1.7002|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904242202
3756410|pass|200904|graham@crookham.uklinux.net (grahamc)|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-server|200904242101
3756386|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.6|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200904242158
3756367|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.6|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200904242156
3756303|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904242147
3756299|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-ParseDistribution|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241547
3756175|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|IP35-irix-64int|5.8.8|irix|6.5|200904242129
3756028|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904242105
3755979|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904242055
3755926|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904242044
3755856|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904242034
3755808|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904242021
3755754|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904242010
3755655|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.31|amd64-freebsd|5.10.0|freebsd|7.0-release|200904241946
3755654|pass|200904|srezic@cpan.org|Devel-CheckOS|1.6|amd64-freebsd|5.10.0|freebsd|7.0-release|200904241945
3755595|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904241938
3755535|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.31|amd64-freebsd|5.8.9|freebsd|7.0-release|200904241932
3755524|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.31|amd64-freebsd|5.8.8|freebsd|7.0-release|200904241932
3755500|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904241230
3755480|pass|200904|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.01|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241325
3755475|pass|200904|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241324
3755343|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7002|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904241904
3755265|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904241848
3755251|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904241846
3755239|fail|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|CPAN-FindDependencies|2.31|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904250145
3755155|pass|200904|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904241736
3754984|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904241804
3754822|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904241638
3754804|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.31|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904241736
3754794|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.31|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904241734
3754780|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.31|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904241733
3754773|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.31|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904241732
3754769|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.31|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904241730
3754753|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.31|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904241730
3754728|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.31||0|||200904241728
3754612|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904241617
3754611|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904241617
3754463|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904241558
3754413|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904241552
3754394|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.7002|i86pc-solaris-64int|5.8.9|solaris|2.11|200904241651
3753952|pass|200904|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.3|i686-linux|5.10.0|linux|2.6.24-16-generic|200904241000
3753280|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904241419
3753230|pass|200904|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7002|i686-linux|5.10.0|linux|2.6.24-16-generic|200904240812
3752767|pass|200904|vovkasm@gmail.com|Devel-CheckOS|1.6|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200904232256
3752666|pass|200904|jozef@kutej.net|Data-Compare|1.21|x86_64-linux-gnu-thread-multi|5.8.8|linux|2.6.24-16-server|200904240956
3752411|pass|200904|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.6|i686-linux|5.10.0|linux|2.6.24-16-generic|200904240708
3752077|pass|200904|bingos@cpan.org|Number-Phone|1.7002|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904241127
3752074|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904241126
3752071|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904241125
3752070|pass|200904|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904241125
3752017|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904241216
3751999|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904241212
3751996|pass|200904|graham@crookham.uklinux.net (grahamc)|Devel-CheckOS|1.6|i686-linux-thread-multi|5.8.9|linux|2.6.27-11-server|200904241112
3751994|pass|200904|graham@crookham.uklinux.net (grahamc)|Data-Compare|1.21|i686-linux-thread-multi|5.8.9|linux|2.6.27-11-server|200904241112
3751962|fail|200904|chris@bingosnet.co.uk|Devel-CheckOS|1.6|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904241106
3751957|pass|200904|chris@bingosnet.co.uk|Data-Compare|1.21|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904241103
3751933|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904241156
3751873|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904241143
3751673|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904241106
3751655|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904241102
3751552|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904241046
3751451|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904240931
3751396|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904241024
3751335|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904241013
3751313|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904241010
3751300|pass|200904|stro@cpan.org|Number-Phone|1.7002|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904241106
3751273|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904240959
3751200|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904240943
3751177|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904240935
3750794|pass|200904|conrad@fringehead.org|XML-Tiny|2.02|i686-linux-thread-multi|5.8.8|linux|2.6.15.1|200904240509
3750428|pass|200904|mw487.yahoo.com@gmail.com|Number-Phone|1.7002|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904240323
3750365|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904240507
3750288|pass|200904|rafl@debian.org|Devel-CheckOS|1.6|i686-linux|5.10.0 patch 35117|linux|2.6.26|200904240443
3750112|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|IP35-irix-64int|5.8.8|irix|6.5|200904240351
3749967|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904240251
3749342|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232224
3749341|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232223
3749339|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232222
3749335|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232221
3749333|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232221
3749331|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904232319
3749327|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232218
3749317|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904232216
3749236|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904232247
3749044|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232024
3749043|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232023
3749042|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232023
3749040|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232022
3749039|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232022
3749037|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232020
3749020|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232019
3749019|pass|200904|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232019
3749018|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904232018
3748395|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904231135
3748368|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904231732
3748339|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904231728
3748320|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904231723
3748134|pass|200904|imacat@mail.imacat.idv.tw|Number-Phone|1.7002|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904232345
3747965|pass|200904|david@cantrell.org.uk|Number-Phone|1.7002|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904231613
3747919|pass|200904|carlos.arenas@gmail.com ("Carlos Arenas")|Devel-CheckOS|1.6|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200904220947
3747736|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904231641
3747701|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.3|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200904231630
3747700|pass|200904|srezic@cpan.org|Devel-CheckOS|1.6|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200904231629
3747656|pass|200904|graham@crookham.uklinux.net (grahamc)|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-server|200904222039
3747588|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904231511
3747587|pass|200904|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200904231510
3747207|pass|200904|graham@crookham.uklinux.net (grahamc)|Devel-CheckOS|1.6|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-server|200904222039
3746441|pass|200904|perl@galumph.com|Devel-CheckOS|1.6|darwin-thread-multi-64int-ld-2level|5.10.0|darwin|9.4.0|200904230730
3746082|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.7002|i86pc-solaris-64int|5.10.0|solaris|2.11|200904231336
3745928|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904231250
3745700|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|alpha-netbsd|5.10.0|netbsd|4.0.1|200904231000
3745673|pass|200904|chris@bingosnet.co.uk|Devel-CheckOS|1.6|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200904231049
3745669|pass|200904|chris@bingosnet.co.uk|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200904231048
3745585|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904230417
3745330|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904230929
3745223|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904230759
3745119|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.6|i586-linux-thread-multi|5.8.8|linux|2.6.16|200904230834
3744768|fail|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.6|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904230430
3744206|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904230222
3744118|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Number-Phone|1.7002|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904230953
3743931|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904230209
3743879|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904230202
3743599|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|IP35-irix-64int|5.8.8|irix|6.5|200904230055
3743594|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222350
3743584|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222350
3743572|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222350
3743568|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222349
3743559|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222348
3743554|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222347
3743550|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222346
3743540|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222344
3743533|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222341
3743532|pass|200904|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222341
3743527|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222340
3743517|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200904222338
3743379|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904222316
3743175|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904222342
3743025|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904222324
3742985|pass|200904|vpit@cpan.org|Devel-CheckOS|1.55|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904222321
3742984|pass|200904|vpit@cpan.org|Devel-CheckOS|1.54|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904222321
3742982|pass|200904|vpit@cpan.org|Devel-CheckOS|1.53|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904222321
3742978|fail|200904|vpit@cpan.org|Devel-CheckOS|1.52|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904222321
3742974|pass|200904|srezic@cpan.org|Number-Phone|1.7002|amd64-freebsd|5.8.9|freebsd|7.0-release|200904222317
3742837|pass|200904|srezic@cpan.org|Number-Phone|1.7002|amd64-freebsd|5.8.8|freebsd|7.0-release|200904222223
3742690|pass|200904|vpit@cpan.org|Devel-CheckOS|1.6|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904222235
3742496|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904221500
3742412|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904222132
3742333|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904222111
3742293|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904222102
3742281|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7002|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904222051
3742250|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7002|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904222053
3742248|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7002|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904222052
3742247|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7002|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904222051
3742242|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7002|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904222050
3742241|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7002|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904222050
3742215|cpan|200904|DCANTRELL|Number-Phone|1.7002||0|||200904222047
3741984|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|IP35-irix-64int|5.8.8|irix|6.5|200904222007
3741786|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221823
3741785|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221822
3741782|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221821
3741779|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221821
3741774|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221820
3741772|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221819
3741770|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904221820
3741761|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221816
3741749|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221814
3741746|pass|200904|bingos@cpan.org|Data-Compare|1.21|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221814
3741743|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221813
3741729|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904221810
3741701|pass|200904|andy@hexten.net|Devel-CheckOS|1.6|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904221804
3741646|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|alpha-netbsd|5.10.0|netbsd|4.0.1|200904221657
3741239|pass|200904|hohokus@gmail.com|Devel-CheckOS|1.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200904221134
3741191|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904221625
3741180|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904221622
3741133|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904221617
3740944|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904221541
3740942|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.3|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904221541
3740742|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904221608
3740676|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904221555
3740568|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904221433
3740534|fail|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.3|i86pc-solaris-64int|5.8.9|solaris|2.11|200904221523
3740358|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904221445
3740301|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.6|i86pc-solaris-64int|5.8.9|solaris|2.11|200904221431
3739979|fail|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.3|i86pc-solaris-64int|5.10.0|solaris|2.11|200904221302
3739916|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.6|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904221945
3739805|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904221131
3739804|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904221131
3739803|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.6|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904221131
3739700|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904221207
3739552|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904221121
3739537|pass|200904|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.6|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904221716
3739496|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.6|i86pc-solaris-64int|5.10.0|solaris|2.11|200904221055
3739036|pass|200904|stro@cpan.org|Devel-CheckOS|1.6|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904221022
3738174|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904220526
3738107|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220406
3738105|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220405
3738103|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220405
3738101|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220404
3738099|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220404
3738095|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220403
3738093|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220402
3738092|pass|200904|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220401
3738089|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220401
3738085|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200904220359
3738078|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904220355
3738055|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904220347
3738026|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904220339
3738009|pass|200904|bingos@cpan.org|Devel-CheckOS|1.6|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904220332
3737921|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904220358
3737845|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904220338
3737791|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904220325
3737504|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904220216
3737335|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904220118
3737329|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904220116
3736710|pass|200904|srezic@cpan.org|Devel-CheckOS|1.6|amd64-freebsd|5.8.9|freebsd|7.0-release|200904212244
3736692|pass|200904|srezic@cpan.org|Devel-CheckOS|1.6|amd64-freebsd|5.8.8|freebsd|7.0-release|200904212244
3736586|pass|200904|graham@crookham.uklinux.net ("GRAHAMC")|Devel-CheckOS|1.55|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-generic|200904202225
3736540|pass|200904|srezic@cpan.org|Devel-CheckOS|1.55|amd64-freebsd|5.10.0|freebsd|7.0-release|200904211809
3736538|pass|200904|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.10.0|freebsd|7.0-release|200904211809
3736397|pass|200904|rlane10@wi.rr.com|CPAN-FindDependencies|2.3|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200904211000
3736358|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.6|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904212142
3736355|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.6|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904212142
3736351|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.6|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904212142
3736349|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.6|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904212141
3736348|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.6|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904212141
3736343|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.6|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904212140
3736322|cpan|200904|DCANTRELL|Devel-CheckOS|1.6||0|||200904212139
3736091|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904212045
3735696|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904211933
3735253|pass|200904|rlane10@wi.rr.com|Devel-CheckOS|1.55|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200904210958
3735091|pass|200904|graham@crookham.uklinux.net ("GRAHAMC")|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-generic|200904202224
3734631|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904211738
3733901|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904211517
3733654|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211325
3733649|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211325
3733643|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211324
3733639|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211323
3733638|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211323
3733635|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211321
3733633|pass|200904|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211321
3733628|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211321
3733622|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200904211319
3733541|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904211405
3733498|fail|200904|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904211956
3732673|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904211127
3732601|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904211107
3732504|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904211051
3732497|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904211050
3732484|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904211049
3732414|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904211037
3731214|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904210513
3731188|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904210506
3731153|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904210459
3731123|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.3|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904210449
3731025|pass|200904|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904210421
3730560|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904210315
3730353|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904210221
3729538|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.3|amd64-freebsd|5.8.8|freebsd|7.0-release|200904202316
3729527|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.3|amd64-freebsd|5.8.9|freebsd|7.0-release|200904202316
3729436|pass|200904|andy@hexten.net|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200904202205
3729431|pass|200904|andy@hexten.net|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200904202204
3728970|fail|200904|andy@hexten.net|CPAN-FindDependencies|2.3|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904202118
3728963|pass|200904|andy@hexten.net|Devel-CheckOS|1.55|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904202118
3728847|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202105
3728838|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202104
3728833|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202104
3728830|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202103
3728828|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202102
3728822|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202101
3728820|pass|200904|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202101
3728819|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202100
3728816|na|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200904202058
3728598|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.3|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904202124
3727449|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.3|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904201814
3727441|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.3|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904201813
3727434|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.3|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904201811
3727421|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.3|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904201810
3727412|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.3|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904201809
3727330|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.3|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904201755
3727292|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.3||0|||200904201753
3727080|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904201714
3726992|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904201702
3725988|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904201313
3725493|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.55|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200904201225
3725491|pass|200904|g.grigelionis@computer.org|Data-Compare|1.21|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200904201225
3725319|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904201204
3725238|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904201154
3725219|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904201151
3725199|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904201148
3725185|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904201146
3725160|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904201141
3725088|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904201132
3724938|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904201058
3724880|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904201045
3724819|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904201034
3724790|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904201029
3724755|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904201022
3724728|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904201016
3724709|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904201010
3724684|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904201005
3724668|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904201002
3724639|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904200955
3724609|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904200948
3724601|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904200947
3724411|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200819
3724406|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200818
3724403|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200817
3724398|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200816
3724396|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200815
3724385|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200813
3724381|pass|200904|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200813
3724375|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200812
3724364|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200904200809
3724208|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904200841
3723315|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200348
3723312|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200348
3723310|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200347
3723309|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200346
3723307|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200346
3723306|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200344
3723305|pass|200904|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200344
3723304|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200338
3723300|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200338
3723298|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200337
3723291|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200904200335
3718803|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904190528
3718634|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904190449
3718448|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904190411
3718419|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904190405
3718408|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904190401
3718358|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904190347
3718332|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904190340
3718259|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904190328
3718257|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904190328
3718197|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904190312
3718102|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904190250
3717979|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904190219
3717897|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904190102
3717554|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904190101
3717449|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904190046
3717404|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904190035
3717157|fail|200904|srezic@cpan.org|Devel-CheckOS|1.52|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200904182352
3716898|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904182313
3716711|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904182241
3716329|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182029
3716321|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182028
3716319|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904182127
3716312|pass|200904|bingos@cpan.org|File-Overwrite|1.1|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182026
3716288|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182023
3716273|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182018
3716268|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182016
3716256|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182013
3716250|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182011
3716248|pass|200904|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182011
3716223|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182007
3716217|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904182105
3716205|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904182002
3716195|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200904181959
3715210|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.22|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200904181757
3715154|pass|200904|srezic@cpan.org|Devel-CheckOS|1.55|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200904181755
3715020|pass|200904|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904181616
3714794|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904181659
3714500|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904181617
3713957|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904181510
3713486|pass|200904|srezic@cpan.org|File-Overwrite|1.1|amd64-freebsd|5.10.0|freebsd|7.0-release|200904181405
3713483|pass|200904|srezic@cpan.org|Data-Hexdumper|2.01|amd64-freebsd|5.10.0|freebsd|7.0-release|200904181406
3713482|pass|200904|srezic@cpan.org|Palm-Treo680MessagesDB|1.01|amd64-freebsd|5.10.0|freebsd|7.0-release|200904181406
3713433|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181304
3713425|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181303
3713419|pass|200904|bingos@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181303
3713415|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181302
3713411|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181301
3713406|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181301
3713403|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181301
3713400|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181300
3713394|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181259
3713392|pass|200904|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181259
3713385|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181258
3713375|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181257
3713368|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904181256
3712331|pass|200904|rafl@debian.org|Devel-CheckOS|1.55|i686-linux|5.10.0 patch 35117|linux|2.6.26|200904181111
3712201|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904181042
3712154|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904181031
3712117|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904181018
3712084|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904181009
3712074|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904181005
3712046|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904180957
3711836|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904180850
3711324|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904180605
3711124|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904180440
3711100|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904180431
3710910|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904180322
3710693|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904180217
3710690|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904180217
3710687|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904180217
3710656|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904180208
3710635|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904180042
3710507|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904180134
3710486|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904180127
3710481|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904180126
3710409|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904180107
3710339|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904180056
3710325|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904180052
3710275|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904180043
3710228|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904180037
3710224|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904180036
3710168|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904180024
3710027|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904172350
3709888|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904172317
3709845|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904172309
3709547|fail|200904|pjacklam@online.no|Devel-CheckOS|1.55|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904172205
3709300|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.55|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200904172119
3709298|pass|200904|g.grigelionis@computer.org|Data-Compare|1.21|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200904172119
3709289|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.55|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200904172118
3709288|pass|200904|g.grigelionis@computer.org|Data-Compare|1.21|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200904172118
3709262|pass|200904|hohokus@gmail.com|Devel-CheckOS|1.55|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200904171515
3709250|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904172013
3709199|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904172101
3709033|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.22|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904172019
3708923|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171835
3708914|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171834
3708913|pass|200904|bingos@cpan.org|File-Overwrite|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171834
3708910|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171834
3708906|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171833
3708903|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171832
3708899|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171832
3708893|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171831
3708889|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171830
3708888|pass|200904|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171829
3708885|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171829
3708879|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171827
3708869|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200904171826
3708863|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904171943
3708330|pass|200904|pjacklam@online.no|Data-Compare|1.21|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904171748
3708089|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904171656
3707898|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904171608
3707857|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904171555
3707782|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904171525
3707753|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904171514
3707731|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904171506
3707707|pass|200904|carlos.arenas@gmail.com ("Carlos Arenas")|Devel-CheckOS|1.55|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200904161129
3707562|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.53|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904171427
3707555|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904171325
3707387|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904171339
3707380|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904171338
3707312|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904171327
3707283|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904171322
3707238|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904171314
3707227|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904171312
3707175|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904171258
3707132|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904171249
3707109|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904171244
3707088|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904171238
3707079|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904171236
3707014|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904171230
3706926|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904171221
3706888|fail|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.54|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170516
3706881|fail|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.53|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170515
3706862|pass|200904|mw487.yahoo.com@gmail.com|CPAN-FindDependencies|2.21|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170513
3706504|pass|200904|carlos.arenas@gmail.com ("Carlos Arenas")|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200904161129
3705964|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904170932
3705943|pass|200904|g.grigelionis@computer.org|Devel-CheckOS|1.55|i586-linux-thread-multi|5.8.8|linux|2.6.16|200904170928
3705571|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904170603
3705570|pass|200904|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904170603
3705385|pass|200904|mw487.yahoo.com@gmail.com|XML-DoubleEncodedEntities|1.0|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170421
3705110|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904170506
3705102|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170403
3705097|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170402
3705096|pass|200904|bingos@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170401
3705092|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170401
3705090|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170400
3705088|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170400
3705086|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170359
3705083|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170359
3705077|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170357
3705076|pass|200904|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170357
3705074|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170357
3705069|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170355
3705061|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904170354
3704959|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904170420
3704463|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904170236
3704357|fail|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.52|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170015
3704350|pass|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.51|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170015
3704346|fail|200904|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.55|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170014
3704326|pass|200904|mw487.yahoo.com@gmail.com|CPAN-FindDependencies|2.2|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200904170012
3703291|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904162243
3703107|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904162151
3703003|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904162132
3702871|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904162057
3702825|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904162043
3702192|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904161825
3701958|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|alpha-netbsd|5.10.0|netbsd|4.0.1|200904161547
3701950|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|alpha-netbsd|5.10.0|netbsd|4.0.1|200904161544
3701926|pass|200904|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904161625
3701866|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904161629
3701865|pass|200904|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904161629
3701563|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904161628
3701409|pass|200904|rbo@cpan.org|Devel-CheckOS|1.55|i86pc-solaris|5.8.9|solaris|2.11|200904161251
3701376|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.54|IP35-irix-64int|5.8.8|irix|6.5|200904161557
3701375|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.54|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904160857
3701364|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.53|IP35-irix-64int|5.8.8|irix|6.5|200904161556
3701360|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.53|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904160856
3701336|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|IP35-irix-64int|5.8.8|irix|6.5|200904161553
3701328|fail|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904160852
3701313|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904160850
3701302|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904160847
3701292|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.21|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904160844
3701283|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|IP35-irix-64int|5.8.8|irix|6.5|200904161542
3701280|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.54|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904161441
3701274|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|IP35-irix-64int|5.8.8|irix|6.5|200904161540
3701272|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.53|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904161440
3701250|fail|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904161439
3701244|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904161438
3701233|pass|200904|perl@galumph.com|Devel-CheckOS|1.55|darwin-thread-multi-64int-ld-2level|5.10.0|darwin|9.4.0|200904160836
3701231|pass|200904|perl@galumph.com|Data-Compare|1.21|darwin-thread-multi-64int-ld-2level|5.10.0|darwin|9.4.0|200904160835
3701100|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904161508
3700932|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.54|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904161338
3700923|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.53|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904161337
3700915|fail|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904161337
3700906|pass|200904|vovkasm@gmail.com|Devel-CheckOS|1.55|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200904161244
3700898|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904161336
3700897|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904161335
3700893|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904161335
3700892|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.22|i86pc-solaris-64int|5.8.9|solaris|2.11|200904161435
3700889|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904161335
3700879|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.21|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904161333
3700878|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.21|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904161333
3700876|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.21|IP35-irix-64int|5.8.8|irix|6.5|200904161433
3700855|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.55|i86pc-solaris-64int|5.8.9|solaris|2.11|200904161429
3700802|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904161420
3700699|pass|200904|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.22|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904162002
3700689|pass|200904|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.55|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904162002
3700675|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904161300
3700608|pass|200904|rlane10@wi.rr.com|CPAN-FindDependencies|2.21|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200904151607
3700607|pass|200904|rbo@cpan.org|Devel-CheckOS|1.53|i86pc-solaris|5.8.9|solaris|2.11|200904161252
3700606|fail|200904|rlane10@wi.rr.com|Devel-CheckOS|1.52|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200904151613
3700604|pass|200904|rbo@cpan.org|Data-Compare|1.21|i86pc-solaris|5.8.9|solaris|2.11|200904161251
3700580|pass|200904|vovkasm@gmail.com|Data-Compare|1.21|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200904161244
3700429|pass|200904|rbo@cpan.org|Devel-CheckOS|1.54|i86pc-solaris|5.8.9|solaris|2.11|200904161253
3700428|fail|200904|rbo@cpan.org|Devel-CheckOS|1.52|i86pc-solaris|5.8.9|solaris|2.11|200904161252
3700255|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904161257
3700181|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.22|i86pc-solaris-64int|5.10.0|solaris|2.11|200904161247
3700159|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.55|i86pc-solaris-64int|5.10.0|solaris|2.11|200904161243
3700142|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904161239
3700137|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904161238
3700046|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904161220
3700013|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904161213
3699890|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904161154
3699865|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904161151
3699835|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.55|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904161141
3699793|pass|200904|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.55|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904161132
3699689|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904161108
3699634|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904161058
3699452|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.54|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904160924
3699448|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.53|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904160924
3699446|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904160923
3699436|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.54|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904160921
3699435|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.53|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904160921
3699434|fail|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904160921
3699412|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904161016
3699364|pass|200904|srezic@cpan.org|Devel-CheckOS|1.55|amd64-freebsd|5.8.9|freebsd|7.0-release|200904161006
3699362|pass|200904|srezic@cpan.org|Devel-CheckOS|1.55|amd64-freebsd|5.8.8|freebsd|7.0-release|200904161006
3699356|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904160905
3699354|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.22|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904160905
3699352|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904160905
3699349|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.55|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904160905
3699347|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.21|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904160904
3699335|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904160901
3699334|fail|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904160901
3699309|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904160856
3699305|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904160855
3699302|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904160855
3699296|fail|200904|bingos@cpan.org|Devel-CheckOS|1.54|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904160854
3699294|pass|200904|bingos@cpan.org|Devel-CheckOS|1.54|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904160853
3699130|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904160920
3699108|pass|200904|stro@cpan.org|Devel-CheckOS|1.55|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904161014
3699083|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904160905
3698962|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904160810
3698936|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904160756
3698931|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904160656
3698928|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904160655
3698921|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904160654
3698918|fail|200904|bingos@cpan.org|Devel-CheckOS|1.53|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904160653
3698917|pass|200904|bingos@cpan.org|Devel-CheckOS|1.53|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904160653
3698897|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904160744
3698868|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904160731
3698660|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904160513
3698659|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904160612
3698651|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904160510
3698644|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904160509
3698638|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904160506
3698635|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904160505
3698629|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904160502
3698622|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904160501
3698621|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.22|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904160501
3698615|pass|200904|bingos@cpan.org|Devel-CheckOS|1.55|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904160458
3698588|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.55|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904161250
3698542|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904160440
3698515|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904160535
3698483|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904160416
3698229|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904160318
3698225|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904160317
3698221|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904160317
3698216|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904160316
3698215|fail|200904|bingos@cpan.org|Devel-CheckOS|1.52|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904160315
3698196|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904160311
3698182|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904160308
3698170|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904160406
3698167|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904160305
3698154|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904160302
3698066|fail|200904|bingos@cpan.org|Devel-CheckOS|1.51|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904160241
3697852|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.53|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904160955
3697839|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|CPAN-FindDependencies|2.22|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904160953
3697696|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.55|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904160216
3697695|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.54|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904160212
3697691|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.55|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904160217
3697689|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.55|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904160217
3697687|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.55|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904160217
3697686|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.55|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904160216
3697681|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.55|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904160216
3697679|cpan|200904|DCANTRELL|Devel-CheckOS|1.55||0|||200904160215
3697672|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.54|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904160213
3697667|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.54|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904160213
3697664|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.54|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904160213
3697661|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.54|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904160212
3697658|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.54|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904160212
3697653|cpan|200904|DCANTRELL|Devel-CheckOS|1.54||0|||200904160211
3697443|fail|200904|david@cantrell.org.uk|Devel-CheckOS|1.52|alpha-netbsd|5.10.0|netbsd|4.0.1|200904160029
3697051|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.22|amd64-freebsd|5.8.9|freebsd|7.0-release|200904152339
3697047|pass|200904|srezic@cpan.org|Devel-CheckOS|1.52|amd64-freebsd|5.8.9|freebsd|7.0-release|200904152339
3696985|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.53|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904160010
3696942|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.53|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904160010
3696933|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.53|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904160010
3696932|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.53|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904160009
3696929|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.53|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904160009
3696927|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.53|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904160009
3696921|cpan|200904|DCANTRELL|Devel-CheckOS|1.53||0|||200904160008
3696881|fail|200904|srezic@cpan.org|Devel-CheckOS|1.52|amd64-freebsd|5.10.0|freebsd|7.0-release|200904152339
3696760|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904152341
3696740|pass|200904|srezic@cpan.org|Devel-CheckOS|1.52|amd64-freebsd|5.8.8|freebsd|7.0-release|200904152339
3696737|pass|200904|srezic@cpan.org|CPAN-FindDependencies|2.22|amd64-freebsd|5.8.8|freebsd|7.0-release|200904152339
3696638|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904152324
3696610|fail|200904|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.52|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904152322
3696603|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.22|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904152321
3696534|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200904152309
3696474|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904152156
3696455|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.52|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904160549
3696445|fail|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|CPAN-FindDependencies|2.21|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904160547
3696211|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.22|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904152150
3696204|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.22|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904152149
3696198|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.22|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904152147
3696185|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.22|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904152146
3696173|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.22|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904152145
3696172|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.22|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904152145
3696161|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.22||0|||200904152143
3696112|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904152134
3695991|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904152113
3695971|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904152000
3695966|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904151959
3695963|pass|200904|bingos@cpan.org|File-Overwrite|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904151958
3695962|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200904151958
3695873|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.21|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904152033
3695871|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.52|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904152040
3695825|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.52|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904152040
3695822|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.52|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904152040
3695818|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.52|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904152039
3695814|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.52|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904152039
3695809|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.21|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904152038
3695802|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.21|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904152037
3695792|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.21|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904152036
3695784|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.21|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904152034
3695776|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.52|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904152033
3695769|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.21|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904152033
3695763|cpan|200904|DCANTRELL|Devel-CheckOS|1.52||0|||200904152031
3695760|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.21||0|||200904152031
3695614|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.01|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904152003
3695611|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904152002
3695533|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Overwrite|1.1|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904151943
3695317|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904151204
3695260|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904151157
3695208|pass|200904|andy@hexten.net|Data-Hexdumper|2.01|i686-linux-thread-multi|5.10.0|linux|2.6.28-11-generic|200904151752
3695030|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904151821
3695019|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904151719
3695015|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904151718
3695012|pass|200904|bingos@cpan.org|File-Overwrite|1.1|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904151717
3695011|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904151716
3694843|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904151746
3694762|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.2|i86pc-solaris-64int|5.10.0|solaris|2.11|200904151733
3694404|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.51|i86pc-solaris-64int|5.8.9|solaris|2.11|200904151644
3694378|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.51|i86pc-solaris-64int|5.10.0|solaris|2.11|200904151642
3694355|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200904151640
3694346|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.1|amd64-freebsd|5.10.0|freebsd|7.0-release|200904151639
3694242|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|alpha-netbsd|5.10.0|netbsd|4.0.1|200904151425
3694015|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|alpha-netbsd|5.10.0|netbsd|4.0.1|200904151340
3693957|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904151415
3693866|pass|200904|hohokus@gmail.com|Devel-CheckOS|1.51|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200904150913
3693832|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904151507
3693827|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904151506
3693788|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904151400
3693785|pass|200904|bingos@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904151359
3693782|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904151359
3693736|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904151446
3693711|pass|200904|rbo@cpan.org|Devel-CheckOS|1.51|i86pc-solaris|5.8.9|solaris|2.11|200904150933
3693685|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904151438
3693548|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904151416
3693395|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|IP35-irix-64int|5.8.8|irix|6.5|200904151359
3693389|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904151259
3693374|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|IP35-irix-64int|5.8.8|irix|6.5|200904151357
3693370|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904151355
3693348|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904151253
3693332|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904151252
3693331|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904151252
3693326|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904151252
3693322|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904151252
3693164|pass|200904|vovkasm@gmail.com|Devel-CheckOS|1.51|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200904151423
3692814|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904151230
3692765|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904151219
3692721|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904151214
3692688|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904151209
3692643|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200904151200
3692624|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904151103
3692612|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904151102
3692610|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.2|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904151102
3692582|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904151159
3692469|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.2|amd64-freebsd|5.8.8|freebsd|7.0-release|200904151136
3692454|pass|200904|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.51|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904151736
3692451|fail|200904|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904151736
3692406|pass|200904|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.51|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904151126
3692403|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904151125
3692395|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.2|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904151124
3692360|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904151117
3692304|pass|200904|david@cantrell.org.uk|Devel-CheckOS|1.51|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904151003
3692215|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904151038
3692177|pass|200904|srezic@cpan.org|Devel-CheckOS|1.51|amd64-freebsd|5.8.9|freebsd|7.0-release|200904150956
3692160|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.2|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904151026
3692064|pass|200904|srezic@cpan.org|Devel-CheckOS|1.51|amd64-freebsd|5.8.8|freebsd|7.0-release|200904150955
3692050|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904150954
3691976|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.51|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904150938
3691796|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904150736
3691690|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904150706
3691685|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904150704
3691680|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904150701
3691669|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.2|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904150658
3691297|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904150508
3691292|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904150508
3691289|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904150507
3691282|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904150506
3691281|pass|200904|bingos@cpan.org|Devel-CheckOS|1.51|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904150506
3691104|fail|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|CPAN-FindDependencies|2.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904151219
3690870|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-Scalar-Decay|1.1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904150413
3690863|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-Scalar-Decay|1.1.1|x86_64-linux|5.10.0|linux|2.6.26-1-amd64|200904150412
3690836|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|x86_64-linux|5.10.0|linux|2.6.26-1-amd64|200904150405
3690766|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904150349
3690486|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904150132
3690482|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904150131
3690480|pass|200904|bingos@cpan.org|File-Overwrite|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904150131
3690479|pass|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904150130
3690235|pass|200904|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.51|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200904150832
3690177|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904150111
3690164|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904150115
3690156|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904150114
3690149|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904150113
3690145|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904150112
3690140|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.2|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904150111
3690126|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.2||0|||200904150108
3689645|pass|200904|bingos@cpan.org|XML-Tiny|2.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904142219
3689581|pass|200904|chris@bingosnet.co.uk|CPAN-FindDependencies|2.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200904142204
3689445|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904142121
3689420|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.51|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904142217
3689418|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.51|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904142216
3689416|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.51|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904142216
3689414|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.51|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904142216
3689410|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.51|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904142215
3689380|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.51|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904142158
3689350|cpan|200904|DCANTRELL|Devel-CheckOS|1.51||0|||200904142157
3689111|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904142102
3689000|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904142030
3688168|pass|200904|andy@hexten.net|Data-Hexdumper|2.01|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-generic|200904141611
3687598|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904141418
3687414|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200904141441
3687408|fail|200904|srezic@cpan.org|CPAN-FindDependencies|2.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200904141441
3687369|pass|200904|srezic@cpan.org|Palm-Treo680MessagesDB|1.01|amd64-freebsd|5.8.9|freebsd|7.0-release|200904141439
3687345|pass|200904|srezic@cpan.org|File-Overwrite|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200904141439
3687289|pass|200904|srezic@cpan.org|Palm-Treo680MessagesDB|1.01|amd64-freebsd|5.8.8|freebsd|7.0-release|200904141439
3687255|pass|200904|srezic@cpan.org|File-Overwrite|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200904141439
3687107|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904141257
3686714|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904141231
3685710|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904140738
3685611|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904140556
3685129|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904140321
3685123|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904140320
3685120|pass|200904|bingos@cpan.org|File-Overwrite|1.1|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904140319
3685119|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200904140319
3685097|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904140317
3684604|pass|200904|mw487.yahoo.com@gmail.com|CPAN-FindDependencies|2.1|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904140025
3683778|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.01|i86pc-solaris-64int|5.8.9|solaris|2.11|200904132322
3683773|pass|200904|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200904132322
3683680|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200904132306
3683427|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904132100
3683424|pass|200904|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904132059
3683420|pass|200904|bingos@cpan.org|File-Overwrite|1.1|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904132059
3683418|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904132058
3682970|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904131945
3682785|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904131854
3682605|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.01|i86pc-solaris-64int|5.10.0|solaris|2.11|200904131757
3682604|pass|200904|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200904131757
3682515|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200904131742
3682211|pass|200904|rlane10@wi.rr.com|CPAN-FindDependencies|2.1|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200904121400
3682126|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904131554
3682050|pass|200904|rlane10@wi.rr.com|XML-Tiny|2.02|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200904130132
3681861|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904131503
3681236|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904131004
3681040|pass|200904|wiml@hhhh.org ("Wim Lewis")|XML-DoubleEncodedEntities|1.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904130612
3680714|pass|200904|mw487.yahoo.com@gmail.com|Palm-Treo680MessagesDB|1.01|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904130252
3680698|pass|200904|mw487.yahoo.com@gmail.com|Data-Hexdumper|2.01|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904130243
3680687|pass|200904|mw487.yahoo.com@gmail.com|File-Overwrite|1.1|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904130235
3680428|pass|200904|wiml@hhhh.org ("Wim Lewis")|Palm-Treo680MessagesDB|1.01|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904130020
3680421|pass|200904|wiml@hhhh.org ("Wim Lewis")|File-Overwrite|1.1|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904130014
3680401|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904130004
3680213|fail|200904|wiml@hhhh.org ("Wim Lewis")|CPAN-FindDependencies|2.1|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904121332
3680058|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|IP35-irix-64int|5.8.8|irix|6.5|200904122346
3679985|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904122256
3679848|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.1|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200904122207
3679791|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904122029
3679622|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|alpha-netbsd|5.10.0|netbsd|4.0.1|200904121834
3679336|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904121852
3678549|pass|200904|rbo@cpan.org|File-Overwrite|1.1|i86pc-solaris|5.8.9|solaris|2.11|200904111029
3678342|pass|200904|rbo@cpan.org|Data-Hexdumper|2.01|i86pc-solaris|5.8.9|solaris|2.11|200904111029
3678067|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904121309
3677760|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904120544
3677718|fail|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904121140
3677715|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|IP35-irix-64int|5.8.8|irix|6.5|200904121239
3677705|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904121138
3677694|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904121136
3677690|pass|200904|david@cantrell.org.uk|CPAN-FindDependencies|2.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904121136
3677679|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|IP35-irix-64int|5.8.8|irix|6.5|200904121233
3677677|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904120533
3677585|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904121218
3677582|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904121218
3677558|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904121214
3677215|fail|200904|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904121654
3677176|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904120949
3676811|pass|200904|stro@cpan.org|Palm-Treo680MessagesDB|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904121020
3676770|pass|200904|stro@cpan.org|File-Overwrite|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904121012
3676567|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904120727
3676555|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904120724
3676538|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904120721
3676520|fail|200904|bingos@cpan.org|CPAN-FindDependencies|2.1|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904120718
3675939|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904120451
3675877|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904120543
3675380|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904120326
3674725|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904120026
3674685|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904120011
3674592|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904112246
3674568|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904112239
3674302|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904112236
3674301|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904112136
3674255|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904112228
3673976|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904112031
3673524|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|alpha-netbsd|5.10.0|netbsd|4.0.1|200904111808
3673462|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|alpha-netbsd|5.10.0|netbsd|4.0.1|200904111759
3673347|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.1|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904111945
3673341|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904111944
3673330|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.1|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904111943
3673327|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904111941
3673322|fail|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-FindDependencies|2.1|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904111940
3673312|cpan|200904|DCANTRELL|CPAN-FindDependencies|2.1||0|||200904111938
3673158|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904111849
3672303|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904111514
3671989|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904111425
3671987|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904111425
3671986|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904111425
3671968|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904111423
3671966|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904111422
3671958|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200904111422
3671933|pass|200904|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904111420
3671921|pass|200904|david@cantrell.org.uk|File-Overwrite|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904111418
3671841|pass|200904|stro@cpan.org|CPAN-ParseDistribution|1.1|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.00|200904110604
3671812|na|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904111457
3670972|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200904111148
3670787|pass|200904|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.01|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904111705
3670786|pass|200904|imacat@mail.imacat.idv.tw|File-Overwrite|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904111703
3670581|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904110950
3670344|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904110838
3670253|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904110808
3669983|pass|200904|rbo@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris|5.8.9|solaris|2.11|200904110110
3669701|pass|200904|stsc@refcnt.org ("Steven Schubiger")|Palm-Treo680MessagesDB|1.01|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904110547
3669697|pass|200904|stsc@refcnt.org ("Steven Schubiger")|File-Overwrite|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904110546
3669587|na|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904110524
3668637|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904110025
3668344|pass|200904|bingos@cpan.org|File-Overwrite|1.1|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904102315
3668341|pass|200904|bingos@cpan.org|File-Overwrite|1.1|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904102315
3668336|pass|200904|bingos@cpan.org|File-Overwrite|1.1|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904102314
3668333|pass|200904|bingos@cpan.org|File-Overwrite|1.1|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904102314
3668332|pass|200904|bingos@cpan.org|File-Overwrite|1.1|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904102314
3668313|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904102309
3668307|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904102307
3668303|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904102305
3668290|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904102303
3668285|pass|200904|bingos@cpan.org|Palm-Treo680MessagesDB|1.01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904102302
3667997|pass|200904|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904102211
3667850|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904102241
3667809|pass|200904|david@cantrell.org.uk|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904102136
3667265|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904102034
3667070|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904101935
3666693|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904101757
3666533|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Overwrite|1.1|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904101717
3666532|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Overwrite|1.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904101716
3666530|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Overwrite|1.1|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904101716
3666529|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Overwrite|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904101716
3666526|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Overwrite|1.1|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904101716
3666523|cpan|200904|DCANTRELL|File-Overwrite|1.1||0|||200904101715
3666422|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-Treo680MessagesDB|1.01|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904101649
3666419|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-Treo680MessagesDB|1.01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904101648
3666418|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-Treo680MessagesDB|1.01|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904101648
3666412|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-Treo680MessagesDB|1.01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904101647
3666408|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-Treo680MessagesDB|1.01|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904101647
3666403|cpan|200904|DCANTRELL|Palm-Treo680MessagesDB|1.01||0|||200904101646
3665463|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904101317
3665462|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904101316
3665282|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904101142
3663135|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904100259
3663130|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904100257
3663113|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904100254
3663059|pass|200904|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904100252
3661720|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|darwin-64int-2level|5.11.0|darwin|8.11.0|200904092115
3661200|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904091955
3661194|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904091953
3661190|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904091951
3660953|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904091859
3660950|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904091859
3660936|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904091856
3659521|pass|200904|rbo@consol.de|CPAN-ParseDistribution|1.1|i86pc-solaris|5.8.9|solaris|2.11|200904091411
3659207|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i686-linux-64int-ld|5.11.0|linux|2.6.16.38|200904091315
3659046|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904090625
3657898|na|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904090900
3657444|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904090703
3657442|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904090703
3656711|pass|200904|dwheeler@cpan.org|Tie-STDOUT|1.0401|darwin-2level|5.10.0|darwin|9.4.0|200904090220
3656269|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904090258
3656214|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904090249
3656098|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904090224
3655288|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-64int-ld|5.11.0|linux|2.6.16.38|200904082327
3654298|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904082250
3653457|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200904081927
3652872|pass|200904|dwheeler@cpan.org|Tie-STDOUT|1.0401|i686-linux|5.10.0|linux|2.6.22-14-server|200904080943
3652871|pass|200904|dwheeler@cpan.org|Tie-STDOUT|1.0401|i686-linux|5.10.0|linux|2.6.22-14-server|200904080954
3651762|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200904081425
3651243|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|IP35-irix-64int|5.8.8|irix|6.5|200904081359
3650941|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200904081220
3649671|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-ParseDistribution|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904080929
3649102|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200904080645
3647539|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200904080156
3647537|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200904080155
3646314|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904072225
3645865|pass|200904|andy@hexten.net|XML-Tiny|2.02|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200904072111
3645341|pass|200904|andy@hexten.net|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200904072008
3645311|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904072005
3645140|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904071949
3645002|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904071931
3644998|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200904071930
3644809|pass|200904|vpit@cpan.org|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904072002
3643446|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904071410
3643421|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200904071505
3643337|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200904071452
3643334|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904071351
3643326|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904071350
3643257|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200904071440
3643160|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904071425
3642897|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200904071341
3642682|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904071314
3642489|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200904071249
3642009|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904071147
3641619|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904070943
3641616|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200904070943
3640896|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200904070835
3640599|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200904070646
3640458|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904070559
3640428|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904070543
3640031|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904070350
3639943|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904070330
3639874|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200904070314
3639841|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904070308
3639839|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904070307
3639750|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.10.0|linux|2.6.18-92.el5|200904070253
3639671|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904070144
3639258|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904070125
3639113|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904070103
3638592|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200904062356
3638390|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904062331
3638137|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200904062316
3638024|fail|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-ParseDistribution|1.1|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904062301
3637732|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904062128
3637726|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200904062127
3636897|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200904061911
3635562|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904061337
3635492|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904061420
3635458|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200904061318
3635443|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200904061316
3635202|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904061240
3635115|pass|200904|stro@cpan.org|CPAN-ParseDistribution|1.1|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904061419
3635106|pass|200904|stro@cpan.org|CPAN-ParseDistribution|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200904061418
3634380|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904060948
3634376|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200904060949
3634114|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904060955
3633915|fail|200904|wiml@hhhh.org ("Wim Lewis")|CPAN-ParseDistribution|1.1|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200904060702
3633731|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904060817
3633726|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904060815
3633672|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200904060656
3633340|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904060524
3633189|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904060535
3633188|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904060534
3633178|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904060532
3632981|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904060348
3632410|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200904060150
3632405|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200904060149
3632095|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|alpha-netbsd|5.10.0|netbsd|4.0.1|200904060047
3631909|fail|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904060019
3631872|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904060012
3631841|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904060003
3631839|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200904060002
3631667|pass|200904|stro@cpan.org|CPAN-ParseDistribution|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904060130
3630964|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904052201
3630962|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904052200
3630957|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904052157
3630881|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-ParseDistribution|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200904052136
3630802|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-ParseDistribution|1.1|i86pc-solaris-64int|5.10.0|solaris|2.11|200904052110
3630586|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904051943
3630585|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904051943
3630550|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904051940
3629968|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904051104
3629950|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904051636
3629655|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904051007
3629363|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904051554
3629350|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200904051451
3629249|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.1|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904051436
3628529|pass|200904|gbnewby@pglaf.org|Palm-ProjectGutenberg|1.0|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200904041432
3628489|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904051248
3628488|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904050715
3628484|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200904051247
3628483|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200904050713
3628315|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200904051221
3627914|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904051201
3627619|pass|200904|imacat@mail.imacat.idv.tw|CPAN-ParseDistribution|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904051702
3627363|pass|200904|srezic@cpan.org|CPAN-ParseDistribution|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200904050733
3627237|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904050906
3627107|pass|200904|srezic@cpan.org|CPAN-ParseDistribution|1.1|amd64-freebsd|5.10.0|freebsd|7.0-release|200904050646
3627105|pass|200904|srezic@cpan.org|CPAN-ParseDistribution|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200904050646
3626985|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904050812
3626468|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200904050446
3626467|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904050545
3626460|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200904050443
3626411|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200904050529
3626151|pass|200904|mw487.yahoo.com@gmail.com|CPAN-ParseDistribution|1.1|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904050206
3626144|pass|200904|mw487.yahoo.com@gmail.com|CPAN-ParseDistribution|1.0|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904050203
3625954|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904050300
3624896|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904042142
3624520|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904042025
3624218|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904041950
3624211|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904041949
3624209|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904041948
3624201|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.1|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904041947
3624010|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200904041918
3623642|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904041853
3623390|pass|200904|imacat@mail.imacat.idv.tw|CPAN-ParseDistribution|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200904050025
3623373|pass|200904|vpit@cpan.org|Devel-CheckLib|0.5|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904041825
3622967|pass|200904|vpit@cpan.org|CPAN-ParseDistribution|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200904041740
3622955|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200904041037
3622917|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200904041534
3622465|pass|200904|stsc@refcnt.org ("Steven Schubiger")|CPAN-ParseDistribution|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200904041634
3622234|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200904041256
3621743|fail|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200904041401
3621740|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|IP35-irix-64int|5.8.8|irix|6.5|200904041500
3621707|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200904041355
3621706|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200904041354
3621703|pass|200904|david@cantrell.org.uk|CPAN-ParseDistribution|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200904041354
3621175|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200904041231
3621125|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200904041324
3621107|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-ParseDistribution|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200904041323
3620876|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.1|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200904041249
3620872|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.1|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200904041248
3620869|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.1|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200904041248
3620865|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.1|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200904041247
3620861|pass|200904|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.1|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200904041247
3620843|cpan|200904|DCANTRELL|CPAN-ParseDistribution|1.1||0|||200904041245
3620694|pass|200904|JOST@cpan.org ("Josts Smokehouse")|CPAN-ParseDistribution|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200904041222
3620321|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904041102
3619818|pass|200904|stro@cpan.org|CPAN-ParseDistribution|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200904041027
3619170|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904040529
3618574|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904040115
3618573|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904040115
3618569|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904040114
3618566|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904040114
3618277|na|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904040114
3618080|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200904032301
3618034|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904040011
3618032|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200904040010
3617530|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904032208
3617528|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200904032207
3616172|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904031524
3616166|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904031524
3616164|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904031523
3616162|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904031523
3616110|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200904031517
3615943|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904031548
3615941|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.6.2|linux|2.6.27.19-5-default|200904031548
3615882|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200904031539
3615737|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200904031416
3615594|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200904031454
3615019|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200904031219
3615017|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200904031219
3615015|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200904031218
3615014|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200904031219
3614296|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200904031032
3612953|pass|200904|mw487.yahoo.com@gmail.com|Devel-CheckLib|0.5|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200904030345
3612661|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200904030220
3612599|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904030204
3612594|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904030204
3612589|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904030204
3612586|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904030203
3612451|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904030237
3612304|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200904030106
3612224|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200904030145
3612013|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200904030058
3611855|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904030024
3611311|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904022227
3611184|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200904022028
3611182|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200904022028
3611181|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200904022028
3610818|pass|200904|bingos@cpan.org|Devel-CheckLib|0.5|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904021855
3610654|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200904021824
3610573|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904021909
3609947|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904021720
3609715|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904021648
3609434|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904021455
3609430|pass|200904|bingos@cpan.org|XML-Tiny|2.02|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904021454
3609427|pass|200904|bingos@cpan.org|XML-Tiny|2.01|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904021453
3609426|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904021453
3608920|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904021323
3608917|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904021323
3608915|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904021322
3608913|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904021322
3608783|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200904021255
3608456|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200904021148
3608085|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904021139
3608076|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200904021138
3607793|pass|200904|robertl@apnic.net ("Robert Loomans")|Devel-CheckOS|1.50|darwin-2level|5.8.9|darwin|9.6.0|200904021324
3607633|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904021027
3606847|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200904020743
3606611|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904020700
3606229|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200904020534
3605961|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200904020326
3605959|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200904020325
3605956|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200904020325
3605954|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200904020325
3605913|pass|200904|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904020311
3605912|pass|200904|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904020311
3605911|pass|200904|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904020310
3605909|pass|200904|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904020310
3605817|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200904020337
3605518|pass|200904|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200904020140
3604365|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904012126
3604363|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904012126
3604311|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200904012112
3604078|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200904012003
3603920|pass|200904|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200904011913
3603860|unknown|200904|JOST@cpan.org ("Josts Smokehouse")|Acme-Pony|1.1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200904011903
3603155|pass|200904|srezic@cpan.org|CPAN-ParseDistribution|1.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200904011512
3602020|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904010952
3602019|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904010952
3602008|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200904010949
3601744|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200904010901
3601437|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904010735
3601436|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904010735
3601360|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200904010719
3601310|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904010701
3601308|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904010700
3601251|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200904010645
3600555|unknown|200904|JOST@cpan.org ("Josts Smokehouse")|Acme-Licence|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200904010340
3600460|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200904010225
3600458|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200904010224
3600452|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200904010224
3600447|fail|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200904010223
3600446|pass|200904|bingos@cpan.org|CPAN-ParseDistribution|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200904010223
3600381|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904010308
3600378|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904010307
3600318|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200904010256
3600096|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904010131
3600092|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904010131
3600007|pass|200904|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200904010109
3599358|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200903312122
3599357|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200903312122
3599296|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200903312058
3599216|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903312027
3599215|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903312027
3599123|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903312016
3599005|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|CPAN-FindDependencies|2.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200903311949
3599000|pass|200903|vpit@cpan.org|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903311950
3598489|fail|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903311758
3598426|pass|200903|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200903311745
3598037|pass|200903|JOST@cpan.org ("Josts Smokehouse")|NestedMap|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903311601
3597540|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Statistics-ChiSquare|0.5|i86pc-solaris-64int|5.8.9|solaris|2.11|200903311348
3597535|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Longest|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200903311348
3597111|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903311237
3596921|pass|200903|srezic@cpan.org|CPAN-ParseDistribution|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200903311140
3596674|pass|200903|srezic@cpan.org|CPAN-ParseDistribution|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200903311022
3596610|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903310949
3594939|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903310035
3594935|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903310035
3594932|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903310035
3594930|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903310034
3594740|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903310102
3594739|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903310102
3594737|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903310101
3594735|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903310101
3594734|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|CPAN-ParseDistribution|1.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903310101
3594727|cpan|200903|DCANTRELL|CPAN-ParseDistribution|1.0||0|||200903310100
3594715|pass|200903|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903302358
3593460|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903302108
3593115|pass|200903|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.10.0|freebsd|7.0-release|200903301938
3592645|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903301725
3592550|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903301631
3592316|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903301425
3592268|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903301410
3591761|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200903301050
3590365|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300245
3590364|pass|200903|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300244
3590362|pass|200903|bingos@cpan.org|XML-Tiny|2.01|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300244
3590358|pass|200903|bingos@cpan.org|XML-Tiny|2.0|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300243
3590346|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300243
3590330|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300240
3590214|pass|200903|dpavlin@mail.carnet.hr ((Dobrica Pavlinusic))|XML-Tiny|2.02|i486-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-686|200903291243
3590024|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903300139
3590022|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903300139
3590019|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903300139
3590016|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903300138
3590013|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903300138
3590005|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903300137
3589667|pass|200903|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.10.0 patch 35117|darwin|8.11.0|200903300020
3589441|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903292307
3589438|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903292307
3589434|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903292306
3589430|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903292306
3589424|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903292305
3588299|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-TreoPhoneCallDB|1.1|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903291947
3588290|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903291943
3587659|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903291708
3587657|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903291707
3587445|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200903291553
3587202|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903291437
3587117|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200903291405
3587096|pass|200903|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903291258
3587090|pass|200903|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903291257
3587086|pass|200903|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903291256
3587082|pass|200903|bingos@cpan.org|Devel-CheckLib|0.5|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903291255
3586908|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200903291305
3586763|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200903291146
3586596|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200903291145
3586590|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903291142
3586589|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903291141
3586482|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200903291132
3586479|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200903291131
3586415|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903291116
3586405|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903291116
3586140|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200903290842
3586133|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200903290841
3585613|pass|200903|jrockway@cpan.org|Devel-CheckOS|1.50|x86_64-linux|5.10.0|linux|2.6.22-3-amd64|200903290223
3585039|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903290725
3584925|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903290703
3584921|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903290702
3584614|pass|200903|wiml@hhhh.org ("Wim Lewis")|XML-Tiny-DOM|1.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903290354
3584613|pass|200903|wiml@hhhh.org ("Wim Lewis")|XML-Tiny|2.02|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903290354
3584480|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200903290511
3584453|pass|200903|wiml@hhhh.org ("Wim Lewis")|Palm-ProjectGutenberg|1.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903290302
3584223|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903290346
3584068|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903290215
3584067|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903290215
3584065|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903290214
3584063|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903290214
3584060|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903290213
3584059|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903290213
3584057|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903290213
3584056|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903290213
3584054|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903290212
3584053|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903290212
3584046|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903290210
3584044|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903290210
3583310|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200903282254
3582472|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200903281951
3582251|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903281900
3582218|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.8.8|linux|2.6.27.19-5-default|200903281847
3582142|pass|200903|mw487.yahoo.com@gmail.com|XML-Tiny-DOM|1.0|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903281228
3582122|pass|200903|mw487.yahoo.com@gmail.com|Palm-ProjectGutenberg|1.0|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903281219
3581947|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.8.9|linux|2.6.27.19-5-default|200903281733
3581939|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200903281730
3581937|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200903281730
3581811|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903281710
3581810|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux-thread-multi-perlio|5.6.2|linux|2.6.27.19-5-default|200903281709
3581807|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200903281709
3581806|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390x-linux|5.6.2|linux|2.6.27.19-5-default|200903281709
3581646|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390x-linux|5.10.0|linux|2.6.27.19-5-default|200903281617
3581513|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.27.19-5-default|200903281541
3581327|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.9|linux|2.6.27.19-5-default|200903281511
3581301|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903281506
3581125|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200903281437
3581058|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.10.0|linux|2.6.27.19-5-default|200903281428
3580816|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200903281344
3580618|pass|200903|vpit@cpan.org|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903281310
3579993|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903281104
3579381|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903280830
3579379|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903280830
3579378|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903280829
3579377|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903280829
3579374|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903280829
3579366|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903280828
3577869|pass|200903|stro@cpan.org|XML-Tiny|2.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|4.0|200903280542
3577608|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903280342
3577583|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903280328
3577575|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903280324
3577573|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903280324
3577450|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903280242
3577393|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903280229
3577308|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-ProjectGutenberg|1.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903280207
3577010|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903280004
3577009|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903280004
3577008|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903280004
3577006|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903280003
3577004|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903280003
3577002|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903280002
3576703|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903272234
3576699|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903272233
3576697|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903272233
3576696|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903272232
3576695|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903272232
3576693|pass|200903|bingos@cpan.org|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903272232
3576692|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903272231
3576690|pass|200903|bingos@cpan.org|XML-Tiny|2.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903272231
3576687|pass|200903|bingos@cpan.org|XML-Tiny|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903272230
3576684|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903272230
3576682|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903272229
3576677|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903272228
3576611|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903272313
3576573|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903272304
3576535|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny-DOM|1.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903272253
3576094|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200903272138
3575715|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903271834
3575386|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903271648
3575190|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903270951
3575183|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903270950
3575178|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|IP35-irix-64int|5.8.8|irix|6.5|200903271549
3575165|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|IP35-irix-64int|5.8.8|irix|6.5|200903271548
3575157|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903271446
3575155|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903271446
3575152|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903271446
3575150|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903271446
3575139|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903271443
3575131|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903271443
3575128|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903271442
3575127|pass|200903|david@cantrell.org.uk|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903271442
3575087|pass|200903|imacat@mail.imacat.idv.tw|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903272207
3575086|pass|200903|imacat@mail.imacat.idv.tw|Palm-ProjectGutenberg|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903272207
3574992|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny-DOM|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200903271407
3574948|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Palm-ProjectGutenberg|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200903271359
3574754|pass|200903|srezic@cpan.org|XML-Tiny|2.02|amd64-freebsd|5.10.0|freebsd|7.0-release|200903271248
3574752|pass|200903|srezic@cpan.org|XML-Tiny-DOM|1.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200903271248
3574721|pass|200903|david@cantrell.org.uk|XML-Tiny-DOM|1.0|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903271155
3574616|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny-DOM|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903271242
3574597|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Palm-ProjectGutenberg|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903271236
3574522|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903271207
3574412|pass|200903|srezic@cpan.org|Palm-ProjectGutenberg|1.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200903271144
3574389|pass|200903|srezic@cpan.org|XML-Tiny-DOM|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200903271139
3574386|pass|200903|srezic@cpan.org|Palm-ProjectGutenberg|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200903271139
3574273|pass|200903|srezic@cpan.org|XML-Tiny-DOM|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200903271109
3574271|pass|200903|srezic@cpan.org|Palm-ProjectGutenberg|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200903271109
3574086|pass|200903|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny-DOM|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903271010
3574074|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Palm-ProjectGutenberg|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903271006
3573661|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-ProjectGutenberg|1.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903270714
3573653|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-ProjectGutenberg|1.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903270715
3573650|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-ProjectGutenberg|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903270714
3573648|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-ProjectGutenberg|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903270714
3573647|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-ProjectGutenberg|1.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903270713
3573416|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903270432
3573412|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903270431
3573410|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903270430
3573406|pass|200903|bingos@cpan.org|Palm-ProjectGutenberg|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903270429
3573322|pass|200903|mw487.yahoo.com@gmail.com|XML-Tiny|2.02|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903270408
3573302|pass|200903|mw487.yahoo.com@gmail.com|XML-Tiny|2.01|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903270404
3573289|pass|200903|mw487.yahoo.com@gmail.com|XML-Tiny|2.0|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903270359
3573273|pass|200903|mw487.yahoo.com@gmail.com|Tie-STDOUT|1.0401|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903270354
3573267|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903270450
3573263|pass|200903|mw487.yahoo.com@gmail.com|App-Rsnapshot|1.999_00002|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903270349
3573233|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903270437
3573092|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903270225
3573091|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903270224
3573090|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903270224
3573087|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903270223
3573086|pass|200903|bingos@cpan.org|XML-Tiny-DOM|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903270223
3572566|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903262207
3572544|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903262208
3572542|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903262208
3572541|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903262208
3572539|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny-DOM|1.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903262207
3572458|cpan|200903|DCANTRELL|Palm-ProjectGutenberg|1.0||0|||200903262148
3572382|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903262109
3572189|fail|200903|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903261438
3572107|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|alpha-netbsd|5.10.0|netbsd|4.0.1|200903261745
3572078|cpan|200903|DCANTRELL|XML-Tiny-DOM|1.0||0|||200903261822
3571722|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903261630
3571513|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903261537
3571424|pass|200903|vpit@cpan.org|XML-Tiny|2.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903261519
3570936|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903261306
3570331|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903260921
3569889|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903260619
3569832|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200903260602
3569685|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.02|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903260448
3569676|pass|200903|stro@cpan.org|XML-Tiny|2.02|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903260543
3569673|pass|200903|stro@cpan.org|XML-Tiny|2.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903260543
3569614|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200903260408
3569364|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200903260232
3569002|na|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.0401|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200903260131
3568971|pass|200903|stro@cpan.org|Tie-STDOUT|1.0401|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903260328
3568750|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200903260210
3568671|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903260159
3568012|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24-16-generic|200903251813
3567935|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903252301
3567789|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903252221
3567755|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200903252209
3567665|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903252155
3567226|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903252029
3566933|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903251423
3565957|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200903251638
3565703|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.10.0|linux|2.6.18-92.el5|200903251535
3565643|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903250926
3565581|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200903251518
3564984|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|IP35-irix-64int|5.8.8|irix|6.5|200903251300
3564872|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903251143
3564868|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903251143
3564861|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903251142
3564847|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903251141
3564403|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200903251051
3564388|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|2.02|i86pc-solaris-64int|5.10.0|solaris|2.11|200903251050
3564369|pass|200903|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903250546
3564250|pass|200903|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903250530
3564228|pass|200903|david@cantrell.org.uk|XML-Tiny|2.02|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903250928
3564100|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|2.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200903251008
3564052|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200903250958
3563825|pass|200903|imacat@mail.imacat.idv.tw|XML-Tiny|2.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903251621
3563141|pass|200903|jeff@zeroclue.com|XML-Tiny|2.01|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903250057
3563098|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200903250709
3562737|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903250551
3562318|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903250329
3562303|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903250325
3561374|pass|200903|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903250022
3561371|pass|200903|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903250022
3561370|pass|200903|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903250021
3561367|pass|200903|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903250021
3561365|pass|200903|bingos@cpan.org|XML-Tiny|2.02|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903250020
3561086|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|XML-Tiny|2.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903250817
3561062|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200903250014
3560711|pass|200903|srezic@cpan.org|XML-Tiny|2.0|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200903242257
3560710|fail|200903|srezic@cpan.org|App-Rsnapshot|1.999_00002|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200903242257
3560630|pass|200903|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|2.02|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903242303
3560610|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.0401|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903241757
3560239|pass|200903|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903242158
3560100|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200903242141
3560064|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200903242134
3560034|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903242129
3559959|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200903242120
3559935|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903242113
3559875|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200903242101
3559860|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200903241559
3559855|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.02|i686-linux|5.8.9|linux|2.6.24-16-generic|200903241557
3559662|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|alpha-netbsd|5.10.0|netbsd|4.0.1|200903241939
3558930|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200903241926
3558927|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.01|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903241925
3558630|pass|200903|srezic@cpan.org|XML-Tiny|2.02|amd64-freebsd|5.8.8|freebsd|7.0-release|200903241856
3558628|pass|200903|srezic@cpan.org|XML-Tiny|2.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200903241856
3558594|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903241353
3558529|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903241346
3558484|unknown|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux|5.8.9|linux|2.6.24-16-generic|200903241341
3558301|pass|200903|vpit@cpan.org|XML-Tiny|2.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903241824
3557365|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903241617
3557364|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903241617
3557362|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903241616
3557359|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903241616
3557357|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.02|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903241616
3557350|cpan|200903|DCANTRELL|XML-Tiny|2.02||0|||200903241615
3556968|unknown|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903241025
3556768|pass|200903|wiml@hhhh.org ("Wim Lewis")|Tie-STDOUT|1.0401|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903240520
3556133|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|IP35-irix-64int|5.8.8|irix|6.5|200903241304
3556057|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200903241247
3556013|pass|200903|srezic@cpan.org|Tie-STDOUT|1.0401|amd64-freebsd|5.10.0|freebsd|7.0-release|200903241134
3555877|pass|200903|imacat@mail.imacat.idv.tw|XML-Tiny|2.01|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903241923
3555674|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903240557
3555546|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903241029
3555544|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903241029
3555538|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903241029
3555528|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903241028
3555426|pass|200903|david@cantrell.org.uk|XML-Tiny|2.01|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903241009
3554502|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|2.01|i86pc-solaris-64int|5.8.9|solaris|2.11|200903240941
3554433|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|2.01|i86pc-solaris-64int|5.10.0|solaris|2.11|200903240929
3554305|pass|200903|d@0x64.nl|XML-Tiny|2.01|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-vserver-amd64|200903240903
3554188|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903240844
3554135|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903240839
3554087|unknown|200903|stsc@refcnt.org ("Steven Schubiger")|Acme-Pony|1.1.2|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903240834
3553704|pass|200903|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|2.01|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903240746
3553294|pass|200903|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903240140
3553106|pass|200903|wiml@hhhh.org ("Wim Lewis")|XML-Tiny|2.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903240507
3552945|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.01|i686-linux|5.8.9|linux|2.6.24-16-generic|200903240036
3552779|pass|200903|stro@cpan.org|XML-Tiny|2.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903240557
3552589|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|XML-Tiny|2.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903241218
3552585|pass|200903|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux|5.8.9|linux|2.6.24-16-generic|200903240318
3552584|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903240317
3552560|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200903240412
3552290|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200903240333
3552246|pass|200903|bingos@cpan.org|XML-Tiny|2.01|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903240228
3552241|pass|200903|bingos@cpan.org|XML-Tiny|2.01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903240228
3552235|pass|200903|bingos@cpan.org|XML-Tiny|2.01|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903240227
3552227|pass|200903|bingos@cpan.org|XML-Tiny|2.01|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903240227
3552224|pass|200903|bingos@cpan.org|XML-Tiny|2.01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903240226
3552169|pass|200903|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903240220
3552096|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200903240305
3552082|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200903240304
3552056|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200903240301
3552004|pass|200903|wiml@hhhh.org ("Wim Lewis")|Tie-STDOUT|1.04|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903240157
3551770|pass|200903|imacat@mail.imacat.idv.tw|XML-Tiny|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903240140
3551318|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903240220
3551079|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200903240155
3550666|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200903240013
3550663|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|alpha-netbsd|5.10.0|netbsd|4.0.1|200903240013
3550185|pass|200903|srezic@cpan.org|XML-Tiny|2.01|amd64-freebsd|5.8.9|freebsd|7.0-release|200903240027
3550179|pass|200903|srezic@cpan.org|XML-Tiny|2.01|amd64-freebsd|5.8.8|freebsd|7.0-release|200903240027
3549959|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903232350
3549789|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903232333
3549697|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903232326
3549583|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903232310
3549574|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903232309
3549316|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903231904
3549305|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903231843
3548747|pass|200903|srezic@cpan.org|XML-Tiny|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200903230936
3548700|pass|200903|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903231709
3548675|fail|200903|srezic@cpan.org|App-Rsnapshot|1.999_00002|amd64-freebsd|5.10.0|freebsd|7.0-release|200903230950
3548634|fail|200903|srezic@cpan.org|App-Rsnapshot|1.999_00002|amd64-freebsd|5.8.9|freebsd|7.0-release|200903230959
3548571|pass|200903|srezic@cpan.org|XML-Tiny|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200903230950
3548272|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903232044
3548140|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903232138
3548134|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.9|linux|2.6.24-16-generic|200903232038
3548000|pass|200903|srezic@cpan.org|XML-Tiny|2.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200903230951
3547996|fail|200903|srezic@cpan.org|App-Rsnapshot|1.999_00002|amd64-freebsd|5.8.8|freebsd|7.0-release|200903230959
3547988|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903232133
3547942|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200903232132
3547644|pass|200903|vpit@cpan.org|XML-Tiny|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903232125
3547641|pass|200903|vpit@cpan.org|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903232124
3547639|pass|200903|vpit@cpan.org|App-Rsnapshot|1.999_00002|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903232124
3547604|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|2.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903232122
3547513|pass|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903231619
3547445|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903231219
3547429|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903232015
3547300|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|2.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903231825
3547241|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903232010
3547067|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903232002
3546847|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903231959
3546789|pass|200903|stro@cpan.org|XML-Tiny|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903231721
3546579|pass|200903|joerg.meltzer@tngtech.com|Statistics-ChiSquare|0.5|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200903202041
3546325|pass|200903|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903231957
3546188|pass|200903|stro@cpan.org|XML-Tiny|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903231721
3546021|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903230850
3546015|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|IP35-irix-64int|5.8.8|irix|6.5|200903231446
3546005|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903231219
3545942|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903231146
3545524|pass|200903|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903231921
3545396|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903231146
3545359|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903231219
3545356|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903231220
3545344|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903231219
3545341|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903231219
3545318|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|IP35-irix-64int|5.8.8|irix|6.5|200903231445
3545305|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00002|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903230850
3545236|pass|200903|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903231914
3545036|pass|200903|stro@cpan.org|Tie-STDOUT|1.0401|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903232010
3544999|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-STDOUT|1.0401|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903231804
3544836|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux|5.8.9|linux|2.6.24-16-generic|200903231902
3544698|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903231220
3544694|pass|200903|david@cantrell.org.uk|XML-Tiny|2.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903231220
3544677|cpan|200903|DCANTRELL|XML-Tiny|2.01||0|||200903231919
3544595|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00002|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903231858
3544589|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00002|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903231857
3544582|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00002|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903231857
3544576|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00002|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903231857
3544566|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903231857
3544542|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903231856
3544536|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903231856
3544523|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903231856
3544518|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903231856
3544510|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00002|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903231856
3544390|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|XML-Tiny|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903231722
3544061|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903231846
3544052|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903231846
3544041|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903231846
3544026|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903231846
3544025|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903231845
3543950|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.01|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903231919
3543947|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903231920
3543940|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.01|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903231920
3543931|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903231919
3543917|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|2.01|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903231920
3543775|pass|200903|bingos@cpan.org|XML-Tiny|2.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903231842
3543767|pass|200903|bingos@cpan.org|XML-Tiny|2.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903231841
3543743|pass|200903|bingos@cpan.org|XML-Tiny|2.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903231841
3543725|pass|200903|bingos@cpan.org|XML-Tiny|2.0|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903231841
3543712|pass|200903|bingos@cpan.org|XML-Tiny|2.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903231841
3543663|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00002|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903231839
3543480|cpan|200903|DCANTRELL|App-Rsnapshot|1.999_00002||0|||200903222104
3543273|cpan|200903|DCANTRELL|XML-Tiny|2.0||0|||200903222031
3543019|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|alpha-netbsd|5.10.0|netbsd|4.0.1|200903221846
3542439|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux|5.8.9|linux|2.6.24-16-generic|200903221302
3541592|pass|200903|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903221019
3540905|unknown|200903|stsc@refcnt.org ("Steven Schubiger")|Acme-Licence|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903221312
3539726|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903220404
3538595|pass|200903|stsc@refcnt.org ("Steven Schubiger")|NestedMap|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903220542
3537940|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903220152
3537731|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903220057
3537553|pass|200903|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux|5.8.9|linux|2.6.24-16-generic|200903211915
3537440|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903212340
3537410|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903212337
3537407|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903212336
3536683|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903212122
3534903|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903211012
3534902|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux|5.8.9|linux|2.6.24-16-generic|200903211012
3534829|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux|5.8.9|linux|2.6.24-16-generic|200903210950
3534550|pass|200903|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux|5.8.9|linux|2.6.24-16-generic|200903210856
3534161|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903210730
3534113|pass|200903|stro@cpan.org|Data-Hexdumper|2.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903211313
3533794|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903210609
3533709|pass|200903|stro@cpan.org|Data-Hexdumper|2.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903211154
3533527|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux-thread-multi|5.8.9|linux|2.6.5-7.315-s390|200903211019
3533420|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux|5.8.9|linux|2.6.5-7.315-s390|200903211005
3533360|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903210959
3533310|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903210954
3533218|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200903210942
3533115|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903210929
3532966|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903210906
3532274|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200903210706
3531128|pass|200903|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903210333
3531127|pass|200903|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200903210333
3530152|pass|200903|jeff@zeroclue.com|XML-Tiny|1.12|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903210049
3528491|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200903202153
3528444|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903202141
3528282|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903202120
3528227|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.0401|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903202116
3528147|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903201608
3527602|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903201955
3526658|pass|200903|jeff@zeroclue.com|Tie-STDOUT|1.0401|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903201232
3526283|pass|200903|mw487.yahoo.com@gmail.com|Tie-STDOUT|1.04|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903201125
3526262|pass|200903|mw487.yahoo.com@gmail.com|App-Rsnapshot|1.999_00001|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903201123
3526141|pass|200903|stsc@refcnt.org ("Steven Schubiger")|File-Overwrite|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903201711
3525648|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.03|i686-linux|5.8.9|linux|2.6.24-16-generic|200903201059
3525455|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903200934
3525361|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|IP35-irix-64int|5.8.8|irix|6.5|200903201527
3525231|pass|200903|bingos@cpan.org|Tie-STDOUT|1.0401|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200903201411
3525228|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200903201411
3525224|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200903201411
3525084|pass|200903|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903200957
3524561|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.12|i686-linux|5.8.9|linux|2.6.24-16-generic|200903200852
3524017|pass|200903|mw487.yahoo.com@gmail.com|Data-Compare|1.21|cygwin-thread-multi-64int|0|cygwin|1.5.25(0.15642)|200903200644
3523774|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.0401|i686-linux|5.8.9|linux|2.6.24-16-generic|200903200709
3523570|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903201050
3523565|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.0401|i86pc-solaris-64int|5.10.0|solaris|2.11|200903201150
3523542|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903201048
3523482|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903201032
3523443|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903201125
3523302|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.0401|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903201002
3522453|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903200831
3521945|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903200649
3521637|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200903200451
3521634|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200903200450
3520943|pass|200903|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.11.0|darwin|8.11.0|200903200234
3520693|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903200149
3520688|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903200149
3520270|pass|200903|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903200039
3519930|pass|200903|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.0401|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903200737
3519476|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.0401|i86pc-solaris-64int|5.8.9|solaris|2.11|200903192252
3519294|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903192213
3518902|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Tie-STDOUT|1.0401|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903200505
3518630|pass|200903|bingos@cpan.org|Tie-STDOUT|1.0401|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903191905
3518626|pass|200903|bingos@cpan.org|Tie-STDOUT|1.0401|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903191905
3518622|pass|200903|bingos@cpan.org|Tie-STDOUT|1.0401|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903191905
3518618|pass|200903|bingos@cpan.org|Tie-STDOUT|1.0401|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903191904
3517857|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Net-Random|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903191747
3517591|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|alpha-netbsd|5.10.0|netbsd|4.0.1|200903191613
3517545|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.8.9|solaris|2.11|200903191708
3517530|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|alpha-netbsd|5.10.0|netbsd|4.0.1|200903191606
3517408|pass|200903|vpit@cpan.org|Tie-STDOUT|1.04|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903191657
3517406|pass|200903|vpit@cpan.org|App-Rsnapshot|1.999_00001|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903191657
3517288|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903191042
3517280|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903191041
3516820|pass|200903|srezic@cpan.org|Tie-STDOUT|1.0401|amd64-freebsd|5.8.9|freebsd|7.0-release|200903191542
3516815|pass|200903|srezic@cpan.org|Tie-STDOUT|1.0401|amd64-freebsd|5.8.8|freebsd|7.0-release|200903191542
3516228|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903191430
3515841|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-STDOUT|1.03|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903191348
3515460|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.04|i86pc-solaris-64int|5.8.9|solaris|2.11|200903191309
3515451|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.04|i86pc-solaris-64int|5.10.0|solaris|2.11|200903191309
3514941|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|IP35-irix-64int|5.8.8|irix|6.5|200903191200
3514932|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|IP35-irix-64int|5.8.8|irix|6.5|200903191159
3514929|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903191059
3514928|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903191058
3514927|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903191058
3514926|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903191058
3514920|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903191058
3514919|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903191058
3514837|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200903191147
3514825|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200903191145
3514762|pass|200903|david@cantrell.org.uk|Tie-STDOUT|1.04|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903191037
3514760|pass|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903191037
3514703|fail|200903|david@cantrell.org.uk|App-Rsnapshot|1.999_00001|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903191026
3514646|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.0401|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903191115
3514644|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.0401|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903191115
3514642|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.0401|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903191115
3514636|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.0401|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903191114
3514633|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.0401|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903191114
3514629|cpan|200903|DCANTRELL|Tie-STDOUT|1.0401||0|||200903191113
3514626|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200903191112
3514603|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200903191106
3514450|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200903191050
3514284|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200903191032
3514265|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200903191031
3514222|pass|200903|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.02|i686-linux|5.8.9|linux|2.6.24-16-generic|200903190527
3514131|pass|200903|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|2.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903190513
3513752|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200903190918
3513705|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.10.0|linux|2.6.18-92.el5|200903190911
3513667|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200903190903
3513627|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903190858
3513319|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.04|i686-linux|5.8.9|linux|2.6.24-16-generic|200903190313
3513190|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200903190741
3513172|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200903190738
3513171|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200903190737
3513168|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Tie-STDOUT|1.04|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903190736
3513067|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.04|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903190214
3512870|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.04|i686-linux|5.10.0|linux|2.6.24-16-generic|200903190118
3512770|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903190445
3512768|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903190445
3512765|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903190444
3512763|fail|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903190443
3512761|pass|200903|bingos@cpan.org|App-Rsnapshot|1.999_00001|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903190443
3512756|pass|200903|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200903190442
3512745|pass|200903|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200903190441
3512732|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903190437
3512730|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903190436
3512726|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903190436
3512723|pass|200903|bingos@cpan.org|Tie-STDOUT|1.04|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903190435
3512282|pass|200903|imacat@mail.imacat.idv.tw|XML-Tiny|1.12|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191039
3512281|pass|200903|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191039
3512280|pass|200903|imacat@mail.imacat.idv.tw|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191039
3512279|pass|200903|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191038
3512278|pass|200903|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191038
3512277|pass|200903|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191037
3512276|pass|200903|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191037
3512274|pass|200903|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191037
3512272|pass|200903|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191036
3512270|pass|200903|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191036
3512269|pass|200903|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191036
3512267|pass|200903|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191035
3512260|pass|200903|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191033
3512254|pass|200903|imacat@mail.imacat.idv.tw|Palm-SMS|0.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191030
3512252|pass|200903|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191030
3512249|pass|200903|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191029
3512244|pass|200903|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191027
3512243|pass|200903|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191027
3512241|pass|200903|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191027
3512240|pass|200903|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191026
3512238|pass|200903|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191026
3512236|pass|200903|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191025
3512235|pass|200903|imacat@mail.imacat.idv.tw|Data-Hexdumper|2.01|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191024
3512223|pass|200903|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191017
3512220|pass|200903|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191016
3512212|pass|200903|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191015
3512211|pass|200903|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191014
3512204|fail|200903|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191010
3512194|pass|200903|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191006
3512193|unknown|200903|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191006
3512192|unknown|200903|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903191006
3512182|fail|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Tie-STDOUT|1.04|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903191101
3511659|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Net-Random|2.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903190022
3511126|fail|200903|srezic@cpan.org|App-Rsnapshot|1.999_00001|amd64-freebsd|5.8.9|freebsd|7.0-release|200903182307
3511122|pass|200903|srezic@cpan.org|Tie-STDOUT|1.04|amd64-freebsd|5.8.9|freebsd|7.0-release|200903182307
3510799|pass|200903|andy@hexten.net|Data-Hexdumper|2.01|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.15.7|200903182137
3510719|pass|200903|dagolden@cpan.org (DAGOLDEN)|WWW-Facebook-Go-SGF|1.0|i686-linux|5.8.9|linux|2.6.24-16-generic|200903181731
3510539|fail|200903|srezic@cpan.org|App-Rsnapshot|1.999_00001|amd64-freebsd|5.8.8|freebsd|7.0-release|200903182159
3510507|pass|200903|srezic@cpan.org|Tie-STDOUT|1.04|amd64-freebsd|5.8.8|freebsd|7.0-release|200903182159
3510272|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00001|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903182104
3510271|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00001|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903182104
3510270|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00001|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903182104
3510267|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00001|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903182103
3510266|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|App-Rsnapshot|1.999_00001|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903182103
3510257|cpan|200903|DCANTRELL|App-Rsnapshot|1.999_00001||0|||200903182101
3510221|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.04|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903182053
3510219|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.04|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903182052
3510216|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.04|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903182052
3510213|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.04|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903182052
3510210|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Tie-STDOUT|1.04|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903182051
3510206|cpan|200903|DCANTRELL|Tie-STDOUT|1.04||0|||200903182050
3509667|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903181331
3509470|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200903181802
3509047|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.8.9|solaris|2.11|200903181634
3508796|pass|200903|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903181554
3508025|pass|200903|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903180845
3506577|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200903180948
3505952|pass|200903|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903180809
3505747|pass|200903|wiml@hhhh.org ("Wim Lewis")|Palm-SMS|0.03|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903180634
3505486|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903180644
3504009|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903180121
3503401|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.8.9|solaris|2.11|200903172316
3502969|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903172207
3501453|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux|5.8.9|linux|2.6.24-16-generic|200903171324
3501411|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7001|i686-linux|5.8.9|linux|2.6.24-16-generic|200903171321
3500825|pass|200903|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux|5.8.9|linux|2.6.24-16-generic|200903171239
3500366|pass|200903|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Z80|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903171702
3500365|pass|200903|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903171701
3500158|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903171644
3499565|pass|200903|stsc@refcnt.org ("Steven Schubiger")|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903171548
3499162|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux|5.8.9|linux|2.6.24-16-generic|200903171005
3498811|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200903171326
3498810|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200903171325
3498489|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903171356
3497708|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903171141
3496132|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200903170623
3495035|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903170142
3494921|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903170108
3493752|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200903162119
3492140|pass|200903|davidp@preshweb.co.uk ("David Precious")|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.10.0|linux|2.6.25-arch|200903161551
3491538|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903161711
3491374|pass|200903|stsc@refcnt.org ("Steven Schubiger")|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903161647
3490918|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903161442
3490226|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903161107
3490224|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903161107
3489194|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Palm-SMS|0.03|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903160913
3488911|fail|200903|stsc@refcnt.org ("Steven Schubiger")|Bryar|3.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903160816
3488580|pass|200903|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|1.12|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903160713
3487569|pass|200903|wiml@hhhh.org ("Wim Lewis")|Bryar|3.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903160205
3486512|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903152339
3486458|pass|200903|stsc@refcnt.org ("Steven Schubiger")|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903152329
3486175|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903152227
3485297|pass|200903|stsc@refcnt.org ("Steven Schubiger")|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903151937
3484593|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Data-Hexdumper|2.01|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903151623
3484466|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Number-Phone|1.7001|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903151603
3484221|pass|200903|wiml@hhhh.org ("Wim Lewis")|XML-Tiny|1.12|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903150641
3480725|pass|200903|altblue@gmail.com ("Marius Feraru")|File-Find-Rule-Permissions|2.0|x86_64-linux-ld|5.11.0 patch 34814|linux|2.6.18-8.1.14.el5|200903150508
3480603|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200903150257
3480589|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200903150254
3480135|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903150231
3479831|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903150155
3479784|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.18-14-ovz-686|200903150152
3479437|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903150100
3478862|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903142218
3475598|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-64int|5.10.0|dragonfly|2.0.1-release|200903141059
3475587|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-64int|5.8.9|dragonfly|2.0.1-release|200903141057
3474262|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903140457
3473457|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903140126
3473280|na|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903140119
3473195|unknown|200903|stsc@refcnt.org ("Steven Schubiger")|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903140004
3473021|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903132326
3472893|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903140030
3472752|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903140007
3471944|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903131707
3471747|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903132143
3471716|pass|200903|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200903132137
3471674|pass|200903|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200903132130
3471591|pass|200903|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200903132118
3470390|unknown|200903|stsc@refcnt.org ("Steven Schubiger")|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903131832
3469863|pass|200903|stro@cpan.org|Number-Phone|1.7001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903131838
3469791|fail|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903131232
3468132|pass|200903|stsc@refcnt.org ("Steven Schubiger")|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903131416
3466849|pass|200903|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200903131138
3466799|pass|200903|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200903131129
3466703|pass|200903|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903130953
3466396|pass|200903|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.50|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903131717
3465821|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903130753
3465694|pass|200903|imacat@mail.imacat.idv.tw|Number-Phone|1.7001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903130752
3465343|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903130506
3465138|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903130506
3463931|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903130248
3463819|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903130108
3462720|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903122020
3461344|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903121559
3460510|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903121341
3460507|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903121340
3459049|pass|200903|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903121113
3459045|pass|200903|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|i86pc-solaris-64int|5.8.9|solaris|2.11|200903121113
3458680|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200903121023
3457736|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903120723
3457645|pass|200903|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.26-1-amd64|200903121351
3457511|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903120621
3456798|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903120344
3455393|pass|200903|wiml@hhhh.org ("Wim Lewis")|Class-DBI-ClassGenerator|1.02|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903111319
3454728|pass|200903|wiml@hhhh.org ("Wim Lewis")|File-Find-Rule-Permissions|2.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903111057
3454359|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903111825
3454103|pass|200903|stsc@refcnt.org ("Steven Schubiger")|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903111728
3451064|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110530
3451017|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110525
3450966|unknown|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110521
3449905|unknown|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110303
3449473|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903110155
3449044|pass|200903|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110055
3448665|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903110407
3448116|pass|200903|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110152
3448115|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110152
3447750|pass|200903|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903110110
3445958|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903102314
3444689|pass|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903101529
3444566|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.03|i86pc-solaris-64int|5.8.9|solaris|2.11|200903102015
3444113|fail|200903|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903101914
3443590|pass|200903|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|1.12|i86pc-solaris-64int|5.8.9|solaris|2.11|200903101721
3443517|pass|200903|bingos@cpan.org|Number-Phone|1.7001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200903101600
3443504|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200903101557
3442981|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903101451
3440035|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903100910
3439120|pass|200903|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903100249
3438753|pass|200903|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903100206
3438300|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200903100603
3438231|pass|200903|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|2.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903100553
3436660|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|darwin-2level|5.10.0|darwin|8.11.1|200903100300
3436378|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903100127
3436264|pass|200903|bingos@cpan.org|Number-Phone|1.7001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200903100111
3436251|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200903100107
3436248|pass|200903|bingos@cpan.org|Number-Phone|1.7001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200903100107
3436244|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200903100107
3436234|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200903100102
3436233|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200903100102
3435889|pass|200903|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903100012
3433824|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|darwin-2level|5.10.0|darwin|8.11.1|200903092103
3433735|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903091554
3433223|pass|200903|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903091448
3433115|pass|200903|JOST@cpan.org ("Josts Smokehouse")|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200903091937
3432825|pass|200903|bingos@cpan.org|Number-Phone|1.7001|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200903091752
3432823|pass|200903|bingos@cpan.org|Number-Phone|1.7001|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903091752
3432792|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200903091744
3432791|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903091744
3432782|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200903091743
3432780|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200903091743
3432266|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|2.01|i86pc-solaris-64int|5.8.9|solaris|2.11|200903091737
3432109|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.7001|i86pc-solaris-64int|5.8.9|solaris|2.11|200903091715
3431433|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.8.9|solaris|2.11|200903091553
3431136|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.8.9|solaris|2.11|200903091507
3430799|pass|200903|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200903091326
3430774|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903090923
3430161|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200903091224
3430157|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200903091224
3430089|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903090817
3429200|pass|200903|andy@hexten.net|XML-Tiny|1.12|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200903091037
3429125|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200903091028
3429122|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200903091028
3429036|pass|200903|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903090621
3428320|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|darwin-2level|5.10.0|darwin|8.11.1|200903090944
3427864|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|darwin-2level|5.10.0|darwin|8.11.1|200903090908
3427647|pass|200903|robbiebow@gmail.com|XML-Tiny|1.12|i686-linux|5.10.0|linux|2.6.27.12-bytemark-kvm-2009-01-19|200903081933
3426959|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903090206
3425196|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903090334
3424877|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903090300
3424260|pass|200903|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903090043
3423098|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200903082257
3423089|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200903082257
3423020|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200903082250
3423019|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200903082250
3422238|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903082142
3422231|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903082142
3422121|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200903082230
3421549|pass|200903|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903082153
3421225|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200903082133
3420511|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200903081949
3420461|pass|200903|mw487.yahoo.com@gmail.com|Data-Hexdumper|2.01|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200903081447
3420457|pass|200903|mw487.yahoo.com@gmail.com|Data-Hexdumper|2.0|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200903081447
3420231|pass|200903|andy@hexten.net|Data-Hexdumper|2.01|i686-linux-thread-multi|5.10.0|linux|2.6.20-16-server|200903081934
3420027|pass|200903|andy@hexten.net|Data-Hexdumper|2.01|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200903081844
3419998|pass|200903|andy@hexten.net|Number-Phone|1.7001|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200903081913
3419484|pass|200903|andy@hexten.net|Data-Hexdumper|2.01|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200903081843
3419060|pass|200903|andy@hexten.net|Data-Hexdumper|2.01|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200903081816
3418859|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200903081707
3418403|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903081817
3417556|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903081217
3417554|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903081217
3417264|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903081156
3416662|pass|200903|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903081106
3415688|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903080949
3414778|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200903081336
3414735|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200903081233
3414730|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200903081232
3414706|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903080832
3414071|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|x86_64-linux-ld|5.10.0|linux|2.6.26-1-amd64|200903081224
3413618|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903080943
3413612|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903080942
3413609|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903080941
3413250|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903080807
3413246|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903080806
3413241|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903080806
3411223|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903080355
3410785|pass|200903|wiml@hhhh.org ("Wim Lewis")|WWW-Facebook-Go-SGF|1.0|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903080117
3410701|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200903080057
3410697|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200903080056
3410629|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200903080045
3410628|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200903080045
3409433|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903071534
3409224|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903071958
3409221|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903071958
3409218|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903071957
3409217|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903071957
3409216|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903071956
3409215|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903071956
3408610|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903071849
3407689|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903071005
3407411|fail|200903|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070908
3407116|pass|200903|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070810
3406352|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200903071031
3406350|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200903071031
3405396|pass|200903|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070223
3405351|pass|200903|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070215
3405059|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200903070411
3405056|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200903070411
3405051|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200903070410
3405049|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200903070410
3403736|pass|200903|dagolden@cpan.org (DAGOLDEN)|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070313
3403137|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070122
3403117|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903070118
3403108|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903070117
3403100|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903070116
3403043|pass|200903|wiml@hhhh.org ("Wim Lewis")|Data-Hexdumper|2.01|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903070103
3402619|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903061853
3402578|pass|200903|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200903061847
3402294|pass|200903|wiml@hhhh.org ("Wim Lewis")|Number-Phone|1.7001|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903061436
3402123|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200903062227
3402102|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200903062222
3402101|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200903062221
3402073|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903061715
3402047|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200903062208
3402043|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200903062207
3402040|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200903062206
3400760|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200903061726
3400566|pass|200903|jeff@zeroclue.com|Class-DBI-ClassGenerator|1.02|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903061141
3400328|pass|200903|jeff@zeroclue.com|File-Find-Rule-Permissions|2.0|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903061106
3399606|fail|200903|srezic@cpan.org|Data-Hexdumper|2.0|amd64-freebsd|5.6.2|freebsd|7.0-release|200903061321
3399415|pass|200903|wiml@hhhh.org ("Wim Lewis")|Devel-CheckOS|1.50|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903060359
3399200|pass|200903|wiml@hhhh.org ("Wim Lewis")|Data-Compare|1.21|powerpc-netbsd-thread-multi|5.8.8|netbsd|4.0|200903060635
3398719|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903061048
3398718|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903061048
3398715|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903061047
3398714|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200903061047
3398359|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903060922
3398356|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903060921
3398353|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903060920
3398350|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200903060920
3397007|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903060153
3396972|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903060145
3396937|unknown|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200903060140
3396073|fail|200903|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903060413
3395620|unknown|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903060320
3394812|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200903060006
3394811|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200903060006
3394001|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903052026
3393998|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903052025
3393994|pass|200903|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903052024
3393993|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903052024
3393992|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200903052023
3393991|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903052022
3393987|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903052022
3393983|pass|200903|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903052021
3393979|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903052020
3393978|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903052119
3393977|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200903052019
3393896|pass|200903|srezic@cpan.org|Number-Phone|1.7|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200903052053
3392975|pass|200903|stro@cpan.org|Data-Hexdumper|2.01|i686-linux|5.8.8|linux|2.4.27-2-386|200903051811
3392941|pass|200903|stro@cpan.org|Number-Phone|1.7001|i686-linux|5.8.8|linux|2.4.27-2-386|200903051756
3392548|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903050934
3392252|pass|200903|stro@cpan.org|Number-Phone|1.7001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903051347
3392232|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903051216
3392087|pass|200903|stro@cpan.org|Data-Hexdumper|2.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903051257
3392030|pass|200903|dagolden@cpan.org (DAGOLDEN)|WWW-Facebook-Go-SGF|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903050542
3391742|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903050910
3391738|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903050910
3391733|pass|200903|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903050909
3391729|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903050909
3391727|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200903050909
3390892|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903050437
3390605|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200903050315
3390594|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200903050313
3390571|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200903050309
3390548|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200903050305
3390534|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200903050302
3390420|pass|200903|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903050332
3390155|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200903050154
3390143|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200903050151
3390121|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903050207
3390118|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200903050148
3390068|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200903050144
3390053|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200903050143
3389619|pass|200903|imacat@mail.imacat.idv.tw|Number-Phone|1.7001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200903050937
3389617|pass|200903|imacat@mail.imacat.idv.tw|Data-Hexdumper|2.01|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200903050937
3389170|pass|200903|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.24-16-generic|200903050034
3389169|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903050033
3388761|pass|200903|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903041839
3388595|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|darwin-2level|5.8.8|darwin|8.10.1|200903050014
3388547|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|darwin-2level|5.8.8|darwin|8.10.1|200903050018
3387618|pass|200903|vpit@cpan.org|Number-Phone|1.7001|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903042226
3387615|pass|200903|vpit@cpan.org|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903042226
3386966|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903041909
3386964|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903041909
3386963|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903041408
3386960|pass|200903|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903041908
3386956|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903041907
3386953|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200903041906
3386935|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903041902
3386933|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903041901
3386926|pass|200903|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903041900
3386923|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903041859
3386922|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200903041859
3386892|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903041953
3386876|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7001|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903041348
3386772|pass|200903|jeff@zeroclue.com|Data-Hexdumper|2.01|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903041429
3386561|pass|200903|jeff@zeroclue.com|Number-Phone|1.7001|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200903041403
3386212|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903041734
3386102|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.03|i686-linux|5.10.0|linux|2.6.24-16-generic|200903041112
3385967|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903041643
3385592|pass|200903|david@cantrell.org.uk|Data-Compare|1.21|alpha-netbsd|5.10.0|netbsd|4.0.1|200903041453
3385358|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903041508
3385186|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903040730
3385185|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|IP35-irix-64int|5.8.8|irix|6.5|200903041430
3385180|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|alpha-netbsd|5.10.0|netbsd|4.0.1|200903041330
3385170|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|IP35-irix-64int|5.8.8|irix|6.5|200903041429
3385169|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903040729
3385165|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|alpha-netbsd|5.10.0|netbsd|4.0.1|200903041329
3384898|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903041349
3384765|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200903041328
3384755|pass|200903|david@cantrell.org.uk|Data-Compare|1.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903041228
3384433|pass|200903|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903040627
3384430|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200903041226
3384414|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903041225
3384394|pass|200903|david@cantrell.org.uk|Data-Compare|1.21|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903041122
3384369|pass|200903|david@cantrell.org.uk|Data-Compare|1.21|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903041119
3384344|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903041116
3384341|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903041116
3384306|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903041109
3384304|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903041109
3384281|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200903041202
3384205|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200903041158
3384135|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200903041143
3384109|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903041039
3384107|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903041038
3384103|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903041038
3384102|pass|200903|david@cantrell.org.uk|Number-Phone|1.7001|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903041038
3384100|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903041037
3384092|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.01|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903041038
3383913|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903041127
3383890|pass|200903|srezic@cpan.org|Number-Phone|1.7001|amd64-freebsd|5.8.9|freebsd|7.0-release|200903041124
3383884|pass|200903|srezic@cpan.org|Data-Hexdumper|2.01|amd64-freebsd|5.8.9|freebsd|7.0-release|200903041124
3383873|pass|200903|srezic@cpan.org|Number-Phone|1.7001|amd64-freebsd|5.8.8|freebsd|7.0-release|200903041124
3383871|pass|200903|srezic@cpan.org|Data-Hexdumper|2.01|amd64-freebsd|5.8.8|freebsd|7.0-release|200903041124
3383712|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903041059
3383696|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200903041057
3383680|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903041055
3383645|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Number-Phone|1.7001|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903041052
3383581|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200903041040
3383578|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200903041039
3383537|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200903041031
3383530|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200903040931
3383529|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903041029
3383528|pass|200903|bingos@cpan.org|Number-Phone|1.7001|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200903040930
3383522|pass|200903|bingos@cpan.org|Number-Phone|1.7|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200903040930
3383518|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200903040929
3383516|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200903040929
3383506|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200903041024
3383503|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200903041024
3383439|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200903041013
3383436|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903041012
3383430|pass|200903|bingos@cpan.org|Number-Phone|1.7001|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903040911
3383427|pass|200903|bingos@cpan.org|Number-Phone|1.7001|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903040910
3383426|pass|200903|bingos@cpan.org|Number-Phone|1.7001|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903040909
3383419|pass|200903|bingos@cpan.org|Number-Phone|1.7001|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903040908
3383416|pass|200903|bingos@cpan.org|Number-Phone|1.7001|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903040907
3383392|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903041003
3383352|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903040959
3383344|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903040958
3383333|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200903040957
3383298|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|2.01|i86pc-solaris-64int|5.10.0|solaris|2.11|200903040951
3383297|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903040951
3383285|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200903040948
3383266|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.7001|i86pc-solaris-64int|5.10.0|solaris|2.11|200903040947
3383265|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200903040946
3383214|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200903040939
3383133|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200903040923
3383082|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200903040916
3383081|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200903040916
3383067|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200903040914
3383058|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903040913
3383018|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903040908
3383013|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200903040906
3383001|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903040905
3382976|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903040850
3382968|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|x86_64-linux|5.10.0|linux|2.6.26-1-amd64|200903040850
3382952|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903040901
3382782|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Number-Phone|1.7001|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903041638
3382737|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7001|s390x-linux|5.10.0|linux|2.6.18-92.el5|200903040830
3382726|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903040229
3382546|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7001|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903040757
3382543|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7001|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903040756
3382539|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7001|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903040755
3382536|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7001|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903040755
3382532|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7001|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903040755
3382356|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903040724
3382353|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903040724
3382352|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903040724
3382351|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903040724
3382348|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.01|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903040723
3382093|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903040530
3382088|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903040530
3382085|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903040529
3382082|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903040529
3382079|pass|200903|bingos@cpan.org|Data-Hexdumper|2.01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903040529
3381971|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.01|i686-linux|5.10.0|linux|2.6.24-16-generic|200903040447
3381952|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7001|i686-linux|5.10.0|linux|2.6.24-16-generic|200903040438
3381314|pass|200903|rafl@debian.org|Devel-CheckOS|1.50|i686-linux|5.10.0 patch 35117|linux|2.6.26|200903040244
3381165|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200903040109
3381161|pass|200903|bingos@cpan.org|Number-Phone|1.7|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200903040109
3381153|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200903040107
3381107|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Hexdumper|2.01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903040954
3380947|cpan|200903|DCANTRELL|Number-Phone|1.7001||0|||200903040118
3380793|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200903032350
3380790|fail|200903|bingos@cpan.org|Number-Phone|1.7|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200903032349
3380787|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200903032349
3380383|pass|200903|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200903031730
3380352|pass|200903|srezic@cpan.org|Data-Hexdumper|2.0|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200903032321
3379920|pass|200903|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903031608
3379675|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.0|s390-linux|5.8.8|linux|2.6.5-7.315-s390|200903032132
3379639|cpan|200903|DCANTRELL|Data-Hexdumper|2.01||0|||200903032129
3379242|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903031833
3379239|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903031832
3379235|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903031831
3379056|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.0|s390-linux-thread-multi|5.10.0|linux|2.6.5-7.315-s390|200903032006
3378905|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903031945
3378779|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903031920
3378774|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903031711
3378769|fail|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903031710
3378766|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903031709
3378760|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903031914
3378717|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|2.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903031901
3378296|pass|200903|vpit@cpan.org|Data-Hexdumper|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903031724
3377827|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200903031443
3377817|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200903031440
3377808|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200903031436
3377737|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.24-16-generic|200903030918
3377483|pass|200903|stro@cpan.org|Data-Hexdumper|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903031520
3377445|fail|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|IP35-irix-64int|5.8.8|irix|6.5|200903031410
3377188|pass|200903|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903030649
3377164|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200903031144
3377162|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200903030544
3377104|pass|200903|srezic@cpan.org|Number-Phone|1.7|amd64-freebsd|5.10.0|freebsd|7.0-release|200903031227
3377010|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200903031110
3377008|pass|200903|bingos@cpan.org|Number-Phone|1.7|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200903031109
3377004|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200903031109
3376999|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200903031108
3376995|pass|200903|bingos@cpan.org|Number-Phone|1.7|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200903031108
3376983|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200903031106
3376977|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200903031107
3376690|pass|200903|stsc@refcnt.org ("Steven Schubiger")|Data-Hexdumper|2.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200903031126
3376628|pass|200903|srezic@cpan.org|WWW-Facebook-Go-SGF|1.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200903031114
3376603|fail|200903|srezic@cpan.org|Data-Hexdumper|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200903031107
3376402|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200903030924
3376399|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200903030924
3376396|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200903030924
3376394|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200903030923
3376391|fail|200903|bingos@cpan.org|Data-Hexdumper|2.0|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200903030923
3376388|pass|200903|bingos@cpan.org|Data-Hexdumper|2.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200903030923
3376245|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|2.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200903031006
3376205|fail|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200903030901
3376204|fail|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200903030901
3376143|pass|200903|david@cantrell.org.uk|Data-Hexdumper|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200903030846
3376003|pass|200903|stro@cpan.org|Data-Hexdumper|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903031017
3375935|fail|200903|srezic@cpan.org|Data-Hexdumper|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200903030842
3375906|pass|200903|srezic@cpan.org|Data-Hexdumper|2.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200903030846
3375854|pass|200903|srezic@cpan.org|Number-Phone|1.7|amd64-freebsd|5.8.8|freebsd|7.0-release|200903030841
3375678|pass|200903|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Data-Hexdumper|2.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200903031600
3375181|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903030438
3375107|pass|200903|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.24-16-generic|200903030405
3374750|pass|200903|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903030203
3374547|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903030105
3374545|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903030104
3374541|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903030103
3374537|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903030102
3374233|fail|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200903030049
3374230|fail|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200903030048
3374228|fail|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200903030048
3374226|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200903030048
3374224|pass|200903|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|2.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200903030048
3374215|cpan|200903|DCANTRELL|Data-Hexdumper|2.0||0|||200903030047
3374116|pass|200903|srezic@cpan.org|Number-Phone|1.7|amd64-freebsd|5.8.9|freebsd|7.0-release|200903022347
3373799|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903021749
3373129|pass|200903|srezic@cpan.org|WWW-Facebook-Go-SGF|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200903021740
3373083|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux|5.10.0|linux|2.6.5-7.315-s390|200903022124
3373060|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200903021515
3372917|pass|200903|srezic@cpan.org|WWW-Facebook-Go-SGF|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200903022059
3372150|pass|200903|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.24-16-generic|200903021324
3371992|pass|200903|vpit@cpan.org|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903021851
3371990|pass|200903|vpit@cpan.org|Number-Phone|1.7|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200903021851
3371966|pass|200903|stro@cpan.org|WWW-Facebook-Go-SGF|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903021943
3371534|pass|200903|celavie@reflec.net|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-stable|200903030026
3371319|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903021641
3371208|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903021527
3371207|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200903021526
3371042|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903021605
3370863|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903021540
3370646|pass|200903|stro@cpan.org|Number-Phone|1.7|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200903021601
3370624|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903021356
3370622|fail|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200903021355
3370474|pass|200903|mw487.yahoo.com@gmail.com|WWW-Facebook-Go-SGF|1.0|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200903020825
3369636|pass|200903|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.7|i86pc-solaris-64int|5.10.0|solaris|2.11|200903021131
3369508|pass|200903|JOST@cpan.org ("Josts Smokehouse")|WWW-Facebook-Go-SGF|1.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200903021113
3368772|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903020338
3367403|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux-thread-multi-perlio|5.6.2|linux|2.6.5-7.315-s390|200903020548
3367317|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux-thread-multi|5.6.2|linux|2.6.5-7.315-s390|200903020511
3367215|pass|200903|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390-linux|5.6.2|linux|2.6.5-7.315-s390|200903020448
3366830|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200903020207
3366828|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200903020206
3365938|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903012259
3365934|pass|200903|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200903012259
3365916|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200903012256
3365757|pass|200903|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.24-16-generic|200903011738
3365498|pass|200903|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200903012102
3363239|pass|200903|stro@cpan.org|WWW-Facebook-Go-SGF|1.0|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200903011245
3363187|pass|200903|ddascaNOSPAMlescu+perl@gmail.com|Data-Hexdumper|1.4|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200903010109
3362672|pass|200902|fraterm@gmail.com ("Michael Costello")|Games-Dice-Advanced|1.1|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902251649
3362477|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010501
3362476|pass|200903|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010501
3362437|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010430
3362264|pass|200903|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010236
3361987|pass|200903|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010417
3361986|pass|200903|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010417
3361773|pass|200903|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200903010203
3359467|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200902281413
3359435|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200902281406
3359423|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200902280802
3359404|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|IP35-irix-64int|5.8.8|irix|6.5|200902281458
3359184|fail|200902|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902281433
3359183|pass|200902|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902281433
3359047|pass|200902|imacat@mail.imacat.idv.tw|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200902282106
3358806|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Number-Phone|1.7|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902281317
3358653|pass|200902|stsc@refcnt.org ("Steven Schubiger")|WWW-Facebook-Go-SGF|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902281256
3358523|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200902281236
3358486|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200902281231
3358473|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200902281129
3358451|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902281225
3358375|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902281214
3358334|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200902281207
3358327|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200902281205
3358296|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200902281159
3358230|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200902281051
3358228|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200902281051
3358222|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200902281050
3358215|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200902281049
3358211|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200902281147
3358192|pass|200902|david@cantrell.org.uk|WWW-Facebook-Go-SGF|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200902281044
3357936|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200902281101
3357935|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200902281101
3357927|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200902281058
3357896|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|WWW-Facebook-Go-SGF|1.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200902281049
3357179|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200902280652
3357177|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200902280652
3357172|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200902280651
3357170|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200902280651
3357167|pass|200902|bingos@cpan.org|WWW-Facebook-Go-SGF|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200902280650
3356934|pass|200902|jeff@zeroclue.com|WWW-Facebook-Go-SGF|1.0|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200902280232
3356927|pass|200902|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.7|i686-linux|5.10.0|linux|2.6.24-16-generic|200902280043
3356777|pass|200902|dagolden@cpan.org (DAGOLDEN)|WWW-Facebook-Go-SGF|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200902280441
3356623|pass|200902|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200902280316
3356612|pass|200902|bingos@cpan.org|Number-Phone|1.7|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200902280311
3356531|pass|200902|imacat@mail.imacat.idv.tw|Number-Phone|1.7|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200902281038
3355866|pass|200902|jeff@zeroclue.com|Number-Phone|1.7|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200902271806
3355857|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|WWW-Facebook-Go-SGF|1.0|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200902272218
3355855|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|WWW-Facebook-Go-SGF|1.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200902272218
3355853|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|WWW-Facebook-Go-SGF|1.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200902272217
3355852|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|WWW-Facebook-Go-SGF|1.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200902272217
3355851|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|WWW-Facebook-Go-SGF|1.0|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200902272217
3355848|cpan|200902|DCANTRELL|WWW-Facebook-Go-SGF|1.0||0|||200902272216
3355302|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200902271753
3355279|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902271747
3355254|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902271734
3355219|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200902271720
3355113|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200902271647
3355060|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200902271632
3355038|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200902271628
3355007|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200902271622
3354988|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200902271610
3354973|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux|5.10.0|linux|2.6.18-92.el5|200902271607
3354922|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200902271551
3354908|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.7|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200902271547
3354607|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902271440
3354296|pass|200902|bingos@cpan.org|Number-Phone|1.7|i386-dragonfly-64int|5.8.8|dragonfly|2.0.0-release|200902271144
3354110|pass|200902|stro@cpan.org|Number-Phone|1.7|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200902271249
3353992|pass|200902|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200902271007
3353951|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|alpha-netbsd|5.10.0|netbsd|4.0.1|200902270944
3353946|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|IP35-irix-64int|5.8.8|irix|6.5|200902271043
3353916|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200902270333
3353827|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200902270912
3353791|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200902270858
3353789|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200902270858
3353774|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200902270856
3353773|pass|200902|david@cantrell.org.uk|Number-Phone|1.7|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200902270856
3353687|fail|200902|bingos@cpan.org|Number-Phone|1.7|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200902270834
3353370|pass|200902|bingos@cpan.org|Number-Phone|1.7|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200902270504
3353368|pass|200902|bingos@cpan.org|Number-Phone|1.7|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200902270503
3353366|pass|200902|bingos@cpan.org|Number-Phone|1.7|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200902270502
3353364|fail|200902|bingos@cpan.org|Number-Phone|1.7|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200902270502
3353363|pass|200902|bingos@cpan.org|Number-Phone|1.7|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200902270501
3353126|pass|200902|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Number-Phone|1.7|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200902271203
3352100|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7|i686-linux-64int|5.8.6|linux|2.6.22-1-k7|200902262122
3352093|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200902262121
3352085|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200902262120
3352080|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200902262119
3352078|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Number-Phone|1.7|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200902262119
3352076|cpan|200902|DCANTRELL|Number-Phone|1.7||0|||200902262116
3349261|pass|200902|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200902260148
3347107|pass|200902|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24-16-generic|200902251135
3346932|fail|200902|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200902251044
3346325|pass|200902|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.12|i686-linux|5.10.0|linux|2.6.24-16-generic|200902250724
3342039|pass|200902|stsc@refcnt.org ("Steven Schubiger")|DBIx-Class-SingletonRows|0.11|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902241926
3341904|pass|200902|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.02|i686-linux|5.10.0|linux|2.6.24-16-generic|200902241255
3341737|pass|200902|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|2.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200902241227
3341582|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Palm-Treo680MessagesDB|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902241803
3341581|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902241802
3341552|pass|200902|rlane10@wi.rr.com|XML-Tiny|1.12|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.18.el5|200902240849
3340932|pass|200902|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.24-16-generic|200902241007
3338316|pass|200902|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.26-1-686|200902240333
3334176|pass|200902|robbiebow@gmail.com|XML-Tiny|1.12|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18-6-686|200902210007
3333867|pass|200902|amoore@cpan.org|Devel-CheckOS|1.50|x86_64-linux-gnu-thread-multi|5.10.0|linux|2.6.26-1-vserver-amd64|200902201428
3328741|pass|200902|rafl@debian.org|Data-Compare|1.21|i686-linux|5.10.0 patch 35117|linux|2.6.26|200902220600
3327140|pass|200902|srezic@cpan.org|File-Find-Rule-Permissions|2.0|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200902212229
3324808|pass|200902|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200902211006
3320276|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200902200023
3319956|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200902192303
3319696|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200902192157
3319416|pass|200902|srezic@cpan.org|File-Find-Rule-Permissions|2.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200902192108
3317709|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200902190325
3317708|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200902190325
3316832|pass|200902|nothingmuch@woobling.org|Number-Phone|1.6|darwin-2level|5.10.0|darwin|9.5.0|200902180429
3316609|pass|200902|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.10.0|freebsd|7.0-release|200902182306
3315924|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200902181859
3315923|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200902181859
3315922|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200902181923
3315628|fail|200902|david@cantrell.org.uk|Devel-CheckOS|1.50|alpha-netbsd|5.10.0|netbsd|4.0.1|200902181717
3314697|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200902180908
3314380|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200902180628
3314147|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200902180251
3314146|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200902180251
3314145|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200902180250
3313856|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200902180101
3312547|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200902171858
3312528|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200902171852
3312527|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200902171852
3312523|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200902171851
3311838|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200902171628
3311833|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200902171626
3311770|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200902171615
3311763|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200902171613
3310986|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200902171257
3310984|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200902171257
3310981|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200902171256
3310951|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200902171245
3310417|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200902171056
3310414|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200902171055
3308834|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902170441
3308828|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902170439
3308827|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902170439
3308787|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200902170331
3308262|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200902170043
3308261|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200902170043
3308194|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902170123
3308048|pass|200902|bingos@cpan.org|XML-Tiny|1.12|i686-linux-64int-ld|5.11.0 patch GitLive-blead-623-g4705144|linux|2.6.21.5-smp|200902162341
3307158|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200902161944
3306331|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-64int-ld|5.11.0 patch GitLive-blead-623-g4705144|linux|2.6.21.5-smp|200902161540
3306119|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200902161438
3306114|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200902161437
3306061|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200902161419
3306057|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200902161418
3305989|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200902161345
3305575|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-64int-ld|5.11.0 patch GitLive-blead-623-g4705144|linux|2.6.21.5-smp|200902161141
3305574|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-64int-ld|5.11.0 patch GitLive-blead-623-g4705144|linux|2.6.21.5-smp|200902161140
3305573|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-64int-ld|5.11.0 patch GitLive-blead-623-g4705144|linux|2.6.21.5-smp|200902161140
3305227|pass|200902|andy@hexten.net|Data-Hexdumper|1.4|x86_64-linux-thread-multi|5.10.0|linux|2.6.26-1-amd64|200902161030
3305048|pass|200902|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.50|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200902160458
3304723|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200902160838
3303730|na|200902|mw487.yahoo.com@gmail.com|File-Find-Rule-Permissions|2.0|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200902160237
3303160|pass|200902|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.8.9|dragonfly|2.0.0-release|200902152222
3303098|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200902152207
3302727|pass|200902|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.26-1-686|200902152131
3302467|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200902151946
3302206|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-dragonfly-64int|5.8.8|dragonfly|2.0.0-release|200902151819
3301764|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200902151648
3301170|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200902151405
3301165|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-dragonfly-64int|5.8.8|dragonfly|2.0.0-release|200902151403
3301130|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200902151348
3300774|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200902151202
3300729|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200902151252
3300040|pass|200902|stro@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux|5.8.8|linux|2.4.27-2-386|200902150627
3299776|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200902150211
3299336|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-dragonfly-64int|5.8.9|dragonfly|2.0.0-release|200902142358
3299284|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200902142345
3299270|na|200902|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|2.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902141839
3298349|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200902142025
3297827|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200902141805
3297826|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200902141805
3297672|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-dragonfly-64int|5.8.9|dragonfly|2.0.0-release|200902141721
3297671|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.8.9|dragonfly|2.0.0-release|200902141721
3297056|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200902141525
3296553|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200902141348
3296543|pass|200902|vpit@cpan.org|Data-Hexdumper|1.4|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200902141449
3296521|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200902141346
3295481|pass|200902|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902140941
3294438|pass|200902|DCOLLINS@cpan.org|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902140611
3293864|pass|200902|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902140448
3292994|pass|200902|DCOLLINS@cpan.org|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902140226
3292741|pass|200902|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902140201
3292665|pass|200902|DCOLLINS@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902140153
3291362|pass|200902|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.11.0 patch GitLive-blead-599-gd9acc78|linux|2.6.16.38|200902132304
3291093|pass|200902|DCOLLINS@cpan.org|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131736
3290989|pass|200902|DCOLLINS@cpan.org|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131707
3290684|pass|200902|DCOLLINS@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131711
3289850|pass|200902|DCOLLINS@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131553
3288267|pass|200902|DCOLLINS@cpan.org|Sub-WrapPackages|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131331
3288094|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.11.0 patch GitLive-blead-599-gd9acc78|linux|2.6.16.38|200902131824
3287853|pass|200902|DCOLLINS@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131303
3287792|pass|200902|DCOLLINS@cpan.org|Statistics-ChiSquare|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902131259
3286793|pass|200902|david@cantrell.org.uk|Data-Hexdumper|1.4|alpha-netbsd|5.10.0|netbsd|4.0.1|200902131639
3286678|pass|200902|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.11.0 patch GitLive-blead-586-gbf8d130|dragonfly|2.0.1-release|200902131602
3285662|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.11.0 patch GitLive-blead-599-gd9acc78|linux|2.6.16.38|200902131449
3285656|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.11.0 patch GitLive-blead-599-gd9acc78|linux|2.6.16.38|200902131449
3285410|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200902131427
3285123|pass|200902|david@cantrell.org.uk|Data-Hexdumper|1.4|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200902131414
3284645|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200902131337
3284640|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200902131336
3284616|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200902131335
3284607|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200902131335
3283396|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902131210
3282049|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200902131032
3280434|pass|200902|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130208
3280379|pass|200902|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130241
3280220|pass|200902|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902130218
3280195|pass|200902|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902130217
3280118|pass|200902|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130242
3280113|pass|200902|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130242
3280014|pass|200902|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902130206
3279887|pass|200902|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130223
3279884|pass|200902|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130222
3279881|pass|200902|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130222
3279851|pass|200902|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902130218
3279831|pass|200902|DCOLLINS@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130143
3279795|pass|200902|DCOLLINS@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130211
3279783|pass|200902|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130209
3279782|pass|200902|DCOLLINS@cpan.org|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130209
3279767|pass|200902|DCOLLINS@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902130206
3279762|pass|200902|DCOLLINS@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902130205
3279584|pass|200902|DCOLLINS@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902130140
3279575|pass|200902|DCOLLINS@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130135
3279442|pass|200902|DCOLLINS@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902130117
3278906|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200902130508
3278888|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200902130506
3278687|fail|200902|DCOLLINS@cpan.org|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130437
3278647|fail|200902|DCOLLINS@cpan.org|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130430
3278624|pass|200902|DCOLLINS@cpan.org|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902130426
3278514|pass|200902|DCOLLINS@cpan.org|Data-Compare|1.21|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902130404
3278510|pass|200902|DCOLLINS@cpan.org|Data-Compare|1.21|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902130403
3278265|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200902130228
3278261|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200902130226
3278253|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200902130224
3278252|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200902130224
3278249|pass|200902|bingos@cpan.org|Data-Hexdumper|1.4|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200902130222
3277641|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200902122330
3277605|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200902122321
3277323|pass|200902|vpit@cpan.org|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200902122314
3277322|pass|200902|vpit@cpan.org|Class-DBI-ClassGenerator|1.02|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200902122314
3274801|pass|200902|DCOLLINS@cpan.org|NestedMap|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902121009
3274757|pass|200902|DCOLLINS@cpan.org|NestedMap|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902121026
3274420|pass|200902|DCOLLINS@cpan.org|NestedMap|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902121148
3274403|pass|200902|DCOLLINS@cpan.org|NestedMap|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902121136
3274314|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200902121642
3274307|pass|200902|DCOLLINS@cpan.org|NestedMap|1.0|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902121120
3272255|pass|200902|srezic@cpan.org|File-Find-Rule-Permissions|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200902121410
3271836|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-dragonfly-64int|5.11.0 patch GitLive-blead-586-gbf8d130|dragonfly|2.0.1-release|200902121302
3271833|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.11.0 patch GitLive-blead-586-gbf8d130|dragonfly|2.0.1-release|200902121301
3271687|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|IP35-irix-64int|5.8.8|irix|6.5|200902121350
3271644|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200902120647
3270557|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200902121139
3270551|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200902121139
3270540|pass|200902|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|2.0|i86pc-solaris-64int|5.10.0|solaris|2.11|200902121239
3270521|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200902121138
3270447|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200902121135
3270416|pass|200902|david@cantrell.org.uk|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200902121133
3270411|pass|200902|david@cantrell.org.uk|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200902121133
3270337|fail|200902|DCOLLINS@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902120317
3270080|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200902121212
3269792|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.10.0|linux|2.6.18-92.el5|200902121156
3269616|pass|200902|stsc@refcnt.org ("Steven Schubiger")|File-Find-Rule-Permissions|2.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902121046
3269601|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.02|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200902121144
3268661|pass|200902|DCOLLINS@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902120317
3268096|pass|200902|stsc@refcnt.org ("Steven Schubiger")|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902120906
3267897|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200902120849
3267890|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200902120848
3267710|pass|200902|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200902121636
3267164|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200902120827
3266967|fail|200902|DCOLLINS@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902120156
3266942|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200902120829
3266930|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200902120826
3266923|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200902120826
3266862|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902120822
3266826|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200902120819
3266798|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200902120817
3266782|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200902120816
3266717|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200902120811
3266713|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902120810
3266624|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200902120803
3266568|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|File-Find-Rule-Permissions|2.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200902120759
3266336|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200902120644
3266324|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200902120644
3266310|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200902120643
3266296|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200902120642
3266287|pass|200902|bingos@cpan.org|File-Find-Rule-Permissions|2.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200902120641
3266094|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200902120728
3265384|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|2.0|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200902120644
3265374|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|2.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200902120644
3265360|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|2.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200902120643
3265350|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|File-Find-Rule-Permissions|2.0|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200902120642
3262299|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902120219
3262294|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902120218
3262234|pass|200902|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902120113
3262138|pass|200902|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902120103
3261996|pass|200902|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902120052
3261898|pass|200902|DCOLLINS@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902120041
3261709|pass|200902|srezic@cpan.org|File-Find-Rule-Permissions|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200902120108
3261096|cpan|200902|DCANTRELL|File-Find-Rule-Permissions|2.0||0|||200902120000
3259025|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902112039
3259008|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902112038
3258974|pass|200902|DCOLLINS@cpan.org|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902111436
3257998|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902111642
3257963|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Palm-SMS|0.03|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902111626
3257851|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200902111702
3257662|pass|200902|stsc@refcnt.org ("Steven Schubiger")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902111527
3257582|fail|200902|stsc@refcnt.org ("Steven Schubiger")|Bryar|3.0|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902111508
3256844|pass|200902|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|1.12|i686-linux|5.10.0|linux|2.6.18-14-ovz-686|200902111334
3255941|pass|200902|DCOLLINS@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902110701
3255810|pass|200902|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.02|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902110651
3255635|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200902111125
3255279|pass|200902|stro@cpan.org|Class-DBI-ClassGenerator|1.02|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200902111226
3255045|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200902111054
3254633|pass|200902|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.02|i86pc-solaris-64int|5.10.0|solaris|2.11|200902111002
3254544|pass|200902|stro@cpan.org|Class-DBI-ClassGenerator|1.02|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200902111048
3253196|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200902110022
3253193|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200902110021
3253190|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200902110021
3253184|pass|200902|bingos@cpan.org|Class-DBI-ClassGenerator|1.02|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200902110020
3251800|fail|200902|srezic@cpan.org|Class-DBI-ClassGenerator|1.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200902102137
3251797|fail|200902|srezic@cpan.org|Class-DBI-ClassGenerator|1.02|amd64-freebsd|5.8.8|freebsd|7.0-release|200902102136
3250744|pass|200902|vpit@cpan.org|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200902101730
3250474|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.02|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200902101619
3250472|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.02|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200902101619
3250471|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.02|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200902101618
3250417|pass|200902|andreas.koenig.gmwojprw@franz.ak.mind.de|Class-DBI-ClassGenerator|1.02|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200902101618
3250415|cpan|200902|DCANTRELL|Class-DBI-ClassGenerator|1.02||0|||200902101617
3248801|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200902100738
3248516|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.11.0 patch GitLive-blead-571-gabd7186|freebsd|7.0-release|200902100616
3241426|pass|200902|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902090952
3238067|pass|200902|cpan.dcollins.s3@gmail.com|File-Overwrite|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902081811
3237946|fail|200902|cpan.dcollins.s3@gmail.com|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902081759
3236420|pass|200902|cpan.dcollins.5100rc2@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902090231
3235347|pass|200902|cpan.dcollins.s1@gmail.com|File-Overwrite|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902081818
3235226|fail|200902|cpan.dcollins.s1@gmail.com|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902081804
3234863|pass|200902|cpan.dcollins.s3@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902080058
3234610|pass|200902|cpan.dcollins.s3@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902080440
3234425|pass|200902|cpan.dcollins.s3@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902080247
3231564|pass|200902|cpan.dcollins.589@gmail.com|Devel-CheckLib|0.5|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902081118
3229992|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200902081022
3229661|pass|200902|cpan.dcollins.s1@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080429
3229624|pass|200902|cpan.dcollins.5100rc2@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080412
3229365|pass|200902|cpan.dcollins.blead@gmail.com (cpanblead)|Data-Transactional|1.02|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902080337
3229258|pass|200902|cpan.dcollins.s2@gmail.com|Data-Transactional|1.02|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080325
3228672|pass|200902|cpan.dcollins.5100rc2@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080223
3228645|pass|200902|cpan.dcollins.s1@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080221
3228535|pass|200902|cpan.dcollins.blead@gmail.com (cpanblead)|Data-Hexdumper|1.4|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902080204
3228468|pass|200902|cpan.dcollins.s2@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080154
3228096|pass|200902|cpan.dcollins.s1@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080048
3228080|pass|200902|cpan.dcollins.s2@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080047
3228078|pass|200902|cpan.dcollins.5100rc2@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902080046
3228066|pass|200902|cpan.dcollins.blead@gmail.com (cpanblead)|Data-Compare|1.21|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902080045
3226001|pass|200902|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902072146
3225616|fail|200902|cpan.dcollins.589@gmail.com|Bryar|3.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902071557
3225185|fail|200902|cpan.dcollins.5100@gmail.com|Bryar|3.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902071454
3225052|fail|200902|cpan.dcollins.blead@gmail.com (cpanblead)|Bryar|3.0|i686-linux-thread-multi|5.11.0|linux|2.6.24-19-generic|200902071438
3224768|fail|200902|cpan.dcollins.5100rc2@gmail.com|Bryar|3.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902071412
3224567|fail|200902|cpan.dcollins.5100rc1@gmail.com|Bryar|3.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902071349
3221614|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200902071240
3220795|pass|200902|conrad@fringehead.org|XML-Tiny|1.12|i686-linux-thread-multi|5.8.8|linux|2.6.15.1|200902070724
3219384|pass|200902|cpan.dcollins.589@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902070214
3219367|pass|200902|cpan.dcollins.5100rc1@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902070212
3219189|pass|200902|cpan.dcollins.blead@gmail.com (cpanblead)|Acme-Licence|1.0|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902070159
3218884|pass|200902|cpan.dcollins.5100rc2@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902070140
3218624|pass|200902|cpan.dcollins.5100rc1@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902070118
3218461|pass|200902|cpan.dcollins.blead@gmail.com (cpanblead)|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902070106
3217037|pass|200902|cpan.dcollins.5100rc2@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902061620
3217019|pass|200902|cpan.dcollins.5100@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902061619
3216816|pass|200902|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902061600
3216711|pass|200902|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902061552
3215542|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200902061735
3213352|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200902061023
3213350|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200902061023
3208246|pass|200902|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.03|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902051031
3206339|pass|200902|cpan.dcollins.s3@gmail.com|XML-Tiny|1.12|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902050349
3205407|fail|200902|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902050311
3204156|pass|200902|cpan.dcollins.s1@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902050246
3204141|pass|200902|cpan.dcollins.s2@gmail.com|Games-Dice-Advanced|1.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902050245
3203703|pass|200902|cpan.dcollins.5100rc2@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902050153
3203592|pass|200902|cpan.dcollins.5100@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902050138
3203507|pass|200902|cpan.dcollins.589@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902050126
3203291|pass|200902|cpan.dcollins.blead@gmail.com|XML-Tiny|1.12|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902050051
3202691|pass|200902|cpan.dcollins.5100rc1@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902050005
3202413|pass|200902|cpan.dcollins.blead@gmail.com|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902041845
3201739|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200902042227
3201070|pass|200902|cpan.dcollins.5100@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902041642
3201003|pass|200902|cpan.dcollins.589@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902041637
3200634|pass|200902|cpan.dcollins.5100rc1@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902041611
3199931|pass|200902|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.12|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902041348
3197695|pass|200902|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200902041136
3192224|pass|200902|cpan.dcollins.589@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200902031541
3192181|pass|200902|cpan.dcollins.5100rc1@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902031533
3191738|pass|200902|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200902031317
3184408|pass|200902|david@cantrell.org.uk|Devel-CheckOS|1.50|alpha-netbsd|5.10.0|netbsd|4.0.1|200902021525
3183902|pass|200902|mw487.yahoo.com@gmail.com|XML-Tiny|1.12|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200902020835
3183037|pass|200902|"Rick Myers <rick@jrmyers.net>"|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.24.3|200902020456
3182198|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.10.0|linux|2.6.18-92.el5|200902020811
3181192|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.10.0|linux|2.6.18-92.el5|200902020221
3180592|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200902020000
3179541|pass|200902|cpan.dcollins.s3@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902010426
3179540|pass|200902|cpan.dcollins.s3@gmail.com|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200902010425
3175292|pass|200902|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckLib|0.5|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200902010655
3175190|pass|200902|cpan.dcollins.s2@gmail.com|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902010026
3175187|pass|200902|cpan.dcollins.s2@gmail.com|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902010025
3175183|pass|200902|cpan.dcollins.s2@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902010025
3174834|pass|200902|cpan.dcollins.s2@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200902010413
3172441|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.18-4-686|200901312121
3170596|pass|200901|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901311033
3170593|pass|200901|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901311032
3170148|pass|200901|cpan.dcollins.5100rc2@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901310956
3169338|pass|200901|cpan.dcollins.5100@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901310904
3167132|pass|200901|stro@cpan.org|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.00|200901310037
3166563|pass|200901|stro@cpan.org|Palm-SMS|0.03|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.00|200901301436
3165345|pass|200901|stsc@refcnt.org ("Steven Schubiger")|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200901302022
3164789|pass|200901|stro@cpan.org|Data-Compare|1.21|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.00|200901300941
3164246|fail|200901|stro@cpan.org|Bryar|3.0|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.00|200901300758
3162691|pass|200901|rafl@debian.org|XML-Tiny|1.12|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200901301249
3161599|pass|200901|vovkasm@gmail.com|Devel-CheckOS|1.50|amd64-freebsd|5.10.0|freebsd|6.2-release-p1|200901291221
3160855|pass|200901|stro@cpan.org|XML-Tiny|1.12|MSWin32-x86-multi-thread|5.8.9|MSWin32|5.00|200901300559
3160093|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200901300249
3157695|pass|200901|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200901291950
3156187|pass|200901|andy@hexten.net|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.27-11-generic|200901291442
3155540|pass|200901|rhaen@cpan.org (Ulrich Habel)|File-Find-Rule-Permissions|1.3|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901291236
3155515|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200901291200
3153623|pass|200901|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200901290218
3153617|pass|200901|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200901290218
3153540|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200901290155
3153489|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200901290133
3152896|pass|200901|cpan.dcollins.589@gmail.com|XML-Tiny|1.12|i686-linux-thread-multi|0|linux|2.6.24-19-generic|200901290345
3152825|fail|200901|cpan.dcollins.5100@gmail.com|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901290342
3152777|fail|200901|cpan.dcollins.blead@gmail.com|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901290338
3152204|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.10.0|dragonfly|2.0.1-release|200901290241
3151148|pass|200901|bingos@cpan.org|XML-Tiny|1.12|alpha-linux-thread-multi|5.8.9|linux|2.6.18-5-alpha-generic|200901281647
3150835|pass|200901|bingos@cpan.org|XML-Tiny|1.12|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200901281559
3150295|pass|200901|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|x86_64-linux-ld|5.10.0|linux|2.6.27-9-generic|200901281506
3150091|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200901281345
3149104|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200901281115
3148341|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.8.8|dragonfly|2.0.1-release|200901280757
3148256|pass|200901|rlane10@wi.rr.com|XML-Tiny|1.12|darwin-2level-ld|5.10.0|darwin|8.11.0|200901271703
3148184|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.8.9|dragonfly|2.0.1-release|200901280748
3147913|pass|200901|rhaen@cpan.org (Ulrich Habel)|CPAN-FindDependencies|2.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901281125
3146470|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200901280201
3146371|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200901280126
3144749|pass|200901|cjfields@bioperl.org|XML-Tiny|1.12|darwin-2level|5.10.0|darwin|9.2.0|200901271052
3143418|pass|200901|rlane10@wi.rr.com|Devel-CheckOS|1.50|darwin-2level-ld|5.10.0|darwin|8.11.0|200901270153
3142519|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200901271139
3142492|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.8.8|dragonfly|2.0.1-release|200901271224
3141201|pass|200901|rhaen@cpan.org (Ulrich Habel)|Devel-CheckLib|0.5|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901270439
3139208|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200901270020
3139120|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200901270014
3138299|pass|200901|srezic@cpan.org|XML-Tiny|1.12|amd64-freebsd|5.6.2|freebsd|7.0-release|200901262311
3137864|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.8.9|dragonfly|2.0.1-release|200901262051
3136871|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.18-4-686|200901261733
3135813|pass|200901|srezic@cpan.org|XML-Tiny|1.12|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200901261309
3135783|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200901261204
3135680|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.8.8|dragonfly|2.0.0-release|200901261141
3133852|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200901260346
3133836|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-dragonfly-64int|5.8.9|dragonfly|2.0.0-release|200901260340
3133678|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200901260246
3133636|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200901260223
3132000|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.18-4-686|200901251705
3130866|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200901251109
3130481|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|x86_64-linux|5.10.0|linux|2.6.26-1-amd64|200901251022
3129721|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200901250314
3129686|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200901250159
3128767|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200901242313
3128380|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200901242139
3127495|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200901241718
3126741|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200901241546
3126734|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200901241543
3126707|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200901241438
3126500|pass|200901|vpit@cpan.org|XML-Tiny|1.12|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200901241509
3126151|fail|200901|bingos@cpan.org|Bryar|3.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200901241232
3125882|pass|200901|srezic@cpan.org|XML-Tiny|1.12|amd64-freebsd|5.10.0|freebsd|7.0-release|200901241208
3125747|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i686-linux-thread-multi-64int|5.8.8|linux|2.4.26|200901241042
3125657|pass|200901|rhaen@cpan.org (Ulrich Habel)|Number-Phone|1.6|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901241015
3124997|pass|200901|rhaen@cpan.org (Ulrich Habel)|CPU-Emulator-Z80|1.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901240538
3124996|pass|200901|rhaen@cpan.org (Ulrich Habel)|CPU-Emulator-Memory|1.1001|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901240538
3124901|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200901240508
3124845|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200901240449
3124819|pass|200901|rhaen@cpan.org (Ulrich Habel)|Data-Transactional|1.02|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901240442
3124660|pass|200901|bingos@cpan.org|Bryar|3.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200901240355
3123823|pass|200901|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200901240027
3123194|pass|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200901232207
3122728|pass|200901|bingos@cpan.org|Bryar|3.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200901232029
3122534|pass|200901|stro@cpan.org|XML-Tiny|1.12|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200901232155
3121964|pass|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200901231757
3121822|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200901231134
3121805|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|alpha-netbsd|5.10.0|netbsd|4.0.1|200901231732
3121783|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200901231729
3121780|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200901231729
3121519|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200901231637
3121468|pass|200901|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200901231624
3121414|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901231614
3121412|fail|200901|bingos@cpan.org|Bryar|3.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200901231613
3121313|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901231606
3121194|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|IP35-irix-64int|5.8.8|irix|6.5|200901231653
3121016|pass|200901|david@cantrell.org.uk|XML-Tiny|1.12|x86_64-linux-thread-multi|5.6.2|linux|2.6.18-4-xen-amd64|200901231510
3120883|pass|200901|bingos@cpan.org|XML-Tiny|1.12|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200901231435
3120191|pass|200901|stro@cpan.org|XML-Tiny|1.12|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200901231349
3120019|pass|200901|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|1.12|i86pc-solaris-64int|5.10.0|solaris|2.11|200901231250
3119868|pass|200901|bingos@cpan.org|XML-Tiny|1.12|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200901231104
3119819|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200901231154
3119755|pass|200901|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|1.12|i686-linux|5.10.0|linux|2.6.18-4-686|200901231144
3119557|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200901231119
3119547|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200901231117
3119538|pass|200901|rhaen@cpan.org (Ulrich Habel)|Palm-Treo680MessagesDB|1.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901231015
3119536|pass|200901|rhaen@cpan.org (Ulrich Habel)|Data-Hexdumper|1.4|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901231015
3119496|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200901231107
3119468|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200901231102
3119467|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901231101
3119457|fail|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200901230847
3119456|pass|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200901230847
3119384|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200901231045
3119175|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200901231013
3119145|pass|200901|imacat@mail.imacat.idv.tw|XML-Tiny|1.12|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200901231703
3119057|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901230944
3118982|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901230931
3118883|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901230915
3118848|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.12|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901230907
3118327|pass|200901|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|XML-Tiny|1.12|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200901231542
3118228|pass|200901|rhaen@cpan.org (Ulrich Habel)|Sort-MultipleFields|1.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901230618
3118210|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200901230712
3118208|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200901230712
3118207|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200901230711
3118206|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.12|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200901230711
3117644|pass|200901|davidrw@cpan.org|XML-Tiny|1.12|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901230321
3117367|pass|200901|bingos@cpan.org|XML-Tiny|1.12|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200901230058
3117366|pass|200901|bingos@cpan.org|XML-Tiny|1.12|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200901230057
3117364|pass|200901|bingos@cpan.org|XML-Tiny|1.12|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200901230057
3117362|pass|200901|bingos@cpan.org|XML-Tiny|1.12|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200901230056
3117361|pass|200901|bingos@cpan.org|XML-Tiny|1.12|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200901230056
3117087|pass|200901|srezic@cpan.org|XML-Tiny|1.12|amd64-freebsd|5.8.8|freebsd|7.0-release|200901222352
3117085|pass|200901|srezic@cpan.org|XML-Tiny|1.12|amd64-freebsd|5.8.9|freebsd|7.0-release|200901222352
3117051|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200901230015
3116396|pass|200901|bingos@cpan.org|Bryar|3.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200901221937
3115892|pass|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200901221741
3115655|cpan|200901|DCANTRELL|XML-Tiny|1.12||0|||200901221756
3115222|pass|200901|jeff@zeroclue.com|Devel-CheckOS|1.50|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200901221507
3114664|pass|200901|jeff@zeroclue.com|Palm-SMS|0.03|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200901221238
3114002|pass|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200901220912
3113680|fail|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200901220746
3113641|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200901220740
3113403|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200901220631
3110600|pass|200901|jeff@zeroclue.com|Data-Compare|1.21|OpenBSD.i386-openbsd|5.8.9|openbsd|4.4|200901211747
3110542|pass|200901|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200901211610
3108972|pass|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200901211210
3107565|pass|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200901210553
3107068|pass|200901|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200901210203
3106869|fail|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200901210230
3106475|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901210141
3105743|pass|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200901202105
3104736|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200901201620
3104600|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901201745
3104594|pass|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200901201511
3103902|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200901201433
3103802|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901201514
3103416|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Palm-SMS|0.03|i686-linux|5.10.0|linux|2.6.18-4-686|200901201357
3102350|pass|200901|david@cantrell.org.uk|Devel-CheckLib|0.5|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200919
3102349|pass|200901|david@cantrell.org.uk|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200919
3102344|pass|200901|david@cantrell.org.uk|Devel-CheckLib|0.5|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901200919
3102343|pass|200901|david@cantrell.org.uk|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901200919
3102336|pass|200901|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200918
3102335|pass|200901|david@cantrell.org.uk|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901200918
3102333|pass|200901|david@cantrell.org.uk|Data-Hexdumper|1.4|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200917
3102331|pass|200901|david@cantrell.org.uk|Data-Hexdumper|1.4|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901200917
3102316|pass|200901|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200914
3102315|pass|200901|david@cantrell.org.uk|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901200914
3102298|pass|200901|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901200911
3102288|pass|200901|david@cantrell.org.uk|DBIx-Class-SingletonRows|0.11|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200910
3102255|pass|200901|david@cantrell.org.uk|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200902
3102253|pass|200901|david@cantrell.org.uk|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901200902
3101854|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200901200624
3101452|fail|200901|bingos@cpan.org|Bryar|3.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200901200437
3101252|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200901200336
3101214|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200901200327
3100701|pass|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200901200112
3100693|pass|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200901200111
3099593|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200901192222
3098209|pass|200901|vpit@cpan.org|Data-Compare|1.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200901192012
3096218|fail|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200901191507
3094992|pass|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200901191135
3094816|pass|200901|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200901191104
3094021|pass|200901|mw487.yahoo.com@gmail.com|Palm-SMS|0.03|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200901190317
3093903|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200901190842
3093840|pass|200901|rhaen@cpan.org (Ulrich Habel)|XML-Tiny|1.11|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901190727
3093276|pass|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200901190415
3092109|pass|200901|davidrw@cpan.org|Palm-SMS|0.03|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901181726
3091579|fail|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200901182024
3091576|pass|200901|bingos@cpan.org|Bryar|3.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200901182023
3090667|pass|200901|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.6.2|freebsd|7.0-release|200901181839
3088650|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|x86_64-linux-thread-multi|5.10.0|linux|2.6.18-4-xen-amd64|200901181322
3088616|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|x86_64-linux-thread-multi|5.8.9|linux|2.6.18-4-xen-amd64|200901181318
3088536|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200901181254
3087651|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.18-4-686|200901181040
3087642|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200901181036
3087406|pass|200901|bingos@cpan.org|Data-Compare|1.21|MirBSD.i386-mirbsd-thread-multi-64int|5.10.0|mirbsd|10|200901180805
3086173|fail|200901|rhaen@cpan.org (Ulrich Habel)|Bryar|3.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901180013
3085556|fail|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200901172221
3085423|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200901172306
3085316|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200901172246
3085202|fail|200901|david@cantrell.org.uk|Bryar|3.0|alpha-netbsd|5.10.0|netbsd|4.0.1|200901172135
3085078|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200901172216
3084958|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200901172201
3084921|fail|200901|david@cantrell.org.uk|Bryar|3.0|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200901172056
3084851|pass|200901|bingos@cpan.org|Palm-SMS|0.03|MirBSD.i386-mirbsd-thread-multi-64int|5.10.0|mirbsd|10|200901172048
3084638|fail|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200901172128
3084438|fail|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901172106
3084426|fail|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200901172105
3084385|fail|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200901172058
3084160|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901172023
3084047|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901172007
3083867|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200901171848
3083833|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901171941
3083730|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Bryar|3.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901171928
3083183|fail|200901|srezic@cpan.org|Bryar|3.0|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200901171727
3083181|fail|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200901171631
3083075|fail|200901|bingos@cpan.org|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200901171606
3082367|fail|200901|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.10.0|freebsd|7.0-release|200901171254
3082188|pass|200901|bingos@cpan.org|Data-Compare|1.21|MirBSD.i386-mirbsd-thread-multi-64int|5.8.8|mirbsd|10|200901171047
3082067|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200901171023
3082000|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200901171011
3081216|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.18-4-686|200901170539
3080893|pass|200901|bingos@cpan.org|Palm-SMS|0.03|MirBSD.i386-mirbsd-thread-multi-64int|5.8.8|mirbsd|10|200901170310
3080878|pass|200901|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200901170309
3080494|pass|200901|bingos@cpan.org|Bryar|3.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200901170135
3080489|pass|200901|bingos@cpan.org|Bryar|3.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200901170134
3080479|fail|200901|bingos@cpan.org|Bryar|3.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200901170133
3080154|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200901162347
3079684|pass|200901|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200901162124
3078878|pass|200901|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200901161836
3078818|pass|200901|stro@cpan.org|Palm-SMS|0.03|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200901162017
3078481|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|alpha-netbsd|5.10.0|netbsd|4.0.1|200901161717
3078438|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200901161710
3078430|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Bryar|3.0|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200901161710
3078106|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200901161557
3077829|pass|200901|davidrw@cpan.org|Tie-Scalar-Decay|1.1.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901161016
3077761|pass|200901|davidrw@cpan.org|Tie-Hash-Rank|1.0.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901161000
3077674|unknown|200901|davidrw@cpan.org|Acme-Pony|1.1.2|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901160944
3076974|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200901161005
3076468|pass|200901|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200901160745
3075501|pass|200901|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200901160244
3075384|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.6.2|solaris|2.10|200901160205
3075263|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200901160127
3073733|unknown|200901|davidrw@cpan.org|Acme-Licence|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901151351
3072868|pass|200901|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.50|x86_64-linux-thread-multi-ld|5.8.9|linux|2.6.26-1-amd64|200901151555
3071928|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200901150934
3071881|pass|200901|billf@FreeBSD.org ("Bill Fumerola")|Data-Compare|1.21|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18-6-686|200901150502
3071702|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200901150848
3070623|pass|200901|davidrw@cpan.org|NestedMap|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901150300
3069556|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200901142247
3069447|pass|200901|g.grigelionis@computer.org|Statistics-SerialCorrelation|1.1|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200901142332
3069414|pass|200901|g.grigelionis@computer.org|Statistics-ChiSquare|0.5|i386-linux-thread-multi|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200901142328
3068845|pass|200901|davidrw@cpan.org|Statistics-ChiSquare|0.5|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901141522
3068842|pass|200901|davidrw@cpan.org|Tie-Hash-Longest|1.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901141522
3068176|pass|200901|davidrw@cpan.org|Statistics-SerialCorrelation|1.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901141202
3066864|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200901141226
3064835|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200901140213
3064812|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200901140209
3063656|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200901131705
3062953|pass|200901|davidrw@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901131345
3062904|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.1-release|200901131439
3062876|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200901131934
3062866|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200901131833
3062786|fail|200901|davidrw@cpan.org|Palm-SMS|0.02|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901131320
3062551|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200901131827
3061988|pass|200901|bingos@cpan.org|Palm-SMS|0.03|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200901131530
3061210|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200901131417
3061039|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901131400
3060926|na|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901131342
3060676|na|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901131256
3060174|pass|200901|rhaen@cpan.org (Ulrich Habel)|Palm-SMS|0.03|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200901131039
3059229|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200901130719
3058988|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200901130645
3057888|pass|200901|bingos@cpan.org|Palm-SMS|0.03|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200901130306
3057594|pass|200901|bingos@cpan.org|Palm-SMS|0.03|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200901130224
3056648|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.10.0|linux|2.6.18-4-686|200901130030
3056023|pass|200901|davidrw@cpan.org|Class-CanBeA|1.2|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901121639
3055888|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901122223
3055877|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200901122120
3055379|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.1-release|200901122016
3055374|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200901122015
3055305|pass|200901|davidrw@cpan.org|File-Overwrite|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901121509
3055128|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.1-release|200901121940
3055025|pass|200901|vpit@cpan.org|Palm-SMS|0.03|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200901122043
3052903|fail|200901|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901121535
3052901|unknown|200901|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901121535
3052262|pass|200901|mohsen_basirat@yahoo.com ("Mohsen Basirat")|Devel-CheckOS|1.50|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.15.7|200901121712
3052076|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200901121238
3051993|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200901121217
3051972|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200901121204
3051816|na|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901121104
3051755|na|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-SMS|0.03|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901121042
3051732|pass|200901|bingos@cpan.org|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200901120939
3050815|pass|200901|davidrw@cpan.org|Sub-WrapPackages|1.2|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901120204
3049614|pass|200901|davidrw@cpan.org|Games-Dice-Advanced|1.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901120250
3049355|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200901120147
3049043|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i386-netbsd-thread-multi-64int|5.8.9|netbsd|3.1|200901120101
3048049|pass|200901|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200901112144
3047835|pass|200901|bingos@cpan.org|Number-Phone|1.6|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200901111319
3047586|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200901111422
3047439|pass|200901|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.03|i86pc-solaris-64int|5.10.0|solaris|2.11|200901112123
3046858|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200901111813
3046741|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|IP35-irix-64int|5.8.8|irix|6.5|200901111842
3046733|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200901111741
3046732|pass|200901|david@cantrell.org.uk|Palm-SMS|0.03|i686-linux|5.8.8|linux|2.4.27-3-686|200901111739
3046521|pass|200901|davidrw@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901111201
3046396|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901111740
3045969|pass|200901|imacat@mail.imacat.idv.tw|Palm-SMS|0.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200901112222
3045905|pass|200901|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200901110805
3045453|na|200901|srezic@cpan.org|Palm-SMS|0.03|amd64-freebsd|5.6.2|freebsd|7.0-release|200901111226
3045241|pass|200901|bingos@cpan.org|Palm-SMS|0.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200901111027
3045176|pass|200901|stro@cpan.org|Palm-SMS|0.03|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200901111157
3045008|pass|200901|davidrw@cpan.org|Net-Random|2.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901110346
3044048|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901110423
3043574|pass|200901|bingos@cpan.org|Palm-SMS|0.03|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200901110037
3043571|pass|200901|bingos@cpan.org|Palm-SMS|0.03|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200901110036
3043569|pass|200901|bingos@cpan.org|Palm-SMS|0.03|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200901110035
3043567|pass|200901|bingos@cpan.org|Palm-SMS|0.03|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200901110034
3043291|pass|200901|srezic@cpan.org|Palm-SMS|0.03|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200901110038
3042788|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901102307
3042724|pass|200901|davidrw@cpan.org|Palm-TreoPhoneCallDB|1.1|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901101653
3042646|pass|200901|srezic@cpan.org|Palm-SMS|0.03|amd64-freebsd|5.10.0|freebsd|7.0-release|200901102230
3042384|pass|200901|srezic@cpan.org|Palm-SMS|0.03|amd64-freebsd|5.8.8|freebsd|7.0-release|200901102120
3042349|pass|200901|srezic@cpan.org|Palm-SMS|0.03|amd64-freebsd|5.8.9|freebsd|7.0-release|200901102120
3042321|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901102116
3042195|pass|200901|cpan.dcollins.5100rc2@gmail.com|Devel-CheckLib|0.5|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901101450
3042141|pass|200901|cpan.dcollins.5100rc1@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901101443
3041909|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckLib|0.5|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901101400
3041654|pass|200901|davidrw@cpan.org|Tie-STDOUT|1.03|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901101247
3041195|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-SMS|0.03|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200901101721
3041179|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-SMS|0.03|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200901101722
3041177|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-SMS|0.03|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200901101722
3041175|pass|200901|andreas.koenig.gmwojprw@franz.ak.mind.de|Palm-SMS|0.03|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200901101721
3041172|cpan|200901|DCANTRELL|Palm-SMS|0.03||0|||200901101720
3038236|pass|200901|davidrw@cpan.org|File-Find-Rule-Permissions|1.3|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901100102
3037456|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901100503
3036705|pass|200901|davidrw@cpan.org|Bryar|3.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901100141
3035762|pass|200901|davidrw@cpan.org|CPAN-FindDependencies|2.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901091803
3035629|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901092346
3035363|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901092309
3034980|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.4-release|200901092019
3034562|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901092059
3034476|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901092038
3032394|pass|200901|davidrw@cpan.org|Devel-CheckLib|0.5|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901090906
3031056|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901090940
3030465|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901090743
3030353|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901090720
3030345|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200901090719
3028034|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.4-release|200901082205
3028031|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.4-release|200901082204
3026216|pass|200901|stsc@refcnt.org ("Steven Schubiger")|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.18-4-686|200901081931
3026194|pass|200901|davidrw@cpan.org|Number-Phone|1.6|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901081254
3025405|pass|200901|davidrw@cpan.org|CPU-Emulator-Z80|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901081046
3025402|pass|200901|davidrw@cpan.org|CPU-Emulator-Memory|1.1001|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901081046
3025313|pass|200901|davidrw@cpan.org|Data-Transactional|1.02|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901081022
3025042|pass|200901|davidrw@cpan.org|DBIx-Class-SingletonRows|0.11|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901080854
3023203|pass|200901|davidrw@cpan.org|Palm-Treo680MessagesDB|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901080417
3023078|pass|200901|davidrw@cpan.org|Data-Hexdumper|1.4|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901080347
3022740|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901080310
3022738|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901080309
3021121|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200901072157
3021091|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200901072153
3020801|pass|200901|cpan.dcollins.5100rc2@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901071613
3020466|pass|200901|davidrw@cpan.org|Sort-MultipleFields|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901071550
3018177|pass|200901|cpan.dcollins.589@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901070730
3018162|pass|200901|cpan.dcollins.589@gmail.com|Tie-STDOUT|1.03|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901070729
3018049|pass|200901|cpan.dcollins.blead@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070717
3018043|pass|200901|cpan.dcollins.blead@gmail.com|Tie-STDOUT|1.03|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070717
3017965|pass|200901|cpan.dcollins.589@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901070709
3017959|pass|200901|cpan.dcollins.589@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901070708
3017848|pass|200901|cpan.dcollins.blead@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070657
3017842|pass|200901|cpan.dcollins.blead@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070657
3017659|pass|200901|cpan.dcollins.589@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901070642
3017425|pass|200901|g.grigelionis@computer.org|Statistics-SerialCorrelation|1.1|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200901071218
3017384|pass|200901|g.grigelionis@computer.org|Statistics-ChiSquare|0.5|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200901071212
3017173|pass|200901|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901070523
3016985|pass|200901|g.grigelionis@computer.org|Statistics-SerialCorrelation|1.1|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200901071103
3016961|pass|200901|g.grigelionis@computer.org|Statistics-ChiSquare|0.5|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200901071102
3016712|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070434
3016659|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070428
3016625|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070424
3016455|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070404
3016330|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901070353
3014615|pass|200901|davidrw@cpan.org|Class-DBI-ClassGenerator|1.01|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901070159
3014349|pass|200901|cpan.dcollins.s3@gmail.com|Tie-STDOUT|1.03|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901061658
3013892|pass|200901|cpan.dcollins.s3@gmail.com|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901061659
3013872|pass|200901|cpan.dcollins.s3@gmail.com|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901061637
3013870|pass|200901|cpan.dcollins.s3@gmail.com|Tie-Hash-Longest|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901061635
3012853|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901062203
3012845|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200901062203
3012451|pass|200901|davidrw@cpan.org|XML-Tiny|1.11|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200901061526
3011821|fail|200901|imacat@mail.imacat.idv.tw|Palm-SMS|0.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200901070326
3010481|pass|200901|stsc@refcnt.org ("Steven Schubiger")|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.18-4-686|200901061820
3009704|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.8.9|dragonfly|2.0.0-release|200901061600
3009674|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200901061558
3008585|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200901061355
3007551|pass|200901|cpan.dcollins.s2@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901060633
3007544|pass|200901|cpan.dcollins.s1@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901060632
3006895|pass|200901|alceu.freitas.jr@gmail.com|Devel-CheckOS|1.50|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.24-14-server|200901060041
3003702|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200901060245
3002946|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.8.8|dragonfly|2.0.0-release|200901060033
3002714|pass|200901|cpan.dcollins.589@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901060007
3002553|pass|200901|cpan.dcollins.blead@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901051852
3002521|pass|200901|cpan.dcollins.5100rc2@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901051851
3002332|pass|200901|cpan.dcollins.5100@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901051843
3002265|pass|200901|cpan.dcollins.5100rc1@gmail.com|Class-CanBeA|1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901051840
3002205|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.10.0|dragonfly|2.0.0-release|200901052336
3002020|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200901052322
2998888|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901051710
2998880|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200901051709
2998800|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901051657
2998795|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200901051656
2997005|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200901051045
2995926|pass|200901|fayland@gmail.com ("Fayland Lam")|Devel-CheckOS|1.50|i386-linux-thread-multi|5.8.6|linux|2.6.9-34.elsmp|200901050300
2995415|pass|200901|bobtfish@bobtfish.net|Devel-CheckOS|1.50|darwin-thread-multi-2level|5.8.6|darwin|8.0|200901042109
2995379|pass|200901|cpan.dcollins.s1@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901041443
2994599|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-64int|5.8.9|dragonfly|2.0.1-release|200901050025
2993573|pass|200901|cpan.dcollins.589@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901041638
2993275|pass|200901|cpan.dcollins.5100rc2@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041611
2993227|pass|200901|g.grigelionis@computer.org|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200901042207
2993215|pass|200901|cpan.dcollins.5100rc1@gmail.com|Data-Hexdumper|1.4|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041606
2993181|pass|200901|g.grigelionis@computer.org|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200901042205
2992512|pass|200901|cpan.dcollins.589@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901041452
2992423|pass|200901|cpan.dcollins.5100@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041444
2992362|pass|200901|cpan.dcollins.5100rc2@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041439
2992353|pass|200901|cpan.dcollins.5100rc1@gmail.com|Data-Compare|1.21|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041438
2992341|pass|200901|cpan.dcollins.5100rc1@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041437
2990736|pass|200901|cpan.dcollins.5100@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041218
2990588|pass|200901|cpan.dcollins.5100@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041158
2990404|pass|200901|cpan.dcollins.blead@gmail.com|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901041140
2990207|pass|200901|cpan.dcollins.blead@gmail.com|Acme-Pony|1.1.2|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901041121
2990077|pass|200901|cpan.dcollins.5100@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901041115
2989644|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200901041630
2989567|pass|200901|cpan.dcollins.blead@gmail.com|Acme-Licence|1.0|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901041021
2989367|pass|200901|cpan.dcollins.5100rc2@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901040956
2989302|pass|200901|cpan.dcollins.5100@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.24-19-generic|200901040946
2989280|pass|200901|cpan.dcollins.589@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.8.9|linux|2.6.24-19-generic|200901040940
2989226|pass|200901|cpan.dcollins.blead@gmail.com|Devel-CheckOS|1.50|i686-linux-thread-multi|5.11.0 patch GitLive-blead-163-g28b1dae|linux|2.6.24-19-generic|200901040921
2988399|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200901041000
2984385|fail|200901|stsc@refcnt.org ("Steven Schubiger")|Bryar|3.0|i686-linux|5.10.0|linux|2.6.18-4-686|200901031821
2983106|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200901031244
2980384|pass|200901|vutral@gmail.com|Devel-CheckOS|1.50|x86_64-linux-gnu-thread-multi|5.8.8|linux|2.6.26-1-vserver-amd64|200901021136
2977742|pass|200901|pjacklam@online.no|Statistics-SerialCorrelation|1.1|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200901021812
2977034|pass|200901|pjacklam@online.no|Statistics-ChiSquare|0.5|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200901021648
2975607|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200901020814
2975571|pass|200901|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200901020803
2972816|pass|200901|stsc@refcnt.org ("Steven Schubiger")|CPAN-FindDependencies|2.0|i686-linux|5.10.0|linux|2.6.18-4-686|200901020241
2970431|pass|200812|schwern@pobox.com|Data-Compare|1.21|darwin-thread-multi-2level|5.10.0|darwin|8.11.1|200812311350
2966094|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200901010440
2965522|pass|200901|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200901010245
2964386|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812312216
2964154|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812312109
2964005|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812312004
2964004|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812312003
2963683|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200812311840
2963351|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812311746
2963177|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812311740
2963147|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812311739
2962681|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812311512
2957197|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|DBIx-Class-SingletonRows|0.11|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812300611
2956497|pass|200812|bingos@cpan.org|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300219
2956495|pass|200812|bingos@cpan.org|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300219
2956494|pass|200812|bingos@cpan.org|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956487|pass|200812|bingos@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300221
2956474|pass|200812|bingos@cpan.org|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300222
2956465|pass|200812|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300218
2956463|pass|200812|bingos@cpan.org|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300222
2956461|pass|200812|bingos@cpan.org|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300221
2956457|pass|200812|bingos@cpan.org|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300221
2956456|pass|200812|bingos@cpan.org|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956454|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300224
2956453|pass|200812|bingos@cpan.org|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956452|pass|200812|bingos@cpan.org|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300223
2956432|pass|200812|bingos@cpan.org|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300217
2956431|pass|200812|bingos@cpan.org|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300218
2956429|pass|200812|bingos@cpan.org|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300216
2956428|pass|200812|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300215
2956422|pass|200812|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812300215
2956271|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812300130
2955398|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Number-Phone|1.6|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812292207
2953998|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812291628
2953993|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812291627
2953742|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812291512
2953740|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812291511
2951440|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812290139
2950457|pass|200812|bingos@cpan.org|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812282054
2949939|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200812281750
2949187|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200812281409
2948935|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812281405
2947328|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812280915
2947073|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812280846
2943341|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812272329
2943274|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0|linux|2.6.22-1-k7|200812272317
2940826|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200812271408
2939983|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.18-4-686|200812270958
2939739|pass|200812|davidrw@cpan.org|XML-DoubleEncodedEntities|1.0|i486-linux-gnu-thread-multi|5.8.8|linux|2.6.18.3|200812270346
2938494|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200812262114
2933659|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.21.5-smp|200812252043
2933026|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200812251348
2933025|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200812251348
2930317|pass|200812|yanick@cpan.org ("Yanick Champoux")|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.22-gentoo-r2enkidu|200812241237
2929898|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200812250031
2929897|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200812250031
2929486|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200812242118
2929091|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200812241937
2928116|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200812241554
2926853|pass|200812|chris@bingosnet.co.uk|Data-Compare|1.21|i686-linux-thread-multi-64int|5.6.2|linux|2.4.26|200812241109
2926641|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200812240957
2924535|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240004
2924504|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240457
2924466|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240452
2923611|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812240212
2923002|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200812232325
2922744|pass|200812|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231759
2922125|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200812232106
2921957|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231537
2921955|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231537
2921664|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231501
2921115|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200812231853
2919615|pass|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231050
2919546|fail|200812|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812231042
2919291|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200812231502
2918722|pass|200812|imacat@mail.imacat.idv.tw|XML-Tiny|1.11|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232209
2918721|pass|200812|imacat@mail.imacat.idv.tw|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232209
2918720|pass|200812|imacat@mail.imacat.idv.tw|Tie-Scalar-Decay|1.1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232209
2918718|pass|200812|imacat@mail.imacat.idv.tw|Tie-STDOUT|1.03|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232208
2918716|pass|200812|imacat@mail.imacat.idv.tw|Tie-Hash-Rank|1.0.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232208
2918712|pass|200812|imacat@mail.imacat.idv.tw|Tie-Hash-Longest|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232207
2918710|pass|200812|imacat@mail.imacat.idv.tw|Sub-WrapPackages|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232207
2918704|pass|200812|imacat@mail.imacat.idv.tw|Statistics-SerialCorrelation|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232207
2918700|pass|200812|imacat@mail.imacat.idv.tw|Statistics-ChiSquare|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232206
2918699|pass|200812|imacat@mail.imacat.idv.tw|Sort-MultipleFields|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232206
2918697|pass|200812|imacat@mail.imacat.idv.tw|Palm-TreoPhoneCallDB|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232206
2918680|pass|200812|imacat@mail.imacat.idv.tw|Palm-Treo680MessagesDB|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232204
2918668|pass|200812|imacat@mail.imacat.idv.tw|Number-Phone-UK-DetailedLocations|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232202
2918661|pass|200812|imacat@mail.imacat.idv.tw|Net-Random|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232201
2918650|pass|200812|imacat@mail.imacat.idv.tw|NestedMap|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232200
2918646|pass|200812|imacat@mail.imacat.idv.tw|Games-Dice-Advanced|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232159
2918643|pass|200812|imacat@mail.imacat.idv.tw|File-Overwrite|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232159
2918638|pass|200812|imacat@mail.imacat.idv.tw|File-Find-Rule-Permissions|1.3|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232159
2918634|pass|200812|imacat@mail.imacat.idv.tw|Devel-CheckLib|0.5|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232158
2918631|pass|200812|imacat@mail.imacat.idv.tw|Data-Transactional|1.02|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232158
2918628|pass|200812|imacat@mail.imacat.idv.tw|Data-Hexdumper|1.4|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232157
2918602|pass|200812|imacat@mail.imacat.idv.tw|Class-CanBeA|1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232152
2918601|pass|200812|imacat@mail.imacat.idv.tw|CPU-Emulator-Z80|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232151
2918589|pass|200812|imacat@mail.imacat.idv.tw|CPU-Emulator-Memory|1.1001|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232150
2918586|pass|200812|imacat@mail.imacat.idv.tw|CPAN-FindDependencies|2.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232150
2918583|fail|200812|imacat@mail.imacat.idv.tw|Bryar|3.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232149
2918563|pass|200812|imacat@mail.imacat.idv.tw|Acme-Scurvy-Whoreson-BilgeRat|1.1|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232146
2918561|unknown|200812|imacat@mail.imacat.idv.tw|Acme-Pony|1.1.2|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232145
2918558|unknown|200812|imacat@mail.imacat.idv.tw|Acme-Licence|1.0|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812232145
2917138|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230432
2916888|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230351
2915507|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200812230557
2915479|pass|200812|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200812230552
2915226|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230458
2914905|pass|200812|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230350
2913921|pass|200812|rhaen@cpan.org (Ulrich Habel)|Data-Compare|1.21|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200812230056
2913601|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812230007
2913521|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Z80|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812230057
2913520|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|CPU-Emulator-Memory|1.1001|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812230056
2913208|pass|200812|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812221817
2912851|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Transactional|1.02|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812222326
2911789|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200812221905
2911775|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200812221903
2910076|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200812221239
2910034|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200812221231
2909708|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220645
2909264|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220542
2908670|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220416
2908326|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Palm-Treo680MessagesDB|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812220926
2908320|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Hexdumper|1.4|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812220925
2907217|pass|200812|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220121
2906994|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812220523
2906678|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812220428
2905324|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812220006
2904781|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211756
2903009|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211211
2902634|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200812211616
2902457|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211052
2902449|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211052
2902335|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812211037
2902320|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200812211533
2901366|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210758
2901362|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210758
2900706|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.16.38|200812211142
2900660|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210640
2900393|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200812211106
2898774|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210302
2897833|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812210048
2896124|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200812210021
2895875|pass|200812|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200812202348
2895545|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200812202304
2894325|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812201534
2893502|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.6.2|netbsd|4.0|200812201849
2893169|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200812201808
2892522|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200812201714
2892266|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200812201627
2892048|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200812201605
2891047|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200931
2891001|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200928
2887212|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.9|darwin|8.11.0|200812200322
2887113|pass|200812|imacat@mail.imacat.idv.tw|Number-Phone|1.6|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812201113
2887031|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Scalar-Decay|1.1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200305
2886959|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Rank|1.0.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200256
2886887|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Pony|1.1.2|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200250
2886280|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200812200127
2885630|pass|200812|rhaen@cpan.org (Ulrich Habel)|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi|5.10.0|netbsd|5.0_beta|200812200028
2885389|unknown|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Licence|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812200009
2884187|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200812192209
2883928|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200812192238
2883908|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200812192235
2882982|pass|200812|dagolden@cpan.org (DAGOLDEN)|NestedMap|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191448
2882117|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-Hash-Longest|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191310
2882111|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-ChiSquare|0.5|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191310
2881720|pass|200812|dagolden@cpan.org (DAGOLDEN)|Statistics-SerialCorrelation|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812191233
2881650|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200812191723
2880250|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200812191538
2879718|pass|200812|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190830
2879660|fail|200812|dagolden@cpan.org (DAGOLDEN)|Palm-SMS|0.02|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190819
2877419|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812190843
2876685|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-CanBeA|1.2|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190216
2876360|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Overwrite|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190134
2874953|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sub-WrapPackages|1.2|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190259
2874665|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812190220
2874399|pass|200812|dagolden@cpan.org (DAGOLDEN)|Games-Dice-Advanced|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812190150
2873610|pass|200812|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.50|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812190833
2873297|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.6.2|freebsd|7.0-release|200812190105
2873201|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200812190000
2871924|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone-UK-DetailedLocations|1.3|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181711
2871351|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200812182222
2871313|pass|200812|dagolden@cpan.org (DAGOLDEN)|Net-Random|2.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181620
2871153|pass|200812|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200812182009
2870944|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200812182149
2870795|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200812182132
2870726|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200812182127
2870612|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200812182118
2870523|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200812182112
2870415|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812182102
2870238|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812182045
2870198|pass|200812|imacat@mail.imacat.idv.tw|Data-Compare|1.21|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.24-etchnhalf.1-amd64|200812190343
2869551|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-TreoPhoneCallDB|1.1|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181348
2869146|pass|200812|dagolden@cpan.org (DAGOLDEN)|Tie-STDOUT|1.03|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181305
2868906|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200812181830
2868649|pass|200812|dagolden@cpan.org (DAGOLDEN)|File-Find-Rule-Permissions|1.3|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812181203
2868648|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200812181801
2868546|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200812181749
2867800|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200812181506
2867688|pass|200812|dagolden@cpan.org (DAGOLDEN)|Bryar|3.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180954
2867138|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPAN-FindDependencies|2.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180825
2866826|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200812181205
2865836|pass|200812|vutral@gmail.com|Data-Compare|1.21|x86_64-linux-gnu|5.8.8|linux|2.6.26-1-vserver-amd64|200812180200
2864805|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckLib|0.5|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180358
2864717|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200812180844
2864474|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200812180809
2864465|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200812180809
2864423|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200812180724
2864299|pass|200812|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200812180750
2863975|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200812180758
2862970|pass|200812|dagolden@cpan.org (DAGOLDEN)|Number-Phone|1.6|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180409
2862849|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812180450
2862576|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180311
2862575|pass|200812|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Memory|1.1001|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180311
2862505|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180258
2862222|pass|200812|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180223
2861966|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200812180150
2861592|pass|200812|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180115
2861591|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Hexdumper|1.4|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180115
2861439|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Sort-MultipleFields|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200812180152
2861126|pass|200812|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812180001
2859596|pass|200812|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812171524
2858912|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-Tiny|1.11|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812171346
2857853|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812171615
2857770|pass|200812|rhaen@cpan.org (Ulrich Habel)|Devel-CheckOS|1.50|i386-netbsd|5.10.0|netbsd|5.0_beta|200812171618
2857148|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.9|linux|2.6.16.60-0.31-default|200812171628
2856657|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.18-4-686|200812171543
2856409|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171511
2856215|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171459
2855399|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.9|linux|2.6.18-92.1.18.el5|200812171405
2854913|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171312
2854842|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812171258
2854457|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.9|linux|2.6.18-92.1.18.el5|200812171144
2853285|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.9|linux|2.6.16.60-0.31-default|200812170858
2853227|pass|200812|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812170252
2851400|pass|200812|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812170447
2849440|pass|200812|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.50|i686-linux-64int-ld|5.8.9|linux|2.6.24-16-generic|200812170143
2849289|pass|200812|wolfman.ncsu2000@gmail.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.5.0|200812161813
2849275|pass|200812|wolfman.ncsu2000@gmail.com|Devel-CheckOS|1.50|darwin-2level|5.10.0|darwin|9.5.0|200812161800
2849222|pass|200812|carlos.arenas@gmail.com ("Carlos Arenas")|Devel-CheckOS|1.50|i686-linux-thread-multi|5.10.0|linux|2.6.25-arch|200812161530
2847950|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812170001
2847277|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200812162242
2846671|pass|200812|stro@cpan.org|DBIx-Class-SingletonRows|0.11|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200812162354
2846388|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.18.el5|200812162214
2845909|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.1.18.el5|200812162124
2845878|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812162120
2845633|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.18-92.1.18.el5|200812162048
2845107|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.18-92.1.18.el5|200812161942
2844746|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35117|linux|2.6.22-1-k7|200812161906
2843801|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200812161611
2839840|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200812160329
2836740|pass|200812|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200812151502
2836735|pass|200812|stsc@refcnt.org ("Steven Schubiger")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.18-4-686|200812151501
2835370|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200812150522
2833867|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812142334
2832930|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.18-4-686|200812142100
2832650|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200812141901
2832569|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141947
2831500|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141707
2831034|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141557
2830300|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|darwin-2level|5.10.0|darwin|8.11.1|200812141411
2830281|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|darwin-2level|5.10.0|darwin|8.11.1|200812141408
2830148|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141427
2830115|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 35088|linux|2.6.22-1-k7|200812141423
2829396|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.1-release|200812141130
2829387|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200812141129
2828047|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200812140608
2828016|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200812140459
2824656|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812131413
2823660|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812131155
2823527|fail|200812|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|amd64-freebsd|5.8.9|freebsd|7.0-release|200812131152
2822282|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.17.el5|200812130920
2821425|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.8|darwin|8.11.0|200812130252
2821406|unknown|200812|srezic@cpan.org|Acme-Pony|1.1.2|amd64-freebsd|5.8.8|freebsd|7.0-release|200812130246
2821357|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.6|darwin|8.11.0|200812130239
2821344|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int|5.6.2|darwin|8.11.0|200812130233
2821240|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.10.0|darwin|8.11.0|200812130223
2821229|unknown|200812|srezic@cpan.org|Acme-Pony|1.1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200812130246
2821122|pass|200812|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812130257
2821110|pass|200812|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812130257
2821016|pass|200812|srezic@cpan.org|Tie-Hash-Rank|1.0.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812130246
2821002|pass|200812|srezic@cpan.org|Tie-Hash-Rank|1.0.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812130246
2820804|pass|200812|andy@hexten.net|Devel-CheckOS|1.50|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200812130117
2820611|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.2|200812130038
2819533|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.0-release|200812122018
2818742|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121927
2817255|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121637
2817254|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121634
2817048|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35072|linux|2.6.22-1-k7|200812121621
2817047|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Palm-Treo680MessagesDB|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200812121621
2817042|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.18-4-686|200812121620
2814813|unknown|200812|srezic@cpan.org|Acme-Licence|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812120848
2814386|unknown|200812|srezic@cpan.org|Acme-Licence|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812120840
2814278|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int|5.10.0|linux|2.4.26|200812120728
2813038|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200812120105
2813036|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200812120105
2812566|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200812112335
2812369|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.0-release|200812112308
2811921|pass|200812|srezic@cpan.org|NestedMap|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812112230
2811883|pass|200812|srezic@cpan.org|NestedMap|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812112229
2811405|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.6.2|freebsd|7.0-release|200812112151
2810112|pass|200812|srezic@cpan.org|Tie-Hash-Longest|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812111619
2809905|pass|200812|srezic@cpan.org|Statistics-ChiSquare|0.5|amd64-freebsd|5.8.9|freebsd|7.0-release|200812111619
2809851|pass|200812|srezic@cpan.org|Statistics-ChiSquare|0.5|amd64-freebsd|5.8.8|freebsd|7.0-release|200812111619
2809848|pass|200812|srezic@cpan.org|Tie-Hash-Longest|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812111619
2809562|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd-thread-multi|5.10.0|freebsd|7.0-release|200812111431
2808263|pass|200812|srezic@cpan.org|Statistics-SerialCorrelation|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812111228
2808229|pass|200812|srezic@cpan.org|Statistics-SerialCorrelation|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812111228
2806541|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.10.0|freebsd|7.0-release|200812110753
2805789|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200812110053
2805174|pass|200812|srezic@cpan.org|Palm-SMS|0.02|amd64-freebsd|5.8.8|freebsd|7.0-release|200812102313
2804813|pass|200812|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812102313
2804681|pass|200812|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812102312
2804353|pass|200812|srezic@cpan.org|Palm-SMS|0.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200812102246
2802464|pass|200812|srezic@cpan.org|Class-CanBeA|1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200812101824
2801623|pass|200812|stsc@refcnt.org ("Steven Schubiger")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200812101728
2801494|pass|200812|srezic@cpan.org|Class-CanBeA|1.2|amd64-freebsd|5.8.8|freebsd|7.0-release|200812101628
2801403|pass|200812|fayland@gmail.com ("Fayland Lam")|Data-Compare|1.21|i386-linux|5.8.6|linux|2.6.9-34.elsmp|200812100817
2800876|pass|200812|srezic@cpan.org|File-Overwrite|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812101524
2800852|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200812101452
2800622|pass|200812|srezic@cpan.org|File-Overwrite|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812101524
2798388|pass|200812|srezic@cpan.org|Games-Dice-Advanced|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812101005
2798287|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812101018
2798039|pass|200812|srezic@cpan.org|Games-Dice-Advanced|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812100855
2796907|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812100621
2796290|pass|200812|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.8.8|freebsd|7.0-release|200812100246
2796282|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812100345
2795329|pass|200812|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812100013
2795306|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812100101
2795233|pass|200812|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.8.8|freebsd|7.0-release|200812100006
2794647|pass|200812|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|amd64-freebsd|5.8.9|freebsd|7.0-release|200812092124
2794124|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200812092252
2793718|pass|200812|srezic@cpan.org|Net-Random|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812092201
2791702|pass|200812|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091135
2791699|pass|200812|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091636
2791311|pass|200812|srezic@cpan.org|Data-Transactional|1.02|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091130
2790751|pass|200812|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091637
2789569|pass|200812|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091145
2789378|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091137
2789178|pass|200812|srezic@cpan.org|Number-Phone|1.6|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091232
2788306|pass|200812|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091137
2788027|pass|200812|srezic@cpan.org|Tie-STDOUT|1.03|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091415
2787559|pass|200812|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091415
2787491|pass|200812|srezic@cpan.org|File-Find-Rule-Permissions|1.3|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091414
2786578|pass|200812|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091332
2785705|pass|200812|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091234
2785638|pass|200812|srezic@cpan.org|Sort-MultipleFields|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091256
2784544|pass|200812|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091124
2784434|pass|200812|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091209
2783242|pass|200812|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091127
2783051|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091126
2782996|pass|200812|srezic@cpan.org|CPU-Emulator-Z80|1.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091124
2782974|pass|200812|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091123
2782968|pass|200812|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812091102
2782913|pass|200812|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.8|freebsd|7.0-release|200812091107
2782725|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812091105
2782107|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 35055|linux|2.6.22-1-k7|200812090949
2779448|pass|200812|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.9|freebsd|7.0-release|200812082312
2777481|pass|200812|srezic@cpan.org|Number-Phone|1.6|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081700
2776950|pass|200812|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081622
2776837|pass|200812|srezic@cpan.org|CPU-Emulator-Z80|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081622
2776594|pass|200812|srezic@cpan.org|Data-Transactional|1.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081622
2776152|pass|200812|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081527
2776080|pass|200812|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081449
2776027|pass|200812|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081449
2775728|pass|200812|srezic@cpan.org|Sort-MultipleFields|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812081416
2774364|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.8.8|linux|2.6.18-92.el5|200812080935
2773440|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080645
2773286|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080534
2772792|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080334
2772224|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812080152
2770543|pass|200812|srezic@cpan.org|XML-Tiny|1.11|amd64-freebsd|5.8.9|freebsd|7.0-release|200812072153
2770111|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812072059
2768458|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 35054|linux|2.6.22-1-k7|200812071756
2768263|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200812071746
2767155|pass|200812|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.8.9|freebsd|7.0-release|200812071006
2767043|pass|200812|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200812071009
2766527|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812071127
2765084|pass|200812|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.8.9|freebsd|7.0-release|200812071006
2764240|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812070834
2763250|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812070354
2763041|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812070419
2762696|pass|200812|salvacpan@gmail.com ("Salvador Fandino")|Data-Compare|1.21|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200812070205
2761654|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.9|linux|2.6.24.5-smp|200812062319
2760698|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812062121
2760549|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35041|linux|2.6.22-1-k7|200812062053
2760109|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.9|solaris|2.10|200812061806
2757875|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200812061144
2757570|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200812060938
2757104|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.17.el5|200812060801
2756192|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200812060213
2755272|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200812052216
2754877|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.01|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200812052029
2753936|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812051658
2753040|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812051434
2753021|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812051438
2752288|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200812051129
2751713|pass|200812|salvacpan@gmail.com ("Salvador Fandino")|Devel-CheckOS|1.50|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200812051112
2749841|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200812050514
2749471|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812050437
2749116|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812050401
2748988|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 35013|linux|2.6.22-1-k7|200812050338
2748728|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.4|200812050103
2747547|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200812042206
2747434|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812042145
2746682|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812041910
2746236|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200812041816
2745522|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812041617
2744247|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812041138
2743415|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812040920
2743368|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 35005|linux|2.6.22-1-k7|200812040926
2743095|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200812040643
2743024|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200812040607
2741814|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0|200812032155
2739594|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812031554
2739588|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812031553
2739030|pass|200812|stro@cpan.org|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200812031531
2737577|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812031025
2736968|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812030852
2736963|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812030850
2736945|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34986|linux|2.6.22-1-k7|200812030843
2736235|pass|200812|stsc@refcnt.org ("Steven Schubiger")|XML-Tiny|1.11|i686-linux|5.10.0|linux|2.6.18-4-686|200812030547
2736160|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200812030416
2735721|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.9|openbsd|4.3|200812022258
2734063|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021726
2733597|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021605
2732948|fail|200812|rivy@cpan.org ("Roy Ivy III")|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200812020454
2732946|fail|200812|rivy@cpan.org ("Roy Ivy III")|Devel-CheckLib|0.5|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.00|200812020341
2732934|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021428
2731466|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021113
2731451|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812021111
2731076|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812020307
2730753|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34974|linux|2.6.22-1-k7|200812020858
2730352|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200812020629
2730254|pass|200812|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200812020607
2729332|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200812020300
2726822|pass|200812|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200812011210
2726411|pass|200812|salvacpan@gmail.com ("Salvador Fandino")|XML-DoubleEncodedEntities|1.0|sparc64-netbsd|5.10.0|netbsd|4.0_rc5|200812011403
2724627|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200812010311
2724251|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200812010203
2724097|pass|200812|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200812010137
2723860|pass|200812|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200812010046
2722147|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Class-DBI-ClassGenerator|1.01|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811302019
2720556|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200811301658
2720540|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200811301659
2719984|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34928|linux|2.6.22-1-k7|200811301552
2718220|pass|200811|"Rick Myers <rick@jrmyers.net>"|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24.3|200811300444
2717845|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200811300924
2716120|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811300042
2716111|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811300040
2714520|pass|200811|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291334
2714072|pass|200811|srezic@cpan.org|Palm-SMS|0.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291337
2713044|pass|200811|srezic@cpan.org|File-Overwrite|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291335
2712827|pass|200811|srezic@cpan.org|Class-CanBeA|1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200811291334
2712234|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811291301
2710866|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290937
2710754|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290915
2710189|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290641
2709871|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811290459
2709271|pass|200811|srezic@cpan.org|Sub-WrapPackages|1.2|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290315
2709107|pass|200811|srezic@cpan.org|Games-Dice-Advanced|1.1|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290314
2708838|pass|200811|srezic@cpan.org|Bryar|3.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290207
2708650|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290213
2707901|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34916|linux|2.6.22-1-k7|200811290043
2707819|pass|200811|srezic@cpan.org|CPAN-FindDependencies|2.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290036
2707483|pass|200811|srezic@cpan.org|Devel-CheckLib|0.5|amd64-freebsd|5.8.9|freebsd|7.0-release|200811290019
2706439|pass|200811|srezic@cpan.org|Number-Phone|1.6|amd64-freebsd|5.8.9|freebsd|7.0-release|200811282003
2706119|pass|200811|srezic@cpan.org|Data-Transactional|1.02|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281903
2706021|pass|200811|srezic@cpan.org|CPU-Emulator-Z80|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281903
2706019|pass|200811|srezic@cpan.org|CPU-Emulator-Memory|1.1001|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281903
2705560|pass|200811|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281702
2705307|pass|200811|srezic@cpan.org|Data-Hexdumper|1.4|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281656
2705095|pass|200811|srezic@cpan.org|Sort-MultipleFields|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811281600
2704593|fail|200811|ackack1024@gmail.com|Bryar|3.0|x86_64-linux-gnu|5.10.0|linux|2.6.22-3-vserver-amd64|200811280034
2704505|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281414
2704176|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281304
2703503|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281252
2703484|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811281247
2702042|pass|200811|srezic@cpan.org|Data-Compare|1.21|amd64-freebsd|5.8.9|freebsd|7.0-release|200811280745
2701374|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811280622
2700276|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34948|linux|2.6.22-1-k7|200811280245
2699400|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.9|netbsd|4.0.1|200811272249
2699083|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|amd64-freebsd|5.8.9|freebsd|7.0-release|200811272254
2698909|pass|200811|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|amd64-freebsd|5.8.9|freebsd|7.0-release|200811272231
2697611|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271846
2696650|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271534
2696614|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271526
2696271|pass|200811|rafl@debian.org|Devel-CheckOS|1.50|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200811271420
2695139|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271036
2695035|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271013
2694990|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34943|linux|2.6.22-1-k7|200811271010
2694091|pass|200811|stsc@refcnt.org ("Steven Schubiger")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.18-4-686|200811270500
2692351|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261736
2691588|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811261259
2691558|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811261249
2691526|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261302
2691266|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200811261148
2691249|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200811261146
2691240|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261244
2690890|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811261142
2690322|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811260930
2687773|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811260025
2687313|pass|200811|stsc@refcnt.org ("Steven Schubiger")|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.18-4-686|200811252254
2686586|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811252018
2686529|fail|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200811252007
2685559|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200811250431
2684539|pass|200811|stsc@refcnt.org ("Steven Schubiger")|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.18-4-686|200811250930
2684229|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|6.2-release|200811251034
2683184|pass|200811|stro@cpan.org|Devel-CheckOS|1.50|armv5tejl-linux|5.10.0|linux|2.6.17.14|200811251247
2682752|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200811250844
2682515|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34904|linux|2.6.22-1-k7|200811250743
2682122|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811250602
2680361|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.8.9|solaris|2.11|200811242131
2680013|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811241933
2679219|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811240814
2678887|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.9|freebsd|7.0-release|200811240554
2678038|pass|200811|rafl@debian.org|Class-DBI-ClassGenerator|1.01|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811240747
2676161|pass|200811|rafl@debian.org|Data-Compare|1.21|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811232125
2675217|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811231348
2674615|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811230922
2673626|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811230505
2673381|pass|200811|rafl@debian.org|XML-DoubleEncodedEntities|1.0|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811230651
2672888|pass|200811|rafl@debian.org|Devel-CheckOS|1.50|x86_64-linux-multi|5.11.0|linux|2.6.26-1-xen-amd64|200811230413
2672840|pass|200811|jeff@zeroclue.com|Devel-CheckOS|1.50|i686-linux|5.10.0|linux|2.6.18.8|200811230307
2669975|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811221546
2669945|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811221537
2669079|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811221129
2668317|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220920
2667763|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220614
2667663|pass|200811|stro@cpan.org|Data-Compare|1.21|armv5tejl-linux|5.10.0|linux|2.6.17.14|200811220848
2667220|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220401
2667070|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34899|linux|2.6.22-1-k7|200811220320
2666633|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811212337
2666546|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811212250
2664815|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811211137
2664721|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.8.9|solaris|2.11|200811211129
2664535|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811211039
2664231|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811210935
2663127|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-TreoPhoneCallDB|1.1|i86pc-solaris-64int|5.8.9|solaris|2.11|200811210309
2662766|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.8.9|solaris|2.11|200811210158
2662093|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|i86pc-solaris-64int|5.8.9|solaris|2.11|200811210002
2661386|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811202109
2660920|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811202100
2660902|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811202057
2660519|fail|200811|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811202023
2660119|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811201917
2659938|pass|200811|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811201900
2659844|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811201838
2659533|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34893|linux|2.6.22-1-k7|200811201758
2658954|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811201415
2658696|pass|200811|stro@cpan.org|Data-Compare|1.21|armv5tejl-linux|5.8.8|linux|2.6.17.14|200811201730
2658653|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200811201322
2658270|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.9|dragonfly|2.0.1-release|200811201152
2657474|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200932
2657124|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.8.9|solaris|2.11|200811200832
2656392|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200620
2655915|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811200359
2655089|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200228
2654660|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200044
2654271|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811200003
2654013|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.6|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192332
2654011|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-Tiny|1.11|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811192320
2653778|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34883|linux|2.6.22-1-k7|200811192255
2653319|pass|200811|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192205
2653317|pass|200811|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192205
2653123|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.8.9|solaris|2.11|200811192129
2652993|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200811191942
2652489|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811191830
2652487|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris-64int|5.8.9|solaris|2.11|200811191829
2652026|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811191553
2651864|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811191400
2650706|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.01|i86pc-solaris-64int|5.8.9|solaris|2.11|200811190859
2649448|pass|200811|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris-64int|5.8.9|solaris|2.11|200811190447
2647682|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811182002
2647154|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200811181754
2646760|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.8.9|solaris|2.11|200811181722
2646137|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.8.9|solaris|2.11|200811181505
2645990|pass|200811|JOST@cpan.org ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.8.9|solaris|2.11|200811181441
2645781|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-thread-multi-64int|5.8.8|dragonfly|2.0.1-release|200811181218
2645722|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811181254
2644766|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811180946
2644577|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811180914
2643742|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.1-release|200811180453
2643594|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180512
2643566|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180515
2643414|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180445
2643177|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180351
2642985|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.1-release|200811180237
2642369|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd|5.6.2|freebsd|6.1-release|200811180117
2642350|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200811180144
2642177|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811180126
2642172|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200811180125
2641757|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd|5.10.0|freebsd|6.1-release|200811180029
2641746|pass|200811|srezic@cpan.org|Devel-CheckOS|1.50|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200811180027
2641438|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811172217
2641399|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34869|linux|2.6.22-1-k7|200811172327
2640048|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811171609
2639731|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200811171306
2638283|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811170451
2637720|pass|200811|jeff@zeroclue.com|Devel-CheckOS|1.49_01|darwin-2level|5.10.0|darwin|9.2.0|200811161554
2637310|pass|200811|jeff@zeroclue.com|Devel-CheckOS|1.50|darwin-2level|5.10.0|darwin|9.2.0|200811161412
2637111|pass|200811|jeff@zeroclue.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.2.0|200811161337
2637023|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811162226
2636444|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811162117
2635457|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811161927
2634691|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811161641
2634650|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811161637
2634399|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811161651
2634255|pass|200811|rlane10@wi.rr.com|Devel-CheckOS|1.50|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200811151624
2633552|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34839|linux|2.6.22-1-k7|200811161440
2633214|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200811161305
2633154|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200811161249
2632879|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200811161137
2632873|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200811161135
2629266|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150819
2628823|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150634
2628664|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200811150458
2628657|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200811150456
2628475|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150517
2628382|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811150359
2627681|pass|200811|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811150124
2627138|pass|200811|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811150006
2626763|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200811142331
2626762|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811150024
2626667|pass|200811|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811150007
2626630|pass|200811|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811150007
2626493|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811142318
2626311|pass|200811|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811150006
2625842|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34836|linux|2.6.22-1-k7|200811142302
2624447|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811141816
2624397|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|IP35-irix-64int|5.8.8|irix|6.5|200811141813
2623107|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|i686-linux|5.8.8|linux|2.4.27-3-686|200811141519
2623103|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200811141519
2623102|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200811141519
2623012|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|i686-linux|5.6.2|linux|2.4.27-3-686|200811141453
2622725|pass|200811|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.50|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200811140858
2621900|pass|200811|srezic@cpan.org|Data-Compare|1.21|i686-linux|5.8.9|linux|2.6.18-4-486|200811141203
2621832|pass|200811|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141209
2621677|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.50|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200811140501
2621490|pass|200811|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141123
2621323|pass|200811|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141123
2620625|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811141025
2620572|pass|200811|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811140748
2620460|pass|200811|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811141004
2619006|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140834
2618695|pass|200811|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811140750
2618435|pass|200811|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811140748
2618156|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811140733
2617636|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140518
2617329|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140403
2617296|pass|200811|mthurn@cpan.org|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200811140257
2616757|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811140150
2616015|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811140029
2615288|pass|200811|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132348
2615275|pass|200811|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132348
2614137|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34793|linux|2.6.22-1-k7|200811132239
2613115|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.16.60-0.31-default|200811132141
2612701|pass|200811|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132107
2612452|pass|200811|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811132052
2611689|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811131825
2611588|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811131811
2611359|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200811131852
2611260|pass|200811|vpit@cpan.org|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811131847
2611233|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200811131921
2611223|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200811131840
2611193|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200811131835
2610645|pass|200811|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131727
2610452|pass|200811|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131727
2610415|pass|200811|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.50|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200811140034
2610256|pass|200811|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131615
2610198|pass|200811|robbiebow@gmail.com|CPAN-FindDependencies|2.0|i486-linux-gnu|5.10.0|linux|2.6.26.1|200811122351
2610048|pass|200811|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811131522
2609908|pass|200811|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811131500
2609715|pass|200811|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131532
2609496|pass|200811|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131443
2609436|pass|200811|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131443
2608821|pass|200811|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131306
2607802|pass|200811|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811131049
2606926|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811130910
2606908|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811130906
2606456|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811130758
2606331|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811130802
2606310|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811130757
2604699|pass|200811|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130026
2604575|pass|200811|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130001
2604455|pass|200811|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130001
2604312|pass|200811|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811130001
2604162|pass|200811|g.grigelionis@computer.org|Devel-CheckOS|1.50|i586-linux-thread-multi|5.8.8|linux|2.6.16|200811122335
2603695|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200811121923
2603694|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200811121922
2603691|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200811121920
2603493|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200811121914
2603489|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811121720
2603481|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811121719
2603473|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200811121908
2602468|pass|200811|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121457
2602130|pass|200811|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121457
2601943|pass|200811|hohokus@gmail.com|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200811120834
2601746|pass|200811|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121403
2601594|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121411
2601466|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.50|i86pc-solaris-64int|5.10.0|solaris|2.11|200811121400
2600709|pass|200811|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811121131
2600480|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-Scalar-Decay|1.1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121135
2600283|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Rank|1.0.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121101
2600248|unknown|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Pony|1.1.2|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811121057
2599568|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200811120954
2599401|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811121000
2598573|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811120825
2598531|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811120822
2598466|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811120819
2598372|pass|200811|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811120809
2598113|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux-thread-multi|5.6.2|linux|2.6.18-92.1.17.el5|200811120803
2598086|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.50|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200811120801
2598013|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811120757
2597996|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811120755
2597982|pass|200811|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.9|freebsd|6.1-release-p23|200811120755
2597818|pass|200811|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.50|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200811121544
2597814|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200811120644
2597801|pass|200811|bingos@cpan.org|Devel-CheckOS|1.50|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200811120642
2597637|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.10.0|linux|2.6.16.60-0.31-default|200811120708
2597576|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811120650
2597159|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.16.60-0.31-default|200811120500
2596797|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200811120242
2596294|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200811120059
2595952|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200811120057
2595950|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200811120056
2595949|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200811120056
2595947|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200811120056
2595946|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200811120056
2595945|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.50|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200811120055
2595943|cpan|200811|DCANTRELL|Devel-CheckOS|1.50||0|||200811120055
2595600|pass|200811|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.46|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200811111753
2594983|pass|200811|gabor@pti.co.il ("Gabor Szabo")|Devel-CheckOS|1.46|i486-linux-gnu|5.10.0|linux|2.6.24-15-server|200811112313
2593732|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|alpha-linux-thread-multi|5.10.0|linux|2.6.18-5-alpha-generic|200811111649
2593627|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.4|200811111630
2593601|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|alpha-linux-thread-multi|5.8.8|linux|2.6.18-5-alpha-generic|200811111628
2593570|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.4|200811111625
2593304|unknown|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Licence|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811111642
2593091|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811111247
2593003|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200811110859
2591883|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811111127
2591868|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811111124
2591312|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811111117
2590531|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200811110903
2590240|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110919
2589391|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.6.2|linux|2.6.18-92.1.17.el5|200811110737
2589185|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110713
2589161|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110650
2587984|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.9 patch 34813|linux|2.6.22-1-k7|200811110308
2587697|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|i686-linux|5.6.2|linux|2.4.27-3-686|200811110046
2587691|pass|200811|JOST@cpan.org ("Josts Smokehouse")|NestedMap|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811110154
2587663|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200811110042
2587641|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|IP35-irix-64int|5.8.8|irix|6.5|200811110131
2587640|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200811110031
2587638|pass|200811|david@cantrell.org.uk|Devel-CheckOS|1.49_01|i686-linux|5.8.8|linux|2.4.27-3-686|200811110029
2587527|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811110103
2587440|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811110042
2587312|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811110028
2586407|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811102044
2585863|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Statistics-ChiSquare|0.5|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811101759
2585862|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-Hash-Longest|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811101759
2585532|pass|200811|mw487.yahoo.com@gmail.com|Data-Compare|1.21|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200811101037
2585044|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|XML-DoubleEncodedEntities|1.0|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811101523
2584335|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200811101218
2583840|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811101204
2583698|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811101153
2583005|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811100952
2582385|pass|200811|gabor@pti.co.il ("Gabor Szabo")|Data-Compare|1.21|i486-linux-gnu|5.10.0|linux|2.6.24-15-server|200811100919
2582326|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811100809
2582078|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811100726
2582050|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34785|linux|2.6.22-1-k7|200811100723
2580967|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200811092243
2580961|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200811092241
2580238|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811092001
2579751|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811091712
2579710|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200811091605
2579703|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200811091604
2579695|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200811091603
2579270|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811090245
2578531|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811090232
2578208|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811090209
2578105|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811091341
2577802|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.49_01|s390x-linux|5.10.0|linux|2.6.18-92.el5|200811091331
2577669|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811090432
2577534|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.10.0|linux|2.6.18-92.el5|200811090310
2577302|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811082145
2576423|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811090541
2576354|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.18-92.el5|200811090437
2576216|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux|5.8.8|linux|2.6.18-92.el5|200811090327
2576097|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811090239
2575964|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.8.8|linux|2.6.16.60-0.31-default|200811090137
2575535|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Data-Compare|1.21|s390x-linux-thread-multi|5.10.0|linux|2.6.16.60-0.31-default|200811082110
2575313|pass|200811|cpan@sourcentral.org ("Oliver Paukstadt")|Devel-CheckOS|1.46|s390x-linux|5.6.2|linux|2.6.16.60-0.31-default|200811081751
2575228|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811091048
2575216|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811091043
2574416|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090653
2574056|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090553
2573577|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090313
2573394|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811090239
2572453|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811082341
2571947|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34778|linux|2.6.22-1-k7|200811082138
2571920|pass|200811|alexchorny@gmail.com|Sub-WrapPackages|1.2|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200811081546
2570723|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200811081452
2570711|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200811081443
2570655|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Statistics-SerialCorrelation|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811081538
2569847|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200811081210
2567904|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811080031
2567855|fail|200811|JOST@cpan.org ("Josts Smokehouse")|Palm-SMS|0.02|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811080017
2567524|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811072307
2565716|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071809
2565220|pass|200811|vpit@cpan.org|Devel-CheckOS|1.49_01|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811071653
2564732|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071535
2564363|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200811071500
2564255|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071441
2563595|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200811071225
2563590|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200811071224
2563414|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200811071300
2563389|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071301
2563372|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34757|linux|2.6.22-1-k7|200811071258
2562584|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Class-CanBeA|1.2|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811070944
2562184|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Overwrite|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811070616
2562127|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200811070401
2562116|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200811070355
2561849|pass|200811|bingos@cpan.org|Devel-CheckOS|1.49_01|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200811070206
2561391|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200811062353
2561375|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200811062340
2561302|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200811062317
2560842|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811062111
2560840|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811062110
2560242|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Sub-WrapPackages|1.2|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811061914
2558802|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.49_01|i86pc-solaris-64int|5.10.0|solaris|2.11|200811061140
2558755|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.49_01|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811061133
2558649|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811061111
2557975|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200811060755
2557935|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200811060740
2557834|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811060723
2557660|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811060652
2556889|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811060446
2556513|pass|200811|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.49_01|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200811061104
2556397|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Games-Dice-Advanced|1.1|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811060121
2556243|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200811052354
2556227|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200811052357
2556218|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200811052355
2556214|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200811052355
2556200|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.49_01|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200811052353
2556191|cpan|200811|DCANTRELL|Devel-CheckOS|1.49_01||0|||200811052352
2555720|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811052249
2555567|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811052217
2555227|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811052117
2555055|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Bryar|3.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811052048
2555052|pass|200811|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200811051938
2555016|pass|200811|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200811051933
2554662|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34736|linux|2.6.22-1-k7|200811051937
2553764|pass|200811|rlane10@wi.rr.com|Devel-CheckOS|1.46|darwin-2level-ld|5.10.0|darwin|8.11.0|200811041145
2553083|fail|200811|cbrown@mail ((root))|Devel-CheckOS|1.46|i386-linux|5.8.8|linux|2.6.18-53.1.14.el5xen|200811041414
2552776|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051334
2552720|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051322
2552717|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051322
2552614|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811051252
2551320|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811050829
2550539|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34732|linux|2.6.22-1-k7|200811050439
2550270|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200811050048
2548813|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Number-Phone-UK-DetailedLocations|1.3|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811042025
2548496|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811041751
2547087|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811041339
2547071|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811041235
2546981|pass|200811|altblue@gmail.com ("Marius Feraru")|File-Find-Rule-Permissions|1.3|x86_64-linux|5.11.0 patch 34595|linux|2.6.18-8.1.14.el5|200811040044
2546913|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811041319
2546290|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|sun4-solaris-64int|5.8.4|solaris|2.10|200811041057
2545569|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040753
2545541|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040747
2545268|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040626
2544952|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200811040434
2544949|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200811040433
2544934|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.10.0 patch 34707|linux|2.6.22-1-k7|200811040531
2544575|pass|200811|JOST@cpan.org ("Josts Smokehouse")|Tie-STDOUT|1.03|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811040415
2544179|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200811040109
2544168|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200811040106
2544048|pass|200811|JOST@cpan.org ("Josts Smokehouse")|File-Find-Rule-Permissions|1.3|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200811040119
2543970|pass|200811|vpit@cpan.org|XML-DoubleEncodedEntities|1.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811040106
2543298|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i486-linux-gnu|5.8.8|linux|2.6.22.10|200811032250
2543230|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811032210
2542832|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811032156
2542819|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811032155
2542678|pass|200811|rlane10@wi.rr.com|Devel-CheckOS|1.46|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200811031045
2542522|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811032027
2542036|pass|200811|vpit@cpan.org|Devel-CheckOS|1.46|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200811031927
2541577|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811031621
2541079|pass|200811|schwern@pobox.com|Devel-CheckOS|1.46|darwin-thread-multi-2level|5.10.0|darwin|8.11.1|200811030129
2539727|pass|200811|srezic@cpan.org|DBIx-Class-SingletonRows|0.11|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200811030815
2539183|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811030150
2539180|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200811030147
2538385|pass|200811|srezic@cpan.org|Class-DBI-ClassGenerator|1.01|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200811021942
2538157|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811021941
2537248|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811021456
2537140|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811021430
2536237|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200811021009
2536236|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200811021008
2535874|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811020919
2535111|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.6.2|freebsd|6.1-release|200811012242
2534583|pass|200811|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34693|linux|2.6.22-1-k7|200811012020
2534123|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200811011854
2534107|pass|200811|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200811011558
2534035|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200811011812
2532885|pass|200811|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200811011018
2532782|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811010932
2532502|pass|200811|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811010844
2532466|pass|200811|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200811010839
2532160|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200811010744
2532014|pass|200811|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200811010701
2531165|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810312125
2531155|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810312117
2530598|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200810311724
2530558|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200810311714
2530332|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810311512
2530145|pass|200810|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.45|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200810311042
2530130|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200810311510
2529997|pass|200810|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.46|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.10.el5|200810311256
2529946|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810311309
2529744|pass|200810|frank.wiegand@gmail.com ("Frank Wiegand")|XML-Tiny|1.11|i686-linux|5.11.0 patch 34669|linux|2.6.24-21-generic|200810311244
2529657|pass|200810|frank.wiegand@gmail.com ("Frank Wiegand")|Devel-CheckOS|1.45|i686-linux|5.11.0 patch 34669|linux|2.6.24-21-generic|200810311221
2529589|pass|200810|frank.wiegand@gmail.com ("Frank Wiegand")|Number-Phone|1.6|i686-linux|5.11.0 patch 34669|linux|2.6.24-21-generic|200810311207
2529526|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200810311155
2529508|pass|200810|JOST@cpan.org ("Josts Smokehouse")|CPAN-FindDependencies|2.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810311157
2529333|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810310847
2529118|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810310945
2529027|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810311016
2529022|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810310743
2528663|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810310637
2528355|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810310720
2528306|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810310543
2528209|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310756
2528098|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310740
2526519|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310305
2525689|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810310106
2525121|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckLib|0.5|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810302318
2524941|pass|200810|alexchorny@gmail.com|Devel-CheckOS|1.46|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200810301734
2524729|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810302110
2524431|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810302038
2524206|pass|200810|alexchorny@gmail.com|Devel-CheckOS|1.46|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.1|200810301605
2524061|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34662|linux|2.6.22-1-k7|200810302041
2522998|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810301500
2522809|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810301439
2521791|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810301311
2521755|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810301308
2521744|pass|200810|rafl@debian.org|Devel-CheckOS|1.46|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200810301406
2521584|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810301248
2521267|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810301300
2521123|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200810301043
2520628|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200810301125
2520307|pass|200810|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200810301052
2519534|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200810300822
2519407|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810300804
2519379|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810300700
2519368|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810300657
2519361|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810300654
2518625|pass|200810|srezic@cpan.org|Devel-CheckOS|1.46|i386-freebsd|5.10.0|freebsd|6.1-release|200810300137
2518304|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810292335
2518295|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-DoubleEncodedEntities|1.0|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200810292333
2518115|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200810292218
2518110|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24.5-smp|200810292217
2518078|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|i686-linux|5.6.2|linux|2.4.27-3-686|200810292107
2518031|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200810292138
2518027|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24.5-smp|200810292138
2517906|pass|200810|bingos@cpan.org|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200810292043
2517761|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200810291837
2517712|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200810291315
2517664|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|IP35-irix-64int|5.8.8|irix|6.5|200810291900
2517662|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200810291800
2517605|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|i686-linux|5.8.8|linux|2.4.27-3-686|200810291747
2517459|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.46|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200810291723
2517027|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291722
2516935|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200810291700
2516933|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.24.5-smp|200810291659
2516753|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291637
2516478|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Number-Phone|1.6|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810291616
2516446|pass|200810|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810291504
2516371|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291556
2516057|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34633|linux|2.6.22-1-k7|200810291518
2515288|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810291044
2514837|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290901
2514753|pass|200810|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Z80|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290912
2514752|pass|200810|JOST@cpan.org ("Josts Smokehouse")|CPU-Emulator-Memory|1.1001|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290912
2514504|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Data-Transactional|1.02|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290834
2514498|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200810290725
2513793|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200810290519
2513133|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290409
2513016|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Palm-Treo680MessagesDB|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290401
2513012|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Data-Hexdumper|1.4|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810290401
2512798|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810290216
2512788|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810290214
2512344|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290145
2512110|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810290105
2511798|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34613|linux|2.6.22-1-k7|200810282356
2511490|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Sort-MultipleFields|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810282313
2510931|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200810282020
2510919|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200810282017
2510832|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.46|i586-linux-thread-multi|5.8.8|linux|2.6.16|200810282005
2510628|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200810281827
2510625|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.3|200810281825
2510615|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200810281822
2510613|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.3|200810281821
2509990|pass|200810|JOST@cpan.org ("Josts Smokehouse")|XML-DoubleEncodedEntities|1.0|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810281554
2509878|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Class-DBI-ClassGenerator|1.01|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810281524
2509570|pass|200810|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810281247
2509343|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.46|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810281250
2509286|pass|200810|bingos@cpan.org|Data-Compare|1.21|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200810281129
2509020|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.46|i86pc-solaris-64int|5.10.0|solaris|2.11|200810281028
2508952|pass|200810|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.46|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200810281653
2508800|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810280819
2508796|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810280818
2508788|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810280809
2508475|pass|200810|srezic@cpan.org|Tie-Scalar-Decay|1.1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810280801
2508250|pass|200810|srezic@cpan.org|Tie-Hash-Rank|1.0.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810280801
2507698|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810280609
2507696|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810280608
2507692|pass|200810|bingos@cpan.org|Devel-CheckOS|1.46|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810280606
2507534|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810280300
2507269|pass|200810|gabor@pti.co.il ("Gabor Szabo")|Devel-CheckOS|1.45|i486-linux-gnu|5.8.8|linux|2.6.15.7|200810280542
2507019|pass|200810|JOST@cpan.org ("Josts Smokehouse")|XML-Tiny|1.11|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810280325
2506737|pass|200810|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.46|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200810280933
2506068|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200810272158
2506067|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200810272158
2506064|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810272157
2506063|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200810272156
2506062|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200810272155
2505988|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.46|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200810272139
2505985|cpan|200810|DCANTRELL|Devel-CheckOS|1.46||0|||200810272138
2505629|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200810271709
2505308|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200810271749
2505240|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810271740
2505236|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810271551
2504937|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.45|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810271734
2504471|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Data-Compare|1.21|i86pc-solaris-64int|5.8.8 patch 34559|solaris|2.11|200810271551
2504185|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.6.2|freebsd|7.0-release|200810271210
2504009|pass|200810|stro@cpan.org|Devel-CheckOS|1.45|i686-linux|5.10.0|linux|2.4.27-2-386|200810270430
2503585|pass|200810|autarch@urth.org|Devel-CheckOS|1.45|x86_64-linux-gnu|5.10.0|linux|2.6.24-15-server|200810270459
2502865|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810270725
2502805|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810270652
2502687|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810270422
2501723|pass|200810|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262157
2501608|pass|200810|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262151
2501355|pass|200810|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262157
2501284|pass|200810|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262302
2500744|pass|200810|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262157
2500305|pass|200810|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810262156
2499468|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810261939
2499318|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200810261821
2499297|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810261714
2499275|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810261639
2499251|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810261651
2498986|pass|200810|rafl@debian.org|Devel-CheckOS|1.45|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200810261508
2498600|pass|200810|mw487.yahoo.com@gmail.com|Devel-CheckOS|1.44|cygwin-thread-multi-64int|5.8.8|cygwin|1.5.24(0.15642)|200810260600
2497640|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200810260550
2497613|pass|200810|bingos@cpan.org|Data-Compare|1.21|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200810260509
2496914|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810260107
2496562|pass|200810|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810260009
2496323|pass|200810|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810260009
2496163|pass|200810|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252229
2496066|pass|200810|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252228
2495552|pass|200810|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252226
2495509|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|MirBSD.i386-mirbsd-thread-multi-64int|5.10.0|mirbsd|10|200810252129
2495081|pass|200810|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252225
2494749|pass|200810|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252223
2494614|pass|200810|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810252226
2492857|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200810251722
2492844|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810251819
2492486|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200810251609
2492206|pass|200810|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251540
2492134|pass|200810|stro@cpan.org|Devel-CheckOS|1.45|i686-linux|5.8.8|linux|2.4.27-2-386|200810251741
2492058|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810251638
2491173|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810251546
2490625|pass|200810|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251454
2490399|pass|200810|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251455
2490325|pass|200810|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810251455
2490042|pass|200810|bingos@cpan.org|Data-Compare|1.21|MirBSD.i386-mirbsd-thread-multi-64int|5.8.8|mirbsd|10|200810251236
2488744|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810250758
2488319|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.10.0 patch 34563|linux|2.6.22-1-k7|200810250250
2487352|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200810241400
2486113|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810241439
2486095|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810241442
2486019|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|i586-linux-thread-multi|5.8.8|linux|2.6.16|200810241428
2485503|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|IP35-irix-64int|5.8.8|irix|6.5|200810241215
2485089|pass|200810|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240958
2484591|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810240947
2484461|pass|200810|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484233|pass|200810|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484151|pass|200810|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484114|pass|200810|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240822
2484095|pass|200810|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240824
2483233|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810240655
2483115|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8|freebsd|6.2-release|200810240215
2482565|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200810240001
2482494|pass|200810|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240123
2481785|pass|200810|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810240003
2481577|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200810232239
2481358|pass|200810|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232100
2481082|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.6.2|freebsd|6.1-release|200810232229
2481074|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|i686-linux|5.8.8|linux|2.4.27-3-686|200810232130
2480989|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232057
2480933|pass|200810|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232151
2480107|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.8.8 patch 34559|freebsd|6.1-release-p23|200810232057
2479548|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200810231842
2479494|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200810231726
2478847|fail|200810|hohokus@gmail.com|Devel-CheckOS|1.45|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200810230941
2478685|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200810231416
2478663|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.10.0|freebsd|6.1-release|200810231414
2478516|pass|200810|srezic@cpan.org|Devel-CheckOS|1.45|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200810231244
2478478|pass|200810|david@cantrell.org.uk|Devel-CheckOS|1.45|i686-linux|5.6.2|linux|2.4.27-3-686|200810231222
2478472|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200810231132
2478379|pass|200810|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.45|i86pc-solaris-64int|5.10.0|solaris|2.11|200810231248
2478227|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810231107
2477945|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.8.8 patch 34560|linux|2.6.22-1-k7|200810231041
2477713|pass|200810|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.45|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200810231631
2477363|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810230636
2477361|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810230635
2477330|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810230626
2477139|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810230451
2477135|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810230450
2477132|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810230449
2476895|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810230222
2476865|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810230218
2476833|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810230207
2476744|pass|200810|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810230030
2476647|pass|200810|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230042
2476637|pass|200810|srezic@cpan.org|NestedMap|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230040
2476585|pass|200810|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230042
2476412|pass|200810|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810230109
2476362|pass|200810|srezic@cpan.org|Tie-Hash-Longest|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810230007
2476350|pass|200810|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810230106
2476257|pass|200810|bingos@cpan.org|Data-Compare|1.21|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810230102
2475879|pass|200810|bingos@cpan.org|Devel-CheckOS|1.45|i386-dragonfly-thread-multi-64int|5.10.0|dragonfly|2.0.0-release|200810230118
2475451|pass|200810|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810230042
2475248|fail|200810|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.45|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200810230753
2474805|pass|200810|srezic@cpan.org|Statistics-ChiSquare|0.5|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810222317
2474135|pass|200810|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810222317
2473532|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200810222320
2473339|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200810222312
2473302|pass|200810|g.grigelionis@computer.org|Devel-CheckOS|1.45|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200810222304
2473143|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810222219
2473065|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810222204
2473064|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200810222203
2473060|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|x86_64-linux|5.10.0|linux|2.6.24-1-amd64|200810222159
2473033|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200810222205
2473031|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200810222204
2473021|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.45|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200810222203
2473005|cpan|200810|DCANTRELL|Devel-CheckOS|1.45||0|||200810222158
2470971|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810221344
2470612|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810221236
2470404|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810221158
2469824|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810220953
2469182|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.11.0 patch 34555|linux|2.6.22-1-k7|200810220802
2468226|pass|200810|davidp@preshweb.co.uk ("David Precious")|Acme-Scurvy-Whoreson-BilgeRat|1.1|i486-linux-gnu|5.10.0|linux|2.6.26.1|200810211951
2464860|fail|200810|srezic@cpan.org|Palm-SMS|0.02|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810210001
2464503|pass|200810|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810210004
2463982|pass|200810|srezic@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810210000
2459015|pass|200810|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810191305
2458962|pass|200810|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810191305
2457553|pass|200810|srezic@cpan.org|File-Overwrite|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810191252
2457513|pass|200810|srezic@cpan.org|Class-CanBeA|1.2|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810191252
2457101|pass|200810|stro@cpan.org|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.4.27-2-386|200810190906
2454809|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810181733
2454735|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.8.8|solaris|2.10|200810181713
2453261|pass|200810|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810180826
2453135|pass|200810|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810180826
2451763|pass|200810|srezic@cpan.org|Sub-WrapPackages|1.2|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810172340
2451647|pass|200810|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810172339
2449165|pass|200810|andy@hexten.net|Data-Hexdumper|1.4|darwin-thread-multi-2level|5.10.0|darwin|9.2.2|200810170121
2448136|pass|200810|srezic@cpan.org|Number-Phone-UK-DetailedLocations|1.3|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810162113
2446197|fail|200810|alexchorny@gmail.com|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200810160657
2445048|fail|200810|mthurn@cpan.org|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200810160055
2443632|pass|200810|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810152300
2443542|pass|200810|srezic@cpan.org|Net-Random|2.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810152301
2442929|pass|200810|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810152045
2442624|pass|200810|srezic@cpan.org|Palm-TreoPhoneCallDB|1.1|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810152056
2442170|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810151943
2442108|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810151925
2441749|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151724
2441374|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151755
2441288|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151737
2441253|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151730
2441130|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34482|linux|2.6.22-1-k7|200810151701
2440340|pass|200810|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810151129
2439931|pass|200810|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810151111
2439551|pass|200810|srezic@cpan.org|Bryar|3.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810151054
2439526|pass|200810|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810150851
2439360|pass|200810|srezic@cpan.org|CPAN-FindDependencies|2.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810151026
2438808|pass|200810|srezic@cpan.org|File-Find-Rule-Permissions|1.3|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810150854
2438472|pass|200810|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810150851
2438337|pass|200810|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810150855
2437120|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0.1|200810141738
2437103|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0.1|200810141734
2435795|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810141034
2435778|pass|200810|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810141032
2435512|pass|200810|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140948
2435324|pass|200810|srezic@cpan.org|Devel-CheckLib|0.5|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140906
2435083|pass|200810|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140831
2434782|pass|200810|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140000
2434685|pass|200810|srezic@cpan.org|CPU-Emulator-Z80|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140745
2434099|pass|200810|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140626
2434066|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140623
2433845|pass|200810|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140601
2433775|pass|200810|srezic@cpan.org|Data-Hexdumper|1.4|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140552
2433436|pass|200810|srezic@cpan.org|XML-Tiny|1.11|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140518
2433208|pass|200810|srezic@cpan.org|Sort-MultipleFields|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140451
2433004|pass|200810|srezic@cpan.org|Data-Transactional|1.02|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140422
2432263|pass|200810|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140315
2431812|pass|200810|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140236
2430749|pass|200810|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140113
2430375|pass|200810|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140054
2430283|pass|200810|srezic@cpan.org|XML-DoubleEncodedEntities|1.0|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140049
2430274|pass|200810|srezic@cpan.org|Data-Compare|1.21|i386-freebsd|5.8.8|freebsd|6.1-release-p23|200810140048
2430265|pass|200810|srezic@cpan.org|CPU-Emulator-Memory|1.1001|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140047
2429590|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0|freebsd|6.2-release|200810140051
2428930|pass|200810|srezic@cpan.org|Number-Phone|1.6|i386-freebsd|5.8.8 patch 34467|freebsd|6.1-release-p23|200810140103
2426648|pass|200810|andy@hexten.net|Devel-CheckOS|1.44|darwin-thread-multi-2level|5.10.0|darwin|9.1.0|200810131955
2423624|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.16.38|200810121825
2423591|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.16.38|200810121816
2422180|pass|200810|bobtfish@bobtfish.net|XML-Tiny|1.11|darwin-thread-multi-2level|5.8.6|darwin|8.0|200810112343
2421100|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.8.4|solaris|2.10|200810111405
2419263|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.26.3-29.fc9.i686|200810110123
2418784|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.26.3-29.fc9.i686|200810102304
2418723|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i86pc-solaris-thread-multi-64int|5.10.0|solaris|2.10|200810102248
2416834|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd-64int|5.8.8|freebsd|6.1-release|200810101125
2415612|pass|200810|rafl@debian.org|Devel-CheckOS|1.44|i686-linux-multi|5.10.0|linux|2.6.24-rc5|200810100449
2415526|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.8.8|freebsd|7.0-release|200810100234
2412109|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810091157
2411847|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.6|linux|2.6.26.6|200810091249
2411451|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.7|linux|2.6.26.6|200810091200
2411206|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.6.26.6|200810091126
2410831|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.10.0|linux|2.6.26.6|200810091027
2408741|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810090314
2408656|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810090900
2407740|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.6|linux|2.6.26.5|200810081940
2407241|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081808
2407181|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.6|linux|2.6.26.5|200810081801
2406981|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810082229
2406680|pass|200810|bingos@cpan.org|XML-DoubleEncodedEntities|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810082151
2406514|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.6|linux|2.6.26.5|200810081629
2406205|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.6|linux|2.6.26.5|200810081530
2405826|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081450
2405496|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.6|linux|2.6.26.5|200810081413
2405457|pass|200810|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810081909
2405326|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081953
2405256|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.6|linux|2.6.26.5|200810081345
2405186|pass|200810|bingos@cpan.org|Data-Hexdumper|1.4|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810081836
2404994|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081912
2404816|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081854
2404578|pass|200810|bingos@cpan.org|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810081730
2404428|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081213
2404417|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.6|linux|2.6.26.5|200810081212
2404347|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.6|linux|2.6.26.5|200810081205
2404177|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081739
2404034|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.6|linux|2.6.26.5|200810081132
2404020|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.6|linux|2.6.26.5|200810081130
2403986|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.6|linux|2.6.26.5|200810081127
2403309|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0 patch 34465|linux|2.6.22-1-k7|200810081519
2401687|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.6|linux|2.6.26.5|200810080346
2401425|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080304
2401256|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.6|linux|2.6.26.5|200810080232
2401241|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080229
2400962|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080145
2400758|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080117
2400672|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.6|linux|2.6.26.5|200810080103
2400381|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080025
2400375|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080024
2400344|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080020
2400205|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080357
2400201|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080355
2400188|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080354
2399570|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080323
2399550|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.6|linux|2.6.26.5|200810080217
2399545|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.6|linux|2.6.26.5|200810080216
2399488|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.6|linux|2.6.26.5|200810080204
2399487|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.6|linux|2.6.26.5|200810080203
2399311|pass|200810|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080240
2399310|pass|200810|bingos@cpan.org|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080240
2399215|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080123
2399200|pass|200810|bingos@cpan.org|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080220
2399199|pass|200810|bingos@cpan.org|Acme-Pony|1.1.2|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080220
2398627|pass|200810|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080112
2398624|pass|200810|bingos@cpan.org|Acme-Licence|1.0|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810080112
2398537|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810080002
2398034|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-19-generic|200810080002
2397861|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071841
2397824|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.7|linux|2.6.26.5|200810071837
2397382|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071728
2397374|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071726
2397323|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071721
2397253|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071712
2397252|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810071712
2397251|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.7|linux|2.6.26.5|200810071712
2396275|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.7|linux|2.6.26.5|200810071516
2395781|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.7|linux|2.6.26.5|200810071408
2395613|pass|200810|bingos@cpan.org|XML-Tiny|1.11|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.24-19-generic|200810071843
2395438|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071326
2395145|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.7|linux|2.6.26.5|200810071250
2395006|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.7|linux|2.6.26.5|200810071238
2394616|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071151
2394604|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.7|linux|2.6.26.5|200810071150
2394559|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.7|linux|2.6.26.5|200810071146
2394350|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.7|linux|2.6.26.5|200810071118
2394327|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.7|linux|2.6.26.5|200810071115
2394310|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.7|linux|2.6.26.5|200810071113
2393415|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.7|linux|2.6.26.5|200810070845
2393059|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.7|linux|2.6.26.5|200810070757
2392700|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.7|linux|2.6.26.5|200810070706
2392485|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.7|linux|2.6.26.5|200810070643
2392480|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.7|linux|2.6.26.5|200810070643
2392462|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.7|linux|2.6.26.5|200810070641
2392460|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.7|linux|2.6.26.5|200810070640
2392379|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810071232
2390588|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810070830
2389583|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810070515
2389245|pass|200810|imacat@mail.imacat.idv.tw|Devel-CheckOS|1.44|x86_64-linux-thread-multi-ld|5.10.0|linux|2.6.22.10|200810071033
2388111|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810070108
2387762|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810062317
2387692|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8 patch 34468|linux|2.6.22-1-k7|200810062332
2386863|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|4.0|200810062112
2386783|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|4.0|200810062050
2384752|pass|200810|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810061018
2382914|pass|200810|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.8.6|linux|2.6.26.5|200810060430
2381463|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.11.0 patch 34456|linux|2.6.26.5|200810041930
2380687|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.6|linux|2.6.26.5|200810041718
2379530|pass|200810|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.6|linux|2.6.26.5|200810041330
2378123|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.6|linux|2.6.26.5|200810040907
2374876|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|OpenBSD.i386-openbsd-thread-multi-64int|5.11.0 patch 34456|openbsd|4.2|200810040217
2374313|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.7|linux|2.6.26.5|200810040228
2373780|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.8.8 patch 34450|freebsd|7.0-release|200810040031
2373731|pass|200810|bingos@cpan.org|Data-Compare|1.21|i386-freebsd-thread-multi-64int|5.10.0 patch 34445|freebsd|7.0-release|200810040024
2373652|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.7|linux|2.6.26.5|200810040007
2373001|pass|200810|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.7|linux|2.6.26.5|200810031758
2372448|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.7|linux|2.6.26.5|200810031624
2366669|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200810030525
2366665|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200810030523
2366661|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200810030522
2366361|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|OpenBSD.i386-openbsd-thread-multi-64int|5.8.8|openbsd|4.2|200810030315
2366104|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|OpenBSD.i386-openbsd-thread-multi-64int|5.10.0|openbsd|4.2|200810030138
2365777|pass|200810|rlane10@wi.rr.com|Devel-CheckOS|1.44|darwin-2level-ld|5.10.0|darwin|8.11.0|200810021301
2364436|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810022223
2363838|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.8.8 patch 34450|freebsd|7.0-release|200810021830
2363552|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0 patch 34445|freebsd|7.0-release|200810021809
2363266|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810022037
2361236|pass|200810|rlane10@wi.rr.com|Devel-CheckOS|1.44|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5xen|200810020617
2359337|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810021013
2359161|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.6.2|linux|2.6.21.5-smp|200810020905
2358794|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810020757
2358649|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810020842
2358373|pass|200810|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34450|linux|2.6.22-1-k7|200810020824
2357391|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8|netbsd|3.1|200810020510
2357386|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.7|netbsd|3.1|200810020510
2357380|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.6|netbsd|3.1|200810020509
2357371|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.5|netbsd|3.1|200810020508
2357367|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.4|netbsd|3.1|200810020507
2357363|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.3|netbsd|3.1|200810020507
2357359|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.2|netbsd|3.1|200810020506
2357354|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.1|netbsd|3.1|200810020505
2357351|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0|netbsd|3.1|200810020504
2355309|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.11.0 patch 34451|freebsd|7.0-release|200810012041
2355042|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.21.5-smp|200810012104
2355010|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i686-linux-thread-multi-64int-ld|5.8.8|linux|2.6.21.5-smp|200810012054
2354265|pass|200810|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.10.0|freebsd|7.0-release|200810011736
2352829|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010943
2352827|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010943
2352706|pass|200810|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi|5.8.8|freebsd|6.1-release|200810011515
2352544|pass|200810|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010829
2352512|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010820
2352502|pass|200810|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010817
2352501|pass|200810|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010816
2352415|pass|200810|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010758
2352286|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010718
2352285|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010718
2352283|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010718
2352196|fail|200810|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010701
2352098|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010644
2352092|pass|200810|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010644
2352031|pass|200810|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010625
2351970|pass|200810|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010615
2351912|pass|200810|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010603
2351900|pass|200810|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010600
2351853|pass|200810|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010553
2351852|pass|200810|rick@jrmyers.net ("Rick Myers")|Palm-TreoPhoneCallDB|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010553
2351822|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010547
2351821|pass|200810|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010547
2351789|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010542
2351637|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010509
2351580|pass|200810|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200810010457
2350694|pass|200810|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.6.26.5|200810010205
2348696|fail|200809|jeff@zeroclue.com|Class-DBI-ClassGenerator|1.0|darwin-2level|5.10.0|darwin|9.2.0|200809300310
2348573|pass|200809|artur+cpan@niif.spb.su (ARTO)|Devel-CheckOS|1.44|i386-linux-thread-multi|5.10.0|linux|2.6.18-92.1.6.el5xen|200809300833
2347621|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi-64int|5.11.0 patch 34448|freebsd|6.2-release|200809301612
2346554|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.11.0 patch 34444|netbsd|4.0|200809301957
2346541|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809302059
2346495|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd-thread-multi|5.10.0|freebsd|6.1-release|200809302053
2346083|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.10.0|linux|2.6.26.5|200809301352
2345902|pass|200809|vpit@cpan.org|Devel-CheckOS|1.44|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809301921
2345835|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.10.0 patch 34437|netbsd|4.0|200809301800
2345731|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|i586-linux-thread-multi|5.8.8|linux|2.6.16|200809301848
2345644|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|i586-linux-thread-multi|5.8.5|linux|2.6.8.1|200809301820
2345489|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|i386-linux|5.8.0|linux|2.6.9-22.18.bz155725.elsmp|200809301750
2345463|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|x86_64-linux-thread-multi|5.8.3|linux|2.6.4|200809301747
2345442|pass|200809|g.grigelionis@computer.org|Devel-CheckOS|1.44|x86_64-linux|5.8.8|linux|2.6.5-7.276-smp|200809301745
2345369|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.6.2|freebsd|6.1-release|200809301737
2345328|pass|200809|bingos@cpan.org|Devel-CheckOS|1.44|i386-netbsd-thread-multi-64int|5.8.8 patch 34439|netbsd|4.0|200809301628
2345303|pass|200809|martin.evans@easysoft.com|Devel-CheckOS|1.44|i686-linux-ld|5.10.0|linux|2.6.24-19-generic|200809301631
2345254|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|alpha-netbsd|5.10.0|netbsd|2.1.0_stable|200809301522
2344936|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|sun4-solaris-thread-multi|5.10.0|solaris|2.9|200809300930
2344441|pass|200809|martin.evans@easysoft.com|XML-Tiny|1.11|i686-linux-ld|5.10.0|linux|2.6.24-19-generic|200809301419
2344321|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|i686-linux|5.6.2|linux|2.4.27-3-686|200809301358
2344168|fail|200809|hohokus@gmail.com|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809300838
2343970|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|i686-linux|5.8.8|linux|2.4.27-3-686|200809301306
2343943|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809301300
2343916|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|IP35-irix-64int|5.8.8|irix|6.5|200809301357
2343785|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809300738
2343727|pass|200809|david@cantrell.org.uk|Devel-CheckOS|1.44|darwin-thread-multi-2level|5.10.0|darwin|8.10.1|200809301219
2342845|pass|200809|JOST@cpan.org ("Josts Smokehouse")|Devel-CheckOS|1.44|i86pc-solaris-64int|5.10.0|solaris|2.11|200809301041
2342820|pass|200809|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809300936
2342758|pass|200809|chris@bingosnet.co.uk|XML-Tiny|1.11|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809300923
2342658|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.44|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809300359
2342112|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.10.0|freebsd|6.1-release|200809300836
2340018|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809300215
2339931|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809300003
2339791|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809300132
2339645|pass|200809|flemming@ipk-gatersleben.de|Data-Compare|1.21|x86_64-linux|5.8.5|linux|2.6.9-55.0.9.elsmp|200809291527
2339578|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809300028
2339210|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.8|darwin|8.10.0|200809292348
2339198|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.8.6|darwin|8.10.0|200809292346
2339185|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int|5.6.2|darwin|8.10.0|200809292344
2339181|pass|200809|chris@bingosnet.co.uk|Devel-CheckOS|1.44|darwin-thread-multi-64int-2level|5.10.0|darwin|8.10.0|200809292343
2339049|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809291830
2337779|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809292158
2337627|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809291552
2337521|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809292123
2337451|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.10.0 patch 34437|linux|2.6.22-1-k7|200809292130
2337323|fail|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.11.0 patch 34435|linux|2.6.26.5|200809291512
2337154|pass|200809|srezic@cpan.org|Devel-CheckOS|1.44|i386-freebsd|5.8.8 patch 33662|freebsd|6.1-release-p23|200809292051
2337033|fail|200809|taro-nishino@wine.ocn.ne.jp ("Taro Nishino")|Devel-CheckOS|1.44|MSWin32-x86-multi-thread|5.8.8|MSWin32|5.0|200809300340
2336854|fail|200809|chris@bingosnet.co.uk|Palm-SMS|0.02|darwin-thread-multi-64int-2level|5.8.8 patch 34327|darwin|8.10.0|200809291916
2335392|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-thread-multi-64int|5.8.7|linux|2.6.14|200809291730
2335389|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8|linux|2.6.14|200809291730
2335386|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8|linux|2.6.22-1-k7|200809291730
2335382|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.8.8 patch 33994|linux|2.6.22-1-k7|200809291729
2335380|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0|linux|2.6.22-1-k7|200809291729
2335378|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.44|i686-linux-64int|5.10.0 patch 33955|linux|2.6.22-1-k7|200809291729
2335260|cpan|200809|DCANTRELL|Devel-CheckOS|1.44||0|||200809291718
2327831|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809282012
2326598|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281754
2325883|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281411
2325540|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281607
2324323|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281350
2323841|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34435|linux|2.6.22-1-k7|200809281313
2322337|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809280900
2319498|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271713
2319483|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.10.0|linux|2.6.26.5|200809271708
2319157|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271616
2319035|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.10.0|linux|2.6.26.5|200809271557
2319010|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.10.0|linux|2.6.26.5|200809271551
2319004|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271550
2318903|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.10.0|linux|2.6.26.5|200809271524
2318880|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.10.0|linux|2.6.26.5|200809271520
2318877|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271520
2318871|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.10.0|linux|2.6.26.5|200809271519
2318831|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809272029
2318736|fail|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.10.0|linux|2.6.26.5|200809271505
2318395|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.10.0|linux|2.6.26.5|200809271436
2318391|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.10.0|linux|2.6.26.5|200809271435
2318062|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271359
2317935|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.10.0|linux|2.6.26.5|200809271348
2317821|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.10.0|linux|2.6.26.5|200809271335
2317778|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271330
2317727|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809271923
2317715|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.10.0|linux|2.6.26.5|200809271322
2317671|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.10.0|linux|2.6.26.5|200809271317
2317661|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.10.0|linux|2.6.26.5|200809271316
2317583|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271305
2317491|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.10.0|linux|2.6.26.5|200809271256
2317421|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809271242
2317418|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.8.8|linux|2.6.26.5|200809271241
2317346|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.10.0|linux|2.6.26.5|200809271226
2316816|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270913
2316677|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.8.8|linux|2.6.26.5|200809270844
2316652|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.8.8|linux|2.6.26.5|200809270840
2316644|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270838
2316478|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.8.8|linux|2.6.26.5|200809270808
2316406|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.8.8|linux|2.6.26.5|200809270758
2316405|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270758
2316400|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.8.8|linux|2.6.26.5|200809270757
2316346|pass|200809|chris@bingosnet.co.uk|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809271246
2316271|pass|200809|eshank@drwholdings.com|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809261140
2316178|pass|200809|jeff@zeroclue.com|Devel-CheckOS|1.43|darwin-2level|5.10.0|darwin|9.2.0|200809261645
2316116|fail|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.8.8|linux|2.6.26.5|200809270712
2315835|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Overwrite|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270635
2315825|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.01|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270633
2315816|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.8|linux|2.6.26.5|200809270633
2315811|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.8.8|linux|2.6.26.5|200809270632
2315682|pass|200809|jeff@zeroclue.com|XML-DoubleEncodedEntities|1.0|darwin-2level|5.10.0|darwin|9.2.0|200809261539
2315664|pass|200809|jeff@zeroclue.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.2.0|200809261459
2315436|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270540
2315331|pass|200809|rick@jrmyers.net ("Rick Myers")|Games-Dice-Advanced|1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270527
2315312|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.8.8|linux|2.6.26.5|200809270525
2315184|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.8.8|linux|2.6.26.5|200809270459
2315158|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270455
2315100|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.8.8|linux|2.6.26.5|200809270445
2315047|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.8.8|linux|2.6.26.5|200809270438
2315044|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.8.8|linux|2.6.26.5|200809270437
2315002|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270429
2314935|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.8.8|linux|2.6.26.5|200809270418
2314821|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.8.8|linux|2.6.26.5|200809270350
2314737|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-STDOUT|1.03|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270327
2314721|pass|200809|rick@jrmyers.net ("Rick Myers")|Net-Random|2.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270324
2314715|pass|200809|rick@jrmyers.net ("Rick Myers")|Sort-MultipleFields|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270323
2314486|pass|200809|rick@jrmyers.net ("Rick Myers")|Sub-WrapPackages|1.2|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270235
2313923|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-SerialCorrelation|1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270027
2313873|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-CanBeA|1.2|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270027
2313864|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone-UK-DetailedLocations|1.3|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270025
2313639|fail|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.20|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270349
2313316|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270310
2313310|pass|200809|rick@jrmyers.net ("Rick Myers")|Number-Phone|1.6|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270310
2313151|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809270448
2312978|pass|200809|rick@jrmyers.net ("Rick Myers")|Class-DBI-ClassGenerator|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270224
2312795|pass|200809|rick@jrmyers.net ("Rick Myers")|File-Find-Rule-Permissions|1.3|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270201
2312628|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Transactional|1.02|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270134
2312597|pass|200809|rick@jrmyers.net ("Rick Myers")|NestedMap|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270128
2312544|pass|200809|rick@jrmyers.net ("Rick Myers")|Statistics-ChiSquare|0.5|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270119
2312505|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Z80|1.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270111
2312499|pass|200809|rick@jrmyers.net ("Rick Myers")|CPU-Emulator-Memory|1.1001|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270110
2312446|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Scalar-Decay|1.1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270100
2312313|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Longest|1.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270037
2312149|pass|200809|rick@jrmyers.net ("Rick Myers")|Tie-Hash-Rank|1.0.1|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809270000
2309548|pass|200809|gabor@pti.co.il ("Gabor Szabo")|Devel-CheckOS|1.43|i486-linux-gnu|5.8.8|linux|2.6.15.7|200809261932
2309012|pass|200809|mgardner@usatech.com|Data-Compare|1.21|cygwin-thread-multi-64int|5.10.0 patch 34065|cygwin|1.5.25(0.15642)|200809261128
2302099|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.10.0|linux|2.6.26.5|200809251442
2302018|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.8.8|linux|2.6.26.5|200809251429
2297597|pass|200809|dagolden@cpan.org (DAGOLDEN)|CPU-Emulator-Z80|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250331
2297496|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Transactional|1.02|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250318
2296959|pass|200809|dagolden@cpan.org (DAGOLDEN)|DBIx-Class-SingletonRows|0.11|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250204
2296227|pass|200809|dagolden@cpan.org (DAGOLDEN)|Palm-Treo680MessagesDB|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250025
2294629|pass|200809|dagolden@cpan.org (DAGOLDEN)|Sort-MultipleFields|1.0|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809250030
2292092|pass|200809|rick@jrmyers.net ("Rick Myers")|Palm-SMS|0.02|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809241509
2291763|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.10.0|linux|2.6.26.5|200809241416
2291076|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.8.8|linux|2.6.26.5|200809241242
2288619|pass|200809|jeff@zeroclue.com|Data-Compare|1.21|darwin-2level|5.10.0|darwin|9.2.0|200809240350
2286699|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809240451
2286365|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.8.8 patch 34412|linux|2.6.22-1-k7|200809240446
2285740|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.8.8|linux|2.6.26.5|200809240143
2284690|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.00|200809240157
2284488|pass|200809|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809231825
2284464|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux-thread-multi-64int-ld|5.10.0|linux|2.6.24-16-generic|200809231822
2283878|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Hexdumper|1.4|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809231648
2283100|pass|200809|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231436
2282898|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231419
2282820|pass|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.8.8|linux|2.6.26.5|200809231412
2282765|pass|200809|dagolden@cpan.org (DAGOLDEN)|Class-DBI-ClassGenerator|1.01|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231409
2282157|pass|200809|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|MSWin32-x86-multi-thread|5.10.0|MSWin32|5.1|200809231319
2279896|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.26.5|200809230902
2279583|fail|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.10.0|linux|2.6.26.5|200809230813
2278891|pass|200809|rick@jrmyers.net ("Rick Myers")|Data-Compare|1.21|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809230655
2278630|pass|200809|andy@hexten.net|Devel-CheckLib|0.5|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809231241
2278342|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230630
2277905|fail|200809|rick@jrmyers.net ("Rick Myers")|Bryar|3.0|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809230605
2276822|pass|200809|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230510
2276210|pass|200809|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230438
2275372|pass|200809|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230356
2274657|pass|200809|dagolden@cpan.org (DAGOLDEN)|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230315
2273987|pass|200809|dagolden@cpan.org (DAGOLDEN)|Acme-Scurvy-Whoreson-BilgeRat|1.1|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230243
2272391|pass|200809|dagolden@cpan.org (DAGOLDEN)|XML-DoubleEncodedEntities|1.0|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230118
2272202|pass|200809|dagolden@cpan.org (DAGOLDEN)|Data-Compare|1.21|i686-linux|5.10.0|linux|2.6.24-16-generic|200809230108
2270334|pass|200809|andy@hexten.net|Data-Hexdumper|1.4|i386-freebsd-thread-multi|5.10.0|freebsd|6.2-release|200809230309
2267038|pass|200809|stro@cpan.org|Class-DBI-ClassGenerator|1.01|i686-linux|5.10.0|linux|2.4.27-2-386|200809220120
2264023|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809220007
2263304|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Sub-WrapPackages|1.2|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809212219
2262089|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|XML-Tiny|1.11|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809211948
2261749|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Hexdumper|1.4|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809211920
2261013|fail|200809|vpit@cpan.org|Bryar|3.0|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809211725
2259935|pass|200809|vpit@cpan.org|Data-Compare|1.21|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809211442
2259900|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.8.8|linux|2.6.26.5|200809210835
2259896|pass|200809|shlomif@cpan.org|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.27-desktop-0.rc6.3.2mnb|200809211534
2259096|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.11.0 patch 34383|linux|2.6.26.5|200809210730
2258972|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckOS|1.43|i686-linux|5.10.0|linux|2.6.26.5|200809210654
2258126|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckOS|1.43|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809210954
2257805|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Data-Compare|1.21|i686-linux-64int|5.11.0 patch 34384|linux|2.6.22-1-k7|200809210901
2254479|pass|200809|rick@jrmyers.net ("Rick Myers")|Devel-CheckLib|0.5|i686-linux|5.8.8|linux|2.6.24.3|200809200851
2253129|pass|200809|andreas.koenig.gmwojprw@franz.ak.mind.de|Devel-CheckLib|0.5|i686-linux-64int|5.8.8 patch 34376|linux|2.6.22-1-k7|200809200458
2252554|pass|200809|bingos@cpan.org|Devel-CheckOS|1.43|i386-dragonfly-thread-multi-64int|5.10.0 patch 34372|dragonfly|2.0.0-release|200809200237
2250965|pass|200809|vpit@cpan.org|Devel-CheckOS|1.43|x86_64-linux-thread-multi|5.10.0|linux|2.6.26.3-fuuka.profvince.com|200809192322
2250695|pass|200809|srezic@cpan.org|Tie-STDOUT|1.03|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192113
2250570|pass|200809|srezic@cpan.org|Palm-Treo680MessagesDB|1.0|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192112
2250153|pass|200809|srezic@cpan.org|Statistics-SerialCorrelation|1.1|i386-freebsd|5.8.8 patch 34313|freebsd|6.1-release-p23|200809192113
2248907|pass|200809|srezic@cpan.org|Games-Dice-Advanced|1.1|i386-free