package Evo;
use strict;
use warnings;
use feature ();
use Carp();
use parent 'Exporter';
use Module::Load;
use Module::Loaded;
use List::Util;

my @DEP_FUNCS = qw(dep dep_exists dep_new dep_single dep_module);
our @EXPORT_OK = (@DEP_FUNCS, qw( cb call fn));
our %EXPORT_TAGS = (all => \@EXPORT_OK, dep => \@DEP_FUNCS);

our $_DEPS = {};
my $_SINGLES = {};

my $CLASS = do {
  local $@;
  !$ENV{NO_EVO_CLASS_XS}
    && eval { require Class::XSAccessor; Class::XSAccessor->VERSION('1.19'); }
    ? 'Evo::Class::XSA'
    : 'Evo::Class';
};

sub import {
  $_->import for qw(strict warnings utf8);
  feature::->import(':5.16');

  my ($class, $target, @list) = (shift, scalar caller(), @_);
  return unless @list;

  my %remaining = map { $_ => 1 } @list;

  # modern
  if (delete $remaining{'-modern'}) {
    feature::->import('signatures', 'postderef');
    warnings::->unimport('experimental::signatures',
      'experimental::postderef');
  }

  # inline class hasn't new method, mark as loaded
  if (delete $remaining{'-loaded'}) {
    mark_as_loaded($target);
  }

  # oo
  if (delete $remaining{'-class'}) {
    load $CLASS;
    $CLASS->init_class($target);
  }
  elsif (my @bases = grep { $_ =~ /::/ } keys %remaining) {
    delete $remaining{$_} for @bases;
    load $CLASS;
    $CLASS->init_class($target, @bases);
  }

  Exporter::export_to_level($class, 1, $target, keys %remaining);
}


sub dep($) {
  my $key = _trim($_[0]);
  Carp::croak "Unmet dependency $key" unless dep_exists($key);
  $_DEPS->{$key};
}

sub dep_exists ($) { exists $_DEPS->{_trim($_[0])} }

sub dep_module (*) {
  my $key = shift;
  my $class = dep_exists($key) ? dep($key) : _trim($key);
  load($class) unless is_loaded($class);
  $class;
}

sub dep_new (*;@) { dep_module(shift)->new(@_) }

sub dep_single(*) {

  # for perf
  my $key = _trim($_[0]);
  $_SINGLES->{(exists $_DEPS->{$key} ? $_DEPS->{$key} : $key)}
    ||= dep_new($key);
}

sub fn($$) {
  my $fn = pop;
  my $extend = shift || {};
  Carp::croak "syntax: evo {foo=> 'bar'}, sub {}" unless ref $fn eq 'CODE';

  my %new = ((map { (_trim($_), $extend->{$_}) } keys %$extend), %$_DEPS);
  sub {
    local $_DEPS;
    $_DEPS = \%new;
    $fn->(@_);
  };
}

sub _trim { $_[0] =~ /(.+)::$/ ? $1 : $_[0] }

sub call($$) { fn($_[0], $_[1])->() }

sub cb(&) {
  my $fn   = $_[0];
  my $deps = $_DEPS;
  sub { local $_DEPS; $_DEPS = $deps; $fn->(@_) };
}


our $VERSION = '0.0171';    # VERSION

1;

# ABSTRACT: Perl Evo design pattern

__END__

=pod

=encoding UTF-8

=head1 NAME

Evo - Perl Evo design pattern

=head1 VERSION

version 0.0171

=head1 DESCRIPTION

Tiny framework for organization non-trivial code into a simple logic.
It includes benefits from functional and OO styles.

Write fast, reusable and testable code.

=head1 ATTENTION

Attention. Don't use it right now. Wait for the stable release.
Documentation is on it's way.

=head1 IMPORTS

With or without options, C<use Evo> does following:

  use strict;
  use warnings;
  use feature ':5.16';

=head2 -modern

Enable cool experimental features. Works with 5.20.0 only

  use feature ('signatures', 'postderef');
  no warnings('experimental::signatures', 'experimental::postderef');

=head2 -loaded

This marks inline or generated classes as loaded, so can be used with
C<require> or C<use>. So this code won't die

  require My::Inline;

  {
    package My::Inline;
    use Evo -loaded;
    sub foo {'foo'}
  }

=head2 -class

Makes an OO for current module.
Also import C<-loaded> key

  package My::Base;
  use Evo -class;
  has 'foo';

  package My::Child;
  use Evo 'My::Base';

  package My::Child2;
  use Evo -class;
  extends 'My::Base';

  has bar => 33;
  has bas => sub { [1, 2] };

=head2 :dep

  use Evo qw(dep dep_exists dep_new dep_single);

=head2 :all

  Export all (functions) without C<-> prefix

=head1 FUNCTIONS

=head2 fn

  my $fn = Evo::fn {dep => 'MyDep'}, sub {...};

Return a fn that start a dependencies zone. Actually L</call> just
invokes a result of this function

=head2 call

  Evo::call {foo => 'MyFoo'}, sub {...};

Start new dsl zone with dependencies using L</fn> and invoke it. If this method is
called inside other dsl zone, dependencies will not be overriden, if they already exist

  # print MyFoo in both cases
  Evo::call {foo => 'MyFoo'}, sub {
    say dep 'foo';
    Evo::call {foo => 'NotChanded'} => sub { say dep 'foo' };
  };

=head2 cb

Returns a function that calls a given function with current envirement. Is usefull
for using in event loops

  my $cb;
  evo {foo => 'MyFoo'} => sub {
    $cb = cb { dep(shift) };
  };

  say $cb->('foo');

The behaviour is almost the same 

  $fn = Evo::fn {}, sub {...};
  $fn = Evo::cb {...};

but while it only copies current dependencies, using it is a little bit
faster, and you don't need to write C<sub>

=head2 dep

Return a dependensy by key

  # croak outside evo dsl, because no dep
  # my $dep = dep 'foo';
  Evo::call {file => 'IO::File'}, sub {
    say dep('file');

    # the same
    my $fh = IO::File->new;
    my $fh = dep('file')->new;

  };

If dependency is missing, die.  Last '::' in key are trimmed, so:

  Evo::dep('My::') == Evo::dep('My');

=head2 dep_module

Loads a dependency or die. If dependency isn't exists, loads a package by given key without dying

  {
    package My::Class;
    use Evo -loaded;
    sub foo {'foo'}
  }

  # My::Class->foo
  say dep_module('My::Class')->foo;

  Evo::call {my => 'My::Class'}, sub {
    # My::Class->foo
    say dep_module('My::Class')->foo();
  };

=head2 dep_new

Treats dependency as class and invokes a C<new> method on this class. Return an result of that invocation

  my $fh1 = Evo::dep_new 'IO::File';
  my $fh2 = Evo::dep_new IO::File, '/etc/passwd', 'r';

  Evo::call {dir => 'IO::Dir'}, sub {
    my $d = Evo::dep_new 'dir', '.';
    say $d;
  };

First this function tries to find a dependency by key passed as the first
argument. If found, us it as class. If dependency doesn't exists by that key,
uses the key as a class

Remaining argument will be passed as arguments for the C<new> method.

This function also loads the class if  wasn't loaded, so you don't need to
manually call C<require IO::File;>

=head2 dep_single

Creates a single instance of the class using L</dep_new>

  # the same object
  my $mdn = Evo::dep_single 'Evo::Mdn';
  say $mdn == Evo::dep_single 'Evo::Mdn';

You can use it instead of singleton design pattern, because single dependency
is much-much(-much) better.

=head2 dep_exists

  say dep_exists '404';

Safely checks if a dependency exists

=head1 AUTHOR

alexbyk.com

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by alexbyk.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
