package Art::World {

  use v5.16;
  use utf8;

  our $VERSION = '0.16_04';

  use Zydeco version => $VERSION, authority => 'cpan:SMONFF';

  class Playground;

  role Concept {
    method idea_of_project {}

    method idea_file {}
    # etc.
  }

  class Abstraction with Concept {
    has idea!, process, file!, discourse, time, project;
  }

  role Showable {
    #requires exhibition;
    method exhibit {
      say "Show";
    }
  }

  role Collectionable {
    # Should be an object of type Agent
    has owner;
    has value ( is => rw );
    has status (
      enum     => ['for_sale', 'sold'],
      handles  => 1,
      default  => 'for_sale'
    );
    method belongs_to {
      return $self->owner;
    }
  }

  role Buyer {
    requires money;
    method acquire ( Num $price ) {
      say "I bought !";
    }

    method sale ( Num $price ) {
      say "I sold !";
    }
  }

  role Exhibit {
    has exhibition ( type => ArrayRef );
    has artist ( type => ArrayRef );
    has artwork ( type => ArrayRef );
    method display {
      say "Shoooow";
    }
  }

  role Active {
    method participate {
      say "That's interesting";
    }
  }


  class Wildlife {

    class Agent with Active {
      # Should be required but will be moved to the Crudable area
      has id         ( type => Int );
      has name!       ( type => Str );
      has reputation ( type => Int );

      class Artist {

        has artworks   ( type => ArrayRef );
        has collectors ( type => ArrayRef, default => sub { [] } );
        has collected  ( type => Bool, default => false, is => rw );
        has status (
          enum => [ 'underground', 'homogenic' ],
          handles => 1,
          default => sub {
            my $self = shift;
            $self->has_collectors ? 'homogenic' : 'underground' });

        method create {
          say $self->name . " create !";
        }

        method have_idea {
          say $self->name . ' have shitty idea' if true;
        }

        method has_collectors {
          if ( scalar @{ $self->collectors }  > 1 ) {
            $self->collected( true );
          }
        }

        # method new ($id, $name, @artworks, @collectors) {
        #     self.bless(:$id, :$name, :@artworks, :@collectors);
        # }
      }

      class Collector with Active, Buyer {
        has money! ( type => Num );
        # Actually an ArrayRef of Artworks
        has collection (
          type    => ArrayRef[Any, 0, 100],
          default => sub { [] }
        ) ;
      }

      class Public {
        method visit {
          say "I visited";
        }
      }
    }


    #use Art::Behavior::Crudable;
    # does Art::Behavior::Crudable;
    # has relations

  }

  class Place {
    has space;

    class Gallery with Exhibit, Buyer {

      has artwork (  type => ArrayRef );
      has artist (  type => ArrayRef );
      has event ( type => ArrayRef );
      has owner;
      has public;
      has money;

      # Should be moved to an Opening role
      method serve {
        say "What would you drink?";
      }
    }
  }

  class Work {

    has creation_date;
    has creator (
      is => ro,
      # ArrayRed of Artists
      type => ArrayRef[ Object ]
    );

    class Artwork with Showable, Collectionable  {

      has creation_date;
      has creator (
        is => ro,
        # Should be ArrayRed of Artists
        type => ArrayRef[ Object ]
      );
      has material;
      has size;
    }
  }
}

1;
__END__

=encoding UTF-8

=head1 NAME

Art::World - Agents interactions modeling  🎨

=head1 SYNOPSIS

  use Art::World;

  my $artwork = Art->new_artwork(
    creator => [ $artist, $another_artist ]  ,
    value => 100,
    owner => $f->person_name );

=head1 DESCRIPTION

C<Art::World> is an attempt to model and simulate a system describing the
interactions and influences between the various I<agents> of the art world.

More informations about the purposes and aims of this project can be found in
it's L<manual|Art::World::Manual>. Especially, the
L<history|Art::World::Manual/"HISTORY"> and the
L<objectives|Art::World::Manual/"OBJECTIVES"> section could be very handy to
understand how this is an artwork using programming.

=head1 OBJECTS

=head2 Art

Will be what you decide it to be

=head2 Wildlife

Activists of the Art World.

  my $agent = Art::World::Wildlife->new_agent( name => $f->person_name );

  $agent->participate;    # ==>  "That's interesting"

A generic entity that can be any activist of the L<Art::World>. Provides all
kind of C<Agents> classes and roles.

=head3 Roles

=over

=item C<Active>

Provide a C<participate> method.

=item C<Buyer>

Provide a C<aquire> method requiring some C<money>.

=back

=head3 Classes

=head4 C<Artist>

The artist got a lots of wonderful powers:

=over

=item C<create>

=item C<have_idea> all day long

In the beginning of their carreer they are usually underground, but this can
change in time.

  $artist->is_underground if not $artist->has_collectors;

=back

=head2 Artwork

The base thing producted by artists. Artwork is a Zydeco subclass of
L<Art::Work> that have a Showable and Collectionable role..

=head2 Place

Since it implements the Buyer role, a gallery can both acquire() and sell(). All
this is encapsulated in the Buyer role because there is no such thing as
somebody in the art world that buy but doesn't sale.

=cut

=head1 AUTHOR

Seb. Hu-Rillettes <shr@balik.network>

=head1 CONTRIBUTORS

Sébastien Feugère <sebastien@feugere.net>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2020 Seb. Hu-Rillettes

This library is free software; you can redistribute it and/or modify it under the Artistic License 2.0.
