package Firewall::PaloAlto::Common;

use Moose::Role;

use 5.006;
use strict;
use warnings;

use Carp;
use XML::Simple;
use LWP::UserAgent;
use Modern::Perl;

use Data::Dumper;

our $VERSION = '0.01';

=head1 NAME

Firewall::PaloAlto::Common - Moose role for Palo Alto API calls.

=cut

=head1 CLASS METHODS

=head2 new(%parameters)

The constructor generates a new object to interact with a specific firewall.

    my $palo = Firewall::PaloAlto->new(host => 'paloalto.local', username => 'admin', password => 'admin', ssl => 0, debug => 1);

=cut

has host => ( is => 'ro',
                isa => 'Str',
                required => '1',
);

has username => ( is => 'ro',
                    isa => 'Str',
                    required => '1',
);

has password => ( is => 'ro',
                    isa => 'Str',
                    required => '1',
);

has api_key => ( is => 'rw', isa => 'Str', init_arg => undef, writer => '_api_key');

has ssl => ( is => 'ro', isa => 'Bool', default => 1 );
has base_uri => (   is => 'ro', 
                    isa => 'Str', 
                    lazy => '1',
                    init_arg => undef,
                    default => sub { return 'http' . ($_[0]->ssl ? 's' : '') . '://' . $_[0]->host . '/api/?'; } 

);

has 'debug' => ( is => 'ro', isa => 'Bool', required => 0, default => 0 );



sub _create_requester {
    my $self = shift;
    my %args = @_;

    my $request =  $self->base_uri. 'key=' . $self->api_key;

    for my $key (keys %args) {
        #Add the argument on to the command line
        $request .= "&$key=$args{$key}";
        
        $self->_debug_print("API Request Creation", "$key=$args{$key}");
    }


    return sub {
        my (%request_args) = @_;
        my $user_agent = LWP::UserAgent->new(ssl_opts => { verify_hostname => 0, SSL_verify_mode => 0x00 });

        for my $key (keys %request_args) {
            $self->_debug_print("API Request String", "$key=$request_args{$key}");
            $request .= "&$key=$request_args{$key}";
        }

        my $http_request = HTTP::Request->new(GET => $request);
        my $http_response = $user_agent->request($http_request);

        return $self->_check_response_and_parse($http_response);
    }
}



sub _check_response_and_parse {
    my $self = shift;
    my $http_response = shift;

    #We locally redefine croak so we can get some nice red printing around it.
    no warnings 'redefine';
    local *original_croak = \&croak;
    local *croak = sub {  original_croak("\e[31m".$_[0]."\e[0m"); };

    #Check the http response message - croak if its not successful
    croak "*[HTTP Request] Failed: ".$http_response->content if !$http_response->is_success;

    my $palo_response = XMLin($http_response->content, KeyAttr => 'name');

    #If the response doesn't contain a code (i.e. the keygen request doesn't), then we check the status string.
    #If its successful, we assign our own 'custom code' as a workaround
    if (!defined $palo_response->{code} && $palo_response->{status} eq "success") {
        $palo_response->{code} = 1023;
    }

    #Check the response from the Palo Alto:
    my $code_info = $self->_get_response_info($palo_response->{code});
    my ($api_error) = $code_info->{parse}->($palo_response);

    carp "*[API Request] $code_info->{string}: $api_error" and return 0 if $code_info->{is_error};

    $self->_debug_print("API Response", $palo_response->{status});

    return $palo_response;
}




sub _get_response_info {
    my ($self, $code) = @_;

    my $response_codes = {
                            400 => {is_error => 1, string => "Bad Request",
                                parse => sub { "Error 400" } },
                            403 => { is_error => 1, string => "Forbidden",
                                parse => sub { $_[0]->{result}->{msg}; }
                            },
                            1 => { is_error => 1, string => "Unknown Command",
                                parse => sub { "Error 1" } },
                            2 => { is_error => 1, string => "Internal Error (2)",
                                parse => sub { "Error 2" } },
                            3 => { is_error => 1, string => "Internal Error (3)",
                                parse => sub { "Error 3" } },
                            4 => { is_error => 1, string => "Internal Error (4)",
                                parse => sub { "Error 4" } },
                            5 => { is_error => 1, string => "Internal Error (4)",
                                parse => sub { "Error 5" } },
                            6 => { is_error => 1, string => "Bad XPath",
                                parse => sub { "Error 6" } },
                            7 => { is_error => 1, string => "Object not present",
                                parse => sub { "Error 7" } },
                            8 => { is_error => 1, string => "Object not unique",
                                parse => sub { "Error 8" } },
                            9 => { is_error => 1, string => "Internal Error (9)",
                                parse => sub { "Error 9" } },
                            10 => { is_error => 1, string => "Reference count not zero",
                                parse => sub { "Error 10" } },
                            11 => { is_error => 1, string => "Internal Error (11)",
                                parse => sub { "Error 11" } },
                            12 => { is_error => 1, string => "Invalid Object",
                                parse => sub { flat ($_[0]->{msg}->{line}) } },
                            13 => { is_error => 1, string => "Operation Failed",
                                parse => sub { $_[0]->{msg}->{line} } },
                            14 => { is_error => 1, string => "Operation Not Possible",
                                parse => sub { "Error 14" } },
                            15 => { is_error => 1, string => "Operation Denied",
                                parse => sub { "Error 15" } },
                            16 => { is_error => 1, string => "Unauthorized",
                                parse => sub { "Error 16" } },
                            17 => { is_error => 1, string => "Invalid Command",
                                parse => sub { flat ($_[0]->{msg}->{line}) } },
                            18 => { is_error => 1, string => "Malformed XML",
                                parse => sub { $_[0]->{msg}->{line} } },
                            19 => { is_error => 0, string => "Get Request Successful",
                                parse => sub{} },
                            20 => { is_error => 0, string => "Set Request Successful",
                                parse => sub{} },
                            21 => { is_error => 1, string => "Internal Error (21)",
                                parse => sub { "Error 21" } },
                            22 => { is_error => 1, string => "Session Timed Out",
                                parse => sub { "Error 22" } },
                            #Custom code for keygen success
                            1023 => { is_error => 0, string => "KeyGen Successful", parse => sub {} },
                        };

    #Return the hash, or 'undef' if the code doesn't exist.
    return $response_codes->{$code};
}




sub _generate_elements {
    my $self = shift;
    my %element_hash = @_;
    my $element_string = "";

    for my $key (keys %element_hash) {
        #If the key is named "entry", this is an entry element
        if ($key eq 'entry') {
            $element_string .= "<entry name=\"$element_hash{$key}\"/>";
            next;
        }

        $element_string .= "<$key>";

        #If our hash points to an array reference, we iterate through the array and add member.
        #This creates <key><member>a</member><member>b</member></key>
        if (ref($element_hash{$key}) eq "ARRAY") {
            for my $member (@{ $element_hash{$key} }) {
                   $element_string .= "<member>$member</member>";
            }
        }
        #If we're pointing to another hash, we recurse down, as the structure will be the same.
        #This allows us to do <key><another-key>data</another-key></key>
        elsif (ref($element_hash{$key}) eq "HASH") {
            $element_string .= $self->_generate_elements(%{ $element_hash{$key} });
        }
        #Otherwise its just a normal <key>value</key>
        else {
            $element_string .= "$element_hash{$key}";
        }

        $element_string .= "</$key>";
    }

    return $element_string;
}


sub _debug_print {
    my $self = shift;
    my ($category, $debug_string) = @_;
    my $string_colour = "\e[0;36m";
    my $string_norm = "\e[0m";

    if (!$self->debug()) {
        return 0;
    }

    say "*[".$string_colour.$category.$string_norm."] $debug_string";
}


=head1 CONFIGURATION METHODS 

=head2 connect()

The connect functions connects to the Palo Alto, validates and saves the API key.
It has no parameters

 $pa->connect();

=cut

sub connect {
    my $self = shift; 
    my $ssl_options;

    my $user_agent = LWP::UserAgent->new(ssl_opts => { verify_hostname => 0, SSL_verify_mode => 0x00 });
    my $key_request = HTTP::Request->new(GET => $self->base_uri.'type=keygen&user='.$self->username.'&password='.$self->password);

    $self->_debug_print("Key Request", "");

    my $key_response = $user_agent->request($key_request);

    my $pa_response = $self->_check_response_and_parse($key_response);

    if ($pa_response->{status} eq 'success') {
        $self->_api_key($pa_response->{result}->{key});
        return 1;
    }
    else {
        return 0;
    }   
}





=head1 AUTHOR

Greg Foletta, C<< <greg at foletta.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-firewall-paloalto-common at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Firewall-PaloAlto-Common>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.




=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Firewall::PaloAlto::Common


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Firewall-PaloAlto-Common>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Firewall-PaloAlto-Common>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Firewall-PaloAlto-Common>

=item * Search CPAN

L<http://search.cpan.org/dist/Firewall-PaloAlto-Common/>

=back


=head1 ACKNOWLEDGEMENTS


=head1 LICENSE AND COPYRIGHT

Copyright 2015 Greg Foletta.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of Firewall::PaloAlto::Common
