package Statocles::Document;
# ABSTRACT: Base class for all Statocles documents
$Statocles::Document::VERSION = '0.053';
use Statocles::Base 'Class';


has path => (
    is => 'rw',
    isa => Path,
    coerce => Path->coercion,
);


has title => (
    is => 'rw',
    isa => Str,
);


has author => (
    is => 'rw',
    isa => Str,
);


has content => (
    is => 'rw',
    isa => Str,
);


has tags => (
    is => 'rw',
    isa => ArrayRef,
    default => sub { [] },
    coerce => sub {
        return [] unless $_[0];
        if ( !ref $_[0] ) {
            return [ split /\s*,\s*/, $_[0] ];
        }
        return $_[0];
    },
);


has links => (
    is => 'rw',
    isa => LinkHash,
    default => sub { +{} },
    coerce => LinkHash->coercion,
);


has date => (
    is => 'rw',
    isa => InstanceOf['Time::Piece'],
    predicate => 'has_date',
);


has template => (
    is => 'rw',
    isa => Maybe[ArrayRef[Str]],
    coerce => sub {
        return $_[0] if ref $_[0];
        return [ grep { $_ ne '' } split m{/}, $_[0] ];
    },
    predicate => 'has_template',
);


has layout => (
    is => 'rw',
    isa => Maybe[ArrayRef[Str]],
    coerce => sub {
        return $_[0] if ref $_[0];
        return [ grep { $_ ne '' } split m{/}, $_[0] ];
    },
    predicate => 'has_layout',
);


has data => (
    is => 'rw',
);

1;

__END__

=pod

=head1 NAME

Statocles::Document - Base class for all Statocles documents

=head1 VERSION

version 0.053

=head1 DESCRIPTION

A Statocles::Document is the base unit of content in Statocles.
L<Applications|Statocles::App> take documents to build
L<pages|Statocles::Page>.

This is the Model class in the Model-View-Controller pattern.

=head1 ATTRIBUTES

=head2 path

The path to this document. This is not settable from the frontmatter.

=head2 title

The title of this document.

=head2 author

The author of this document.

=head2 content

The raw content of this document, in markdown. This is everything below
the frontmatter.

=head2 tags

The tags for this document. Tags are used to categorize documents.

Tags may be specified as an array or as a comma-seperated string of
tags.

=head2 links

Related links for this document. Links are used to build relationships
to other web addresses. Link categories are named based on their
relationship. Some possible categories are:

    alternate - The same document posted to another web site

Each category contains an arrayref of hashrefs of L<link objects|Statocles::Link>.
See the L<Statocles::Link|Statocles::Link> documentation for a full list of
supported attributes. The most common attributes are:

    text - The text of the link
    href - The URL for the link

=head2 date

The date/time this document is for. For pages, this is the last modified date.
For blog posts, this is the post's date.

=head2 template

A template override for this document. If set, the L<document
page|Statocles::Page::Document> will use this instead of the template provided
by the application.

=head2 layout

A layout template override for this document. If set, the L<document
page|Statocles::Page::Document> will use this instead of the layout provided
by the application.

=head2 data

Any kind of miscellaneous data. This is available immediately in the document
content.

=head1 SEE ALSO

=over 4

=item L<Statocles::Help::Content>

The content guide describes how to edit content in Statocles sites, which are
represented by Document objects.

=back

=head1 AUTHOR

Doug Bell <preaction@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by Doug Bell.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
