package Data::Throttler_CHI;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2020-02-19'; # DATE
our $DIST = 'Data-Throttler_CHI'; # DIST
our $VERSION = '0.001'; # VERSION

use strict;
use warnings;

sub new {
    my ($package, %args) = @_;
    bless \%args, $package;
}

my $counter = 0;
sub try_push {
    my $self = shift;
    my $now = time();
    $counter++;
    $counter = 0 if $counter == 2e31; # wraparound 32bit int
    $self->{cache}->set(sprintf("%010d|%d", $now, $counter), 1, $self->{interval}); # Y2286!
    my @keys0 = $self->{cache}->get_keys;

    my @keys;
    for my $key (@keys0) {
        my ($key_time, $key_serial) = split /\|/, $key, 2;
        if ($key_time >= $now - $self->{interval}) {
            push @keys, $key;
        }
    }

    # these drivers return expired keys: Memory. so we need to purge these keys
    my $do_purge = rand() < 0.05; # probabilistic
    $self->{cache}->purge if $do_purge && @keys < @keys0;

    return @keys <= $self->{max_items} ? 1:0;
}

1;
# ABSTRACT: Data::Throttler-like throttler with CHI backend

__END__

=pod

=encoding UTF-8

=head1 NAME

Data::Throttler_CHI - Data::Throttler-like throttler with CHI backend

=head1 VERSION

This document describes version 0.001 of Data::Throttler_CHI (from Perl distribution Data-Throttler_CHI), released on 2020-02-19.

=head1 SYNOPSIS

 use Data::Throttler_CHI;
 use CHI;

 my $throttler = Data::Throttler_CHI->new(
     max_items => 100,
     interval  => 3600,
     cache     => CHI->new(driver=>"Memory"),
 );

 if ($throttle->try_push) {
     print "Item can be pushed\n";
 } else {
     print "Item must wait\n";
 }

=head1 DESCRIPTION

EXPERIMENTAL, PROOF OF CONCEPT.

This module tries to use L<CHI> as the backend for data throttling. It presents
an interface similar to, but simpler than, L<Data::Throttler>.

=head1 METHODS

=head2 new

Usage:

 my $throttler = Data::Throttler_CHI->new(%args);

Known arguments (C<*> means required):

=over

=item * max_items*

=item * interval*

=item * cache*

CHI instance.

=back

=head2 try_push

Usage:

 $bool = $throttler->try_push(%args);

Return 1 if data can be pushed, or 0 if it must wait.

Known arguments:

=over

=back

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Data-Throttler_CHI>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Data-Throttler_CHI>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Data-Throttler_CHI>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Data::Throttler>

L<CHI>

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2020 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
