package # hide from PAUSE
App::YTDL::Helper;

use warnings;
use strict;
use 5.010000;

use Exporter qw( import );
our @EXPORT_OK = qw( term_size sec_to_time insert_sep choose_a_dir choose_a_number print_hash get_filename
                     timestamp_to_upload_date encode_fs encode_stdout_lax encode_stdout write_json read_json
                     sanitize_for_path HIDE_CURSOR SHOW_CURSOR );

use Encode             qw( encode );
use Time::Local        qw( timelocal );
use Unicode::Normalize qw( NFC );

use Encode::Locale     qw();
use File::Touch        qw();
use JSON               qw();
use Term::Choose::Util qw( print_hash choose_a_dir choose_a_number term_size insert_sep unicode_trim );
use Unicode::GCString  qw();


use constant {
    HIDE_CURSOR => "\e[?25l",
    SHOW_CURSOR => "\e[?25h",
};



sub get_filename {
    my ( $opt, $title, $ext, $fmt ) = @_;
    $ext //= 'unknown';
    $fmt //= '';
    if ( length $fmt ) {
        $fmt = '_' . $fmt;
    }
    my $len_ext   = Unicode::GCString->new( $ext )->columns();
    my $len_fmt   = Unicode::GCString->new( $fmt )->columns();
    my $max_len_title = $opt->{max_len_f_name} - ( $len_fmt + 1 + $len_ext );
    $title = unicode_trim( sanitize_for_path( $title ), $max_len_title );
    my $file_name = $title . $fmt . '.' . $ext;
    return $file_name;
}

sub sanitize_for_path {
    my ( $path ) = @_;
    $path =~ s/^\s+|\s+\z//g;
    $path =~ s/\s/_/g;
    $path =~ s/^\.+//;
    #path =~ s/[^\p{Word}.()]/-/g;
    $path =~ s/["\/\\:*?<>|]/-/g;
    # NTFS and FAT unsupported characters:  / \ : " * ? < > |
    return $path;
}

sub timestamp_to_upload_date {
    my ( $info, $ex, $video_id, $file ) = @_;
    if ( $info->{$ex}{$video_id}{upload_datetime} =~ /^(\d\d\d\d)-(\d\d)-(\d\d)T(\d\d):(\d\d):(\d\d)\z/ ) {
        my $time = timelocal( $6, $5, $4, $3, $2 - 1, $1 );
        my $ref = File::Touch->new( time => $time );
        my $count = $ref->touch( encode_fs( $file ) );
    }
}


sub encode_fs {
    my ( $filename ) = @_;
    return encode( 'locale_fs', NFC( $filename ), Encode::FB_HTMLCREF );
}


sub encode_stdout_lax {
    my ( $string ) = @_;
    return encode( 'console_out', NFC( $string ), sub { '*' } );
}


sub encode_stdout {
    my ( $string ) = @_;
    return encode( 'console_out', NFC( $string ), Encode::FB_HTMLCREF );
}


sub sec_to_time {
    my ( $seconds, $long ) = @_;
    die 'seconds: not defined'      if ! defined $seconds;
    die 'seconds: "' . $seconds . '" invalid datatype' if $seconds !~ /^[0-9]+\z/;
    my ( $minutes, $hours );
    if ( $seconds ) {
        $minutes = int( $seconds / 60 );
        $seconds = $seconds % 60;
    }
    if ( $minutes ) {
        $hours   = int( $minutes / 60 );
        $minutes = $minutes % 60;
    }
    if ( $long ) {
        return sprintf( "%d:%02d:%02d", $hours // 0, $minutes // 0, $seconds );
    }
    if ( $hours ) {
        return sprintf( "%d:%02d:%02d", $hours, $minutes, $seconds );
    }
    elsif ( $minutes ) {
        return sprintf( "%d:%02d", $minutes, $seconds );
    }
    else {
        return sprintf( "0:%02d", $seconds );
    }
}


sub write_json {
    my ( $file, $h_ref ) = @_;
    my $json = JSON->new->pretty->canonical->utf8->encode( $h_ref );
    open my $fh, '>', encode_fs( $file ) or die $!;
    print $fh $json;
    close $fh;
}


sub read_json {
    my ( $file ) = @_;
    return {} if ! -f encode_fs( $file );
    open my $fh, '<', encode_fs( $file ) or die $!;
    my $json = do { local $/; <$fh> };
    close $fh;
    my $h_ref = JSON->new->utf8->decode( $json ) if $json;
    return $h_ref;
}



1;


__END__
