# MyMakeMakerExtras.pm -- my shared MakeMaker extras

# Copyright 2009 Kevin Ryde

# MyMakeMakerExtras.pm is shared by several distributions.
#
# MyMakeMakerExtras.pm is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 3, or (at your option) any later
# version.
#
# MyMakeMakerExtras.pm is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
# for more details.
#
# You should have received a copy of the GNU General Public License along
# with this file.  If not, see <http://www.gnu.org/licenses/>.

package MyMakeMakerExtras;
use strict;
use warnings;
use File::Basename;

use constant DEBUG => 0;

sub strip_comments {
  my ($str) = @_;
  $str =~ s/^\s*#.*\n//mg;
  $str
}

sub postamble {
  my ($makemaker) = @_;

  if (DEBUG) {
    require Data::Dumper;
    print Data::Dumper::Dumper($makemaker);
  }

  my $post = <<'HERE';

#------------------------------------------------------------------------------
# docs stuff -- from inc/MyMakeMakerExtras.pm

MY_POD2HTML = perl -MPod::Simple::HTML -e Pod::Simple::HTML::go
MY_MUNGHTML = \
  sed -e 's!http://search.cpan.org/perldoc[?]Glib%3A%3A\([^"]*\)!http://gtk2-perl.sourceforge.net/doc/pod/Glib/\1.html!'g \
      -e 's!http://search.cpan.org/perldoc[?]Gtk2%3A%3AGdk%3A%3A\([^"%]*\)"!http://gtk2-perl.sourceforge.net/doc/pod/Gtk2/Gdk/\1.html"!'g \
      -e 's!http://search.cpan.org/perldoc[?]Gtk2%3A%3A\([^"%]*\)"!http://gtk2-perl.sourceforge.net/doc/pod/Gtk2/\1.html"!'g \
      -e 's!http://search.cpan.org/perldoc[?]AptPkg!http://packages.debian.org/libapt-pkg-perl!' \
      -e 's!http://search.cpan.org/perldoc[?]apt-file!http://packages.debian.org/apt-file!'

HERE

  my @pmfiles = keys %{$makemaker->{'PM'}};
  my @exefiles = (defined $makemaker->{'EXE_FILES'}
                  ? @{$makemaker->{'EXE_FILES'}}
                  : ());
  my %html_files;

  foreach my $pm (@exefiles, @pmfiles) {
    my $html = File::Basename::basename($pm,'.pm') . '.html';
    next if $html_files{$html}++;
    $post .= <<"HERE";
$html: $pm Makefile
	\$(MY_POD2HTML) $pm \\
	  | \$(MY_MUNGHTML) >$html
HERE
  }

  $post .= "HTML_FILES = " . join(' ', keys %html_files) . "\n";
  $post .= <<'HERE';
html: $(HTML_FILES)
HERE


  $post .= <<'HERE';

#------------------------------------------------------------------------------
# development stuff -- from inc/MyMakeMakerExtras.pm

version:
	$(NOECHO) $(ECHO) $(VERSION)

HERE

  my $lint_files = $makemaker->{'MyMakeMakerExtras_LINT_FILES'};
  if (! defined $lint_files) {
    $lint_files = '$(EXE_FILES) $(TO_INST_PM)';
    # would prefer not to lock down the 't' dir existance at ./Makefile.PL
    # time, but it's a bit hard without without GNU make extensions
    if (-d 't') { $lint_files .= ' t/*.t'; }

    foreach my $dir ('examples', 'devel') {
      my $pattern = "$dir/*.pl";
      if (glob ($pattern)) {
        $lint_files .= " $pattern";
      }
    }
  }

  my $podcoverage = '';
  foreach my $class (@{$makemaker->{'MyMakeMakerExtras_Pod_Coverage'}}) {
    # the "." obscures it from MyExtractUse.pm
    $podcoverage .= "\t-perl -e 'use "."Pod::Coverage package=>$class'\n";
  }

  $post .= "LINT_FILES = $lint_files\n"
    . <<'HERE';
lint:
	perl -MO=Lint $(LINT_FILES)
pc:
HERE
  $post .= $podcoverage . <<'HERE';
	-podchecker $(LINT_FILES)
	perlcritic $(LINT_FILES)
unused:
	for i in $(LINT_FILES); do perl -Mwarnings::unused -I lib -c $$i; done

HERE

  $post .= <<'HERE';
myman:
	-mv MANIFEST MANIFEST.old
	touch SIGNATURE
	(make manifest 2>&1; diff -u MANIFEST.old MANIFEST) |less

# find files in the dist with mod times this year, but without this year in
# the copyright line
check-copyright-years:
	year=`date +%Y`; \
	tar tvfz $(DISTVNAME).tar.gz \
	| egrep '$$year-|debian/copyright' \
	| sed 's:^.*$(DISTVNAME)/::' \
	| (result=0; \
	  while read i; do \
	    case $$i in \
	      '' | */ \
	      | debian/changelog | debian/compat \
	      | COPYING | MANIFEST* | SIGNATURE | META.yml) \
	      continue ;; \
	    esac; \
	    if test -e "$(srcdir)/$$i"; then f="$(srcdir)/$$i"; \
	    else f="$$i"; fi; \
	    if ! grep -q "Copyright.*$$year" $$f; then \
	      echo "$$i":"1: this file"; \
	      grep Copyright $$f; \
	      result=1; \
	    fi; \
	  done; \
	  exit $$result)

# only a non-zero number is bad, allow an expression to copy a debug from
# another package
check-debug-constants:
	if egrep -n 'DEBUG => [1-9]' $(EXE_FILES) $(TO_INST_PM); then exit 1; else exit 0; fi

diff-prev:
	rm -rf diff.tmp
	mkdir diff.tmp
	cd diff.tmp \
	&& tar xfz ../$(DISTNAME)-`expr $(VERSION) - 1`.tar.gz \
	&& tar xfz ../$(DISTNAME)-$(VERSION).tar.gz
	-cd diff.tmp; diff -ur $(DISTNAME)-`expr $(VERSION) - 1` \
	                       $(DISTNAME)-$(VERSION) >tree.diff
	-$${PAGER:-less} diff.tmp/tree.diff
	rm -rf diff.tmp

TAG_FILES = $(TO_INST_PM)
TAGS: $(TAG_FILES)
	etags $(TAG_FILES)

HERE

  my $have_XS = scalar %{$makemaker->{'XS'}};
  my $arch = ($have_XS
              ? `dpkg --print-architecture`
              : 'all');
  chomp($arch);
  my $debname = (defined $makemaker->{'EXE_FILES'}
                 ? '$(DISTNAME)'
                 : "\Llib$makemaker->{'DISTNAME'}-perl");
  $post .=
    "DEBNAME = $debname\n"
    . "DPKG_ARCH = $arch\n"
    . <<'HERE';
DEBVNAME = $(DEBNAME)_$(VERSION)-1
DEBFILE = $(DEBVNAME)_$(DPKG_ARCH).deb

# ExtUtils::MakeMaker 6.42 of perl 5.10.0 makes "$(DISTVNAME).tar.gz" depend
# on "$(DISTVNAME)" distdir directory, which is always non-existant after a
# successful dist build, so the .tar.gz is always rebuilt.
#
# So although the .deb depends on the .tar.gz don't express that here or it
# rebuilds the .tar.gz every time.
#
# The right rule for the .tar.gz would be to depend on the files which go
# into it of course ...
#
# DISPLAY is unset for making a deb since under fakeroot gtk stuff may try
# to read config files like ~/.pangorc from root's home dir /root/.pangorc,
# and that dir will be unreadable by ordinary users (normally), provoking
# warnings and possible failures from Test::NoWarnings.
#
$(DEBFILE) deb:
	test -f $(DISTVNAME).tar.gz
	debver="`dpkg-parsechangelog -c1 | sed -n -r -e 's/^Version: (.*)-[0-9.]+$$/\1/p'`"; \
	  echo "debver $$debver", want $(VERSION); \
	  test "$$debver" = "$(VERSION)"
	rm -rf $(DISTVNAME)
	tar xfz $(DISTVNAME).tar.gz
	unset DISPLAY; export DISPLAY; \
	  cd $(DISTVNAME) \
	  && dpkg-checkbuilddeps debian/control \
	  && fakeroot debian/rules binary
	rm -rf $(DISTVNAME)

lintian-deb: $(DEBFILE)
	lintian -i -X new-package-should-close-itp-bug $(DEBFILE)
lintian-source:
	rm -rf temp-lintian; \
	mkdir temp-lintian; \
	cd temp-lintian; \
	cp ../$(DISTVNAME).tar.gz $(DEBNAME)_$(VERSION).orig.tar.gz; \
	tar xfz $(DEBNAME)_$(VERSION).orig.tar.gz; \
        echo 'empty-debian-diff' \
             >$(DISTVNAME)/debian/source.lintian-overrides; \
	mv -T $(DISTVNAME) $(DEBNAME)-$(VERSION); \
	dpkg-source -b $(DEBNAME)-$(VERSION) \
	               $(DEBNAME)_$(VERSION).orig.tar.gz; \
	lintian -i *.dsc; \
	cd ..; \
	rm -rf temp-lintian

HERE

  return $post;
}

1;
__END__
