package Math::DWT::Wavelet::DiscreteMeyer;


=head1 NAME

Math::DWT::Wavelet::DiscreteMeyer - FIR lo- & hi-pass filter coefficients for the DiscreteMeyer wavelet.

=head1 VERSION

Version 0.01_1

=cut

our $VERSION = '0.01_1';

=head1 SYNOPSIS

This module provides the lo- and hi-pass decomposition and reconstruction filter coefficients for the DiscreteMeyer wavelet.  It is meant to be used with other Math::DWT modules:

    use Math::DWT;
    use Math::DWT::UDWT;

    my $dwt = Math::DWT->new('DiscreteMeyer');
    my $udwt = Math::DWT::UDWT->new('DiscreteMeyer');


=cut

=head1 SUBROUTINES/METHODS

=head2 new(VAR)

Takes a single argument, var, which determines the particular filter set which the object will provide.  You can get a list of available filter sets with the vars() method.

    my $discretemeyer62 = Math::DWT::Wavelet::DiscreteMeyer->new(62);

This method returns a Math::DWT::Wavelet::DiscreteMeyer object;

=head2 vars()

This method returns a list of possible choices for VAR when creating a new object, e.g.:

    my @v = Math::DWT::Wavelet::DiscreteMeyer->vars();
    print scalar(@v); # 2


=head2 filters()

Depending on the context in which it is called, returns an array or an arrayref containing (lo_d, hi_d, lo_r, hi_r) - the set of filters which are defined with the instantiated object.


=head2 lo_d()

=head2 hi_d()

=head2 lo_r()

=head2 hi_r()

Returns the requested set of filter coefficients as either an array or arrayref, depending on calling context.

=head1 SEE ALSO

Math::DWT(3pm), Math::DWT::UDWT(3pm), Math::DWT::Wavelet::Haar(3pm), Math::DWT::Wavelet::Coiflet(3pm), Math::DWT::Wavelet::Symlet(3pm), Math::DWT::Wavelet::Biorthogonal(3pm), Math::DWT::Wavelet::ReverseBiorthogonal(3pm), Math::DWT::Wavelet::Daubechies(3pm), perl(1)


=head1 AUTHOR


Mike Kroh, C<< <kroh at cpan.org> >>

=head1 BUGS


Please report any bugs or feature requests to C<bug-math-dwt-wavelet-discretemeyer at rt.cpan.org>, or through the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Math-DWT-Wavelet-DiscreteMeyer>.  I will be notified, and then you'll automatically be notified of progress on your bug as I make changes.



=head1 ACKNOWLEDGEMENTS

These wavelet filter coefficients were scraped from this site: L<http://wavelets.pybytes.com/>.

=head1 LICENSE AND COPYRIGHT


Copyright 2016 Mike Kroh.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut




my @vars=qw/62 102/;

my %lo_d; my %hi_d; my %lo_r; my %hi_r;


$lo_d{62}=[qw/0.0 -1.009999956941423e-12 8.519459636796214e-09 -1.111944952595278e-08 -1.0798819539621958e-08 6.066975741351135e-08 -1.0866516536735883e-07 8.200680650386481e-08 1.1783004497663934e-07 -5.506340565252278e-07 1.1307947017916706e-06 -1.489549216497156e-06 7.367572885903746e-07 3.20544191334478e-06 -1.6312699734552807e-05 6.554305930575149e-05 -0.0006011502343516092 -0.002704672124643725 0.002202534100911002 0.006045814097323304 -0.006387718318497156 -0.011061496392513451 0.015270015130934803 0.017423434103729693 -0.03213079399021176 -0.024348745906078023 0.0637390243228016 0.030655091960824263 -0.13284520043622938 -0.035087555656258346 0.44459300275757724 0.7445855923188063 0.44459300275757724 -0.035087555656258346 -0.13284520043622938 0.030655091960824263 0.0637390243228016 -0.024348745906078023 -0.03213079399021176 0.017423434103729693 0.015270015130934803 -0.011061496392513451 -0.006387718318497156 0.006045814097323304 0.002202534100911002 -0.002704672124643725 -0.0006011502343516092 6.554305930575149e-05 -1.6312699734552807e-05 3.20544191334478e-06 7.367572885903746e-07 -1.489549216497156e-06 1.1307947017916706e-06 -5.506340565252278e-07 1.1783004497663934e-07 8.200680650386481e-08 -1.0866516536735883e-07 6.066975741351135e-08 -1.0798819539621958e-08 -1.111944952595278e-08 8.519459636796214e-09 -1.009999956941423e-12/];
$lo_r{62}=[qw/-1.009999956941423e-12 8.519459636796214e-09 -1.111944952595278e-08 -1.0798819539621958e-08 6.066975741351135e-08 -1.0866516536735883e-07 8.200680650386481e-08 1.1783004497663934e-07 -5.506340565252278e-07 1.1307947017916706e-06 -1.489549216497156e-06 7.367572885903746e-07 3.20544191334478e-06 -1.6312699734552807e-05 6.554305930575149e-05 -0.0006011502343516092 -0.002704672124643725 0.002202534100911002 0.006045814097323304 -0.006387718318497156 -0.011061496392513451 0.015270015130934803 0.017423434103729693 -0.03213079399021176 -0.024348745906078023 0.0637390243228016 0.030655091960824263 -0.13284520043622938 -0.035087555656258346 0.44459300275757724 0.7445855923188063 0.44459300275757724 -0.035087555656258346 -0.13284520043622938 0.030655091960824263 0.0637390243228016 -0.024348745906078023 -0.03213079399021176 0.017423434103729693 0.015270015130934803 -0.011061496392513451 -0.006387718318497156 0.006045814097323304 0.002202534100911002 -0.002704672124643725 -0.0006011502343516092 6.554305930575149e-05 -1.6312699734552807e-05 3.20544191334478e-06 7.367572885903746e-07 -1.489549216497156e-06 1.1307947017916706e-06 -5.506340565252278e-07 1.1783004497663934e-07 8.200680650386481e-08 -1.0866516536735883e-07 6.066975741351135e-08 -1.0798819539621958e-08 -1.111944952595278e-08 8.519459636796214e-09 -1.009999956941423e-12 0.0/];
$hi_d{62}=[qw/1.009999956941423e-12 8.519459636796214e-09 1.111944952595278e-08 -1.0798819539621958e-08 -6.066975741351135e-08 -1.0866516536735883e-07 -8.200680650386481e-08 1.1783004497663934e-07 5.506340565252278e-07 1.1307947017916706e-06 1.489549216497156e-06 7.367572885903746e-07 -3.20544191334478e-06 -1.6312699734552807e-05 -6.554305930575149e-05 -0.0006011502343516092 0.002704672124643725 0.002202534100911002 -0.006045814097323304 -0.006387718318497156 0.011061496392513451 0.015270015130934803 -0.017423434103729693 -0.03213079399021176 0.024348745906078023 0.0637390243228016 -0.030655091960824263 -0.13284520043622938 0.035087555656258346 0.44459300275757724 -0.7445855923188063 0.44459300275757724 0.035087555656258346 -0.13284520043622938 -0.030655091960824263 0.0637390243228016 0.024348745906078023 -0.03213079399021176 -0.017423434103729693 0.015270015130934803 0.011061496392513451 -0.006387718318497156 -0.006045814097323304 0.002202534100911002 0.002704672124643725 -0.0006011502343516092 -6.554305930575149e-05 -1.6312699734552807e-05 -3.20544191334478e-06 7.367572885903746e-07 1.489549216497156e-06 1.1307947017916706e-06 5.506340565252278e-07 1.1783004497663934e-07 -8.200680650386481e-08 -1.0866516536735883e-07 -6.066975741351135e-08 -1.0798819539621958e-08 1.111944952595278e-08 8.519459636796214e-09 1.009999956941423e-12 0.0/];
$hi_r{62}=[qw/0.0 1.009999956941423e-12 8.519459636796214e-09 1.111944952595278e-08 -1.0798819539621958e-08 -6.066975741351135e-08 -1.0866516536735883e-07 -8.200680650386481e-08 1.1783004497663934e-07 5.506340565252278e-07 1.1307947017916706e-06 1.489549216497156e-06 7.367572885903746e-07 -3.20544191334478e-06 -1.6312699734552807e-05 -6.554305930575149e-05 -0.0006011502343516092 0.002704672124643725 0.002202534100911002 -0.006045814097323304 -0.006387718318497156 0.011061496392513451 0.015270015130934803 -0.017423434103729693 -0.03213079399021176 0.024348745906078023 0.0637390243228016 -0.030655091960824263 -0.13284520043622938 0.035087555656258346 0.44459300275757724 -0.7445855923188063 0.44459300275757724 0.035087555656258346 -0.13284520043622938 -0.030655091960824263 0.0637390243228016 0.024348745906078023 -0.03213079399021176 -0.017423434103729693 0.015270015130934803 0.011061496392513451 -0.006387718318497156 -0.006045814097323304 0.002202534100911002 0.002704672124643725 -0.0006011502343516092 -6.554305930575149e-05 -1.6312699734552807e-05 -3.20544191334478e-06 7.367572885903746e-07 1.489549216497156e-06 1.1307947017916706e-06 5.506340565252278e-07 1.1783004497663934e-07 -8.200680650386481e-08 -1.0866516536735883e-07 -6.066975741351135e-08 -1.0798819539621958e-08 1.111944952595278e-08 8.519459636796214e-09 1.009999956941423e-12/];
;

$lo_d{102}=[qw/0
  -0.000001509740857423615368067119392459
  0.0000012787667568234988185092932191922
 0.00000044958556044886894257115545966985
 -0.0000020965688704949424373349693012658
  0.0000017232235544806816379817060794211
 0.00000069808227631073860176537599145363
 -0.0000028794080326548469234463571014837
  0.0000023831483945189297583150400683394
  0.0000009825156022293385846413460785187
  -0.000004217789186342479173499799699476
  0.0000033535015380894437405246560901517
  0.0000016747218588365072067676011732806
 -0.0000060345013418603467021238823120299
  0.0000048375558015595789058851776409664
  0.0000024022880228828376897201417328187
 -0.0000095563098456654446672233274573394
  0.0000072165276947634148549219623713924
  0.0000048490782997767487258228454038989
  -0.000014206928580564191070038058961611
   0.000010503914270783866119550291340357
  0.0000061875802981115544029451329410829
   -0.00002443800584565461033272927571236
   0.000020106387690909483331345650181099
   0.000014993523600015133676109473082061
  -0.000046428764283651689564677567112483
   0.000032341311913679686516464445444896
   0.000037409665760249840606426613431879
   -0.00010277900508488476988971926928684
   0.000024461956844602301730561347281601
    0.00014971351538925736215576711707342
  -0.000075592870255167127495517231938749
   -0.00013991314821741802128934883953093
  -0.000093512893880113803378104075925137
    0.00016118981972534630843149616730159
    0.00085950021376237750302812257174878
   -0.00057818579527344111674419524504742
    -0.0027021687339390796679661210788481
     0.0021947753364594444445978549396159
       0.00604551059645607770376418699243
    -0.0063867286185481262836272087213274
     -0.011044641900538889126015540398384
      0.015250913158585904447206438305784
      0.017403888210177406031808899911084
     -0.032094063354505306118191754194413
     -0.024321783959518776530028816296181
      0.063667300884468314148278977882001
      0.030621243943424570482791935432942
      -0.13269661535886173586717973194027
     -0.035048287390595032897433469543103
       0.44409503076652878661434442619793
       0.74375100490378698303572946315398
       0.44409503076652878661434442619793
     -0.035048287390595032897433469543103
      -0.13269661535886173586717973194027
      0.030621243943424570482791935432942
      0.063667300884468314148278977882001
     -0.024321783959518776530028816296181
     -0.032094063354505306118191754194413
      0.017403888210177406031808899911084
      0.015250913158585904447206438305784
     -0.011044641900538889126015540398384
    -0.0063867286185481262836272087213274
       0.00604551059645607770376418699243
     0.0021947753364594444445978549396159
    -0.0027021687339390796679661210788481
   -0.00057818579527344111674419524504742
    0.00085950021376237750302812257174878
    0.00016118981972534630843149616730159
  -0.000093512893880113803378104075925137
   -0.00013991314821741802128934883953093
  -0.000075592870255167127495517231938749
    0.00014971351538925736215576711707342
   0.000024461956844602301730561347281601
   -0.00010277900508488476988971926928684
   0.000037409665760249840606426613431879
   0.000032341311913679686516464445444896
  -0.000046428764283651689564677567112483
   0.000014993523600015133676109473082061
   0.000020106387690909483331345650181099
   -0.00002443800584565461033272927571236
  0.0000061875802981115544029451329410829
   0.000010503914270783866119550291340357
  -0.000014206928580564191070038058961611
  0.0000048490782997767487258228454038989
  0.0000072165276947634148549219623713924
 -0.0000095563098456654446672233274573394
  0.0000024022880228828376897201417328187
  0.0000048375558015595789058851776409664
 -0.0000060345013418603467021238823120299
  0.0000016747218588365072067676011732806
  0.0000033535015380894437405246560901517
  -0.000004217789186342479173499799699476
  0.0000009825156022293385846413460785187
  0.0000023831483945189297583150400683394
 -0.0000028794080326548469234463571014837
 0.00000069808227631073860176537599145363
  0.0000017232235544806816379817060794211
 -0.0000020965688704949424373349693012658
 0.00000044958556044886894257115545966985
  0.0000012787667568234988185092932191922
  -0.000001509740857423615368067119392459/];

$hi_d{102}=[qw/0.000001509740857423615368067119392459
   0.0000012787667568234988185092932191922
 -0.00000044958556044886894257115545966985
  -0.0000020965688704949424373349693012658
  -0.0000017232235544806816379817060794211
  0.00000069808227631073860176537599145363
   0.0000028794080326548469234463571014837
   0.0000023831483945189297583150400683394
  -0.0000009825156022293385846413460785187
   -0.000004217789186342479173499799699476
  -0.0000033535015380894437405246560901517
   0.0000016747218588365072067676011732806
   0.0000060345013418603467021238823120299
   0.0000048375558015595789058851776409664
  -0.0000024022880228828376897201417328187
  -0.0000095563098456654446672233274573394
  -0.0000072165276947634148549219623713924
   0.0000048490782997767487258228454038989
    0.000014206928580564191070038058961611
    0.000010503914270783866119550291340357
  -0.0000061875802981115544029451329410829
    -0.00002443800584565461033272927571236
   -0.000020106387690909483331345650181099
    0.000014993523600015133676109473082061
    0.000046428764283651689564677567112483
    0.000032341311913679686516464445444896
   -0.000037409665760249840606426613431879
    -0.00010277900508488476988971926928684
   -0.000024461956844602301730561347281601
     0.00014971351538925736215576711707342
    0.000075592870255167127495517231938749
    -0.00013991314821741802128934883953093
    0.000093512893880113803378104075925137
     0.00016118981972534630843149616730159
    -0.00085950021376237750302812257174878
    -0.00057818579527344111674419524504742
      0.0027021687339390796679661210788481
      0.0021947753364594444445978549396159
       -0.00604551059645607770376418699243
     -0.0063867286185481262836272087213274
       0.011044641900538889126015540398384
       0.015250913158585904447206438305784
      -0.017403888210177406031808899911084
      -0.032094063354505306118191754194413
       0.024321783959518776530028816296181
       0.063667300884468314148278977882001
      -0.030621243943424570482791935432942
       -0.13269661535886173586717973194027
       0.035048287390595032897433469543103
        0.44409503076652878661434442619793
       -0.74375100490378698303572946315398
        0.44409503076652878661434442619793
       0.035048287390595032897433469543103
       -0.13269661535886173586717973194027
      -0.030621243943424570482791935432942
       0.063667300884468314148278977882001
       0.024321783959518776530028816296181
      -0.032094063354505306118191754194413
      -0.017403888210177406031808899911084
       0.015250913158585904447206438305784
       0.011044641900538889126015540398384
     -0.0063867286185481262836272087213274
       -0.00604551059645607770376418699243
      0.0021947753364594444445978549396159
      0.0027021687339390796679661210788481
    -0.00057818579527344111674419524504742
    -0.00085950021376237750302812257174878
     0.00016118981972534630843149616730159
    0.000093512893880113803378104075925137
    -0.00013991314821741802128934883953093
    0.000075592870255167127495517231938749
     0.00014971351538925736215576711707342
   -0.000024461956844602301730561347281601
    -0.00010277900508488476988971926928684
   -0.000037409665760249840606426613431879
    0.000032341311913679686516464445444896
    0.000046428764283651689564677567112483
    0.000014993523600015133676109473082061
   -0.000020106387690909483331345650181099
    -0.00002443800584565461033272927571236
  -0.0000061875802981115544029451329410829
    0.000010503914270783866119550291340357
    0.000014206928580564191070038058961611
   0.0000048490782997767487258228454038989
  -0.0000072165276947634148549219623713924
  -0.0000095563098456654446672233274573394
  -0.0000024022880228828376897201417328187
   0.0000048375558015595789058851776409664
   0.0000060345013418603467021238823120299
   0.0000016747218588365072067676011732806
  -0.0000033535015380894437405246560901517
   -0.000004217789186342479173499799699476
  -0.0000009825156022293385846413460785187
   0.0000023831483945189297583150400683394
   0.0000028794080326548469234463571014837
  0.00000069808227631073860176537599145363
  -0.0000017232235544806816379817060794211
  -0.0000020965688704949424373349693012658
 -0.00000044958556044886894257115545966985
   0.0000012787667568234988185092932191922
    0.000001509740857423615368067119392459
                                         0/];

$lo_r{102}=[qw/-0.000001509740857423615368067119392459
  0.0000012787667568234988185092932191922
 0.00000044958556044886894257115545966985
 -0.0000020965688704949424373349693012658
  0.0000017232235544806816379817060794211
 0.00000069808227631073860176537599145363
 -0.0000028794080326548469234463571014837
  0.0000023831483945189297583150400683394
  0.0000009825156022293385846413460785187
  -0.000004217789186342479173499799699476
  0.0000033535015380894437405246560901517
  0.0000016747218588365072067676011732806
 -0.0000060345013418603467021238823120299
  0.0000048375558015595789058851776409664
  0.0000024022880228828376897201417328187
 -0.0000095563098456654446672233274573394
  0.0000072165276947634148549219623713924
  0.0000048490782997767487258228454038989
  -0.000014206928580564191070038058961611
   0.000010503914270783866119550291340357
  0.0000061875802981115544029451329410829
   -0.00002443800584565461033272927571236
   0.000020106387690909483331345650181099
   0.000014993523600015133676109473082061
  -0.000046428764283651689564677567112483
   0.000032341311913679686516464445444896
   0.000037409665760249840606426613431879
   -0.00010277900508488476988971926928684
   0.000024461956844602301730561347281601
    0.00014971351538925736215576711707342
  -0.000075592870255167127495517231938749
   -0.00013991314821741802128934883953093
  -0.000093512893880113803378104075925137
    0.00016118981972534630843149616730159
    0.00085950021376237750302812257174878
   -0.00057818579527344111674419524504742
    -0.0027021687339390796679661210788481
     0.0021947753364594444445978549396159
       0.00604551059645607770376418699243
    -0.0063867286185481262836272087213274
     -0.011044641900538889126015540398384
      0.015250913158585904447206438305784
      0.017403888210177406031808899911084
     -0.032094063354505306118191754194413
     -0.024321783959518776530028816296181
      0.063667300884468314148278977882001
      0.030621243943424570482791935432942
      -0.13269661535886173586717973194027
     -0.035048287390595032897433469543103
       0.44409503076652878661434442619793
       0.74375100490378698303572946315398
       0.44409503076652878661434442619793
     -0.035048287390595032897433469543103
      -0.13269661535886173586717973194027
      0.030621243943424570482791935432942
      0.063667300884468314148278977882001
     -0.024321783959518776530028816296181
     -0.032094063354505306118191754194413
      0.017403888210177406031808899911084
      0.015250913158585904447206438305784
     -0.011044641900538889126015540398384
    -0.0063867286185481262836272087213274
       0.00604551059645607770376418699243
     0.0021947753364594444445978549396159
    -0.0027021687339390796679661210788481
   -0.00057818579527344111674419524504742
    0.00085950021376237750302812257174878
    0.00016118981972534630843149616730159
  -0.000093512893880113803378104075925137
   -0.00013991314821741802128934883953093
  -0.000075592870255167127495517231938749
    0.00014971351538925736215576711707342
   0.000024461956844602301730561347281601
   -0.00010277900508488476988971926928684
   0.000037409665760249840606426613431879
   0.000032341311913679686516464445444896
  -0.000046428764283651689564677567112483
   0.000014993523600015133676109473082061
   0.000020106387690909483331345650181099
   -0.00002443800584565461033272927571236
  0.0000061875802981115544029451329410829
   0.000010503914270783866119550291340357
  -0.000014206928580564191070038058961611
  0.0000048490782997767487258228454038989
  0.0000072165276947634148549219623713924
 -0.0000095563098456654446672233274573394
  0.0000024022880228828376897201417328187
  0.0000048375558015595789058851776409664
 -0.0000060345013418603467021238823120299
  0.0000016747218588365072067676011732806
  0.0000033535015380894437405246560901517
  -0.000004217789186342479173499799699476
  0.0000009825156022293385846413460785187
  0.0000023831483945189297583150400683394
 -0.0000028794080326548469234463571014837
 0.00000069808227631073860176537599145363
  0.0000017232235544806816379817060794211
 -0.0000020965688704949424373349693012658
 0.00000044958556044886894257115545966985
  0.0000012787667568234988185092932191922
  -0.000001509740857423615368067119392459
                                        0/];

$hi_r{102}=[qw/0
    0.000001509740857423615368067119392459
   0.0000012787667568234988185092932191922
 -0.00000044958556044886894257115545966985
  -0.0000020965688704949424373349693012658
  -0.0000017232235544806816379817060794211
  0.00000069808227631073860176537599145363
   0.0000028794080326548469234463571014837
   0.0000023831483945189297583150400683394
  -0.0000009825156022293385846413460785187
   -0.000004217789186342479173499799699476
  -0.0000033535015380894437405246560901517
   0.0000016747218588365072067676011732806
   0.0000060345013418603467021238823120299
   0.0000048375558015595789058851776409664
  -0.0000024022880228828376897201417328187
  -0.0000095563098456654446672233274573394
  -0.0000072165276947634148549219623713924
   0.0000048490782997767487258228454038989
    0.000014206928580564191070038058961611
    0.000010503914270783866119550291340357
  -0.0000061875802981115544029451329410829
    -0.00002443800584565461033272927571236
   -0.000020106387690909483331345650181099
    0.000014993523600015133676109473082061
    0.000046428764283651689564677567112483
    0.000032341311913679686516464445444896
   -0.000037409665760249840606426613431879
    -0.00010277900508488476988971926928684
   -0.000024461956844602301730561347281601
     0.00014971351538925736215576711707342
    0.000075592870255167127495517231938749
    -0.00013991314821741802128934883953093
    0.000093512893880113803378104075925137
     0.00016118981972534630843149616730159
    -0.00085950021376237750302812257174878
    -0.00057818579527344111674419524504742
      0.0027021687339390796679661210788481
      0.0021947753364594444445978549396159
       -0.00604551059645607770376418699243
     -0.0063867286185481262836272087213274
       0.011044641900538889126015540398384
       0.015250913158585904447206438305784
      -0.017403888210177406031808899911084
      -0.032094063354505306118191754194413
       0.024321783959518776530028816296181
       0.063667300884468314148278977882001
      -0.030621243943424570482791935432942
       -0.13269661535886173586717973194027
       0.035048287390595032897433469543103
        0.44409503076652878661434442619793
       -0.74375100490378698303572946315398
        0.44409503076652878661434442619793
       0.035048287390595032897433469543103
       -0.13269661535886173586717973194027
      -0.030621243943424570482791935432942
       0.063667300884468314148278977882001
       0.024321783959518776530028816296181
      -0.032094063354505306118191754194413
      -0.017403888210177406031808899911084
       0.015250913158585904447206438305784
       0.011044641900538889126015540398384
     -0.0063867286185481262836272087213274
       -0.00604551059645607770376418699243
      0.0021947753364594444445978549396159
      0.0027021687339390796679661210788481
    -0.00057818579527344111674419524504742
    -0.00085950021376237750302812257174878
     0.00016118981972534630843149616730159
    0.000093512893880113803378104075925137
    -0.00013991314821741802128934883953093
    0.000075592870255167127495517231938749
     0.00014971351538925736215576711707342
   -0.000024461956844602301730561347281601
    -0.00010277900508488476988971926928684
   -0.000037409665760249840606426613431879
    0.000032341311913679686516464445444896
    0.000046428764283651689564677567112483
    0.000014993523600015133676109473082061
   -0.000020106387690909483331345650181099
    -0.00002443800584565461033272927571236
  -0.0000061875802981115544029451329410829
    0.000010503914270783866119550291340357
    0.000014206928580564191070038058961611
   0.0000048490782997767487258228454038989
  -0.0000072165276947634148549219623713924
  -0.0000095563098456654446672233274573394
  -0.0000024022880228828376897201417328187
   0.0000048375558015595789058851776409664
   0.0000060345013418603467021238823120299
   0.0000016747218588365072067676011732806
  -0.0000033535015380894437405246560901517
   -0.000004217789186342479173499799699476
  -0.0000009825156022293385846413460785187
   0.0000023831483945189297583150400683394
   0.0000028794080326548469234463571014837
  0.00000069808227631073860176537599145363
  -0.0000017232235544806816379817060794211
  -0.0000020965688704949424373349693012658
 -0.00000044958556044886894257115545966985
   0.0000012787667568234988185092932191922
    0.000001509740857423615368067119392459/];

sub new {
	my $class=shift;
	my $self={};
	my $var=shift || $vars[0];

	$self={lo_d=>$lo_d{$var},
		hi_d=>$hi_d{$var},
		lo_r=>$lo_r{$var},
		hi_r=>$hi_r{$var}
		};
	
	bless $self,$class;
	return $self;
};

sub vars {
	my $self=shift;
	if (wantarray()) {
		return (@vars);
	};
	return \@vars;
};

sub filters {
	my $self=shift;
	my $lo_d=$self->lo_d;
	my $hi_d=$self->hi_d;
	my $lo_r=$self->lo_r;
	my $hi_r=$self->hi_r;
	my @a=( $lo_d,$hi_d,$lo_r,$hi_r);
	if (wantarray()) {
		return (@a);
	};
	return \@a;
};

sub lo_d {
	my $self=shift;
	my $a=$self->{lo_d};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub hi_d {
	my $self=shift;
	my $a=$self->{hi_d};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub lo_r {
	my $self=shift;
	my $a=$self->{lo_r};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	
sub hi_r {
	my $self=shift;
	my $a=$self->{hi_r};
	if (wantarray()) {
		return (@{$a});
	};
	return $a;
};	

		
1;
