## $Id: Formatting.pm,v 1.1 2002/04/01 22:13:10 allane Exp $
##
## Associates package: Routines to handle Amazon.com XML interface
##                     for Amazon Associates.
## Formatting:         Documents visual formatting
##
## Copyright  2002 Allan Engelhardt <allane@cybaea.com>
##
## Amazon, Amazon.com, and other names used may be trademarks of Amazon
## in the United States of America and other jurisdictions.
##
## ************************************************************************
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
## ************************************************************************
##

package Business::Associates::Formatting;

use strict;

require Exporter;

BEGIN {
    use Exporter();
    our ($VERSION, @ISA, @EXPORT, @EXPORT_OK, %EXPORT_TAGS);
    $VERSION = do { my @r = (q$Revision: 1.1 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r }; # must be all one line, for MakeMaker
    @ISA = qw(Exporter);
    @EXPORT = qw();
    @EXPORT_OK = qw();
    %EXPORT_TAGS = (
                    );
}


1;

__END__


=head1 NAME

Business::Associates::Formatting - Information on visual formatting for the Associates package

=head1 INTRODUCTION

This document is all about visual formatting, i.e. Cascading Style
Sheets.  We will assume that you are familiar with
L<Business::Associates>.

=head2 Overview of HTML structure

The output generated by the standard Associates routines and data
transformation style sheets is fundamentally a HTML table with the
ads.  The table contains up to three rows with the top image in the
first row, the left image, another table with the actual ads, and the
right image in the second row, and the bottom image in the last row:

  +----------------------------------------------+
  |                 Top    image                 |
  +--------------+----------------+--------------+
  | Left   image | Table with ads | Right  image |
  +--------------+----------------+--------------+
  |                 Bottom image                 |
  +----------------------------------------------+

Of course, any of the image may be missing, depending on the wishes of
the web designer.

Two divisions help to navigate us more precisely.  The whole structure
is enclosed in a C<div> entity with the fixed, named class name
I<Associates>.  This ensures that our cascading style sheet (CSS) does
not interfere with your site style: all CSS rules begin with
C<div.Associates>.  This division is then immediately followed by
another C<div> entity with a variable class name that is the product
category (e.g. "Books" or "DVDs").  This means that we can easily have
different rules for different product types; useful, for example, to
select the fields that are displayed in the ad.

These two divisions are then followed by the actual table, as outlined
in the figure above.  The class for this table is variable and can be
defined by the web designer, but it defaults to I<A_catalog>.

The structure, then, is logically this:

  <div class="Associates">
   <div class="Books">
    <table class="A_catalog">
     <!- ... -->
    </table>
   </div>
  </div>

where, as we have mentioned, "Books" is replaced by the product
category (or one of the product categories, in multi-category
searches) and "A_catalog" is the default, but can be changed by the we
designer (for example, by the C<class> parameter to the supplied
F<a.pl> CGI script).

This allows us to (a) define styles for the entire Associates module,

    div.Associates {
        page-break-inside:      avoid;
        margin-top:             0.5ex;
        margin-bottom:          0.5ex;
    }

(b) define per-product-type styles,

    div.Associates > div.DVDs > table
    {
        border-width:           medium;
        border-style:           ridge;
    }

and (c), by using the class parameter to set the class of the C<table>
entity, we can define styles for ad hoc advert styles:

    div.Associates table.right_float
    {
        float:                  right;
    }

The four images are in a table data (C<td>) entity with the class name
I<A__image_(left|right|top|bottom)>.  Each image is a link (C<a>
entity) of class I<Associate_link>; the image itself (C<img>) does not
need a class as it is the unique C<img> child of C<a.Associate_link>.

=head2 Inner Table

The inner table is, as shown on the figure above, the table that
contains the actual ads.  Each ad consists of a product image (known
as the cover image because I was thinking books when I developed this)
and the ad text; each of these two items is a separate table data
element (C<td> entity).

The inner table is of class I<A__data_table> (which is also the class
of the C<td> entity in which the inner table resides).  The table data
elements for the product image and the ad text information are of
class I<A__cover_image> and I<A__text>, respectively.

Within the text, each data item from the original XML is presented in
a separate C<div> entity with a class name that is the concatenation
of I<A_> and the name of the data item (i.e. the original XML entity
name).  For example, the author is available in I<A_author> and the
title in I<A_title>.  This allows us to easily select which items to
display

    div.Associates table.A__data_table td.A__text div.A_ranking {
        display:                none;
    }

and how to display them:

    /* In the following section we make the title follow the author
       on a single line with a colon to separate them. */
    
    div.Associates table.A__data_table td.A__text div.A_author,
    div.Associates table.A__data_table td.A__text div.A_director
    {
        display:                inline;
    }
    div.Associates table.A__data_table td.A__text div.A_director:after {
        content:                ": ";
    }
    
    div.Associates table.A__data_table td.A__text div.A_title {
        display:                inline;
    }

And that's really all there is to it.  Please see the file
F<demo/Associates.css> for more examples.

=head2 Summary

This is the structure of the output.  Variable class names are
surrounded by stars.

  <div class="Associates">
   <div class="*Product Category*">
    <table class="*User-defined, default is A_catalog*">
     <tr>
      <td class="A__image_top">
       <a class="Associate_link">
        <img />
       </a>
      </td>
     </tr>
     <tr>
      <td class="A__image_left">
       <a class="Associate_link">
        <img />
       </a>
      </td>
      <td class="A__data_table">
       <table class="A__data_table">
        <tr>
         <td class="A__cover_image">
          <a></a>
         </td>
         <td class="A__text">
          <div class="A_ranking"></div>
          <div class="A_author"></div>
          <div class="A_director"></div>
          <div class="A_title"><a></a></div>
          <div class="A_release_date"></div>
          <div class="A_binding"></div>
          <div class="A_availability"></div>
          <div class="A_asin"></div>
         </td>
         <td class="A__cover_image">
          ...
         </td>
         <td class="A__text">
          ...
         </td>
         ...
        </tr>
        <tr>
         <td class="A__cover_image">
          ...
         </td>
         <td class="A__text">
          ...
         </td>
         ...
        </tr>
        ...
       </table>
      <td class="A__image_right">
       <a class="Associate_link">
        <img />
       </a>
      </td>
     </tr>
     <tr>
      <td class="A__image_top">
       <a class="Associate_link">
        <img />
       </a>
      </td>
     </tr>
    </table>
   </div>
  </div>


=head1 COMBINING STYLE SHEETS

You may want to combine the supplied cascading style sheet
(F<demo/Associates.css>) with your site-wide cascading style sheet.
This is easiest to do with the C<@import> directive which must appear
in your site-wide style sheet before any formatting instructions.  For
example, my style sheet starts with the three lines:

    @charset "ISO-8859-1";
    /* $Id: style.css,v 1.10 2002/02/24 13:35:57 allane Exp $ */
    @import url("/Associates.css");

which works because I copied F<demo/Associates.css> from the
distribution to the document root of my web server.

=head1 SEE ALSO

See also L<Business::Associates> for a general overview and
L<Business::Associates::Stylesheets> for details of the data
processing.

=head1 FILES

See the file F<demo/Associates.css> in the distribution for more
examples.


=head1 TRADEMARKS

Amazon, Amazon.com, and other names used in this document may be the
registered trademark of Amazon in the United States of America and
other jurisdictions.  Please see http://www.amazon.com/ for more
information.

Other names may be trademarks of their respective owners.

=head1 AUTHOR

This package was written by Allan Engelhardt E<lt>allane@cybaea.comE<gt>

=head1 COPYING

Copyright (c) 2002 Allan Engelhardt E<lt>allane@cybaea.comE<gt>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
USA



=cut
