package Yandex::Translate;

use strict;
use warnings;
use HTTP::Tiny;
use JSON;

$Yandex::Translate::VERSION = '0.01';

=head1 NAME

Yandex::Translate - Perl extension for using Yandex API

=head1 SYNOPSIS
  #Calling the module
  use Yandex::Translate; #Calling the module
  
  #Create an instance
  my $tr = Yandex::Translate->new;

  #Set API key, you can get from https://translate.yandex.com/developers/keys
  $tr->set_key('your_api_key');
  
  #Select the lang that you want to translate from
  $tr->set_from_lang("en");
  
  #Select the lang that you want  to translate to
  $tr->set_to_lang("ru");
   
  #Set the text that you want to translate
  $tr->set_to_lang("Hello Perl");
  
  #Get the result
  print $tr->translate();

=head1 DESCRIPTION

Yandex.Translate (previously Yandex.Translation) is a web service 
provided by Yandex intended for the translation of text or web pages 
into another language.

The service uses a self-learning statistical machine translation,
developed by Yandex.
The system constructs the dictionary of correspondences based 
on the analysis of millions of translated texts.

=head1 SEE ALSO
For more info please visit Yandex https://translate.yandex.com/developers


=head1 AUTHOR

James Axl, E<lt>axlrose112@gmail.comE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2017 by James Axl

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.24.1 or,
at your option, any later version of Perl 5 you may have available.


=cut

sub new
{
    my $class = shift;
    my $self = {
	_key_        => shift,
        _text_       => shift,
        _from_lang_  => shift,
        _to_lang_    => shift,
	_base_       => "https://translate.yandex.net/api/v1.5/tr.json",
	_post_       => undef
    };

    bless $self, $class;
    return $self
}

sub set_key
{
    my ( $self, $key ) = @_;
    $self->{_key_} = $key if defined($key);
}

sub get_langs_list
{
    my $self = shift;
    my $query = '/getLangs?';
    $self->{_post_} = 'ui=en&key='.$self->{_key_};
    my $response = HTTP::Tiny->new->get($self->{_base_} . $query  . $self->{_post_});
    die "Invalid Api\n" if $response->{status} eq '403';
    die "Blocked Api\n" if $response->{status} eq '402';
    die "Failed!\n" unless $response->{success};
    my $result = undef;
    if (length $response->{content}) {
        my $json_respond =  JSON->new->utf8->decode($response->{content});
	my $index = 0;
	while ($index < 10) {
	    $result .= ' | '.$json_respond->{dirs}[$index];
	    $index++;
	}
    }

    return $result.'... for more info visit https://translate.yandex.com\n' if defined($result);
}

sub set_text 
{
    my ( $self, $text ) = @_;
    $self->{_text_} = $text if defined($text);
}

sub set_from_lang
{
    my ($self, $from_lang) = @_;
    $self->{_from_lang_} = $from_lang if defined($from_lang);
}

sub set_to_lang
{
    my ($self, $to_lang) = @_;
    $self->{_to_lang_} = $to_lang if defined($to_lang);
}

sub detect_lang
{
    my $self = shift;
    my $query = '/detect?';
    $self->{_post_} = 'text='.$self->{_text_}.'&key='.$self->{_key_};
    my $response = HTTP::Tiny->new->get($self->{_base_} . $query  . $self->{_post_});
    
    die "Failed!\n" unless $response->{success};
   
    if (length $response->{content}) {
        my $json_respond =  JSON->new->utf8->decode($response->{content});
	return $json_respond->{lang}."\n";
    }
}

sub translate
{
    my $self = shift;
    my $query = '/translate?';
    $self->{_text_} =~ s/ /+/ig;
    $self->{_post_} = 'lang='.$self->{_from_lang_}.'-'.$self->{_to_lang_}.'&key='.$self->{_key_}.'&text='.$self->{_text_};
    my $response = HTTP::Tiny->new->get($self->{_base_} . $query  . $self->{_post_});
    die "Invalid Api\n" if $response->{status} eq '401';
    die "Invalid Api\n" if $response->{status} eq '403';
    die "Exceeded the daily limit on the amount of translated text\n" if $response->{status} eq '403';
    die "Exceeded the maximum text size\n" if $response->{status} eq '413';
    die "The text cannot be translated\n" if $response->{status} eq '422';
    die "The specified translation direction is not supported\n" if $response->{status} eq '501';
    die "Failed!\n" unless $response->{success};

    if (length $response->{content}) {
        my $json_respond =  JSON->new->utf8->decode($response->{content});
        return $json_respond->{text}[0]."\n";
    }
}

1;
__END__

