#!/usr/bin/perl

use Test::More 'no_plan';

use Crypt::Rijndael;

while( $hash = get_group() )
	{
	print Dumper( $hash ); use Data::Dumper;
	
	my $cr = Crypt::Rijndael->new( $hash->{key},  Crypt::Rijndael::MODE_CBC() );
	
	$cr->set_iv( pack 'H*', $hash->{iv} );
	
	$plain = unpack 'H*', $cr->decrypt( $hash->{ct} );
	
	is( $plain, $hash->{pt}, "Iteration $$hash{i} works" );
	
	last;
	}
	

sub get_group
	{
	local $/ = "\n\n";
		
	<DATA> =~ m/
		I=(\d+)
			\s+
		KEY=(\S+)
			\s+
		IV=(\S+)
			\s+
		CT=(\S+)
			\s+
		PT=(\S+)
		
	/gcx;

	my( $i, $key, $iv, $ct, $pt ) = ( $1, $2, $3, $4, $5 );
	return defined $i ? { 
	 	iteration => $i,
	 	key       => $key,
	 	iv        => $iv,
	 	ct        => $ct,
	 	pt        => $pt,
	 	} : ();
	}

__DATA__
I=0
KEY=00000000000000000000000000000000
IV=00000000000000000000000000000000
CT=00000000000000000000000000000000
PT=FACA37E0B0C85373DF706E73F7C9AF86

I=1
KEY=FACA37E0B0C85373DF706E73F7C9AF86
IV=52D0C29FF8793A519BD6A8289FC80E6A
CT=FACA37E0B0C85373DF706E73F7C9AF86
PT=F5372F9735C5685F1DA362AF6ECB2940

I=2
KEY=0FFD1877850D3B2CC2D30CDC990286C6
IV=DD74BB1AC6F0F866C7992C61F59D5594
CT=F5372F9735C5685F1DA362AF6ECB2940
PT=5496A4C29C7670F61B5D5DF6181F5947

I=3
KEY=5B6BBCB5197B4BDAD98E512A811DDF81
IV=3795C88134F7C011433397D1443FEB3A
CT=5496A4C29C7670F61B5D5DF6181F5947
PT=940CC5A2AF4F1F8D1862B47BCF63E4CA

I=4
KEY=CF677917B6345457C1ECE5514E7E3B4B
IV=0FC1413900225B47AF9E139E1650EA23
CT=940CC5A2AF4F1F8D1862B47BCF63E4CA
PT=08832415D97820DE305A58A9AD111A9E

I=5
KEY=C7E45D026F4C7489F1B6BDF8E36F21D5
IV=3032F64674FA243D61DF8C16313D81F3
CT=08832415D97820DE305A58A9AD111A9E
PT=BD4089775FD1BDB0A6C4F36D1DDAA93E

I=6
KEY=7AA4D475309DC93957724E95FEB588EB
IV=47D7F440B43888F173AE5A492F1A99F8
CT=BD4089775FD1BDB0A6C4F36D1DDAA93E
PT=AB9955F74612859267D61FEA85A75ADC

I=7
KEY=D13D8182768F4CAB30A4517F7B12D237
IV=6D9E57242195D27771D1C7A748378AB0
CT=AB9955F74612859267D61FEA85A75ADC
PT=168F213FB678D8A46D0E55EFD1C49BB3

I=8
KEY=C7B2A0BDC0F7940F5DAA0490AAD64984
IV=1B98968FB5ABD95930525C1257DEB364
CT=168F213FB678D8A46D0E55EFD1C49BB3
PT=3C4CEECFF9560501C5D8C901D87C8E7B

I=9
KEY=FBFE4E7239A1910E9872CD9172AAC7FF
IV=35EB7B6D3F63AAE927C5346C9C2CB91D
CT=3C4CEECFF9560501C5D8C901D87C8E7B
PT=9D206BE0CC60296BF3A486A8C69778EF

I=10
KEY=66DE2592F5C1B8656BD64B39B43DBF10
IV=226646B36D8E9B9279F94378CCBF8FBA
CT=9D206BE0CC60296BF3A486A8C69778EF
PT=9DA7B8094BD0F0BEA26848C84B8E083C

I=11
KEY=FB799D9BBE1148DBC9BE03F1FFB3B72C
IV=3E3A2B0E1C70A19BCD7222708AA94F26
CT=9DA7B8094BD0F0BEA26848C84B8E083C
PT=6A93B2F318C14C2D3E58455B86A4F5D7

I=12
KEY=91EA2F68A6D004F6F7E646AA791742FB
IV=1F256942B3CD691858A20664BB24545E
CT=6A93B2F318C14C2D3E58455B86A4F5D7
PT=4A19AC6C6FEB654D3675A2BE32E5A820

I=13
KEY=DBF38304C93B61BBC193E4144BF2EADB
IV=631F2EDE05C1C950BCE67DF53BA53ED6
CT=4A19AC6C6FEB654D3675A2BE32E5A820
PT=799EAB8E9C864A13490FD6B089F764C5

I=14
KEY=A26D288A55BD2BA8889C32A4C2058E1E
IV=F6AD064E0DBF4AF4AA5165E1FE0E86FA
CT=799EAB8E9C864A13490FD6B089F764C5
PT=2FA3F38CA1216C6D911C1779F333413E

I=15
KEY=8DCEDB06F49C47C5198025DD3136CF20
IV=DA039AC5F4EDCD84472B23CFA1252EC3
CT=2FA3F38CA1216C6D911C1779F333413E
PT=2FC703AD6A7DAF8B0BDDD8586C1D67D0

I=16
KEY=A209D8AB9EE1E84E125DFD855D2BA8F0
IV=341B5FDA8AB434461A7B7A87EF0C530C
CT=2FC703AD6A7DAF8B0BDDD8586C1D67D0
PT=54723CABFBE32DEEDFEBF4292A5710FA

I=17
KEY=F67BE4006502C5A0CDB609AC777CB80A
IV=2C42AD577B0BE057555C3FCB6FC0089E
CT=54723CABFBE32DEEDFEBF4292A5710FA
PT=6B1E2A8C1A9CBB95D1B34AAFE15A78D3

I=18
KEY=9D65CE8C7F9E7E351C0543039626C0D9
IV=E5E06583A834BF16118ED5FA401A2D6F
CT=6B1E2A8C1A9CBB95D1B34AAFE15A78D3
PT=F4E57BFAB65375B3AD49CD78814F0C16

I=19
KEY=6980B576C9CD0B86B14C8E7B1769CCCF
IV=71E59389FF6F992225B32960593F9362
CT=F4E57BFAB65375B3AD49CD78814F0C16
PT=90A31B5934FCF6822526C973114BD953

I=20
KEY=F923AE2FFD31FD04946A47080622159C
IV=2A4B31492B9BAEC402C255978BB96D40
CT=90A31B5934FCF6822526C973114BD953
PT=61F88BA8D438A0FD23F0101465C7FD4E

I=21
KEY=98DB258729095DF9B79A571C63E5E8D2
IV=607640CEA12DCF7F188B2B65C7F4153D
CT=61F88BA8D438A0FD23F0101465C7FD4E
PT=F62D206DF53F41CA898F3A8EB6A843FF

I=22
KEY=6EF605EADC361C333E156D92D54DAB2D
IV=D69C32E12303361B465BE9A484BCA723
CT=F62D206DF53F41CA898F3A8EB6A843FF
PT=364BC3F9FDA5FB1AEEA7691550A58320

I=23
KEY=58BDC6132193E729D0B2048785E8280D
IV=C207B8347B8DB8BC6845827BB6986B0E
CT=364BC3F9FDA5FB1AEEA7691550A58320
PT=DAA6386A073CD1653E9A9DB0A97970BA

I=24
KEY=821BFE7926AF364CEE2899372C9158B7
IV=6C5D2B186393280253D6922A8421C3A9
CT=DAA6386A073CD1653E9A9DB0A97970BA
PT=AAF74F0E0A0141A811853AE8B120BC00

I=25
KEY=28ECB1772CAE77E4FFADA3DF9DB1E4B7
IV=3CB62EF0E649027DDDB76206D0F4D355
CT=AAF74F0E0A0141A811853AE8B120BC00
PT=22A0F9611447C1A85F397DD79E3206F9

I=26
KEY=0A4C481638E9B64CA094DE080383E24E
IV=C4320FA90686AE0F9E979CBF1391F6D6
CT=22A0F9611447C1A85F397DD79E3206F9
PT=1871ABDEAA7E057E76D41ED9191FAC39

I=27
KEY=123DE3C89297B332D640C0D11A9C4E77
IV=FBFD18AADF7C5E8E96EB976A535BF312
CT=1871ABDEAA7E057E76D41ED9191FAC39
PT=BF52994F41B1DEF78C6A057642C52D89

I=28
KEY=AD6F7A87D3266DC55A2AC5A7585963FE
IV=C195141578ADF4C92113C8A9A63FF072
CT=BF52994F41B1DEF78C6A057642C52D89
PT=3EAFB00E3B4A0E6DF74743A7EDAD0FE4

I=29
KEY=93C0CA89E86C63A8AD6D8600B5F46C1A
IV=D1C4E98C4A9AC79B7CCBC052ACF78913
CT=3EAFB00E3B4A0E6DF74743A7EDAD0FE4
PT=4393544AB2387A93496D683C2B8970C7

I=30
KEY=D0539EC35A54193BE400EE3C9E7D1CDD
IV=80517C6341DAF4DCBCCAE452F3040147
CT=4393544AB2387A93496D683C2B8970C7
PT=51DAC594293708A918E9EFC04F7F2F1C

I=31
KEY=81895B5773631192FCE901FCD10233C1
IV=69DE50406B4E3E8A94C1367E1E3D5464
CT=51DAC594293708A918E9EFC04F7F2F1C
PT=035B64F06C8166000C82BF15CC88446D

I=32
KEY=82D23FA71FE27792F06BBEE91D8A77AC
IV=8EA2E38145BBD79881B7EB31B8818A21
CT=035B64F06C8166000C82BF15CC88446D
PT=F3EFFA770A21741551F309E1967CC2E4

I=33
KEY=713DC5D015C30387A198B7088BF6B548
IV=0DCAA309668E11509828152553FF2306
CT=F3EFFA770A21741551F309E1967CC2E4
PT=62BA0548E7AD136BEB4E38AECD7942D9

I=34
KEY=1387C098F26E10EC4AD68FA6468FF791
IV=D031FCA5F3C88926CC3BDDC448537B69
CT=62BA0548E7AD136BEB4E38AECD7942D9
PT=38E3D4C7AF0C3E494F1D580AC5430634

I=35
KEY=2B64145F5D622EA505CBD7AC83CCF1A5
IV=6FFA5A45EDA60294B0224EEB8C36C455
CT=38E3D4C7AF0C3E494F1D580AC5430634
PT=2231EB6A6588CBCB56B930B7F1CF9E04

I=36
KEY=0955FF3538EAE56E5372E71B72036FA1
IV=B5375674631ECC3334A318993CB81F9A
CT=2231EB6A6588CBCB56B930B7F1CF9E04
PT=59692274C130D6820287EA10F7E64602

I=37
KEY=503CDD41F9DA33EC51F50D0B85E529A3
IV=A93D6E028B5C2BD17A29C6CA1617A0A2
CT=59692274C130D6820287EA10F7E64602
PT=5975A95327DE7749EA8DEDC580E5E8C7

I=38
KEY=09497412DE0444A5BB78E0CE0500C164
IV=26E00CB161145FCD00C39BC4FB1E5CBB
CT=5975A95327DE7749EA8DEDC580E5E8C7
PT=D1F7E36BC8FF4439820CB617E745A9CC

I=39
KEY=D8BE977916FB009C397456D9E24568A8
IV=72E78D9E15C8FBA864C6C40CEBB190B8
CT=D1F7E36BC8FF4439820CB617E745A9CC
PT=76FB4F20C902E9AA89854440631092B5

I=40
KEY=AE45D859DFF9E936B0F112998155FA1D
IV=4B6C585FEF52490EA62BAAD08FAA3DCF
CT=76FB4F20C902E9AA89854440631092B5
PT=6D7D4487DCD624B2FC5002DFE7D2D347

I=41
KEY=C3389CDE032FCD844CA110466687295A
IV=E94FD23A598DDE077CD1DAEE97FDADD9
CT=6D7D4487DCD624B2FC5002DFE7D2D347
PT=7297F8F85B21E1A5CB902DCC5F6738F9

I=42
KEY=B1AF6426580E2C2187313D8A39E011A3
IV=86E495D31EEFCBC30064414FBD78FAE5
CT=7297F8F85B21E1A5CB902DCC5F6738F9
PT=4A6978AACC6968931FB1B83D53D31FEF

I=43
KEY=FBC61C8C946744B2988085B76A330E4C
IV=6BA7664AB5ED4F7956D6C3FAA7C42961
CT=4A6978AACC6968931FB1B83D53D31FEF
PT=6B63FAAF9DB43A2B6025F29B1D6A331C

I=44
KEY=90A5E62309D37E99F8A5772C77593D50
IV=37D17D1702EA34A3C75376AA0F9D24AD
CT=6B63FAAF9DB43A2B6025F29B1D6A331C
PT=8317A07BEE0BF16B88D7D197884AC6CF

I=45
KEY=13B24658E7D88FF27072A6BBFF13FB9F
IV=29B89B238062495C260AF44C63AA809F
CT=8317A07BEE0BF16B88D7D197884AC6CF
PT=0443BFEBCA7C85CBD0CB66AB9D75425D

I=46
KEY=17F1F9B32DA40A39A0B9C0106266B9C2
IV=08CF82854AE07429C4A866AA727CEA2E
CT=0443BFEBCA7C85CBD0CB66AB9D75425D
PT=6B1EB033CB844F9BDD96BA2C67D139C7

I=47
KEY=7CEF4980E62045A27D2F7A3C05B78005
IV=05579F3E0017EBF6083756DC0A50A2E9
CT=6B1EB033CB844F9BDD96BA2C67D139C7
PT=E7904CBFB41EBE4178B4D633AFC9B311

I=48
KEY=9B7F053F523EFBE3059BAC0FAA7E3314
IV=E6B2CB0CE57E02A230DCD16C08614823
CT=E7904CBFB41EBE4178B4D633AFC9B311
PT=96EAAA148F05FAD1F0FC06E135EC5DF8

I=49
KEY=0D95AF2BDD3B0132F567AAEE9F926EEC
IV=4268294B10AD58BE9E570A9485963682
CT=96EAAA148F05FAD1F0FC06E135EC5DF8
PT=9F2F712F62DE747EE479A0A74384B196

I=50
KEY=92BADE04BFE5754C111E0A49DC16DF7A
IV=A5C558F5D4658089CD9FB295EDAE64DB
CT=9F2F712F62DE747EE479A0A74384B196
PT=9CF7F00094C4E2D96D752C2312FC135E

I=51
KEY=0E4D2E042B2197957C6B266ACEEACC24
IV=1CE301F294BDF4C906E5F0A6FA7C22BB
CT=9CF7F00094C4E2D96D752C2312FC135E
PT=ABA58D548DB6E9BE977FDDC85E16C2DE

I=52
KEY=A5E8A350A6977E2BEB14FBA290FC0EFA
IV=66C1FF1E4B52A3542B48306EC33F171D
CT=ABA58D548DB6E9BE977FDDC85E16C2DE
PT=802948E1A5177FBE97ABA26B996FDF1B

I=53
KEY=25C1EBB1038001957CBF59C90993D1E1
IV=197A546DEB2E19E17507276B9ED95135
CT=802948E1A5177FBE97ABA26B996FDF1B
PT=67D931C0946ECDF2F8E41000726D1A56

I=54
KEY=4218DA7197EECC67845B49C97BFECBB7
IV=D11DA3CE92ADA0686F7BF12A4796D9F7
CT=67D931C0946ECDF2F8E41000726D1A56
PT=386C68821385E8CFCDD5678FE2AA89F8

I=55
KEY=7A74B2F3846B24A8498E2E469954424F
IV=4DDE413EBC3F30E2F3A9F48DA1258DAF
CT=386C68821385E8CFCDD5678FE2AA89F8
PT=1638C741EFE3B1CC45C7754783350825

I=56
KEY=6C4C75B26B8895640C495B011A614A6A
IV=5B4F013B085E3A04A16A409E8C3BB270
CT=1638C741EFE3B1CC45C7754783350825
PT=91AB00C3828D3DF72BEC487EF4652567

I=57
KEY=FDE77571E905A89327A5137FEE046F0D
IV=98074F522144682FF1498C61581F8B0E
CT=91AB00C3828D3DF72BEC487EF4652567
PT=BC5BA874D2F3039837528993FB275EBA

I=58
KEY=41BCDD053BF6AB0B10F79AEC152331B7
IV=B65C1A6F80CA024B92176D57061ECDFE
CT=BC5BA874D2F3039837528993FB275EBA
PT=B7DE1BEE1867F4DC3F974E0D8B416DB0

I=59
KEY=F662C6EB23915FD72F60D4E19E625C07
IV=7268C08FA807D83C9830F54A59F32662
CT=B7DE1BEE1867F4DC3F974E0D8B416DB0
PT=19769B9EB30B8DE5FD640ACE78F01F0A

I=60
KEY=EF145D75909AD232D204DE2FE692430D
IV=B260792BAB22BAAB6CCF3AE1DD3A0F85
CT=19769B9EB30B8DE5FD640ACE78F01F0A
PT=2079F2C0A0DE164CD41FCE29D8CE3178

I=61
KEY=CF6DAFB53044C47E061B10063E5C7275
IV=6ABCE7B41E112B91B8E6BC0DBE37600C
CT=2079F2C0A0DE164CD41FCE29D8CE3178
PT=0D3A678FAB06E743A9134C103B39B61D

I=62
KEY=C257C83A9B42233DAF085C160565C468
IV=CEB1E776DBEF46BF8B949C0AF0EBEAD2
CT=0D3A678FAB06E743A9134C103B39B61D
PT=80339ADB34BCDE3384D66170373AEEAC

I=63
KEY=426452E1AFFEFD0E2BDE3D66325F2AC4
IV=EAAC373FCA05E610C1C5D4437AFD5631
CT=80339ADB34BCDE3384D66170373AEEAC
PT=38D52D75E5783A08FBFE622633D6CF47

I=64
KEY=7AB17F944A86C706D0205F400189E583
IV=2A25ED6933CAAA95569ADA1AC7480942
CT=38D52D75E5783A08FBFE622633D6CF47
PT=3204F1EACBB520A69F0939F45BC82041

I=65
KEY=48B58E7E8133E7A04F2966B45A41C5C2
IV=9C6819799DAF61111ACC3C0ECEB0F8BF
CT=3204F1EACBB520A69F0939F45BC82041
PT=16BD33EE541798BEE114E9D7F9141A9C

I=66
KEY=5E08BD90D5247F1EAE3D8F63A355DF5E
IV=C181F4DF1EE0E13BF76044A3DA5E3BC4
CT=16BD33EE541798BEE114E9D7F9141A9C
PT=A0A68CF016C6AD8DAC4D5634EAADEBE5

I=67
KEY=FEAE3160C3E2D2930270D95749F834BB
IV=2A4EE2CD011AF421616789DB790354FD
CT=A0A68CF016C6AD8DAC4D5634EAADEBE5
PT=FCE7D92FD0B3A1C7920558F653E1019B

I=68
KEY=0249E84F13517354907581A11A193520
IV=A7F7E2B701F34F308C0F28514DFAA15F
CT=FCE7D92FD0B3A1C7920558F653E1019B
PT=F16A084B2DE16D5E3E5405665DE3C623

I=69
KEY=F323E0043EB01E0AAE2184C747FAF303
IV=6D319828AAA75DFE99BEEDD21B983C7D
CT=F16A084B2DE16D5E3E5405665DE3C623
PT=CE20AE629887DF02064F2D9DB092E1DB

I=70
KEY=3D034E66A637C108A86EA95AF76812D8
IV=9484E76149F41901AF8AFD3152C56506
CT=CE20AE629887DF02064F2D9DB092E1DB
PT=B457A655282213C1B7C5C2A206C7FCB9

I=71
KEY=8954E8338E15D2C91FAB6BF8F1AFEE61
IV=329E089DC915BD71B1F8D0713BFC4F97
CT=B457A655282213C1B7C5C2A206C7FCB9
PT=A89B8F7541FFF719A3C7DE1577CE1C3F

I=72
KEY=21CF6746CFEA25D0BC6CB5ED8661F25E
IV=FA381CAC5777234E6DF8A28A4E6671D8
CT=A89B8F7541FFF719A3C7DE1577CE1C3F
PT=C4E73D25647350527C1F93F6C1D90296

I=73
KEY=E5285A63AB997582C073261B47B8F0C8
IV=082ACF3D2661514F0E00BDABC9015946
CT=C4E73D25647350527C1F93F6C1D90296
PT=34BF6DECECFE1C9AE3AA34B6DB3741EF

I=74
KEY=D197378F4767691823D912AD9C8FB127
IV=3E56B658F899B6DBBA4EAF6E629F3555
CT=34BF6DECECFE1C9AE3AA34B6DB3741EF
PT=FD86C78538176DC5D48BE2EC6B89D581

I=75
KEY=2C11F00A7F7004DDF752F041F70664A6
IV=765B823193F2738E6CACBB0154C941B1
CT=FD86C78538176DC5D48BE2EC6B89D581
PT=524E035AB9DEB29E467B9586A90F3A42

I=76
KEY=7E5FF350C6AEB643B12965C75E095EE4
IV=DE2B3EC0A3FF29EC67BBE1B124E9451A
CT=524E035AB9DEB29E467B9586A90F3A42
PT=83DE1E768EED94BA09E623845A057834

I=77
KEY=FD81ED26484322F9B8CF4643040C26D0
IV=A5CF4D996D8460E71DDA938461BB884B
CT=83DE1E768EED94BA09E623845A057834
PT=D8381D17B9BCF942F408DBB2B7B2D6B7

I=78
KEY=25B9F031F1FFDBBB4CC79DF1B3BEF067
IV=899696B8EDC4E8B60489742A2559C909
CT=D8381D17B9BCF942F408DBB2B7B2D6B7
PT=BF1D8175015D0A0341427DE1F7DDE379

I=79
KEY=9AA47144F0A2D1B80D85E0104463131E
IV=AD76D13B2F2A3F938F5C810A38A83D09
CT=BF1D8175015D0A0341427DE1F7DDE379
PT=58114DEFF8CFF10F06B390EB01B13394

I=80
KEY=C2B53CAB086D20B70B3670FB45D2208A
IV=9EC540C144AB57D4E5A9185F55E71E50
CT=58114DEFF8CFF10F06B390EB01B13394
PT=53E72B9603CE221F0160791DF51F6B3E

I=81
KEY=9152173D0BA302A80A5609E6B0CD4BB4
IV=B780257238CA60FD998819B2D04CEB92
CT=53E72B9603CE221F0160791DF51F6B3E
PT=AF0238554AAFFB0FB5DFAC35E3CE8F4C

I=82
KEY=3E502F68410CF9A7BF89A5D35303C4F8
IV=3509BC884089ADEB90137A39AEA75CFC
CT=AF0238554AAFFB0FB5DFAC35E3CE8F4C
PT=20844D3F4F637B974D38B7E5F1C84205

I=83
KEY=1ED462570E6F8230F2B11236A2CB86FD
IV=52BAB07EB72804630E4BD769F0172FDA
CT=20844D3F4F637B974D38B7E5F1C84205
PT=68404E006D2F16572EDFAE636A62C537

I=84
KEY=76942C5763409467DC6EBC55C8A943CA
IV=E4B417D85F36F3A25CF04AB05157F247
CT=68404E006D2F16572EDFAE636A62C537
PT=13C8DF4117D4EA3645F1825700F8FFA5

I=85
KEY=655CF31674947E51999F3E02C851BC6F
IV=3FE0683442B6B5DF515209DE16F8FD4B
CT=13C8DF4117D4EA3645F1825700F8FFA5
PT=42307D929439F2723202A5206D640431

I=86
KEY=276C8E84E0AD8C23AB9D9B22A535B85E
IV=33FAD11D9D4057E7ED989F1520450C32
CT=42307D929439F2723202A5206D640431
PT=5ED8AF875ACF37BE8FAAC16D4AB8072B

I=87
KEY=79B42103BA62BB9D24375A4FEF8DBF75
IV=807852E5AD0EB5E0C3A2AAF18DFB14F9
CT=5ED8AF875ACF37BE8FAAC16D4AB8072B
PT=79C950A45ECDC8B676A5CFF324A81DA8

I=88
KEY=007D71A7E4AF732B529295BCCB25A2DD
IV=516BC35B52C26C99E21C0147CB7DD99E
CT=79C950A45ECDC8B676A5CFF324A81DA8
PT=595C3DDCFBD90661A3A70B94C470A26B

I=89
KEY=59214C7B1F76754AF1359E280F5500B6
IV=FDABDD4BD60725F5E640629416F417D3
CT=595C3DDCFBD90661A3A70B94C470A26B
PT=7957CF3989083D25B54C6890EC58DB0C

I=90
KEY=20768342967E486F4479F6B8E30DDBBA
IV=CADFADF417898DE3786D5DF869396611
CT=7957CF3989083D25B54C6890EC58DB0C
PT=6C1E8DF34FC80DD2ECA00A1EDC5FC239

I=91
KEY=4C680EB1D9B645BDA8D9FCA63F521983
IV=A7DF9FA44AA2B49B9E26C6148CCAF36F
CT=6C1E8DF34FC80DD2ECA00A1EDC5FC239
PT=72B5A426C9DC3B336756FB10B00626CE

I=92
KEY=3EDDAA97106A7E8ECF8F07B68F543F4D
IV=50FF574915D407A97A74CC399DE2D901
CT=72B5A426C9DC3B336756FB10B00626CE
PT=194826159D7FAB9166D64F1DD6F627B0

I=93
KEY=27958C828D15D51FA95948AB59A218FD
IV=A1E3BC848C019B6FC3F7072F9A866C52
CT=194826159D7FAB9166D64F1DD6F627B0
PT=0186B2316FC29104D3549C49F92AE414

I=94
KEY=26133EB3E2D7441B7A0DD4E2A088FCE9
IV=4DBA42F1031AE32016524C7F86781528
CT=0186B2316FC29104D3549C49F92AE414
PT=373384DB7383D3A4914C0141BB104B81

I=95
KEY=1120BA68915497BFEB41D5A31B98B768
IV=FFCB87107AC596D71BA807902D524247
CT=373384DB7383D3A4914C0141BB104B81
PT=25465CE5FA2AD65D75586064A18516A8

I=96
KEY=3466E68D6B7E41E29E19B5C7BA1DA1C0
IV=55879596C3F0A295FA00CFEF1B056248
CT=25465CE5FA2AD65D75586064A18516A8
PT=2311A52CEC4A16B5F4FCA613BC0BDB83

I=97
KEY=177743A1873457576AE513D406167A43
IV=B1A7DABF3D2384BF8754B829659B3989
CT=2311A52CEC4A16B5F4FCA613BC0BDB83
PT=357F4BF62308CD5C48F34E9E644DD515

I=98
KEY=22080857A43C9A0B22165D4A625BAF56
IV=72BF477FE8CF428CFA856323E18B3135
CT=357F4BF62308CD5C48F34E9E644DD515
PT=17758539D1896B92BFB811D30C9BA134

I=99
KEY=357D8D6E75B5F1999DAE4C996EC00E62
IV=4CBA048DFA50910EC94FE5D6AA5DD502
CT=17758539D1896B92BFB811D30C9BA134
PT=372B6DCEE44D18D3791C4E8FB0A91971

I=100
KEY=0256E0A091F8E94AE4B20216DE691713
IV=86F4F670CA3B191B860D6C8A9A35A872
CT=372B6DCEE44D18D3791C4E8FB0A91971
PT=66D3FABC4885C1F4897D9160C152D570

I=101
KEY=64851A1CD97D28BE6DCF93761F3BC263
IV=08490B2E8588FE564F0112D76D6108E9
CT=66D3FABC4885C1F4897D9160C152D570
PT=926278E71894F32B5E2A5A1F5FA695E7

I=102
KEY=F6E762FBC1E9DB9533E5C969409D5784
IV=D3A2E4AFFDD8A7CC1498EF0E812087B7
CT=926278E71894F32B5E2A5A1F5FA695E7
PT=A455F5D65D91563D9C15E6B745F29531

I=103
KEY=52B2972D9C788DA8AFF02FDE056FC2B5
IV=6CB8822EA67C247A6A0CBAD830B3562C
CT=A455F5D65D91563D9C15E6B745F29531
PT=4544371A5E3A8E7AD9885AECD3868AF3

I=104
KEY=17F6A037C24203D276787532D6E94846
IV=93E3CAD8BE023F6451ADAA525988A974
CT=4544371A5E3A8E7AD9885AECD3868AF3
PT=8473F4F1D77AC1F46706EBD108CB48A5

I=105
KEY=938554C61538C226117E9EE3DE2200E3
IV=A7E7B354606EF4C1D00BF4F128CCE966
CT=8473F4F1D77AC1F46706EBD108CB48A5
PT=787ADE453DCE2FA7F6F24CD33C9A5CDC

I=106
KEY=EBFF8A8328F6ED81E78CD230E2B85C3F
IV=97A2CF706C1871B502D15B028C5F0D80
CT=787ADE453DCE2FA7F6F24CD33C9A5CDC
PT=E6A7E5B65DC124FBE1C1FFFE18A34D8A

I=107
KEY=0D586F357537C97A064D2DCEFA1B11B5
IV=9BBDCAF2A3CC7FC5BF6285631A4636FD
CT=E6A7E5B65DC124FBE1C1FFFE18A34D8A
PT=6A391E9F475546720836312BBEB0B3A9

I=108
KEY=676171AA32628F080E7B1CE544ABA21C
IV=603EEDE91F925A2228537E8BDD26C96B
CT=6A391E9F475546720836312BBEB0B3A9
PT=D75086BA446563AAB1873FDEBE611D67

I=109
KEY=B031F7107607ECA2BFFC233BFACABF7B
IV=546B8CA37AACE2BC2C85014F2ACD6315
CT=D75086BA446563AAB1873FDEBE611D67
PT=63A4D826625CBD9F10083E62330BCFBC

I=110
KEY=D3952F36145B513DAFF41D59C9C170C7
IV=BC2E6E9DD10152C8281B798F711A9334
CT=63A4D826625CBD9F10083E62330BCFBC
PT=41275EFDFC86B5BDCCCDCA2BBFA5B90D

I=111
KEY=92B271CBE8DDE4806339D7727664C9CA
IV=D940A8D1B3448AA9148C7F6DFC266B00
CT=41275EFDFC86B5BDCCCDCA2BBFA5B90D
PT=56AC1BCB285044710D26EFC42A0ABA56

I=112
KEY=C41E6A00C08DA0F16E1F38B65C6E739C
IV=214238FF5F4D01F3A6694EC80E15BCF7
CT=56AC1BCB285044710D26EFC42A0ABA56
PT=5AF2877973AB9787E42202AC0C6E7538

I=113
KEY=9EECED79B32637768A3D3A1A500006A4
IV=561B4934904E327F3B3639CED6C7A995
CT=5AF2877973AB9787E42202AC0C6E7538
PT=787F52F6D9CD9085D782B95FD68F70F8

I=114
KEY=E693BF8F6AEBA7F35DBF8345868F765C
IV=06B93FB1A99BD387A091509B8C6D3482
CT=787F52F6D9CD9085D782B95FD68F70F8
PT=E7B1DB917C92E488C6B02AC4C42AFD79

I=115
KEY=0122641E1679437B9B0FA98142A58B25
IV=1D57A61FED51218BD8C1357CE58F21EB
CT=E7B1DB917C92E488C6B02AC4C42AFD79
PT=2FBAB770D511EACA849F654DE429DE84

I=116
KEY=2E98D36EC368A9B11F90CCCCA68C55A1
IV=CB311C43EC32D59941CEF81BB9CF4E57
CT=2FBAB770D511EACA849F654DE429DE84
PT=A8FE17AEA961C2A450DD5EAABAA19613

I=117
KEY=8666C4C06A096B154F4D92661C2DC3B2
IV=B9D7B74D62DD89A01EE3093A5C2A0470
CT=A8FE17AEA961C2A450DD5EAABAA19613
PT=D9E598A4DBAF82BDF9474BE13B997FF5

I=118
KEY=5F835C64B1A6E9A8B60AD98727B4BC47
IV=6398D8A38C38A9B54DCF18C0D76758BA
CT=D9E598A4DBAF82BDF9474BE13B997FF5
PT=291DB2BC325CD7A94032E7C1208921A5

I=119
KEY=769EEED883FA3E01F6383E46073D9DE2
IV=4774C7E1BB97117BCA26A897B9302EFA
CT=291DB2BC325CD7A94032E7C1208921A5
PT=DFC55866F85055F06E752E04AAC4C36A

I=120
KEY=A95BB6BE7BAA6BF1984D1042ADF95E88
IV=BDA0616700CD2817FE4425F6F541E8C1
CT=DFC55866F85055F06E752E04AAC4C36A
PT=7CACB3A88E7114F7737B05D0FB3AF7F3

I=121
KEY=D5F70516F5DB7F06EB36159256C3A97B
IV=1EF4836BD363B89EE73911B14EBE9262
CT=7CACB3A88E7114F7737B05D0FB3AF7F3
PT=3282F1F6B3C04E325FB13E85D18CDBBC

I=122
KEY=E775F4E0461B3134B4872B17874F72C7
IV=E35D47CEDBD30701A64FE504C6BC4199
CT=3282F1F6B3C04E325FB13E85D18CDBBC
PT=BFB9748A3C825CDA3EA5D001F6AD305F

I=123
KEY=58CC806A7A996DEE8A22FB1671E24298
IV=7A3676377E0B4F815B4B8C9A98DCAB4D
CT=BFB9748A3C825CDA3EA5D001F6AD305F
PT=B33787380A4C3BD5B32EC335916F9229

I=124
KEY=EBFB075270D5563B390C3823E08DD0B1
IV=B3E40B27DBF6D985D4F57BD4D2899DC8
CT=B33787380A4C3BD5B32EC335916F9229
PT=BCD5E40A3B4885ACA82BEC952BA8A973

I=125
KEY=572EE3584B9DD3979127D4B6CB2579C2
IV=897608FE3238F80CD24192CC64F4F695
CT=BCD5E40A3B4885ACA82BEC952BA8A973
PT=793E22E0B01FFE63E89C84A0D0FE1216

I=126
KEY=2E10C1B8FB822DF479BB50161BDB6BD4
IV=D84401CF83566FEA8FB859C446C22632
CT=793E22E0B01FFE63E89C84A0D0FE1216
PT=96858A96FEB19712B10A44FBF8E633D9

I=127
KEY=B8954B2E0533BAE6C8B114EDE33D580D
IV=7210E127BFB706EF9CD24139C0927E01
CT=96858A96FEB19712B10A44FBF8E633D9
PT=28A34CC75A215117331D6B88387E16F8

I=128
KEY=903607E95F12EBF1FBAC7F65DB434EF5
IV=744B8DD281919F79FC529DA15AA0F18F
CT=28A34CC75A215117331D6B88387E16F8
PT=43FE41DC9FFB88F93ADC43D41BA3E4B2

I=129
KEY=D3C84635C0E96308C1703CB1C0E0AA47
IV=5C7379A90F1E47F88C7424E8CD31E55D
CT=43FE41DC9FFB88F93ADC43D41BA3E4B2
PT=66D6D0E84CEB8672D38DDE557CCCFFF6

I=130
KEY=B51E96DD8C02E57A12FDE2E4BC2C55B1
IV=529B92CD068415627C4BE49874347F98
CT=66D6D0E84CEB8672D38DDE557CCCFFF6
PT=F817D013CB041CA41B6D611ED261FAE9

I=131
KEY=4D0946CE4706F9DE099083FA6E4DAF58
IV=9B4452DD48C3EECEFDB6A6CFAE670117
CT=F817D013CB041CA41B6D611ED261FAE9
PT=56420C70CC459A131CD531EDC39ED6E9

I=132
KEY=1B4B4ABE8B4363CD1545B217ADD379B1
IV=264ABBC30CC98DA3C06B935B1364A317
CT=56420C70CC459A131CD531EDC39ED6E9
PT=650DA8C4CB11E4ABFE8841D20443BB04

I=133
KEY=7E46E27A40528766EBCDF3C5A990C2B5
IV=DBABF7B18568FED50C69DA0C9AA5395D
CT=650DA8C4CB11E4ABFE8841D20443BB04
PT=94E6CE02AE253F3BE4A59775CAB7B11B

I=134
KEY=EAA02C78EE77B85D0F6864B0632773AE
IV=F0D2E456CE063CB68988C1C397038480
CT=94E6CE02AE253F3BE4A59775CAB7B11B
PT=124F09DAC6A45E6077DCECE971B24ED1

I=135
KEY=F8EF25A228D3E63D78B4885912953D7F
IV=C19A6D2D35E606E552580B002023C966
CT=124F09DAC6A45E6077DCECE971B24ED1
PT=AAD649FD1168DFCB431A7216571ED4C1

I=136
KEY=52396C5F39BB39F63BAEFA4F458BE9BE
IV=DF1A307B1D25BC2751ECDC791EC51B16
CT=AAD649FD1168DFCB431A7216571ED4C1
PT=992749C940CBBB0944F5248E8D65268C

I=137
KEY=CB1E2596797082FF7F5BDEC1C8EECF32
IV=D8B0466FC4344C6BF4E57584E5F48A36
CT=992749C940CBBB0944F5248E8D65268C
PT=FF16B9246C325A3B96250875A9D5C4BE

I=138
KEY=34089CB21542D8C4E97ED6B4613B0B8C
IV=B8C3E4ECF57631CE1EEA74AD99545BA2
CT=FF16B9246C325A3B96250875A9D5C4BE
PT=72EDB906165FE6A19E02435AC6457C9A

I=139
KEY=46E525B4031D3E65777C95EEA77E7716
IV=9C4F78D05257C64D752C576B5BB16245
CT=72EDB906165FE6A19E02435AC6457C9A
PT=0C2F00362B8733D343DCAF884A6BFD46

I=140
KEY=4ACA2582289A0DB634A03A66ED158A50
IV=C51122198D2FD566ACA1BD07C28BCC7A
CT=0C2F00362B8733D343DCAF884A6BFD46
PT=C7F5C00F42695CF7870240DA73D487F7

I=141
KEY=8D3FE58D6AF35141B3A27ABC9EC10DA7
IV=14769B7962371925822C42F0411BDC65
CT=C7F5C00F42695CF7870240DA73D487F7
PT=F4BC8A4310A4CD2F396F43B81BD060EB

I=142
KEY=79836FCE7A579C6E8ACD390485116D4C
IV=CDA4414DF23427B937D586C1A8164479
CT=F4BC8A4310A4CD2F396F43B81BD060EB
PT=1B996A525174C35466103BCBD53ECB9C

I=143
KEY=621A059C2B235F3AECDD02CF502FA6D0
IV=1DA9E7C02BE868E118E4B9BF5928BFCF
CT=1B996A525174C35466103BCBD53ECB9C
PT=2B4EC5CB9DE50257B5901C6D49B5EB54

I=144
KEY=4954C057B6C65D6D594D1EA2199A4D84
IV=55433149AA93215BF53CB8FE66CD683A
CT=2B4EC5CB9DE50257B5901C6D49B5EB54
PT=5A5B9ACA3BDA951DC4C4DFB7DD15AD18

I=145
KEY=130F5A9D8D1CC8709D89C115C48FE09C
IV=695C69EA8C4257D2758364AFF6387DA0
CT=5A5B9ACA3BDA951DC4C4DFB7DD15AD18
PT=5BB08593BA52E94F07CB14FF564AF299

I=146
KEY=48BFDF0E374E213F9A42D5EA92C51205
IV=665463164601CD0045C684729CDD5E7D
CT=5BB08593BA52E94F07CB14FF564AF299
PT=DB1840467889551150026A90E3FC272F

I=147
KEY=93A79F484FC7742ECA40BF7A7139352A
IV=355C4B87A682712460C80ABA630F7BE8
CT=DB1840467889551150026A90E3FC272F
PT=9F7BD8EF96A1ECC75B4E3031E65FE8FC

I=148
KEY=0CDC47A7D96698E9910E8F4B9766DDD6
IV=FBF20FEC0A9E9B530DD3745955A1960C
CT=9F7BD8EF96A1ECC75B4E3031E65FE8FC
PT=DCF003E1DDD4B52F5B680568B42EEECF

I=149
KEY=D02C444604B22DC6CA668A2323483319
IV=04EADC638E309466B9500B1C3A308856
CT=DCF003E1DDD4B52F5B680568B42EEECF
PT=DDD2F2AEFEF4A4641747699170DF0129

I=150
KEY=0DFEB6E8FA4689A2DD21E3B253973230
IV=FB521123E5BB45F1AC7083C382B95E10
CT=DDD2F2AEFEF4A4641747699170DF0129
PT=7D6C54E4369E965513DE1E95E8A37433

I=151
KEY=7092E20CCCD81FF7CEFFFD27BB344603
IV=0E71300101915AEE97563493B4B2EFB9
CT=7D6C54E4369E965513DE1E95E8A37433
PT=B7F5C4275EF7D3A23E8E9A4BC2BCA700

I=152
KEY=C767262B922FCC55F071676C7988E103
IV=8BBC970D4E24F6F1A92859BBAE70B757
CT=B7F5C4275EF7D3A23E8E9A4BC2BCA700
PT=679FA9A8AB3EE2F0663D4DC95FEC3847

I=153
KEY=A0F88F8339112EA5964C2AA52664D944
IV=B8D28B6903859465F996A0DF491DDAAA
CT=679FA9A8AB3EE2F0663D4DC95FEC3847
PT=914DCEAC5B00940A415C2144DAA1EF03

I=154
KEY=31B5412F6211BAAFD7100BE1FCC53647
IV=06E1FF99E63E0EDDA1D95682B541D026
CT=914DCEAC5B00940A415C2144DAA1EF03
PT=A718031A531395DA071B828A10A2B9DC

I=155
KEY=96AD423531022F75D00B896BEC678F9B
IV=F18546028693CC0A72E218460AC932E9
CT=A718031A531395DA071B828A10A2B9DC
PT=19D996B9FF549029D800CF977C0E0A88

I=156
KEY=8F74D48CCE56BF5C080B46FC90698513
IV=7E98DAFA8DE35537A0B0AA8FA32E01DF
CT=19D996B9FF549029D800CF977C0E0A88
PT=6A7D514FBE9963987E0EE5DD22FDC8BD

I=157
KEY=E50985C370CFDCC47605A321B2944DAE
IV=071F44277D12AD6E653FB747C11BEFB7
CT=6A7D514FBE9963987E0EE5DD22FDC8BD
PT=AB07C464DF72DF58411A7CBF906F43A6

I=158
KEY=4E0E41A7AFBD039C371FDF9E22FB0E08
IV=BDA8979A0DE78999AE0659D2744BC21B
CT=AB07C464DF72DF58411A7CBF906F43A6
PT=A8ABC34FF51FD998427A7837BB0F22BB

I=159
KEY=E6A582E85AA2DA047565A7A999F42CB3
IV=3CA763C4375A6C70F072B52C42A02560
CT=A8ABC34FF51FD998427A7837BB0F22BB
PT=9E82A41AB2A2748695663075C5BF1C09

I=160
KEY=782726F2E800AE82E00397DC5C4B30BA
IV=B1DD4DC8715CA0B8BA3426395B61ABA4
CT=9E82A41AB2A2748695663075C5BF1C09
PT=3C891AEFC6645F3C23A2E3A2E1A522EF

I=161
KEY=44AE3C1D2E64F1BEC3A1747EBDEE1255
IV=932F28E3FC0F85A0E703AD6079A68CC0
CT=3C891AEFC6645F3C23A2E3A2E1A522EF
PT=D4FDD9C3D36E80E0AFA95ABEC2E62FB9

I=162
KEY=9053E5DEFD0A715E6C082EC07F083DEC
IV=1AD1530DE6274BBA3A097004D85DF8FA
CT=D4FDD9C3D36E80E0AFA95ABEC2E62FB9
PT=9FACBB106FBF34D9B25A83AD4F45AA38

I=163
KEY=0FFF5ECE92B54587DE52AD6D304D97D4
IV=8412EAA361AB8193AD529FA36944FE3D
CT=9FACBB106FBF34D9B25A83AD4F45AA38
PT=E95628B6A14326AC153B228C0DD25063

I=164
KEY=E6A9767833F6632BCB698FE13D9FC7B7
IV=9F1D906FD25CCCA7565D82CDA44B4A57
CT=E95628B6A14326AC153B228C0DD25063
PT=7B1ECD07E452D593D00E326F8FD9F2B5

I=165
KEY=9DB7BB7FD7A4B6B81B67BD8EB2463502
IV=9C91B12734A86451492400EE5759C190
CT=7B1ECD07E452D593D00E326F8FD9F2B5
PT=A99874C1ECA463D335B37179348AEF4F

I=166
KEY=342FCFBE3B00D56B2ED4CCF786CCDA4D
IV=9060C6E8E2B51E5F1000860ECB5A608C
CT=A99874C1ECA463D335B37179348AEF4F
PT=F9296E658C3048C58FE6404AB6E0186C

I=167
KEY=CD06A1DBB7309DAEA1328CBD302CC221
IV=ADCB427405EDF8078622AFC6C731D3EF
CT=F9296E658C3048C58FE6404AB6E0186C
PT=8D3A4ABFB8F57AEBB1284B78BFDD240D

I=168
KEY=403CEB640FC5E745101AC7C58FF1E62C
IV=4CED30477756892A49FE6358700F5DE7
CT=8D3A4ABFB8F57AEBB1284B78BFDD240D
PT=7C71310D5241B8B461847FA7E2B4D506

I=169
KEY=3C4DDA695D845FF1719EB8626D45332A
IV=52A3B8E21588368AA3AD2131B283B34C
CT=7C71310D5241B8B461847FA7E2B4D506
PT=380A1E2925FC8061FBC69B08B4C7C6C8

I=170
KEY=0447C4407878DF908A58236AD982F5E2
IV=B7AD265DF50ABD77C1D1D02BAF06CEBD
CT=380A1E2925FC8061FBC69B08B4C7C6C8
PT=6931E262570F27732A75629A384492CA

I=171
KEY=6D7626222F77F8E3A02D41F0E1C66728
IV=DD7D3CA7F6CE54775836734B10685A99
CT=6931E262570F27732A75629A384492CA
PT=336D0AB3F9A793469C544E98CFDC8AD6

I=172
KEY=5E1B2C91D6D06BA53C790F682E1AEDFE
IV=54087C05437049FBAFC9BF6C56AD16CC
CT=336D0AB3F9A793469C544E98CFDC8AD6
PT=0098A640F1D04E5557F748B80215082A

I=173
KEY=5E838AD1270025F06B8E47D02C0FE5D4
IV=DD13A93C7FAEF14820D6A168AF9969DB
CT=0098A640F1D04E5557F748B80215082A
PT=48660EDDC3ACAC1A814AE088650DE007

I=174
KEY=16E5840CE4AC89EAEAC4A758490205D3
IV=589C43A680F7AE6E15DE25FD50F18BED
CT=48660EDDC3ACAC1A814AE088650DE007
PT=047740C48160A14563C0049316C7B78E

I=175
KEY=1292C4C865CC28AF8904A3CB5FC5B25D
IV=22DB095AF8268D37AC536FDA3A81B7C2
CT=047740C48160A14563C0049316C7B78E
PT=D9679ABB1AED1524E8DA2ECA89172853

I=176
KEY=CBF55E737F213D8B61DE8D01D6D29A0E
IV=77FDCCC9BAB3B7E81D7B9D14540ED3A7
CT=D9679ABB1AED1524E8DA2ECA89172853
PT=C774AD9EF683A59747AB41DA044471C4

I=177
KEY=0C81F3ED89A2981C2675CCDBD296EBCA
IV=2E266970512C0F6D404367978C70B5C6
CT=C774AD9EF683A59747AB41DA044471C4
PT=927C98AFDE6F108268A306A1A818EE89

I=178
KEY=9EFD6B4257CD889E4ED6CA7A7A8E0543
IV=E2B0A0F146E931136932D8188B7DC980
CT=927C98AFDE6F108268A306A1A818EE89
PT=D3C11AF1EF255D65EAAEE5B5C5BF3CD9

I=179
KEY=4D3C71B3B8E8D5FBA4782FCFBF31399A
IV=ABA118A590A8A69D41D20ED13F7DA4C9
CT=D3C11AF1EF255D65EAAEE5B5C5BF3CD9
PT=760B1C47EB476A246CE41163B7425C7B

I=180
KEY=3B376DF453AFBFDFC89C3EAC087365E1
IV=8A395FFEE93FF820FBC086932216324F
CT=760B1C47EB476A246CE41163B7425C7B
PT=4EEAE25D1654A461D6A56896EDFFEAB7

I=181
KEY=75DD8FA945FB1BBE1E39563AE58C8F56
IV=82D68DE458D38127E2A17DA736BC1651
CT=4EEAE25D1654A461D6A56896EDFFEAB7
PT=BD7ABD27F3515B9FD337ADADBA5E9EC9

I=182
KEY=C8A7328EB6AA4021CD0EFB975FD2119F
IV=95757C38F4CB26C29DF174FD8B1F35E8
CT=BD7ABD27F3515B9FD337ADADBA5E9EC9
PT=E32EF56DE91CC21C158822BDB841A02C

I=183
KEY=2B89C7E35FB6823DD886D92AE793B1B3
IV=AFFB434005E300E99E815A8E5B46E159
CT=E32EF56DE91CC21C158822BDB841A02C
PT=578B7692D68BD4D65A159C9CCDC81F64

I=184
KEY=7C02B171893D56EB829345B62A5BAED7
IV=15E783551969F275EF25A429F68BA0DB
CT=578B7692D68BD4D65A159C9CCDC81F64
PT=0F70D77F31DE8F0010628377C8E4A1C3

I=185
KEY=7372660EB8E3D9EB92F1C6C1E2BF0F14
IV=078B6E3FC7C79D52A5C49847A7C5D0BB
CT=0F70D77F31DE8F0010628377C8E4A1C3
PT=A4F29F6A50A4A15789D4647774C96997

I=186
KEY=D780F964E84778BC1B25A2B696766683
IV=C9B2C903F131856C0D8218146479513A
CT=A4F29F6A50A4A15789D4647774C96997
PT=7AD4F764911F8F1E8C9356F00B1468EE

I=187
KEY=AD540E007958F7A297B6F4469D620E6D
IV=780DAD9DBA12DDC713ECF44057B8B56E
CT=7AD4F764911F8F1E8C9356F00B1468EE
PT=A00EAFD28A95C7F39137773F28F9A0D3

I=188
KEY=0D5AA1D2F3CD305106818379B59BAEBE
IV=6643BF26488FB21BCDAA7D4FABAA5BC3
CT=A00EAFD28A95C7F39137773F28F9A0D3
PT=FC884EEEB73B2F66C97D742A441B0463

I=189
KEY=F1D2EF3C44F61F37CFFCF753F180AADD
IV=7B707442550BF1FA4C1B74F0DC3237FC
CT=FC884EEEB73B2F66C97D742A441B0463
PT=9C260AA82C6902B068495B4844507F7D

I=190
KEY=6DF4E594689F1D87A7B5AC1BB5D0D5A0
IV=8329E8DE2980225DA61158BF467B1D79
CT=9C260AA82C6902B068495B4844507F7D
PT=1C12FEAA35AD2DD21DE6411C3A3BB6D5

I=191
KEY=71E61B3E5D323055BA53ED078FEB6375
IV=EFD514D7CF3FAC49DBB5BB3505B9B18F
CT=1C12FEAA35AD2DD21DE6411C3A3BB6D5
PT=15F26D34E8E1037A5A2393CEA5DB396B

I=192
KEY=6414760AB5D3332FE0707EC92A305A1E
IV=9EFC37C295552B965B29A23AC2788D8B
CT=15F26D34E8E1037A5A2393CEA5DB396B
PT=76F0C79955566A0A091AA46E0C59455E

I=193
KEY=12E4B193E0855925E96ADAA726691F40
IV=5BA54E33128A7526A9130E632BE00D65
CT=76F0C79955566A0A091AA46E0C59455E
PT=1EA06231EBEEB995E2F94B88A4004333

I=194
KEY=0C44D3A20B6BE0B00B93912F82695C73
IV=9B1C6FA42510E530D2282BF8C5516DCE
CT=1EA06231EBEEB995E2F94B88A4004333
PT=16C36348884B32AC1CB1DE2D3C158981

I=195
KEY=1A87B0EA8320D21C17224F02BE7CD5F2
IV=F755F641EED4C828DCBECC327D93F4AF
CT=16C36348884B32AC1CB1DE2D3C158981
PT=6581A3FB5F61C6C1904F35E787BF552D

I=196
KEY=7F061311DC4114DD876D7AE539C380DF
IV=5D799F356C1672707716C47DA4384C01
CT=6581A3FB5F61C6C1904F35E787BF552D
PT=0BA11F8EE845E03641928825B3344994

I=197
KEY=74A70C9F3404F4EBC6FFF2C08AF7C94B
IV=89A5592031795CEAECF1C1F274813818
CT=0BA11F8EE845E03641928825B3344994
PT=1CF907A6C8A33C7806819EB97B854588

I=198
KEY=685E0B39FCA7C893C07E6C79F1728CC3
IV=BBE87558AF456066B9CE3EBB8729D554
CT=1CF907A6C8A33C7806819EB97B854588
PT=722BB2E53CE7846F40B5EA18AF1430B0

I=199
KEY=1A75B9DCC0404CFC80CB86615E66BC73
IV=2CC42E6DA36883482ACB5D9D92823186
CT=722BB2E53CE7846F40B5EA18AF1430B0
PT=E667D6A95E9C56BB56E4436CC1B69874

I=200
KEY=FC126F759EDC1A47D62FC50D9FD02407
IV=F18DD0388BF5692B7394BAFDC4854308
CT=E667D6A95E9C56BB56E4436CC1B69874
PT=3C9A76798C468470EEBC5F7B915879A3

I=201
KEY=C088190C129A9E3738939A760E885DA4
IV=8262D8603FE0F4AE8BF5795690992F1C
CT=3C9A76798C468470EEBC5F7B915879A3
PT=3CAFB994BDEF76370D8A5129B63BECDB

I=202
KEY=FC27A098AF75E8003519CB5FB8B3B17F
IV=AB090F7A86CFD9A7C1BD8869BBB4D27B
CT=3CAFB994BDEF76370D8A5129B63BECDB
PT=50D037733D2C3242024BD322ADDF01CB

I=203
KEY=ACF797EB9259DA423752187D156CB0B4
IV=3581D6724722C3892C712B2AA1066824
CT=50D037733D2C3242024BD322ADDF01CB
PT=0D7CB755305EF09692FCED530CCC1336

I=204
KEY=A18B20BEA2072AD4A5AEF52E19A0A382
IV=9D0DAD6448895ECBE05F8DC73E014E3C
CT=0D7CB755305EF09692FCED530CCC1336
PT=D66389E01F44C60B26BCF3F9B39DAB55

I=205
KEY=77E8A95EBD43ECDF831206D7AA3D08D7
IV=7BDC3CD62CD79C7E6CB8FAE104175DE4
CT=D66389E01F44C60B26BCF3F9B39DAB55
PT=CC9F59409D636BE5C555AC2841650CD7

I=206
KEY=BB77F01E2020873A4647AAFFEB580400
IV=A4FBE42983B236097E40C1E244B14748
CT=CC9F59409D636BE5C555AC2841650CD7
PT=34804672FCEF79481269B1D254DDCAF9

I=207
KEY=8FF7B66CDCCFFE72542E1B2DBF85CEF9
IV=700B37E771E88E5DCD9C81D6FD16B995
CT=34804672FCEF79481269B1D254DDCAF9
PT=62FC00FDF1F6BF44FFBEB5BED5589A64

I=208
KEY=ED0BB6912D394136AB90AE936ADD549D
IV=F3A9954079382ACD97C3EC198522AE69
CT=62FC00FDF1F6BF44FFBEB5BED5589A64
PT=F7901904A02544E064400A2625121C4C

I=209
KEY=1A9BAF958D1C05D6CFD0A4B54FCF48D1
IV=3799F5DEF76C56F8B0A69889B85760AA
CT=F7901904A02544E064400A2625121C4C
PT=A915D55AE8E14497F2693A55F83041F5

I=210
KEY=B38E7ACF65FD41413DB99EE0B7FF0924
IV=5421F3F4CDD3C6F9BF38184BC61904AA
CT=A915D55AE8E14497F2693A55F83041F5
PT=66EA0D0968631F3BDA46921078E495C1

I=211
KEY=D56477C60D9E5E7AE7FF0CF0CF1B9CE5
IV=3B87243EE38BD474E4CBF34A7C592CFC
CT=66EA0D0968631F3BDA46921078E495C1
PT=283826367D00718849A6539EE8CAD8BF

I=212
KEY=FD5C51F0709E2FF2AE595F6E27D1445A
IV=0FB53077D2BC27C544CD400A28212291
CT=283826367D00718849A6539EE8CAD8BF
PT=B15EA2718FB406D457C8B96530EDC014

I=213
KEY=4C02F381FF2A2926F991E60B173C844E
IV=8BFE654E422E9BCCE7F1357053E1130C
CT=B15EA2718FB406D457C8B96530EDC014
PT=44238CE0F1765C7983BF408D5ECFA2BA

I=214
KEY=08217F610E5C755F7A2EA68649F326F4
IV=86841D597FA91F54CBAD3C57FC1FC492
CT=44238CE0F1765C7983BF408D5ECFA2BA
PT=BB6408099CA64F1216D204865449E967

I=215
KEY=B345776892FA3A4D6CFCA2001DBACF93
IV=849198CEA19F4AD76013D172867A0D20
CT=BB6408099CA64F1216D204865449E967
PT=B7E109C6813E686ADA396DDB277AD960

I=216
KEY=04A47EAE13C45227B6C5CFDB3AC016F3
IV=9C5E0182124701DA958CF68E1B9F8CC1
CT=B7E109C6813E686ADA396DDB277AD960
PT=65C77CB822CA10AE1F95B91F0A3F6436

I=217
KEY=61630216310E4289A95076C430FF72C5
IV=2BB9026F793A64A325E02B922E3A51DD
CT=65C77CB822CA10AE1F95B91F0A3F6436
PT=48E9DB9FC3F883AD28A0299561444197

I=218
KEY=298AD989F2F6C12481F05F5151BB3352
IV=6BE0B03639C858D8F2CAE00EFE8FECCA
CT=48E9DB9FC3F883AD28A0299561444197
PT=02B277AB327BCB5C91831BD376DC5211

I=219
KEY=2B38AE22C08D0A781073448227676143
IV=ADC033E3EEC3705031D2FB24205823A2
CT=02B277AB327BCB5C91831BD376DC5211
PT=FE288D6703CF6AB1635483305A82F192

I=220
KEY=D5102345C34260C97327C7B27DE590D1
IV=F637C38CB5DF86458225143E157B9415
CT=FE288D6703CF6AB1635483305A82F192
PT=B07BF02F795D13D972D5D6E556E8BA37

I=221
KEY=656BD36ABA1F731001F211572B0D2AE6
IV=824B7BBB3A339BF045C3CFEA9FD842B9
CT=B07BF02F795D13D972D5D6E556E8BA37
PT=D4661C2E102C6644724DA54BE291215D

I=222
KEY=B10DCF44AA33155473BFB41CC99C0BBB
IV=21FA242B2B9FFF0C7DC6B712A38A333B
CT=D4661C2E102C6644724DA54BE291215D
PT=F7EDEE448559A6ACD68815B6EEAF6E38

I=223
KEY=46E021002F6AB3F8A537A1AA27336583
IV=390906ECE51C92800D620E1CD1FD3574
CT=F7EDEE448559A6ACD68815B6EEAF6E38
PT=07257C91424F65E6752470A6960DF6D9

I=224
KEY=41C55D916D25D61ED013D10CB13E935A
IV=918A010EC2B106DE32A220315999DF7D
CT=07257C91424F65E6752470A6960DF6D9
PT=CCB4F3FCFFDEE9E93DD3AB4CC2DB870E

I=225
KEY=8D71AE6D92FB3FF7EDC07A4073E51454
IV=7CF6B4948C4E04ED235F2CC96B0A6B49
CT=CCB4F3FCFFDEE9E93DD3AB4CC2DB870E
PT=C455220E8C8847FF2592074A53104105

I=226
KEY=49248C631E737808C8527D0A20F55551
IV=4839E74C868786D0A109603CF98F1100
CT=C455220E8C8847FF2592074A53104105
PT=B4D0F5B4C15D7AD556ED4964A94C898F

I=227
KEY=FDF479D7DF2E02DD9EBF346E89B9DCDE
IV=5C6D9649E47FAA8A2DCAFB317B6BC0EA
CT=B4D0F5B4C15D7AD556ED4964A94C898F
PT=B515D2904555793DDC142618DCFF8464

I=228
KEY=48E1AB479A7B7BE042AB1276554658BA
IV=E3573B4137C13BBA6CDD10D71042B8F6
CT=B515D2904555793DDC142618DCFF8464
PT=2C272D6589E2E6EE09067882795E9FB9

I=229
KEY=64C6862213999D0E4BAD6AF42C18C703
IV=08E335104FC5D18E2C11EF34B805F060
CT=2C272D6589E2E6EE09067882795E9FB9
PT=B38E39322245C4A2E170EBF36D764AD5

I=230
KEY=D748BF1031DC59ACAADD8107416E8DD6
IV=E134D0DEA38A23983766DA18DADA70E0
CT=B38E39322245C4A2E170EBF36D764AD5
PT=E7556255602B48D7CAC710A004FFFD2E

I=231
KEY=301DDD4551F7117B601A91A7459170F8
IV=301C2808833418F45D5F97FB09A99504
CT=E7556255602B48D7CAC710A004FFFD2E
PT=D4356D51B1C486AAC8BCF2F65312758C

I=232
KEY=E428B014E03397D1A8A6635116830574
IV=264D556FA9A9D73510F562BBF983A4F7
CT=D4356D51B1C486AAC8BCF2F65312758C
PT=B4D003D6C542B576A3EAEC8F1B017F5B

I=233
KEY=50F8B3C2257122A70B4C8FDE0D827A2F
IV=8821647AD3E9E529909E60D0F75DA400
CT=B4D003D6C542B576A3EAEC8F1B017F5B
PT=10056819F314FF21114BE6209AB5508C

I=234
KEY=40FDDBDBD665DD861A0769FE97372AA3
IV=CBD574F4DB1EFEE4E2EC9485D1192E04
CT=10056819F314FF21114BE6209AB5508C
PT=6E7AA061FE85B6F8972CB0E793BE0CB0

I=235
KEY=2E877BBA28E06B7E8D2BD91904892613
IV=4F98EE1501681EC0E7834FE941DF6C6B
CT=6E7AA061FE85B6F8972CB0E793BE0CB0
PT=45E0C0014219D77686E82E74E60608CA

I=236
KEY=6B67BBBB6AF9BC080BC3F76DE28F2ED9
IV=1D34CAB2FF836A7E6568823784519170
CT=45E0C0014219D77686E82E74E60608CA
PT=ACF686F9D6723EAFDBFE512AB3A8C6B1

I=237
KEY=C7913D42BC8B82A7D03DA6475127E868
IV=BEA74B4435F9F79D39B8CCE1F090D4B8
CT=ACF686F9D6723EAFDBFE512AB3A8C6B1
PT=F11FE34D158D40DEA10033BCE0253F30

I=238
KEY=368EDE0FA906C279713D95FBB102D758
IV=DF30EE5D8CA298BE630DDE4B6D8AD7AA
CT=F11FE34D158D40DEA10033BCE0253F30
PT=4E970ACAB8B19C942C85FAA7825C5E67

I=239
KEY=7819D4C511B75EED5DB86F5C335E893F
IV=DC3CCC928C02710EE91391032A3B557D
CT=4E970ACAB8B19C942C85FAA7825C5E67
PT=0FF6C292507CF135036E04A391210466

I=240
KEY=77EF165741CBAFD85ED66BFFA27F8D59
IV=7CC57A486E05E36F852278450C03D39F
CT=0FF6C292507CF135036E04A391210466
PT=BBEF94C8179E7A124492A252507D58E7

I=241
KEY=CC00829F5655D5CA1A44C9ADF202D5BE
IV=BBB92F65197F8D6B388191E78E90A858
CT=BBEF94C8179E7A124492A252507D58E7
PT=BAF11150DD418ED7C28072A718815954

I=242
KEY=76F193CF8B145B1DD8C4BB0AEA838CEA
IV=99DA5A29F934BE23EB74B6DE4E8CC740
CT=BAF11150DD418ED7C28072A718815954
PT=A935099B91A35655C850939D2FED1569

I=243
KEY=DFC49A541AB70D4810942897C56E9983
IV=92401B7E92A49FA5A6905E51E49D51C2
CT=A935099B91A35655C850939D2FED1569
PT=F60E1B5D10CDA17FC762798B86B8708D

I=244
KEY=29CA81090A7AAC37D7F6511C43D6E90E
IV=C2A57521674EB0FB3E46DE2CC1DCF419
CT=F60E1B5D10CDA17FC762798B86B8708D
PT=0B976079B61F46E7F3CCCD17E23D872F

I=245
KEY=225DE170BC65EAD0243A9C0BA1EB6E21
IV=66970AE807FED1E3581496C9FBE5FF50
CT=0B976079B61F46E7F3CCCD17E23D872F
PT=FB44F0094A5FC03BD0A27391E58D0AA3

I=246
KEY=D9191179F63A2AEBF498EF9A44666482
IV=ED7B2ED1AFF7DB9B514D6E4DFAEA484B
CT=FB44F0094A5FC03BD0A27391E58D0AA3
PT=41A534A69AD25FEB39BB9F754F15D8FE

I=247
KEY=98BC25DF6CE87500CD2370EF0B73BC7C
IV=B7E8C7F08BCE1CF1CC2568E18C18C92B
CT=41A534A69AD25FEB39BB9F754F15D8FE
PT=022B28D4F7355E14F99F654B6C35BC16

I=248
KEY=9A970D0B9BDD2B1434BC15A46746006A
IV=F2EF32464FC7E5F3D397C259CB963B7F
CT=022B28D4F7355E14F99F654B6C35BC16
PT=28F18EA03ADEB36C40FD8FBC590EC6C8

I=249
KEY=B26683ABA103987874419A183E48C6A2
IV=687E822CCFC5800E88E59ED7DEC1409D
CT=28F18EA03ADEB36C40FD8FBC590EC6C8
PT=661F9AEA300100333D64FCDD1212C663

I=250
KEY=D47919419102984B492566C52C5A00C1
IV=3922BB08EE1B4CC75881D1D6F66B8EE0
CT=661F9AEA300100333D64FCDD1212C663
PT=11C4D360F585EDF544D5479AE1ABDBE6

I=251
KEY=C5BDCA21648775BE0DF0215FCDF1DB27
IV=9B7C0D7D0EA8CAD77B331F2E0CA1E639
CT=11C4D360F585EDF544D5479AE1ABDBE6
PT=2E4695733B01CC78993E6EB464784223

I=252
KEY=EBFB5F525F86B9C694CE4FEBA9899904
IV=A2BF273F12FE1F4B2F56A86CAFD9C5BF
CT=2E4695733B01CC78993E6EB464784223
PT=1CC514E9F4B13FFEF3B076DAC8C62877

I=253
KEY=F73E4BBBAB378638677E3931614FB173
IV=1D4960EBE667AC5DE5ED767183838846
CT=1CC514E9F4B13FFEF3B076DAC8C62877
PT=3B1A7D179DDED66A37F3053B8B231EEA

I=254
KEY=CC2436AC36E95052508D3C0AEA6CAF99
IV=974643FDFDB695A44B7EF8676A2B87DD
CT=3B1A7D179DDED66A37F3053B8B231EEA
PT=620A4E7A25E8D05779C0FB8CEBA55A74

I=255
KEY=AE2E78D613018005294DC78601C9F5ED
IV=C9A1D92DB7D553DCAEB3BB81B489257A
CT=620A4E7A25E8D05779C0FB8CEBA55A74
PT=93508503278ECECE65D67B5F50962C4E

I=256
KEY=3D7EFDD5348F4ECB4C9BBCD9515FD9A3
IV=108E736F1DCD178A8648F6AA6D527732
CT=93508503278ECECE65D67B5F50962C4E
PT=EA9E953D8B62FFAFF9FAFB97F614D65C

I=257
KEY=D7E068E8BFEDB164B561474EA74B0FFF
IV=B893A94302C0DAFB19D7B6656085DCA9
CT=EA9E953D8B62FFAFF9FAFB97F614D65C
PT=1452ED26E774029EF30D272F8548C2F8

I=258
KEY=C3B285CE5899B3FA466C60612203CD07
IV=668FF6719E5A29A99876227EEDE5818F
CT=1452ED26E774029EF30D272F8548C2F8
PT=4083167E5ED1A1A449C1825B84F321E1

I=259
KEY=833193B00648125E0FADE23AA6F0ECE6
IV=9F6375A022543AB8303368A235813F02
CT=4083167E5ED1A1A449C1825B84F321E1
PT=9206639D076685FD2078F9B984B7A03F

I=260
KEY=1137F02D012E97A32FD51B8322474CD9
IV=1B95128E90B110AED9D543D2A996271F
CT=9206639D076685FD2078F9B984B7A03F
PT=AAA3F2BD0126DFD5007DD2835CCB0109

I=261
KEY=BB940290000848762FA8C9007E8C4DD0
IV=BB2E26A9CDEF9D1B2940EFFBA1971A5D
CT=AAA3F2BD0126DFD5007DD2835CCB0109
PT=B58BB3761519C1726D75AE96313FB7DE

I=262
KEY=0E1FB1E61511890442DD67964FB3FA0E
IV=68D3EC37EE43E3F24072F69F2DFA768D
CT=B58BB3761519C1726D75AE96313FB7DE
PT=4558CD47C012B7293396D7EDFE3CEA85

I=263
KEY=4B477CA1D5033E2D714BB07BB18F108B
IV=F7E107DBF4DDBF7974B699536A4E2803
CT=4558CD47C012B7293396D7EDFE3CEA85
PT=6F43B6B28AEF5DD1B66279ACAAB9A589

I=264
KEY=2404CA135FEC63FCC729C9D71B36B502
IV=B3E6DA8DE5DDF7ED6F832D1306A1C04A
CT=6F43B6B28AEF5DD1B66279ACAAB9A589
PT=1BA743E148BC164C4C8324D665B5A3BD

I=265
KEY=3FA389F2175075B08BAAED017E8316BF
IV=8BBBB56B39893C8F286561EF6E47E66A
CT=1BA743E148BC164C4C8324D665B5A3BD
PT=49A40365092DD1360D916AAEAFD03FC1

I=266
KEY=76078A971E7DA486863B87AFD153297E
IV=48BA08703A306F772E78F660173A26B3
CT=49A40365092DD1360D916AAEAFD03FC1
PT=DF7DEB5968D683AAA274B0D5D3F9AA03

I=267
KEY=A97A61CE76AB272C244F377A02AA837D
IV=0C7832F9CDF65DADA7FA809B8DE709C3
CT=DF7DEB5968D683AAA274B0D5D3F9AA03
PT=69689E97E9B166DA53E25BDFBCAF4A97

I=268
KEY=C012FF599F1A41F677AD6CA5BE05C9EA
IV=05BC03F59CD8C5FAE4A7240E6168CF41
CT=69689E97E9B166DA53E25BDFBCAF4A97
PT=5F87DC71EF2918F75A94E43007FF50E6

I=269
KEY=9F952328703359012D398895B9FA990C
IV=4233E654802B55155E158B650B64718E
CT=5F87DC71EF2918F75A94E43007FF50E6
PT=4F170790775A6A147E99D251C401969A

I=270
KEY=D08224B80769331553A05AC47DFB0F96
IV=4AEAE53003D5085F5963CC36C4589941
CT=4F170790775A6A147E99D251C401969A
PT=E566D46360EF38E5CFF0C2E7E7E0D679

I=271
KEY=35E4F0DB67860BF09C5098239A1BD9EF
IV=F6CBFF51F6E59B7AAA9EB1261707429A
CT=E566D46360EF38E5CFF0C2E7E7E0D679
PT=F7189C4B9E1048ED66D9B7CCD089C5F9

I=272
KEY=C2FC6C90F996431DFA892FEF4A921C16
IV=0A05A4BFBBCC8B3E218C123A95FFF475
CT=F7189C4B9E1048ED66D9B7CCD089C5F9
PT=68FB0909789DBFA672BEEF5DF0017B63

I=273
KEY=AA076599810BFCBB8837C0B2BA936775
IV=DF1A3AF147E72C2CEAE5D8713D52F572
CT=68FB0909789DBFA672BEEF5DF0017B63
PT=AF2784590089186BED3A48ADC1B68562

I=274
KEY=0520E1C08182E4D0650D881F7B25E217
IV=BFDC5ECF596F5F3CFFEAAF4B961209FA
CT=AF2784590089186BED3A48ADC1B68562
PT=E8C29D3C129AB8FA797EFDCE4CE4D4F2

I=275
KEY=EDE27CFC93185C2A1C7375D137C136E5
IV=49195E71516D5117C26814F98FD94C31
CT=E8C29D3C129AB8FA797EFDCE4CE4D4F2
PT=6ACF2AE12C38DB520F49E6C90D4B8F03

I=276
KEY=872D561DBF208778133A93183A8AB9E6
IV=E4FA7788D6E7AFBD08884BFCFA74D30C
CT=6ACF2AE12C38DB520F49E6C90D4B8F03
PT=78FEA3B57D6176C44CE753FB01416CA6

I=277
KEY=FFD3F5A8C241F1BC5FDDC0E33BCBD540
IV=08E333F43615229D95E7C425A04EA2B7
CT=78FEA3B57D6176C44CE753FB01416CA6
PT=5030D7131ABBD7AC323BA31B602B776E

I=278
KEY=AFE322BBD8FA26106DE663F85BE0A22E
IV=64BDE35106D9A66A4216EA20F6C52D06
CT=5030D7131ABBD7AC323BA31B602B776E
PT=14B32E8BD3BD9D82613E32536E6A9F91

I=279
KEY=BB500C300B47BB920CD851AB358A3DBF
IV=8FB9E6BDCD50610E97CF731216CFE698
CT=14B32E8BD3BD9D82613E32536E6A9F91
PT=1F22542AFCE63AA92CCDAF45F8DAEEF0

I=280
KEY=A472581AF7A1813B2015FEEECD50D34F
IV=E18B3C5EDA8CDA9E94A376CEEF687D9F
CT=1F22542AFCE63AA92CCDAF45F8DAEEF0
PT=FC986510E3ED40994D1C33201D0EDB27

I=281
KEY=58EA3D0A144CC1A26D09CDCED05E0868
IV=D1828C9DDB263064AD259A2360B3DFBA
CT=FC986510E3ED40994D1C33201D0EDB27
PT=06C1CE557C9AC5D40560CF2D40DCC47C

I=282
KEY=5E2BF35F68D60476686902E39082CC14
IV=3454FD0698CC8CD949970509C23E50F6
CT=06C1CE557C9AC5D40560CF2D40DCC47C
PT=4287FA420FD9AFCAB7E290273BDF4054

I=283
KEY=1CAC091D670FABBCDF8B92C4AB5D8C40
IV=C2F0C044A975ADBD20DE56858A074886
CT=4287FA420FD9AFCAB7E290273BDF4054
PT=447AFDD3ACF0DEEB0385799789FBAFF0

I=284
KEY=58D6F4CECBFF7557DC0EEB5322A623B0
IV=299C337821B0779A7ADC3A04316A6373
CT=447AFDD3ACF0DEEB0385799789FBAFF0
PT=BA0BBC980827E87EEA51E66C92C96B31

I=285
KEY=E2DD4856C3D89D29365F0D3FB06F4881
IV=2368291AD179C3C143E636898EB3E104
CT=BA0BBC980827E87EEA51E66C92C96B31
PT=77A6131415B251049423FFFB1F3A8A3A

I=286
KEY=957B5B42D66ACC2DA27CF2C4AF55C2BB
IV=14300EA81B57E102AEDD97CCD466E18A
CT=77A6131415B251049423FFFB1F3A8A3A
PT=A7AEC0CC599D56347A72D934DDAC7899

I=287
KEY=32D59B8E8FF79A19D80E2BF072F9BA22
IV=FD62415C90DACCD22EC364877DB0CB24
CT=A7AEC0CC599D56347A72D934DDAC7899
PT=1B1100FCB9F85D8CCFDEC5865083B44D

I=288
KEY=29C49B72360FC79517D0EE76227A0E6F
IV=6CB24892D26E43E63819C372D5EB4F6D
CT=1B1100FCB9F85D8CCFDEC5865083B44D
PT=33E9BBD018604F301C2543878FCCCA5C

I=289
KEY=1A2D20A22E6F88A50BF5ADF1ADB6C433
IV=50D84A03998B897D1AE217FEA9A955C4
CT=33E9BBD018604F301C2543878FCCCA5C
PT=07163D652DB035C091830042D6047DF3

I=290
KEY=1D3B1DC703DFBD659A76ADB37BB2B9C0
IV=092DF26C855F07690836203E3E1790D0
CT=07163D652DB035C091830042D6047DF3
PT=AD6647AF8EFCD4F3B2ECDB4D6FFFC1EB

I=291
KEY=B05D5A688D236996289A76FE144D782B
IV=68634C941957E3F58F71BE37D7C5C78B
CT=AD6647AF8EFCD4F3B2ECDB4D6FFFC1EB
PT=D2B51122154D10F67D2C07BD75D5E8E2

I=292
KEY=62E84B4A986E796055B67143619890C9
IV=3C85044F6438FAA021BB7985A63A959D
CT=D2B51122154D10F67D2C07BD75D5E8E2
PT=4131E9296FECD2DF35EDFBF0CBD0E54B

I=293
KEY=23D9A263F782ABBF605B8AB3AA487582
IV=46D47F3F12B6ADD5580CE8CE7C7359CD
CT=4131E9296FECD2DF35EDFBF0CBD0E54B
PT=5AEA3E73A6EC0D9D683A3526AC1C0E58

I=294
KEY=79339C10516EA6220861BF9506547BDA
IV=D3C58B190FDA0838C770C2185C553D97
CT=5AEA3E73A6EC0D9D683A3526AC1C0E58
PT=A21B17CB4588C30CAD6DF25A5E83E273

I=295
KEY=DB288BDB14E6652EA50C4DCF58D799A9
IV=870E7F0E3109712E84C58BEC0E7032F6
CT=A21B17CB4588C30CAD6DF25A5E83E273
PT=0021FF3BB8336E607B8DE10117005230

I=296
KEY=DB0974E0ACD50B4EDE81ACCE4FD7CB99
IV=2FC82B298A60FC40DA3EFD4E800DB063
CT=0021FF3BB8336E607B8DE10117005230
PT=2EFA32680ECD84891B447393F7C1AC88

I=297
KEY=F5F34688A2188FC7C5C5DF5DB8166711
IV=E1B86C9FC466B72BE40F65973FC41FC4
CT=2EFA32680ECD84891B447393F7C1AC88
PT=722117C1EE83FBA840EFEE86A930579F

I=298
KEY=87D251494C9B746F852A31DB1126308E
IV=48E347A705B6E322153BFE567931346F
CT=722117C1EE83FBA840EFEE86A930579F
PT=A684344231EC4C03726A0DC20EADA36D

I=299
KEY=2156650B7D77386CF7403C191F8B93E3
IV=4CD618E5F959FC4C66602263F6C5C652
CT=A684344231EC4C03726A0DC20EADA36D
PT=F40BFBEAC1C1BD1621FBAE1B605BD092

I=300
KEY=D55D9EE1BCB6857AD6BB92027FD04371
IV=5A41E8952E6B1C551C4691DE9C0A4B6D
CT=F40BFBEAC1C1BD1621FBAE1B605BD092
PT=6ECDBFB3947C7E81C7BD4BE93B603728

I=301
KEY=BB90215228CAFBFB1106D9EB44B07459
IV=52E2726B0855F993E1075077B3AB1E77
CT=6ECDBFB3947C7E81C7BD4BE93B603728
PT=71E051375A3B974FE523DA47862E24AE

I=302
KEY=CA70706572F16CB4F42503ACC29E50F7
IV=76B6637161FBE4760FCE90E9868B441A
CT=71E051375A3B974FE523DA47862E24AE
PT=CC6CFBCF9A62A3AE55F6B040673B25B3

I=303
KEY=061C8BAAE893CF1AA1D3B3ECA5A57544
IV=5563B0C2350A4ED76EE0C322480884FB
CT=CC6CFBCF9A62A3AE55F6B040673B25B3
PT=DA71B26C4EE3FB1FD413FD0757B379E6

I=304
KEY=DC6D39C6A670340575C04EEBF2160CA2
IV=45C1C5DAF8DE9170C01E554DD36C564E
CT=DA71B26C4EE3FB1FD413FD0757B379E6
PT=BCD4BC394FFB46207EF0A8EC9883BD55

I=305
KEY=60B985FFE98B72250B30E6076A95B1F7
IV=38E5B69E6D82A760BFD65CBAACC56F8A
CT=BCD4BC394FFB46207EF0A8EC9883BD55
PT=FD997E7A7B77E5DF637182399779AF75

I=306
KEY=9D20FB8592FC97FA6841643EFDEC1E82
IV=8EB39B261910419E43B178DCD6803572
CT=FD997E7A7B77E5DF637182399779AF75
PT=71749C51136457B604CD8B4C193928A8

I=307
KEY=EC5467D48198C04C6C8CEF72E4D5362A
IV=6DFA447305215066A0B5671E5AB80951
CT=71749C51136457B604CD8B4C193928A8
PT=DF48D76E54B4264304B0EC036DFDADDE

I=308
KEY=331CB0BAD52CE60F683C037189289BF4
IV=8C2B316ECA96DB57322853E07BE47AB3
CT=DF48D76E54B4264304B0EC036DFDADDE
PT=5E2BD374B9856C573A4DEBF1B8F453EC

I=309
KEY=6D3763CE6CA98A585271E88031DCC818
IV=E541AC59CFDBEA97D1AE5221F849838E
CT=5E2BD374B9856C573A4DEBF1B8F453EC
PT=D7F97B76A8D3076D343A22459EC4765D

I=310
KEY=BACE18B8C47A8D35664BCAC5AF18BE45
IV=B359889A22B9502571E4817C47BC5348
CT=D7F97B76A8D3076D343A22459EC4765D
PT=E24F05EEE812BD3389AC0AAC4172F5C6

I=311
KEY=58811D562C683006EFE7C069EE6A4B83
IV=33F8D881A783051987103096B6C734C3
CT=E24F05EEE812BD3389AC0AAC4172F5C6
PT=A2AB44CC9C41A8748201DA6F9429AB39

I=312
KEY=FA2A599AB02998726DE61A067A43E0BA
IV=2028704442AE252BCF1D081C5EE2D80B
CT=A2AB44CC9C41A8748201DA6F9429AB39
PT=E8FA100D1D28186196BAF990A80109AA

I=313
KEY=12D04997AD018013FB5CE396D242E910
IV=6384E3612062BC6850C94C3B14934A6F
CT=E8FA100D1D28186196BAF990A80109AA
PT=683AEF47A8990E2F8AAD8813789069B1

I=314
KEY=7AEAA6D005988E3C71F16B85AAD280A1
IV=12F78728F3B0EFAC7D304257E47ED5C8
CT=683AEF47A8990E2F8AAD8813789069B1
PT=F3121C95A5BACDEAEFA87C4EE40B6C6D

I=315
KEY=89F8BA45A02243D69E5917CB4ED9ECCC
IV=FE654BC69677E13383C4206BFB7896A6
CT=F3121C95A5BACDEAEFA87C4EE40B6C6D
PT=AE9701378E128847A2669B516795D586

I=316
KEY=276FBB722E30CB913C3F8C9A294C394A
IV=E79D5C468BCBA36A79B74F32F28FA635
CT=AE9701378E128847A2669B516795D586
PT=D97D988A40C5B323794F160FE8639262

I=317
KEY=FE1223F86EF578B245709A95C12FAB28
IV=4F3055450C627AD4712E042666AC0812
CT=D97D988A40C5B323794F160FE8639262
PT=32381BA55D5C0C91D683D57593772496

I=318
KEY=CC2A385D33A9742393F34FE052588FBE
IV=172165B468B83B62B41A2B4310461193
CT=32381BA55D5C0C91D683D57593772496
PT=4D555F31A60F1007B96FBB458B06A619

I=319
KEY=817F676C95A664242A9CF4A5D95E29A7
IV=5D5EEBEA67DCD5F7851619F1BAA6414E
CT=4D555F31A60F1007B96FBB458B06A619
PT=23D1AC0F18C20FCE7E5CC4619F8376D5

I=320
KEY=A2AECB638D646BEA54C030C446DD5F72
IV=54154B82B5820D5892DABD15B403345D
CT=23D1AC0F18C20FCE7E5CC4619F8376D5
PT=2E55905A38F818CC04D8A792D2007332

I=321
KEY=8CFB5B39B59C73265018975694DD2C40
IV=AC47D7619A587DC8213B749918FCFEA8
CT=2E55905A38F818CC04D8A792D2007332
PT=010F3396885F8F6BF15170B8E6D221BD

I=322
KEY=8DF468AF3DC3FC4DA149E7EE720F0DFD
IV=6CE249D2E89C61ACD029AA3C122427D1
CT=010F3396885F8F6BF15170B8E6D221BD
PT=4DB3972D9851B8A4B3150A43210D36F0

I=323
KEY=C047FF82A59244E9125CEDAD53023B0D
IV=07A578889914E113C1C4DE01DC4B1DE4
CT=4DB3972D9851B8A4B3150A43210D36F0
PT=84F64CFFD705E82704FF3DA4FC074B34

I=324
KEY=44B1B37D7297ACCE16A3D009AF057039
IV=CE73ADCD7BCF1807C084276EDD765256
CT=84F64CFFD705E82704FF3DA4FC074B34
PT=5964DF8E8D656846EE574635787F7B18

I=325
KEY=1DD56CF3FFF2C488F8F4963CD77A0B21
IV=679903574013011E62CC5798B9633607
CT=5964DF8E8D656846EE574635787F7B18
PT=E28E7E3AA93802BEEFE258F38F5D2E5C

I=326
KEY=FF5B12C956CAC6361716CECF5827257D
IV=678D8583A022C60BCE2C227627A548B2
CT=E28E7E3AA93802BEEFE258F38F5D2E5C
PT=D297684A7C50765C413BFA37B5E1E256

I=327
KEY=2DCC7A832A9AB06A562D34F8EDC6C72B
IV=4EF9AACC21733B65B4DD0B1F105D4037
CT=D297684A7C50765C413BFA37B5E1E256
PT=9EF5F9A2B14B090A201DE74E4CF714C8

I=328
KEY=B33983219BD1B9607630D3B6A131D3E3
IV=5CC41FCC9EC19619BA06D10BBC6E1F5F
CT=9EF5F9A2B14B090A201DE74E4CF714C8
PT=81EF000CF6CA737E0F449927E66F6392

I=329
KEY=32D6832D6D1BCA1E79744A91475EB071
IV=6DE60CE2986CF80C5D588FA1F9A8C3A0
CT=81EF000CF6CA737E0F449927E66F6392
PT=3BE177EE9F8F7BE3BCE0FAA70C0A5025

I=330
KEY=0937F4C3F294B1FDC594B0364B54E054
IV=BE2F86A52BEED4497790D6134CF0D64E
CT=3BE177EE9F8F7BE3BCE0FAA70C0A5025
PT=DD313134DB126529F349808B3E766795

I=331
KEY=D406C5F72986D4D436DD30BD752287C1
IV=A89E52478FDEF91AF151097CED84579C
CT=DD313134DB126529F349808B3E766795
PT=330203EA36484C8E091F00CD2255E599

I=332
KEY=E704C61D1FCE985A3FC2307057776258
IV=1AEF7BA63A3E3B6D2EB8EB1980D3E581
CT=330203EA36484C8E091F00CD2255E599
PT=E28A47CDB7DB0351006C1D9FB227FCEF

I=333
KEY=058E81D0A8159B0B3FAE2DEFE5509EB7
IV=F67C7FCF8C5AA1574A165D8CBE694BF4
CT=E28A47CDB7DB0351006C1D9FB227FCEF
PT=ACF869797690853BE142BB7BAC735FC8

I=334
KEY=A976E8A9DE851E30DEEC96944923C17F
IV=636F181C4735E700897928BFC60A43EB
CT=ACF869797690853BE142BB7BAC735FC8
PT=60FF9BF36809DAEB94F823FC6458B26A

I=335
KEY=C989735AB68CC4DB4A14B5682D7B7315
IV=0829C0EE6B170C490C7A73401761CEAA
CT=60FF9BF36809DAEB94F823FC6458B26A
PT=17630CBEAB904F7996301515F24A1F27

I=336
KEY=DEEA7FE41D1C8BA2DC24A07DDF316C32
IV=A4E0624D86F1098D2DD4D0408897ACED
CT=17630CBEAB904F7996301515F24A1F27
PT=A1E35C3E2FA1B1DB11DB96E7DD58AB37

I=337
KEY=7F0923DA32BD3A79CDFF369A0269C705
IV=D3639A296FF168ED773785CFCACD1015
CT=A1E35C3E2FA1B1DB11DB96E7DD58AB37
PT=14F5408460708A69A22A2023B01969CE

I=338
KEY=6BFC635E52CDB0106FD516B9B270AECB
IV=37CD0428A0C08CC6AABAAA05D8830557
CT=14F5408460708A69A22A2023B01969CE
PT=516991CA6530D20AB121394F51288880

I=339
KEY=3A95F29437FD621ADEF42FF6E358264B
IV=FB52F5A95F2E3665D44F0AEACB88B108
CT=516991CA6530D20AB121394F51288880
PT=FB0C32B66729736F8E5EC8C92D80FDC5

I=340
KEY=C199C02250D4117550AAE73FCED8DB8E
IV=403B43C740215D94E8FBAE22846D3B0A
CT=FB0C32B66729736F8E5EC8C92D80FDC5
PT=C243054D022E7E9EFC21F965C71D63EB

I=341
KEY=03DAC56F52FA6FEBAC8B1E5A09C5B865
IV=663796B211E7F6E05B4A107B16A904B0
CT=C243054D022E7E9EFC21F965C71D63EB
PT=6CA56F6792355B2A7A60A4467A85F71E

I=342
KEY=6F7FAA08C0CF34C1D6EBBA1C73404F7B
IV=A23B73DF1477EC98202135164C6EF1AB
CT=6CA56F6792355B2A7A60A4467A85F71E
PT=BC590C8B427B013F873B2F5F8D66D5D1

I=343
KEY=D326A68382B435FE51D09543FE269AAA
IV=E84DF462C62D594AAD94C33C446E18FC
CT=BC590C8B427B013F873B2F5F8D66D5D1
PT=D37CAA9BEA300FF6219349DBA25DA0A6

I=344
KEY=005A0C1868843A087043DC985C7B3A0C
IV=9AAD322884EBE93D0FA084E144600174
CT=D37CAA9BEA300FF6219349DBA25DA0A6
PT=AE50C328AFFC39A103C767C1FD337449

I=345
KEY=AE0ACF30C77803A97384BB59A1484E45
IV=9CEE91362FB0D50ABCEB58EF49EC3614
CT=AE50C328AFFC39A103C767C1FD337449
PT=882939453C6E156CD023B468EDD16C95

I=346
KEY=2623F675FB1616C5A3A70F314C9922D0
IV=5626126CC424BF26D92C822442E4221E
CT=882939453C6E156CD023B468EDD16C95
PT=B9A97FE0DEF66604F6CE928AAB532EC2

I=347
KEY=9F8A899525E070C155699DBBE7CA0C12
IV=E5BB374C14E36FD0C2D5F9191C967EC7
CT=B9A97FE0DEF66604F6CE928AAB532EC2
PT=6ED0FF2FE8DA900A33AE20F7C1C2E457

I=348
KEY=F15A76BACD3AE0CB66C7BD4C2608E845
IV=4493B9F0FF63D392273302B6932C8F77
CT=6ED0FF2FE8DA900A33AE20F7C1C2E457
PT=B7583A56BED55067A520B4F5E5C1528F

I=349
KEY=46024CEC73EFB0ACC3E709B9C3C9BACA
IV=12E2BFD3EAB69E5132E3CDF8BAFBD45E
CT=B7583A56BED55067A520B4F5E5C1528F
PT=28874485575E292B0C83E1B568019BF0

I=350
KEY=6E85086924B19987CF64E80CABC8213A
IV=3CEFDC726C919CA265CD6C28A05DE542
CT=28874485575E292B0C83E1B568019BF0
PT=A35D16BBB96BF3F9E06D96EDA9BB44BB

I=351
KEY=CDD81ED29DDA6A7E2F097EE102736581
IV=4AE79901700F018D1E5D95A9FAD762F2
CT=A35D16BBB96BF3F9E06D96EDA9BB44BB
PT=367E6880D369D482E7E64CFCD1BCC870

I=352
KEY=FBA676524EB3BEFCC8EF321DD3CFADF1
IV=52EEDB058A8E1FE7C651D484F86F941F
CT=367E6880D369D482E7E64CFCD1BCC870
PT=32B33E97C94953392A738A964DB121FC

I=353
KEY=C91548C587FAEDC5E29CB88B9E7E8C0D
IV=B3704291187E9D55901DD237EC1E2F24
CT=32B33E97C94953392A738A964DB121FC
PT=965A9A0C0287AB8535436E95CC9BFB99

I=354
KEY=5F4FD2C9857D4640D7DFD61E52E57794
IV=B3878FC256FA4F9ED0BA05FF51C0CE2A
CT=965A9A0C0287AB8535436E95CC9BFB99
PT=367A5C93099ACF72353DAC320A3B766E

I=355
KEY=69358E5A8CE78932E2E27A2C58DE01FA
IV=774EA78AECAF5A1D7CC1785DE91CAB01
CT=367A5C93099ACF72353DAC320A3B766E
PT=116EDEB1D0E6F8D1BB58AF648429B792

I=356
KEY=785B50EB5C0171E359BAD548DCF7B668
IV=FB1EC4218C5F733A54DEA7F753F45D68
CT=116EDEB1D0E6F8D1BB58AF648429B792
PT=EF3C679822E3A906F380CD07EBE65F09

I=357
KEY=976737737EE2D8E5AA3A184F3711E961
IV=8BBE09AFE41C5A78EBA0524212AC570F
CT=EF3C679822E3A906F380CD07EBE65F09
PT=291B066A30A3B11681D414A4BB5311D5

I=358
KEY=BE7C31194E4169F32BEE0CEB8C42F8B4
IV=4E8F651C06778A964AA7D43999E41FD6
CT=291B066A30A3B11681D414A4BB5311D5
PT=764F8FD22FE0BD98852F259EA6DC6F7C

I=359
KEY=C833BECB61A1D46BAEC129752A9E97C8
IV=E90A35007EE7F001D1F5982BF2477C82
CT=764F8FD22FE0BD98852F259EA6DC6F7C
PT=66016373A6B1A3797A1B2D2B19B73126

I=360
KEY=AE32DDB8C7107712D4DA045E3329A6EE
IV=46FF289808B1AF6B290CB5AF05747085
CT=66016373A6B1A3797A1B2D2B19B73126
PT=3367064ECF5CA978001887DBDE40DE5A

I=361
KEY=9D55DBF6084CDE6AD4C28385ED6978B4
IV=9BAC8819E1CA613CFAA730FBC2034168
CT=3367064ECF5CA978001887DBDE40DE5A
PT=5C4F01D47C9DA94DE95FC26763CC1146

I=362
KEY=C11ADA2274D177273D9D41E28EA569F2
IV=A5CE34A846E1D0A737963C49FAD6378E
CT=5C4F01D47C9DA94DE95FC26763CC1146
PT=D41D854E42863F161E76A2397020C321

I=363
KEY=15075F6C3657483123EBE3DBFE85AAD3
IV=2CC823722DD3E8F15035875ED43513E2
CT=D41D854E42863F161E76A2397020C321
PT=1C542DC0B54A053471AF583C909AB3C4

I=364
KEY=095372AC831D4D055244BBE76E1F1917
IV=9799D03B6AD46BECD981279B35DF707C
CT=1C542DC0B54A053471AF583C909AB3C4
PT=958D0254D7AFE1E678A9844DF4F4D18D

I=365
KEY=9CDE70F854B2ACE32AED3FAA9AEBC89A
IV=E0E401C7BA5E319F0D60AB765CBEE176
CT=958D0254D7AFE1E678A9844DF4F4D18D
PT=C54888169B30587E732BD7213EE5B1A3

I=366
KEY=5996F8EECF82F49D59C6E88BA40E7939
IV=711DA6521732E6868111DCC1C835F9DF
CT=C54888169B30587E732BD7213EE5B1A3
PT=446F234FCADA809A8B2E167A86C9EB36

I=367
KEY=1DF9DBA105587407D2E8FEF122C7920F
IV=B7008BC907C546A56743C292185C2B02
CT=446F234FCADA809A8B2E167A86C9EB36
PT=6AD93BC7E18B48810664CD1C528BA6C4

I=368
KEY=7720E066E4D33C86D48C33ED704C34CB
IV=4D7DD98183E56F71D72DB76187FC5F95
CT=6AD93BC7E18B48810664CD1C528BA6C4
PT=0D75BD2B8683FB819EB7F71E2E0836A7

I=369
KEY=7A555D4D6250C7074A3BC4F35E44026C
IV=F4DCFB73BBE7EB5348D56F81BCBC0971
CT=0D75BD2B8683FB819EB7F71E2E0836A7
PT=8CBE14862E93FE011A4F45518C7258FA

I=370
KEY=F6EB49CB4CC33906507481A2D2365A96
IV=5432FBF1480AA398F09CA5AF375FA394
CT=8CBE14862E93FE011A4F45518C7258FA
PT=4026D669905762F0790B6497646A2D4D

I=371
KEY=B6CD9FA2DC945BF6297FE535B65C77DB
IV=CA462DC9F029A8311CA45A4116DADC1D
CT=4026D669905762F0790B6497646A2D4D
PT=A8CD2E187CF274756E5736D420853435

I=372
KEY=1E00B1BAA0662F834728D3E196D943EE
IV=4FC9E7688352F4700C5F380BFD1F4FB2
CT=A8CD2E187CF274756E5736D420853435
PT=A99410B60322340E279C87D3D26793AA

I=373
KEY=B794A10CA3441B8D60B4543244BED044
IV=4D90B1A30EB7E957FAE3DC8C132F4F42
CT=A99410B60322340E279C87D3D26793AA
PT=D136F990196C667EC35E3A0492C39968

I=374
KEY=66A2589CBA287DF3A3EA6E36D67D492C
IV=6FAD87F358F0C702C5E9E599093219ED
CT=D136F990196C667EC35E3A0492C39968
PT=8CE49C1BAACAB69432DDA2FF4250C7B2

I=375
KEY=EA46C48710E2CB679137CCC9942D8E9E
IV=2B6FF495B7E482360E31949C7C9B2E2B
CT=8CE49C1BAACAB69432DDA2FF4250C7B2
PT=7942B80F57EA24FA200451B37899F9E2

I=376
KEY=93047C884708EF9DB1339D7AECB4777C
IV=21FDC4C9B03A1A9A3F87FF9AAC80FAAE
CT=7942B80F57EA24FA200451B37899F9E2
PT=DE156D56E199FDBB2600DBF9FC254003

I=377
KEY=4D1111DEA6911226973346831091377F
IV=387612E399C62916B394A9FC6BC0F059
CT=DE156D56E199FDBB2600DBF9FC254003
PT=C302C494D910640271695324E64B0992

I=378
KEY=8E13D54A7F817624E65A15A7F6DA3EED
IV=34BCABDE4646BCC05CAF7E61CBF1F8D2
CT=C302C494D910640271695324E64B0992
PT=110CE84E5B81867CE4FCCD1BBB5FD792

I=379
KEY=9F1F3D042400F05802A6D8BC4D85E97F
IV=C7898DA9ABF2A3D364C228A83525B04F
CT=110CE84E5B81867CE4FCCD1BBB5FD792
PT=7B94C0E4849AD65291F0F7BD464C544C

I=380
KEY=E48BFDE0A09A260A93562F010BC9BD33
IV=00E079EEBC47AF3ED3796DBF4A2B55FC
CT=7B94C0E4849AD65291F0F7BD464C544C
PT=D6518596BB53E44452393BD4440D805C

I=381
KEY=32DA78761BC9C24EC16F14D54FC43D6F
IV=03E168F8AF2303593E0F936648D253DC
CT=D6518596BB53E44452393BD4440D805C
PT=0C4B51FB33380BB215F29F3FCE7CA0DA

I=382
KEY=3E91298D28F1C9FCD49D8BEA81B89DB5
IV=15E1ECD59BC092C974FA4738C66D9604
CT=0C4B51FB33380BB215F29F3FCE7CA0DA
PT=3FC98D86A558B711807930AAA391C9BF

I=383
KEY=0158A40B8DA97EED54E4BB402229540A
IV=303BD4A407015A9E899816E0FABF288D
CT=3FC98D86A558B711807930AAA391C9BF
PT=85D77D90C312FE938F6C1983014BD572

I=384
KEY=848FD99B4EBB807EDB88A2C323628178
IV=E018FA96FD433CFD162E8E67D4F37015
CT=85D77D90C312FE938F6C1983014BD572
PT=4F49C1740EA88B5767CE36C95158DD63

I=385
KEY=CBC618EF40130B29BC46940A723A5C1B
IV=84013D3E2AA0C77650644CD6862096A7
CT=4F49C1740EA88B5767CE36C95158DD63
PT=E6F1409ABA890AC3A50FC4AAD82E1872

I=386
KEY=2D375875FA9A01EA194950A0AA144469
IV=8BFD137593BFF254414349F1147B9BFF
CT=E6F1409ABA890AC3A50FC4AAD82E1872
PT=66F6E29696CFDC72EE905636614D2A72

I=387
KEY=4BC1BAE36C55DD98F7D90696CB596E1B
IV=8EC295602FB66C4A2DCA65F66B07E010
CT=66F6E29696CFDC72EE905636614D2A72
PT=553B45792AE0625B357C4B68A4BB767F

I=388
KEY=1EFAFF9A46B5BFC3C2A54DFE6FE21864
IV=5999E221475D06295451E52FF88F9F97
CT=553B45792AE0625B357C4B68A4BB767F
PT=FAB6F53166BFDB89DF68D46D423C243E

I=389
KEY=E44C0AAB200A644A1DCD99932DDE3C5A
IV=49D1944CBD7BE00C671B8E996F9A3EB9
CT=FAB6F53166BFDB89DF68D46D423C243E
PT=308DDB3FE0415C46AE450C78FC5F2558

I=390
KEY=D4C1D194C04B380CB38895EBD1811902
IV=9785FEE180794318514967B4199CFB93
CT=308DDB3FE0415C46AE450C78FC5F2558
PT=6659969D1BD493A3851E75981BACCCB0

I=391
KEY=B2984709DB9FABAF3696E073CA2DD5B2
IV=4FD1DAE14DB75AC0E6C94388805AC3B5
CT=6659969D1BD493A3851E75981BACCCB0
PT=B56B950D84DCA131FD8A5E2601037B36

I=392
KEY=07F3D2045F430A9ECB1CBE55CB2EAE84
IV=3ED415E7DDDADD2DC7F0376E6CA42EFE
CT=B56B950D84DCA131FD8A5E2601037B36
PT=8648F40FA13018563E6F1A799D9FCD91

I=393
KEY=81BB260BFE7312C8F573A42C56B16315
IV=FEAF76717C0905A601509BD68C6D08E5
CT=8648F40FA13018563E6F1A799D9FCD91
PT=66DE6D7A74B36C7EE2737DA526F5C8CD

I=394
KEY=E7654B718AC07EB61700D9897044ABD8
IV=2B808C340320DD64AB9278A9D9A3E247
CT=66DE6D7A74B36C7EE2737DA526F5C8CD
PT=9BB0D29CDA21452F60B5897186F3FAFB

I=395
KEY=7CD599ED50E13B9977B550F8F6B75123
IV=D2EB48117E2E9FB2058800DE1A442799
CT=9BB0D29CDA21452F60B5897186F3FAFB
PT=489CDC3A2C1AE7510E03AE36D5EAB0E5

I=396
KEY=344945D77CFBDCC879B6FECE235DE1C6
IV=427765E0671AA6D32634447687E9B209
CT=489CDC3A2C1AE7510E03AE36D5EAB0E5
PT=1D65D0E1235AE47DBCC2C65B59892C4C

I=397
KEY=292C95365FA138B5C57438957AD4CD8A
IV=D75E9F074D965FE902287B3BB1483315
CT=1D65D0E1235AE47DBCC2C65B59892C4C
PT=E75B7AE5A8C3B93F15D0E3EBD04E1B4B

I=398
KEY=CE77EFD3F762818AD0A4DB7EAA9AD6C1
IV=5455FD9A01E04E48CA777CF8185697ED
CT=E75B7AE5A8C3B93F15D0E3EBD04E1B4B
PT=11F4A35F6C225A9DF1CA6BAF92E11B07

I=399
KEY=DF834C8C9B40DB17216EB0D1387BCDC6
IV=3C608F664492626B2208DC92E819411A
CT=11F4A35F6C225A9DF1CA6BAF92E11B07
PT=9B8FB71E035CEFF9CBFA1346E5ACEFE0

