package App::Manoc::Controller::Root;
#ABSTRACT: Root controller for Manon

use Moose;

our $VERSION = '2.99.4'; ##TRIAL VERSION

use namespace::autoclean;

BEGIN { extends 'Catalyst::Controller' }

# Sets the actions in this controller to be registered with no prefix
# so they function identically to actions created in Manoc.pm
#
__PACKAGE__->config( namespace => '' );


sub index : Path : Args(0) {
    my ( $self, $c ) = @_;

    $c->response->redirect( $c->uri_for_action('/search/index') );
    $c->detach();
}


sub auto : Private {
    my ( $self, $c ) = @_;

    # Do not use csrf protection for APIs
    $c->stash->{is_api} and
        $c->stash->{skip_csrf} = 1;

    # Disable csrf in tests
    $c->config->{test_mode} && $ENV{MANOC_SKIP_CSRF} and
        $c->stash->{skip_csrf} = 1;

    ##  XHR detection ##
    if ( my $req_with = $c->req->header('X-Requested-With') ) {
        $c->stash->{is_xhr} = $req_with eq 'XMLHttpRequest';
    }
    else {
        $c->stash->{is_xhr} = 0;
    }
    $c->log->debug( "is_xhr = ", $c->stash->{is_xhr} );

    ## output format selection ##
    if ( my $fmt = $c->req->param('format') ) {
        $fmt eq 'fragment' and $c->stash( no_wrapper => 1 );
        delete $c->req->params->{'format'};
    }

    ## check authentication ##
    if ( !$self->check_auth($c) ) {
        $c->log->debug("Not authenticated") if $c->debug;

        if ( $c->stash->{is_api} || $c->stash->{is_xhr} ) {
            $c->detach('access_denied');
        }
        else {
            $c->response->redirect(
                $c->uri_for_action(
                    '/auth/login',
                    {
                        login_redirect => $c->request->path
                    }
                )
            );
        }
        return 0;
    }

    # CSRF protection
    $c->stash->{skip_csrf} //= 0;
    $c->log->debug( "Manoc root: skip CSRF = ", $c->stash->{skip_csrf} );
    if ( $c->req->method eq 'POST' && !$c->stash->{skip_csrf} ) {
        $c->log->debug("POST method, token validation required");
        $c->require_valid_token();
    }

    return 1;
}


sub check_auth {
    my ( $self, $c ) = @_;

    $c->controller eq $c->controller('Auth') and
        return 1;

    # already authenticated by API controller
    $c->stash->{is_api} and
        return 1;

    # user must be authenticated
    return 0 unless $c->user_exists;

    # users with agent flag can only access API controller
    return 0 if $c->user->agent;

    return 1;
}


sub default : Path {
    my ( $self, $c ) = @_;
    $c->detach('error/http_404');
}


sub message : Path('message') Args(0) {
    my ( $self, $c ) = @_;
    my $page = $c->request->param('page');
    $c->stash( template => 'message.tt' );
}


sub end : ActionClass('RenderView') {
}


sub access_denied : Private {
    my ( $self, $c, $action ) = @_;
    $c->log->debug("Error 403");
    $c->detach('error/http_403');
}

__PACKAGE__->meta->make_immutable;

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::Root - Root controller for Manon

=head1 VERSION

version 2.99.4

=head1 DESCRIPTION

The Root controller is used to implement global actions.

=head1 ACTIONS

=head2 index

The root page (/), redirect to search page.

=head2 auto

Perform CSRF checks for POST requests, sets is_xhr for async request,
check authentication.

=head2 default

Shows 404 error page for Manoc using Error controller

=head2 message

Basic minimal page for showing messages

=head2 end

Attempt to render a view, if needed.

=head2 access_denied

Action called by ACL rules for failed checks.
Shows 403 error page for Manoc using Error controller

=head1 METHODS

=head2 check_auth

Check if user is authenticated and restricts a2a users to API controllers

=head1 NAME

App::Manoc::Controller::Root - Root Controller for Manoc

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
